/*	Public domain	*/

#ifndef _AGAR_GUI_TEXT_H_
#define _AGAR_GUI_TEXT_H_

#include <agar/gui/drv.h>
#include <agar/gui/font.h>

#include <agar/gui/begin.h>

/* Maximum height of AG_{Push,Pop}TextState() stack. */
#ifndef AG_TEXT_STATES_MAX
#define AG_TEXT_STATES_MAX (AG_MODEL >> 1)
#endif

/* Maximum length of AG_TextParseFontSpec() font specification strings. */
#ifndef AG_TEXT_FONTSPEC_MAX
#define AG_TEXT_FONTSPEC_MAX AG_MODEL
#endif

/*
 * Maximum length of ANSI control sequences. Longer sequences are ignored.
 * Note: Increasing this impacts the performance of backward ansi-scanning
 * routines such as AG_Editable's CursorLeft().
 */
#ifndef AG_TEXT_ANSI_SEQ_MAX
#define AG_TEXT_ANSI_SEQ_MAX 64
#endif

/* Maximum ANSI parameter bytes (for those sequences we care about). */
#ifndef AG_TEXT_ANSI_PARAM_MAX
#define AG_TEXT_ANSI_PARAM_MAX 32
#endif

/* Way to justify text horizontally in a space. */
enum ag_text_justify {
	AG_TEXT_LEFT,
	AG_TEXT_CENTER,
	AG_TEXT_RIGHT
};

/* Way to align text vertically in a space. */
enum ag_text_valign {
	AG_TEXT_TOP,
	AG_TEXT_MIDDLE,
	AG_TEXT_BOTTOM
};

/* Canned dialog types. */
enum ag_text_msg_title {
	AG_MSG_ERROR,
	AG_MSG_WARNING,
	AG_MSG_INFO
};

/* ANSI 3-bit and 4-bit colors. */
enum ag_ansi_color {
	AG_ANSI_BLACK,		/* fg=30 bg=40 */
	AG_ANSI_RED,
	AG_ANSI_GREEN,
	AG_ANSI_YELLOW,
	AG_ANSI_BLUE,
	AG_ANSI_MAGENTA,
	AG_ANSI_CYAN,
	AG_ANSI_WHITE,		/* fg=37 bg=47 */
	AG_ANSI_BRIGHT_BLACK,	/* fg=90 bg=100 */
	AG_ANSI_BRIGHT_RED,
	AG_ANSI_BRIGHT_GREEN,
	AG_ANSI_BRIGHT_YELLOW,
	AG_ANSI_BRIGHT_BLUE,
	AG_ANSI_BRIGHT_MAGENTA,
	AG_ANSI_BRIGHT_CYAN,
	AG_ANSI_BRIGHT_WHITE,	/* fg=97 bg=107 */
	AG_ANSI_COLOR_LAST
};

/* Raw ANSI escape codes. */
enum ag_text_ansi_control {
	AG_ANSI_SS2,			/* G2 character set (xterm) */
	AG_ANSI_SS3,			/* G3 character set (xterm) */
	AG_ANSI_DCS,			/* User-defined keys/termcap (xterm) */
	AG_ANSI_CSI_CUU,		/* Cursor Up */
	AG_ANSI_CSI_CUD,		/* Cursor Down */
	AG_ANSI_CSI_CUF,		/* Cursor Forward */
	AG_ANSI_CSI_CUB,		/* Cursor Back */
	AG_ANSI_CSI_CNL,		/* Cursor Next Line */
	AG_ANSI_CSI_CPL,		/* Cursor Prev Line */
	AG_ANSI_CSI_CHA,		/* Cursor Horizontal Absolute */
	AG_ANSI_CSI_CUP,		/* Cursor Position */
	AG_ANSI_CSI_ED,			/* Erase in Display */
	AG_ANSI_CSI_EL,			/* Erase in Line */
	AG_ANSI_CSI_SU,			/* Scroll Up */
	AG_ANSI_CSI_SD,			/* Scroll Down */
	AG_ANSI_CSI_SGR,		/* Select Graphic Rendition */
	AG_ANSI_CSI_DSR,		/* Device Status Report */
	AG_ANSI_CSI_SCP,		/* Save Cursor Position */
	AG_ANSI_CSI_RCP,		/* Restore Cursor Position */
	AG_ANSI_ST,			/* String Terminator */
	AG_ANSI_OSC,			/* Operating System Command */
	AG_ANSI_PM,			/* Ignore (ST-terminated) */
	AG_ANSI_APC,			/* Ignore (ST-terminated) */
	AG_ANSI_SOS,			/* Ignore (ST-terminated) */
	AG_ANSI_RIS,			/* Reset to initial state */
	AG_ANSI_PRIVATE,
	AG_ANSI_LAST
};
enum ag_text_sgr_parameter {
	AG_SGR_RESET           = 0,	/* Reset all attributes */
	AG_SGR_BOLD            = 1,	/* Bold / increase intensity */
	AG_SGR_FAINT           = 2,	/* Faint / decrease intensity */
	AG_SGR_ITALIC          = 3,	/* Italic (or sometimes inverse) */
	AG_SGR_UNDERLINE       = 4,	/* Single underline */
	AG_SGR_BLINK_SLOW      = 5,	/* Under 400ms */
	AG_SGR_BLINK_FAST      = 6,	/* Over 400ms */
	AG_SGR_REVERSE         = 7,	/* Reverse video */
	AG_SGR_CONCEAL         = 8,	/* Hidden */
	AG_SGR_CROSSED_OUT     = 9,	/* Marked for deletion */
	AG_SGR_PRI_FONT        = 10,	/* Switch to primary font */
	AG_SGR_ALT_FONT_1      = 11,	/* Switch to alt font #1 */
	AG_SGR_ALT_FONT_2      = 12,	/* Switch to alt font #2 */
	AG_SGR_ALT_FONT_3      = 13,	/* Switch to alt font #3 */
	AG_SGR_ALT_FONT_4      = 14,	/* Switch to alt font #4 */
	AG_SGR_ALT_FONT_5      = 15,	/* Switch to alt font #5 */
	AG_SGR_ALT_FONT_6      = 16,	/* Switch to alt font #6 */
	AG_SGR_ALT_FONT_7      = 17,	/* Switch to alt font #7 */
	AG_SGR_ALT_FONT_8      = 18,	/* Switch to alt font #8 */
	AG_SGR_ALT_FONT_9      = 19,	/* Switch to alt font #9 */
	AG_SGR_FRAKTUR         = 20,	/* Switch to Fraktur */
	AG_SGR_UNDERLINE_2     = 21,	/* Double underline (or bold off) */
	AG_SGR_NO_INTENSITY    = 22,	/* Neither bold nor faint */
	AG_SGR_NO_FONT_STYLE   = 23,	/* Not italic, not Fraktur */
	AG_SGR_NO_UNDERLINE    = 24,	/* No single or double underline */
	AG_SGR_NO_BLINK        = 25,	/* Blink off */
	AG_SGR_NO_INVERSE      = 27,	/* Inverse video off */
	AG_SGR_REVEAL          = 28,	/* Not concealed */
	AG_SGR_NOT_CROSSED_OUT = 29,	/* Not crossed-out */
	AG_SGR_FG1             = 30,	/* Set FG color #1 */
	AG_SGR_FG2             = 31,	/* Set FG color #2 */
	AG_SGR_FG3             = 32,	/* Set FG color #3 */
	AG_SGR_FG4             = 33,	/* Set FG color #4 */
	AG_SGR_FG5             = 34,	/* Set FG color #5 */
	AG_SGR_FG6             = 35,	/* Set FG color #6 */
	AG_SGR_FG7             = 36,	/* Set FG color #7 */
	AG_SGR_FG8             = 37,	/* Set FG color #8 */
	AG_SGR_FG              = 38,	/* Set FG color (256-color) */
	AG_SGR_NO_FG           = 39,	/* Set default FG color */
	AG_SGR_BG1             = 40,	/* Set BG color #1 */
	AG_SGR_BG2             = 41,	/* Set BG color #2 */
	AG_SGR_BG3             = 42,	/* Set BG color #3 */
	AG_SGR_BG4             = 43,	/* Set BG color #4 */
	AG_SGR_BG5             = 44,	/* Set BG color #5 */
	AG_SGR_BG6             = 45,	/* Set BG color #6 */
	AG_SGR_BG7             = 46,	/* Set BG color #7 */
	AG_SGR_BG8             = 47,	/* Set BG color #8 */
	AG_SGR_BG              = 48,	/* Set BG color (256-color) */
	AG_SGR_NO_BG           = 49,	/* Set default BG color */
	AG_SGR_FRAMED          = 51,	/* Square frame */
	AG_SGR_ENCIRCLED       = 52,	/* Circular frame */
	AG_SGR_OVERLINED       = 53,	/* Overlined */
	AG_SGR_NO_FRAMES       = 54,	/* Not framed or encircled */
	AG_SGR_NOT_OVERLINED   = 55,	/* Not overlined */
					/* (56-59 unused) */
	AG_SGR_NO_FG_NO_BG     = 56,	/* Both NO_FG and NO_BG (internal) */
	AG_SGR_IDEOGRAM_1      = 60,    /* Ideogram underline or right side line */
	AG_SGR_IDEOGRAM_2      = 61,    /* Ideogram double underline, or
	                                   double line on the right side */
	AG_SGR_IDEOGRAM_3      = 62,    /* Ideogram overline or left side line */
	AG_SGR_IDEOGRAM_4      = 63,    /* Ideogram double overline, or
	                                   double line on the left side */
	AG_SGR_IDEOGRAM_5      = 64,    /* Ideogram stress marking */
	AG_SGR_IDEOGRAM_6      = 65,    /* No Ideogram attributes */
	AG_SGR_ALT_FONT_11     = 66,    /* Switch to alt font #11 (Agar ext) */
	AG_SGR_ALT_FONT_12     = 67,    /* Switch to alt font #12 (Agar ext) */
	AG_SGR_ALT_FONT_13     = 68,    /* Switch to alt font #13 (Agar ext) */
	AG_SGR_ALT_FONT_14     = 69,    /* Switch to alt font #14 (Agar ext) */
	AG_SGR_ALT_FONT_15     = 70,    /* Switch to alt font #15 (Agar ext) */
	AG_SGR_ALT_FONT_16     = 71,    /* Switch to alt font #16 (Agar ext) */
	AG_SGR_ALT_FONT_17     = 72,    /* Switch to alt font #17 (Agar ext) */
	AG_SGR_SUPERSCRIPT     = 73,    /* Superscript */
	AG_SGR_SUBSCRIPT       = 74,    /* Subscript */
	AG_SGR_NO_SUPSUBSCRIPT = 75,    /* No superscript or subscript */
	AG_SGR_BRIGHT_FG_1     = 90,
	AG_SGR_BRIGHT_FG_2     = 91,
	AG_SGR_BRIGHT_FG_3     = 92,
	AG_SGR_BRIGHT_FG_4     = 93,
	AG_SGR_BRIGHT_FG_5     = 94,
	AG_SGR_BRIGHT_FG_6     = 95,
	AG_SGR_BRIGHT_FG_7     = 96,
	AG_SGR_BRIGHT_FG_8     = 97,
					/* (98-99 unused) */
	AG_SGR_BRIGHT_BG_1     = 100,
	AG_SGR_BRIGHT_BG_2     = 101,
	AG_SGR_BRIGHT_BG_3     = 102,
	AG_SGR_BRIGHT_BG_4     = 103,
	AG_SGR_BRIGHT_BG_5     = 104,
	AG_SGR_BRIGHT_BG_6     = 105,
	AG_SGR_BRIGHT_BG_7     = 106,
	AG_SGR_BRIGHT_BG_8     = 107,
	AG_SGR_LAST            = 108
};

/* Normalized ANSI escape code. */
typedef struct ag_text_ansi {
	enum ag_text_ansi_control ctrl;   /* Type of control sequence */
	Uint len;                         /* Number of chars in sequence */
	int  n;                           /* Parameter count */
	enum ag_text_sgr_parameter sgr;   /* SGR parameter */
	AG_Color color;	                  /* For BG_COLOR / FG_COLOR */
} AG_TextANSI;

/*
 * An element of the rendering attribute stack.
 * Sync with CompareTextStates() in gui/text_cache.c.
 */
typedef struct ag_text_state {
	AG_Font *_Nonnull font;		/* Font face */
	AG_Color color;			/* Foreground text color */
	AG_Color colorBG;		/* Background color */
	AG_Color *_Nullable colorANSI;	/* ANSI color palette (3/4-bit mode) */
	enum ag_text_justify justify;	/* Justification mode */
	enum ag_text_valign valign;	/* Vertical alignment mode */
	int tabWd;			/* Width of \t in pixels */
	Uint32 _pad;
} AG_TextState;

/* Measures of rendered text. */
typedef struct ag_text_metrics {
	int w, h;			/* Dimensions in pixels */
	Uint *_Nullable wLines;		/* Width of each line */
	Uint            nLines;		/* Total line count */
	Uint32 _pad;
} AG_TextMetrics;

#ifdef AG_DEBUG
# define AG_TEXT_STATE_CUR() (((agTextStateCur >= 0 && agTextStateCur < AG_TEXT_STATES_MAX)) ? &agTextStateStack[agTextStateCur] : (AG_TextState *)AG_GenericMismatch("AG_TEXT_STATE"))
#else
# define AG_TEXT_STATE_CUR() (&agTextStateStack[agTextStateCur])
#endif

/* Begin generated block */
__BEGIN_DECLS
extern DECLSPEC int agTextFontHeight; 
extern DECLSPEC int agTextFontAscent; 
extern DECLSPEC int agTextFontLineSkip; 
extern DECLSPEC int agFontconfigInited; 
extern DECLSPEC _Nonnull_Mutex AG_Mutex agTextLock;
extern DECLSPEC AG_FontQ agFontCache;
extern DECLSPEC AG_TextState agTextStateStack[AG_TEXT_STATES_MAX];
extern DECLSPEC int agTextStateCur;
extern DECLSPEC AG_Font *_Nullable agDefaultFont;
extern DECLSPEC const AG_FontAdjustment agFontAdjustments[];
extern DECLSPEC const char *agCoreFonts[];
extern DECLSPEC const char *agTextMsgTitles[];
extern DECLSPEC const char *agTextJustifyNames[];
extern DECLSPEC const char *agTextValignNames[];
extern DECLSPEC void AG_PushTextState(void);
extern DECLSPEC void AG_CopyTextState(AG_TextState *);
extern DECLSPEC void AG_TextColorANSI(enum ag_ansi_color, const AG_Color *_Nonnull);
extern DECLSPEC AG_Font *_Nullable AG_TextFontLookup(const char *_Nullable, float, Uint);
extern DECLSPEC AG_Font *_Nullable AG_TextFontPts(float);
extern DECLSPEC AG_Font *_Nullable AG_TextFontPct(int);
extern DECLSPEC AG_Font *_Nullable AG_TextFontPctFlags(int, Uint);
extern DECLSPEC void AG_PopTextState(void);
extern DECLSPEC void AG_TextClearGlyphCache(AG_Driver *_Nonnull);
extern DECLSPEC void AG_TextSize(const char *_Nullable, int *_Nullable, int *_Nullable);
extern DECLSPEC void AG_TextSizeMulti(const char *_Nonnull, int *_Nonnull, int *_Nonnull, Uint *_Nullable *_Nonnull, Uint *_Nullable);
extern DECLSPEC void AG_TextSizeInternal(const AG_Char *_Nullable, int *_Nullable, int *_Nullable);
extern DECLSPEC void AG_TextSizeMultiInternal(const AG_Char *_Nonnull, int *_Nullable, int *_Nullable, Uint *_Nullable *_Nonnull, Uint *_Nonnull);
#ifdef AG_UNICODE
#define AG_TextSizeUCS4(s,w,h) AG_TextSizeInternal((s),(w),(h))
#define AG_TextSizeMultiUCS4(s,w,h,wL,nL) AG_TextSizeMultiInternal((s),(w),(h),(wL),(nL))
#endif
extern DECLSPEC void AG_TextMsgS(enum ag_text_msg_title, const char *_Nonnull);
extern DECLSPEC void AG_TextMsg(enum ag_text_msg_title, const char *_Nonnull, ...) FORMAT_ATTRIBUTE(printf,2,3);
extern DECLSPEC void AG_TextMsgFromError(void);
extern DECLSPEC void AG_TextTmsgS(enum ag_text_msg_title, Uint32, const char *_Nonnull);
extern DECLSPEC void AG_TextTmsg(enum ag_text_msg_title, Uint32, const char *_Nonnull, ...) FORMAT_ATTRIBUTE(printf,3,4);
extern DECLSPEC void AG_TextInfoS(const char *_Nullable, const char *_Nonnull);
extern DECLSPEC void AG_TextInfo(const char *_Nullable, const char *_Nonnull, ...) FORMAT_ATTRIBUTE(printf,2,3);
extern DECLSPEC void AG_TextWarningS(const char *_Nonnull, const char *_Nonnull);
extern DECLSPEC void AG_TextWarning(const char *_Nonnull, const char *_Nonnull, ...) FORMAT_ATTRIBUTE(printf,2,3);
extern DECLSPEC void AG_TextErrorS(const char *_Nonnull);
extern DECLSPEC void AG_TextError(const char *_Nonnull, ...) FORMAT_ATTRIBUTE(printf,1,2);
extern DECLSPEC void AG_TextAlign(int *_Nonnull, int *_Nonnull, int,int, int,int, int,int,int, int, enum ag_text_justify, enum ag_text_valign);
extern DECLSPEC AG_Surface *_Nonnull AG_TextRenderF(const char *_Nonnull, ...) _Warn_Unused_Result;
extern DECLSPEC AG_Surface *_Nonnull AG_TextRender(const char *_Nonnull) _Warn_Unused_Result;
extern DECLSPEC AG_Surface *_Nonnull AG_TextRenderRTL(const char *_Nonnull) _Warn_Unused_Result;
extern DECLSPEC AG_Surface *_Nonnull AG_TextRenderCropped(const char *_Nonnull) _Warn_Unused_Result;
extern DECLSPEC AG_Surface *_Nonnull AG_TextRenderInternal(const AG_Char *_Nonnull, AG_Font *_Nonnull, const AG_Color *_Nonnull, const AG_Color *_Nonnull) _Warn_Unused_Result;
#ifdef AG_UNICODE
# define AG_TextRenderUCS4(s) AG_TextRenderInternal((s), AG_TEXT_STATE_CUR()->font, &AG_TEXT_STATE_CUR()->colorBG, &AG_TEXT_STATE_CUR()->color)
#endif
extern DECLSPEC AG_Glyph *_Nonnull AG_TextRenderGlyph(AG_Driver *_Nonnull, AG_Font *_Nonnull, const AG_Color *_Nonnull, const AG_Color *_Nonnull, AG_Char) _Warn_Unused_Result;
extern DECLSPEC int AG_TextParseANSI(const AG_TextState *_Nonnull, AG_TextANSI *_Nonnull, const AG_Char *_Nonnull);
extern DECLSPEC int AG_TextExportUnicode_StripANSI(const char *, char *, const AG_Char *, AG_Size);
extern DECLSPEC void AG_TextColor(const AG_Color *_Nonnull);
extern DECLSPEC void AG_TextColorRGB(Uint8, Uint8, Uint8);
extern DECLSPEC void AG_TextColorRGBA(Uint8, Uint8, Uint8, Uint8);
extern DECLSPEC void AG_TextColorHex(Uint32);
extern DECLSPEC void AG_TextBGColor(const AG_Color *_Nonnull);
extern DECLSPEC void AG_TextBGColorRGB(Uint8, Uint8, Uint8);
extern DECLSPEC void AG_TextBGColorRGBA(Uint8, Uint8, Uint8, Uint8);
extern DECLSPEC void AG_TextBGColorHex(Uint32);
extern DECLSPEC void AG_TextFont(AG_Font *_Nonnull);
extern DECLSPEC void AG_TextJustify(enum ag_text_justify);
extern DECLSPEC void AG_TextValign(enum ag_text_valign);
extern DECLSPEC void AG_TextTabWidth(int);
extern DECLSPEC AG_Font *AG_SetDefaultFont(AG_Font *_Nullable);
extern DECLSPEC void AG_TextParseFontSpec(const char *_Nullable);
extern DECLSPEC int AG_InitTextSubsystem(void);
extern DECLSPEC void AG_DestroyTextSubsystem(void);
/*
 * Test whether the given character should be considered a space
 * (for word wrapping and word selection).
 */

static __inline__ _Const_Attribute int
AG_CharIsSpace(AG_Char c)
{
	switch (c) {
	case ' ':		
	case '\t':		
		return (1);
#ifdef AG_UNICODE
	case 0x00a0:		
	case 0x1680:		
	case 0x180e:		
	case 0x202f:		
	case 0x205f:		
	case 0x3000:		
	case 0xfeff:		
		return (1);
#endif
	}
#ifdef AG_UNICODE
	if (c >= 0x2000 && c <= 0x200b)	
		return (1);
#endif
	return (0);
}
/*
 * Test whether the given character should be considered punctuation
 * (for word selection).
 */

static __inline__ _Const_Attribute int
AG_CharIsPunct(AG_Char c)
{
	switch (c) {
	case '!':
	case '"':
	case '#':
	case '$':
	case '%':
	case '&':
	case '\'':
	case '(':
	case ')':
	case '*':
	case '+':
	case ',':
	case '-':
	case '.':
	case '/':
	case ':':
	case ';':
	case '<':
	case '=':
	case '>':
	case '?':
	case '@':
	case '[':
	case '\\':
	case ']':
	case '^':
	case '_':
	case '`':
	case '{':
	case '|':
	case '}':
	case '~':
		return (1);
#ifdef AG_UNICODE
	case 0x037e:		
	case 0x0589:		
	case 0x058a:		
	case 0x061b:		
	case 0x061f:		
	case 0x07f8:		
	case 0x07f9:		
	case 0x10fb:		
	case 0x2e18:		
	case 0x2e2e:		
	case 0xa4fe:		
	case 0xa4ff:		
		return (1);
#endif
	}
#ifdef AG_UNICODE
	if ((c >= 0x00a1 &&	
	     c <= 0x00bf) ||	
	    (c >= 0x055a &&	
	     c <= 0x055f) ||	
	    (c >= 0x2010 &&	
	     c <= 0x205e) ||	
	    (c >= 0x3008 &&	
	     c <= 0x301f))	
		return (1);
#endif
	return (0);
}
#define AG_CharIsSpaceOrPunct(ch) (AG_CharIsSpace(ch) || AG_CharIsPunct(ch))
#define AG_CharIsSpaceOrLF(ch) (AG_CharIsSpace(ch) || (ch == '\n'))
__END_DECLS
/* Close generated block */

#include <agar/gui/close.h>
#endif	/* _AGAR_GUI_TEXT_H_ */
