/*
 *  $Id: xyz_average.c 28808 2025-11-05 18:26:20Z yeti-dn $
 *  Copyright (C) 2019-2025 David Necas (Yeti).
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <string.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <gwy.h>

#define RUN_MODES (GWY_RUN_IMMEDIATE)

typedef struct {
    GwySurface *surface;
    GwySurface *result;
} ModuleArgs;

static gboolean module_register(void);
static void     module_main    (GwyFile *data,
                                GwyRunModeFlags mode);
static gboolean execute        (ModuleArgs *args,
                                GwyFile *data,
                                gint id);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Averages coinciding points in XYZ data."),
    "Yeti <yeti@gwyddion.net>",
    "2.0",
    "David Nečas (Yeti)",
    "2018",
};

GWY_MODULE_QUERY2(module_info, xyz_average)

static gboolean
module_register(void)
{
    gwy_xyz_func_register("xyz_average",
                          module_main,
                          N_("/_Average Identical Positions"),
                          NULL,
                          RUN_MODES,
                          GWY_MENU_FLAG_XYZ,
                          N_("Average coinciding XYZ points"));

    return TRUE;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    g_return_if_fail(mode & RUN_MODES);

    ModuleArgs args;
    gwy_clear1(args);
    gint id;

    gwy_data_browser_get_current(GWY_APP_SURFACE, &args.surface,
                                 GWY_APP_SURFACE_ID, &id,
                                 0);
    g_return_if_fail(GWY_IS_SURFACE(args.surface));

    if (execute(&args, data, id)) {
        gint newid = gwy_file_add_xyz(data, args.result);
        gwy_file_set_visible(data, GWY_FILE_XYZ, newid, TRUE);
        gwy_file_set_title(data, GWY_FILE_XYZ, newid, _("Points averaged"), TRUE);
        gwy_file_sync_items(data, GWY_FILE_XYZ, id,
                            data, GWY_FILE_XYZ, newid,
                            GWY_FILE_ITEM_PALETTE, FALSE);
        g_object_unref(args.result);
    }
}

static int
compare_xy(gconstpointer a, gconstpointer b)
{
    return memcmp(a, b, sizeof(GwyXY));
}

/* Merge exact matches.  We do not promise anything cleverer. */
static guint
merge_coninciding_xyz(GwyXYZ *xyz, guint n)
{
    qsort(xyz, n, sizeof(GwyXYZ), compare_xy);

    guint len, bstart = 0;
    guint i, pos = 0;
    for (i = 1; i < n; i++) {
        if (xyz[i].x != xyz[bstart].x || xyz[i].y != xyz[bstart].y) {
            len = i-bstart;
            xyz[pos] = xyz[bstart++];
            while (bstart < i)
                xyz[pos].z += xyz[bstart++].z;
            xyz[pos++].z /= len;
        }
    }
    len = i - bstart;
    xyz[pos] = xyz[bstart++];
    while (bstart < i)
        xyz[pos].z += xyz[bstart++].z;
    xyz[pos++].z /= len;

    gwy_debug("merged %u points", n-pos);
    return pos;
}

static gboolean
execute(ModuleArgs *args, GwyFile *data, gint id)
{
    GwySurface *surface = args->surface;
    GwySurface *out = args->result = gwy_surface_new_alike(surface);
    guint n = gwy_surface_get_npoints(surface);
    const GwyXYZ *xyz1 = gwy_surface_get_data_const(surface);
    GwyXYZ *xyz = g_new(GwyXYZ, n);

    gwy_app_wait_start(gwy_data_browser_get_window_for_data(data, GWY_FILE_XYZ, id), _("Averaging..."));
    gwy_assign(xyz, xyz1, n);
    n = merge_coninciding_xyz(xyz, n);
    gwy_app_wait_finish();

    gwy_surface_set_data_full(out, xyz, n);
    g_free(xyz);

    return TRUE;
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
