/*
 * BRLTTY - A background process providing access to the console screen (when in
 *          text mode) for a blind person using a refreshable braille display.
 *
 * Copyright (C) 1995-2025 by The BRLTTY Developers.
 *
 * BRLTTY comes with ABSOLUTELY NO WARRANTY.
 *
 * This is free software, placed under the terms of the
 * GNU Lesser General Public License, as published by the Free Software
 * Foundation; either version 2.1 of the License, or (at your option) any
 * later version. Please see the file LICENSE-LGPL for details.
 *
 * Web Page: http://brltty.app/
 *
 * This software is maintained by Dave Mielke <dave@mielke.cc>.
 */

#include "prologue.h"

#include <string.h>
#include <errno.h>

#include "cmdline.h"
#include "cmdput.h"
#include "log.h"
#include "cldr.h"
#include "utf8.h"

#define DEFAULT_OUTPUT_FORMAT "%s\\t%n\\n"

static char *opt_outputFormat;

BEGIN_COMMAND_LINE_OPTIONS(programOptions)
  { .word = "output-format",
    .letter = 'f',
    .argument = strtext("string"),
    .setting.string = &opt_outputFormat,
    .internal.setting = DEFAULT_OUTPUT_FORMAT,
    .description = strtext("The format of each output line.")
  },
END_COMMAND_LINE_OPTIONS(programOptions)

static const char *inputFile;

BEGIN_COMMAND_LINE_PARAMETERS(programParameters)
  { .name = "file",
    .description = "the CLDR annotations file to process",
    .setting = &inputFile,
  },
END_COMMAND_LINE_PARAMETERS(programParameters)

BEGIN_COMMAND_LINE_NOTES(programNotes)
  "The output format is printf-like -",
  "arbitrary text which may contain",
  "field specifiers (introduced via a percent sign [%])",
  "and/or special characters (introduced via a backslash [\\]).",
  "The default format, excluding the quotes, is \"" DEFAULT_OUTPUT_FORMAT "\".",
  "",
  "These field specifiers are recognized:",
  "  %n  the name of the character sequence",
  "  %s  the character sequence itself",
  "  %x  the character sequence in hexadecimal",
  "  %%  a literal percent sign",
  "",
  "These special characters are recognized:",
  "  \\a  alert (bell)",
  "  \\b  backspace",
  "  \\e  escape",
  "  \\f  form feed",
  "  \\n  new line",
  "  \\r  carriage return",
  "  \\t  horizontal tab",
  "  \\v  vertical tab",
  "  \\\\  literal backslasha  ",
END_COMMAND_LINE_NOTES

BEGIN_COMMAND_LINE_DESCRIPTOR(programDescriptor)
  .name = "brltty-cldr",
  .purpose = strtext("List the characters defined within a CLDR (Common Locale Data Repository Project) annotations file."),

  .options = &programOptions,
  .parameters = &programParameters,
  .notes = COMMAND_LINE_NOTES(programNotes),
END_COMMAND_LINE_DESCRIPTOR

static void
onFormatError (void) {
  exit(PROG_EXIT_SYNTAX);
}

static void
onMissingCharacter (const char *type) {
  logMessage(LOG_ERR, "missing %s character", type);
  onFormatError();
}

static void
onUnrecognizedCharacter (const char *type, int byte) {
  logMessage(LOG_ERR, "unrecognized %s character: %c", type, byte);
  onFormatError();
}

static void
putHexadecimal (const char *string) {
  size_t size = strlen(string) + 1;
  wchar_t characters[size];

  const char *byte = string;
  wchar_t *character = characters;
  wchar_t *end = character;

  convertUtf8ToWchars(&byte, &end, size);
  putHexadecimalCharacters(character, (end - character));
}

static
CLDR_ANNOTATION_HANDLER(handleAnnotation) {
  typedef enum {LITERAL, FORMAT, ESCAPE} State;
  State state = LITERAL;
  const char *format = opt_outputFormat;

  while (*format) {
    int byte = *format & 0XFF;

    switch (state) {
      case LITERAL: {
        switch (byte) {
          case '%':
            state = FORMAT;
            break;

          case '\\':
            state = ESCAPE;
            break;

          default:
            putByte(byte);
            break;
        }

        break;
      }

      case FORMAT: {
        switch (byte) {
          case 'n':
            putString(parameters->name);
            break;

          case 's':
            putString(parameters->sequence);
            break;

          case 'x':
            putHexadecimal(parameters->sequence);
            break;

          case '%':
            putByte(byte);
            break;

          default:
            onUnrecognizedCharacter("format", byte);
            return 0;
        }

        state = LITERAL;
        break;
      }

      case ESCAPE: {
        static const char escapes[] = {
          ['a'] = '\a',
          ['b'] = '\b',
          ['e'] = '\e',
          ['f'] = '\f',
          ['n'] = '\n',
          ['r'] = '\r',
          ['t'] = '\t',
          ['v'] = '\v',
          ['\\'] = '\\'
        };

        switch (byte) {
          default: {
            if (byte < ARRAY_COUNT(escapes)) {
              char escape = escapes[byte];

              if (escape) {
                putByte(escape);
                break;
              }
            }

            onUnrecognizedCharacter("escape", byte);
            return 0;
          }
        }

        state = LITERAL;
        break;
      }
    }

    format += 1;
  }

  switch (state) {
    case LITERAL:
      return 1;

    case FORMAT:
      onMissingCharacter("format");
      break;

    case ESCAPE:
      onMissingCharacter("escape");
      break;
  }

  return 0;
}

int
main (int argc, char *argv[]) {
  PROCESS_COMMAND_LINE(programDescriptor, argc, argv);

  return cldrParseFile(inputFile, handleAnnotation, NULL)?
         PROG_EXIT_SUCCESS:
         PROG_EXIT_FATAL;
}
