use core:io;
use core:net;
use crypto;

/**
 * A class that represents the state associated with a HTTP client.
 *
 * Encapsulates the state needed for persistent connections, cookies, etc. even though those
 * features are not implemented yet.
 */
class Client {
	// SSL context.
	private ClientContext context;

	// Timeout.
	private Duration timeoutDuration = 60 s;

	// Regex to match the end of the response.
	private lang:bnf:Regex endOfHeader = lang:bnf:Regex(".*\r\n\r\n");

	// Create.
	init() {
		init {
			context = ClientContext:systemDefault();
		}
	}

	// Set timeout.
	assign timeout(Duration t) {
		timeoutDuration = t;
	}

	// Perform requests. Assumes that `path` is a Http url.
	Result request(Url path) {
		unless (proto = path.protocol as HttpProtocol)
			throw HttpError("The path passed to 'request' must be a http url!");

		Request request(Method:GET, Version:HTTP_1_1, path);
		Str hostname = path[0];
		return rawRequest(hostname, proto.secure, request);
	}

	// Perform a request. This is a low-level function that does minimal conversion. Use `request`
	// for everyday use instead.
	Result rawRequest(Str host, Bool secure, Request request) {
		Nat port = if (secure) {
			443n;
		} else {
			80n;
		};

		unless (socket = connect(host, port))
			throw HttpError("Failed to connect to ${host}");

		socket.input.timeout = timeoutDuration;

		OStream out = socket.output;
		IStream in = socket.input;

		Session? ssl;
		if (secure) {
			var connected = context.connect(socket, host);
			out = connected.output;
			in = connected.input;
			ssl = connected;
		}

		// Send the request.
		BufferedOStream buffer(out, 4096);
		request.write(buffer);
		buffer.flush();

		Buffer inputBuffer;
		Response response = if (request.version == Version:HTTP_0_9) {
			// HTTP 0.9 has no headers.
			Response(request.version, Status:OK);
		} else {
			// Read the response.
			inputBuffer = in.read(4096);
			while (endOfHeader.match(inputBuffer).empty) {
				if (!in.more())
					throw HttpError("Invalid response header.");

				if (inputBuffer.filled <= 0)
					inputBuffer = inputBuffer.grow(inputBuffer.count + 4096);
				in.read(inputBuffer);
			}

			var parsedHeader = parseResponseHeader(inputBuffer);
			unless (response = parsedHeader.value) {
				throw HttpError("Failed to parse response:\n${parsedHeader.error}");
			}

			inputBuffer.shift(parsedHeader.end);
			response;
		};

		Nat? bytes;
		if (length = response.header("content-length"))
			bytes = length.nat;

		return Result(Response(), socket, ssl, inputBuffer, bytes);
	}

	// Close any open connections and cleanup any lingering state.
	void close() {
		// Nothing needs to be done at the moment.
	}


	/**
	 * Stream returned from `request` to handle the stream lifetime.
	 */
	private class Result extends IStream {
		// HTTP response. The body is always empty.
		Response response;

		// Socket.
		private NetStream socket;

		// SSL context, if any.
		private Session? ssl;

		// Remaining bytes to read.
		private Nat? remainingBytes;

		// Input stream.
		private IStream input;

		// Remaining input.
		private Buffer remainingInput;

		// Read position in the remaining input.
		private Nat remainingPos;

		// Create.
		init(Response response, NetStream socket, Session? ssl, Buffer remainingInput, Nat? remainingBytes) {
			init {
				response = response;
				socket = socket;
				ssl = ssl;
				remainingInput = remainingInput;
				remainingBytes = remainingBytes;
				input = if (ssl) { ssl.input; } else { socket.input; };
			}
		}

		// More data?
		Bool more() : override {
			if (!input.more)
				return false;

			if (remainingBytes) {
				return remainingBytes > 0;
			} else {
				return true;
			}
		}

		// Read.
		Buffer read(Buffer to) : override {
			if (remainingBytes) {
				if (to.free > remainingBytes) {
					Buffer out = readI(buffer(remainingBytes));
					for (Nat i = 0; i < out.filled; i++)
						to.push(out[i]);
					this.remainingBytes = remainingBytes - out.filled;
					return out;
				} else {
					Nat oldFilled = to.filled;
					Buffer out = readI(to);
					this.remainingBytes = remainingBytes - (out.filled - oldFilled);
					return out;
				}
			} else {
				return readI(to);
			}
		}

		// Helper for read.
		private Buffer readI(Buffer to) {
			if (remainingPos < remainingInput.filled) {
				while ((remainingPos < remainingInput.filled) & (to.free > 0))
					to.push(remainingInput[remainingPos++]);
				if (remainingPos >= remainingInput.filled)
					remainingInput = Buffer();
				return to;
			} else {
				return input.read(to);
			}
		}

		// Peek.
		Buffer peek(Buffer to) : override {
			if (remainingBytes) {
				if (to.free > remainingBytes) {
					Buffer out = peekI(buffer(remainingBytes));
					for (Nat i = 0; i < out.filled; i++)
						to.push(out[i]);
					return out;
				} else {
					Nat oldFilled = to.filled;
					return peekI(to);
				}
			} else {
				return peekI(to);
			}
		}

		// Helper for peek.
		private Buffer peekI(Buffer to) {
			if (remainingPos < remainingInput.filled) {
				Nat pos = remainingPos;
				while ((pos < remainingInput.filled) & (to.free > 0))
					to.push(remainingInput[pos++]);
				return to;
			} else {
				return input.peek(to);
			}
		}

		// Close.
		void close() : override {
			if (ssl)
				ssl.close();
			socket.close();
		}

		// Error.
		ErrorCode error() : override { input.error(); }
	}

}
