extern crate criterion;
extern crate rle_decode_fast;

use criterion::{BatchSize, Bencher, black_box, Criterion, criterion_group, criterion_main, ParameterizedBenchmark};

use std::fmt::{Debug, Formatter, Error};
use std::rc::Rc;

#[derive(Clone)]
struct Inputs {
    buffer: Vec<u8>,
    lookbehind: Vec<usize>,
    length: Vec<usize>,
    name: &'static str,
}

impl Debug for Inputs {
    fn fmt<'a>(&self, f: &mut Formatter<'a>) -> Result<(), Error> {
        write!(f, "{}", self.name)
    }
}

fn naive(bencher: &mut Bencher, inputs: &Rc<Inputs>) {
    bencher.iter_batched(
        move || inputs.as_ref().clone(),
        move |input| {
            let input = black_box(input);
            let mut buffer = input.buffer;
            for (lookbehind, length) in input.lookbehind.into_iter().zip(input.length) {
                for _ in 0..length {
                    let val = buffer[buffer.len() - lookbehind];
                    buffer.push(val);
                }
            }
            buffer
        },
        BatchSize::SmallInput,
    )
}

fn vulnerable(bencher: &mut Bencher, inputs: &Rc<Inputs>) {
    bencher.iter_batched(
        move || inputs.as_ref().clone(),
        move |input| {
            let input = black_box(input);
            let mut buffer = input.buffer;
            for (lookbehind, length) in input.lookbehind.into_iter().zip(input.length) {
                buffer.reserve(length); // allocate required memory immediately, it's faster this way
                unsafe {
                    // set length of the buffer up front so we can set elements in a slice instead of pushing
                    let len = buffer.len();
                    buffer.set_len(len + length);
                }
                for i in (buffer.len() - length)..buffer.len() {
                    unsafe {
                        let cpy = *buffer.get_unchecked(i - lookbehind);
                        *buffer.get_unchecked_mut(i) = cpy;
                    }
                }
            }
            buffer
        },
        BatchSize::SmallInput,
    )
}

fn lib(bencher: &mut Bencher, inputs: &Rc<Inputs>) {
    bencher.iter_batched(
        move || inputs.as_ref().clone(),
        move |input| {
            let input = black_box(input);
            let mut buffer = input.buffer;
            for (lookbehind, length) in input.lookbehind.into_iter().zip(input.length) {
                rle_decode_fast::rle_decode(&mut buffer, lookbehind, length);
            }
            buffer
        },
        BatchSize::SmallInput,
    )
}

fn criterion_benchmark(c: &mut Criterion) {
    let mut initial = Vec::new();
    for i in 0..5000 {
        initial.push((i & 255) as u8);
    }

    let small_lookback = vec![27, 40, 29, 15, 34, 35, 35, 22, 10, 12, 1, 31, 1, 37, 1, 5, 7, 3, 35, 40, 23, 27, 29, 34, 36, 14, 15, 31, 4, 15, 16, 16, 1, 11, 16, 34, 34, 7, 32, 31, 8, 10, 28, 21, 3, 7, 16, 11, 39, 34, 40, 14, 9, 21, 19, 4, 14, 14, 6, 34, 7, 35, 6, 24, 38, 1, 1, 3, 28, 38, 17, 40, 3, 29, 31, 23, 24, 34, 13, 25, 36, 33, 10, 1, 33, 18, 30, 26, 16, 34, 33, 29, 2, 14, 30, 24, 27, 24, 13, 22, 31, 10, 7, 29, 9, 2, 22, 16, 16, 19, 13, 10, 40, 35, 39, 26, 4, 36, 17, 24, 8, 26, 15, 4, 38, 22, 39, 15, 4, 33, 17, 6, 2, 35, 33, 8, 40, 35, 3, 20, 37, 34, 2, 16, 4, 35, 22, 10, 3, 29, 3, 19, 13, 31, 28, 23, 34, 34, 1, 24, 9, 4, 15, 20, 20, 27, 40, 22, 21, 33, 22, 1, 2, 9, 2, 29, 20, 6, 11, 23, 27, 38, 22, 20, 13, 21, 10, 35, 33, 12, 6, 40, 36, 3, 37, 15, 40, 28, 35, 14, 18, 25, 39, 33, 10, 13, 16, 22, 16, 8, 1, 19, 3, 29, 30, 35, 26, 25, 35, 5, 20, 18, 13, 29, 32, 12, 38, 18, 29, 9, 22, 19, 11, 19, 25, 8, 18, 14, 5, 3, 19, 2, 39, 15, 40, 13, 7, 5, 23, 13, 8, 34, 36, 19, 6, 25, 16, 17, 29, 40, 35, 25, 24, 3, 33, 29, 37, 15, 34, 18, 31, 36, 13, 11, 7, 19, 26, 27, 25, 26, 13, 23, 24, 40, 6, 11, 29, 24, 22, 8, 38, 4, 2, 28, 14, 35, 7, 12, 25, 14, 26, 34, 16, 3, 31, 35, 20, 26, 35, 32, 39, 2, 38, 25, 9, 5, 16, 26, 33, 7, 28, 27, 28, 19, 14, 11, 16, 22, 5, 3, 39, 30, 18, 9, 33, 24, 16, 3, 21, 36, 34, 25, 9, 7, 24, 37, 25, 27, 1, 6, 9, 9, 19, 20, 26, 24, 22, 16, 7, 24, 25, 12, 32, 12, 37, 5, 28, 7, 4, 25, 40, 12, 12, 15, 18, 6, 13, 19, 17, 11, 22, 11, 5, 31, 39, 8, 35, 19, 6, 34, 22, 26, 29, 33, 35, 20, 21, 30, 32, 13, 25, 11, 5, 33, 26, 6, 12, 25, 9, 7, 27, 34, 6, 16, 16, 15, 30, 32, 23, 20, 16, 40, 1, 21, 19, 9, 8, 38, 31, 24, 10, 39, 24, 37, 34, 23, 8, 32, 35, 27, 39, 27, 19, 13, 39, 10, 32, 27, 6, 24, 22, 37, 14, 35, 32, 26, 4, 1, 21, 33, 36, 8, 25, 20, 8, 11, 20, 10, 23, 19, 3, 3, 16, 31, 10, 19, 34, 13, 32, 18, 7, 34, 39, 20, 19, 15, 30, 25, 35, 22, 25, 3, 20, 4, 19, 10, 29, 35, 22, 35, 6, 23, 13, 15, 17, 14, 22, 38, 38, 7, 4, 28, 34, 30, 16, 21, 12, 13, 17, 13, 11, 29, 1, 33, 20, 40, 19, 20, 35, 25, 21, 2, 32, 22, 3, 35, 39, 5, 1, 15, 12, 12, 26, 12, 40, 16, 37, 33, 12, 34, 8, 18, 32, 29, 10, 19, 17, 27, 4, 10, 26, 6, 18, 39, 13, 24, 37, 39, 15, 31, 28, 9, 29, 29, 39, 2, 17, 35, 24, 2, 17, 30, 10, 38, 23, 17, 10, 32, 30, 25, 8, 25, 4, 29, 20, 9, 33, 13, 24, 40, 30, 12, 20, 12, 14, 36, 14, 31, 15, 17, 22, 14, 23, 40, 39, 12, 13, 30, 4, 12, 37, 40, 34, 5, 8, 9, 2, 12, 2, 22, 36, 33, 31, 29, 1, 14, 25, 8, 12, 14, 23, 14, 28, 20, 3, 39, 30, 31, 38, 39, 30, 22, 14, 24, 23, 4, 4, 12, 12, 17, 20, 21, 7, 5, 34, 33, 7, 19, 39, 10, 5, 32, 1, 17, 5, 15, 13, 19, 1, 27, 20, 34, 4, 13, 14, 37, 22, 39, 32, 28, 22, 11, 28, 37, 22, 17, 8, 27, 11, 37, 28, 16, 2, 34, 40, 38, 20, 33, 24, 18, 37, 7, 35, 18, 32, 24, 16, 6, 27, 40, 26, 39, 39, 9, 35, 23, 21, 33, 13, 12, 38, 37, 31, 27, 15, 31, 37, 12, 36, 10, 15, 24, 31, 5, 16, 4, 8, 26, 11, 6, 23, 23, 10, 35, 33, 17, 34, 32, 13, 15, 31, 3, 28, 40, 29, 23, 23, 8, 16, 5, 4, 26, 7, 3, 15, 40, 30, 19, 23, 36, 5, 36, 36, 33, 36, 19, 26, 1, 13, 8, 28, 30, 30, 22, 10, 31, 12, 8, 28, 11, 39, 39, 11, 38, 26, 3, 8, 21, 14, 35, 23, 18, 37, 19, 18, 2, 37, 32, 21, 28, 38, 24, 6, 21, 36, 1, 31, 27, 4, 30, 39, 8, 12, 17, 24, 15, 40, 1, 37, 20, 10, 20, 37, 36, 18, 23, 29, 37, 21, 20, 2, 37, 39, 17, 23, 11, 34, 17, 39, 2, 22, 16, 2, 7, 38, 27, 7, 10, 34, 38, 16, 5, 39, 14, 9, 25, 26, 19, 14, 6, 31, 32, 38, 3, 12, 17, 13, 10, 39, 8, 12, 40, 5, 25, 1, 10, 11, 35, 19, 36, 13, 39, 38, 3, 32, 3, 23, 31, 38, 15, 17, 21, 8, 27, 21, 4, 29, 37, 12, 32, 40, 36, 5, 39, 20, 19, 39, 25, 14, 21, 40, 4, 22, 34, 11, 20, 27, 8, 35, 1, 8, 13, 27, 8, 4, 36, 9, 1, 4, 40, 36, 13, 1, 35, 16, 5, 20, 9, 23, 28, 5, 5, 25, 9, 9, 9, 20, 1, 29, 9, 36, 14, 25, 3, 40, 9, 32, 27, 22, 6, 15, 27, 12, 33, 17, 39, 13, 37, 24, 24, 16, 6, 32, 36, 23, 37, 37, 38, 18, 20];
    let small_lengths = vec![38, 9, 7, 17, 8, 37, 14, 4, 13, 27, 1, 2, 26, 34, 23, 18, 10, 19, 4, 24, 30, 23, 21, 29, 23, 17, 24, 34, 5, 7, 2, 25, 18, 34, 15, 1, 6, 11, 26, 15, 35, 11, 35, 19, 23, 37, 19, 5, 2, 15, 39, 3, 24, 36, 29, 27, 29, 25, 28, 12, 8, 20, 17, 27, 28, 8, 3, 36, 4, 14, 35, 11, 3, 22, 13, 3, 17, 28, 33, 10, 31, 16, 14, 34, 32, 28, 40, 10, 1, 7, 4, 36, 9, 20, 25, 15, 22, 24, 19, 28, 9, 9, 16, 38, 27, 16, 1, 10, 33, 12, 33, 13, 20, 17, 15, 25, 7, 15, 26, 12, 20, 14, 27, 40, 20, 30, 26, 29, 24, 27, 30, 3, 24, 31, 31, 4, 27, 19, 7, 38, 29, 17, 19, 23, 32, 9, 31, 21, 1, 20, 21, 35, 34, 32, 12, 24, 30, 25, 5, 40, 22, 14, 11, 17, 2, 17, 38, 40, 18, 31, 2, 7, 5, 29, 37, 26, 35, 36, 40, 9, 4, 4, 12, 25, 8, 3, 8, 28, 12, 12, 9, 19, 8, 36, 36, 2, 9, 35, 23, 29, 22, 30, 18, 19, 16, 1, 15, 20, 13, 20, 38, 11, 18, 30, 40, 27, 29, 26, 26, 40, 24, 33, 18, 21, 38, 32, 6, 16, 3, 11, 16, 29, 37, 14, 37, 29, 32, 24, 30, 38, 35, 36, 6, 12, 27, 15, 11, 12, 7, 35, 28, 28, 34, 17, 13, 18, 32, 25, 28, 25, 14, 32, 40, 32, 15, 33, 11, 30, 12, 4, 4, 14, 37, 22, 22, 10, 15, 20, 25, 18, 21, 4, 24, 38, 6, 34, 13, 3, 5, 38, 11, 32, 19, 14, 20, 17, 11, 11, 9, 15, 16, 29, 13, 16, 15, 9, 2, 38, 8, 3, 20, 4, 16, 17, 20, 35, 20, 1, 35, 38, 38, 16, 14, 12, 5, 18, 34, 14, 6, 3, 9, 3, 21, 39, 28, 2, 5, 15, 22, 30, 17, 8, 31, 16, 4, 13, 39, 35, 14, 4, 4, 14, 23, 4, 33, 21, 12, 26, 16, 21, 31, 34, 15, 12, 35, 22, 18, 36, 2, 24, 22, 40, 7, 16, 30, 22, 19, 16, 8, 13, 28, 29, 5, 5, 39, 23, 6, 32, 25, 10, 39, 33, 11, 17, 9, 9, 27, 31, 3, 28, 35, 12, 17, 11, 36, 12, 28, 5, 7, 8, 26, 4, 17, 30, 39, 25, 20, 27, 22, 7, 7, 23, 35, 9, 29, 2, 11, 6, 21, 5, 12, 26, 25, 16, 30, 34, 40, 37, 31, 17, 36, 18, 39, 40, 21, 1, 11, 19, 12, 39, 33, 36, 17, 4, 33, 8, 13, 18, 32, 20, 34, 19, 30, 7, 30, 25, 7, 12, 15, 17, 34, 21, 23, 26, 14, 4, 10, 35, 25, 37, 12, 34, 6, 32, 12, 21, 36, 20, 25, 17, 27, 24, 36, 36, 20, 17, 40, 40, 27, 7, 19, 17, 2, 19, 10, 23, 22, 20, 31, 14, 29, 4, 1, 13, 22, 21, 21, 9, 1, 15, 23, 35, 38, 22, 34, 16, 1, 7, 17, 7, 11, 15, 34, 13, 22, 37, 11, 17, 10, 17, 3, 18, 21, 15, 3, 7, 21, 39, 15, 8, 5, 31, 29, 23, 15, 2, 16, 34, 2, 29, 8, 34, 18, 13, 37, 4, 18, 17, 39, 12, 27, 30, 33, 29, 18, 24, 38, 35, 30, 6, 12, 17, 35, 21, 9, 40, 30, 28, 37, 19, 29, 14, 12, 6, 34, 37, 7, 36, 19, 40, 21, 28, 12, 3, 32, 1, 28, 21, 31, 37, 23, 8, 21, 32, 19, 25, 30, 29, 20, 28, 22, 30, 1, 35, 5, 16, 10, 14, 28, 17, 40, 15, 38, 34, 11, 28, 33, 19, 8, 9, 1, 1, 17, 5, 33, 4, 20, 37, 28, 24, 6, 35, 36, 36, 30, 30, 33, 3, 2, 27, 30, 35, 29, 4, 29, 38, 11, 29, 3, 22, 26, 5, 5, 29, 15, 19, 17, 1, 3, 33, 9, 34, 12, 39, 35, 26, 40, 40, 23, 2, 28, 37, 17, 35, 14, 21, 35, 7, 33, 40, 24, 18, 12, 30, 5, 5, 10, 18, 9, 24, 33, 18, 35, 37, 29, 5, 14, 32, 3, 40, 35, 22, 15, 12, 14, 36, 5, 29, 39, 40, 20, 10, 14, 15, 2, 20, 38, 25, 16, 36, 36, 36, 23, 10, 31, 22, 12, 35, 4, 18, 17, 18, 15, 11, 6, 32, 7, 22, 40, 9, 23, 5, 31, 14, 33, 36, 3, 24, 23, 18, 24, 29, 32, 7, 37, 35, 11, 13, 24, 33, 27, 38, 9, 35, 29, 12, 38, 18, 30, 1, 40, 3, 12, 10, 4, 35, 21, 11, 18, 36, 16, 13, 6, 14, 12, 40, 28, 28, 6, 1, 38, 11, 13, 22, 25, 36, 38, 23, 15, 8, 36, 9, 25, 13, 24, 24, 20, 26, 29, 1, 24, 11, 40, 4, 24, 33, 40, 34, 9, 25, 1, 22, 18, 4, 12, 10, 24, 39, 38, 18, 13, 9, 21, 11, 20, 5, 36, 24, 2, 17, 32, 18, 30, 7, 34, 8, 18, 14, 16, 22, 34, 16, 29, 30, 26, 31, 36, 21, 19, 4, 12, 37, 29, 38, 35, 40, 38, 2, 29, 34, 19, 38, 16, 8, 28, 22, 11, 20, 22, 25, 6, 28, 13, 24, 16, 35, 17, 6, 10, 6, 22, 21, 36, 34, 40, 10, 2, 17, 40, 30, 9, 3, 7, 33, 19, 17, 25, 8, 26, 32, 19, 11, 8, 28, 4, 30, 1, 13, 37, 18, 14, 19, 28, 24, 7, 36, 30, 16, 14, 21, 1, 36, 25, 32, 27, 16, 27, 8, 18, 37, 30, 9, 15, 33, 5, 38, 4, 27, 10, 29, 3, 24, 40, 14, 27, 15, 33, 2, 30, 13, 7, 27, 22, 18, 17, 40, 38, 39, 4, 10, 35, 27, 38, 24, 8, 30, 24, 7, 28, 2];
    let big_lookback = vec![1083, 2905, 1772, 3351, 3786, 3410, 3491, 2588, 4141, 1118, 3947, 4880, 2311, 4715, 3463, 4206, 2328, 4743, 4851, 2679, 3033, 2511, 4036, 1010, 2830, 3254, 2014, 2917, 2315, 4290, 1118, 4964, 4659, 2817, 4985, 3236, 3234, 2391, 1789, 2490, 2666, 3759, 2289, 2558, 2767, 4762, 4198, 2998, 3360, 2013, 4173, 3797, 3510, 3864, 1417, 2340, 4513, 4403, 2775, 3223, 1078, 4108, 1501, 4771, 3648, 4018, 1672, 4874, 1924, 3240, 1114, 4137, 2087, 4463, 2124, 2721, 1347, 2437, 3449, 3819, 3264, 4317, 4720, 2247, 2218, 1469, 1052, 2953, 1786, 3005, 2860, 3312, 3665, 1351, 2173, 3435, 3288, 1250, 2253, 4717, 2359, 2133, 1650, 4373, 2141, 2937, 2120, 4256, 2613, 3021, 1084, 3933, 1140, 2688, 3786, 4562, 2880, 2477, 3885, 4572, 3168, 1354, 3581, 3685, 4257, 3155, 2414, 1219, 1107, 3742, 4953, 2110, 2670, 2337, 3822, 4276, 3749, 4418, 3206, 3045, 2027, 2107, 1296, 3059, 2148, 4792, 4007, 1037, 2774, 1366, 2839, 4237, 2191, 1165, 3433, 4337, 2771, 1980, 4839, 4701, 1746, 4192, 4707, 1249, 1426, 4547, 1698, 4188, 1032, 3834, 1629, 2905, 3935, 1293, 3647, 1520, 1441, 1589, 2988, 4282, 3389, 3178, 4827, 1131, 3894, 4894, 1530, 3871, 1121, 2020, 3275, 2021, 2696, 2707, 3965, 2389, 1341, 4364, 4909, 2582, 1935, 2140, 2509, 4960, 1173, 3180, 4732, 4615, 1190, 4246, 4178, 3739, 3212, 4545, 1474, 2650, 1558, 3777, 1217, 4263, 4413, 4924, 4331, 3474, 2368, 1871, 1899, 2653, 3953, 4853, 1976, 2701, 2409, 1250, 2660, 1603, 2916, 4350, 4040, 3062, 2398, 1232, 3590, 1494, 2717, 2302, 1434, 1916, 3491, 4576, 4027, 1473, 2152, 3152, 3409, 3322, 2439, 2977, 1823, 3837, 3581, 4202, 1779, 3614, 1682, 3291, 4726, 3982, 2305, 2379, 3687, 1858, 4232, 3585, 1884, 1853, 3305, 4496, 2344, 4179, 3372, 1719, 1152, 3815, 2226, 4089, 2868, 1151, 1947, 2540, 3795, 2588, 4943, 4386, 2464, 2583, 3871, 1031, 1742, 1775, 3195, 2636, 4685, 4830, 4614, 1628, 1597, 4948, 3960, 3419, 1360, 3117, 4414, 2188, 4082, 2957, 3558, 3595, 1560, 2639, 4344, 3388, 2332, 1992, 4533, 1490, 4072, 4497, 4884, 4338, 4502, 3852, 3331, 3255, 2914, 3277, 4326, 4124, 1543, 2817, 4275, 4179, 1661, 2003, 2333, 3264, 4106, 1417, 3255, 4392, 1964, 3789, 1351, 3104, 2727, 1629, 1348, 4157, 4709, 3372, 2450, 3627, 3028, 4691, 2337, 4515, 1071, 4176, 4099, 2685, 1974, 3220, 4552, 2733, 1969, 4744, 3912, 2148, 3353, 2410, 4902, 4323, 1769, 2958, 2116, 2744, 3342, 1787, 2002, 3170, 2363, 2399, 3951, 1241, 4687, 1836, 4183, 1553, 3943, 3175, 2291, 4998, 2691, 4593, 1706, 4001, 3414, 3325, 2862, 1249, 1895, 4157, 1241, 1104, 4814, 3538, 4582, 1247, 2826, 2057, 1331, 4849, 3337, 1226, 1837, 1907, 3022, 2301, 3978, 4627, 4408, 3036, 1770, 1917, 3657, 1982, 3931, 2887, 4914, 4458, 4029, 3679, 2187, 2550, 3727, 2010, 4977, 1058, 4061, 3052, 1193, 1505, 4209, 4542, 4811, 2794, 4579, 1444, 2320, 1955, 2454, 1231, 4044, 3759, 1963, 3413, 1529, 4850, 3709, 2227, 2015, 3233, 1055, 4259, 4084, 4663, 1654, 4642, 2670, 3032, 3510, 3404, 4050, 2284, 2881, 1260, 2671, 2473, 4747, 4261, 1829, 4215, 2802, 1953, 1258, 4406, 2953, 2520, 2407, 3576, 3066, 1182, 1090, 4927, 1702, 1788, 2048, 2800, 3997, 4064, 4388, 4752, 1311, 1198, 2679, 2011, 4636, 4700, 4537, 4763, 2099, 1772, 2543, 2080, 3340, 4442, 1721, 2971, 3000, 2246, 4919, 1209, 1370, 3632, 1330, 1919, 3755, 3852, 2124, 3457, 4356, 4992, 4071, 4093, 4243, 4493, 2877, 2402, 4890, 1584, 1585, 3920, 4202, 4848, 1271, 2975, 1638, 2582, 1246, 3026, 3759, 3835, 3229, 2290, 1238, 3852, 2390, 4000, 1746, 1600, 3177, 1173, 3605, 1676, 4835, 4458, 2946, 2431, 1677, 3107, 1492, 4031, 2680, 2610, 4584, 1710, 3113, 1128, 3911, 4424, 4631, 1939, 4236, 3512, 1469, 4867, 1479, 1472, 4353, 3046, 2663, 2307, 1504, 1666, 2797, 2092, 4592, 2136, 4823, 3720, 1462, 2622, 3695, 4451, 2938, 2774, 2721, 3064, 2477, 1711, 3546, 2713, 2351, 4636, 1369, 3111, 2587, 3571, 4123, 3116, 1017, 1802, 4370, 1736, 3660, 2174, 4440, 4254, 1035, 1437, 2966, 2682, 3873, 3362, 3198, 2982, 3311, 4324, 2354, 3427, 1907, 4524, 3530, 1862, 4143, 4937, 2119, 4248, 3876, 4392, 1881, 1672, 1154, 3880, 2998, 2083, 1528, 1046, 1191, 2807, 3117, 2687, 1111, 3772, 4863, 2650, 4925, 4504, 1945, 4125, 4542, 3361, 3003, 4405, 1640, 2268, 4491, 3691, 4381, 1301, 1305, 3461, 4877, 4591, 2659, 3837, 3591, 1526, 4296, 4115, 4008, 3227, 2001, 1861, 3808, 4847, 4407, 4207, 1802, 1306, 4721, 1282, 1281, 4260, 2807, 3053, 2767, 3511, 4380, 1240, 3461, 1039, 1644, 4452, 3025, 3173, 2791, 2294, 3956, 3208, 3749, 1818, 2597, 1949, 4179, 3317, 3165, 2662, 4903, 2170, 3808, 2642, 3754, 4575, 2283, 2845, 2202, 2745, 4208, 3947, 3461, 1256, 4427, 4863, 3839, 1425, 3198, 4816, 3642, 3927, 2720, 4988, 2537, 2015, 3487, 2120, 4092, 4194, 3940, 2802, 3023, 2313, 1838, 2864, 4781, 3634, 3514, 3690, 3833, 1736, 1130, 4549, 3257, 4793, 4773, 4021, 3145, 3141, 3207, 2733, 1393, 1970, 2094, 4134, 1974, 4023, 1393, 4329, 4669, 1394, 4305, 2208, 4903, 4607, 4672, 4956, 1070, 2759, 2241, 4583, 4105, 2004, 1935, 1890, 3844, 2046, 4699, 4701, 2475, 1430, 3502, 4498, 3578, 2990, 2435, 2564, 1203, 3506, 3666, 2275, 4288, 3912, 2038, 3954, 2801, 4998, 4418, 2051, 4055, 2536, 3868, 2005, 3240, 2432, 4090, 4517, 3395, 4393, 3762, 4277, 4192, 3155, 3017, 4744, 4296, 1324, 1181, 2266, 4829, 3102, 4507, 3736, 1630, 2984, 4567, 4038, 2225, 1567, 2012, 2624, 2248, 4193, 2813, 3287, 2749, 3969, 3478, 4885, 1688, 1480, 4530, 4513, 1173, 2218, 2441, 1217, 3195, 4891, 1471, 1719, 1801, 3007, 3700, 4410, 3043, 1340, 3222, 1440, 2216, 3195, 1352, 3927, 1147, 1963, 2593, 1195, 3001, 2140, 1544, 1870, 2134, 1667, 3416, 1541, 4293, 1149, 2017, 2516, 3216, 3734, 2360, 1278, 1500, 3137, 2028, 1839, 4463, 4820, 2956, 2903, 4210, 3851, 2860, 3590, 4985, 3583, 1943, 2180, 3243, 3061, 2949, 4496, 1094, 2249, 4858, 2447, 3589, 4771, 3052, 1209, 3654, 4184, 2377, 4127, 2283, 2672, 3035, 1122, 2546, 2282, 4268, 2681, 4990, 4196, 2497, 2017, 2746, 2487, 1846, 3345, 3640, 2406, 4039, 2157, 4506, 1172, 2039, 3836, 3571, 1483, 2761, 2121, 4446, 2608, 3077, 2442, 4726, 1076, 1274, 4114, 4649, 1012, 4960, 3430, 4188, 3741, 3964, 2352, 3453, 4244];
    let big_lengths = vec![89, 30, 270, 287, 135, 17, 167, 117, 42, 90, 221, 290, 395, 267, 256, 172, 247, 162, 274, 240, 360, 341, 271, 265, 123, 92, 357, 375, 178, 348, 341, 251, 114, 322, 11, 174, 182, 88, 96, 10, 52, 57, 82, 72, 146, 76, 196, 29, 365, 175, 366, 235, 375, 291, 21, 39, 28, 218, 40, 61, 93, 344, 193, 193, 192, 58, 312, 305, 39, 20, 252, 21, 73, 301, 228, 341, 191, 353, 201, 215, 293, 359, 204, 50, 354, 130, 6, 346, 274, 75, 324, 44, 238, 275, 190, 65, 51, 41, 62, 387, 148, 123, 109, 265, 247, 191, 249, 158, 137, 188, 258, 122, 245, 373, 379, 114, 203, 210, 132, 113, 334, 69, 195, 12, 6, 191, 385, 223, 299, 212, 343, 149, 52, 61, 301, 115, 164, 309, 323, 206, 374, 381, 10, 126, 49, 268, 276, 65, 368, 35, 8, 337, 119, 295, 222, 57, 178, 92, 385, 299, 56, 166, 256, 12, 368, 329, 391, 245, 32, 292, 224, 6, 232, 215, 97, 172, 217, 221, 51, 378, 195, 220, 212, 164, 71, 15, 25, 31, 113, 298, 278, 71, 294, 279, 174, 31, 267, 181, 266, 48, 100, 315, 358, 73, 333, 374, 48, 276, 101, 6, 197, 62, 149, 189, 26, 60, 13, 341, 167, 82, 314, 279, 220, 178, 367, 121, 74, 310, 287, 166, 290, 224, 242, 388, 50, 114, 314, 6, 39, 195, 364, 370, 314, 148, 206, 205, 310, 281, 180, 162, 218, 210, 145, 321, 238, 268, 162, 322, 326, 329, 138, 351, 349, 374, 358, 376, 172, 50, 277, 44, 266, 162, 388, 64, 313, 41, 280, 286, 327, 302, 346, 335, 222, 373, 7, 285, 257, 242, 211, 11, 14, 252, 337, 268, 46, 331, 52, 114, 400, 139, 148, 62, 186, 97, 165, 315, 302, 335, 373, 100, 72, 195, 182, 87, 188, 91, 244, 3, 49, 2, 359, 161, 282, 215, 70, 359, 229, 57, 95, 11, 259, 41, 99, 81, 33, 127, 14, 94, 11, 234, 6, 248, 9, 341, 280, 373, 10, 162, 97, 179, 399, 134, 155, 398, 348, 314, 275, 84, 23, 121, 326, 59, 36, 283, 115, 279, 24, 320, 326, 129, 399, 130, 396, 87, 303, 94, 356, 248, 311, 137, 52, 162, 206, 320, 149, 173, 29, 141, 313, 13, 33, 331, 22, 375, 72, 365, 2, 112, 330, 93, 201, 294, 67, 180, 317, 353, 268, 247, 254, 326, 177, 358, 72, 84, 86, 323, 152, 56, 293, 56, 252, 393, 265, 141, 385, 324, 77, 215, 316, 87, 169, 128, 145, 375, 340, 227, 193, 26, 238, 31, 242, 142, 224, 392, 48, 131, 107, 72, 284, 172, 334, 26, 57, 124, 252, 185, 239, 288, 109, 356, 140, 306, 6, 286, 214, 393, 396, 201, 375, 171, 218, 196, 177, 210, 256, 385, 28, 57, 58, 44, 34, 85, 175, 10, 40, 240, 216, 196, 113, 132, 368, 116, 11, 18, 252, 221, 263, 70, 219, 324, 370, 306, 246, 253, 25, 15, 380, 9, 342, 277, 133, 163, 34, 301, 71, 50, 319, 117, 130, 91, 168, 40, 77, 41, 195, 168, 381, 76, 236, 131, 385, 382, 289, 82, 246, 38, 347, 60, 89, 281, 217, 99, 170, 381, 66, 80, 279, 315, 351, 242, 252, 236, 210, 111, 96, 111, 89, 285, 199, 76, 164, 2, 211, 359, 59, 249, 186, 186, 262, 345, 326, 329, 335, 109, 334, 60, 287, 227, 381, 127, 215, 316, 298, 129, 82, 223, 287, 177, 297, 49, 137, 77, 320, 248, 223, 373, 273, 300, 37, 324, 184, 293, 400, 137, 377, 5, 2, 20, 183, 118, 277, 141, 276, 341, 321, 154, 290, 350, 292, 291, 98, 278, 224, 26, 297, 9, 357, 390, 120, 129, 155, 60, 205, 384, 186, 228, 130, 127, 176, 311, 369, 45, 43, 57, 229, 397, 262, 364, 98, 246, 193, 23, 98, 263, 347, 279, 136, 133, 285, 355, 107, 262, 88, 200, 124, 62, 18, 1, 98, 19, 212, 137, 154, 115, 85, 376, 356, 72, 59, 212, 201, 203, 292, 118, 183, 268, 127, 14, 303, 149, 339, 290, 54, 137, 29, 1, 353, 280, 194, 199, 68, 123, 109, 293, 296, 358, 36, 255, 151, 365, 312, 158, 239, 316, 93, 10, 147, 203, 47, 214, 336, 105, 85, 11, 310, 338, 338, 26, 366, 270, 150, 373, 361, 121, 332, 349, 298, 319, 274, 20, 246, 303, 251, 200, 188, 333, 20, 241, 283, 66, 156, 122, 226, 229, 317, 217, 54, 387, 199, 166, 273, 102, 236, 297, 202, 86, 208, 125, 326, 353, 392, 69, 200, 106, 2, 72, 188, 9, 289, 15, 250, 359, 307, 47, 165, 203, 131, 69, 102, 133, 340, 269, 223, 22, 145, 44, 261, 92, 331, 22, 11, 155, 217, 84, 295, 313, 76, 71, 69, 277, 365, 55, 251, 198, 256, 70, 207, 329, 112, 203, 61, 306, 318, 304, 62, 98, 112, 57, 130, 21, 384, 285, 35, 288, 158, 171, 302, 47, 382, 23, 217, 258, 389, 95, 277, 5, 272, 53, 44, 281, 100, 326, 50, 371, 86, 161, 306, 128, 319, 121, 197, 130, 341, 281, 33, 133, 208, 3, 27, 348, 34, 275, 233, 77, 226, 112, 306, 12, 383, 331, 180, 212, 120, 164, 311, 31, 122, 56, 26, 319, 314, 70, 204, 225, 69, 228, 153, 211, 286, 374, 35, 387, 206, 347, 306, 39, 167, 367, 143, 280, 370, 283, 89, 201, 307, 178, 388, 113, 379, 384, 281, 265, 245, 23, 154, 375, 3, 400, 88, 259, 49, 262, 70, 73, 282, 116, 161, 35, 239, 257, 271, 195, 149, 303, 153, 107, 258, 287, 102, 150, 395, 288, 12, 118, 172, 252, 124, 340, 386, 214, 370, 343, 149, 227, 345, 41, 249, 272, 22, 352, 273, 274, 248, 337, 252, 181, 295, 244, 110, 113, 298, 335, 149, 354, 260, 12, 266, 379, 388, 37, 71, 105, 99, 134, 183, 290, 226, 393, 323, 305];

    let inputs = vec![
        Rc::new(Inputs {
            buffer: initial.clone(),
            lookbehind: small_lookback.clone(),
            length: small_lengths.clone(),
            name: "small-lb-small-length",
        }),
        Rc::new(Inputs {
            buffer: initial.clone(),
            lookbehind: small_lookback.clone(),
            length: big_lengths.clone(),
            name: "small-lb-big-length",
        }),
        Rc::new(Inputs {
            buffer: initial.clone(),
            lookbehind: big_lookback.clone(),
            length: small_lengths.clone(),
            name: "big-lb-small-length",
        }),
        Rc::new(Inputs {
            buffer: initial.clone(),
            lookbehind: big_lookback.clone(),
            length: big_lengths.clone(),
            name: "big-lb-big-length",
        }),
    ];

    c.bench("rle-impl",
            ParameterizedBenchmark::new("naive", naive, inputs)
                .with_function("vulnerable", vulnerable)
                .with_function("lib", lib)
    );
}

criterion_group!(benches, criterion_benchmark);
criterion_main!(benches);
