package org.opentest4j.reporting.tooling.spi.htmlreport;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link Labels}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code new Labels.Builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableLabels implements Labels {
  private final List<String> content;

  private ImmutableLabels(List<String> content) {
    this.content = content;
  }

  /**
   * @return The value of the {@code content} attribute
   */
  @Override
  public List<String> getContent() {
    return content;
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Labels#getContent() content}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableLabels withContent(String... elements) {
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableLabels(newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Labels#getContent() content}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of content elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableLabels withContent(Iterable<String> elements) {
    if (this.content == elements) return this;
    List<String> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableLabels(newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableLabels} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableLabels
        && equalsByValue((ImmutableLabels) another);
  }

  private boolean equalsByValue(ImmutableLabels another) {
    return content.equals(another.content);
  }

  /**
   * Computes a hash code from attributes: {@code content}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + content.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Labels} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Labels{"
        + "content=" + content
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Labels} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Labels instance
   */
  public static Labels copyOf(Labels instance) {
    if (instance instanceof ImmutableLabels) {
      return (ImmutableLabels) instance;
    }
    return new Labels.Builder()
        .addAllContent(instance.getContent())
        .build();
  }

  /**
   * Builds instances of type {@link Labels Labels}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static class Builder {
    private List<String> content = new ArrayList<String>();

    /**
     * Creates a builder for {@link Labels Labels} instances.
     * <pre>
     * new Labels.Builder()
     *    .addContent|addAllContent(String) // {@link Labels#getContent() content} elements
     *    .build();
     * </pre>
     */
    public Builder() {
      if (!(this instanceof Labels.Builder)) {
        throw new UnsupportedOperationException("Use: new Labels.Builder()");
      }
    }

    /**
     * Adds one element to {@link Labels#getContent() content} list.
     * @param element A content element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Labels.Builder addContent(String element) {
      this.content.add(Objects.requireNonNull(element, "content element"));
      return (Labels.Builder) this;
    }

    /**
     * Adds elements to {@link Labels#getContent() content} list.
     * @param elements An array of content elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Labels.Builder addContent(String... elements) {
      for (String element : elements) {
        this.content.add(Objects.requireNonNull(element, "content element"));
      }
      return (Labels.Builder) this;
    }


    /**
     * Sets or replaces all elements for {@link Labels#getContent() content} list.
     * @param elements An iterable of content elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Labels.Builder content(Iterable<String> elements) {
      this.content.clear();
      return addAllContent(elements);
    }

    /**
     * Adds elements to {@link Labels#getContent() content} list.
     * @param elements An iterable of content elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Labels.Builder addAllContent(Iterable<String> elements) {
      for (String element : elements) {
        this.content.add(Objects.requireNonNull(element, "content element"));
      }
      return (Labels.Builder) this;
    }

    /**
     * Builds a new {@link Labels Labels}.
     * @return An immutable instance of Labels
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public Labels build() {
      return new ImmutableLabels(createUnmodifiableList(true, content));
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<? extends T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
