package org.opentest4j.reporting.tooling.spi.htmlreport;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link PreFormattedOutput}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code new PreFormattedOutput.Builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutablePreFormattedOutput
    implements PreFormattedOutput {
  private final String content;

  private ImmutablePreFormattedOutput(String content) {
    this.content = content;
  }

  /**
   * @return The value of the {@code content} attribute
   */
  @Override
  public String getContent() {
    return content;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link PreFormattedOutput#getContent() content} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for content
   * @return A modified copy or the {@code this} object
   */
  public final ImmutablePreFormattedOutput withContent(String value) {
    String newValue = Objects.requireNonNull(value, "content");
    if (this.content.equals(newValue)) return this;
    return new ImmutablePreFormattedOutput(newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutablePreFormattedOutput} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutablePreFormattedOutput
        && equalsByValue((ImmutablePreFormattedOutput) another);
  }

  private boolean equalsByValue(ImmutablePreFormattedOutput another) {
    return content.equals(another.content);
  }

  /**
   * Computes a hash code from attributes: {@code content}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + content.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code PreFormattedOutput} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "PreFormattedOutput{"
        + "content=" + content
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link PreFormattedOutput} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable PreFormattedOutput instance
   */
  public static PreFormattedOutput copyOf(PreFormattedOutput instance) {
    if (instance instanceof ImmutablePreFormattedOutput) {
      return (ImmutablePreFormattedOutput) instance;
    }
    return new PreFormattedOutput.Builder()
        .content(instance.getContent())
        .build();
  }

  /**
   * Builds instances of type {@link PreFormattedOutput PreFormattedOutput}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static class Builder {
    private static final long INIT_BIT_CONTENT = 0x1L;
    private long initBits = 0x1L;

    private String content;

    /**
     * Creates a builder for {@link PreFormattedOutput PreFormattedOutput} instances.
     * <pre>
     * new PreFormattedOutput.Builder()
     *    .content(String) // required {@link PreFormattedOutput#getContent() content}
     *    .build();
     * </pre>
     */
    public Builder() {
      if (!(this instanceof PreFormattedOutput.Builder)) {
        throw new UnsupportedOperationException("Use: new PreFormattedOutput.Builder()");
      }
    }

    /**
     * Initializes the value for the {@link PreFormattedOutput#getContent() content} attribute.
     * @param content The value for content 
     * @return {@code this} builder for use in a chained invocation
     */
    public final PreFormattedOutput.Builder content(String content) {
      this.content = Objects.requireNonNull(content, "content");
      initBits &= ~INIT_BIT_CONTENT;
      return (PreFormattedOutput.Builder) this;
    }

    /**
     * Builds a new {@link PreFormattedOutput PreFormattedOutput}.
     * @return An immutable instance of PreFormattedOutput
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public PreFormattedOutput build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutablePreFormattedOutput(content);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_CONTENT) != 0) attributes.add("content");
      return "Cannot build PreFormattedOutput, some of required attributes are not set " + attributes;
    }
  }
}
