% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scale_GammaDose.R
\name{scale_GammaDose}
\alias{scale_GammaDose}
\title{Calculate the gamma dose deposited within a sample taking layer-to-layer
variations in radioactivity into account (according to Aitken, 1985)}
\usage{
scale_GammaDose(
  data,
  conversion_factors = c("Cresswelletal2018", "Guerinetal2011", "AdamiecAitken1998",
    "Liritzisetal2013")[1],
  fractional_gamma_dose = c("Aitken1985")[1],
  verbose = TRUE,
  plot = TRUE,
  plot_single = TRUE,
  ...
)
}
\arguments{
\item{data}{\link{data.frame} (\strong{required}):
A table containing all relevant information for each individual layer. The
table must have the following named columns:
\itemize{
\item \code{id} (\link{character}): an arbitrary id or name of each layer
\item \code{thickness} (\link{numeric}): vertical extent of each layer in cm
\item \code{sample_offset} (\link{logical}): distance of the sample in cm,
\strong{measured from the BOTTOM OF THE TARGET LAYER}. Except for the target layer
all values must be \code{NA}.
\item \code{K} (\link{numeric}): K nuclide content in \%
\item \code{K_se} (\link{numeric}): error on the K content
\item \code{Th} (\link{numeric}): Th nuclide content in ppm
\item \code{Th_se} (\link{numeric}): error on the Th content
\item \code{U} (\link{numeric}): U nuclide content in ppm
\item \code{U_se} (\link{numeric}): error on the U content
\item \code{water_content} (\link{numeric}): water content of each layer in \%
\item \code{water_content_se} (\link{numeric}):  error on the water content
\item \code{density} (\link{numeric}): bulk density of each layer in g/cm^-3
}}

\item{conversion_factors}{\link{character} (\emph{optional}):
The conversion factors used to calculate the dose rate from sediment
nuclide contents. Valid options are:
\itemize{
\item \code{"Cresswelletal2018"} (default)
\item \code{"Liritzisetal2013"}
\item \code{"Guerinetal2011"}
\item \code{"AdamiecAitken1998"}
}}

\item{fractional_gamma_dose}{\link{character} (\emph{optional}):
Factors to scale gamma dose rate values. Valid options are:
\itemize{
\item \code{"Aitken1985"} (default): Table H1 in the appendix
}}

\item{verbose}{\link{logical} (\emph{optional}):
Show or hide console output (defaults to \code{TRUE}).}

\item{plot}{\link{logical} (\emph{optional}):
Show or hide the plot (defaults to \code{TRUE}).}

\item{plot_single}{\link{logical} (\emph{optional}):
Show all plots in one panel (defaults to \code{TRUE}).}

\item{...}{Further parameters passed to \link{barplot}.}
}
\value{
After performing the calculations the user is provided with different outputs.
\enumerate{
\item The total gamma dose rate received by the sample (+/- uncertainties) as a
print in the console.
\item A plot showing the sediment sequence, the user input sample information
and the contribution to total gamma dose rate.
\item RLum Results. If the user wishes to save these results, writing a script
to run the function and to save the results would look like this:
}

\if{html}{\out{<div class="sourceCode">}}\preformatted{mydata <- read.table("c:/path/to/input/file.txt")
results <- scale_GammaDose(mydata)
table <- get_RLum(results)
write.csv(table, "c:/path/to/results.csv")
}\if{html}{\out{</div>}}

-----------------------------------\cr
\verb{[ NUMERICAL OUTPUT ]}\cr
-----------------------------------\cr

\strong{\code{RLum.Results}}-object

\strong{slot:} \strong{\verb{@data}}

\tabular{lll}{
\strong{Element} \tab \strong{Type} \tab \strong{Description}\cr
\verb{$summary} \tab \code{data.frame} \tab summary of the model results \cr
\verb{$data} \tab \code{data.frame} \tab the original input data \cr
\verb{$dose_rates} \tab \code{list} \tab two \code{data.frames} for the scaled and infinite matrix dose rates \cr
\verb{$tables} \tab \code{list} \tab several \code{data.frames} containing intermediate results \cr
\verb{$args} \tab \code{character} \tab arguments of the call \cr
\verb{$call} \tab \code{call} \tab the original function call \cr
}

\strong{slot:} \strong{\verb{@info}}

Currently unused.

------------------------\cr
\verb{[ PLOT OUTPUT ]}\cr
------------------------\cr

Three plots are produced:
\itemize{
\item A visualisation of the provided sediment layer structure to quickly
assess whether the data was provided and interpreted correctly.
\item A scatter plot of the nuclide contents per layer (K, Th, U) as well as the
water content. This may help to correlate the dose rate contribution of
specific layers to the layer of interest.
\item A barplot visualising the contribution of each layer to the total dose rate
received by the sample in the target layer.
}
}
\description{
This function calculates the gamma dose deposited in a luminescence sample
taking into account layer-to-layer variations in sediment radioactivity .
The function scales user inputs of uranium, thorium and potassium based on
input parameters for sediment density, water content and given layer
thicknesses and distances to the sample.
}
\details{
\strong{User Input}

To calculate the gamma dose which is deposited in a sample, the user needs
to provide information on those samples influencing the luminescence sample.
As a rule of thumb, all sediment layers within at least 30 cm radius from
the luminescence sample taken should be taken into account when calculating
the gamma dose rate. However, the actual range of gamma radiation might be
different, depending on the emitting radioelement, the water content and the
sediment density of each layer (Aitken, 1985). Therefore the user is
advised to provide as much detail as possible and physically sensible.

The function requires a \link{data.frame} that is to be structured
in columns and rows, with samples listed in rows. The first column contains
information on the layer/sample ID, the second on the thickness (in cm) of
each layer, whilst column 3 should contain \code{NA} for all layers that are not
sampled for OSL/TL. For the layer the OSL/TL sample was taken from a numerical
value must be provided, which is the distance (in cm) measured from \strong{bottom}
of the layer of interest. If the whole layer was sampled insert \code{0}. If the
sample was taken from \emph{within} the layer, insert a numerical value \verb{>0},
which describes the distance from the middle of the sample to the bottom of
the layer in cm. Columns 4 to 9 should  contain radionuclide concentrations
and their standard errors for
potassium (in \%), thorium (in ppm) and uranium (in ppm). Columns 10 and 11
give information on the water content and its uncertainty (standard error)
in \%. The layer density (in g/cm3) should be given in column 12. No cell
should be left blank. Please ensure to keep the column titles as given in
the example dataset (\code{data('ExampleData.ScaleGammaDose')}, see examples).

The user can decide which dose rate
conversion factors should be used to calculate the gamma dose rates.
The options are:
\itemize{
\item \code{"Cresswelletal2018"} (Cresswell et al., 2018)
\item \code{"Liritzisetal2013"} (Liritzis et al., 2013)
\item \code{"Guerinetal2011"} (Guerin et al., 2011)
\item \code{"AdamiecAitken1998"} (Adamiec and Aitken, 1998)
}

\strong{Water content}

The water content provided by the user should be calculated according to:

\deqn{ ( Wet weight [g] - Dry weight [g] ) / Dry weight [g] * 100 }

\strong{Calculations}

After converting the radionuclide concentrations into dose rates, the
function will scale the dose rates based on the thickness of the layers,
the distances to the sample, the water content and the density of the sediment.
The calculations are based on Aitken (1985, Appendix H). As an example
(equivalent to Aitken, 1985), assuming three layers of sediment, where \strong{L} is
inert and positioned in between the infinite thick and equally active
layers \strong{A} and \strong{B}, the dose in \strong{L} and \strong{B} due to \strong{A} is given by

\deqn{ {1-f(x)}D_A }

Where \code{x} is the distance into the inert medium, so \code{f(x)} is the weighted
average fractional dose at \code{x} and \code{D_A} denotes that the dose is delivered by \strong{A}.
\code{f(x)} is derived from table H1 (Aitken, 1985), when setting \code{z = x}.
Consequently, the dose in \strong{A} and \strong{L} due to \strong{B} is given by

\deqn{ {1 - f(t-x)}D_B }

Here \code{t} is the thickness of \strong{L} and the other parameters are denoted as above,
just for the dose being delivered by B. \code{f(t-x)} is derived from table H1
(Aitken, 1985), when setting \code{z} equal to \code{t-x}. Following this, the dose in \strong{L}
delivered by \strong{A} and \strong{B} is given by

\deqn{ {2 - f(x) - f(t-x)}D_{AB} }

Since \strong{A} and \strong{B} are equally active \verb{D_\{AB\} = D_A = D_B}.

The function uses the value of the fractional dose rate at the layer
boundary to start the calculation for the next layer. This way, the function
is able to scale the gamma dose rate accurately for distant layers when the
density and water content is not constant for the entire section.
}
\note{
\strong{This function has BETA status. If possible, results should be}
\strong{cross-checked.}
}
\section{Function version}{
 0.1.2
}

\section{Acknowledgements}{


We thank Dr Ian Bailiff for the provision of an excel spreadsheet, which has
been very helpful when writing this function.
}

\examples{

# Load example data
data("ExampleData.ScaleGammaDose", envir = environment())
x <- ExampleData.ScaleGammaDose

# Scale gamma dose rate
results <- scale_GammaDose(data = x,
                           conversion_factors = "Cresswelletal2018",
                           fractional_gamma_dose = "Aitken1985",
                           verbose = TRUE,
                           plot = TRUE)

get_RLum(results)

} 

\section{How to cite}{
Riedesel, S., Autzen, M., Burow, C., 2023. scale_GammaDose(): Calculate the gamma dose deposited within a sample taking layer-to-layer variations in radioactivity into account (according to Aitken, 1985). Function version 0.1.2. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., Mercier, N., Philippe, A., Riedesel, S., Autzen, M., Mittelstrass, D., Gray, H.J., Galharret, J., 2023. Luminescence: Comprehensive Luminescence Dating Data Analysis. R package version 0.9.23. https://CRAN.R-project.org/package=Luminescence
}

\references{
Aitken, M.J., 1985. Thermoluminescence Dating. Academic Press, London.

Adamiec, G., Aitken, M.J., 1998. Dose-rate conversion factors: update.
Ancient TL 16, 37-46.

Cresswell., A.J., Carter, J., Sanderson, D.C.W., 2018.
Dose rate conversion parameters: Assessment of nuclear data.
Radiation Measurements 120, 195-201.

Guerin, G., Mercier, N., Adamiec, G., 2011. Dose-rate conversion
factors: update. Ancient TL, 29, 5-8.

Liritzis, I., Stamoulis, K., Papachristodoulou, C., Ioannides, K., 2013.
A re-evaluation of radiation dose-rate conversion factors. Mediterranean
Archaeology and Archaeometry 13, 1-15.
}
\seealso{
\link{ExampleData.ScaleGammaDose},
\link{BaseDataSet.ConversionFactors}, \link{approx}, \link{barplot}
}
\author{
Svenja Riedesel, Aberystwyth University (United Kingdom) \cr
Martin Autzen, DTU NUTECH Center for Nuclear Technologies (Denmark) \cr
Christoph Burow, University of Cologne (Germany) \cr
Based on an excel spreadsheet and accompanying macro written by Ian Bailiff.
, RLum Developer Team}
\keyword{datagen}
