if (typeof T === 'undefined') require('../setup');

T('minus', function () {

  var t = function (minuend, subtrahend, expected) {
    T.assertEqual(expected, new Decimal(minuend).minus(subtrahend).valueOf());
  }

  Decimal.config({
    precision: 200,
    rounding: 4,
    toExpNeg: -9e9,
    toExpPos: 9e9,
    minE: -9e15,
    maxE: 9e15
  });

  t(1, 0, '1');
  t(1, -0, '1');
  t(-1, 0, '-1');
  t(-1, -0, '-1');
  t(1, 'NaN', 'NaN');
  t(-1, 'NaN', 'NaN');
  t(1, 'Infinity', '-Infinity');
  t(1, '-Infinity', 'Infinity');
  t(-1, 'Infinity', '-Infinity');
  t(-1, '-Infinity', 'Infinity');
  t(0, 1, '-1');
  t(0, -1, '1');
  t(-0, 1, '-1');
  t(-0, -1, '1');

  /*
    IEEE 754 - 2008 section 6.3
    When the difference of two operands with like signs is exactly zero, the
    sign of that difference shall be +0 in all rounding-direction attributes
    except roundTowardNegative; under that attribute, the sign of an exact
    difference shall be −0.
    However, x + x = x −(−x) retains the same sign as x even when x is zero.
  */
  Decimal.rounding = 3;

  t('0', '-0', '0');    //   0 - -0 =  0
  t('-0', '0', '-0');   //  -0 -  0 = -0
  t('0', '0', '-0');    //   0 -  0 = -0
  t('-0', '-0', '-0');  //  -0 - -0 = -0
  t('1', '1', '-0');    //   1 -  1 = -0
  t('-1', '-1', '-0');  //  -1 - -1 = -0

  Decimal.rounding = 4;

  t('0', '-0', '0');    //   0 - -0 =  0
  t('-0', '0', '-0');   //  -0 -  0 = -0
  t('0', '0', '0');     //   0 -  0 =  0
  t('-0', '-0', '0');   //  -0 - -0 =  0
  t('1', '1', '0');     //   1 -  1 =  0
  t('-1', '-1', '0');   //  -1 - -1 =  0

  t(0, NaN, 'NaN');
  t(-0, NaN, 'NaN');
  t(0, Infinity, '-Infinity');
  t(0, -Infinity, 'Infinity');
  t(-0, Infinity, '-Infinity');
  t(-0, -Infinity, 'Infinity');
  t('NaN', 1, 'NaN');
  t('NaN', -1, 'NaN');
  t('NaN', 0, 'NaN');
  t('NaN', -0, 'NaN');
  t('NaN', 'NaN', 'NaN');
  t('NaN', Infinity, 'NaN');
  t('NaN', -Infinity, 'NaN');
  t(Infinity, 1, 'Infinity');
  t(Infinity, -1, 'Infinity');
  t(-Infinity, 1, '-Infinity');
  t(-Infinity, -1, '-Infinity');
  t(Infinity, 0, 'Infinity');
  t(Infinity, -0, 'Infinity');
  t(-Infinity, 0, '-Infinity');
  t(-Infinity, -0, '-Infinity');
  t(Infinity, 'NaN', 'NaN');
  t(-Infinity, 'NaN', 'NaN');
  t(Infinity, Infinity, 'NaN');
  t(Infinity, -Infinity, 'Infinity');
  t(-Infinity, Infinity, '-Infinity');
  t(-Infinity, -Infinity, 'NaN');

  t(1, '0', '1');
  t(1, '1', '0');
  t(1, '-45', '46');
  t(1, '22', '-21');
  t(1, 0144, '-99');
  t(1, '0144', '-143');
  t(1, '6.1915', '-5.1915');
  t(1, '-1.02', '2.02');
  t(1, '0.09', '0.91');
  t(1, '-0.0001', '1.0001');
  t(1, '8e5', '-799999');
  t(1, '9E12', '-8999999999999');
  t(1, '1e-14', '0.99999999999999');
  t(1, '3.345E-9', '0.999999996655');
  t(1, '-345.43e+4', '3454301');
  t(1, '-94.12E+0', '95.12');
  t('0', 0, '0');
  t('0', '0', '0');
  t(3, -0, '3');
  t(9.654, 0, '9.654');
  t(0, '0.001', '-0.001');
  t(0, '111.1111111110000', '-111.111111111');
  t(-1, 1, '-2');
  t(-0.01, 0.01, '-0.02');
  t(54, -54, '108');
  t(9.99, '-9.99', '19.98');
  t('0.0000023432495704937', '-0.0000023432495704937', '0.0000046864991409874');
  t(NaN, NaN, 'NaN');
  t(NaN, 'NaN', 'NaN');
  t('NaN', NaN, 'NaN');
  t('NaN', 4, 'NaN');
  t('NaN', '4534534.45435435', 'NaN');
  t('NaN', 99999.999, 'NaN');
  t('Infinity', '354.345341', 'Infinity');
  t(3, '-Infinity', 'Infinity');
  t('Infinity', 'Infinity', 'NaN');
  t('-Infinity', '-Infinity', 'NaN');
  t('Infinity', '-999e999', 'Infinity');
  t('1.21123e43', '-Infinity', 'Infinity');
  t('-999.0', 'Infinity', '-Infinity');
  t('657.342e-45', '-Infinity', 'Infinity');
  t('Infinity', 123, 'Infinity');
  t(-0, 'Infinity', '-Infinity');
  t(100, 100, '0');
  t(-999.99, '0.01', '-1000');
  t('10', 4, '6');
  t('03.333', -4, '7.333');
  t(-1, -0.1, '-0.9');
  t(43534.5435, '0.054645', '43534.488855');
  t('99999', '1', '99998');

  t = function (minuend, subtrahend, expected, sd, rm) {
    Decimal.precision = sd;
    Decimal.rounding = rm;
    T.assertEqual(expected, new Decimal(minuend).minus(subtrahend).valueOf());
  }

  t('0.000001', '1E-13', '0.0000009999999', 12, 5);
  t('1E-29', '0.1', '-0.09999999999999999999999999999', 30, 4);
  t('1E-9', '1E-40', '0.0000000009999999999999999999999999999999', 72, 6);
  t('1E-12', '1E-7', '-0.000000099999', 42, 3);
  t('1E-14', '1E-13', '-0.00000000000009', 76, 0);
  t('1E-21', '1E-32', '0.00000000000000000000099999999999', 77, 4);
  t('1E-19', '1E-16', '-0.0000000000000000999', 43, 3);
  t('1E-17', '1E-39', '0.000000000000000009999999999999999999999', 68, 2);
  t('1E-17', '1E-16', '-0.00000000000000009', 67, 1);
  t('1E-37', '1E-13', '-0.0000000000000999999999999999999999999', 52, 0);
  t('1E-27', '0.1', '-0.1', 25, 5);
  t('1E-28', '0.0001', '-0.0001', 6, 4);
  t('1E-20', '1E-34', '0.0000000000000000000099999999999999', 26, 0);
  t('1E-20', '0.1', '-0.1', 1, 6);
  t('1E-21', '1E-35', '0.00000000000000000000099999999999999', 57, 3);
  t('1E-30', '0.000001', '-0.000000999999999999999999999999', 81, 0);
  t('0.001', '1E-37', '0.00099999999999', 11, 1);
  t('1E-37', '1E-22', '-0.0000000000000000000000999999999999999', 73, 6);
  t('1E-25', '1E-34', '0.0000000000000000000000000999999999', 77, 5);
  t('1E-35', '1E-15', '-0.00000000000000099999999999999999999', 72, 1);
  t('1E-21', '1E-18', '-0.000000000000000000999', 16, 3);
  t('1E-31', '0.01', '-0.0099999999999999999999999999999', 36, 1);
  t('1E-13', '1E-10', '-0.0000000000999', 47, 6);
  t('1E-18', '0.001', '-0.000999999999999999', 91, 5);
  t('1E-38', '1E-11', '-0.00000000000999999999999999999999999999', 35, 5);
  t('0.001', '1E-23', '0.001', 17, 2);
  t('1E-38', '1E-7', '-0.00000009999999999999999999999999999999', 80, 5);
  t('0.000001', '1E-32', '0.000001', 21, 2);
  t('1E-20', '1E-30', '0.000000000000000000009999999999', 99, 1);
  t('1E-23', '1E-17', '-0.00000000000000000999999', 7, 0);
  t('1E-27', '1E-13', '-0.000000000000099999999999999', 94, 4);
  t('0.000001', '1E-39', '0.000000999999999999999999999999999999999', 77, 0);
  t('1E-32', '1E-23', '-0.00000000000000000000000999999999', 26, 0);
  t('1E-37', '1E-19', '-0.0000000000000000001', 17, 4);
  t('1E-20', '1E-34', '0.0000000000000000000099999999999999', 41, 1);
  t('1E-16', '1E-30', '0.000000000000000099999999999999', 37, 4);
  t('1E-9', '1E-14', '0.00000000099999', 54, 2);
  t('1E-33', '1E-12', '-0.000000000000999999999999999999999', 39, 6);
  t('1E-26', '1E-40', '0.0000000000000000000000000099999999999999', 36, 3);
  t('0.0001', '0.1', '-0.0999', 27, 5);
  t('1E-39', '1E-26', '-0.000000000000000000000000009999999999999', 28, 5);
  t('1E-37', '1E-14', '-0.0000000000000099999999999999999999999', 99, 1);
  t('1E-21', '1E-14', '-0.000000000000009999999', 79, 6);
  t('1E-30', '1E-33', '0.000000000000000000000000000000999', 5, 0);
  t('1E-24', '1E-16', '-0.000000000000000099999999', 8, 4);
  t('0.1', '1E-29', '0.09999999999999999999999999999', 53, 1);
  t('1E-21', '1E-16', '-0.000000000000000099999', 69, 5);
  t('1E-13', '1E-32', '0.00000000000009999999999999999999', 80, 3);
  t('1E-27', '1E-27', '0', 14, 1);
  t('1E-36', '1E-17', '-0.00000000000000001', 10, 5);
  t('1E-20', '1E-39', '0.000000000000000000009999999999999999999', 39, 6);
  t('1E-32', '1E-13', '-0.00000000000009999999999999999999', 69, 0);
  t('1E-35', '1E-22', '-0.00000000000000000000009999999999999', 40, 0);
  t('1E-17', '1E-37', '0.00000000000000001', 8, 2);
  t('1E-19', '1E-9', '-0.0000000009999999999', 60, 5);
  t('1E-33', '1E-28', '-0.000000000000000000000000000099999', 6, 1);
  t('0.000001', '1E-27', '0.000001', 13, 2);
  t('1E-34', '1E-12', '-0.0000000000009999999999999999999999', 66, 2);
  t('1E-36', '1E-27', '-0.000000000000000000000000000999999999', 22, 3);
  t('1E-40', '1E-20', '-0.0000000000000000000099999999999999999999', 91, 5);
  t('1E-26', '1E-17', '-0.00000000000000001', 7, 0);
  t('1E-25', '1E-34', '0.0000000000000000000000000999999999', 76, 5);
  t('1E-37', '1E-29', '-0.0000000000000000000000000000099999999', 29, 0);
  t('1E-16', '1E-9', '-0.0000000009999999', 74, 1);
  t('1E-20', '1E-26', '0.00000000000000000000999999', 88, 1);
  t('1E-35', '1E-22', '-0.00000000000000000000009999999999999', 84, 5);
  t('1E-14', '1E-32', '0.00000000000000999999999999999999', 68, 1);
  t('1E-9', '1E-38', '0.0000000009999999999999999999999999999', 28, 3);
  t('1E-15', '1E-18', '0.000000000000000999', 46, 1);
  t('1E-8', '1E-20', '0.00000001', 1, 5);
  t('1E-33', '1E-33', '0', 98, 2);
  t('1E-30', '1E-38', '0.00000000000000000000000000000099999999', 22, 3);
  t('1E-19', '0.1', '-0.0999999999999999999', 53, 6);
  t('0.000001', '1E-14', '0.00000099999999', 53, 2);
  t('0.000001', '1E-25', '0.0000009999999999999999999', 96, 2);
  t('1E-13', '1E-16', '0.0000000000000999', 89, 2);
  t('1E-25', '1E-40', '0.0000000000000000000000000999999999999999', 74, 5);
  t('1E-33', '1E-18', '-0.000000000000000000999999999999999', 86, 3);
  t('1E-13', '1E-21', '0.000000000000099999999', 44, 5);
  t('1E-29', '0.01', '-0.01', 17, 6);
  t('0.1', '1E-29', '0.09999999999999999999999999999', 30, 5);
  t('1E-9', '1E-18', '0.000000000999999999', 10, 0);
  t('1E-34', '1E-31', '-0.0000000000000000000000000000000999', 90, 1);
  t('1E-35', '0.1', '-0.099999', 5, 2);
  t('1E-32', '1E-7', '-0.00000009999999999999999999999999', 27, 3);
  t('1E-25', '0.00001', '-0.0000099999999999999999999', 45, 5);
  t('1E-30', '1E-22', '-0.000000000000000000000099999999', 29, 6);
  t('1E-21', '1E-23', '0.00000000000000000000099', 4, 4);
  t('1E-23', '1E-36', '0.000000000000000000000009', 1, 3);
  t('1E-31', '1E-37', '0.0000000000000000000000000000000999999', 73, 0);
  t('1E-9', '1E-36', '0.000000000999999999999999999999999999', 90, 4);
  t('1E-39', '1E-31', '-0.000000000000000000000000000000099999999', 45, 6);
  t('0.0001', '0.000001', '0.000099', 7, 0);
  t('1E-7', '1E-36', '0.000000099999999999999999999999999999', 60, 0);
  t('1E-27', '1E-28', '0.0000000000000000000000000009', 95, 5);
  t('0.01', '0.0001', '0.0099', 27, 1);
  t('1E-9', '1E-30', '0.000000001', 6, 2);
  t('0.1', '1E-23', '0.1', 17, 6);
  t('1E-21', '1E-18', '-0.000000000000000000999', 46, 0);
  t('1E-19', '1E-23', '0.00000000000000000009999', 27, 4);
  t('1E-24', '1E-19', '-0.000000000000000000099999', 41, 0);
  t('1E-26', '1E-20', '-0.00000000000000000000999999', 53, 0);
  t('1E-19', '0.1', '-0.0999999999999999999', 80, 5);
  t('1E-14', '1E-24', '0.000000000000009999999999', 12, 3);
  t('1E-19', '1E-17', '-0.0000000000000000099', 59, 5);
  t('1E-27', '1E-22', '-0.000000000000000000000099999', 72, 4);
  t('1E-28', '1E-8', '-0.0000000099999999999999999999', 40, 2);
  t('1E-32', '1E-34', '0.0000000000000000000000000000000099', 60, 3);
  t('1E-23', '0.1', '-0.09999999999999999999999', 32, 5);
  t('0.01', '1E-26', '0.00999999999999999999999999', 100, 4);
  t('1E-22', '0.01', '-0.0099999999999999999999', 62, 1);
  t('1E-15', '1E-7', '-0.000000099999999', 57, 3);
  t('1E-11', '1E-7', '-0.00000009999', 51, 4);
  t('0.001', '1E-17', '0.00099999999999999', 19, 0);
  t('1E-12', '0.001', '-0.00099999', 5, 2);
  t('1E-24', '1E-14', '-0.000000000000009999999999', 22, 2);
  t('0.1', '1E-27', '0.099999999999999999999999999', 27, 3);
  t('1E-18', '1E-33', '0.000000000000000000999999999999999', 98, 2);
  t('0.1', '1E-17', '0.09999999999999999', 48, 6);
  t('1E-30', '1E-33', '0.000000000000000000000000000000999', 34, 2);
  t('1E-31', '1E-10', '-0.0000000000999999999999999999999', 37, 2);
  t('1E-28', '1E-12', '-0.000000000001', 14, 3);
  t('1E-23', '1E-19', '-0.00000000000000000009999', 41, 4);
  t('1E-8', '1E-9', '0.000000009', 31, 2);
  t('1E-17', '1E-8', '-0.00000000999999999', 67, 4);
  t('1E-31', '1E-17', '-0.0000000000000000099999999999999', 67, 6);
  t('0.000001', '1E-24', '0.000000999999999999999999', 93, 0);
  t('1E-17', '1E-33', '0.000000000000000009999999999999999', 57, 2);
  t('1E-28', '1E-10', '-0.0000000000999999999999999999', 71, 4);
  t('1E-17', '1E-26', '0.00000000000000000999999999', 82, 0);
  t('0.1', '0.00001', '0.09999', 67, 2);
  t('1E-21', '1E-25', '0.0000000000000000000009999', 93, 3);
  t('1E-19', '0.1', '-0.0999999999999999999', 64, 1);
  t('1E-10', '1E-21', '0.000000000099999999999', 49, 0);
  t('1E-37', '1E-16', '-0.0000000000000000999999999999999999999', 74, 0);
  t('1E-31', '1E-33', '0.000000000000000000000000000000099', 2, 4);
  t('1E-21', '1E-28', '0.0000000000000000000009999999', 54, 5);
  t('1E-26', '1E-28', '0.0000000000000000000000000099', 5, 3);
  t('1E-28', '1E-32', '0.00000000000000000000000000009999', 72, 5);
  t('1E-9', '1E-26', '0.00000000099999999999999999', 34, 4);
  t('1E-40', '1E-12', '-0.0000000000009999999999999999999999999999', 98, 1);
  t('1E-30', '0.0001', '-0.000099999999999999999999999999', 59, 5);
  t('1E-40', '1E-30', '-0.0000000000000000000000000000009999999999', 39, 0);
  t('1E-33', '1E-17', '-0.000000000000000009999999999999999', 30, 0);
  t('1E-37', '1E-16', '-0.0000000000000000999999999999999999999', 82, 1);
  t('0.1', '1E-11', '0.09999999999', 88, 1);
  t('1E-14', '0.1', '-0.09999999999999', 62, 6);
  t('0.001', '1E-14', '0.00099999999999', 56, 1);
  t('1E-20', '1E-9', '-0.00000000099999999999', 34, 0);
  t('1E-39', '1E-34', '-0.000000000000000000000000000000000099999', 73, 4);
  t('1E-14', '1E-31', '0.0000000000000099999999999999999', 42, 1);
  t('1E-31', '1E-12', '-0.0000000000009999999999999999999', 34, 6);
  t('0.01', '1E-30', '0.009999999999999999999999999999', 60, 6);
  t('1E-30', '1E-14', '-0.000000000000009999999999999999', 70, 2);
  t('0.01', '1E-17', '0.00999999999999999', 31, 5);
  t('1E-22', '0.001', '-0.0009999999999999999999', 46, 4);
  t('1E-35', '1E-31', '-0.00000000000000000000000000000009999', 10, 6);
  t('0.1', '1E-28', '0.1', 23, 5);
  t('0.00001', '1E-40', '0.00001', 2, 4);
  t('0.00001', '1E-35', '0.0000099999999999999999999999', 23, 1);
  t('1E-25', '1E-12', '-0.0000000000009999999999999', 85, 0);
  t('0.00001', '1E-29', '0.00000999999999999999999999999', 90, 0);
  t('0.0001', '1E-22', '0.0000999999999999999999', 20, 3);
  t('1E-19', '1E-25', '0.0000000000000000000999999', 87, 4);
  t('1E-22', '1E-23', '0.00000000000000000000009', 22, 6);
  t('1E-21', '1E-23', '0.00000000000000000000099', 23, 6);
  t('1E-12', '1E-7', '-0.000000099999', 55, 0);
  t('0.000001', '1E-17', '0.00000099999999999', 19, 1);
  t('1E-17', '1E-19', '0.0000000000000000099', 90, 4);
  t('1E-16', '1E-37', '0.0000000000000000999999999999999999999', 50, 1);
  t('1E-9', '1E-31', '0.0000000009999999999999999999999', 37, 2);
  t('1E-35', '0.1', '-0.09999999999999999999999999999999999', 87, 2);
  t('1E-32', '1E-13', '-0.0000000000001', 15, 5);
  t('1E-10', '1E-18', '0.000000000099999999', 91, 3);
  t('0.1', '1E-25', '0.099999999999999999', 17, 3);
  t('1E-32', '1E-21', '-0.000000000000000000001', 10, 0);
  t('1E-23', '1E-10', '-0.00000000009999999999999', 77, 0);
  t('1E-20', '1E-18', '-0.00000000000000000099', 71, 0);
  t('1E-17', '0.01', '-0.00999999999999999', 76, 6);
  t('1E-18', '1E-15', '-0.000000000000000999', 70, 1);
  t('1E-34', '1E-31', '-0.0000000000000000000000000000000999', 21, 0);
  t('0.1', '1E-28', '0.0999999999999999999999999999', 72, 2);
  t('1E-27', '1E-30', '0.000000000000000000000000000999', 76, 1);
  t('1E-14', '1E-11', '-0.00000000000999', 95, 2);
  t('1E-19', '1E-18', '-0.0000000000000000009', 43, 2);
  t('1E-26', '1E-28', '0.0000000000000000000000000099', 87, 5);
  t('1E-39', '1E-36', '-0.000000000000000000000000000000000000999', 97, 5);
  t('0.000001', '1E-35', '0.00000099999999999999999999999999999', 40, 1);
  t('1E-20', '1E-27', '0.000000000000000000009999999', 32, 1);
  t('1E-17', '0.1', '-0.09999999999999999', 30, 0);
  t('1E-37', '0.000001', '-0.0000009999999999999999999999999999999', 33, 6);
  t('1E-8', '1E-36', '0.000000009999999999999999999999999999', 46, 4);
  t('0.1', '1E-25', '0.0999999999999999999999999', 75, 1);
  t('1E-21', '1E-13', '-0.000000000000099999999', 45, 6);
  t('1E-28', '1E-13', '-0.0000000000000999999999999999', 63, 6);
  t('0.0001', '1E-23', '0.00009999999999999999999', 39, 2);
  t('1E-33', '1E-19', '-0.000000000000000000099999999999999', 15, 1);
  t('1E-38', '1E-24', '-0.00000000000000000000000099999999999999', 20, 2);
  t('1E-37', '1E-27', '-0.0000000000000000000000000009999999999', 24, 4);
  t('1E-29', '1E-18', '-0.00000000000000000099999999999', 92, 3);
  t('1E-7', '1E-34', '0.0000000999999999999999999999999999', 96, 1);
  t('0.00001', '1E-25', '0.0000099999999999999999999', 93, 0);
  t('1E-30', '1E-36', '0.000000000000000000000000000000999999', 37, 6);
  t('1E-24', '1E-37', '0.0000000000000000000000009999999999999', 85, 5);
  t('1E-21', '0.001', '-0.001', 17, 6);
  t('1E-31', '1E-23', '-0.0000000000000000000000099999999', 75, 2);
  t('0.000001', '1E-40', '0.0000009', 1, 3);
  t('1E-25', '1E-17', '-0.0000000000000000099999999', 37, 2);
  t('0.1', '1E-31', '0.0999999999999999999999999999999', 50, 3);
  t('1E-33', '1E-23', '-0.000000000000000000000009999999999', 95, 5);
  t('1E-32', '1E-35', '0.00000000000000000000000000000000999', 4, 1);
  t('1E-8', '1E-9', '0.000000009', 64, 5);
  t('1E-11', '1E-17', '0.00000000000999999', 67, 5);
  t('0.00001', '1E-12', '0.000009999999', 80, 4);
  t('0.001', '1E-12', '0.000999999999', 27, 5);
  t('1E-40', '1E-39', '-0.0000000000000000000000000000000000000009', 4, 2);
  t('1E-13', '1E-15', '0.000000000000099', 51, 2);
  t('0.1', '0.01', '0.09', 21, 0);
  t('1E-35', '1E-39', '0.000000000000000000000000000000000009999', 50, 5);
  t('0.1', '1E-37', '0.0999999999999999999999999999999999999', 93, 0);
  t('0.1', '1E-31', '0.0999999999999999999999999999999', 31, 1);
  t('1E-12', '1E-35', '0.00000000000099999999999999999999999', 43, 3);
  t('1E-10', '1E-22', '0.00000000009', 1, 3);
  t('1E-7', '1E-19', '0.0000000999999999999', 33, 2);
  t('1E-23', '1E-20', '-0.00000000000000000000999', 4, 6);
  t('1E-33', '1E-31', '-0.000000000000000000000000000000099', 65, 5);
  t('0.01', '1E-17', '0.00999999999999999', 50, 0);
  t('1E-21', '1E-12', '-0.00000000000099999', 5, 1);
  t('1E-15', '1E-40', '0.000000000000001', 5, 5);
  t('1E-19', '0.00001', '-0.0000099999999999999', 21, 3);
  t('0.1', '0.1', '0', 71, 5);
  t('1E-9', '1E-33', '0.000000000999999999999999999999999', 95, 3);
  t('1E-15', '1E-19', '0.0000000000000009999', 77, 5);
  t('1E-34', '0.01', '-0.0099999999999999999999999999999999', 61, 2);
  t('1E-17', '0.1', '-0.1', 6, 0);
  t('0.000001', '0.000001', '0', 72, 0);
  t('1E-15', '1E-40', '0.0000000000000009999999999999999999999999', 37, 0);
  t('1E-34', '1E-24', '-0.0000000000000000000000009999999999', 14, 3);
  t('1E-38', '0.1', '-0.09999999999999999999', 19, 1);
  t('1E-36', '0.0001', '-0.000099999999999999999999999999999999', 48, 0);
  t('0.01', '1E-39', '0.009999999999999999999999999999999999999', 94, 4);
  t('1E-24', '1E-13', '-0.000000000000099999999999', 48, 5);
  t('1E-31', '0.1', '-0.0999999999999999999999999999999', 50, 3);
  t('0.1', '1E-7', '0.0999999', 79, 5);
  t('0.01', '1E-36', '0.01', 26, 4);
  t('1E-22', '1E-8', '-0.00000001', 3, 6);
  t('1E-25', '0.1', '-0.0999999999999999999999999', 83, 1);
  t('1E-38', '0.000001', '-0.000001', 20, 4);
  t('0.001', '1E-21', '0.000999999999999999999', 100, 1);
  t('1E-26', '1E-35', '0.00000000000000000000000000999999999', 40, 0);
  t('1E-12', '1E-25', '0.0000000000009999999999999', 71, 2);
  t('1E-11', '1E-33', '0.000000000009999999999999999999999', 78, 0);
  t('1E-13', '1E-25', '0.0000000000000999999999999', 17, 0);
  t('1E-25', '0.1', '-0.0999999999999999999999999', 33, 2);
  t('1E-20', '1E-20', '0', 15, 4);
  t('1E-9', '1E-23', '0.000000001', 12, 6);
  t('1E-29', '1E-40', '0.0000000000000000000000000000099999999999', 29, 2);
  t('1E-32', '1E-36', '0.000000000000000000000000000000009999', 73, 2);
  t('1E-29', '0.00001', '-0.00000999999999999999999999999', 69, 3);
  t('0.00001', '1E-33', '0.00001', 16, 4);
  t('1E-10', '0.1', '-0.0999999999', 58, 1);
  t('1E-18', '1E-39', '0.000000000000000000999999999999999999999', 84, 1);
  t('1E-24', '0.00001', '-0.000009999999999999999999', 21, 0);
  t('1E-23', '1E-39', '0.000000000000000000000009999999999999999', 66, 2);
  t('1E-21', '1E-19', '-0.000000000000000000099', 71, 6);
  t('1E-24', '1E-27', '0.000000000000000000000000999', 81, 5);
  t('1E-40', '1E-9', '-0.000000001', 8, 6);
  t('1E-32', '1E-14', '-0.00000000000000999999999999999999', 24, 5);
  t('1E-11', '1E-11', '0', 70, 4);
  t('0.00001', '1E-29', '0.00000999999999999999999999999', 96, 0);
  t('1E-20', '1E-13', '-0.00000000000009999999', 43, 6);
  t('1E-34', '1E-16', '-0.0000000000000000999999999999999999', 69, 3);
  t('1E-29', '1E-33', '0.000000000000000000000000000009999', 83, 2);
  t('1E-26', '1E-27', '0.000000000000000000000000009', 39, 4);
  t('1E-36', '1E-23', '-0.000000000000000000000009999999999999', 64, 0);
  t('1E-35', '1E-28', '-0.00000000000000000000000000009999999', 49, 0);
  t('1E-11', '1E-29', '0.00000000000999999999999999999', 24, 5);
  t('1E-16', '1E-39', '0.00000000000000009999999999999999999', 19, 1);
  t('1E-36', '1E-21', '-0.000000000000000000000999999999999999', 27, 4);
  t('1E-8', '1E-34', '0.0000000099999999999999999999999999', 52, 0);
  t('1E-39', '1E-18', '-0.000000000000000000999999999999999999999', 79, 5);
  t('1E-23', '1E-36', '0.00000000000000000000001', 3, 6);
  t('1E-29', '1E-15', '-0.00000000000000099999999999999', 22, 3);
  t('0.01', '1E-17', '0.00999999999999999', 37, 2);
  t('1E-33', '0.0001', '-0.000099999999999999999999999999999', 43, 5);
  t('1E-33', '1E-32', '-0.000000000000000000000000000000009', 8, 2);
  t('1E-31', '1E-20', '-0.0000000000000000000099999999999', 35, 4);
  t('1E-12', '1E-19', '0.0000000000009999999', 79, 0);
  t('0.000001', '1E-37', '0.00000099999999999999999', 17, 3);
  t('1E-13', '1E-40', '0.0000000000000999999999999999999999999999', 82, 6);
  t('1E-24', '1E-36', '0.000000000000000000000000999999999999', 25, 1);
  t('1E-10', '1E-39', '0.000000000099999999999999999999999999999', 67, 1);
  t('1E-39', '1E-37', '-0.000000000000000000000000000000000000099', 28, 3);
  t('1E-24', '1E-30', '0.00000000000000000000000099', 2, 1);
  t('0.01', '1E-39', '0.009999999999999999999999999999999999999', 37, 3);
  t('0.001', '1E-23', '0.001', 15, 4);
  t('1E-23', '1E-37', '0.0000000000000000000000099999999999999', 55, 1);
  t('1E-10', '1E-21', '0.000000000099999999999', 60, 2);
  t('1E-15', '1E-25', '0.0000000000000009999999999', 82, 0);
  t('1E-35', '1E-21', '-0.00000000000000000000099999999999999', 90, 3);
  t('1E-8', '0.00001', '-0.00000999', 82, 6);
  t('1E-8', '1E-17', '0.00000000999999999', 39, 2);
  t('1E-38', '0.1', '-0.09999999999999999999999999999999999999', 58, 2);
  t('0.0001', '1E-28', '0.0001', 10, 6);
  t('0.1', '1E-39', '0.099999999999999999999999999999999999999', 90, 2);
  t('1E-18', '1E-34', '0.0000000000000000009999999999999999', 16, 3);
  t('1E-8', '1E-33', '0.000000009999999999999999999999999', 31, 0);
  t('1E-27', '1E-39', '0.000000000000000000000000000999999999999', 27, 1);
  t('1E-24', '1E-8', '-0.000000009999999999999999', 45, 5);
  t('0.00001', '1E-27', '0.000009999999999999999999999', 28, 3);
  t('1E-21', '0.01', '-0.009999999999999999999', 52, 2);
  t('1E-33', '1E-9', '-0.000000001', 13, 3);
  t('1E-14', '1E-9', '-0.00000000099999', 60, 3);
  t('1E-28', '1E-27', '-0.0000000000000000000000000009', 86, 1);
  t('1E-33', '1E-33', '-0', 21, 3);
  t('1E-13', '1E-26', '0.00000000000009999999999999', 66, 4);
  t('1E-16', '1E-20', '0.00000000000000009999', 10, 3);
  t('1E-8', '1E-40', '0.0000000099999999999999999999999999999999', 80, 2);
  t('1E-25', '1E-8', '-0.0000000099999999999999999', 53, 2);
  t('1E-22', '1E-21', '-0.0000000000000000000009', 79, 5);
  t('0.000001', '1E-40', '0.0000009999999999999999999999999999999999', 94, 3);
  t('1E-10', '1E-33', '0.000000000099999999999999999999999', 55, 4);
  t('1E-37', '1E-12', '-0.0000000000009999999999999999999999999', 52, 4);
  t('1E-39', '0.01', '-0.009999999999999999999999999999999999999', 55, 2);
  t('1E-16', '0.0001', '-0.0000999999999999', 91, 5);
  t('1E-13', '1E-34', '0.0000000000000999999999999999999999', 22, 2);
  t('0.0001', '0.1', '-0.0999', 16, 4);
  t('0.000001', '1E-16', '0.0000009999999999', 35, 1);
  t('1E-8', '1E-19', '0.0000000099999999999', 24, 6);
  t('1E-17', '1E-7', '-0.00000009999999999', 58, 4);
  t('0.1', '0.000001', '0.099999', 53, 0);
  t('1E-39', '0.1', '-0.099999999999999999999999999999999999999', 38, 6);
  t('1E-7', '1E-22', '0.0000001', 12, 2);
  t('1E-28', '1E-12', '-0.0000000000009999999999999999', 40, 0);
  t('1E-20', '1E-28', '0.00000000000000000001', 5, 4);
  t('0.1', '1E-31', '0.0999999999999999999999999999999', 75, 2);
  t('0.0001', '0.01', '-0.0099', 36, 2);
  t('1E-15', '1E-8', '-0.000000009999999', 77, 0);
  t('1E-30', '1E-13', '-0.000000000000099999999999999999', 45, 4);
  t('1E-26', '0.00001', '-0.00000999999999999999999999', 53, 5);
  t('1E-20', '1E-9', '-0.00000000099999999999', 37, 6);
  t('1E-34', '0.1', '-0.1', 18, 4);
  t('1E-8', '1E-17', '0.00000000999999999', 46, 3);
  t('1E-11', '1E-12', '0.000000000009', 15, 4);
  t('1E-25', '0.0001', '-0.0000999999999999999999999', 54, 1);
  t('1E-12', '0.00001', '-0.000009999999', 50, 1);
  t('0.001', '0.0001', '0.0009', 9, 5);
  t('0.1', '1E-23', '0.0999999999', 9, 1);
  t('1E-30', '1E-29', '-0.000000000000000000000000000009', 5, 0);
  t('1E-25', '1E-11', '-0.0000000000099999999999999', 73, 2);
  t('1E-29', '1E-7', '-0.00000009999999999999999999999', 25, 4);
  t('1E-35', '0.00001', '-0.00000999999999999999999999999999999', 73, 0);
  t('1E-25', '1E-29', '0.00000000000000000000000009999', 35, 2);
  t('0.01', '1E-10', '0.0099999999', 52, 1);
  t('1E-35', '1E-33', '-0.00000000000000000000000000000000099', 53, 5);
  t('1E-24', '0.1', '-0.099999999999999999999999', 29, 5);
  t('1E-34', '0.01', '-0.0099', 2, 2);
  t('0.1', '1E-12', '0.09999', 4, 1);
  t('1E-26', '1E-37', '0.0000000000000000000000000099999999999', 70, 1);
  t('1E-23', '1E-14', '-0.00000000000000999999999', 56, 1);
  t('0.00001', '1E-17', '0.00000999999999999', 42, 4);
  t('1E-40', '1E-38', '-0.0000000000000000000000000000000000000099', 30, 4);
  t('0.1', '1E-38', '0.09999999999999999999999999999999999999', 82, 3);
  t('1E-18', '0.1', '-0.099999999999999999', 29, 0);
  t('1E-13', '0.1', '-0.0999999999999', 20, 6);
  t('1E-21', '1E-27', '0.000000000000000000000999999', 53, 6);
  t('1E-29', '1E-23', '-0.00000000000000000000000999999', 41, 5);
  t('1E-26', '1E-20', '-0.00000000000000000000999999', 33, 2);
  t('1E-7', '1E-15', '0.0000001', 4, 6);
  t('0.01', '1E-20', '0.00999999999999999999', 88, 5);
  t('1E-24', '1E-26', '0.00000000000000000000000099', 67, 3);
  t('1E-20', '0.1', '-0.09999999999999999999', 63, 5);
  t('1E-24', '0.01', '-0.009999999999999999999999', 29, 2);
  t('0.001', '1E-37', '0.0009999999999999999999999999999999999', 92, 3);
  t('1E-21', '0.0001', '-0.000099999999999999999', 39, 1);
  t('0.000001', '0.00001', '-0.000009', 35, 6);
  t('1E-15', '1E-34', '0.0000000000000009999999999999999999', 26, 4);
  t('1E-14', '1E-13', '-0.00000000000009', 63, 5);
  t('1E-27', '1E-25', '-0.000000000000000000000000099', 98, 4);
  t('1E-19', '1E-32', '0.00000000000000000009999999999999', 61, 6);
  t('1E-22', '1E-15', '-0.0000000000000009999999', 53, 3);
  t('1E-40', '1E-30', '-0.0000000000000000000000000000009999999999', 80, 4);
  t('1E-10', '0.1', '-0.0999999999', 26, 1);
  t('1E-31', '1E-33', '0.000000000000000000000000000000099', 38, 5);
  t('1E-8', '1E-27', '0.000000009999999999999999999', 95, 4);
  t('1E-28', '1E-31', '0.0000000000000000000000000000999', 30, 6);
  t('1E-35', '1E-9', '-0.000000001', 21, 4);
  t('1E-17', '0.01', '-0.00999999999999999', 20, 1);
  t('1E-22', '1E-24', '0.000000000000000000000099', 51, 2);
  t('0.000001', '1E-10', '0.0000009999', 96, 2);
  t('1E-26', '1E-21', '-0.00000000000000000000099999', 85, 1);
  t('1E-27', '1E-31', '0.0000000000000000000000000009999', 52, 5);
  t('1E-12', '1E-19', '0.0000000000009999999', 78, 3);
  t('0.000001', '1E-17', '0.00000099999999999', 47, 0);
  t('0.01', '1E-11', '0.00999999999', 57, 5);
  t('1E-30', '1E-30', '0', 19, 2);
  t('0.000001', '1E-33', '0.000000999999999999999999999999999', 90, 1);
  t('1E-34', '1E-27', '-0.0000000000000000000000000009999999', 56, 2);
  t('0.000001', '1E-14', '0.00000099999999', 80, 0);
  t('1E-26', '0.0001', '-0.00009999999999999999999999', 41, 2);
  t('1E-16', '0.001', '-0.0009999999999999', 33, 3);
  t('1E-35', '1E-12', '-0.00000000000099999999999999999999999', 89, 4);
  t('1E-26', '1E-25', '-0.00000000000000000000000009', 85, 2);
  t('1E-29', '1E-17', '-0.00000000000000000999999999999', 15, 5);
  t('1E-27', '0.1', '-0.099999999999999999999999999', 84, 1);
  t('1E-27', '0.0001', '-0.000099999999999999999999999', 54, 5);
  t('1E-31', '1E-7', '-0.0000000999999999999999999999999', 75, 4);
  t('0.000001', '1E-22', '0.000001', 11, 5);
  t('1E-19', '1E-20', '0.00000000000000000009', 24, 3);
  t('0.00001', '1E-35', '0.00000999999999999999999999999999999', 95, 0);
  t('1E-37', '1E-28', '-0.0000000000000000000000000000999999999', 31, 2);
  t('1E-28', '1E-10', '-0.0000000000999999999999999999', 74, 3);
  t('1E-21', '1E-18', '-0.000000000000000000999', 88, 1);
  t('1E-40', '1E-30', '-0.0000000000000000000000000000009999999999', 23, 5);
  t('1E-10', '1E-14', '0.00000000009999', 99, 2);
  t('1E-9', '1E-22', '0.0000000009999999999999', 51, 5);
  t('1E-16', '1E-16', '0', 56, 1);
  t('1E-20', '0.1', '-0.09999999999999999999', 19, 3);
  t('1E-37', '0.000001', '-0.0000009999999999999999999999999999999', 33, 1);
  t('1E-16', '1E-16', '0', 96, 2);
  t('1E-27', '0.00001', '-0.000009999999999999999999999', 93, 2);
  t('1E-17', '1E-22', '0.0000000000000000099999', 38, 4);
  t('1E-26', '1E-10', '-0.00000000009999999999999999', 56, 0);
  t('1E-40', '1E-11', '-0.0000000000099999999999999999999999999999', 75, 2);
  t('1E-8', '1E-10', '0.0000000099', 42, 5);
  t('1E-36', '1E-16', '-0.000000000000000099999999999999999999', 90, 0);
  t('1E-20', '1E-22', '0.0000000000000000000099', 51, 1);
  t('1E-25', '1E-26', '0.00000000000000000000000009', 16, 2);
  t('1E-36', '1E-40', '0.0000000000000000000000000000000000009999', 17, 3);
  t('1E-17', '1E-31', '0.00000000000000001', 8, 0);
  t('1E-32', '1E-19', '-0.00000000000000000009999999999999', 51, 4);
  t('1E-30', '0.000001', '-0.000000999999999999999999999999', 92, 5);
  t('1E-17', '0.01', '-0.00999999999999999', 37, 3);
  t('1E-16', '1E-30', '0.000000000000000099999999999999', 20, 1);
  t('1E-10', '1E-32', '0.00000000009999999999999999999999', 84, 0);
  t('1E-12', '1E-17', '0.00000000000099999', 37, 6);
  t('1E-22', '1E-10', '-0.0000000000999999999999', 87, 1);
  t('1E-9', '1E-11', '0.00000000099', 4, 0);
  t('1E-16', '1E-26', '0.00000000000000009999999999', 13, 0);
  t('1E-37', '0.001', '-0.001', 32, 5);
  t('1E-16', '1E-24', '0.000000000000000099999999', 17, 2);
  t('1E-7', '1E-27', '0.000000099999999999999999999', 29, 3);
  t('1E-25', '1E-26', '0.00000000000000000000000009', 24, 4);
  t('1E-11', '1E-20', '0.00000000000999999999', 47, 1);
  t('0.1', '1E-24', '0.099999999999999999999999', 46, 1);
  t('1E-17', '1E-16', '-0.00000000000000009', 27, 5);
  t('1E-27', '1E-13', '-0.000000000000099999999999999', 42, 0);
  t('1E-25', '0.001', '-0.0009999999999999999999999', 42, 1);
  t('0.01', '1E-35', '0.00999999999999999999999999999999999', 86, 3);
  t('1E-35', '1E-24', '-0.000000000000000000000001', 5, 0);
  t('1E-15', '1E-14', '-0.000000000000009', 62, 6);
  t('1E-28', '1E-10', '-0.0000000000999999999999999999', 91, 3);
  t('1E-37', '1E-35', '-0.0000000000000000000000000000000000099', 67, 0);
  t('1E-7', '1E-35', '0.00000009999999999999999999999999999', 80, 2);
  t('1E-35', '1E-20', '-0.00000000000000000000999999999999999', 84, 4);
  t('1E-24', '1E-27', '0.000000000000000000000000999', 88, 2);
  t('1E-21', '1E-11', '-0.000000000009999999999', 33, 0);
  t('1E-15', '1E-37', '0.00000000000000099999999999', 11, 1);
  t('1E-14', '1E-34', '0.0000000000000099999999999999999999', 35, 0);
  t('1E-8', '1E-34', '0.00000001', 12, 5);
  t('1E-17', '1E-19', '0.0000000000000000099', 52, 2);
  t('1E-33', '1E-33', '0', 100, 4);
  t('0.0001', '0.00001', '0.00009', 55, 5);
  t('1E-8', '1E-32', '0.00000000999999999999999999999999', 83, 0);
  t('1E-7', '1E-33', '0.000000099999999999999999999999999', 26, 0);
  t('1E-35', '1E-10', '-0.00000000009999999999999999999999999', 87, 4);
  t('1E-35', '1E-28', '-0.00000000000000000000000000009999999', 58, 4);
  t('0.01', '1E-26', '0.00999999999999999999999999', 83, 3);
  t('1E-28', '1E-26', '-0.0000000000000000000000000099', 35, 2);
  t('1E-14', '1E-38', '0.00000000000000999999999999999999999999', 54, 3);
  t('1E-16', '1E-24', '0.000000000000000099999999', 11, 4);
  t('1E-25', '1E-21', '-0.0000000000000000000009999', 11, 5);
  t('1E-9', '1E-9', '0', 48, 4);
  t('1E-10', '1E-31', '0.0000000000999999999999999999999', 46, 5);
  t('1E-22', '0.001', '-0.0009999999999999999999', 81, 1);
  t('1E-17', '1E-22', '0.0000000000000000099999', 75, 3);
  t('0.0001', '1E-22', '0.0000999999999999999999', 43, 5);
  t('1E-10', '1E-40', '0.0000000000999999999999999999999999999999', 35, 3);
  t('1E-24', '1E-12', '-0.000000000000999999999999', 20, 1);
  t('1E-40', '1E-33', '-0.0000000000000000000000000000000009999999', 78, 5);
  t('1E-39', '1E-23', '-0.00000000000000000000001', 1, 6);
  t('0.1', '0.01', '0.09', 38, 5);
  t('1E-29', '0.0001', '-0.00009999999999999999999999999', 49, 6);
  t('1E-17', '1E-28', '0.0000000000000000099999999999', 100, 2);
  t('1E-27', '1E-23', '-0.000000000000000000000009999', 94, 5);
  t('1E-12', '1E-17', '0.00000000000099999', 42, 6);
  t('1E-12', '1E-33', '0.000000000000999999999999999999999', 33, 2);

  t('0.9999999999999999999876881224', '640033848998621900.19735', '-640033848998621899.1973500000000000000123118776', 49, 2);
  t('1.000000000000000000000000119823914072125503157964', '3062705906358982193618.98485068689066024569704943195219', '-3062705906358982193617.984850686890660245697049312128275927874496842036', 94, 6);
  t('0.9999999999999998265534546200191793251275036241640334423212917258444307775885354012967962729546', '258319.60193860385931337782', '-258318.60193860385931355126654537998082067487249637583596655767870827415556922241146459870320372705', 98, 4);
  t('0.99999999999999999989761950386191857639251221379227189329555371776218724908031098142396778431218707', '726355355995140387099932991211229400.158448248205932318709', '-726355355995140387099932991211229399.1584482482059323188113804961380814236074877862077281067044463', 97, 5);
  t('1.000000000000000000000000000000000000027702350484831926292132972134993320024115135851747430016246', '79385320209954140084685835.28094025821251260106069305881382567515', '-79385320209954000000000000', 14, 4);
  t('0.99998712469413', '794547422607164224869232243086324.8360594446035869419718131828058', '-794547422607164224869232243086323.836072319909456942', 52, 3);
  t('0.9999999999998534962024501557924401736482182589661527', '5698985430281057199751490373953345194.59621391632', '-5698985430281057199751490373953345193.596213916320146503797549844207559826351781741', 82, 2);
  t('0.9999999999999999999999985533840413028010928652936714305576280330060940667532288645', '78657501050716794990357190861.70489425564556943595076975008', '-78657501050716794990357190000', 25, 5);
  t('0.9999999961220652049373184', '914035313573641721959451176671.848582838348529711402806', '-914035314000000000000000000000', 9, 5);
  t('1.11874482', '48840191373078238086640881533698465.717620778949', '-48840191373100000000000000000000000', 12, 5);
  t('0.281969520841', '525591380542776020976261376042231280.99050263561330377859863953', '-525591380542776020976261376042231280.70853311477230377859863953', 97, 6);
  t('0.9999999999999959369334762196188619130301649399733281630919124260101076514635673054327688686405515', '6892460359743232550483058284.326802388', '-6892460359743232550483058283.326802388000004063067', 49, 3);
  t('1.000000000000000000000000138365204598651999681848', '4.585694782001886784351261812739727', '-3.585694782001886784351261674374522401348000318152', 93, 4);
  t('1.00000000000000000001348089658161552137212593704802977308725958902187035801449881713928406671', '2599076526329799757363350602243099450.6957978861778433554878885203926487289', '-2599076526329800000000000000000000000', 14, 5);
  t('1.0000000000000', '66366654979619144184658698072192.357935388883183470438181288188381307278', '-66366650000000000000000000000000', 7, 5);
  t('1.000000', '18105728986041761640887595533283120477.2742082541073753474', '-18100000000000000000000000000000000000', 4, 1);
  t('0.9999', '560937701245564854434830710385849233.169675002587006894673914472266909', '-560937701245564854434830710385849232.169775002587006894673914472266909', 79, 2);
  t('0.9999998719213481023531897427716397778263888401893910733910220314504983682315253', '38792839790712300540097766016170179.72942773540650288197576877282', '-38792839790712300540097766016170178.72942786348515477', 52, 2);
  t('1.000000000000000000000000000000011337132819044149107', '97753977764753722505638575181479019.579624307127638397405681', '-97753977764753722505638575181479018.579624307127638397405681', 60, 4);
  t('1.01261395071914172725658272307075493652694434213346101402984', '1049710327243884264.3689519321', '-1049710327243884263.35633798138085827274341727692924506347305565786653898597016', 95, 6);
  t('0.99999999998635625720009473958665342273155724466867056964747162778310004832306258429717336387169', '12.1520940727652904005353452973720184', '-11.1520940727789341433352505577853649772684427553313294303525283722168999516769374157', 84, 2);
  t('0.999999999999999999999999999999999524622195377650474620654559769514364455807304047052', '7002515.6669052486740556972355374422866181829657', '-7002514.6669052486740556972355374', 32, 5);
  t('0.99999998878183120416824285037203', '38327.8407', '-38326.8407000112181687958317', 27, 1);
  t('0.999999999999999999999999999', '2.9094462300972', '-1.909446230097200000000000001', 55, 4);
  t('0.9999999999999999999999999996779494408046302584202610482', '530770914537012205776892173.865394333047000705228144653463710904', '-530770914537012205776892172.8653943330470007052281446537857614631953697415797389518', 92, 1);
  t('1.0000000000000000000000000000000000000211924266373601875662343025201383719314762448306414697', '33086682481347.6314289196308731545826716', '-33086682481346.63142891963087315458267159999999999997880757336264', 64, 6);
  t('0.9999999999999999999999999999999999999995086077216389080791500865', '43010632125906095550264.572650474808124', '-43010632125906095550263.572650474808124', 51, 4);
  t('0.999999999999999999999999999999392910865226086344321', '4584324554085375538027273692534240749302.742306700543168212764946899734664', '-4584324554085375538027273692534240749301.742306700543168212764946899735271089134773913655679', 91, 3);
  t('0.9999999999999999999999999999999999999921018515820697887426805224100947352172147724109795779', '830076114723591620059141676694501404709.4930298672', '-830076114723591620059141676694501404708.4930298672', 70, 4);
  t('1.000154688825648481267644205561160017273389804317525860318979525925354', '5590.1166644928260', '-5589.1165098040003515187323557944388', 35, 1);
  t('0.99999999999006027143696724994249662', '88359515632902648.759516639917666812111537547', '-88359515632902600', 15, 1);
  t('1.00000000000000', '95.33659318661235', '-94.33659318661235', 70, 4);
  t('0.9999999999999999999775634801237813332795738416510297362767757177787528', '10739394560873313157162707656670.252363942865247660125793434721595385', '-10739394560873313157163000000000', 23, 6);
  t('0.99999998955431286156626', '4902.7915368894281469434', '-4901.79153689987383408183374', 80, 2);
  t('0.99999999999999898259041082937473685908796752941185241324258874667581114315116781', '5517546011621870332399416576575.811785782758678907878062576', '-5517546011621870332399416576574.81178578275867992528765174662526314091203247058814758675741126', 93, 0);
  t('1.13135445288218095176064019413277569648722343676398316921046471781497209018036761926374980678', '3619072849985061077293.9012666679115988775', '-3619072849985061077292.76991222', 30, 4);
  t('0.999964371653162345075430201881', '168105725527169314379065790662916050221.475269915782886990770332731552414180376', '-168105725527169314379066000000000000000', 24, 5);
  t('0.999999999998343581666045', '21225778421730442449952304083.507476189532490354504704824693390193', '-21225778422000000000000000000', 11, 4);
  t('1.0000000000', '53.0721765572571058497526522442', '-52.0721765572571058497526522442', 68, 1);
  t('1.0000000000000000010868737982065976552678875918917937663867883830744', '72274249298653961307161266.2429171261996071156986500328106121752', '-72274249298653961307161265.2429171261996071146117762346040145199321124081082062336132116', 87, 1);
  t('1.00000000000000000000000000000000001026071005118758019147403591153361193415866494922695', '145.57075488191632841598918', '-144.57075', 8, 4);
  t('1.0000000', '526650518510726171000913231536479079.037380846608', '-526650518510726171000913231536479078.037380846608', 75, 3);
  t('0.999999999999999999999999999999999999997106197957371323499604358266134753009', '3584419408840321948825.676382768', '-3584419408840321948824.6763827680000000000000000000000000000028938020426286765003956', 83, 2);
  t('0.99999999880681185203794773690375608443466417194000936368334849553283470358861975018', '5.0975685392156383162823410521', '-4.0975685404088264642443933152', 30, 3);
  t('0.999999998177929451382948935869', '10103621840183015475172718.257', '-10103621840183015475172717.257000001822070548617051064131', 70, 4);
  t('1.000000000000000000000000000000000000000144374259221118949564393862079', '99403923311484468697407136499693668.59308470138007980955', '-99403923311484468697407136499693667.5930847013800798095499999999999999999998556257407788811', 90, 5);
  t('0.999999975691341207638223751903870453583011108741353767405960466759765078360050260758989020', '4.6084648305890876715', '-3.60846485489774646386177624809612954641698889125864623259403', 60, 2);
  t('0.999999999999999999895045963233918813836521284190202155769449', '429720283413999973159.756715979', '-429720283413999973158.756715979000000000104954036766081186163478715809797', 72, 2);
  t('0.999999999999999999998773564582848665317660249642891385732648080962811751', '59990316469528.036171429611687', '-59990316469527.036171429611687000001226435417151334682339750357108614267351919037188249', 100, 2);
  t('1.000011075533608457976204143394074562732956861700803109887313366700147999338062991286462275342326012', '2792362408227880809678840930573.860305000', '-2792362408227880809678840930572.8602939244663915420237958566059254372670431382991968901126866333', 97, 6);
  t('1.000000000000000000000000000000000000015973553865744063022762207811071071600821959910304633272757', '1809301547973844603902311.5654087887338690', '-1809301547973844603902310.56540878873386899999999999999999999998402644613425593697723779218893', 93, 3);
  t('0.999999999999999999999999999999999999893329386351827298183253105991893674068332982263468835259', '2757167858879.682885', '-2757167858878.6828850000000000000000000000000000001066706136481727018167468940081063259317', 89, 4);
  t('1.0000000000000000000000000000111360158322886005852576303546314011852429419271247', '93939314.46578658799426616', '-93939313.46578658799426615999999999998886398416771139941474236964536859881475705807', 82, 2);
  t('0.9999999999999999999998770154790480869633545879631257941833370190599047521999789229', '199829413575720837193657766.733709709179385212202071225', '-199829413575720837193657765.734', 30, 5);
  t('0.999999999999999999999999998135', '554606063.013454693418100772398897875014035248', '-554606062.01346', 14, 0);
  t('0.98773519715148', '2549541405629529059884688169455754452231.105907313888980241014461140', '-2549541405629529059884688169455754452230.11817211673750024101446114', 74, 6);
  t('0.99999999999999999999999999999999999999988230202698086139860756333417913240', '690065016931922.4085310572334', '-690065016931921', 15, 4);
  t('1.00000000000000', '1289887282480591855765.347633806181', '-1289887282480591855764.347633806181', 51, 0);
  t('0.999999999999999999999999999995658364099102648491089', '414085174968159470504048.51', '-414085174968159470504047.510000000000000000000000001', 51, 3);
  t('0.999999999999999295252487060650057351383150516174879', '6306669915887513524394303183172309.934564178035290474234779480', '-6306669915887513524394303183172308.934564178035291178982292419349942648616849483825121', 99, 4);
  t('1.00000000000000000000081253348145920781026085647520801897170982656182514222627649830571537940570097', '34180955261225283278.9546214472426960215969277072', '-34180955261225283277.954621447242696021596115173718540792189739143524791981028290173438174857773723', 98, 1);
  t('0.99999999999999999999999856479373626889523595783214491795216121174762099354921379755267504653649134', '8233795130549636376322.3369072392151862574878', '-8233800000000000000000', 5, 0);
  t('1.0001', '1607764700164263.45772807689415459406211472014192328', '-1607764700164262.45762807689415459406211472014192328', 65, 3);
  t('0.99999999999999999999999999999999999998756496998614330743704048912', '36423078435.49585469762969072900556033', '-36423078434.49585469762969072900556033000000000001243503', 56, 1);
  t('0.99999891041356952032727215514401427884704660602746473835', '195363860454397925.197522412317', '-195364000000000000', 6, 4);
  t('1.00000', '781339417511868985543776052995.98028599275239777987', '-781339417511868986000000000000', 18, 0);
  t('1.00000000000000001', '72.130607640926138152719501354908', '-71.130607640926138142719501354908', 74, 0);
  t('0.99999999999999999999999999999815156221279091250236161202515555239214607133', '184535429818126810905.866146730925744980274670738002788577', '-184535429818126810904.866146730925744980274670738004637014787209087497', 69, 1);
  t('0.999999999999999999999331856438039650339670670738698379280474551045601', '175218340352452.70507614829', '-175218340000000', 8, 5);
  t('1.0000000', '6626035360301835643744155055681183579.72680430492878230246147491362', '-6626035360301835643744155055681183578.7268', 41, 4);
  t('1.0000000000000000014816657759635935574232', '32818928503629136.252295990697644196564215870800902906892', '-32818928503629135.2522959906976', 30, 5);
  t('1.00000000000000000000000000000', '13043617905561942222230054077.21749663178357', '-13043617910000000000000000000', 10, 6);
  t('1.000000000', '468078670327435157159036637213089.68427', '-468078670327435157159036637213088.68427', 91, 2);
  t('1.000000000000000000000364962235111663114736570947300481441647272089259984421671059783280000', '91128496181.82733', '-91128496180.82733', 27, 0);
  t('1.00000000000000000000000000', '248084803156.38225825687792564146883', '-248084803155.38225825687792564146883', 45, 6);
  t('0.999996430603395547879085578374943218533075889', '752798095516784566364107011187.1828743644733064738998075060481075333023', '-752798095516784566364107011186.182877933869910926020721927673164314769224111', 88, 3);
  t('0.9999999864688912316764271500401957202149887487506193902184177755581162789923303216472682', '434.57654697091569924806', '-433.576546984446808016383572849959804279785011251249380609781582224441883721007669678353', 87, 5);
  t('0.999999999999999999999999999982814515318881641819728269066787504001787393727056279440923', '83875621901513231603980702356510524090.18029474594', '-83875621901513231603980702356510524089.18029474594', 66, 2);
  t('1.10455706926852759636854514325898495936688404757420394401166868002', '35601365816445214252.5525881378161872928170', '-35601365816445214251.44803106854765969644845485674101504063311595242579605598833131998', 98, 1);
  t('1.0000000000000000000000000100777014575445669415762036017243568735466409286660372771012100670802724', '899760814583915.493920081239', '-899760814583914.49392008123899999999999999', 42, 6);
  t('0.984114021239868072775897442427054089380451', '5368174736.76894671250154921560010347054', '-5368174735.784832691262', 22, 0);
  t('0.9999999999380984950253', '83972321187501515671178201780968.114223326', '-83972321187501515670000000000000', 19, 1);
  t('0.9999999999999999999', '977684073047763681239701380533.7210843444785440991219674', '-977684073047763681239701380532.7210843445', 40, 6);
  t('1.000182570092457168104845853335311043776', '571449324605826766.57809543188536449255427224746102', '-571449324605826765.5779129', 25, 5);
  t('1.000002907816994100773837', '728233850755.2332313401141380833836258554', '-728233850754.2332284323', 23, 4);
  t('1.0000000000000000000000', '68592074796033717984633919704192.29865862981318592502795535346119', '-68592074796033720000000000000000', 16, 4);
  t('1.000000000000000', '903783422370949770321379953874545365.79210759958265665013787195048', '-903783422370949770321379953874545364.79210759958265665013787195048', 90, 6);
  t('0.999999999999999999999999999999999999717820449394897427433674919219243501563657735037982299', '302299275353390.8981647238924456616512453522112', '-302299275353389.8981647238924456616512453522112000002821795506', 61, 4);
  t('0.9999999999999999999999999999999999984487898167125641473658922', '3585319.7759370122633314703102', '-3585318.77593701', 15, 5);
  t('1.00000000000000', '203506155814552758148.37264782776486', '-203506155814552758147.37264782776486', 87, 2);
  t('1.00000000000000120579185273161226763034133156520200356939876275446105426421498158283412', '4739163351387537956850958064407251558.61807582', '-4739163351387537956850958064407251557.618075819999998794', 55, 6);
  t('1.000000000000000000143533559194819040300297231250823252', '15868546371105409745483539490.0', '-15868546371100000000000000000', 12, 5);
  t('1.00000000000', '54832392.01861789482959289047969611', '-54832391.0186178948295928904797', 31, 4);
  t('0.9999999999998917142347210910448', '7645882914233858938098515181269111721255.2', '-7645882914233858938098515181269111721254.2', 43, 5);
  t('1.000000000019026254756354259135217413250706', '2347731114172144002848808544526073254840.183', '-2347731114172144002850000000000000000000', 21, 4);
  t('1.00000000000000000000', '5823756950779433872739279788630725.76794045546', '-5823756950779433872739279788630724.76794045546', 89, 3);
  t('0.99999969198488396329717135352024457590848674566799393432083784906001565273156394858272936', '454231689016.0', '-454231689015.000000308015116036702828646479755424091513254332006065679162150939984347268436051417271', 99, 0);
  t('0.999983024504112322076111878226563575778225060676062233866162469025247716629315046304590555306536650', '92208.949978', '-92207.9499949754958876779238881217734364242217749393239377661338375309747522833707', 81, 3);
  t('0.999999999999999999988633888832566250132656497801403179018156138677466111241821611', '9129486452431567225984.698928988441611973802466974072821', '-9129486452431567225983.6989289884416119738139', 44, 0);
  t('0.99822719738396736525667460628136940989020697', '531550942791472583639.938880089389348', '-531550942791472583638.94065289200538063474332539371864', 53, 0);
  t('0.99629097248777667878668059904642041828035606099947021', '55813978579621.14619474187723', '-55813978579620.14990377', 22, 5);
  t('1.000000000000000000000', '99056936495226365315892.123168349533765718909', '-99056936495226365315891.123168349533766', 38, 6);
  t('0.99999999999999999999999999999998537924', '8448121780.740448713003666762457063113', '-8448121779.74044871300366676245706311300001462076', 92, 5);
  t('0.99999999988101657646132948979847662022351119638102579488054328360426024854964444951581938034', '26431528769156126902185.5455342185734', '-26431528769156126902184.5455342186923834235386705102015233797764888', 66, 6);
  t('0.999999999999999999999999999999999773647865082540886511424610602720', '94.88', '-90', 1, 4);
  t('1', '28714265844773879322429567222.0', '-28714265844773879322429567221', 51, 3);
  t('1.00000000000000000000000000000002230745989026902237', '5182909643868660181.616326849296578520834', '-5182909643868660180.61632684929657852083399999999997769254010973097763', 74, 1);
  t('0.99999999998849524198691154110204351283244388511160071147040874333277620620', '487171785.353436844076', '-487171784.35343684408750476', 26, 6);
  t('0.99999999999999999995826254', '623264646061209021513606736676.4070464780349804249191774960065', '-623264646061209021513606736675.4070464780349804249609149560065', 76, 4);
  t('0.9999999999999999999999840316000599556090993211977492120887044730644124505912', '3400141293380019543581.846146', '-3400141293380019543580.84614600000000000000001596839994004439090067880225079', 75, 0);
  t('1.00000000000000000000000000000000017890501993570751536367804830406693776445', '7596063323395135740831786963557300.828122807129946719956', '-7596063323395135740831786963557299.82812280712994671995599999999999982109498006429248463', 87, 1);
  t('0.999999999999999999999999981830537099429946248094516383336914915456449167384292448663913', '6160.9833347271963', '-6159.98333472719630000000001', 27, 0);
  t('1.00000000000000000000000000', '140694913008020347351867336278.2604696945968', '-140694913008020347351867336277.2604696945968', 93, 2);
  t('1.00000000000000183062574240181614811643732437563124607364762306403722737412422201569764119662', '315468725452602772046101176967.0405677893315761072465140617', '-315468725452602772046101176966.040567789331574276620771659883851883562675624368753926352376935962', 96, 2);
  t('1.00000000000000020487871739872142801523484304068051994930096447885572730358527340688870270096190', '3223913266.58144343252766526385', '-3223913265.5814434325276650589712826012785719847651569593194800506990355211442727', 80, 4);
  t('0.83864421727143491142189501983147051332126197163408403622694220682570328285455003', '6177991817255640.0', '-6177991800000000', 8, 4);
  t('1.0000000', '52385.2928549712746987395557277', '-52384.2928549712746987395557277', 31, 6);
  t('0.9999999999999999999999969656277', '189067655662557843433650599.27270881530157259', '-189067655670000000000000000', 11, 0);
  t('1.0000000000', '80884.39846', '-80883.39846', 26, 5);
  t('0.99999999999999999999999620', '8104102.905199872665141004926', '-8104101.905199872666', 19, 3);
  t('0.9999999999999999999', '83532645279145516530203.4814629819050362463403556437845', '-83530000000000000000000', 4, 5);
  t('1.000', '895694428428700219865817784521974121281.8617', '-895694428428700219865817784521974121280.8617', 74, 2);
  t('1.0000000000000000000000000000000000000229021152905485877265916049147377596', '46587749069373859988241253.489', '-46587749069373859988241252.489', 63, 0);
  t('1.00000000000000000000000000003315723137560736009063125977104295', '125786914090542197734.1243843248365770', '-125786914090542197733.124384324836577', 38, 4);
  t('0.99999999999999999999999880956718792058', '7160823998851.8329358', '-7160823998850.8329358000001', 26, 0);
  t('1.0000000000000000000000000', '168185946163980552320.267147742897536930695190349307194331', '-168185946163980552319.267147742897537', 37, 3);
  t('1.013368631237074269967629841470163611', '705721440565168596203735309275.2226143048086127376737122109829328049', '-705721440565168596203735309274.2092456735715384677060823695127691939', 100, 5);
  t('0.9999999999999999999999895953786771511342299983348190561873466109129549287631216974725051767872153303', '564417756540611857691.339744659093279108641571889', '-564417756540611857690.33974465909327910864158229362132284886577000166518094382', 77, 3);
  t('0.99999999999999999998540837496861607141739434483624886985909476644326279641174', '702032398191198338390179.03234814268263959712', '-702032398191198338390178.03234814268263959713459162503138392858260566', 68, 3);
  t('0.99999999997144379431778592551299579284817269428143146335310624910486', '5744510660347753705135441208927833216328.435064197579250286979352170', '-5744510660347753705135441208927833216327.435064197607806492661566244487004', 73, 1);
  t('0.9999999999999999999999999999999999999998703780041291982139945219878674624', '5520834448.5302613321828479018112575569', '-5520834447.5302613321828479018112575569000000000001296219958708017860054780121325376', 90, 0);
  t('1.00000000000000000021756877520948475176291147979521588259993838886521586727381224303946367', '8458303053559696574120.570523', '-8458303053559696574100', 20, 2);
  t('1.0000000000000000000', '5966.1758018', '-5965.1758018', 58, 1);

  t('606.254575304', '10913610393350.2606', '-10913610392744.006024696', 37, 1);
  t('3637644346217992.6517548755488192', '61.23387301712754337', '3637644346217931.41788185842', 27, 6);
  t('8562.3', '3857989719151989094.28436267039053249', '-3858000000000000000', 4, 5);
  t('42885995.8922914272', '706475.796938411', '42170000', 4, 1);
  t('7744926264.9828979895736', '23.42189', '7744926241.561008', 16, 2);
  t('2652458869.8552735', '275855874.003329237660489', '2376602995.851944262339511', 30, 4);
  t('982009515496588214.2961068036', '6139.9730427718831', '982009515496600000', 13, 6);
  t('674668148428.007557069664', '9730741.87736', '674658417686.1302', 16, 6);
  t('724155.70665100', '7644638.16160612893825643332', '-6920482.45495512893825643332', 29, 2);
  t('7489512119755281.0', '5696661.38689', '7489512114058619.61311', 27, 2);
  t('34021417679098405263.210121640063', '469015725494952.249997154054', '34020948663372910310.9601245', 27, 6);
  t('8104166310518.2057288', '331717.89', '8104165978800.3157288', 38, 4);
  t('685.46727959004970110362', '69830719.4666357936145496499', '-69830033.9993562036', 18, 6);
  t('91409621890423.735243', '583761545405875499.298205826415', '-583670135783985075.562962826415', 40, 3);
  t('63.62', '816473804427740.32776832822474', '-816473804427676.70776832822474', 29, 3);
  t('93913790211035413955.0', '37517895256.691410825590', '93913790173517518700', 19, 4);
  t('8171950.0', '7916992.81283069266914681297', '254957.18716930733085318703', 33, 2);
  t('70570310990471372.535740', '306380460.13915815763162', '70570310684090912.396581842368', 29, 6);
  t('904002719243514578.199046601', '392302713454.30007186959883636', '904002326940802000', 15, 2);
  t('61222626759880119720.6', '38.1763782264368', '61222626759880119682.4236217735632', 40, 5);
  t('106992249842.454676', '9783343723646590.895277', '-9783236731396748.44', 18, 5);
  t('315594693.3230', '601427787.397048404134', '-285833094.074048404134', 37, 5);
  t('2624664081780.107816961929532', '638739152166230.6570284141825715', '-636114488084450.5492114522530395', 34, 3);
  t('850761485.0159962973', '16557587439532.009816893', '-16556736678000', 11, 4);
  t('29143073887560442.2732826582829', '817114655795.2399384597929170', '29142256772904650', 16, 0);
  t('13683256304217.3', '79044625035670693.5671293', '-79030941779366476', 17, 4);
  t('6052717165.1446', '65455223170949076520.8060582718286379316', '-65455223164896300000', 15, 1);
  t('54131.622129937389021', '998107.4', '-943975.7778700627', 16, 0);
  t('4134679559878377.722073883', '77947153111703354860.05228739542293', '-80000000000000000000', 1, 3);
  t('5955452.7441572243480999', '916606845979019189.694793932', '-916606845973063000', 15, 1);
  t('1719979601587.803845', '582.660259024590', '1719979601005.14358597541', 33, 6);
  t('179.7986834569433903', '5.3035430636', '174.4951403933433903', 34, 4);
  t('2571182.9104', '5112317485037139357.0326821757855538364', '-5112317485034568174.12228217578555383', 36, 1);
  t('33777941893928234.1403399701827010', '4007790318278125722.91052580512126369', '-3000000000000000000', 1, 1);
  t('1608.6048924999255577935', '272293.285196', '-270680', 5, 4);
  t('53611325447739.249', '633369797029.0', '52977955650710.249', 20, 5);
  t('710416580769.486', '33.64813473834309', '710416580735.8378652', 19, 3);
  t('9611898517968409.49495', '3.912', '9611898517968405.583', 19, 0);
  t('28302362177443.355900', '6595.574858694980', '28302362170847.78104130502', 30, 2);
  t('61066.29222627342560448', '12924.77619098', '48150', 4, 2);
  t('132.65466', '712167.8700932215292641', '-712035.2', 7, 4);
  t('688767986059678709.380854', '32846480.81710', '688767986026832000', 15, 6);
  t('1768.11258632743', '33118961361.6262', '-33118960000', 7, 4);
  t('3648851903469184.5520025290', '616248879.263262', '3648800000000000', 5, 3);
  t('890038068185855.960980', '2709407759.1604115152745810269', '890035358778096.8005684847254189731', 38, 4);
  t('102.0', '1.670386774', '100.329613226', 38, 4);
  t('188638666620373.0', '264.8829877746255031', '188640000000000', 5, 2);
  t('63940.75794950117047', '31279.29426406007672410', '32661', 5, 4);
  t('79999.339185864', '77281398922.8', '-77281318923.460814136', 29, 2);
  t('2043.1155258', '8703901.5', '-8701858.3844742', 20, 2);
  t('6409343803262.388', '823497316.0', '6408520305946.388', 38, 1);
  t('78753063306376138.506170034261398', '394.626015216966077', '78753063306375743.88015481729533', 31, 2);
  t('3396217318606541981.25623355', '8090252540252652303.5496813397654', '-4694035221646110320', 18, 4);
  t('572125232723470402.673', '450102582792560243.4928609852', '122000000000000000', 4, 6);
  t('73.1469871709778', '91.10583849', '-17.958851319022', 14, 5);
  t('35811.99', '6730744506562.519', '-6000000000000', 1, 1);
  t('12948569823926516.1', '6032050833605576786.28750720978465442', '-6019102263781650270.18751', 24, 4);
  t('73.460260127414', '5580002381112368469.58572385874194', '-5580002381112368396.125464', 25, 0);
  t('36533488928501951.648348833', '34832817263.64', '36533454095684688.008348833', 34, 4);
  t('941537063116169.19', '294.64269', '941537063115874.5', 16, 5);
  t('5436517709034324.73139', '2494962.7', '5436517706539362.03139', 36, 2);
  t('3055391195233077250.6394833837101256009', '6427707361341089.0024974560131565626', '3000000000000000000', 1, 6);
  t('29074997967.4033077914', '566332182964.6602808', '-537250000000', 5, 1);
  t('685.6230300327385', '8297.123231678604', '-7611.5003', 8, 0);
  t('9529.144771387', '8895131556.883', '-8895122027.738228613', 34, 1);
  t('265633925199908208.42314005298861', '565128025.144', '265633924634780183.27915', 23, 0);
  t('1376622519151175.50445', '75251508328.9053133011113', '1376547267642846.5991366988887', 40, 1);
  t('3338928425.7256095268', '44744429118812497.041340991400', '-44744425780000000', 10, 5);
  t('6505421540677.424332952118826', '5946601.3849316529', '6505415594076.0394', 17, 3);
  t('2967456.687326291008', '53066388425896878.90368577441', '-53066390000000000', 7, 6);

  t('943058798364.395479068079592238958593748', '4262.860991837775288450967406945', '943058794101.534487230304303787991186803', 40, 5);
  t('3961880712.1739466698112561773101546317748', '768274235.957926640149693329', '3193606476.216020029662', 22, 4);
  t('705240179905.01961035839697848937836', '5.730085107325974', '705240179899.28952525107100448937836', 35, 4);
  t('4.677135032771256809341543004', '368037386167.5782655174332095276', '-368037386162.901130484661952718258456996', 94, 3);
  t('1634482637211224074.5176185357539524070468846', '51959012150.180327166277012544968432200', '1634482585252211924.3372913694769398620784524', 71, 4);
  t('7062679212073143326888924001163.02786', '670367382193775796879309.97397127870279732280890295', '7062678541705761133000000000000', 19, 1);
  t('4159462381849975492409125.0', '659.758349477474619798183023482274815728', '4160000000000000000000000', 3, 2);
  t('386.361433432357526852389133', '161581451832534896068758.824566698062465908442798606434258', '-161581451832534896068372.4631333', 31, 0);
  t('2989170915162203782.110570', '78042238635700889.19875125602523287111', '2911128676526502892.91181874397476712889', 96, 0);
  t('24459694394745404014469850905.84318799775716933', '206361207751899.635824898631920532', '24459694394745197653262099006.207363099125248798', 97, 0);
  t('961453869.879856', '6205777354.0303077454349819728607', '-5244323484.1504517454349819728607', 50, 3);
  t('224978326.9', '46099.92658151985887055028304994', '224932226.97342', 14, 2);
  t('4795500.09497120551913045', '43866652015.225143790341972440190589912', '-43861856515.13017258482284199', 29, 6);
  t('785759919848321456608.53471973157141969206251316', '50515325074711677303149183246092378.4', '-50515325074710892000000000000000000', 17, 6);
  t('1052.813283166117082', '573714957335293801886430730.75448811348282631726775089262354663', '-573714957335293801886429677.94120494736574431726775089262354663', 70, 4);
  t('16271428.62516560', '32898412714198161463717.4889152072', '-32898412714198200000000', 15, 3);
  t('900560899.913104266482415717003', '5952428176343381136666681609845550966809.5', '-5952428176343381136666681609844650405909.586895733517584282997', 79, 0);
  t('96646163996160433101650646872419298.8222', '589774359042273020.4302680978027718659076738429520984', '96646163996160432511876287840000000', 28, 0);
  t('7560498588.138', '7703300420181340100559937071730.98176915487027', '-7703300420181340100552376573142.8437691548703', 44, 0);
  t('8522365103427764558030271.51366810225', '88935863758311187415.13400', '8522276167564006246842856.37966810225', 45, 2);
  t('269622299212099.5658', '67080856574410241738.2', '-67080586952111029638.6342', 43, 1);
  t('178101542842138132551032196400994301.93920152593345597972723530172437819734', '9928207710724533478360637492329686410.096810135482', '-9750000000000000000000000000000000000', 4, 4);
  t('347539.651654349', '999164349246243097554653376.101753100580453', '-999164349246243000000000000', 16, 1);
  t('67.82933602918033391813', '80456255662446245353590549764939356874.08639717592445030759922844', '-80456255662446245353590549764939356806.257061146745', 50, 3);
  t('1286189309439198325290138412924.71091621710744938298299045634143848', '55925281201729145416213246160396310313.033984382012126643460839793', '-55925279915539835977014920870257897388.32306816490467726047784933665856152', 92, 0);
  t('409218380262925.03329205577858741776', '98299234078747591.9319529133359131207968770243284522', '-97890015698484666.8986608575573257030368770243284522', 65, 3);
  t('589630075341582712237298018007.626791554398075374457001806', '37072203775488.5460924874', '589630075341582675165094242519.080699066998075374457001806', 80, 1);
  t('952406921388034785124870359466.4208', '92386404607924941831558.493321724284807184287113592', '952406829001630177199928527907.927478275715192815712886408', 65, 6);
  t('239797538.62924219574590', '62.7179', '239797475.9113421957459', 29, 1);
  t('9117573077.38203829394236794264681559', '9485399161355567.024972352', '-9485390040000000', 9, 6);
  t('195854152327661808049809012.152492452144772559', '52666453496377990502.8656918042014982268', '195854099661208311671818509.2868006479432743322', 61, 6);
  t('68389708692892364357038.550642618', '346589125.87181', '68389708692892017767912.678832618', 74, 1);
  t('22366800802617651725451014763352512558.88384367', '56706199621325498.77362822019', '22366800802617651725394308563731187060.11021544981', 77, 1);
  t('222099131902528.823008', '23424944387813665353358.766033498779988076936', '-23424944165700000000000', 12, 4);
  t('94369.805612979192617825311244420237761233', '1404995131175289820.12435079015027778619', '-1404995131175195450.318737810957659960878755579762238767', 77, 2);
  t('51595640870900081.31194767388', '7760147060689.98166496533407685834688993271909084', '51590000000000000', 4, 6);
  t('9773886938068800575328768856352095468.6307897839608910717952963742165591', '444564767846600894514602578978208.76080300483076311650', '9773442373300953974434254253773117259.8699867791301279552952963742165591', 73, 5);
  t('999567934641977983949863368480.115289355375579041', '1165149936.615876680281694880228', '999567934641977983948698218543.499412675093884160772', 58, 3);
  t('5501088.6084944677846990708389945743093', '664927.656419331398418627625297930574', '4836160.9520751363862804432136966437353', 53, 1);
  t('39.48037672881845226078909407', '5341174765458518115561880076247.366376615978250578161608182949283345', '-5341174765458518115561880076207.885999887159798317372514112949283345', 85, 6);
  t('8117017575502275847692679.264176875515281734668365581147676', '8659713925984172141543085.308002265245744274949016821247', '-542696350481896293850406.043825389730462540280651240099324', 58, 3);
  t('553909918921981594.751577774245505483484884974672361660', '8054222209080795584363855.323441', '-8054221655170876662382260', 24, 2);
  t('962163351664.0', '332173.51965410', '962163019490.4803459', 51, 4);
  t('78570476306710514328422903.566399424510435349164985564', '20645082983476005744595606474867.7059235785374', '-20645004412999699034081278051964.139524154026964650835014436', 71, 3);
  t('666631867650769350383766311.24823253716', '232735963045186976112963962.24048736757366948', '433895904605582374270802349.00774516958633052', 81, 3);
  t('759785915873173118941850738960455977789.03259903086677271090681583525083391', '6239262109019238233981735614.87249900850446524387856346136836303', '759790000000000000000000000000000000000', 5, 5);
  t('47417002283487491793133.162876598099522479641', '5798942052554.5198526293050', '47417002277688549740578.64', 25, 4);
  t('6626460939175437003981392048059079.7245933323187090582585544', '909561620735130820650.3713179', '6626460939174527442360656917238429.3532754323187090582585544', 60, 3);
  t('65718846993876485099499095393267604350.5432361999471902331', '9133254.0', '65718847000000000000000000000000000000', 10, 2);
  t('9831060534.4113885', '40.264960990851888058', '9831060494.146427509148111942', 97, 5);
  t('618365246.16154228299443804064726416551494052021', '4175245358857853690274.6574698427569', '-4200000000000000000000', 2, 3);
  t('5794058568457386772009082293111862955.1723471', '7874.10120139', '5794058568457386772009082293111855081.07114571', 58, 3);
  t('115514655034192092805976.359058975764477', '7321838425189291796508643.82134639311753', '-7206323770155099703703000', 22, 0);
  t('4979836.458', '4343643405.293', '-4338663568.835', 93, 4);
  t('6962719933111298474232811011939872095.31542793502134199201685731559857711019', '45898320993079771121163323807243350900.791045197014396089', '-38935601059968472646930512795303478805.475617261993', 50, 2);
  t('96.45114046', '43788964069986958954302793689691871.503332629337662292231604855967058576', '-43788964069986958954302793689691775.052192169337662292231604855967058576', 83, 0);
  t('19.706629067026369851443323141433225326543', '9130912170755702578573915.749688', '-9130912170755702578573896.04305893297', 36, 2);
  t('999909783662285281933157327721667.775471879628', '2795909691798312648666149730968954.548860473478014471571999555369104559', '-1795999908136027366732992403247280', 33, 1);
  t('9055989.3253617695654064149036971384419806539658', '16700012813767870416936045681260.961743541', '-16700012813767870416936036625271.6363817714345935850963028615580193460342', 92, 2);
  t('3761160947240340619655699804028514166.1545031706631', '90008594170211932230610878657643955.0766771120853201398452422803203', '3671000000000000000000000000000000000', 4, 1);
  t('7052186163764582.7718768047025626908949225631756923472405', '279825326258.044979268268396', '7051906338438324.7268975364341666908949225631756923472405', 85, 6);
  t('55148738462073723350464458331198071161.324235957257244236', '890533822519200952605943.67027149110407422724883', '55148738462072832816641939130245465210', 37, 1);
  t('22048566656678706770922535389009.8800482', '691835258073242811318135316422877.78205857134', '-669786691416564104547212781033867.90201037134', 85, 5);
  t('360228259046240.247611499704462968697207149395', '2086914304371111332610306394896990.291351096738500858888680941788348731605', '-2086914304371110000000000000000000', 15, 4);
  t('237765665865861098214622.6280166600682705261137847840237788088', '49282802218452.8192989767794767678363', '237765665816578295996169.8087176832887937582774847840237788088', 73, 2);
  t('94963576434071265035613300794191.69660087123956630049887827698659406', '991367147561458404473577.06', '94963575442704117474160000000000', 22, 0);
  t('21039.0941271849563698186545060837025468', '61.986522669935', '20000', 1, 3);
  t('5527758.208579037564171467506', '57746543998576.5', '-57746538470818.291420962435828532494', 57, 1);
  t('53.526728183052625', '4566777936616620019069108049674.2001694968892442806307029510968199973907', '-4566777936616620019069108049620.6734413138366192806307029510968199973907', 81, 4);
  t('8398.06881519554924106264640920848', '78419713421175411471008565354902971834.199590637228446719811837169766362', '-78419713421175411471008500000000000000', 24, 2);
  t('46651569150015595210809704661940.8', '96834650155790.328525', '46651569150010000000000000000000', 13, 3);
  t('203662156936266018.56857105801974209381240694301095451', '77270.69753652195683217690846412859892084', '203662156900000000', 10, 3);
  t('69434.88031535436784733509731541971007', '8.5519048231739842309', '69426.3284105311938631', 21, 4);
  t('695265208240345480799.465208690311427717020174', '963908834.1637317007838017919941336969673787756325', '695265208239381571965.3014769895276259250260403030326212243675', 70, 5);
  t('875031885506622202313972150.7266913315589066826744', '581696758787378729955423404234905245415.0607657848516406595', '-581696758786503698069916782100000000000', 28, 0);
  t('3850085716409132708226314757323424740259.29106821935', '318215482803.0830194832761343614525696063', '3850085716409132708226314757005209257456.208048736074', 52, 4);
  t('2077002389327078369687189002.91', '239837588404470046844798445464872593503.053633023008384327820145755683896617', '-239837588402393044455471367095185404500.143633023008384327820145755683896617', 95, 2);
  t('525304094036133422209603266609297.83269311554678194352008621', '3352818947475215852524.43621', '525304094032780603262128100000000', 25, 5);
  t('2481703604661998598312313771116.683575522902', '121527168476013300130172028650996952931.129337247013280289581101702083117322', '-121527165994309695468173430338683181814.4457617241', 49, 5);
  t('729755648.217572562103468478434322387136565', '843447200457297.639006250839608805585207851529624505', '-843446470710000', 11, 3);
  t('387838224.031176138', '502511403087739175653052910412155182.44006582048399018208099', '-502511403087739175653052910024316958.408889682483990182081', 58, 6);
  t('69132135722.58295408468070686447778925874', '1453815723479122700218782754.639093106402300013304188564319839138', '-1453815723479122631086647032.05613902172159314', 45, 2);
  t('456578903803098400.0', '1905158313568197958999531007564638.80281445760356043208682451', '-1905158313568197502420627000000000', 25, 5);
  t('838363997375241408512205.91331625707234', '1817185772520982149742322396855847.0774430711345644250', '-1817185771682618152367080988343641.164126814062224425', 60, 0);
  t('7238789566926067985363334642130.599058758518787968', '6343845415.777256414176861661', '7238789566926067985356990796714.821802344341926307', 51, 3);
  t('9035266192537122485859011581230277071.66843338494296468220058', '360821803539883788027990763992.0', '9035265831000000000000000000000000000', 10, 3);
  t('3516070859853058886674366.675', '777750746025759323785264.6302948541581587174', '2738320113827299562889102.04', 27, 6);
  t('234667843801502022403492664022.51274243509522937067257601080027353748', '7434662831014255662256399.51019323374327636911', '234660409138671008147830407623.00254920135195300156257601080027353748', 96, 3);
  t('801.228464206716', '79633403862832778.892817935631812561652', '-79633403862831977.664353728915812561652', 57, 4);
  t('148761420566882053922085.5584439797471528210930019899', '1048230527982649187568473020741814535186.994351', '-1048230527982000000000000000000000000000', 13, 2);
  t('320094453723.90491004868317673103779600426809241', '6105321.02334476276', '320088348402.89', 14, 2);
  t('4470885354629185573049940913387317010.5199552238351', '584051218937333672.77333925100575115', '4470885354629185572465889694449983337.74661597282934885', 88, 2);
  t('213143.8021123372696220632021010884971460171', '9503836216.62157633547688479807890', '-9503623072.8194639982072627348767989115028539829', 87, 5);

  Decimal.toExpNeg = Decimal.toExpPos = 0;

  t('1.792316612884487564311771281e-23932057', '5.69976456866047320412517423508310354728679470757328490000343592950016294088615989268093866e-32517', '-5.6997645686604732041251742351e-32517', 29, 5);
  t('1.99169479851016450011043646901687400115512860504018833941164718920781076854216762430238626408343644e-9931933', '6.38526426739e+8626025', '-6.38526426739e+8626025', 22, 4);
  t('-6.2542519003586694651026234e+87', '-6.5757404951359531828365479313153601424485747649226579274168996862092441994766035e-49849', '-6.2542519003586694651026234e+87', 53, 3);
  t('-9.604772015363615559582304998733675e+5850', '8.1340843005908425924790547473450988769558068511930082393e+138332923', '-8.1340843005908425924790547473450988769558068511930082393e+138332923', 96, 2);
  t('-4.228107205843224077015863084700000947581407920789023984663576639e+72', '-2.3646076187718393062198539736528060703203e-18947270', '-4.22810720584322407701586308470000094758140792078902398466357663899999e+72', 69, 2);
  t('3.42164002916939306745661852229207679178670363e-739642191', '48413603180304870097616580848566368', '-4.8413603180304870097616580848566368e+34', 43, 5);
  t('-2.7082067257439001248731034e+255', '1.463357067849197668312169937835142233343040728649218403546975321163316191984829045020133415427186882e-813', '-2.708207e+255', 7, 5);
  t('1.80544723819507780400985933621596829e-23912898', '-6.8902351056372730055504911816243114779696717903217778350020023536206268396056597e+8117', '6.8902351056372730055504911816243114779696717903218e+8117', 50, 0);
  t('-1.688322353309930289764085302393645579e-2061', '8.304874790464942866574379155442852235973389929444676317853978559698488e+320', '-8.304874790464942866574379155442852235973389929444676317853978559698488e+320', 80, 6);
  t('5.323551831268452061208291787260167129e+3404403', '1.731761449136485742878138678929997425806089831701987933870812710697622052696577114307158491381462e+545', '5.32355183126845206e+3404403', 18, 5);
  t('-478232737.497', '4.0462091688110303902787237905655428288294712510868875182578760417628600744075185622547055498e+3079062', '-4.04620916881103039027872379056554282882947125e+3079062', 45, 6);
  t('3.0612417422416523917303965310500924897578e+6491', '7.827139363104333343854465329937791735830391528894758013298610462720743920', '3.0612417422416523917303965310500924897578e+6491', 54, 2);
  t('-5.3813883054823380e+324', '-1.2369677988e-57833611', '-5.381388305482338e+324', 75, 5);
  t('9.21805739519561831297761022643364049172592e-918', '-48895883124010996460744080319445610017323327196869255668.4572', '4.88958831240109964607440803194456100173233271968692556684572e+55', 66, 6);
  t('-9.252715780068260418043e+72', '-12590139569136468029213593074424819654932792387400646295775156790571242103916622815884889.5892', '1.25901395691364587764978130061644016119e+88', 39, 6);
  t('-5.71550608647635849585421199679478092798616968053825229082869453927359310584963054663809359831818000e+75134426', '-1e-353086', '-5.7155060864763584958542119968e+75134426', 30, 0);
  t('-1.82078636582034050e-58370', '6.949920119724528258522072178e-45427578', '-1.8207863658203405e-58370', 71, 6);
  t('-2.795864396006197633901037932062163941946590578075449507267908969758559434326789e+742563919', '-3.486025739887413584063563381470972389366187865215964884178573941676391907e-39238980', '-2.79586439600619763390103793206216394195e+742563919', 39, 5);
  t('9.481503849304826928973950761062868210560506712428214283679344428199e+986553613', '-6.44018958804822470871191968385562129218994153583004246052413787928461266956e-183768', '9.4815038493048269289739507610628682105605067124282142836793e+986553613', 59, 5);
  t('8.42058906940015718798e+23', '-5.12818647090105520279002851228627516477188332698656845964525170861373277e+77', '5.12818647091e+77', 12, 0);
  t('-6.66867298424442973991017135004123e-6016', '1.122942426365938279781316944e-50646915', '-6.66867298424442973991017135004123e-6016', 89, 2);
  t('-14434824877681671252494.50', '-4744180884604619155005023739354631157756385352850463825687419180.317893158', '4.744180884604619155005023739354631157756370918025586144016e+63', 58, 1);
  t('3.021351757092074226662594776398163721367430180658722006837511458091887e+754145', '-7.7281874393523698296251147e-342', '3.021351757092074226662594776398163721367430180658722006837511e+754145', 61, 3);
  t('-9.222486718751634618635052066622543e+36', '3.38442051790053e+18', '-9.22248672e+36', 9, 3);
  t('7.09899984575865359734e+84', '7.78137675765375240317008124083145599412855878668147615343173099333e-778', '7.09899984575865359734e+84', 36, 0);
  t('-7.731336555060253e+61', '-3.797617486236881717819332e+113', '3.79761748624e+113', 12, 6);
  t('-3203091812097591345956128', '1.54e+804483', '-1.540000000000001e+804483', 16, 3);
  t('-6.8790104211130242454991601642749891042049402313899949070009698688834394612168393565e-552', '-6.47132322792558510008703e+977', '6.4713232279255851000870299999999999999999999999999999999999999999999e+977', 68, 3);
  t('5.11288463e+804121757', '-3.233654093376025263229529432009804011579062041719e-323', '5.1128846300000000000000000000000000000000000000000000000000000000000000000000000000000000001e+804121757', 92, 0);
  t('2215721.822336', '1450572422564612919038700170868560596636485142539113457.847312756478875123323372035942316', '-1.450572422564612919038700170868561e+54', 34, 0);
  t('3.856021532534216364576565205962498e-680406380', '4.0748021e+910929', '-4.0748020999999999999999999999999999999999999999999999999999999999999999999999999999e+910929', 83, 2);
  t('587477294349.9128', '5.9009418178397792919787923374434637609777539615615913481542114343091e-601914', '5.874772943499128e+11', 53, 0);
  t('63490679413147320151422267902664.025528745', '-1.1511e+1066189', '1.1511e+1066189', 57, 4);
  t('4.339186830776212855076142e+864782108', '6.8227815138786859587759979265847699809e-704', '4.339186830776212855076141999999999999e+864782108', 37, 3);
  t('-42.683', '-2.1402464208082777156219367044346794646671634131699440029334548496490648e+95393976', '2.140246420808278e+95393976', 16, 4);
  t('-5.86269349525893423867699327633641571932e+53447445', '4.4471801597664471649566620984480e+70072792', '-4.447180159766447164956662098448e+70072792', 95, 6);
  t('2.01368392850485088e-7530', '7.80493785717377845822211894268763294620830205670933641096287239110624600304801026405564172615080e+76373', '-7.8049378571737784582221189426876329462083020567093364109628723911062460030480102640556e+76373', 86, 2);
  t('-9.443635658940048035845e-138325', '-1.35559639536973582525930597369597363666745264482626811e-35473', '1.355596395369735825259305973695973636667452644826268e-35473', 52, 5);
  t('1.8616211493256324295274776609561325391953686920009343298139595509785385e-3501', '1.03012623565635246794472661292428160395721926954319341162059243941806e+69', '-1.0301262356563524679447266129242816039572192695431934116205924394e+69', 65, 4);
  t('4.511087857625328082657257239895665113144003584405761127577260517484874613e+778771121', '1.8325842443473932072e-59442002', '4.511087857625328082657257239895665113144003584405761127577260517484874612999999999999999e+778771121', 88, 1);
  t('-1.287219294194087073000916324326305e+96', '7.1699261108103554603864614680e-7547516', '-1.28721929419408707300091632432630500001e+96', 39, 0);
  t('5.0461889643211e-8150', '-5.3418458279074099377871e+49476', '5.3418458279074099377871e+49476', 56, 4);
  t('-6.71547813417393495728240002804006106537518734009661124726571518977786981532e+3318', '-7.2398883759082488170424e+75376', '7.2398883759082488170424e+75376', 56, 5);
  t('1.44735793243277391262764608662589226432402931e+41712893', '-340885521769586478087899909642349839174961017948388220545761927469.281', '1e+41712893', 1, 6);
  t('6.561512285692418871312354839741304442546111797417416511e+94599649', '-440795478877.624', '6.561512285692418871312354839741304442546111797417416511e+94599649', 66, 6);
  t('-688837253042614877399.5', '-3.632441039346975825180266180783847897335359529888336884091855e+188507', '3.6324410393469758251802661807838478973353595298883368840918549999999999999999999999e+188507', 83, 1);
  t('1.0360675995358536872163314493066410401378536425375038250374718871686217599980919432e+553425', '8.704062929984294713140931351834846849951498e-593673477', '1.0360675995358536872163314493066410401378536425375038250374718871686217599980919431e+553425', 83, 3);
  t('7.43404688761493575549516036492094029880287e-26712037', '-7.9066293190521765772848055594328963843287417065639566213443958915377769052389889e+928963854', '7.90662931905217657728480555943289638432874170656395662134439e+928963854', 60, 1);
  t('1686783212633289750026735931738732785983663746521931468526088783025141583828124.465', '8768590549.7241944309880627200749822214571362095236108237501203844509903955330291000151422', '1.686783212633289750026735931739e+78', 31, 4);
  t('-1.30273886137268e+5289457', '-1.2899382417e+647330', '-1.30274e+5289457', 6, 4);
  t('42050878099274497028907191938263035774766181789514211433098485981439.75', '1.43994506473742424499458038815027395361046188494352856430241622193144915292115196755951196e-5923', '4.2050878099274497028907191938e+67', 29, 5);
  t('-4.320626740890915234398134986114517e+94658', '2.78894990560963181013521759e+8249', '-4.32062674089091523439813498e+94658', 27, 1);
  t('4.01525027773499792223184009769672475650267777439e+2993', '-9.7541734788872345649674286932141725163703554472540789341298781328709290320275810e-6547602', '4.01525027773499792223184009769672475650267777439e+2993', 67, 4);
  t('-3.95785198444775477048e-46', '-6.870396030942305e-7108', '-3.957851984447754770479999999999999999999999999999999999999999999999999999e-46', 73, 1);
  t('-9711.6203836557758490544267', '9.188784153327143382542570876093e-23501', '-9.7116203836557758490544267e+3', 29, 4);
  t('7.355460006727234208680888285222108269270481742254887642198770398692713606694685939591959411955545e-487104390', '-32815863771944.87', '3.281586377194487e+13', 24, 5);
  t('-2.6583825751466329584377255122872320456516043330212891532766393709048631359035760266577469045556542e-99686492', '1.3196779556e-3010', '-1.319677955600000000000000000000000000000000000001e-3010', 49, 3);
  t('-7.17914921169565067656826325978347627539814084292339438679298679106912228033991102060926370758854e-7922150', '7.149725093137932026784009177225420e+497084531', '-7.14972509313793202678400917722542e+497084531', 65, 1);
  t('6.8401837269807432719138151869469924006494010653417334710e+59', '-3.3013338878941172505094851600658908071364455751e+267', '3.30133388789411725050948516006589080713644557510000000000000000000000001e+267', 72, 0);
  t('-4.54866584280149572085681473646934865513679804760160868303937609538161216745e+50712', '6008765.20498946344036890775138116504265390042946401212817895579352563718709273657', '-4.5486658e+50712', 8, 5);
  t('-2.17306915251811963448649934897203802911487529061830244254311067431220811484827762498e-8070467', '9.76024516020697735879087332971502859486313033396978241761064994198378347207016488940e-15', '-9.76024516020697735879087332971502859486313033396978241761064994198378347207016489e-15', 81, 6);
  t('1.07099326000068746588224496478208445416181740811806641950018390004007596301502960407795142041621e-48973759', '-6.86643594114206914494853806604921827520006935681793235523223528319863051850507291287e-619', '6.8664359411420691449485380660492182752000693e-619', 44, 3);
  t('8.09229557883260745215088204147149523134000618264060088333656193879542292e-9131', '6.8621966724589426049058030194215748202384645065147845415643636106228614429216889627e+5710', '-6.862196672458942604905803019421574820238464506514784541564363610623e+5710', 67, 6);
  t('9.581451793682701427886005083345394028175040878149375394273513e-378782', '-3.042744613335733864056288278187491527807214836284653551560072721302128870257120169275325e+92742406', '3.04274461333573386405628827818749152780721483628e+92742406', 48, 4);
  t('5.7313e+4575578', '-6.364840640e+39426', '5.7313e+4575578', 73, 6);
  t('8.356458499777e-30', '-4.8e-10939393', '8.356458499777e-30', 40, 1);
  t('6.5195472944708148433514527800921108648346637999108500193846733263849685289748056063562486569260648e-3279', '1.937421712038539810513850428488208588363123168835091110018815864672320921451430e+67547265', '-1.9374217120385398105138504284882085883631231688350911100188158647e+67547265', 65, 5);
  t('-3.57599586681284801380522107189891356706964779329156399436948121466556878573027334286577e+33356', '-6.6883693946947144556866042137800276517141450815792257780747268780004604e+17956', '-3.57599586681284801380522107189891356707e+33356', 39, 4);
  t('-9.7224e-560399300', '2.6205204246262938260068273679375013815833069770430762996326779771138775544825087639482895011e-7856317', '-2.62052042463e-7856317', 12, 0);
  t('-3.715681211128611433143970130865055335882020223143276e+52288355', '287789445446044936290241879917633998688', '-3.715681211128611433143970130865056e+52288355', 34, 0);
  t('-8.2493944690319859817818378061643226370038799191809830064274762378667243798969812972072673e+113', '-1.845641201032288327245060081214715637229922954057240e+444094', '1.8456412010322883272450601e+444094', 26, 5);
  t('-5.89309192204153001500843135046e+69621244', '5.23486399668082298543743497700744461634328561270113191e+51146867', '-5.89309192204153001500843135046e+69621244', 85, 6);
  t('-9.332533364617829013735842749683e-1172', '5.1886033191439624397804333255765500761063229061403068373806210152358593405375e+93591', '-5.1886033191439624397804333255765500761063229061403068373806210152358593405375000000000000000001e+93591', 95, 0);
  t('7.5445762070e+288854366', '3.8063974317472361901e+3723940', '7.544576207e+288854366', 26, 0);
  t('-1.10202097949692682919643989760625683657125288006838604142487011903806380534e-57', '-7.686349839017994404992480435117213704809681451389739987668242695205627997632727166e-756892681', '-1e-57', 1, 2);
  t('-2.6397824546988e+8167', '2.24335709768620433408676393186381185893686419138146661581115354856133911663897900777674634334e-667005', '-2.6397824546988e+8167', 96, 2);
  t('8.95701118636167267321626e+31', '3.866403036598893507073274582036553740764934411841223762724e+692437', '-3.866403036598893507073274582036553741e+692437', 37, 3);
  t('-5.7548994542146425852173016343954785000442443441023113369093428041115489e+5485526', '2.377230710793e+240213000', '-2.3772307107930000000000000000000000000000000000000000000000000000000000000000000000000001e+240213000', 89, 3);
  t('6.82187852561709017305422060576561581323016263253171768648527e+5000', '7.04611109462407074965728925055745664e-47', '6.82187852561709017305422060576561581323016263253171768648527e+5000', 86, 4);
  t('-7e+346', '-5.9739701543133697799688640180015825274829505377320867981068366951396904e+48689101', '5.97397015431337e+48689101', 16, 4);
  t('98575514219130865770354786583585231780940810430604135675363131502461134646460224095552185218.067', '8.6917047455394e-6901', '9.8575514219130865770354786583585231780940810430604135675363132e+91', 62, 2);
  t('-307326217766153207641296337408407178240069815', '-2.08928068512362083349037857638914402632806700732817216924871595627504976e+79', '2.0892806851236208334903785763891439955954452307128514051190822154343319359930185e+79', 92, 4);
  t('-4.555772492857631171176698810e+267939772', '-6.5528127257655266409752878244003501007476069420122758335606274879164568165395899170518329221501437e+1605318', '-4.55577249285763117117669881e+267939772', 32, 4);
  t('5.991948215377e-6123814', '6.38169244547e-151999802', '5.991948e-6123814', 7, 1);
  t('-5.21392008878352879914684467839557603e-6383', '7.9128406143023651297161503149801950528574678665327896050715374145940581e+68652', '-7.91284e+68652', 7, 1);
  t('9.161508948580791329450886540869770096703513327961216315071048543864660403509816151467893300e+47184', '-875581155292196024043167430367.03', '9.161508948580791e+47184', 16, 4);
  t('-4.7064408883914958217230526613884166e-5322567', '4.351015580014501018829827548653770705101061619352333235878130703968596958820676e-219807', '-4.3510155800145010188298275486537707051010616194e-219807', 47, 0);
  t('-5.86864922318646549356624038e-6388', '5.15864e-74', '-5.15864e-74', 67, 6);
  t('6.85870e-682', '-5.937075111928240207419060418231137e+5622', '5.937075111928240207419060418231137e+5622', 60, 5);
  t('-6.2846594717691810204240002443373127012520917569388241491889161091457255e+405', '8.0153128400211520487097219622809800495350464749634911270946865114372247349824051499281e-403780935', '-6.284659471769181020424000244337312701252e+405', 41, 1);
  t('-2.51033165877791728750241775443190891984391066633109045373979918847785114646268e+11923016', '4.55723346737173346604225e+453358630', '-4.5572335e+453358630', 8, 5);
  t('-6.4628942182810825075881700012135312694160551114463952887191657863131398435523742532514461e-6893', '-2.02225243286523769309415121074539512381866987210186008388367868075931147507424746610012554354e+933', '2.0222524328652376930941512107453951238186698721018600838836786807593e+933', 68, 3);
  t('-3.962623644340542242418789633415625567324136758413e-1340635', '-8.525681521139144430830605987294226705653004742663e+143', '8.5256815211391e+143', 14, 6);
  t('5.04188783533720818821892800831885659358875702648575435572892191640197223326922385205881e+94', '-4.044799176079958739979061829854269217248659863992017958880957143157173214518526499880140332437e-114', '5.04188783533720818821892800831885659358875702648575435572892191640197223326922385205881e+94', 93, 1);
  t('1.907022118659156494293209737808617967820791932710407270985579455491981584539790796605649238854197520e-848365946', '1011010673605111764248575166293951178735325592521363033821328588157771619.377853', '-1.0110106736051118e+72', 17, 5);
  t('-2.5823111978958804549209956134391354686605e-39', '-2.3756e+28994781', '2.3756e+28994781', 70, 4);
  t('7.3611e-4667', '-8.2885529813162426964849138072454671342893024340594245978772222282845e-4350', '8.288552981316242696484914e-4350', 25, 6);
  t('2.81803362355775841871831055053034976376902815619094990590e-5055230', '-4.63574008674183265911935834304869991074733393973186872200729701292356572e+896792', '4.6357400867418326591193583430486999107473339397318687220072971e+896792', 62, 0);
  t('7.9e-554925', '-9.560683183921193703634759160036460577238395402766596695837039927663889e+632788428', '9.56068318392119370363475917e+632788428', 27, 0);
  t('-3.32941405604212020e-471060', '5.8742682933034054748993473533381157135243211041287576477331104705911641772138481682906049360e-6416', '-5.87426829330340547489934735333811571352432110412875764773311047059116417721384e-6416', 78, 1);
  t('-5.7215753817629721723180740209343545067304363734183776559544e+941', '7016523421977501249723778592612429967458684107745534160459415336808848152279743423.0668', '-5.721575381762972172318074020934e+941', 31, 2);
  t('-7.02702276643566900354455889172971653055573e+208095', '3.1385249373648119405249018029681510060240311651663321860133526277691069177182640849572656671e+1153567', '-3.13852493736481194052490180296815100602403116516633218601335262776910691771826408495726566710001e+1153567', 96, 3);
  t('-1.7832354748159574129369361988269904804308254487309290902349465951236972e+55700094', '9.90973236504901221625863735557853554449248096229915839712817629426762063100501286822728059080142235e+78756406', '-9.909733e+78756406', 7, 0);
  t('5.806100701116837194078609598286654001909211492267547200e-6351', '-3.1322555058724231886103050203444286e+8361722', '3.1322555058724231886103050203444286e+8361722', 89, 1);
  t('6.858541567594014583569e-92403457', '932344125683735556729917290084013738442958.911626146652618624368048421976479', '-9.323441256837355567e+41', 19, 1);
  t('-515603412562141323976260579.7608', '-4.3459986742e+61871', '4.3459986742e+61871', 24, 0);
  t('-2482.166560', '-8.7529651707683159200571472364119589276981748230359366217157560936540211030e+129', '8.75296518e+129', 9, 2);
  t('7137625499977739961.174647264422176975940443694511971170437331664836965532929116335954039', '6.344746501443024903148745334379114127368257359979286387628469760443123519696786e-564190646', '7.1376254999777399611746472644221769759404436945119711704373317e+18', 62, 5);
  t('-9.718417655944792606998316560239480283922e-4697', '-4.215428946661920636536185801565059268e-4262', '4.21542894666192063653618580156505926799999999999999999999999999999999999999999999999999999999999999e-4262', 99, 3);
  t('-8.73155178653866580e-972', '5.65980788569468390921784398580677872138446786064513360271081e-9214177', '-8.731551786538665800000000000000000000000000000000000000000000000000000000000000000000000000000000001e-972', 100, 3);
  t('-7.4825478541105716433040812574e-2957384', '-8933987726393636136642392373893254836880610561986804046816378730701943482730.285263', '8.93398772639363613664239237389325483688061056198680404681637873070194348e+75', 72, 1);
  t('-6.1868661228611360547121138333173109576855395466826784e-9251', '-3.3407355564230790311536623458601152993159e-206708339', '-6.18686612286113605471211383331731095768553954668267839999999999999999999e-9251', 72, 2);
  t('-8.2894772800817807e-795566101', '-882094911847133130492689954681328382693767501748865135497984118817292143479942556093392300518133.40', '8.820949118471331304926899546813283826937675017488651354979841188172921434799425560933923005181334e+95', 97, 2);
  t('-3.71e-54', '8.1409094135303963371158960125104271236131e-52265', '-3.710000000000000000000000000000000000000000000000000000000000001e-54', 64, 3);
});
