# frozen_string_literal: true

require "reline"

module Byebug
  #
  # Handles byebug's history of commands.
  #
  class History
    attr_reader :size

    def initialize
      @size = 0
    end

    #
    # Array holding the list of commands in history
    #
    def buffer
      Reline::HISTORY.to_a
    end

    #
    # Restores history from disk.
    #
    def restore
      return unless File.exist?(Setting[:histfile])

      File.readlines(Setting[:histfile]).reverse_each { |l| push(l.chomp) }
    end

    #
    # Saves history to disk.
    #
    def save
      n_cmds = Setting[:histsize] > size ? size : Setting[:histsize]

      File.open(Setting[:histfile], "w") do |file|
        n_cmds.times { file.puts(pop) }
      end

      clear
    end

    #
    # Discards history.
    #
    def clear
      size.times { pop }
    end

    #
    # Adds a new command to Reline's history.
    #
    def push(cmd)
      return if ignore?(cmd)

      @size += 1
      Reline::HISTORY.push(cmd)
    end

    #
    # Removes a command from Reline's history.
    #
    def pop
      @size -= 1
      Reline::HISTORY.pop
    end

    #
    # Prints the requested numbers of history entries.
    #
    def to_s(n_cmds)
      show_size = n_cmds ? specific_max_size(n_cmds) : default_max_size

      commands = buffer.last(show_size)

      last_ids(show_size).zip(commands).map do |l|
        format("%<position>5d  %<command>s", position: l[0], command: l[1])
      end.join("\n") + "\n"
    end

    #
    # Array of ids of the last +number+ commands.
    #
    def last_ids(number)
      (1 + size - number..size).to_a
    end

    #
    # Max number of commands to be displayed when no size has been specified.
    #
    # Never more than Setting[:histsize].
    #
    def default_max_size
      [Setting[:histsize], size].min
    end

    #
    # Max number of commands to be displayed when a size has been specified.
    #
    # The only bound here is not showing more items than available.
    #
    def specific_max_size(number)
      [size, number].min
    end

    #
    # Whether a specific command should not be stored in history.
    #
    # For now, empty lines and consecutive duplicates.
    #
    def ignore?(buf)
      return true if /^\s*$/.match?(buf)
      return false if Reline::HISTORY.empty?

      buffer[Reline::HISTORY.length - 1] == buf
    end
  end
end
