/** Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * xine main for X11
 *
 */
/* required for getsubopt(); the __sun test gives us strncasecmp() on solaris */
#if !defined(__sun) && ! defined(__FreeBSD__) && ! defined(__NetBSD__) && ! defined(__DragonFly__)
#define _XOPEN_SOURCE 500
#endif
/* required for strncasecmp() */
#define _BSD_SOURCE 1
#define _DEFAULT_SOURCE 1

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xui_getopt.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <math.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <sys/types.h>
#include <signal.h>

#include <locale.h>

#include <xine.h>
#include <xine/xineutils.h>

#include "common.h"
#include "config_wrapper.h"
#include "kbindings.h"
#include "mediamark.h"
#include "panel.h"
#include "playlist.h"
#include "tvout.h"
#include "videowin.h"
#include "actions.h"
#include "event.h"
#include "osd.h"
#include "stdctl.h"
#include "session.h"
#include "errors.h"
#include "download.h"
#include "xitk.h"
#include "xine-toolkit/cfg_parse.h"
#include "xine-toolkit/image.h"
#include "xine-toolkit/skin.h"

#ifndef XINE_VERSION_CODE
#define XINE_VERSION_CODE  XINE_MAJOR_VERSION*10000+XINE_MINOR_VERSION*100+XINE_SUB_VERSION
#endif

/*
 * global variables
 */
#define CONFIGFILE "config"
#define CONFIGDIR  ".xine"

#define	OPTION_VISUAL		1000
#define	OPTION_INSTALL_COLORMAP	1001
#define OPTION_DISPLAY_KEYMAP   1002
#define OPTION_SK_SERVER        1003
#define OPTION_ENQUEUE          1004
#define OPTION_VERBOSE          1005
#define OPTION_BROADCAST_PORT   1006
#define OPTION_NO_LOGO          1007
#define OPTION_POST             1008
#define OPTION_DISABLE_POST     1009
#define OPTION_NO_SPLASH        1010
#define OPTION_STDCTL           1011
#define OPTION_LIST_PLUGINS     1012
#define OPTION_BUG_REPORT       1013
#define OPTION_NETWORK_PORT     1014
#define OPTION_NO_MOUSE         1015


/* options args */
static const char short_options[] = "?hHgfvn"
#ifdef HAVE_XINERAMA
 "F"
#endif
#ifdef HAVE_LIRC
 "L"
#endif
#ifdef HAVE_XF86VIDMODE
 "F"
#endif
 "u:a:V:A:p::s:RW:G:BN:P:l::S:ZD::r:c:ET:I";

static const struct option long_options[] = {
  {"help"           , no_argument      , 0, 'h'                      },
#ifdef HAVE_LIRC
  {"no-lirc"        , no_argument      , 0, 'L'                      },
#endif
  {"spu-channel"    , required_argument, 0, 'u'                      },
  {"audio-channel"  , required_argument, 0, 'a'                      },
  {"video-driver"   , required_argument, 0, 'V'                      },
  {"audio-driver"   , required_argument, 0, 'A'                      },
  {"auto-play"      , optional_argument, 0, 'p'                      },
  {"auto-scan"      , required_argument, 0, 's'                      },
  {"hide-gui"       , no_argument      , 0, 'g'                      },
  {"hide-video"     , no_argument      , 0, 'H'                      },
  {"fullscreen"     , no_argument      , 0, 'f'                      },
#ifdef HAVE_XINERAMA
  {"xineramafull"   , no_argument      , 0, 'F'                      },
#endif
  {"visual"	    , required_argument, 0,  OPTION_VISUAL           },
  {"install"	    , no_argument      , 0,  OPTION_INSTALL_COLORMAP },
  {"keymap"         , optional_argument, 0,  OPTION_DISPLAY_KEYMAP   },
  {"network"        , no_argument      , 0, 'n'                      },
  {"network-port"   , required_argument, 0,  OPTION_NETWORK_PORT     },
  {"root"           , no_argument      , 0, 'R'                      },
  {"wid"            , required_argument, 0, 'W'                      },
  {"geometry"       , required_argument, 0, 'G'                      },
  {"borderless"     , no_argument      , 0, 'B'                      },
  {"animation"      , required_argument, 0, 'N'                      },
  {"playlist"       , required_argument, 0, 'P'                      },
  {"loop"           , optional_argument, 0, 'l'                      },
  {"skin-server-url", required_argument, 0, OPTION_SK_SERVER         },
  {"enqueue"        , no_argument      , 0, OPTION_ENQUEUE           },
  {"session"        , required_argument, 0, 'S'                      },
  {"version"        , no_argument      , 0, 'v'                      },
  {"deinterlace"    , optional_argument, 0, 'D'                      },
  {"aspect-ratio"   , required_argument, 0, 'r'                      },
  {"config"         , required_argument, 0, 'c'                      },
  {"verbose"        , optional_argument, 0, OPTION_VERBOSE           },
  {"stdctl"         , optional_argument, 0, OPTION_STDCTL            },
#ifdef XINE_PARAM_BROADCASTER_PORT
  {"broadcast-port" , required_argument, 0, OPTION_BROADCAST_PORT    },
#endif
  {"no-logo"        , no_argument      , 0, OPTION_NO_LOGO           },
  {"no-mouse"       , no_argument      , 0, OPTION_NO_MOUSE          },
  {"no-reload"      , no_argument      , 0, 'E'                      },
  {"post"           , required_argument, 0, OPTION_POST              },
  {"disable-post"   , no_argument      , 0, OPTION_DISABLE_POST      },
  {"no-splash"      , no_argument      , 0, OPTION_NO_SPLASH         },
  {"tvout"          , required_argument, 0, 'T'                      },
  {"list-plugins"   , optional_argument, 0, OPTION_LIST_PLUGINS      },
  {"bug-report"     , optional_argument, 0, OPTION_BUG_REPORT        },
  {"no-gui"         , no_argument      , 0, 'I'                      },
  {0                , no_argument      , 0, 0                        }
};

#undef TRACE_RC

/*
 * This function duplicate argv[] to an char** array,,
 * try to open and parse a "~/.xine/xinerc" rc file,
 * and concatenate found entries to the char **array.
 * This allow user to always specify a command line argument.
 */
static char **build_command_line_args (int argc, char *argv[], int *_argc) {
  int i = 1, j;
  size_t rcs;
  char *rcn, *rcf, **_argv = calloc ((argc + 1), sizeof (char *));

  if (!_argv) {
    *_argc = 0;
    return NULL;
  }
  *_argc = argc;
  _argv[0] = strdup (argv[0]);

  rcn = xitk_asprintf ("%s/" CONFIGDIR "/xinerc", xine_get_homedir ());
  rcs = 128 << 10;
  rcf = xitk_cfg_load (rcn, &rcs);
  if (rcf && (rcs < (128 << 10))) {
    char *next = rcf;
    while (1) {
      char **nargv, *line = xitk_cfg_next_line (&next);

      if (!line)
        break;
      nargv = realloc (_argv, sizeof (char *) * ((*_argc) + 2));
      if (!nargv)
        break;
      _argv = nargv;
      _argv[i] = strdup (line);
      i++;
      (*_argc)++;
    }
  }
  xitk_cfg_unload (rcf);
  free (rcn);

  for (j = 1; i < (*_argc); i++, j++)
    _argv[i] = strdup (argv[j]);

  _argv[(*_argc)] = NULL;

  return _argv;
}

static void free_command_line_args(char ***argv, int argc) {
  for (argc--; argc >= 0; argc--)
    free ((*argv)[argc]);
  free (*argv);
  *argv = NULL;
}

static void main_change_logo_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->logo_mrl = cfg->str_value;
}

static void sub_autoload_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->flags &= ~XUI_FLAG_auto_subtitle;
  gui->flags |= xitk_bitmove (cfg->num_value, 1, XUI_FLAG_auto_subtitle);
}

static void warn_too_slow_cb (void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->flags &= ~XUI_FLAG_warn_too_slow;
  gui->flags |= xitk_bitmove (cfg->num_value, 1, XUI_FLAG_warn_too_slow);
}

/*
 *
 */
static void show_version(void) {
  printf (_("This is xine (X11 gui) - a free video player v%s"), VERSION
#ifdef DEBUG
  "-[DEBUG]"
#endif
  );
  printf (_(".\n(c) %s The xine Team.\n"), "2000-2024");
}

static void print_formatted (char *title, const char *const *plugins) {
  const char  *plugin, blanks[] = "     ";
  char buffer[81 + 4], *s = buffer, *p, *e = buffer + sizeof (buffer) - 4;

  printf ("%s", title);

  strcpy (s, blanks); s += strlen (blanks);
  p = s;
  while ((plugin = *plugins++)) {
    unsigned int l = strlen (plugin);
    while (p + l + 3 > e) {
      if (p > s) {
        memcpy (p - 1, "\n", 2);
        printf ("%s", buffer);
        p = s;
      } else {
        l = e - s - 3;
        break;
      }
    }
    memcpy (p, plugin, l); p += l;
    memcpy (p, ", ", 3); p += 2;
  }
  if (p > s) {
    memcpy (p - 2, ".\n\n", 4);
    printf ("%s", buffer);
  }
}

static char *_config_default (xine_t *xine) {
  const char * const h = xine_get_homedir ();

  if (h) {
    const unsigned int l[3] = {strlen (h), strlen (CONFIGDIR), strlen (CONFIGFILE)};
    char *name = malloc (l[0] + l[1] + l[2] + 3);

    if (!name)
      return NULL;

    memcpy (name, h, l[0] + 1);
    memcpy (name + l[0], "/" CONFIGDIR, l[1] + 2);
    if ((mkdir (name, 0755) < 0) && (errno != EEXIST))
      fprintf (stderr, "Error creating %s: %d (%s)\n", name, errno, strerror (errno));
    memcpy (name + l[0] + 1 + l[1], "/" CONFIGFILE, l[2] + 2);
    if (!xine)
      return name;
    xine_config_load (xine, name);
    free (name);
  }
  return NULL;
}

/* FreeBSD: Taking protected visibility symbol address to struct causes link time error:
 *   ld: error: cannot preempt symbol: xine_list_spu_plugins
 * Fix by creating wrapper functions.
 */
#ifdef HAVE_PTR_TO_PROTECTED_VISIBILITY_FUNC
#  define _list_audio_output_plugins xine_list_audio_output_plugins
#  define _list_video_output_plugins xine_list_video_output_plugins
#  define _list_demuxer_plugins xine_list_demuxer_plugins
#  define _list_input_plugins xine_list_input_plugins
#  define _list_spu_plugins xine_list_spu_plugins
#  define _list_post_plugins xine_list_post_plugins
#  define _list_audio_decoder_plugins xine_list_audio_decoder_plugins
#  define _list_video_decoder_plugins xine_list_video_decoder_plugins
#else
#  define LIST_WRAPPER(TYPE) \
     static const char * const * _list_ ## TYPE ## _plugins (xine_t *xine) { return xine_list_ ## TYPE ## _plugins (xine); }
   LIST_WRAPPER(audio_output)
   LIST_WRAPPER(video_output)
   LIST_WRAPPER(demuxer)
   LIST_WRAPPER(input)
   LIST_WRAPPER(spu)
   LIST_WRAPPER(post)
   LIST_WRAPPER(audio_decoder)
   LIST_WRAPPER(video_decoder)
#  undef LIST_WRAPPER
#endif
static void list_plugins (const char *type) {
  const char   *const  *plugins;
  xine_t               *xine;
  int                   i;
  static const struct {
    const char *const *(*func)(xine_t *);
    char                name[24];
    char                type[16];
  } list_functions[] = {
    { _list_audio_output_plugins,  N_("   -Audio output:\n"),    "audio_out"     },
    { _list_video_output_plugins,  N_("   -Video output:\n"),    "video_out"     },
    { _list_demuxer_plugins,       N_("   -Demuxer:\n"),         "demux"         },
    { _list_input_plugins,         N_("   -Input:\n"),           "input"         },
    { _list_spu_plugins,           N_("   -Subpicture:\n"),      "sub"           },
    { _list_post_plugins,          N_("   -Post processing:\n"), "post"          },
    { _list_audio_decoder_plugins, N_("   -Audio decoder:\n"),   "audio_decoder" },
    { _list_video_decoder_plugins, N_("   -Video decoder:\n"),   "video_decoder" },
    { NULL,                            "",                           ""              }
  };

  xine = xine_new ();
  if (!xine)
    return;
  _config_default (xine);
  xine_init (xine);

  show_version ();

  if (type && type[0]) {
    unsigned int tl = strlen (type);

    for (i = 0; list_functions[i].func; i++) {
      if (!strncasecmp (type, list_functions[i].type, tl)) {
        if ((plugins = list_functions[i].func (xine))) {
          printf (_("\n Available xine's plugins:\n"));
          print_formatted (gettext (list_functions[i].name), plugins);
          break;
	}
      }
    }
    if (!list_functions[i].func)
      printf (_("No available plugins found of %s type!\n"), type);
  } else {
    printf (_("\n Available xine's plugins:\n"));
    for (i = 0; list_functions[i].func; i++) {
      if ((plugins = list_functions[i].func (xine)))
        print_formatted (gettext (list_functions[i].name), plugins);
    }
  }

  xine_exit(xine);
}

/*
 *
 */
static void show_usage (void) {
  const char   *const *driver_ids;
  const char   *driver_id;
  xine_t       *xine;
  const char * const *backends;
  const char *backend;

  xine = xine_new();
  if (xine) {
    _config_default (xine);
    xine_init (xine);
  }

  printf("\n");
  printf(_("Usage: xine [OPTIONS]... [MRL]\n"));
  printf("\n");
  printf(_("OPTIONS are:\n"));
  printf(_("  -v, --version                Display version.\n"));
  printf(_("      --verbose [=level]       Set verbosity level. Default is 1.\n"));
  printf(_("  -c, --config <file>          Use config file instead of default one.\n"));
  printf(_("  -V, --video-driver <drv>     Select video driver by id. Available drivers: \n"));
  printf("                               ");
  if((driver_ids = xine_list_video_output_plugins (xine))) {
    driver_id  = *driver_ids++;
    while (driver_id) {
      printf ("%s ", driver_id);
      driver_id  = *driver_ids++;
    }
  }
  printf ("\n");

  printf(_("  -A, --audio-driver <drv>     Select audio driver by id. Available drivers: \n"));
  printf("                               null ");
  if((driver_ids = xine_list_audio_output_plugins (xine))) {
    driver_id  = *driver_ids++;
    while (driver_id) {
      printf ("%s ", driver_id);
      driver_id  = *driver_ids++;
    }
  }
  printf ("\n");
  printf(_("  -u, --spu-channel <#>        Select SPU (subtitle) channel '#'.\n"));
  printf(_("  -a, --audio-channel <#>      Select audio channel '#'.\n"));
  printf(_("  -p, --auto-play [opt]        Play on start. Can be followed by:\n"
	   "                    'f': in fullscreen mode.\n"
	   "                    'h': hide GUI (panel, etc.).\n"
	   "                    'w': hide video window.\n"
	   "                    'q': quit when play is done.\n"
	   "                    'd': retrieve playlist from DVD. (deprecated. use -s DVD)\n"
	   "                    'v': retrieve playlist from VCD. (deprecated. use -s VCD)\n"));
#ifdef HAVE_XINERAMA
  printf(_("                    'F': in xinerama fullscreen mode.\n"));
#endif
  printf(_("  -s, --auto-scan <plugin>     auto-scan play list from <plugin>\n"));
  printf(_("  -f, --fullscreen             start in fullscreen mode,\n"));
#ifdef HAVE_XINERAMA
  printf(_("  -F, --xineramafull           start in xinerama fullscreen (display on several screens),\n"));
#endif
  printf(_("  -g, --hide-gui               hide GUI (panel, etc.)\n"));
  printf(_("  -I, --no-gui                 disable GUI\n"));
  printf(_("      --no-mouse               disable mouse event\n"));
  printf(_("  -H, --hide-video             hide video window\n"));
#ifdef HAVE_LIRC
  printf(_("  -L, --no-lirc                Turn off LIRC support.\n"));
#endif
  printf(_("      --visual <class-or-id>   Use the specified X11 visual. <class-or-id>\n"
	   "                               is either an X11 visual class, or a visual id.\n"));
  printf(_("      --install                Install a private colormap.\n"));
  printf(_("      --keymap [=option]       Display keymap. Option are:\n"
	   "                                 'default': display default keymap table,\n"
	   "                                 'lirc': display draft of a .lircrc config file.\n"
	   "                                 'remapped': user remapped keymap table.\n"
	   "                                 'file:<filename>': use <filename> as keymap.\n"
	   "                                 -if no option is given, 'default' is selected.\n"));
  printf(_("  -n, --network                Enable network remote control server.\n"));
  printf(_("      --network-port           Specify network server port number.\n"));
  printf(_("  -R, --root                   Use root window as video window.\n"));
  printf(_("  -W, --wid                    Use a window id.\n"));
  printf(_("  -G, --geometry <WxH[+X+Y]>   Set output window geometry (X style).\n"));
  printf(_("  -B, --borderless             Borderless video output window.\n"));
  printf(_("  -N, --animation <mrl>        Specify mrl to play when video output isn't used.\n"
	   "                                 -can be used more than one time.\n"));
  printf(_("  -P, --playlist <filename>    Load a playlist file.\n"));
  printf(_("  -l, --loop [=mode]           Set playlist loop mode (default: loop). Modes are:\n"
	   "                                 'loop': loop entire playlist.\n"
	   "                                 'repeat': repeat current playlist entry.\n"
	   "                                 'shuffle': select randomly a yet unplayed entry from playlist\n"
	   "                                 'shuffle+': same as 'shuffle', but indefinetely replay the playlist.\n"));
  printf(_("      --skin-server-url <url>  Define the skin server url.\n"));
  printf(_("      --enqueue <mrl> ...      Enqueue mrl of a running session (session 0)\n"));
  printf(_("  -S, --session <option1,option2,...>\n"
	   "                               Session managements. Options can be:\n"
	   "                    session=n   specify session <n> number,\n"
	   "                    mrl=m       add mrl <m> to the playlist,\n"
	   "                    audio=c     select audio channel (<c>: 'next' or 'prev'),\n"
	   "                    spu=c       select spu channel (<c>: 'next' or 'prev'),\n"
	   "                    volume=v    set audio volume,\n"
	   "                    amp=v       set amplification level,\n"
	   "                    loop=m      set loop mode (<m>: 'none' 'loop' 'repeat' 'shuffle' or 'shuffle+'),\n"
           "                    get_speed   get speed value (see man page for returned value).\n"
	   "                    (playlist|pl)=p\n"
	   "                                 <p> can be:\n"
	   "                                   'clear'  clear the playlist,\n"
	   "                                   'first'  play first entry in the playlist,\n"
	   "                                   'prev'   play previous playlist entry,\n"
	   "                                   'next'   play next playlist entry,\n"
	   "                                   'last'   play last entry in the playlist,\n"
	   "                                   'load:s' load playlist file <s>,\n"
	   "                                   'stop'   stop playback at the end of current playback.\n"
	   "                                   'cont'   continue playback at the end of current playback.\n"
	   "                    play, slow2, slow4, pause, fast2,\n"
	   "                    fast4, stop, quit, fullscreen, eject.\n"));
  printf(_("  -Z                           Don't automatically start playback (smart mode).\n"));
  printf(_("  -D, --deinterlace [post]...  Deinterlace video output. One or more post plugin\n"
	   "                                 can be specified, with optional parameters.\n"
	   "                                 Syntax is the same as --post option.\n"));
  printf(_("  -r, --aspect-ratio <mode>    Set aspect ratio of video output. Modes are:\n"
	   "                                 'auto', 'square', '4:3', 'anamorphic', 'dvb'.\n"));
#ifdef XINE_PARAM_BROADCASTER_PORT
  printf(_("      --broadcast-port <port>  Set port of xine broadcaster (master side)\n"
	   "                               Slave is started with 'xine slave://address:port'\n"));
#endif
  printf(_("      --no-logo                Don't display the logo.\n"));
  printf(_("  -E, --no-reload              Don't reload old playlist.\n"));
  printf(_("      --post <plugin>[:parameter=value][,...][;...]\n"
	   "                               Load one or many post plugin(s).\n"
	   "                               Parameters are comma separated.\n"
	   "                               This option can be used more than one time.\n"));
  printf(_("      --disable-post           Don't enable post plugin(s).\n"));
  printf(_("      --no-splash              Don't display the splash screen.\n"));
  printf(_("      --stdctl                 Control xine by the console, using lirc commands.\n"));
  printf(_("  -T, --tvout <backend>        Turn on tvout support. Backend can be:\n"));
  printf("                               ");
  if((backends = tvout_get_backend_names())) {
    backend = *backends++;
    while(backend) {
      printf("%s ", backend);
      backend = *backends++;
    }
  }
  printf("\n");
  printf(_("      --list-plugins [=type]   Display the list of all available plugins,\n"
	   "                               Optional <type> can be:\n"
	   "                    audio_out, video_out, demux, input, sub, post,\n"
	   "                    audio_decoder, video_decoder.\n"));
  printf(_("      --bug-report [=mrl]      Enable bug report mode:\n"
	   "                                 Turn on verbosity, gather all output\n"
	   "                                 messages and write them into a file named\n"
	   "                                 BUG-REPORT.TXT.\n"
	   "                                 If <mrl> is given, xine will play the mrl\n"
	   "                                 then quit (like -pq).\n"));
  printf("\n\n");
  printf(_("examples for valid MRLs (media resource locator):\n"
	   "  File:  'path/foo.vob'\n"
	   "         '/path/foo.vob'\n"
	   "         'file://path/foo.vob'\n"
	   "         'fifo://[[mpeg1|mpeg2]:/]path/foo'\n"
	   "         'stdin://[mpeg1|mpeg2]' or '-' (mpeg2 only)\n"
	   "  DVD:   'dvd://VTS_01_2.VOB'\n"
	   "  VCD:   'vcd://<track number>'\n"));
  printf("\n");

  xine_exit(xine);
}

/* Try to load video output plugin, by stored name or probing. */
static xine_video_port_t *load_vo (gGui_t *gui, const char *name, int *hide_win) {
  void *visual;
  const char * const *avail;
  const char *list[32], **p = list, **e = list + sizeof (list) / sizeof (list[0] - 1);
  xine_video_port_t *res;
  int di_cfg, vtype = XINE_VISUAL_TYPE_NONE;
  /* add the "auto" entry first. */
  *p++ = "auto";
  /* query available vos. */
  avail = xine_list_video_output_plugins (gui->xine);
  while (*avail && (p < e))
    *p++ = *avail++;
  *p = NULL;
  /* register and query the config entry. */
  di_cfg = xine_config_register_enum (gui->xine, "video.driver", 0, (char **)list,
    _("video driver to use"),
    _("Choose video driver. NOTE: you may restart xine to use the new driver"),
    CONFIG_LEVEL_ADV, CONFIG_NO_CB, CONFIG_NO_DATA);
  if (!strcasecmp (list[di_cfg], "dxr3")) {
    xine_cfg_entry_t cfg_entry;
    if (xine_config_lookup_entry (gui->xine, "dxr3.output.mode", &cfg_entry)) {
      if (!strcmp (cfg_entry.enum_values[cfg_entry.num_value], "letterboxed tv") ||
          !strcmp (cfg_entry.enum_values[cfg_entry.num_value], "widescreen tv"))
        *hide_win = 1;
    }
  } else if (!strcasecmp (list[di_cfg], "fb")) {
    *hide_win = 1;
  }
  /* get our preferred visual. */
  visual = video_window_get_xine_visual (gui->vwin, &vtype);
  /* the command line -V option. */
  if (name && name[0]) {
    unsigned int u;
    for (u = 0; list[u] && strcasecmp (name, list[u]); u++) ;
    if (list[u])
      di_cfg = u;
    else
      di_cfg = e - list;
  } else {
    name = list[di_cfg];
  }
  /* try user selection. */
  if (!strcasecmp (name, "none") || !strcasecmp (name, "null"))
    res = xine_open_video_driver (gui->xine, name, XINE_VISUAL_TYPE_NONE, NULL);
  else if (!strcasecmp (name, "fb"))
    res = xine_open_video_driver (gui->xine, name, XINE_VISUAL_TYPE_FB, NULL);
  else if (strcasecmp (name, "auto"))
    res = xine_open_video_driver (gui->xine, name, vtype, visual);
  else /* "auto" */
    name = list[di_cfg], res = NULL;
  if (res)
    return res;
  /* -V failed? */
  if (name != list[di_cfg]) {
    printf (_("main: video driver <%s> failed\n"), name);
    exit (1);
  }
  /* probe. */
  for (p = list + 1; *p; p++) {
    if (gui->verbosity >= 2)
      printf (_("main: probing <%s> video output plugin\n"), *p);
    res = xine_open_video_driver (gui->xine, *p, vtype, visual);
    if (res)
      return res;
  }
  if (gui->verbosity >= 1) {
      printf (_("main: all available video drivers failed.\n"));
      exit (1);
  }
  return NULL;
}

/* Try to load audio output plugin, by stored name or probing */
static xine_audio_port_t *load_ao (gGui_t *gui, const char *name) {
  const char * const *avail;
  const char *list[32], **p = list, **e = list + sizeof (list) / sizeof (list[0] - 1);
  xine_audio_port_t *res = NULL;
  int di_cfg;
  /* add the "auto" entry first. */
  *p++ = "auto";
  *p++ = "null";
  /* query available vos. */
  avail = xine_list_audio_output_plugins (gui->xine);
  while (*avail && (p < e))
    *p++ = *avail++;
  *p = NULL;
  /* register and query the config entry. */
  di_cfg = xine_config_register_enum (gui->xine, "audio.driver", 0, (char **)list,
    _("audio driver to use"),
    _("Choose audio driver. NOTE: you may restart xine to use the new driver"),
    CONFIG_LEVEL_ADV, CONFIG_NO_CB, CONFIG_NO_DATA);
  /* the command line -A option. */
  if (name && name[0]) {
    unsigned int u;
    for (u = 0; list[u] && strcasecmp (name, list[u]); u++) ;
    if (list[u])
      di_cfg = u;
    else
      di_cfg = e - list;
  } else {
    name = list[di_cfg];
  }
  /* try user selection. */
  if (!strcasecmp (name, "null")) {
    /* calling -A null is useful to developers, but we should not save it at config.
     * user without a sound card may go to setup screen changing audio.driver to NULL
     * in order to make xine start a bit faster. */
    if (gui->verbosity >= 2)
      printf (_("main: not using any audio driver (as requested).\n"));
    return NULL;
  }
  if (strcasecmp (name, "auto")) {
    res = xine_open_audio_driver (gui->xine, name, NULL);
    if (res)
      return res;
    /* -A failed? */
    if (name != list[di_cfg]) {
      if (gui->verbosity >= 1)
        printf (_("main: audio driver <%s> failed\n"), name);
      exit (1);
    }
  }
  /* probe. */
  for (p = list + 2; *p; p++) {
    if (!strcmp (*p, "none")) {
      if (gui->verbosity >= 2)
        printf (_("main: skipping <%s> audio output plugin from auto-probe\n"), *p);
      continue;
    }
    if (gui->verbosity >= 2)
      printf(_("main: probing <%s> audio output plugin\n"), *p);
    res = xine_open_audio_driver (gui->xine, *p, NULL);
    if (res)
      return res;
  }
  if (gui->verbosity >= 1)
    printf(_("main: audio driver probing failed => no audio output\n"));
  return NULL;
}

/*
 *
 */
static void event_listener (void *user_data, const xine_event_t *event) {
  gGui_t *gui = user_data;
  struct timeval tv;
#if XINE_VERSION_CODE < 10200
  static int mrl_ext = 0; /* set if we get an MRL_REFERENCE_EXT */
#endif

  /*
   * Ignoring finished event logo is displayed (or played), that save us
   * from a loop of death
   */
  if(gui->logo_mode && (event->type == XINE_EVENT_UI_PLAYBACK_FINISHED))
    return;

  xitk_gettime_tv (&tv);

  if(labs(tv.tv_sec - event->tv.tv_sec) > 3) {
    fprintf(stderr, "Event too old, discarding\n");
    return;
  }

  if (gui->stdctl_enable)
    stdctl_event (gui, event);

  switch(event->type) {

  /* frontend can e.g. move on to next playlist entry */
  case XINE_EVENT_UI_PLAYBACK_FINISHED:
    if(event->stream == gui->stream) {
#ifdef XINE_PARAM_GAPLESS_SWITCH
      if( xine_check_version(1,1,1) )
        xine_set_param(gui->stream, XINE_PARAM_GAPLESS_SWITCH, 1);
#endif
      xitk_lock (gui->xitk, 1);
      gui_playlist_start_next (gui, 0);
      xitk_lock (gui->xitk, 0);
    }
    else if(event->stream == gui->visual_anim.stream) {
      /* printf("xitk/main.c: restarting visual stream...\n"); */
      xitk_lock (gui->xitk, 1);
      visual_anim_play (gui, 2);
      xitk_lock (gui->xitk, 0);
    }
    break;

    /* inform ui that new channel info is available */
  case XINE_EVENT_UI_CHANNELS_CHANGED:
    if (event->stream == gui->stream) {
      xitk_lock (gui->xitk, 1);
      panel_update_channel_display (gui->panel);
      xitk_lock (gui->xitk, 0);
    }
    break;

    /* request title display change in ui */
  case XINE_EVENT_UI_SET_TITLE:
    if (event->stream == gui->stream) {
      xine_ui_data_t *uevent = (xine_ui_data_t *) event->data;
      mediamark_t *m = &gui->mmk;
      int changed;

      gui_playlist_set_str_val (gui, uevent->str, MMK_VAL_IDENT, GUI_MMK_CURRENT);
      gui_playlist_lock (gui);
      changed = mediamark_set_str_val (&m, uevent->str, MMK_VAL_IDENT);
      gui_playlist_unlock (gui);

      if (changed) {
        xitk_lock (gui->xitk, 1);
        video_window_set_mrl (gui->vwin, uevent->str);
        playlist_update_playlist (gui);
        panel_message (gui->panel, NULL);
        xitk_lock (gui->xitk, 0);
      }
    }
    break;

    /* message (dialog) for the ui to display */
  case XINE_EVENT_UI_MESSAGE:
    if(event->stream == gui->stream) {
      xine_ui_message_data_t *data = (xine_ui_message_data_t *) event->data;
      char                    buffer[8192], *q = buffer, *e = buffer + sizeof (buffer) - 16;
      const char             *s1, *s2 = NULL;
#define _MSG_INFO      1
#define _MSG_WARN      2
#define _MSG_WITH_EXPL 4
#define _MSG_WITH_ARGS 8
#define _LAST_TYPE XINE_MSG_AUDIO_OUT_UNAVAILABLE
#if defined(XINE_MSG_PERMISSION_ERROR) && (XINE_MSG_PERMISSION_ERROR > _LAST_TYPE)
#  undef _LAST_TYPE
#  define _LAST_TYPE XINE_MSG_PERMISSION_ERROR
#endif
#if defined(XINE_MSG_FILE_EMPTY) && (XINE_MSG_FILE_EMPTY > _LAST_TYPE)
#  undef _LAST_TYPE
#  define _LAST_TYPE XINE_MSG_FILE_EMPTY
#endif
#if defined(XINE_MSG_AUTHENTICATION_NEEDED) && (XINE_MSG_AUTHENTICATION_NEEDED > _LAST_TYPE)
#  undef _LAST_TYPE
#  define _LAST_TYPE XINE_MSG_AUTHENTICATION_NEEDED
#endif
#if defined(XINE_MSG_RECORDING_DONE) && (XINE_MSG_RECORDING_DONE > _LAST_TYPE)
#  undef _LAST_TYPE
#  define _LAST_TYPE XINE_MSG_RECORDING_DONE
#endif
      unsigned int type, flags;
      static const char * const tmsg[_LAST_TYPE + 2] = {
        [XINE_MSG_NO_ERROR]              = NULL,
        [XINE_MSG_GENERAL_WARNING]       = N_("*drum roll*, xine lib wants to take your attention "
                                              "to deliver an important message to you ;-):"),
        [XINE_MSG_UNKNOWN_HOST]          = N_("The host you're trying to connect is unknown.\n"
                                              "Check the validity of the specified hostname."),
        [XINE_MSG_UNKNOWN_DEVICE]        = N_("The device name you specified seems invalid."),
        [XINE_MSG_NETWORK_UNREACHABLE]   = N_("The network looks unreachable.\nCheck your network "
                                              "setup and/or the server name."),
        [XINE_MSG_CONNECTION_REFUSED]    = N_("The connection was refused.\nCheck the host name."),
        [XINE_MSG_FILE_NOT_FOUND]        = N_("The specified file or MRL could not be found. Please check it twice."),
        [XINE_MSG_READ_ERROR]            = N_("The source can't be read.\nMaybe you don't have enough rights for this, "
                                              "or source doesn't contain data (e.g: not disc in drive)."),
        [XINE_MSG_LIBRARY_LOAD_ERROR]    = N_("A problem occurred while loading a library or a decoder"),
        [XINE_MSG_ENCRYPTED_SOURCE]      = N_("The source seems encrypted, and can't be read."),
        [XINE_MSG_SECURITY]              = "",
        [XINE_MSG_AUDIO_OUT_UNAVAILABLE] = N_("The audio device is unavailable. "
                                              "Please verify if another program already uses it."),
#ifdef XINE_MSG_PERMISSION_ERROR
        [XINE_MSG_PERMISSION_ERROR]      = N_("You have no permission to read:"),
#endif
#ifdef XINE_MSG_FILE_EMPTY
        [XINE_MSG_FILE_EMPTY]            = N_("The file is empty:"),
#endif
#ifdef XINE_MSG_AUTHENTICATION_NEEDED
        [XINE_MSG_AUTHENTICATION_NEEDED] = N_("Sorry, not implemented (authentication)"),
#endif
#ifdef XINE_MSG_RECORDING_DONE
        [XINE_MSG_RECORDING_DONE]        = N_("The following stream has finished recording:"),
#endif
        [_LAST_TYPE + 1]                 = N_("*sight*, unknown error.")
      };
      static const uint8_t tflag[_LAST_TYPE + 2] = {
        [XINE_MSG_NO_ERROR]              = _MSG_INFO | _MSG_WITH_ARGS, /* (messages to UI) */
        [XINE_MSG_GENERAL_WARNING]       = _MSG_WARN | _MSG_WITH_EXPL | _MSG_WITH_ARGS, /* (warning message) */
        [XINE_MSG_UNKNOWN_HOST]          = _MSG_WITH_ARGS, /* (host name) */
        [XINE_MSG_UNKNOWN_DEVICE]        = _MSG_WITH_ARGS, /* (device name) */
        [XINE_MSG_NETWORK_UNREACHABLE]   = _MSG_WITH_ARGS, /* none */
        [XINE_MSG_CONNECTION_REFUSED]    = _MSG_WITH_ARGS, /* (host name) */
        [XINE_MSG_FILE_NOT_FOUND]        = _MSG_WITH_ARGS, /* (file name or mrl) */
        [XINE_MSG_READ_ERROR]            = _MSG_WITH_ARGS, /* (device/file/mrl) */
        [XINE_MSG_LIBRARY_LOAD_ERROR]    = _MSG_WITH_ARGS, /* (library/decoder)  */
        [XINE_MSG_ENCRYPTED_SOURCE]      = _MSG_WITH_ARGS, /* none */
        [XINE_MSG_SECURITY]              = _MSG_WARN | _MSG_WITH_EXPL | _MSG_WITH_ARGS, /* (security message) */
        [XINE_MSG_AUDIO_OUT_UNAVAILABLE] = 0, /* none */
#ifdef XINE_MSG_PERMISSION_ERROR
        [XINE_MSG_PERMISSION_ERROR]      = _MSG_WITH_ARGS, /* (file name or mrl) */
#endif
#ifdef XINE_MSG_FILE_EMPTY
        [XINE_MSG_FILE_EMPTY]            = _MSG_WITH_ARGS, /* file is empty */
#endif
#ifdef XINE_MSG_AUTHENTICATION_NEEDED
        [XINE_MSG_AUTHENTICATION_NEEDED] = _MSG_WITH_ARGS, /* (mrl, likely http) */
#endif
#ifdef XINE_MSG_RECORDING_DONE
        [XINE_MSG_RECORDING_DONE]        = _MSG_INFO | _MSG_WITH_ARGS, /* just what it says */
#endif
        [_LAST_TYPE + 1]                 = _MSG_WARN | _MSG_WITH_EXPL | _MSG_WITH_ARGS
      };

      type = XITK_0_TO_MAX_MINUS_1 (data->type, _LAST_TYPE + 1) ? data->type : _LAST_TYPE + 1;
      *q = 0;

      switch (type) {
        case XINE_MSG_NO_ERROR: /* (messages to UI) */
          { /* copy strings, and replace '\0' separators by '\n' */
            const char *p = data->messages ? data->messages : "\0";
            while (*p) {
              size_t l = strlcpy (q, p, e - q);
              q += l;
              if (q + 1 >= e)
                break;
              *q++ = '\n';
              p += l + 1;
            }
          }
          *q = 0;
          break;
        case XINE_MSG_ENCRYPTED_SOURCE: /* none */
          {
            int i;
            gui_playlist_lock (gui);
            i = strncasecmp (gui->mmk.mrl, "dvd:/", 5);
            gui_playlist_unlock (gui);
            if (!i)
              s2 = _("\nYour DVD is probably crypted. According to your country laws, you can or can't "
                     "install/use libdvdcss to be able to read this disc, which you bought.");
          }
          break;
        case XINE_MSG_AUDIO_OUT_UNAVAILABLE:
          gui_stop (NULL, gui);
          break;
#ifdef XINE_MSG_AUTHENTICATION_NEEDED
        case XINE_MSG_AUTHENTICATION_NEEDED: /* mrl */
          /* FIXME: implement auth dialogue box */
          break;
#endif
        default: ;
      }

      flags = tflag[type];
      if (gui->verbosity >= XINE_VERBOSITY_DEBUG)
        flags |= _MSG_WITH_EXPL | _MSG_WITH_ARGS;

      s1 = tmsg[type];
      if (s1) {
        q += strlcpy (q, gettext (s1), e - q);
        if (s2 && (q < e)) {
          *q++ = '\n';
          q += strlcpy (q, s2, e - q);
        }
      }
      if ((flags & _MSG_WITH_EXPL) && (q < e)) {
        if (data->explanation) {
          *q++ = '\n';
          q += strlcpy (q, (const char *)data + data->explanation, e - q);
        }
      }
      if ((flags & _MSG_WITH_ARGS) && (q < e)) {
        *q++ = '\n';
        if (data->parameters) {
          q += strlcpy (q, (const char *)data + data->parameters, e - q);
        } else {
          q += strlcpy (q, _("No information available."), e - q);
        }
      }

      if (buffer[0]) {
        unsigned int xui_flags = (flags & _MSG_INFO) ? XUI_MSG_INFO
                               : (flags & _MSG_WARN) ? (XUI_MSG_WARN | XUI_MSG_MORE)
                               : (XUI_MSG_ERROR | XUI_MSG_MORE);
        pthread_mutex_lock (&gui->no_messages.mutex);
        if ((gui->no_messages.used >= 0) &&
          ((event->tv.tv_sec < gui->no_messages.until.tv_sec)
          || ((event->tv.tv_sec == gui->no_messages.until.tv_sec) && (event->tv.tv_usec < gui->no_messages.until.tv_usec)))) {
          if ((unsigned int)gui->no_messages.used < sizeof (gui->no_messages.msg) / sizeof (gui->no_messages.msg[0])) {
            memcpy (gui->no_messages.msg[gui->no_messages.used], buffer, q - buffer + 1);
            gui->no_messages.flags[gui->no_messages.used] = xui_flags;
            gui->no_messages.used++;
            pthread_mutex_unlock (&gui->no_messages.mutex);
          } else {
            pthread_mutex_unlock (&gui->no_messages.mutex);
            if (gui->verbosity >= XINE_VERBOSITY_DEBUG)
              printf ("xine-ui: suppressed message (%" PRId64 ".%06d):\n%s\n",
                (int64_t)event->tv.tv_sec, (int)event->tv.tv_usec, buffer);
          }
        } else {
          pthread_mutex_unlock (&gui->no_messages.mutex);
          xitk_lock (gui->xitk, 1);
          gui_msg (gui, xui_flags, "%s", buffer);
          xitk_lock (gui->xitk, 0);
        }
      }

    }
    break;

    /* e.g. aspect ratio change during dvd playback */
  case XINE_EVENT_FRAME_FORMAT_CHANGE:
    break;

    /* report current audio vol level (l/r/mute) */
  case XINE_EVENT_AUDIO_LEVEL:
    if(event->stream == gui->stream) {
      xine_audio_level_data_t *aevent = (xine_audio_level_data_t *) event->data;

      xitk_lock (gui->xitk, 1);
      gui->mixer.level[SOUND_CARD_MIXER] = (aevent->left + aevent->right) / 2;
      gui->mixer.mute[SOUND_CARD_MIXER] = aevent->mute;
      panel_update_mixer_display (gui->panel);
      xitk_lock (gui->xitk, 0);
    }
    break;

#ifdef XINE_EVENT_AUDIO_AMP_LEVEL	/* Precaution for backward compatibility, will be removed some time */
    /* report current audio amp level (l/r/mute) */
  case XINE_EVENT_AUDIO_AMP_LEVEL:
    if(event->stream == gui->stream) {
      xine_audio_level_data_t *aevent = (xine_audio_level_data_t *) event->data;

      xitk_lock (gui->xitk, 1);
      gui->mixer.level[SOFTWARE_MIXER] = (aevent->left + aevent->right) / 2;
      gui->mixer.mute[SOFTWARE_MIXER] = aevent->mute;
      panel_update_mixer_display (gui->panel);
      xitk_lock (gui->xitk, 0);
    }
    break;
#endif

    /* last event sent when stream is disposed */
  case XINE_EVENT_QUIT:
    break;

    /* index creation/network connections */
  case XINE_EVENT_PROGRESS:
    if(event->stream == gui->stream) {
      xine_progress_data_t *pevent = (xine_progress_data_t *) event->data;
      char                  buffer[1024];

      snprintf(buffer, sizeof(buffer), "%s [%d%%]\n", pevent->description, pevent->percent);
      xitk_lock (gui->xitk, 1);
      panel_message (gui->panel, buffer);
      osd_message (gui, "%s", buffer);
      xitk_lock (gui->xitk, 0);
    }
    break;

#if XINE_VERSION_CODE < 10200
  case XINE_EVENT_MRL_REFERENCE:
    if(!mrl_ext && (event->stream == gui->stream) && gui->playlist.num) {
      xine_mrl_reference_data_t *ref = (xine_mrl_reference_data_t *) event->data;

      if (gui->verbosity)
	printf("XINE_EVENT_MRL_REFERENCE got mrl [%s] (alternative=%d)\n",
               ref->mrl, ref->alternative);

      gui_playlist_lock (gui);
      if(ref->alternative == 0) {
        gui->playlist.ref_append++;
        gui_playlist_insert (gui, gui->playlist.ref_append, ref->mrl, ref->mrl, NULL, 0, -1, 0, 0);
      } else {
        gui_playlist_set_str_val (gui, ref->mrl, MMK_VAL_ADD_ALTER, gui->playlist.ref_append);
      }
      gui_playlist_unlock (gui);
      xitk_lock (gui->xitk, 1);
      gui_pl_updated (gui, MMK_CHANGED_MRL);
      xitk_lock (gui->xitk, 0);
    }
    break;
#endif

#ifndef XINE_EVENT_MRL_REFERENCE_EXT
/* present in 1.1.0 but not 1.0.2 */
#define XINE_EVENT_MRL_REFERENCE_EXT 13
typedef struct {
  int alternative, start_time, duration;
  const char mrl[1];
} xine_mrl_reference_data_ext_t;
#endif
  case XINE_EVENT_MRL_REFERENCE_EXT:
    if((event->stream == gui->stream) && gui->playlist.num) {
      xine_mrl_reference_data_ext_t *ref = (xine_mrl_reference_data_ext_t *) event->data;
      const char *title = ref->mrl + strlen (ref->mrl) + 1;
#if XINE_VERSION_CODE < 10200
      mrl_ext = 1; /* use this to ignore MRL_REFERENCE events */
#endif
      if (gui->verbosity)
	printf("XINE_EVENT_MRL_REFERENCE_EXT got mrl [%s] (alternative=%d)\n",
               ref->mrl, ref->alternative);

      gui_playlist_lock (gui);
      if(ref->alternative == 0) {
        gui->playlist.ref_append++;
        /* FIXME: duration handled correctly? */
        gui_playlist_insert (gui, gui->playlist.ref_append, ref->mrl, *title ? title : NULL, NULL,
                               ref->start_time,
                               ref->duration ? (int)(ref->start_time + ref->duration) : -1,
                               0, 0);
      } else {
        /* FIXME: title? start? duration? */
        gui_playlist_set_str_val (gui, ref->mrl, MMK_VAL_ADD_ALTER, gui->playlist.ref_append);
      }
      gui_playlist_unlock (gui);
      xitk_lock (gui->xitk, 1);
      gui_pl_updated (gui, MMK_CHANGED_MRL);
      xitk_lock (gui->xitk, 0);
    }
    break;

  case XINE_EVENT_UI_NUM_BUTTONS:
    break;

  case XINE_EVENT_SPU_BUTTON:
    {
      xine_spu_button_t *spubtn = (xine_spu_button_t *) event->data;

      xitk_lock (gui->xitk, 1);
      video_window_set_cursor (gui->vwin, spubtn->direction ? CURSOR_HAND : CURSOR_ARROW);
      xitk_lock (gui->xitk, 0);
    }
    break;

  case XINE_EVENT_DROPPED_FRAMES:
    if (xine_get_param(gui->stream, XINE_PARAM_SPEED) <= XINE_SPEED_NORMAL) {
      xitk_lock (gui->xitk, 1);
      too_slow_window (gui);
      xitk_lock (gui->xitk, 0);
    }
    break;

  }
}

/*
 *
 */
int main(int argc, char *argv[]) {
  gGui_t *gui;
  /* command line options will end up in these variables: */
  int                     c = '?';
  int                     option_index    = 0;
  int                     audio_channel   = -1;
  int                     spu_channel     = -1;
  char                   *audio_driver_id = NULL;
  char                   *video_driver_id = NULL;
  sigset_t                vo_mask;
  char                  **_argv;
  int                     _argc;
  int                     session         = -1;
  int                     aspect_ratio    = XINE_VO_ASPECT_AUTO ;
  int                     no_auto_start   = 0;
  int                     old_playlist_cfg, no_old_playlist = 0;
  char                  **pplugins        = NULL;
  int                     pplugins_num    = 0;
  char                   *tvout            = NULL;
  char                   *pdeinterlace     = NULL;
  int                     enable_deinterlace = 0;
  char                  **session_argv     = NULL;
  int                     session_argv_num = 0;
  int                     retval           = 0;

  gui_init_params_t       gui_params;

  /* Set stdout always line buffered to get every     */
  /* message line immediately, needed esp. for stdctl */
  setlinebuf(stdout);

#ifdef ENABLE_NLS
  if((xitk_set_locale()) != NULL) {
    setlocale(LC_ALL, "");
    setlocale(LC_NUMERIC, "C");
  }
#endif

  bindtextdomain(PACKAGE, XITK_LOCALE);
  textdomain(PACKAGE);

  /* Check xine library version */
  if(!xine_check_version(1, 0, 0)) {
    int major, minor, sub;

    xine_get_version (&major, &minor, &sub);
    fprintf(stderr, _("Require xine library version 1.0.0, found %d.%d.%d.\n"),
	    major, minor,sub);
    exit(1);
  }

  sigemptyset(&vo_mask);
  sigaddset(&vo_mask, SIGALRM);
  if (sigprocmask (SIG_BLOCK,  &vo_mask, NULL))
    fprintf (stderr, "sigprocmask() failed.\n");

  gui = (gGui_t *)calloc (1, sizeof (*gui));
  if (!gui)
    return 1;

  gui->nextprev[0]            =
  gui->nextprev[1]            =
  gui->nextprev[2]            = gui;

  if (xitk_init_NULL ()) {
    gui->stream            = NULL;
    gui->autoscan_plugin   = NULL;
    gui->skin_server_url   = NULL;
    gui->nongui_error_msg  = NULL;
    gui->splash_win        = NULL;
    gui->panel             = NULL;
    gui->vwin              = NULL;
    gui->setup             = NULL;
    gui->mrlb              = NULL;
    gui->actrl             = NULL;
    gui->vctrl             = NULL;
    gui->viewlog           = NULL;
    gui->help              = NULL;
    gui->eventer           = NULL;
    gui->keyedit           = NULL;
    gui->streaminfo        = NULL;
    gui->tvset             = NULL;
    gui->tvout             = NULL;
    gui->mmkedit           = NULL;
    gui->plwin             = NULL;
    gui->lirc              = NULL;
    gui->pl_fb[0]          = NULL;
    gui->pl_fb[1]          = NULL;
    gui->load_stream       = NULL;
    gui->load_sub          = NULL;
    gui->osd               = NULL;
  }

#if 0
  gui->cursor_grabbed         = 0;
  gui->network                = 0;
  gui->network_port           = 0;
  gui->use_root_window        = 0;
  gui->verbosity              = 0;
  gui->broadcast_port         = 0;
  gui->no_gui                 = 0;
  gui->no_mouse               = 0;
#endif

  gui->display_logo           = 1;
  gui->flags                  = XUI_FLAG_splash | XUI_FLAG_post_v | XUI_FLAG_post_a | XUI_FLAG_start_lirc;
  gui->report                 = stdout;
  gui->ssaver_enabled         = 1;
  gui->orig_stdout            = stdout;

  gui->last_playback_speed    = XINE_FINE_SPEED_NORMAL;

  memset(&gui_params, 0, sizeof(gui_params));

  _argv = build_command_line_args (argc, argv, &_argc);
#ifdef TRACE_RC
  {
    int i;

    printf ("%s(): argc %d\n", __XINE_FUNCTION__, argc);
    for (i = 0; i < _argc; i++)
        printf ("argv[%d] = '%s'\n", i, _argv[i]);
  }
#endif

  {
    pthread_mutexattr_t attr;
    pthread_mutexattr_init (&attr);
    pthread_mutexattr_settype (&attr, PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init (&gui->mmk_mutex, &attr);
    pthread_mutexattr_destroy (&attr);
  }

  visual_anim_init (gui);

  gui_playlist_init (gui);

  /*
   * parse command line
   */
  opterr = 0;
  while ((c = getopt_long (_argc, _argv, short_options, long_options, &option_index)) != EOF) {
    char empty[1] = "", *sub = optarg ? xine_chomp (optarg) : empty;

    switch (c) {
      case 'L': /* Disable LIRC support */
        gui->flags &= ~XUI_FLAG_start_lirc;
        break;
      case 'u': /* Select SPU channel */
        if (!sub[0])
          break;
        sscanf (sub, "%i", &spu_channel);
        break;
      case 'a': /* Select audio channel */
        if (!sub[0])
          break;
        sscanf (sub, "%i", &audio_channel);
        break;
      case 'V': /* select video driver by plugin id */
        if (sub[0]) {
          video_driver_id = strdup (sub);
        } else {
          fprintf (stderr, _("video driver id required for -V option\n"));
          exit (1);
        }
        break;
      case 'A': /* Select audio driver */
        if (sub[0]) {
          audio_driver_id = strdup (sub);
        } else {
          fprintf (stderr, _("audio driver id required for -A option\n"));
          exit (1);
        }
        break;
      case 'p': /* Play [[in fullscreen][then quit]] on start */
        gui->flags |= XUI_FLAG_start_play;
        if (!sub[0])
          break;
        if (strrchr (sub, 'f'))
          gui->flags ^= XUI_FLAG_start_fullscr;
#ifdef HAVE_XINERAMA
        if (strrchr (sub, 'F'))
          gui->flags ^= XUI_FLAG_start_xinerama;
#endif
        if (strrchr (sub, 'h'))
          gui->flags ^= XUI_FLAG_start_vis;
        if (strrchr (sub, 'q'))
          gui->flags ^= XUI_FLAG_start_quit;
        if (strrchr (sub, 'w'))
          gui->flags ^= XUI_FLAG_start_vo_vis;
        if (strrchr (sub, 'd'))
          gui->autoscan_plugin = "DVD";
        if (strrchr (sub, 'v'))
          gui->autoscan_plugin = "VCD";
        break;
      case 'g': /* hide panel on start */
        gui->flags ^= XUI_FLAG_start_vis;
        break;
      case 'H': /* hide video on start */
        gui->flags ^= XUI_FLAG_start_vo_vis;
        break;
      case 'f': /* full screen mode on start */
        gui->flags ^= XUI_FLAG_start_fullscr;
        break;
#ifdef HAVE_XINERAMA
      case 'F': /* xinerama full screen mode on start */
        gui->flags ^= XUI_FLAG_start_xinerama;
        break;
#endif
      case 'D':
        if (pdeinterlace)
          break;
        enable_deinterlace++;
        if (!sub[0])
          break;
        pdeinterlace = strdup (sub);
        break;
      case 'r':
        if (!sub[0])
          break;
        if (!strcasecmp (sub, "auto"))
          aspect_ratio = XINE_VO_ASPECT_AUTO;
        else if (!strcasecmp (sub, "square"))
          aspect_ratio = XINE_VO_ASPECT_SQUARE;
        else if (!strcasecmp (sub, "4:3"))
          aspect_ratio = XINE_VO_ASPECT_4_3;
        else if (!strcasecmp (sub, "anamorphic"))
          aspect_ratio = XINE_VO_ASPECT_ANAMORPHIC;
        else if (!strcasecmp (sub, "dvb"))
          aspect_ratio = XINE_VO_ASPECT_DVB;
        else {
          fprintf (stderr, _("Bad aspect ratio mode '%s', see xine --help\n"), sub);
          exit (1);
        }
        break;
      case 's': /* autoscan on start */
        gui->autoscan_plugin = sub;
        break;
      case OPTION_VISUAL:
        if (!sub[0])
          break;
        free (gui_params.prefered_visual);
        gui_params.prefered_visual = strdup (sub);
        break;
      case OPTION_INSTALL_COLORMAP:
        gui_params.install_colormap = 1;
        break;
      case OPTION_DISPLAY_KEYMAP:
        {
          kbindings_dump_mode_t mode = KBT_DISPLAY_MODE_DEFAULT;
          if (sub[0]) {
            /* if (!strcasecmp (sub, "default"))
                 mode = KBT_DISPLAY_MODE_DEFAULT; */
            if (!strcasecmp (sub, "lirc")) {
              mode = KBT_DISPLAY_MODE_LIRC;
            } else if (!strcasecmp (sub, "remapped")) {
              gui->keymap_file = xitk_asprintf ("%s/" CONFIGDIR "/keymap", xine_get_homedir ());
              mode = KBT_DISPLAY_MODE_CURRENT;
            } else if (!strcasecmp (sub, "man")) {
              gui->keymap_file = xitk_asprintf ("%s/" CONFIGDIR "/keymap", xine_get_homedir ());
              mode = KBT_DISPLAY_MODE_MAN;
            } else if (!strncasecmp(sub, "file:", 5)) {
              if (!gui->keymap_file && sub[5]) {
                gui->keymap_file = xitk_filter_filename (sub + 5);
                continue;
              }
            }
          }
          kbindings_dump (gui, mode);
        }
        SAFE_FREE (gui->keymap_file);
        exit (1);
        break;
      case 'n': /* Enable remote control server */
        gui->network = 1;
        break;
      case OPTION_NETWORK_PORT:
        if (!sub[0])
          break;
        {
          int port = strtol (sub, &sub, 10);
          if ((port >= 1) && (port <= 65535))
            gui->network_port = port;
          else
            fprintf (stderr, _("Bad port number: %d\n"), port);
        }
        break;
      case 'R': /* Use root window for video output */
        gui->use_root_window = 1;
        break;
      case 'G': /* Set geometry */
        if (!sub[0])
          break;
        free (gui_params.geometry);
        gui_params.geometry = strdup (sub);
        break;
      case 'B':
        gui_params.borderless = 1;
        break;
      case 'N':
        if (!sub[0])
          break;
        visual_anim_add (gui, sub, 1);
        break;
      case 'P':
        if (!sub[0])
          break;
        gui->flags |= XUI_FLAG_start_plist;
        gui_playlist_add_item (gui, sub, 1, GUI_ITEM_TYPE_PLAYLIST, 0);
        /* don't load original playlist when loading this one */
        no_old_playlist = 1;
        break;
      case 'l':
        if (!sub[0]) {
          gui->playlist.loop = PLAYLIST_LOOP_LOOP;
          break;
        }
        if (!strcasecmp (sub, "loop"))
          gui->playlist.loop = PLAYLIST_LOOP_LOOP;
        else if (!strcasecmp (sub, "repeat"))
          gui->playlist.loop = PLAYLIST_LOOP_REPEAT;
        else if (!strcasecmp (sub, "shuffle+"))
          gui->playlist.loop = PLAYLIST_LOOP_SHUF_PLUS;
        else if (!strcasecmp (sub, "shuffle"))
          gui->playlist.loop = PLAYLIST_LOOP_SHUFFLE;
        else {
          fprintf (stderr, _("Bad loop mode '%s', see xine --help\n"), sub);
          exit (1);
        }
        break;
      case OPTION_SK_SERVER:
        if (!sub[0])
          break;
        gui->skin_server_url = strdup (sub);
        break;
      case OPTION_ENQUEUE:
        if (is_remote_running(((session >= 0) ? session : 0))) {
          if (session < 0)
            session = 0;
        }
        break;
      case OPTION_VERBOSE:
        gui->verbosity = sub[0] ? strtol (sub, &sub, 10) : 1;
        break;
#ifdef XINE_PARAM_BROADCASTER_PORT
      case OPTION_BROADCAST_PORT:
        if (!sub[0])
          break;
        gui->broadcast_port = strtol (sub, &sub, 10);
        break;
#endif
      case OPTION_NO_LOGO:
        gui->display_logo = 0;
        break;
      case OPTION_NO_MOUSE:
        gui->flags |= XUI_FLAG_no_mouse;
        break;
      case 'S':
        if (!sub[0])
          break;
        if (is_remote_running (((session >= 0) ? session : 0)))
          retval = session_handle_subopt (sub, NULL, &session);
        else {
          session_argv = (char **)realloc (session_argv, sizeof (char *) * (session_argv_num + 2));
          session_argv[session_argv_num++] = strdup (sub);
          session_argv[session_argv_num]   = NULL;
        }
        break;
      case 'Z':
        no_auto_start = 1;
        break;
      case 'c':
        if (!sub[0])
          break;
        gui->cfg_file = xitk_filter_filename (sub);
        break;
      case 'E':
        no_old_playlist = 1;
        break;
      case OPTION_POST:
        if (!sub[0])
          break;
        pplugins = (char **)realloc (pplugins, sizeof (char *) * (pplugins_num + 2));
        pplugins[pplugins_num++] = sub;
        pplugins[pplugins_num] = NULL;
        break;
      case OPTION_DISABLE_POST:
        gui->flags &= ~(XUI_FLAG_post_v | XUI_FLAG_post_a);
        break;
      case OPTION_NO_SPLASH:
        gui->flags &= ~XUI_FLAG_splash;
        break;
      case 'v': /* Display version and exit*/
        show_version ();
        exit (1);
        break;
      case 'h': /* Display usage */
      case '?':
        show_usage ();
        exit (1);
        break;
      case OPTION_STDCTL:
        gui->stdctl_enable = 1;
        break;
      case 'T':
        tvout = optarg;
        break;
      case OPTION_LIST_PLUGINS:
        list_plugins (sub);
        exit (1);
        break;
      case OPTION_BUG_REPORT:
        {
          FILE *f = fopen ("BUG-REPORT.TXT", "w+");
          if (!f) {
            fprintf (stderr, "fopen(%s) failed: %s.\n", "BUG-REPORT.TXT", strerror (errno));
            break;
          }
	  printf(_("*** NOTE ***\n"));
	  printf(_(" Bug Report mode: All output messages will be added in BUG-REPORT.TXT file.\n"));
#ifdef ENABLE_NLS
          if ((xitk_set_locale ()) != NULL) {
            setlocale (LC_ALL, "");
            setlocale (LC_NUMERIC, "C");
          }
#endif
          if (dup2 ((fileno (f)), STDOUT_FILENO) < 0)
            fprintf (stderr, "dup2() failed: %s.\n", strerror (errno));
          else if (dup2 ((fileno (f)), STDERR_FILENO) < 0)
            fprintf (stdout, "dup2() failed: %s.\n", strerror (errno));
          gui->report = f;
          gui->verbosity = 0xff;
          if (sub[0]) {
            session_argv = (char **) realloc(session_argv, sizeof(char *) * (session_argv_num + 2));
            session_argv[session_argv_num] = xitk_asprintf("mrl=%s", sub);
            session_argv[++session_argv_num]   = NULL;
            gui->flags |= XUI_FLAG_start_play | XUI_FLAG_start_quit;
          }
        }
        break;
      case 'I':
        gui->flags |= XUI_FLAG_no_gui;
        break;
      case 'W': /* Select wid */
        if (sub[0]) {
          sscanf (sub, "%i", &gui_params.window_id);
        } else {
          fprintf (stderr, _("window id required for -W option\n"));
          exit (1);
        }
        break;
      default:
        show_usage ();
        fprintf (stderr, _("invalid argument %d => exit\n"), c);
        exit (1);
    }
  }

  if(session >= 0) {
    /* Session feature was used, say good bye */
    if(is_remote_running(session) && (_argc - optind)) {
      /* send all remaining MRL options to session */
      int i;

      if(_argv[optind]) {
	for(i = optind; i < _argc; i++) {
	  char enqueue_mrl[strlen(_argv[i]) + 256];
	  snprintf(enqueue_mrl, sizeof(enqueue_mrl), "session=%d", session);
	  (void) session_handle_subopt(enqueue_mrl, _argv[i], &session);
	}
      }
      else
	fprintf(stderr, _("You should specify at least one MRL to enqueue!\n"));

    }

    free_command_line_args(&_argv, _argc);

    return retval;
  }

  show_version ();
  if (gui->verbosity) {
    int v[3];
    printf (_("Built with xine library %d.%d.%d (%s)\n"),
      XINE_MAJOR_VERSION, XINE_MINOR_VERSION, XINE_SUB_VERSION, XINE_VERSION);
    xine_get_version (v + 0, v + 1, v + 2);
    printf (_("Found xine library version: %d.%d.%d (%s).\n"),
      v[0], v[1], v[2], xine_get_version_string ());
  }

#ifndef DEBUG
  /* Make non-verbose stdout really quiet but keep a copy for special use, e.g. stdctl feedback */
  if (!gui->verbosity) {
    int   guiout_fd, stdout_fd;
    FILE *guiout_fp;

    if((guiout_fd = dup(STDOUT_FILENO)) < 0)
      fprintf(stderr, "cannot dup STDOUT_FILENO: %s.\n", strerror(errno));
    else if((guiout_fp = fdopen(guiout_fd, "w")) == NULL)
      fprintf(stderr, "cannot fdopen guiout_fd: %s.\n", strerror(errno));
    else if((stdout_fd = open("/dev/null", O_WRONLY)) < 0)
      fprintf(stderr, "cannot open /dev/null: %s.\n", strerror(errno));
    else {
      if (fcntl(guiout_fd, F_SETFD, FD_CLOEXEC) < 0) {
        fprintf(stderr, "cannot make guiout_fd uninheritable: %s.\n", strerror(errno));
      }

      if(dup2(stdout_fd, STDOUT_FILENO) < 0)
        fprintf(stderr, "cannot dup2 stdout_fd: %s.\n", strerror(errno));
      else {
        gui->orig_stdout = guiout_fp;
        setlinebuf(gui->orig_stdout);
      }

      close(stdout_fd); /* stdout_fd was intermediate, not needed any longer */
    }
  }
#endif

  /*
   * Initialize config
   */
  if (!gui->cfg_file) {
    struct stat st;
    gui->cfg_file = _config_default (NULL);

    /* Popup setup window if there is no config file */
    if (!gui->cfg_file || stat (gui->cfg_file, &st) < 0)
      gui->flags |= XUI_FLAG_start_setup;
  }

  /*
   * Initialize keymap
   */
  if (!gui->keymap_file)
    gui->keymap_file = xitk_asprintf ("%s/" CONFIGDIR "/keymap", xine_get_homedir ());

  pthread_mutex_init (&gui->seek.mutex, NULL);
  gui->seek.running = 0;
  gui->seek.newpos = -1;
  gui->seek.timestep = 0;

  pthread_mutex_init (&gui->no_messages.mutex, NULL);
  memcpy (gui->no_messages.helipad, "\x00\x01\x02\x03", 4);

  gui->xine = xine_new ();
  if (gui->cfg_file)
    xine_config_load (gui->xine, gui->cfg_file);
  xine_engine_set_param (gui->xine, XINE_ENGINE_PARAM_VERBOSITY, gui->verbosity);

  /*
   * Playlist auto reload
   */
  old_playlist_cfg = xine_config_register_bool (gui->xine, "gui.playlist_auto_reload", 0,
    _("Automatically reload old playlist"),
    _("If it's enabled and if you don't specify any MRL in command "
      "line, xine will automatically load previous playlist."),
    CONFIG_LEVEL_BEG, dummy_config_cb, gui);

  if(old_playlist_cfg && (!(_argc - optind)) && (!no_old_playlist)) {
    char buffer[XITK_PATH_MAX + XITK_NAME_MAX + 2];

    snprintf(buffer, sizeof(buffer), "%s/.xine/xine-ui_old_playlist.tox", xine_get_homedir());
    gui_playlist_add_item (gui, buffer, 1, GUI_ITEM_TYPE_PLAYLIST, 0);
  }

  gui->flags |= xitk_bitmove (xine_config_register_bool (gui->xine, "gui.subtitle_autoload", 1,
    _("Subtitle autoloading"),
    _("Automatically load subtitles if they exist."),
    CONFIG_LEVEL_BEG, sub_autoload_cb, gui), 1, XUI_FLAG_auto_subtitle);

  enable_deinterlace += xine_config_register_bool (gui->xine, "gui.deinterlace_by_default", 0,
    _("Enable deinterlacing by default"),
    _("Deinterlace plugin will be enabled on startup. Progressive streams are automatically "
      "detected with no performance penalty."),
    CONFIG_LEVEL_BEG, NULL, CONFIG_NO_DATA);

  if (enable_deinterlace)
    gui->flags |= XUI_FLAG_start_deint;

  gui->flags |= xitk_bitmove (xine_config_register_bool (gui->xine, "gui.dropped_frames_warning", 1,
    _("Warn user when too much frames are dropped."),
    CONFIG_NO_HELP, CONFIG_LEVEL_BEG, warn_too_slow_cb, gui), 1, XUI_FLAG_warn_too_slow);

  /*
   * init gui
   */

  gui_params.num_files = _argc - optind;
  gui_params.filenames = &_argv[optind];
  gui_init (gui, &gui_params);

  network_dl_init ();

  /* Automatically start playback if new_mode is enabled and playlist is filled */
  if (
    (
      (gui->flags & XUI_FLAG_smart_mode) &&
      (gui->playlist.num || (gui->flags & XUI_FLAG_start_plist)) &&
      !(gui->flags & XUI_FLAG_start_play)
    ) && (no_auto_start == 0)
  ) {
    gui->flags |= XUI_FLAG_start_play;
  }

  /*
   * xine init
   */
  xine_init (gui->xine);

  /* Get old working path from input plugin */
  {
    xine_cfg_entry_t  cfg_entry;

    if(xine_config_lookup_entry (gui->xine, "media.files.origin_path", &cfg_entry))
      strlcpy(gui->curdir, cfg_entry.str_value, sizeof(gui->curdir));
    else
      if (!getcwd(&(gui->curdir[0]), XITK_PATH_MAX))
        gui->curdir[0] = 0;
  }

  /*
   * load and init output drivers
   */
  /* Video out plugin */
  {
    int hide_win = 0;
    gui->vo_port = load_vo (gui, video_driver_id, &hide_win);
    if (hide_win)
      gui->flags |= XUI_FLAG_start_vo_vis;
  }
  SAFE_FREE(video_driver_id);

  /* Audio out plugin */
  gui->ao_port = load_ao (gui, audio_driver_id);
  SAFE_FREE(audio_driver_id);

  post_init (gui);
  post_deinterlace_init (gui, pdeinterlace);

  gui->stream = xine_stream_new (gui->xine, gui->ao_port, gui->vo_port);
#ifdef XINE_PARAM_EARLY_FINISHED_EVENT
  if( xine_check_version(1,1,1) )
      xine_set_param(gui->stream, XINE_PARAM_EARLY_FINISHED_EVENT, 1);
#endif

  gui->vo_none = xine_open_video_driver (gui->xine, "none", XINE_VISUAL_TYPE_NONE, NULL);
  gui->ao_none = xine_open_audio_driver (gui->xine, "none", NULL);

  osd_init (gui);

  /*
   * Setup logo.
   */
#ifdef XINE_LOGO2_MRL
#  define USE_XINE_LOGO_MRL XINE_LOGO2_MRL
#else
#  define USE_XINE_LOGO_MRL XINE_LOGO_MRL
#endif
  gui->logo_mode = 0;
  gui->logo_has_changed = 0;
  gui->logo_mrl = xine_config_register_filename (gui->xine, "gui.logo_mrl", USE_XINE_LOGO_MRL,
    XINE_CONFIG_STRING_IS_FILENAME,
    _("Logo MRL"), CONFIG_NO_HELP, CONFIG_LEVEL_EXP, main_change_logo_cb, gui);

  gui->event_queue = xine_event_new_queue(gui->stream);
  xine_event_create_listener_thread (gui->event_queue, event_listener, gui);

  if (tvout && tvout[0]) {
    if ((gui->tvout = tvout_init (gui, tvout)))
      tvout_setup(gui->tvout);
  }

  xine_set_param(gui->stream, XINE_PARAM_AUDIO_CHANNEL_LOGICAL, audio_channel);
  xine_set_param(gui->stream, XINE_PARAM_SPU_CHANNEL, spu_channel);
  xine_set_param(gui->stream, XINE_PARAM_AUDIO_REPORT_LEVEL, 0);
  xine_set_param(gui->stream, XINE_PARAM_AUDIO_AMP_LEVEL, gui->mixer.level[SOFTWARE_MIXER]);
#ifdef XINE_PARAM_BROADCASTER_PORT
  xine_set_param(gui->stream, XINE_PARAM_BROADCASTER_PORT, gui->broadcast_port);
#endif

  /* Visual animation stream init */
  gui->visual_anim.stream = xine_stream_new (gui->xine, NULL, gui->vo_port);
  gui->visual_anim.event_queue = xine_event_new_queue(gui->visual_anim.stream);
  gui->visual_anim.current = 0;
  xine_event_create_listener_thread (gui->visual_anim.event_queue, event_listener, gui);
  xine_set_param(gui->visual_anim.stream, XINE_PARAM_AUDIO_CHANNEL_LOGICAL, -2);
  xine_set_param(gui->visual_anim.stream, XINE_PARAM_SPU_CHANNEL, -2);
  xine_set_param(gui->visual_anim.stream, XINE_PARAM_AUDIO_REPORT_LEVEL, 0);

  /* subtitle stream */
  gui->spu_stream = xine_stream_new (gui->xine, NULL, gui->vo_port);
  xine_set_param(gui->spu_stream, XINE_PARAM_AUDIO_REPORT_LEVEL, 0);

  /* init the video window */
  video_window_select_visual (gui->vwin);

  xine_set_param(gui->stream, XINE_PARAM_VO_ASPECT_RATIO, aspect_ratio);

  /*
   * hand control over to gui
   */

  /* Initialize posts, if required */
  if(pplugins_num) {
    char             **plugin = pplugins;

    while(*plugin) {
      pplugin_parse_and_store_post (&gui->post_video, (const char *) *plugin);
      pplugin_parse_and_store_post (&gui->post_audio, (const char *) *plugin);
      plugin++;
    }

    pplugin_rewire_posts (&gui->post_video);
    pplugin_rewire_posts (&gui->post_audio);
  }

  gui_run(gui, session_argv);

  xine_exit (gui->xine);

  visual_anim_done (gui);
  free(pplugins);

  if(session_argv_num) {
    int i = 0;

    while(session_argv[i])
      free(session_argv[i++]);

    free(session_argv);
  }

  network_dl_deinit ();

  free(gui_params.geometry);
  free(gui_params.prefered_visual);

  pthread_mutex_destroy (&gui->mmk_mutex);
  pthread_mutex_destroy (&gui->no_messages.mutex);
  pthread_mutex_destroy (&gui->seek.mutex);

  if(gui->report != stdout)
    fclose(gui->report);
  if(gui->orig_stdout != stdout)
    fclose(gui->orig_stdout);

  free_command_line_args(&_argv, _argc);

  free (gui->cfg_file);
  free (gui->keymap_file);

  free (gui->skin_server_url);

  gui_playlist_deinit (gui);

  free (gui);
  return retval;
}

void gui_load_window_pos (gGui_t *gui, const char *name, int *x, int *y) {
  char buf[80], *e;

  memcpy (buf, "gui.", 4);
  for (e = buf + 4; *name; name++)
    *e++ = *name;
  memcpy (e, "_x", 3);
  *x = xine_config_register_num (gui->xine, buf, *x, NULL, NULL, CONFIG_LEVEL_DEB, NULL, NULL);
  e[1] = 'y';
  *y = xine_config_register_num (gui->xine, buf, *y, NULL, NULL, CONFIG_LEVEL_DEB, NULL, NULL);
}

void gui_save_window_pos (gGui_t *gui, const char *name, xitk_register_key_t key) {
  window_info_t wi;

  (void)gui;
  if ((xitk_get_window_info (gui->xitk, key, &wi))) {
    char buf[80], *e;

    memcpy (buf, "gui.", 4);
    for (e = buf + 4; *name; name++)
      *e++ = *name;
    memcpy (e, "_x", 3);
    config_update_num (gui->xine, buf, wi.x);
    e[1] = 'y';
    config_update_num (gui->xine, buf, wi.y);
  }
}

int gui_window_new (gui_new_window_t *nw) {
  int res;

  if (!nw)
    return -1;
  if (!nw->gui)
    return -1;

  if (nw->id)
    gui_load_window_pos (nw->gui, nw->id, &nw->wr.x, &nw->wr.y);

  nw->bg = NULL;
  if (nw->skin) {
    const xitk_skin_element_info_t *info = xitk_skin_get_info (nw->gui->skin_config, nw->skin);
    nw->bg = info ? info->pixmap_img.image : NULL;
  }
  if (nw->bg) {
    nw->wr.width = xitk_image_width (nw->bg);
    nw->wr.height = xitk_image_height (nw->bg);
    res = 1;
  } else {
    if ((nw->wr.width <= 0) || (nw->wr.height <= 0))
      return -3;
    /* HACK: have event sender (250x200) without title frame. */
    nw->bg = (nw->wr.width > 270) ? XITK_WINDOW_BG_FRAME : XITK_WINDOW_BG_SIMPLE;
    res = 0;
  }

  if (nw->adjust)
    nw->adjust (nw);
  nw->xwin = xitk_window_create_window_ext (nw->gui->xitk, nw->wr.x, nw->wr.y, nw->wr.width, nw->wr.height,
    nw->title, NULL, "xine", 0, gui_layer_above (nw->gui, NULL), nw->gui->icon, nw->bg);
  if (!nw->xwin)
    return -2;
  if (res == 0)
    nw->bg = xitk_window_get_background_image (nw->xwin);
  set_window_type_start (nw->gui, nw->xwin);

  nw->wl = xitk_window_widget_list (nw->xwin);

  if (nw->wfskin) {
    xitk_image_widget_t im = {
      .nw = {
        .wl = nw->wl,
        .skin_element_name = nw->wfskin,
        .tips = "",
        .userdata = nw,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
      }
    };
    xitk_widget_t *w = xitk_image_create (&im, nw->gui->skin_config);
    xitk_widget_set_window_focusable (w);
  }

  return res;
}

void gui_window_delete (gui_new_window_t *nw) {
  gui_save_window_pos (nw->gui, nw->id, nw->key);
  xitk_unregister_event_handler (nw->gui->xitk, &nw->key);
  xitk_window_destroy_window (nw->xwin);
  nw->xwin = NULL;
}

