import logging
from typing import Dict

from ...sim_type import SimTypeFunction, SimTypePointer, SimTypeLong, SimStruct, SimTypeInt, SimTypeChar, \
    SimTypeBottom, SimTypeFd, SimTypeLongLong
from .. import SIM_PROCEDURES as P
from . import SimSyscallLibrary

_l = logging.getLogger(__name__)


lib = SimSyscallLibrary()
lib.set_library_names('linux')
lib.add_all_from_dict(P['linux_kernel'])

lib.add('open', P['posix']['open'])
lib.add('read', P['posix']['read'])
lib.add('write', P['posix']['write'])
lib.add('close', P['posix']['close'])
lib.add('exit', P['libc']['exit'])
lib.add('sigaction', P['posix']['sigaction'])
lib.add('dup', P['posix']['dup'])
lib.add('pread64', P['posix']['pread64'])
lib.add('pwrite64', P['posix']['pwrite64'])
lib.add('socket', P['posix']['socket'])
lib.add('recvfrom', P['posix']['recvfrom'])
lib.add('select', P['posix']['select'])
lib.add('chroot', P['posix']['chroot'])

lib.add_alias('exit', 'exit_group')
lib.add_alias('getuid', 'geteuid')
lib.add_alias('getgid', 'getegid')


# syscall prototypes

# angr.utils.library.register_kernel_types()
# print(angr.utils.library.cprotos2py(angr.procedures.definitions.linux_kernel._amd64_c_decls, fd_spots=angr.procedures.definitions.linux_kernel._amd64_fd_spots, remove_sys_prefix=True))
_base_syscall_decls = \
    {
        # long sys_time(time_t *tloc);
        'time': SimTypeFunction([SimTypePointer(SimTypeLong(signed=True, label="time_t"), offset=0)], SimTypeLong(signed=True), arg_names=["tloc"]),
        # long sys_stime(time_t *tptr);
        'stime': SimTypeFunction([SimTypePointer(SimTypeLong(signed=True, label="time_t"), offset=0)], SimTypeLong(signed=True), arg_names=["tptr"]),
        # long sys_gettimeofday(struct timeval *tv, struct timezone *tz);
        'gettimeofday': SimTypeFunction([SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_usec": SimTypeLong(signed=True)}, name="timeval", pack=False, align=None), offset=0), SimTypePointer(SimStruct({}, name="timezone", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["tv", "tz"]),
        # long sys_settimeofday(struct timeval *tv, struct timezone *tz);
        'settimeofday': SimTypeFunction([SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_usec": SimTypeLong(signed=True)}, name="timeval", pack=False, align=None), offset=0), SimTypePointer(SimStruct({}, name="timezone", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["tv", "tz"]),
        # long sys_adjtimex(struct timex *txc_p);
        'adjtimex': SimTypeFunction([SimTypePointer(SimStruct({}, name="timex", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["txc_p"]),
        # long sys_times(struct tms *tbuf);
        'times': SimTypeFunction([SimTypePointer(SimStruct({}, name="tms", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["tbuf"]),
        # long sys_gettid(void);
        'gettid': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_nanosleep(struct timespec *rqtp, struct timespec *rmtp);
        'nanosleep': SimTypeFunction([SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["rqtp", "rmtp"]),
        # long sys_alarm(unsigned int seconds);
        'alarm': SimTypeFunction([SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["seconds"]),
        # long sys_getpid(void);
        'getpid': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_getppid(void);
        'getppid': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_getuid(void);
        'getuid': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_geteuid(void);
        'geteuid': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_getgid(void);
        'getgid': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_getegid(void);
        'getegid': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_getresuid(uid_t *ruid, uid_t *euid, uid_t *suid);
        'getresuid': SimTypeFunction([SimTypePointer(SimTypeInt(signed=False, label="uid_t"), offset=0), SimTypePointer(SimTypeInt(signed=False, label="uid_t"), offset=0), SimTypePointer(SimTypeInt(signed=False, label="uid_t"), offset=0)], SimTypeLong(signed=True), arg_names=["ruid", "euid", "suid"]),
        # long sys_getresgid(gid_t *rgid, gid_t *egid, gid_t *sgid);
        'getresgid': SimTypeFunction([SimTypePointer(SimTypeInt(signed=False, label="gid_t"), offset=0), SimTypePointer(SimTypeInt(signed=False, label="gid_t"), offset=0), SimTypePointer(SimTypeInt(signed=False, label="gid_t"), offset=0)], SimTypeLong(signed=True), arg_names=["rgid", "egid", "sgid"]),
        # long sys_getpgid(pid_t pid);
        'getpgid': SimTypeFunction([SimTypeInt(signed=True, label="pid_t")], SimTypeLong(signed=True), arg_names=["pid"]),
        # long sys_getpgrp(void);
        'getpgrp': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_getsid(pid_t pid);
        'getsid': SimTypeFunction([SimTypeInt(signed=True, label="pid_t")], SimTypeLong(signed=True), arg_names=["pid"]),
        # long sys_getgroups(int gidsetsize, gid_t *grouplist);
        'getgroups': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypeInt(signed=False, label="gid_t"), offset=0)], SimTypeLong(signed=True), arg_names=["gidsetsize", "grouplist"]),
        # long sys_setregid(gid_t rgid, gid_t egid);
        'setregid': SimTypeFunction([SimTypeInt(signed=False, label="gid_t"), SimTypeInt(signed=False, label="gid_t")], SimTypeLong(signed=True), arg_names=["rgid", "egid"]),
        # long sys_setgid(gid_t gid);
        'setgid': SimTypeFunction([SimTypeInt(signed=False, label="gid_t")], SimTypeLong(signed=True), arg_names=["gid"]),
        # long sys_setreuid(uid_t ruid, uid_t euid);
        'setreuid': SimTypeFunction([SimTypeInt(signed=False, label="uid_t"), SimTypeInt(signed=False, label="uid_t")], SimTypeLong(signed=True), arg_names=["ruid", "euid"]),
        # long sys_setuid(uid_t uid);
        'setuid': SimTypeFunction([SimTypeInt(signed=False, label="uid_t")], SimTypeLong(signed=True), arg_names=["uid"]),
        # long sys_setresuid(uid_t ruid, uid_t euid, uid_t suid);
        'setresuid': SimTypeFunction([SimTypeInt(signed=False, label="uid_t"), SimTypeInt(signed=False, label="uid_t"), SimTypeInt(signed=False, label="uid_t")], SimTypeLong(signed=True), arg_names=["ruid", "euid", "suid"]),
        # long sys_setresgid(gid_t rgid, gid_t egid, gid_t sgid);
        'setresgid': SimTypeFunction([SimTypeInt(signed=False, label="gid_t"), SimTypeInt(signed=False, label="gid_t"), SimTypeInt(signed=False, label="gid_t")], SimTypeLong(signed=True), arg_names=["rgid", "egid", "sgid"]),
        # long sys_setfsuid(uid_t uid);
        'setfsuid': SimTypeFunction([SimTypeInt(signed=False, label="uid_t")], SimTypeLong(signed=True), arg_names=["uid"]),
        # long sys_setfsgid(gid_t gid);
        'setfsgid': SimTypeFunction([SimTypeInt(signed=False, label="gid_t")], SimTypeLong(signed=True), arg_names=["gid"]),
        # long sys_setpgid(pid_t pid, pid_t pgid);
        'setpgid': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypeInt(signed=True, label="pid_t")], SimTypeLong(signed=True), arg_names=["pid", "pgid"]),
        # long sys_setsid(void);
        'setsid': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_setgroups(int gidsetsize, gid_t *grouplist);
        'setgroups': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypeInt(signed=False, label="gid_t"), offset=0)], SimTypeLong(signed=True), arg_names=["gidsetsize", "grouplist"]),
        # long sys_acct(const char *name);
        'acct': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["name"]),
        # long sys_capget(cap_user_header_t header, cap_user_data_t dataptr);
        'capget': None,
        # long sys_capset(cap_user_header_t header, const cap_user_data_t data);
        'capset': None,
        # long sys_personality(unsigned int personality);
        'personality': SimTypeFunction([SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["personality"]),
        # long sys_sigpending(old_sigset_t *set);
        'sigpending': None,
        # long sys_sigprocmask(int how, old_sigset_t *set, old_sigset_t *oset);
        'sigprocmask': None,
        # long sys_sigaltstack(const struct sigaltstack *uss, struct sigaltstack *uoss);
        'sigaltstack': SimTypeFunction([SimTypePointer(SimStruct({}, name="sigaltstack", pack=False, align=None), offset=0), SimTypePointer(SimStruct({}, name="sigaltstack", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["uss", "uoss"]),
        # long sys_getitimer(int which, struct itimerval *value);
        'getitimer': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="itimerval", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["which", "value"]),
        # long sys_setitimer(int which, struct itimerval *value, struct itimerval *ovalue);
        'setitimer': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="itimerval", pack=False, align=None), offset=0), SimTypePointer(SimStruct({}, name="itimerval", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["which", "value", "ovalue"]),
        # long sys_timer_create(clockid_t which_clock, struct sigevent *timer_event_spec, timer_t * created_timer_id);
        'timer_create': SimTypeFunction([SimTypeInt(signed=True, label="clockid_t"), SimTypePointer(SimStruct({}, name="sigevent", pack=False, align=None), offset=0), SimTypePointer(SimTypePointer(SimTypeBottom(label="BOT"), label="timer_t", offset=0), offset=0)], SimTypeLong(signed=True), arg_names=["which_clock", "timer_event_spec", "created_timer_id"]),
        # long sys_timer_gettime(timer_t timer_id, struct itimerspec *setting);
        'timer_gettime': SimTypeFunction([SimTypePointer(SimTypeBottom(label="BOT"), label="timer_t", offset=0), SimTypePointer(SimStruct({}, name="itimerspec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["timer_id", "setting"]),
        # long sys_timer_getoverrun(timer_t timer_id);
        'timer_getoverrun': SimTypeFunction([SimTypePointer(SimTypeBottom(label="BOT"), label="timer_t", offset=0)], SimTypeLong(signed=True), arg_names=["timer_id"]),
        # long sys_timer_settime(timer_t timer_id, int flags, const struct itimerspec *new_setting, struct itimerspec *old_setting);
        'timer_settime': SimTypeFunction([SimTypePointer(SimTypeBottom(label="BOT"), label="timer_t", offset=0), SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="itimerspec", pack=False, align=None), offset=0), SimTypePointer(SimStruct({}, name="itimerspec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["timer_id", "flags", "new_setting", "old_setting"]),
        # long sys_timer_delete(timer_t timer_id);
        'timer_delete': SimTypeFunction([SimTypePointer(SimTypeBottom(label="BOT"), label="timer_t", offset=0)], SimTypeLong(signed=True), arg_names=["timer_id"]),
        # long sys_clock_settime(clockid_t which_clock, const struct timespec *tp);
        'clock_settime': SimTypeFunction([SimTypeInt(signed=True, label="clockid_t"), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["which_clock", "tp"]),
        # long sys_clock_gettime(clockid_t which_clock, struct timespec *tp);
        'clock_gettime': SimTypeFunction([SimTypeInt(signed=True, label="clockid_t"), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["which_clock", "tp"]),
        # long sys_clock_adjtime(clockid_t which_clock, struct timex *tx);
        'clock_adjtime': SimTypeFunction([SimTypeInt(signed=True, label="clockid_t"), SimTypePointer(SimStruct({}, name="timex", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["which_clock", "tx"]),
        # long sys_clock_getres(clockid_t which_clock, struct timespec *tp);
        'clock_getres': SimTypeFunction([SimTypeInt(signed=True, label="clockid_t"), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["which_clock", "tp"]),
        # long sys_clock_nanosleep(clockid_t which_clock, int flags, const struct timespec *rqtp, struct timespec *rmtp);
        'clock_nanosleep': SimTypeFunction([SimTypeInt(signed=True, label="clockid_t"), SimTypeInt(signed=True), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["which_clock", "flags", "rqtp", "rmtp"]),
        # long sys_nice(int increment);
        'nice': SimTypeFunction([SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["increment"]),
        # long sys_sched_setscheduler(pid_t pid, int policy, struct sched_param *param);
        'sched_setscheduler': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="sched_param", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["pid", "policy", "param"]),
        # long sys_sched_setparam(pid_t pid, struct sched_param *param);
        'sched_setparam': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypePointer(SimStruct({}, name="sched_param", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["pid", "param"]),
        # long sys_sched_setattr(pid_t pid, struct sched_attr *attr, unsigned int flags);
        'sched_setattr': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypePointer(SimStruct({}, name="sched_attr", pack=False, align=None), offset=0), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["pid", "attr", "flags"]),
        # long sys_sched_getscheduler(pid_t pid);
        'sched_getscheduler': SimTypeFunction([SimTypeInt(signed=True, label="pid_t")], SimTypeLong(signed=True), arg_names=["pid"]),
        # long sys_sched_getparam(pid_t pid, struct sched_param *param);
        'sched_getparam': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypePointer(SimStruct({}, name="sched_param", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["pid", "param"]),
        # long sys_sched_getattr(pid_t pid, struct sched_attr *attr, unsigned int size, unsigned int flags);
        'sched_getattr': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypePointer(SimStruct({}, name="sched_attr", pack=False, align=None), offset=0), SimTypeInt(signed=False), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["pid", "attr", "size", "flags"]),
        # long sys_sched_setaffinity(pid_t pid, unsigned int len, unsigned long *user_mask_ptr);
        'sched_setaffinity': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypeInt(signed=False), SimTypePointer(SimTypeLong(signed=False), offset=0)], SimTypeLong(signed=True), arg_names=["pid", "len", "user_mask_ptr"]),
        # long sys_sched_getaffinity(pid_t pid, unsigned int len, unsigned long *user_mask_ptr);
        'sched_getaffinity': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypeInt(signed=False), SimTypePointer(SimTypeLong(signed=False), offset=0)], SimTypeLong(signed=True), arg_names=["pid", "len", "user_mask_ptr"]),
        # long sys_sched_yield(void);
        'sched_yield': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_sched_get_priority_max(int policy);
        'sched_get_priority_max': SimTypeFunction([SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["policy"]),
        # long sys_sched_get_priority_min(int policy);
        'sched_get_priority_min': SimTypeFunction([SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["policy"]),
        # long sys_sched_rr_get_interval(pid_t pid, struct timespec *interval);
        'sched_rr_get_interval': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["pid", "interval"]),
        # long sys_setpriority(int which, int who, int niceval);
        'setpriority': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["which", "who", "niceval"]),
        # long sys_getpriority(int which, int who);
        'getpriority': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["which", "who"]),
        # long sys_shutdown(int, int);
        'shutdown': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["None", "None"]),
        # long sys_reboot(int magic1, int magic2, unsigned int cmd, void *arg);
        'reboot': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypeInt(signed=False), SimTypePointer(SimTypeBottom(label="void"), offset=0)], SimTypeLong(signed=True), arg_names=["magic1", "magic2", "cmd", "arg"]),
        # long sys_restart_syscall(void);
        'restart_syscall': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_kexec_load(unsigned long entry, unsigned long nr_segments, struct kexec_segment *segments, unsigned long flags);
        'kexec_load': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypePointer(SimStruct({}, name="kexec_segment", pack=False, align=None), offset=0), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["entry", "nr_segments", "segments", "flags"]),
        # long sys_kexec_file_load(int kernel_fd, int initrd_fd, unsigned long cmdline_len, const char *cmdline_ptr, unsigned long flags);
        'kexec_file_load': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypeLong(signed=False), SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["kernel_fd", "initrd_fd", "cmdline_len", "cmdline_ptr", "flags"]),
        # long sys_exit(int error_code);
        'exit': SimTypeFunction([SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["error_code"]),
        # long sys_exit_group(int error_code);
        'exit_group': SimTypeFunction([SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["error_code"]),
        # long sys_wait4(pid_t pid, int *stat_addr, int options, struct rusage *ru);
        'wait4': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypePointer(SimTypeInt(signed=True), offset=0), SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="rusage", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["pid", "stat_addr", "options", "ru"]),
        # long sys_waitid(int which, pid_t pid, struct siginfo *infop, int options, struct rusage *ru);
        'waitid': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True, label="pid_t"), SimTypePointer(SimStruct({}, name="siginfo", pack=False, align=None), offset=0), SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="rusage", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["which", "pid", "infop", "options", "ru"]),
        # long sys_waitpid(pid_t pid, int *stat_addr, int options);
        'waitpid': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypePointer(SimTypeInt(signed=True), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["pid", "stat_addr", "options"]),
        # long sys_set_tid_address(int *tidptr);
        'set_tid_address': SimTypeFunction([SimTypePointer(SimTypeInt(signed=True), offset=0)], SimTypeLong(signed=True), arg_names=["tidptr"]),
        # long sys_futex(u32 *uaddr, int op, u32 val, struct timespec *utime, u32 *uaddr2, u32 val3);
        'futex': SimTypeFunction([SimTypePointer(SimTypeInt(signed=False, label="u32"), offset=0), SimTypeInt(signed=True), SimTypeInt(signed=False, label="u32"), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0), SimTypePointer(SimTypeInt(signed=False, label="u32"), offset=0), SimTypeInt(signed=False, label="u32")], SimTypeLong(signed=True), arg_names=["uaddr", "op", "val", "utime", "uaddr2", "val3"]),
        # long sys_init_module(void *umod, unsigned long len, const char *uargs);
        'init_module': SimTypeFunction([SimTypePointer(SimTypeBottom(label="void"), offset=0), SimTypeLong(signed=False), SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["umod", "len", "uargs"]),
        # long sys_delete_module(const char *name_user, unsigned int flags);
        'delete_module': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["name_user", "flags"]),
        # long sys_rt_sigsuspend(sigset_t *unewset, size_t sigsetsize);
        'rt_sigsuspend': SimTypeFunction([SimTypePointer(SimTypeInt(signed=True, label="sigset_t"), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["unewset", "sigsetsize"]),
        # long sys_sigaction(int, const struct old_sigaction *, struct old_sigaction *);
        'sigaction': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="old_sigaction", pack=False, align=None), offset=0), SimTypePointer(SimStruct({}, name="old_sigaction", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["None", "None", "None"]),
        # long sys_rt_sigaction(int, const struct sigaction *, struct sigaction *, size_t);
        'rt_sigaction': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="sigaction", pack=False, align=None), offset=0), SimTypePointer(SimStruct({}, name="sigaction", pack=False, align=None), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["None", "None", "None", "None"]),
        # long sys_rt_sigprocmask(int how, sigset_t *set, sigset_t *oset, size_t sigsetsize);
        'rt_sigprocmask': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypeInt(signed=True, label="sigset_t"), offset=0), SimTypePointer(SimTypeInt(signed=True, label="sigset_t"), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["how", "set", "oset", "sigsetsize"]),
        # long sys_rt_sigpending(sigset_t *set, size_t sigsetsize);
        'rt_sigpending': SimTypeFunction([SimTypePointer(SimTypeInt(signed=True, label="sigset_t"), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["set", "sigsetsize"]),
        # long sys_rt_sigtimedwait(const sigset_t *uthese, siginfo_t *uinfo, const struct timespec *uts, size_t sigsetsize);
        'rt_sigtimedwait': None,
        # long sys_rt_tgsigqueueinfo(pid_t tgid, pid_t pid, int sig, siginfo_t *uinfo);
        'rt_tgsigqueueinfo': None,
        # long sys_kill(pid_t pid, int sig);
        'kill': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["pid", "sig"]),
        # long sys_tgkill(pid_t tgid, pid_t pid, int sig);
        'tgkill': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypeInt(signed=True, label="pid_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["tgid", "pid", "sig"]),
        # long sys_tkill(pid_t pid, int sig);
        'tkill': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["pid", "sig"]),
        # long sys_rt_sigqueueinfo(pid_t pid, int sig, siginfo_t *uinfo);
        'rt_sigqueueinfo': None,
        # long sys_sgetmask(void);
        'sgetmask': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_ssetmask(int newmask);
        'ssetmask': SimTypeFunction([SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["newmask"]),
        # long sys_signal(int sig, __sighandler_t handler);
        'signal': None,
        # long sys_pause(void);
        'pause': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_sync(void);
        'sync': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_fsync(unsigned int fd);
        'fsync': SimTypeFunction([SimTypeFd()], SimTypeLong(signed=True), arg_names=["fd"]),
        # long sys_fdatasync(unsigned int fd);
        'fdatasync': SimTypeFunction([SimTypeFd()], SimTypeLong(signed=True), arg_names=["fd"]),
        # long sys_bdflush(int func, long data);
        'bdflush': SimTypeFunction([SimTypeInt(signed=True), SimTypeLong(signed=True)], SimTypeLong(signed=True), arg_names=["func", "data"]),
        # long sys_mount(char *dev_name, char *dir_name, char *type, unsigned long flags, void *data);
        'mount': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False), SimTypePointer(SimTypeBottom(label="void"), offset=0)], SimTypeLong(signed=True), arg_names=["dev_name", "dir_name", "type", "flags", "data"]),
        # long sys_umount(char *name, int flags);
        'umount': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["name", "flags"]),
        # long sys_oldumount(char *name);
        'oldumount': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["name"]),
        # long sys_truncate(const char *path, long length);
        'truncate': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=True)], SimTypeLong(signed=True), arg_names=["path", "length"]),
        # long sys_ftruncate(unsigned int fd, unsigned long length);
        'ftruncate': SimTypeFunction([SimTypeFd(), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "length"]),
        # long sys_stat(const char *filename, struct __old_kernel_stat *statbuf);
        'stat': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({}, name="__old_kernel_stat", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["filename", "statbuf"]),
        # long sys_statfs(const char * path, struct statfs *buf);
        'statfs': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({}, name="statfs", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["path", "buf"]),
        # long sys_statfs64(const char *path, size_t sz, struct statfs64 *buf);
        'statfs64': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypePointer(SimStruct({}, name="statfs64", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["path", "sz", "buf"]),
        # long sys_fstatfs(unsigned int fd, struct statfs *buf);
        'fstatfs': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="statfs", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["fd", "buf"]),
        # long sys_fstatfs64(unsigned int fd, size_t sz, struct statfs64 *buf);
        'fstatfs64': SimTypeFunction([SimTypeInt(signed=False), SimTypeLong(signed=False, label="size_t"), SimTypePointer(SimStruct({}, name="statfs64", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["fd", "sz", "buf"]),
        # long sys_lstat(const char *filename, struct __old_kernel_stat *statbuf);
        'lstat': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({}, name="__old_kernel_stat", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["filename", "statbuf"]),
        # long sys_fstat(unsigned int fd, struct __old_kernel_stat *statbuf);
        'fstat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="__old_kernel_stat", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["fd", "statbuf"]),
        # long sys_newstat(const char *filename, struct stat *statbuf);
        'newstat': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({}, name="stat", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["filename", "statbuf"]),
        # long sys_newlstat(const char *filename, struct stat *statbuf);
        'newlstat': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({}, name="stat", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["filename", "statbuf"]),
        # long sys_newfstat(unsigned int fd, struct stat *statbuf);
        'newfstat': SimTypeFunction([SimTypeInt(signed=False), SimTypePointer(SimStruct({}, name="stat", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["fd", "statbuf"]),
        # long sys_ustat(unsigned dev, struct ustat *ubuf);
        'ustat': SimTypeFunction([SimTypeInt(signed=False), SimTypePointer(SimStruct({}, name="ustat", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["dev", "ubuf"]),
        # long sys_stat64(const char *filename, struct stat64 *statbuf);
        'stat64': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({}, name="stat64", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["filename", "statbuf"]),
        # long sys_fstat64(unsigned long fd, struct stat64 *statbuf);
        'fstat64': SimTypeFunction([SimTypeLong(signed=False), SimTypePointer(SimStruct({}, name="stat64", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["fd", "statbuf"]),
        # long sys_lstat64(const char *filename, struct stat64 *statbuf);
        'lstat64': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({}, name="stat64", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["filename", "statbuf"]),
        # long sys_fstatat64(int dfd, const char *filename, struct stat64 *statbuf, int flag);
        'fstatat64': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({}, name="stat64", pack=False, align=None), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["dfd", "filename", "statbuf", "flag"]),
        # long sys_truncate64(const char *path, loff_t length);
        'truncate64': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeLongLong(signed=True, label="loff_t")], SimTypeLong(signed=True), arg_names=["path", "length"]),
        # long sys_ftruncate64(unsigned int fd, loff_t length);
        'ftruncate64': SimTypeFunction([SimTypeInt(signed=False), SimTypeLongLong(signed=True, label="loff_t")], SimTypeLong(signed=True), arg_names=["fd", "length"]),
        # long sys_setxattr(const char *path, const char *name, const void *value, size_t size, int flags);
        'setxattr': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeBottom(label="void"), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["path", "name", "value", "size", "flags"]),
        # long sys_lsetxattr(const char *path, const char *name, const void *value, size_t size, int flags);
        'lsetxattr': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeBottom(label="void"), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["path", "name", "value", "size", "flags"]),
        # long sys_fsetxattr(int fd, const char *name, const void *value, size_t size, int flags);
        'fsetxattr': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeBottom(label="void"), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["fd", "name", "value", "size", "flags"]),
        # long sys_getxattr(const char *path, const char *name, void *value, size_t size);
        'getxattr': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeBottom(label="void"), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["path", "name", "value", "size"]),
        # long sys_lgetxattr(const char *path, const char *name, void *value, size_t size);
        'lgetxattr': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeBottom(label="void"), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["path", "name", "value", "size"]),
        # long sys_fgetxattr(int fd, const char *name, void *value, size_t size);
        'fgetxattr': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeBottom(label="void"), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["fd", "name", "value", "size"]),
        # long sys_listxattr(const char *path, char *list, size_t size);
        'listxattr': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["path", "list", "size"]),
        # long sys_llistxattr(const char *path, char *list, size_t size);
        'llistxattr': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["path", "list", "size"]),
        # long sys_flistxattr(int fd, char *list, size_t size);
        'flistxattr': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["fd", "list", "size"]),
        # long sys_removexattr(const char *path, const char *name);
        'removexattr': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["path", "name"]),
        # long sys_lremovexattr(const char *path, const char *name);
        'lremovexattr': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["path", "name"]),
        # long sys_fremovexattr(int fd, const char *name);
        'fremovexattr': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["fd", "name"]),
        # long sys_brk(unsigned long brk);
        'brk': SimTypeFunction([SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["brk"]),
        # long sys_mprotect(unsigned long start, size_t len, unsigned long prot);
        'mprotect': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False, label="size_t"), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["start", "len", "prot"]),
        # long sys_mremap(unsigned long addr, unsigned long old_len, unsigned long new_len, unsigned long flags, unsigned long new_addr);
        'mremap': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["addr", "old_len", "new_len", "flags", "new_addr"]),
        # long sys_remap_file_pages(unsigned long start, unsigned long size, unsigned long prot, unsigned long pgoff, unsigned long flags);
        'remap_file_pages': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["start", "size", "prot", "pgoff", "flags"]),
        # long sys_msync(unsigned long start, size_t len, int flags);
        'msync': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["start", "len", "flags"]),
        # long sys_fadvise64(int fd, loff_t offset, size_t len, int advice);
        'fadvise64': SimTypeFunction([SimTypeFd(), SimTypeLongLong(signed=True, label="loff_t"), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["fd", "offset", "len", "advice"]),
        # long sys_fadvise64_64(int fd, loff_t offset, loff_t len, int advice);
        'fadvise64_64': SimTypeFunction([SimTypeInt(signed=True), SimTypeLongLong(signed=True, label="loff_t"), SimTypeLongLong(signed=True, label="loff_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["fd", "offset", "len", "advice"]),
        # long sys_munmap(unsigned long addr, size_t len);
        'munmap': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["addr", "len"]),
        # long sys_mlock(unsigned long start, size_t len);
        'mlock': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["start", "len"]),
        # long sys_munlock(unsigned long start, size_t len);
        'munlock': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["start", "len"]),
        # long sys_mlockall(int flags);
        'mlockall': SimTypeFunction([SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["flags"]),
        # long sys_munlockall(void);
        'munlockall': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_madvise(unsigned long start, size_t len, int behavior);
        'madvise': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["start", "len", "behavior"]),
        # long sys_mincore(unsigned long start, size_t len, unsigned char * vec);
        'mincore': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False, label="size_t"), SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["start", "len", "vec"]),
        # long sys_pivot_root(const char *new_root, const char *put_old);
        'pivot_root': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["new_root", "put_old"]),
        # long sys_chroot(const char *filename);
        'chroot': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["filename"]),
        # long sys_mknod(const char *filename, umode_t mode, unsigned dev);
        'mknod': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False, label="umode_t"), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["filename", "mode", "dev"]),
        # long sys_link(const char *oldname, const char *newname);
        'link': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["oldname", "newname"]),
        # long sys_symlink(const char *old, const char *new);
        'symlink': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["old", "new"]),
        # long sys_unlink(const char *pathname);
        'unlink': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["pathname"]),
        # long sys_rename(const char *oldname, const char *newname);
        'rename': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["oldname", "newname"]),
        # long sys_chmod(const char *filename, umode_t mode);
        'chmod': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False, label="umode_t")], SimTypeLong(signed=True), arg_names=["filename", "mode"]),
        # long sys_fchmod(unsigned int fd, umode_t mode);
        'fchmod': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=False, label="umode_t")], SimTypeLong(signed=True), arg_names=["fd", "mode"]),
        # long sys_fcntl(unsigned int fd, unsigned int cmd, unsigned long arg);
        'fcntl': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=False), SimTypeInt()], SimTypeInt(), arg_names=["fd", "cmd", "arg"]),
        # long sys_fcntl64(unsigned int fd, unsigned int cmd, unsigned long arg);
        'fcntl64': SimTypeFunction([SimTypeInt(signed=False), SimTypeInt(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "cmd", "arg"]),
        # long sys_pipe(int *fildes);
        'pipe': SimTypeFunction([SimTypePointer(SimTypeInt(signed=True), offset=0)], SimTypeLong(signed=True), arg_names=["fildes"]),
        # long sys_pipe2(int *fildes, int flags);
        'pipe2': SimTypeFunction([SimTypePointer(SimTypeInt(signed=True), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["fildes", "flags"]),
        # long sys_dup(unsigned int fildes);
        'dup': SimTypeFunction([SimTypeFd()], SimTypeFd(), arg_names=["fildes"]),
        # long sys_dup2(unsigned int oldfd, unsigned int newfd);
        'dup2': SimTypeFunction([SimTypeFd(), SimTypeFd()], SimTypeFd(), arg_names=["oldfd", "newfd"]),
        # long sys_dup3(unsigned int oldfd, unsigned int newfd, int flags);
        'dup3': SimTypeFunction([SimTypeFd(), SimTypeFd(), SimTypeInt(signed=True)], SimTypeFd(), arg_names=["oldfd", "newfd", "flags"]),
        # long sys_ioperm(unsigned long from, unsigned long num, int on);
        'ioperm': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["from", "num", "on"]),
        # long sys_ioctl(unsigned int fd, unsigned int cmd, unsigned long arg);
        'ioctl': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=False), SimTypeInt()], SimTypeInt(), arg_names=["fd", "cmd", "arg"]),
        # long sys_flock(unsigned int fd, unsigned int cmd);
        'flock': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "cmd"]),
        # long sys_io_setup(unsigned nr_reqs, aio_context_t *ctx);
        'io_setup': None,
        # long sys_io_destroy(aio_context_t ctx);
        'io_destroy': None,
        # long sys_io_getevents(aio_context_t ctx_id, long min_nr, long nr, struct io_event *events, struct timespec *timeout);
        'io_getevents': None,
        # long sys_io_submit(aio_context_t, long, struct iocb * *);
        'io_submit': None,
        # long sys_io_cancel(aio_context_t ctx_id, struct iocb *iocb, struct io_event *result);
        'io_cancel': None,
        # long sys_sendfile(int out_fd, int in_fd, off_t *offset, size_t count);
        'sendfile': SimTypeFunction([SimTypeFd(), SimTypeFd(), SimTypePointer(SimTypeLong(signed=False, label="off_t"), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["out_fd", "in_fd", "offset", "count"]),
        # long sys_sendfile64(int out_fd, int in_fd, loff_t *offset, size_t count);
        'sendfile64': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypePointer(SimTypeLongLong(signed=True, label="loff_t"), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["out_fd", "in_fd", "offset", "count"]),
        # long sys_readlink(const char *path, char *buf, int bufsiz);
        'readlink': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["path", "buf", "bufsiz"]),
        # long sys_creat(const char *pathname, umode_t mode);
        'creat': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False, label="umode_t")], SimTypeFd(), arg_names=["pathname", "mode"]),
        # long sys_open(const char *filename, int flags, umode_t mode);
        'open': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True), SimTypeInt(signed=False, label="umode_t")], SimTypeFd(), arg_names=["filename", "flags", "mode"]),
        # long sys_close(unsigned int fd);
        'close': SimTypeFunction([SimTypeFd()], SimTypeLong(signed=True), arg_names=["fd"]),
        # long sys_access(const char *filename, int mode);
        'access': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["filename", "mode"]),
        # long sys_vhangup(void);
        'vhangup': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_chown(const char *filename, uid_t user, gid_t group);
        'chown': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False, label="uid_t"), SimTypeInt(signed=False, label="gid_t")], SimTypeLong(signed=True), arg_names=["filename", "user", "group"]),
        # long sys_lchown(const char *filename, uid_t user, gid_t group);
        'lchown': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False, label="uid_t"), SimTypeInt(signed=False, label="gid_t")], SimTypeLong(signed=True), arg_names=["filename", "user", "group"]),
        # long sys_fchown(unsigned int fd, uid_t user, gid_t group);
        'fchown': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=False, label="uid_t"), SimTypeInt(signed=False, label="gid_t")], SimTypeLong(signed=True), arg_names=["fd", "user", "group"]),
        # long sys_utime(char *filename, struct utimbuf *times);
        'utime': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({}, name="utimbuf", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["filename", "times"]),
        # long sys_utimes(char *filename, struct timeval *utimes);
        'utimes': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_usec": SimTypeLong(signed=True)}, name="timeval", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["filename", "utimes"]),
        # long sys_lseek(unsigned int fd, off_t offset, unsigned int whence);
        'lseek': SimTypeFunction([SimTypeFd(), SimTypeLong(signed=False, label="off_t"), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "offset", "whence"]),
        # long sys_llseek(unsigned int fd, unsigned long offset_high, unsigned long offset_low, loff_t *result, unsigned int whence);
        'llseek': SimTypeFunction([SimTypeInt(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypePointer(SimTypeLongLong(signed=True, label="loff_t"), offset=0), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "offset_high", "offset_low", "result", "whence"]),
        # long sys_read(unsigned int fd, char *buf, size_t count);
        'read': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["fd", "buf", "count"]),
        # long sys_readahead(int fd, loff_t offset, size_t count);
        'readahead': SimTypeFunction([SimTypeFd(), SimTypeLongLong(signed=True, label="loff_t"), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["fd", "offset", "count"]),
        # long sys_readv(unsigned long fd, const struct iovec *vec, unsigned long vlen);
        'readv': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({"iov_base": SimTypePointer(SimTypeBottom(label="void"), offset=0), "iov_len": SimTypeLong(signed=False, label="size_t")}, name="iovec", pack=False, align=None), offset=0), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "vec", "vlen"]),
        # long sys_write(unsigned int fd, const char *buf, size_t count);
        'write': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["fd", "buf", "count"]),
        # long sys_writev(unsigned long fd, const struct iovec *vec, unsigned long vlen);
        'writev': SimTypeFunction([SimTypeLong(signed=False), SimTypePointer(SimStruct({"iov_base": SimTypePointer(SimTypeBottom(label="void"), offset=0), "iov_len": SimTypeLong(signed=False, label="size_t")}, name="iovec", pack=False, align=None), offset=0), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "vec", "vlen"]),
        # long sys_pread64(unsigned int fd, char *buf, size_t count, loff_t pos);
        'pread64': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeLongLong(signed=True, label="loff_t")], SimTypeLong(signed=True), arg_names=["fd", "buf", "count", "pos"]),
        # long sys_pwrite64(unsigned int fd, const char *buf, size_t count, loff_t pos);
        'pwrite64': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeLongLong(signed=True, label="loff_t")], SimTypeLong(signed=True), arg_names=["fd", "buf", "count", "pos"]),
        # long sys_preadv(unsigned long fd, const struct iovec *vec, unsigned long vlen, unsigned long pos_l, unsigned long pos_h);
        'preadv': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({"iov_base": SimTypePointer(SimTypeBottom(label="void"), offset=0), "iov_len": SimTypeLong(signed=False, label="size_t")}, name="iovec", pack=False, align=None), offset=0), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "vec", "vlen", "pos_l", "pos_h"]),
        # long sys_preadv2(unsigned long fd, const struct iovec *vec, unsigned long vlen, unsigned long pos_l, unsigned long pos_h, int flags);
        'preadv2': SimTypeFunction([SimTypeLong(signed=False), SimTypePointer(SimStruct({"iov_base": SimTypePointer(SimTypeBottom(label="void"), offset=0), "iov_len": SimTypeLong(signed=False, label="size_t")}, name="iovec", pack=False, align=None), offset=0), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["fd", "vec", "vlen", "pos_l", "pos_h", "flags"]),
        # long sys_pwritev(unsigned long fd, const struct iovec *vec, unsigned long vlen, unsigned long pos_l, unsigned long pos_h);
        'pwritev': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({"iov_base": SimTypePointer(SimTypeBottom(label="void"), offset=0), "iov_len": SimTypeLong(signed=False, label="size_t")}, name="iovec", pack=False, align=None), offset=0), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "vec", "vlen", "pos_l", "pos_h"]),
        # long sys_pwritev2(unsigned long fd, const struct iovec *vec, unsigned long vlen, unsigned long pos_l, unsigned long pos_h, int flags);
        'pwritev2': SimTypeFunction([SimTypeLong(signed=False), SimTypePointer(SimStruct({"iov_base": SimTypePointer(SimTypeBottom(label="void"), offset=0), "iov_len": SimTypeLong(signed=False, label="size_t")}, name="iovec", pack=False, align=None), offset=0), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["fd", "vec", "vlen", "pos_l", "pos_h", "flags"]),
        # long sys_getcwd(char *buf, unsigned long size);
        'getcwd': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["buf", "size"]),
        # long sys_mkdir(const char *pathname, umode_t mode);
        'mkdir': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False, label="umode_t")], SimTypeLong(signed=True), arg_names=["pathname", "mode"]),
        # long sys_chdir(const char *filename);
        'chdir': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["filename"]),
        # long sys_fchdir(unsigned int fd);
        'fchdir': SimTypeFunction([SimTypeFd()], SimTypeLong(signed=True), arg_names=["fd"]),
        # long sys_rmdir(const char *pathname);
        'rmdir': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["pathname"]),
        # long sys_lookup_dcookie(u64 cookie64, char *buf, size_t len);
        'lookup_dcookie': SimTypeFunction([SimTypeLongLong(signed=False, label="u64"), SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["cookie64", "buf", "len"]),
        # long sys_quotactl(unsigned int cmd, const char *special, qid_t id, void *addr);
        'quotactl': SimTypeFunction([SimTypeInt(signed=False), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True, label="qid_t"), SimTypePointer(SimTypeBottom(label="void"), offset=0)], SimTypeLong(signed=True), arg_names=["cmd", "special", "id", "addr"]),
        # long sys_getdents(unsigned int fd, struct linux_dirent *dirent, unsigned int count);
        'getdents': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="linux_dirent", pack=False, align=None), offset=0), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "dirent", "count"]),
        # long sys_getdents64(unsigned int fd, struct linux_dirent64 *dirent, unsigned int count);
        'getdents64': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="linux_dirent64", pack=False, align=None), offset=0), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "dirent", "count"]),
        # long sys_setsockopt(int fd, int level, int optname, char *optval, int optlen);
        'setsockopt': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["fd", "level", "optname", "optval", "optlen"]),
        # long sys_getsockopt(int fd, int level, int optname, char *optval, int *optlen);
        'getsockopt': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeInt(signed=True), offset=0)], SimTypeLong(signed=True), arg_names=["fd", "level", "optname", "optval", "optlen"]),
        # long sys_bind(int, struct sockaddr *, int);
        'bind': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="sockaddr", pack=False, align=None), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["None", "None", "None"]),
        # long sys_connect(int, struct sockaddr *, int);
        'connect': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="sockaddr", pack=False, align=None), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["None", "None", "None"]),
        # long sys_accept(int, struct sockaddr *, int *);
        'accept': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="sockaddr", pack=False, align=None), offset=0), SimTypePointer(SimTypeInt(signed=True), offset=0)], SimTypeFd(), arg_names=["None", "None", "None"]),
        # long sys_accept4(int, struct sockaddr *, int *, int);
        'accept4': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="sockaddr", pack=False, align=None), offset=0), SimTypePointer(SimTypeInt(signed=True), offset=0), SimTypeInt(signed=True)], SimTypeFd(), arg_names=["None", "None", "None", "None"]),
        # long sys_getsockname(int, struct sockaddr *, int *);
        'getsockname': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="sockaddr", pack=False, align=None), offset=0), SimTypePointer(SimTypeInt(signed=True), offset=0)], SimTypeLong(signed=True), arg_names=["None", "None", "None"]),
        # long sys_getpeername(int, struct sockaddr *, int *);
        'getpeername': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="sockaddr", pack=False, align=None), offset=0), SimTypePointer(SimTypeInt(signed=True), offset=0)], SimTypeLong(signed=True), arg_names=["None", "None", "None"]),
        # long sys_send(int, void *, size_t, unsigned);
        'send': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypeBottom(label="void"), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["None", "None", "None", "None"]),
        # long sys_sendto(int, void *, size_t, unsigned, struct sockaddr *, int);
        'sendto': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeBottom(label="void"), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=False), SimTypePointer(SimStruct({}, name="sockaddr", pack=False, align=None), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["None", "None", "None", "None", "None", "None"]),
        # long sys_sendmsg(int fd, struct user_msghdr *msg, unsigned flags);
        'sendmsg': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="user_msghdr", pack=False, align=None), offset=0), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "msg", "flags"]),
        # long sys_sendmmsg(int fd, struct mmsghdr *msg, unsigned int vlen, unsigned flags);
        'sendmmsg': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="mmsghdr", pack=False, align=None), offset=0), SimTypeInt(signed=False), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "msg", "vlen", "flags"]),
        # long sys_recv(int, void *, size_t, unsigned);
        'recv': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypeBottom(label="void"), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["None", "None", "None", "None"]),
        # long sys_recvfrom(int, void *, size_t, unsigned, struct sockaddr *, int *);
        'recvfrom': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeBottom(label="void"), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=False), SimTypePointer(SimStruct({}, name="sockaddr", pack=False, align=None), offset=0), SimTypePointer(SimTypeInt(signed=True), offset=0)], SimTypeLong(signed=True), arg_names=["None", "None", "None", "None", "None", "None"]),
        # long sys_recvmsg(int fd, struct user_msghdr *msg, unsigned flags);
        'recvmsg': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="user_msghdr", pack=False, align=None), offset=0), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "msg", "flags"]),
        # long sys_recvmmsg(int fd, struct mmsghdr *msg, unsigned int vlen, unsigned flags, struct timespec *timeout);
        'recvmmsg': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="mmsghdr", pack=False, align=None), offset=0), SimTypeInt(signed=False), SimTypeInt(signed=False), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["fd", "msg", "vlen", "flags", "timeout"]),
        # long sys_socket(int, int, int);
        'socket': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypeInt(signed=True)], SimTypeFd(), arg_names=["None", "None", "None"]),
        # long sys_socketpair(int, int, int, int *);
        'socketpair': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypePointer(SimTypeInt(signed=True), offset=0)], SimTypeLong(signed=True), arg_names=["None", "None", "None", "None"]),
        # long sys_socketcall(int call, unsigned long *args);
        'socketcall': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypeLong(signed=False), offset=0)], SimTypeLong(signed=True), arg_names=["call", "args"]),
        # long sys_listen(int, int);
        'listen': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["None", "None"]),
        # long sys_poll(struct pollfd *ufds, unsigned int nfds, int timeout);
        'poll': SimTypeFunction([SimTypePointer(SimStruct({}, name="pollfd", pack=False, align=None), offset=0), SimTypeInt(signed=False), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["ufds", "nfds", "timeout"]),
        # long sys_select(int n, fd_set *inp, fd_set *outp, fd_set *exp, struct timeval *tvp);
        'select': None,
        # long sys_old_select(struct sel_arg_struct *arg);
        'old_select': SimTypeFunction([SimTypePointer(SimStruct({}, name="sel_arg_struct", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["arg"]),
        # long sys_epoll_create(int size);
        'epoll_create': SimTypeFunction([SimTypeInt(signed=True)], SimTypeFd(), arg_names=["size"]),
        # long sys_epoll_create1(int flags);
        'epoll_create1': SimTypeFunction([SimTypeInt(signed=True)], SimTypeFd(), arg_names=["flags"]),
        # long sys_epoll_ctl(int epfd, int op, int fd, struct epoll_event *event);
        'epoll_ctl': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=True), SimTypeFd(), SimTypePointer(SimStruct({}, name="epoll_event", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["epfd", "op", "fd", "event"]),
        # long sys_epoll_wait(int epfd, struct epoll_event *events, int maxevents, int timeout);
        'epoll_wait': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="epoll_event", pack=False, align=None), offset=0), SimTypeInt(signed=True), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["epfd", "events", "maxevents", "timeout"]),
        # long sys_epoll_pwait(int epfd, struct epoll_event *events, int maxevents, int timeout, const sigset_t *sigmask, size_t sigsetsize);
        'epoll_pwait': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="epoll_event", pack=False, align=None), offset=0), SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypePointer(SimTypeInt(signed=True, label="sigset_t"), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["epfd", "events", "maxevents", "timeout", "sigmask", "sigsetsize"]),
        # long sys_gethostname(char *name, int len);
        'gethostname': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["name", "len"]),
        # long sys_sethostname(char *name, int len);
        'sethostname': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["name", "len"]),
        # long sys_setdomainname(char *name, int len);
        'setdomainname': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["name", "len"]),
        # long sys_newuname(struct new_utsname *name);
        'newuname': SimTypeFunction([SimTypePointer(SimStruct({}, name="new_utsname", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["name"]),
        # long sys_uname(struct old_utsname *);
        'uname': SimTypeFunction([SimTypePointer(SimStruct({}, name="old_utsname", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["None"]),
        # long sys_olduname(struct oldold_utsname *);
        'olduname': SimTypeFunction([SimTypePointer(SimStruct({}, name="oldold_utsname", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["None"]),
        # long sys_getrlimit(unsigned int resource, struct rlimit *rlim);
        'getrlimit': SimTypeFunction([SimTypeInt(signed=False), SimTypePointer(SimStruct({}, name="rlimit", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["resource", "rlim"]),
        # long sys_setrlimit(unsigned int resource, struct rlimit *rlim);
        'setrlimit': SimTypeFunction([SimTypeInt(signed=False), SimTypePointer(SimStruct({}, name="rlimit", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["resource", "rlim"]),
        # long sys_prlimit64(pid_t pid, unsigned int resource, const struct rlimit64 *new_rlim, struct rlimit64 *old_rlim);
        'prlimit64': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypeInt(signed=False), SimTypePointer(SimStruct({}, name="rlimit64", pack=False, align=None), offset=0), SimTypePointer(SimStruct({}, name="rlimit64", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["pid", "resource", "new_rlim", "old_rlim"]),
        # long sys_getrusage(int who, struct rusage *ru);
        'getrusage': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="rusage", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["who", "ru"]),
        # long sys_umask(int mask);
        'umask': SimTypeFunction([SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["mask"]),
        # long sys_msgget(key_t key, int msgflg);
        'msgget': SimTypeFunction([SimTypeInt(signed=True, label="key_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["key", "msgflg"]),
        # long sys_msgsnd(int msqid, struct msgbuf *msgp, size_t msgsz, int msgflg);
        'msgsnd': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="msgbuf", pack=False, align=None), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["msqid", "msgp", "msgsz", "msgflg"]),
        # long sys_msgrcv(int msqid, struct msgbuf *msgp, size_t msgsz, long msgtyp, int msgflg);
        'msgrcv': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="msgbuf", pack=False, align=None), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeLong(signed=True), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["msqid", "msgp", "msgsz", "msgtyp", "msgflg"]),
        # long sys_msgctl(int msqid, int cmd, struct msqid_ds *buf);
        'msgctl': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="msqid_ds", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["msqid", "cmd", "buf"]),
        # long sys_semget(key_t key, int nsems, int semflg);
        'semget': SimTypeFunction([SimTypeInt(signed=True, label="key_t"), SimTypeInt(signed=True), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["key", "nsems", "semflg"]),
        # long sys_semop(int semid, struct sembuf *sops, unsigned nsops);
        'semop': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="sembuf", pack=False, align=None), offset=0), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["semid", "sops", "nsops"]),
        # long sys_semctl(int semid, int semnum, int cmd, unsigned long arg);
        'semctl': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["semid", "semnum", "cmd", "arg"]),
        # long sys_semtimedop(int semid, struct sembuf *sops, unsigned nsops, const struct timespec *timeout);
        'semtimedop': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="sembuf", pack=False, align=None), offset=0), SimTypeInt(signed=False), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["semid", "sops", "nsops", "timeout"]),
        # long sys_shmat(int shmid, char *shmaddr, int shmflg);
        'shmat': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["shmid", "shmaddr", "shmflg"]),
        # long sys_shmget(key_t key, size_t size, int flag);
        'shmget': SimTypeFunction([SimTypeInt(signed=True, label="key_t"), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["key", "size", "flag"]),
        # long sys_shmdt(char *shmaddr);
        'shmdt': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["shmaddr"]),
        # long sys_shmctl(int shmid, int cmd, struct shmid_ds *buf);
        'shmctl': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="shmid_ds", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["shmid", "cmd", "buf"]),
        # long sys_ipc(unsigned int call, int first, unsigned long second, unsigned long third, void *ptr, long fifth);
        'ipc': SimTypeFunction([SimTypeInt(signed=False), SimTypeInt(signed=True), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypePointer(SimTypeBottom(label="void"), offset=0), SimTypeLong(signed=True)], SimTypeLong(signed=True), arg_names=["call", "first", "second", "third", "ptr", "fifth"]),
        # long sys_mq_open(const char *name, int oflag, umode_t mode, struct mq_attr *attr);
        'mq_open': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True), SimTypeInt(signed=False, label="umode_t"), SimTypePointer(SimStruct({}, name="mq_attr", pack=False, align=None), offset=0)], SimTypeFd(), arg_names=["name", "oflag", "mode", "attr"]),
        # long sys_mq_unlink(const char *name);
        'mq_unlink': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["name"]),
        # long sys_mq_timedsend(mqd_t mqdes, const char *msg_ptr, size_t msg_len, unsigned int msg_prio, const struct timespec *abs_timeout);
        'mq_timedsend': SimTypeFunction([SimTypeFd(label="mqd_t"), SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=False), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["mqdes", "msg_ptr", "msg_len", "msg_prio", "abs_timeout"]),
        # long sys_mq_timedreceive(mqd_t mqdes, char *msg_ptr, size_t msg_len, unsigned int *msg_prio, const struct timespec *abs_timeout);
        'mq_timedreceive': SimTypeFunction([SimTypeFd(label="mqd_t"), SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypePointer(SimTypeInt(signed=False), offset=0), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["mqdes", "msg_ptr", "msg_len", "msg_prio", "abs_timeout"]),
        # long sys_mq_notify(mqd_t mqdes, const struct sigevent *notification);
        'mq_notify': SimTypeFunction([SimTypeFd(label="mqd_t"), SimTypePointer(SimStruct({}, name="sigevent", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["mqdes", "notification"]),
        # long sys_mq_getsetattr(mqd_t mqdes, const struct mq_attr *mqstat, struct mq_attr *omqstat);
        'mq_getsetattr': SimTypeFunction([SimTypeFd(label="mqd_t"), SimTypePointer(SimStruct({}, name="mq_attr", pack=False, align=None), offset=0), SimTypePointer(SimStruct({}, name="mq_attr", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["mqdes", "mqstat", "omqstat"]),
        # long sys_pciconfig_iobase(long which, unsigned long bus, unsigned long devfn);
        'pciconfig_iobase': SimTypeFunction([SimTypeLong(signed=True), SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["which", "bus", "devfn"]),
        # long sys_pciconfig_read(unsigned long bus, unsigned long dfn, unsigned long off, unsigned long len, void *buf);
        'pciconfig_read': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypePointer(SimTypeBottom(label="void"), offset=0)], SimTypeLong(signed=True), arg_names=["bus", "dfn", "off", "len", "buf"]),
        # long sys_pciconfig_write(unsigned long bus, unsigned long dfn, unsigned long off, unsigned long len, void *buf);
        'pciconfig_write': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypePointer(SimTypeBottom(label="void"), offset=0)], SimTypeLong(signed=True), arg_names=["bus", "dfn", "off", "len", "buf"]),
        # long sys_prctl(int option, unsigned long arg2, unsigned long arg3, unsigned long arg4, unsigned long arg5);
        'prctl': SimTypeFunction([SimTypeInt(signed=True), SimTypeLong(signed=False), SimTypeFd(), SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["option", "arg2", "arg3", "arg4", "arg5"]),
        # long sys_swapon(const char *specialfile, int swap_flags);
        'swapon': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["specialfile", "swap_flags"]),
        # long sys_swapoff(const char *specialfile);
        'swapoff': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["specialfile"]),
        # long sys_sysctl(struct __sysctl_args *args);
        'sysctl': SimTypeFunction([SimTypePointer(SimStruct({}, name="__sysctl_args", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["args"]),
        # long sys_sysinfo(struct sysinfo *info);
        'sysinfo': SimTypeFunction([SimTypePointer(SimStruct({}, name="sysinfo", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["info"]),
        # long sys_sysfs(int option, unsigned long arg1, unsigned long arg2);
        'sysfs': SimTypeFunction([SimTypeInt(signed=True), SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["option", "arg1", "arg2"]),
        # long sys_syslog(int type, char *buf, int len);
        'syslog': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["type", "buf", "len"]),
        # long sys_uselib(const char *library);
        'uselib': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["library"]),
        # long sys_ni_syscall(void);
        'ni_syscall': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_ptrace(long request, long pid, unsigned long addr, unsigned long data);
        'ptrace': SimTypeFunction([SimTypeLong(signed=True), SimTypeLong(signed=True), SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["request", "pid", "addr", "data"]),
        # long sys_add_key(const char *_type, const char *_description, const void *_payload, size_t plen, key_serial_t destringid);
        'add_key': None,
        # long sys_request_key(const char *_type, const char *_description, const char *_callout_info, key_serial_t destringid);
        'request_key': None,
        # long sys_keyctl(int cmd, unsigned long arg2, unsigned long arg3, unsigned long arg4, unsigned long arg5);
        'keyctl': SimTypeFunction([SimTypeInt(signed=True), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["cmd", "arg2", "arg3", "arg4", "arg5"]),
        # long sys_ioprio_set(int which, int who, int ioprio);
        'ioprio_set': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["which", "who", "ioprio"]),
        # long sys_ioprio_get(int which, int who);
        'ioprio_get': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["which", "who"]),
        # long sys_set_mempolicy(int mode, const unsigned long *nmask, unsigned long maxnode);
        'set_mempolicy': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypeLong(signed=False), offset=0), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["mode", "nmask", "maxnode"]),
        # long sys_migrate_pages(pid_t pid, unsigned long maxnode, const unsigned long *from, const unsigned long *to);
        'migrate_pages': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypeLong(signed=False), SimTypePointer(SimTypeLong(signed=False), offset=0), SimTypePointer(SimTypeLong(signed=False), offset=0)], SimTypeLong(signed=True), arg_names=["pid", "maxnode", "from", "to"]),
        # long sys_move_pages(pid_t pid, unsigned long nr_pages, const void * *pages, const int *nodes, int *status, int flags);
        'move_pages': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypeLong(signed=False), SimTypePointer(SimTypePointer(SimTypeBottom(label="void"), offset=0), offset=0), SimTypePointer(SimTypeInt(signed=True), offset=0), SimTypePointer(SimTypeInt(signed=True), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["pid", "nr_pages", "pages", "nodes", "status", "flags"]),
        # long sys_mbind(unsigned long start, unsigned long len, unsigned long mode, const unsigned long *nmask, unsigned long maxnode, unsigned flags);
        'mbind': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypePointer(SimTypeLong(signed=False), offset=0), SimTypeLong(signed=False), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["start", "len", "mode", "nmask", "maxnode", "flags"]),
        # long sys_get_mempolicy(int *policy, unsigned long *nmask, unsigned long maxnode, unsigned long addr, unsigned long flags);
        'get_mempolicy': SimTypeFunction([SimTypePointer(SimTypeInt(signed=True), offset=0), SimTypePointer(SimTypeLong(signed=False), offset=0), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["policy", "nmask", "maxnode", "addr", "flags"]),
        # long sys_inotify_init(void);
        'inotify_init': SimTypeFunction([], SimTypeFd()),
        # long sys_inotify_init1(int flags);
        'inotify_init1': SimTypeFunction([SimTypeInt(signed=True)], SimTypeFd(), arg_names=["flags"]),
        # long sys_inotify_add_watch(int fd, const char *path, u32 mask);
        'inotify_add_watch': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False, label="u32")], SimTypeLong(signed=True), arg_names=["fd", "path", "mask"]),
        # long sys_inotify_rm_watch(int fd, __s32 wd);
        'inotify_rm_watch': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=True, label="__s32")], SimTypeLong(signed=True), arg_names=["fd", "wd"]),
        # long sys_spu_run(int fd, __u32 *unpc, __u32 *ustatus);
        'spu_run': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypeInt(signed=False, label="__u32"), offset=0), SimTypePointer(SimTypeInt(signed=False, label="__u32"), offset=0)], SimTypeLong(signed=True), arg_names=["fd", "unpc", "ustatus"]),
        # long sys_spu_create(const char *name, unsigned int flags, umode_t mode, int fd);
        'spu_create': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False), SimTypeInt(signed=False, label="umode_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["name", "flags", "mode", "fd"]),
        # long sys_mknodat(int dfd, const char * filename, umode_t mode, unsigned dev);
        'mknodat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False, label="umode_t"), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["dfd", "filename", "mode", "dev"]),
        # long sys_mkdirat(int dfd, const char * pathname, umode_t mode);
        'mkdirat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False, label="umode_t")], SimTypeLong(signed=True), arg_names=["dfd", "pathname", "mode"]),
        # long sys_unlinkat(int dfd, const char * pathname, int flag);
        'unlinkat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["dfd", "pathname", "flag"]),
        # long sys_symlinkat(const char * oldname, int newdfd, const char * newname);
        'symlinkat': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["oldname", "newdfd", "newname"]),
        # long sys_linkat(int olddfd, const char *oldname, int newdfd, const char *newname, int flags);
        'linkat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["olddfd", "oldname", "newdfd", "newname", "flags"]),
        # long sys_renameat(int olddfd, const char * oldname, int newdfd, const char * newname);
        'renameat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["olddfd", "oldname", "newdfd", "newname"]),
        # long sys_renameat2(int olddfd, const char *oldname, int newdfd, const char *newname, unsigned int flags);
        'renameat2': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["olddfd", "oldname", "newdfd", "newname", "flags"]),
        # long sys_futimesat(int dfd, const char *filename, struct timeval *utimes);
        'futimesat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_usec": SimTypeLong(signed=True)}, name="timeval", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["dfd", "filename", "utimes"]),
        # long sys_faccessat(int dfd, const char *filename, int mode);
        'faccessat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["dfd", "filename", "mode"]),
        # long sys_fchmodat(int dfd, const char * filename, umode_t mode);
        'fchmodat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False, label="umode_t")], SimTypeLong(signed=True), arg_names=["dfd", "filename", "mode"]),
        # long sys_fchownat(int dfd, const char *filename, uid_t user, gid_t group, int flag);
        'fchownat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False, label="uid_t"), SimTypeInt(signed=False, label="gid_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["dfd", "filename", "user", "group", "flag"]),
        # long sys_openat(int dfd, const char *filename, int flags, umode_t mode);
        'openat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True), SimTypeInt(signed=False, label="umode_t")], SimTypeFd(), arg_names=["dfd", "filename", "flags", "mode"]),
        # long sys_newfstatat(int dfd, const char *filename, struct stat *statbuf, int flag);
        'newfstatat': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({}, name="stat", pack=False, align=None), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["dfd", "filename", "statbuf", "flag"]),
        # long sys_readlinkat(int dfd, const char *path, char *buf, int bufsiz);
        'readlinkat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["dfd", "path", "buf", "bufsiz"]),
        # long sys_utimensat(int dfd, const char *filename, struct timespec *utimes, int flags);
        'utimensat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["dfd", "filename", "utimes", "flags"]),
        # long sys_unshare(unsigned long unshare_flags);
        'unshare': SimTypeFunction([SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["unshare_flags"]),
        # long sys_splice(int fd_in, loff_t *off_in, int fd_out, loff_t *off_out, size_t len, unsigned int flags);
        'splice': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeLongLong(signed=True, label="loff_t"), offset=0), SimTypeFd(), SimTypePointer(SimTypeLongLong(signed=True, label="loff_t"), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["fd_in", "off_in", "fd_out", "off_out", "len", "flags"]),
        # long sys_vmsplice(int fd, const struct iovec *iov, unsigned long nr_segs, unsigned int flags);
        'vmsplice': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({"iov_base": SimTypePointer(SimTypeBottom(label="void"), offset=0), "iov_len": SimTypeLong(signed=False, label="size_t")}, name="iovec", pack=False, align=None), offset=0), SimTypeLong(signed=False), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "iov", "nr_segs", "flags"]),
        # long sys_tee(int fdin, int fdout, size_t len, unsigned int flags);
        'tee': SimTypeFunction([SimTypeFd(), SimTypeFd(), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["fdin", "fdout", "len", "flags"]),
        # long sys_sync_file_range(int fd, loff_t offset, loff_t nbytes, unsigned int flags);
        'sync_file_range': SimTypeFunction([SimTypeFd(), SimTypeLongLong(signed=True, label="loff_t"), SimTypeLongLong(signed=True, label="loff_t"), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["fd", "offset", "nbytes", "flags"]),
        # long sys_sync_file_range2(int fd, unsigned int flags, loff_t offset, loff_t nbytes);
        'sync_file_range2': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=False), SimTypeLongLong(signed=True, label="loff_t"), SimTypeLongLong(signed=True, label="loff_t")], SimTypeLong(signed=True), arg_names=["fd", "flags", "offset", "nbytes"]),
        # long sys_get_robust_list(int pid, struct robust_list_head * *head_ptr, size_t *len_ptr);
        'get_robust_list': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypePointer(SimStruct({}, name="robust_list_head", pack=False, align=None), offset=0), offset=0), SimTypePointer(SimTypeLong(signed=False, label="size_t"), offset=0)], SimTypeLong(signed=True), arg_names=["pid", "head_ptr", "len_ptr"]),
        # long sys_set_robust_list(struct robust_list_head *head, size_t len);
        'set_robust_list': SimTypeFunction([SimTypePointer(SimStruct({}, name="robust_list_head", pack=False, align=None), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["head", "len"]),
        # long sys_getcpu(unsigned *cpu, unsigned *node, struct getcpu_cache *cache);
        'getcpu': SimTypeFunction([SimTypePointer(SimTypeInt(signed=False), offset=0), SimTypePointer(SimTypeInt(signed=False), offset=0), SimTypePointer(SimStruct({}, name="getcpu_cache", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["cpu", "node", "cache"]),
        # long sys_signalfd(int ufd, sigset_t *user_mask, size_t sizemask);
        'signalfd': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeInt(signed=True, label="sigset_t"), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeFd(), arg_names=["ufd", "user_mask", "sizemask"]),
        # long sys_signalfd4(int ufd, sigset_t *user_mask, size_t sizemask, int flags);
        'signalfd4': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeInt(signed=True, label="sigset_t"), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=True)], SimTypeFd(), arg_names=["ufd", "user_mask", "sizemask", "flags"]),
        # long sys_timerfd_create(int clockid, int flags);
        'timerfd_create': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True)], SimTypeFd(), arg_names=["clockid", "flags"]),
        # long sys_timerfd_settime(int ufd, int flags, const struct itimerspec *utmr, struct itimerspec *otmr);
        'timerfd_settime': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="itimerspec", pack=False, align=None), offset=0), SimTypePointer(SimStruct({}, name="itimerspec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["ufd", "flags", "utmr", "otmr"]),
        # long sys_timerfd_gettime(int ufd, struct itimerspec *otmr);
        'timerfd_gettime': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="itimerspec", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["ufd", "otmr"]),
        # long sys_eventfd(unsigned int count);
        'eventfd': SimTypeFunction([SimTypeInt(signed=False)], SimTypeFd(), arg_names=["count"]),
        # long sys_eventfd2(unsigned int count, int flags);
        'eventfd2': SimTypeFunction([SimTypeInt(signed=False), SimTypeInt(signed=True)], SimTypeFd(), arg_names=["count", "flags"]),
        # long sys_memfd_create(const char *uname_ptr, unsigned int flags);
        'memfd_create': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False)], SimTypeFd(), arg_names=["uname_ptr", "flags"]),
        # long sys_userfaultfd(int flags);
        'userfaultfd': SimTypeFunction([SimTypeInt(signed=True)], SimTypeFd(), arg_names=["flags"]),
        # long sys_fallocate(int fd, int mode, loff_t offset, loff_t len);
        'fallocate': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=True), SimTypeLongLong(signed=True, label="loff_t"), SimTypeLongLong(signed=True, label="loff_t")], SimTypeLong(signed=True), arg_names=["fd", "mode", "offset", "len"]),
        # long sys_old_readdir(unsigned int, struct old_linux_dirent *, unsigned int);
        'old_readdir': SimTypeFunction([SimTypeInt(signed=False), SimTypePointer(SimStruct({}, name="old_linux_dirent", pack=False, align=None), offset=0), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["None", "None", "None"]),
        # long sys_pselect6(int, fd_set *, fd_set *, fd_set *, struct timespec *, void *);
        'pselect6': None,
        # long sys_ppoll(struct pollfd *, unsigned int, struct timespec *, const sigset_t *, size_t);
        'ppoll': SimTypeFunction([SimTypePointer(SimStruct({}, name="pollfd", pack=False, align=None), offset=0), SimTypeInt(signed=False), SimTypePointer(SimStruct({"tv_sec": SimTypeLong(signed=True, label="time_t"), "tv_nsec": SimTypeLong(signed=True)}, name="timespec", pack=False, align=None), offset=0), SimTypePointer(SimTypeInt(signed=True, label="sigset_t"), offset=0), SimTypeLong(signed=False, label="size_t")], SimTypeLong(signed=True), arg_names=["None", "None", "None", "None", "None"]),
        # long sys_fanotify_init(unsigned int flags, unsigned int event_f_flags);
        'fanotify_init': SimTypeFunction([SimTypeInt(signed=False), SimTypeInt(signed=False)], SimTypeFd(), arg_names=["flags", "event_f_flags"]),
        # long sys_fanotify_mark(int fanotify_fd, unsigned int flags, u64 mask, int fd, const char *pathname);
        'fanotify_mark': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=False), SimTypeLongLong(signed=False, label="u64"), SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["fanotify_fd", "flags", "mask", "fd", "pathname"]),
        # long sys_syncfs(int fd);
        'syncfs': SimTypeFunction([SimTypeFd()], SimTypeLong(signed=True), arg_names=["fd"]),
        # long sys_fork(void);
        'fork': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_vfork(void);
        'vfork': SimTypeFunction([], SimTypeLong(signed=True)),
        # long sys_clone(unsigned long, unsigned long, int *, int *, unsigned long);
        'clone': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypePointer(SimTypeInt(signed=True), offset=0), SimTypePointer(SimTypeInt(signed=True), offset=0), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["None", "None", "None", "None", "None"]),
        # long sys_execve(const char *filename, const char *const *argv, const char *const *envp);
        'execve': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypePointer(SimTypeChar(), offset=0), offset=0), SimTypePointer(SimTypePointer(SimTypeChar(), offset=0), offset=0)], SimTypeLong(signed=True), arg_names=["filename", "argv", "envp"]),
        # long sys_perf_event_open( struct perf_event_attr *attr_uptr, pid_t pid, int cpu, int group_fd, unsigned long flags);
        'perf_event_open': SimTypeFunction([SimTypePointer(SimStruct({}, name="perf_event_attr", pack=False, align=None), offset=0), SimTypeFd(label="pid_t"), SimTypeInt(signed=True), SimTypeFd(), SimTypeLong(signed=False)], SimTypeFd(), arg_names=["attr_uptr", "pid", "cpu", "group_fd", "flags"]),
        # long sys_mmap_pgoff(unsigned long addr, unsigned long len, unsigned long prot, unsigned long flags, unsigned long fd, unsigned long pgoff);
        'mmap_pgoff': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["addr", "len", "prot", "flags", "fd", "pgoff"]),
        # long sys_old_mmap(struct mmap_arg_struct *arg);
        'old_mmap': SimTypeFunction([SimTypePointer(SimStruct({}, name="mmap_arg_struct", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["arg"]),
        # long sys_name_to_handle_at(int dfd, const char *name, struct file_handle *handle, int *mnt_id, int flag);
        'name_to_handle_at': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimStruct({}, name="file_handle", pack=False, align=None), offset=0), SimTypePointer(SimTypeInt(signed=True), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["dfd", "name", "handle", "mnt_id", "flag"]),
        # long sys_open_by_handle_at(int mountdirfd, struct file_handle *handle, int flags);
        'open_by_handle_at': SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="file_handle", pack=False, align=None), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["mountdirfd", "handle", "flags"]),
        # long sys_setns(int fd, int nstype);
        'setns': SimTypeFunction([SimTypeFd(), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["fd", "nstype"]),
        # long sys_process_vm_readv(pid_t pid, const struct iovec *lvec, unsigned long liovcnt, const struct iovec *rvec, unsigned long riovcnt, unsigned long flags);
        'process_vm_readv': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypePointer(SimStruct({"iov_base": SimTypePointer(SimTypeBottom(label="void"), offset=0), "iov_len": SimTypeLong(signed=False, label="size_t")}, name="iovec", pack=False, align=None), offset=0), SimTypeLong(signed=False), SimTypePointer(SimStruct({"iov_base": SimTypePointer(SimTypeBottom(label="void"), offset=0), "iov_len": SimTypeLong(signed=False, label="size_t")}, name="iovec", pack=False, align=None), offset=0), SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["pid", "lvec", "liovcnt", "rvec", "riovcnt", "flags"]),
        # long sys_process_vm_writev(pid_t pid, const struct iovec *lvec, unsigned long liovcnt, const struct iovec *rvec, unsigned long riovcnt, unsigned long flags);
        'process_vm_writev': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypePointer(SimStruct({"iov_base": SimTypePointer(SimTypeBottom(label="void"), offset=0), "iov_len": SimTypeLong(signed=False, label="size_t")}, name="iovec", pack=False, align=None), offset=0), SimTypeLong(signed=False), SimTypePointer(SimStruct({"iov_base": SimTypePointer(SimTypeBottom(label="void"), offset=0), "iov_len": SimTypeLong(signed=False, label="size_t")}, name="iovec", pack=False, align=None), offset=0), SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["pid", "lvec", "liovcnt", "rvec", "riovcnt", "flags"]),
        # long sys_kcmp(pid_t pid1, pid_t pid2, int type, unsigned long idx1, unsigned long idx2);
        'kcmp': SimTypeFunction([SimTypeInt(signed=True, label="pid_t"), SimTypeInt(signed=True, label="pid_t"), SimTypeInt(signed=True), SimTypeFd(), SimTypeFd()], SimTypeLong(signed=True), arg_names=["pid1", "pid2", "type", "idx1", "idx2"]),
        # long sys_finit_module(int fd, const char *uargs, int flags);
        'finit_module': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["fd", "uargs", "flags"]),
        # long sys_seccomp(unsigned int op, unsigned int flags, const char *uargs);
        'seccomp': SimTypeFunction([SimTypeInt(signed=False), SimTypeInt(signed=False), SimTypePointer(SimTypeChar(), offset=0)], SimTypeLong(signed=True), arg_names=["op", "flags", "uargs"]),
        # long sys_getrandom(char *buf, size_t count, unsigned int flags);
        'getrandom': SimTypeFunction([SimTypePointer(SimTypeChar(), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["buf", "count", "flags"]),
        # long sys_bpf(int cmd, struct bpf_attr *attr, unsigned int size);
        'bpf': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimStruct({}, name="bpf_attr", pack=False, align=None), offset=0), SimTypeInt(signed=False)], SimTypeFd(), arg_names=["cmd", "attr", "size"]),
        # long sys_execveat(int dfd, const char *filename, const char *const *argv, const char *const *envp, int flags);
        'execveat': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypePointer(SimTypePointer(SimTypeChar(), offset=0), offset=0), SimTypePointer(SimTypePointer(SimTypeChar(), offset=0), offset=0), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["dfd", "filename", "argv", "envp", "flags"]),
        # long sys_membarrier(int cmd, int flags);
        'membarrier': SimTypeFunction([SimTypeInt(signed=True), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["cmd", "flags"]),
        # long sys_copy_file_range(int fd_in, loff_t *off_in, int fd_out, loff_t *off_out, size_t len, unsigned int flags);
        'copy_file_range': SimTypeFunction([SimTypeInt(signed=True), SimTypePointer(SimTypeLongLong(signed=True, label="loff_t"), offset=0), SimTypeInt(signed=True), SimTypePointer(SimTypeLongLong(signed=True, label="loff_t"), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=False)], SimTypeLong(signed=True), arg_names=["fd_in", "off_in", "fd_out", "off_out", "len", "flags"]),
        # long sys_mlock2(unsigned long start, size_t len, int flags);
        'mlock2': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["start", "len", "flags"]),
        # long sys_pkey_mprotect(unsigned long start, size_t len, unsigned long prot, int pkey);
        'pkey_mprotect': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False, label="size_t"), SimTypeLong(signed=False), SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["start", "len", "prot", "pkey"]),
        # long sys_pkey_alloc(unsigned long flags, unsigned long init_val);
        'pkey_alloc': SimTypeFunction([SimTypeLong(signed=False), SimTypeLong(signed=False)], SimTypeLong(signed=True), arg_names=["flags", "init_val"]),
        # long sys_pkey_free(int pkey);
        'pkey_free': SimTypeFunction([SimTypeInt(signed=True)], SimTypeLong(signed=True), arg_names=["pkey"]),
        # long sys_statx(int dfd, const char *path, unsigned flags, unsigned mask, struct statx *buffer);
        'statx': SimTypeFunction([SimTypeFd(), SimTypePointer(SimTypeChar(), offset=0), SimTypeInt(signed=False), SimTypeInt(signed=False), SimTypePointer(SimStruct({}, name="statx", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["dfd", "path", "flags", "mask", "buffer"]),
        # void *mmap(void *addr, size_t len, int prot, int flags, int fildes, off_t off);
        'mmap': SimTypeFunction([SimTypePointer(SimTypeBottom(label="void"), offset=0), SimTypeLong(signed=False, label="size_t"), SimTypeInt(signed=True), SimTypeInt(signed=True), SimTypeFd(), SimTypeLong(signed=False, label="off_t")], SimTypePointer(SimTypeBottom(label="void"), offset=0), arg_names=["addr", "len", "prot", "flags", "fildes", "off"]),
    }

_amd64_c_decls = [
    "long sys_time(time_t *tloc);",
    "long sys_stime(time_t *tptr);",
    "long sys_gettimeofday(struct timeval *tv, struct timezone *tz);",
    "long sys_settimeofday(struct timeval *tv, struct timezone *tz);",
    "long sys_adjtimex(struct timex *txc_p);",
    "long sys_times(struct tms *tbuf);",
    "long sys_gettid(void);",
    "long sys_nanosleep(struct timespec *rqtp, struct timespec *rmtp);",
    "long sys_alarm(unsigned int seconds);",
    "long sys_getpid(void);",
    "long sys_getppid(void);",
    "long sys_getuid(void);",
    "long sys_geteuid(void);",
    "long sys_getgid(void);",
    "long sys_getegid(void);",
    "long sys_getresuid(uid_t *ruid, uid_t *euid, uid_t *suid);",
    "long sys_getresgid(gid_t *rgid, gid_t *egid, gid_t *sgid);",
    "long sys_getpgid(pid_t pid);",
    "long sys_getpgrp(void);",
    "long sys_getsid(pid_t pid);",
    "long sys_getgroups(int gidsetsize, gid_t *grouplist);",
    "long sys_setregid(gid_t rgid, gid_t egid);",
    "long sys_setgid(gid_t gid);",
    "long sys_setreuid(uid_t ruid, uid_t euid);",
    "long sys_setuid(uid_t uid);",
    "long sys_setresuid(uid_t ruid, uid_t euid, uid_t suid);",
    "long sys_setresgid(gid_t rgid, gid_t egid, gid_t sgid);",
    "long sys_setfsuid(uid_t uid);",
    "long sys_setfsgid(gid_t gid);",
    "long sys_setpgid(pid_t pid, pid_t pgid);",
    "long sys_setsid(void);",
    "long sys_setgroups(int gidsetsize, gid_t *grouplist);",
    "long sys_acct(const char *name);",
    "long sys_capget(cap_user_header_t header, cap_user_data_t dataptr);",
    "long sys_capset(cap_user_header_t header, const cap_user_data_t data);",
    "long sys_personality(unsigned int personality);",
    "long sys_sigpending(old_sigset_t *set);",
    "long sys_sigprocmask(int how, old_sigset_t *set, old_sigset_t *oset);",
    "long sys_sigaltstack(const struct sigaltstack *uss, struct sigaltstack *uoss);",
    "long sys_getitimer(int which, struct itimerval *value);",
    "long sys_setitimer(int which, struct itimerval *value, struct itimerval *ovalue);",
    "long sys_timer_create(clockid_t which_clock, struct sigevent *timer_event_spec, timer_t * created_timer_id);",
    "long sys_timer_gettime(timer_t timer_id, struct itimerspec *setting);",
    "long sys_timer_getoverrun(timer_t timer_id);",
    "long sys_timer_settime(timer_t timer_id, int flags, const struct itimerspec *new_setting, struct itimerspec *old_setting);",
    "long sys_timer_delete(timer_t timer_id);",
    "long sys_clock_settime(clockid_t which_clock, const struct timespec *tp);",
    "long sys_clock_gettime(clockid_t which_clock, struct timespec *tp);",
    "long sys_clock_adjtime(clockid_t which_clock, struct timex *tx);",
    "long sys_clock_getres(clockid_t which_clock, struct timespec *tp);",
    "long sys_clock_nanosleep(clockid_t which_clock, int flags, const struct timespec *rqtp, struct timespec *rmtp);",
    "long sys_nice(int increment);",
    "long sys_sched_setscheduler(pid_t pid, int policy, struct sched_param *param);",
    "long sys_sched_setparam(pid_t pid, struct sched_param *param);",
    "long sys_sched_setattr(pid_t pid, struct sched_attr *attr, unsigned int flags);",
    "long sys_sched_getscheduler(pid_t pid);",
    "long sys_sched_getparam(pid_t pid, struct sched_param *param);",
    "long sys_sched_getattr(pid_t pid, struct sched_attr *attr, unsigned int size, unsigned int flags);",
    "long sys_sched_setaffinity(pid_t pid, unsigned int len, unsigned long *user_mask_ptr);",
    "long sys_sched_getaffinity(pid_t pid, unsigned int len, unsigned long *user_mask_ptr);",
    "long sys_sched_yield(void);",
    "long sys_sched_get_priority_max(int policy);",
    "long sys_sched_get_priority_min(int policy);",
    "long sys_sched_rr_get_interval(pid_t pid, struct timespec *interval);",
    "long sys_setpriority(int which, int who, int niceval);",
    "long sys_getpriority(int which, int who);",
    "long sys_shutdown(int, int);",
    "long sys_reboot(int magic1, int magic2, unsigned int cmd, void *arg);",
    "long sys_restart_syscall(void);",
    "long sys_kexec_load(unsigned long entry, unsigned long nr_segments, struct kexec_segment *segments, unsigned long flags);",
    "long sys_kexec_file_load(int kernel_fd, int initrd_fd, unsigned long cmdline_len, const char *cmdline_ptr, unsigned long flags);",
    "long sys_exit(int error_code);",
    "long sys_exit_group(int error_code);",
    "long sys_wait4(pid_t pid, int *stat_addr, int options, struct rusage *ru);",
    "long sys_waitid(int which, pid_t pid, struct siginfo *infop, int options, struct rusage *ru);",
    "long sys_waitpid(pid_t pid, int *stat_addr, int options);",
    "long sys_set_tid_address(int *tidptr);",
    "long sys_futex(u32 *uaddr, int op, u32 val, struct timespec *utime, u32 *uaddr2, u32 val3);",
    "long sys_init_module(void *umod, unsigned long len, const char *uargs);",
    "long sys_delete_module(const char *name_user, unsigned int flags);",
    "long sys_rt_sigsuspend(sigset_t *unewset, size_t sigsetsize);",
    "long sys_sigaction(int, const struct old_sigaction *, struct old_sigaction *);",
    "long sys_rt_sigaction(int, const struct sigaction *, struct sigaction *, size_t);",
    "long sys_rt_sigprocmask(int how, sigset_t *set, sigset_t *oset, size_t sigsetsize);",
    "long sys_rt_sigpending(sigset_t *set, size_t sigsetsize);",
    "long sys_rt_sigtimedwait(const sigset_t *uthese, siginfo_t *uinfo, const struct timespec *uts, size_t sigsetsize);",
    "long sys_rt_tgsigqueueinfo(pid_t tgid, pid_t pid, int sig, siginfo_t *uinfo);",
    "long sys_kill(pid_t pid, int sig);",
    "long sys_tgkill(pid_t tgid, pid_t pid, int sig);",
    "long sys_tkill(pid_t pid, int sig);",
    "long sys_rt_sigqueueinfo(pid_t pid, int sig, siginfo_t *uinfo);",
    "long sys_sgetmask(void);",
    "long sys_ssetmask(int newmask);",
    "long sys_signal(int sig, __sighandler_t handler);",
    "long sys_pause(void);",
    "long sys_sync(void);",
    "long sys_fsync(unsigned int fd);",
    "long sys_fdatasync(unsigned int fd);",
    "long sys_bdflush(int func, long data);",
    "long sys_mount(char *dev_name, char *dir_name, char *type, unsigned long flags, void *data);",
    "long sys_umount(char *name, int flags);",
    "long sys_oldumount(char *name);",
    "long sys_truncate(const char *path, long length);",
    "long sys_ftruncate(unsigned int fd, unsigned long length);",
    "long sys_stat(const char *filename, struct __old_kernel_stat *statbuf);",
    "long sys_statfs(const char * path, struct statfs *buf);",
    "long sys_statfs64(const char *path, size_t sz, struct statfs64 *buf);",
    "long sys_fstatfs(unsigned int fd, struct statfs *buf);",
    "long sys_fstatfs64(unsigned int fd, size_t sz, struct statfs64 *buf);",
    "long sys_lstat(const char *filename, struct __old_kernel_stat *statbuf);",
    "long sys_fstat(unsigned int fd, struct __old_kernel_stat *statbuf);",
    "long sys_newstat(const char *filename, struct stat *statbuf);",
    "long sys_newlstat(const char *filename, struct stat *statbuf);",
    "long sys_newfstat(unsigned int fd, struct stat *statbuf);",
    "long sys_ustat(unsigned dev, struct ustat *ubuf);",
    "long sys_stat64(const char *filename, struct stat64 *statbuf);",
    "long sys_fstat64(unsigned long fd, struct stat64 *statbuf);",
    "long sys_lstat64(const char *filename, struct stat64 *statbuf);",
    "long sys_fstatat64(int dfd, const char *filename, struct stat64 *statbuf, int flag);",
    "long sys_truncate64(const char *path, loff_t length);",
    "long sys_ftruncate64(unsigned int fd, loff_t length);",
    "long sys_setxattr(const char *path, const char *name, const void *value, size_t size, int flags);",
    "long sys_lsetxattr(const char *path, const char *name, const void *value, size_t size, int flags);",
    "long sys_fsetxattr(int fd, const char *name, const void *value, size_t size, int flags);",
    "long sys_getxattr(const char *path, const char *name, void *value, size_t size);",
    "long sys_lgetxattr(const char *path, const char *name, void *value, size_t size);",
    "long sys_fgetxattr(int fd, const char *name, void *value, size_t size);",
    "long sys_listxattr(const char *path, char *list, size_t size);",
    "long sys_llistxattr(const char *path, char *list, size_t size);",
    "long sys_flistxattr(int fd, char *list, size_t size);",
    "long sys_removexattr(const char *path, const char *name);",
    "long sys_lremovexattr(const char *path, const char *name);",
    "long sys_fremovexattr(int fd, const char *name);",
    "long sys_brk(unsigned long brk);",
    "long sys_mprotect(unsigned long start, size_t len, unsigned long prot);",
    "long sys_mremap(unsigned long addr, unsigned long old_len, unsigned long new_len, unsigned long flags, unsigned long new_addr);",
    "long sys_remap_file_pages(unsigned long start, unsigned long size, unsigned long prot, unsigned long pgoff, unsigned long flags);",
    "long sys_msync(unsigned long start, size_t len, int flags);",
    "long sys_fadvise64(int fd, loff_t offset, size_t len, int advice);",
    "long sys_fadvise64_64(int fd, loff_t offset, loff_t len, int advice);",
    "long sys_munmap(unsigned long addr, size_t len);",
    "long sys_mlock(unsigned long start, size_t len);",
    "long sys_munlock(unsigned long start, size_t len);",
    "long sys_mlockall(int flags);",
    "long sys_munlockall(void);",
    "long sys_madvise(unsigned long start, size_t len, int behavior);",
    "long sys_mincore(unsigned long start, size_t len, unsigned char * vec);",
    "long sys_pivot_root(const char *new_root, const char *put_old);",
    "long sys_chroot(const char *filename);",
    "long sys_mknod(const char *filename, umode_t mode, unsigned dev);",
    "long sys_link(const char *oldname, const char *newname);",
    "long sys_symlink(const char *old, const char *new);",
    "long sys_unlink(const char *pathname);",
    "long sys_rename(const char *oldname, const char *newname);",
    "long sys_chmod(const char *filename, umode_t mode);",
    "long sys_fchmod(unsigned int fd, umode_t mode);",
    "long sys_fcntl(unsigned int fd, unsigned int cmd, unsigned long arg);",
    "long sys_fcntl64(unsigned int fd, unsigned int cmd, unsigned long arg);",
    "long sys_pipe(int *fildes);",
    "long sys_pipe2(int *fildes, int flags);",
    "long sys_dup(unsigned int fildes);",
    "long sys_dup2(unsigned int oldfd, unsigned int newfd);",
    "long sys_dup3(unsigned int oldfd, unsigned int newfd, int flags);",
    "long sys_ioperm(unsigned long from, unsigned long num, int on);",
    "long sys_ioctl(unsigned int fd, unsigned int cmd, unsigned long arg);",
    "long sys_flock(unsigned int fd, unsigned int cmd);",
    "long sys_io_setup(unsigned nr_reqs, aio_context_t *ctx);",
    "long sys_io_destroy(aio_context_t ctx);",
    "long sys_io_getevents(aio_context_t ctx_id, long min_nr, long nr, struct io_event *events, struct timespec *timeout);",
    "long sys_io_submit(aio_context_t, long, struct iocb * *);",
    "long sys_io_cancel(aio_context_t ctx_id, struct iocb *iocb, struct io_event *result);",
    "long sys_sendfile(int out_fd, int in_fd, off_t *offset, size_t count);",
    "long sys_sendfile64(int out_fd, int in_fd, loff_t *offset, size_t count);",
    "long sys_readlink(const char *path, char *buf, int bufsiz);",
    "long sys_creat(const char *pathname, umode_t mode);",
    "long sys_open(const char *filename, int flags, umode_t mode);",
    "long sys_close(unsigned int fd);",
    "long sys_access(const char *filename, int mode);",
    "long sys_vhangup(void);",
    "long sys_chown(const char *filename, uid_t user, gid_t group);",
    "long sys_lchown(const char *filename, uid_t user, gid_t group);",
    "long sys_fchown(unsigned int fd, uid_t user, gid_t group);",
    "long sys_utime(char *filename, struct utimbuf *times);",
    "long sys_utimes(char *filename, struct timeval *utimes);",
    "long sys_lseek(unsigned int fd, off_t offset, unsigned int whence);",
    "long sys_llseek(unsigned int fd, unsigned long offset_high, unsigned long offset_low, loff_t *result, unsigned int whence);",
    "long sys_read(unsigned int fd, char *buf, size_t count);",
    "long sys_readahead(int fd, loff_t offset, size_t count);",
    "long sys_readv(unsigned long fd, const struct iovec *vec, unsigned long vlen);",
    "long sys_write(unsigned int fd, const char *buf, size_t count);",
    "long sys_writev(unsigned long fd, const struct iovec *vec, unsigned long vlen);",
    "long sys_pread64(unsigned int fd, char *buf, size_t count, loff_t pos);",
    "long sys_pwrite64(unsigned int fd, const char *buf, size_t count, loff_t pos);",
    "long sys_preadv(unsigned long fd, const struct iovec *vec, unsigned long vlen, unsigned long pos_l, unsigned long pos_h);",
    "long sys_preadv2(unsigned long fd, const struct iovec *vec, unsigned long vlen, unsigned long pos_l, unsigned long pos_h, int flags);",
    "long sys_pwritev(unsigned long fd, const struct iovec *vec, unsigned long vlen, unsigned long pos_l, unsigned long pos_h);",
    "long sys_pwritev2(unsigned long fd, const struct iovec *vec, unsigned long vlen, unsigned long pos_l, unsigned long pos_h, int flags);",
    "long sys_getcwd(char *buf, unsigned long size);",
    "long sys_mkdir(const char *pathname, umode_t mode);",
    "long sys_chdir(const char *filename);",
    "long sys_fchdir(unsigned int fd);",
    "long sys_rmdir(const char *pathname);",
    "long sys_lookup_dcookie(u64 cookie64, char *buf, size_t len);",
    "long sys_quotactl(unsigned int cmd, const char *special, qid_t id, void *addr);",
    "long sys_getdents(unsigned int fd, struct linux_dirent *dirent, unsigned int count);",
    "long sys_getdents64(unsigned int fd, struct linux_dirent64 *dirent, unsigned int count);",
    "long sys_setsockopt(int fd, int level, int optname, char *optval, int optlen);",
    "long sys_getsockopt(int fd, int level, int optname, char *optval, int *optlen);",
    "long sys_bind(int, struct sockaddr *, int);",
    "long sys_connect(int, struct sockaddr *, int);",
    "long sys_accept(int, struct sockaddr *, int *);",
    "long sys_accept4(int, struct sockaddr *, int *, int);",
    "long sys_getsockname(int, struct sockaddr *, int *);",
    "long sys_getpeername(int, struct sockaddr *, int *);",
    "long sys_send(int, void *, size_t, unsigned);",
    "long sys_sendto(int, void *, size_t, unsigned, struct sockaddr *, int);",
    "long sys_sendmsg(int fd, struct user_msghdr *msg, unsigned flags);",
    "long sys_sendmmsg(int fd, struct mmsghdr *msg, unsigned int vlen, unsigned flags);",
    "long sys_recv(int, void *, size_t, unsigned);",
    "long sys_recvfrom(int, void *, size_t, unsigned, struct sockaddr *, int *);",
    "long sys_recvmsg(int fd, struct user_msghdr *msg, unsigned flags);",
    "long sys_recvmmsg(int fd, struct mmsghdr *msg, unsigned int vlen, unsigned flags, struct timespec *timeout);",
    "long sys_socket(int, int, int);",
    "long sys_socketpair(int, int, int, int *);",
    "long sys_socketcall(int call, unsigned long *args);",
    "long sys_listen(int, int);",
    "long sys_poll(struct pollfd *ufds, unsigned int nfds, int timeout);",
    "long sys_select(int n, fd_set *inp, fd_set *outp, fd_set *exp, struct timeval *tvp);",
    "long sys_old_select(struct sel_arg_struct *arg);",
    "long sys_epoll_create(int size);",
    "long sys_epoll_create1(int flags);",
    "long sys_epoll_ctl(int epfd, int op, int fd, struct epoll_event *event);",
    "long sys_epoll_wait(int epfd, struct epoll_event *events, int maxevents, int timeout);",
    "long sys_epoll_pwait(int epfd, struct epoll_event *events, int maxevents, int timeout, const sigset_t *sigmask, size_t sigsetsize);",
    "long sys_gethostname(char *name, int len);",
    "long sys_sethostname(char *name, int len);",
    "long sys_setdomainname(char *name, int len);",
    "long sys_newuname(struct new_utsname *name);",
    "long sys_uname(struct old_utsname *);",
    "long sys_olduname(struct oldold_utsname *);",
    "long sys_getrlimit(unsigned int resource, struct rlimit *rlim);",
    "long sys_setrlimit(unsigned int resource, struct rlimit *rlim);",
    "long sys_prlimit64(pid_t pid, unsigned int resource, const struct rlimit64 *new_rlim, struct rlimit64 *old_rlim);",
    "long sys_getrusage(int who, struct rusage *ru);",
    "long sys_umask(int mask);",
    "long sys_msgget(key_t key, int msgflg);",
    "long sys_msgsnd(int msqid, struct msgbuf *msgp, size_t msgsz, int msgflg);",
    "long sys_msgrcv(int msqid, struct msgbuf *msgp, size_t msgsz, long msgtyp, int msgflg);",
    "long sys_msgctl(int msqid, int cmd, struct msqid_ds *buf);",
    "long sys_semget(key_t key, int nsems, int semflg);",
    "long sys_semop(int semid, struct sembuf *sops, unsigned nsops);",
    "long sys_semctl(int semid, int semnum, int cmd, unsigned long arg);",
    "long sys_semtimedop(int semid, struct sembuf *sops, unsigned nsops, const struct timespec *timeout);",
    "long sys_shmat(int shmid, char *shmaddr, int shmflg);",
    "long sys_shmget(key_t key, size_t size, int flag);",
    "long sys_shmdt(char *shmaddr);",
    "long sys_shmctl(int shmid, int cmd, struct shmid_ds *buf);",
    "long sys_ipc(unsigned int call, int first, unsigned long second, unsigned long third, void *ptr, long fifth);",
    "long sys_mq_open(const char *name, int oflag, umode_t mode, struct mq_attr *attr);",
    "long sys_mq_unlink(const char *name);",
    "long sys_mq_timedsend(mqd_t mqdes, const char *msg_ptr, size_t msg_len, unsigned int msg_prio, const struct timespec *abs_timeout);",
    "long sys_mq_timedreceive(mqd_t mqdes, char *msg_ptr, size_t msg_len, unsigned int *msg_prio, const struct timespec *abs_timeout);",
    "long sys_mq_notify(mqd_t mqdes, const struct sigevent *notification);",
    "long sys_mq_getsetattr(mqd_t mqdes, const struct mq_attr *mqstat, struct mq_attr *omqstat);",
    "long sys_pciconfig_iobase(long which, unsigned long bus, unsigned long devfn);",
    "long sys_pciconfig_read(unsigned long bus, unsigned long dfn, unsigned long off, unsigned long len, void *buf);",
    "long sys_pciconfig_write(unsigned long bus, unsigned long dfn, unsigned long off, unsigned long len, void *buf);",
    "long sys_prctl(int option, unsigned long arg2, unsigned long arg3, unsigned long arg4, unsigned long arg5);",
    "long sys_swapon(const char *specialfile, int swap_flags);",
    "long sys_swapoff(const char *specialfile);",
    "long sys_sysctl(struct __sysctl_args *args);",
    "long sys_sysinfo(struct sysinfo *info);",
    "long sys_sysfs(int option, unsigned long arg1, unsigned long arg2);",
    "long sys_syslog(int type, char *buf, int len);",
    "long sys_uselib(const char *library);",
    "long sys_ni_syscall(void);",
    "long sys_ptrace(long request, long pid, unsigned long addr, unsigned long data);",
    "long sys_add_key(const char *_type, const char *_description, const void *_payload, size_t plen, key_serial_t destringid);",
    "long sys_request_key(const char *_type, const char *_description, const char *_callout_info, key_serial_t destringid);",
    "long sys_keyctl(int cmd, unsigned long arg2, unsigned long arg3, unsigned long arg4, unsigned long arg5);",
    "long sys_ioprio_set(int which, int who, int ioprio);",
    "long sys_ioprio_get(int which, int who);",
    "long sys_set_mempolicy(int mode, const unsigned long *nmask, unsigned long maxnode);",
    "long sys_migrate_pages(pid_t pid, unsigned long maxnode, const unsigned long *from, const unsigned long *to);",
    "long sys_move_pages(pid_t pid, unsigned long nr_pages, const void * *pages, const int *nodes, int *status, int flags);",
    "long sys_mbind(unsigned long start, unsigned long len, unsigned long mode, const unsigned long *nmask, unsigned long maxnode, unsigned flags);",
    "long sys_get_mempolicy(int *policy, unsigned long *nmask, unsigned long maxnode, unsigned long addr, unsigned long flags);",
    "long sys_inotify_init(void);",
    "long sys_inotify_init1(int flags);",
    "long sys_inotify_add_watch(int fd, const char *path, u32 mask);",
    "long sys_inotify_rm_watch(int fd, __s32 wd);",
    "long sys_spu_run(int fd, __u32 *unpc, __u32 *ustatus);",
    "long sys_spu_create(const char *name, unsigned int flags, umode_t mode, int fd);",
    "long sys_mknodat(int dfd, const char * filename, umode_t mode, unsigned dev);",
    "long sys_mkdirat(int dfd, const char * pathname, umode_t mode);",
    "long sys_unlinkat(int dfd, const char * pathname, int flag);",
    "long sys_symlinkat(const char * oldname, int newdfd, const char * newname);",
    "long sys_linkat(int olddfd, const char *oldname, int newdfd, const char *newname, int flags);",
    "long sys_renameat(int olddfd, const char * oldname, int newdfd, const char * newname);",
    "long sys_renameat2(int olddfd, const char *oldname, int newdfd, const char *newname, unsigned int flags);",
    "long sys_futimesat(int dfd, const char *filename, struct timeval *utimes);",
    "long sys_faccessat(int dfd, const char *filename, int mode);",
    "long sys_fchmodat(int dfd, const char * filename, umode_t mode);",
    "long sys_fchownat(int dfd, const char *filename, uid_t user, gid_t group, int flag);",
    "long sys_openat(int dfd, const char *filename, int flags, umode_t mode);",
    "long sys_newfstatat(int dfd, const char *filename, struct stat *statbuf, int flag);",
    "long sys_readlinkat(int dfd, const char *path, char *buf, int bufsiz);",
    "long sys_utimensat(int dfd, const char *filename, struct timespec *utimes, int flags);",
    "long sys_unshare(unsigned long unshare_flags);",
    "long sys_splice(int fd_in, loff_t *off_in, int fd_out, loff_t *off_out, size_t len, unsigned int flags);",
    "long sys_vmsplice(int fd, const struct iovec *iov, unsigned long nr_segs, unsigned int flags);",
    "long sys_tee(int fdin, int fdout, size_t len, unsigned int flags);",
    "long sys_sync_file_range(int fd, loff_t offset, loff_t nbytes, unsigned int flags);",
    "long sys_sync_file_range2(int fd, unsigned int flags, loff_t offset, loff_t nbytes);",
    "long sys_get_robust_list(int pid, struct robust_list_head * *head_ptr, size_t *len_ptr);",
    "long sys_set_robust_list(struct robust_list_head *head, size_t len);",
    "long sys_getcpu(unsigned *cpu, unsigned *node, struct getcpu_cache *cache);",
    "long sys_signalfd(int ufd, sigset_t *user_mask, size_t sizemask);",
    "long sys_signalfd4(int ufd, sigset_t *user_mask, size_t sizemask, int flags);",
    "long sys_timerfd_create(int clockid, int flags);",
    "long sys_timerfd_settime(int ufd, int flags, const struct itimerspec *utmr, struct itimerspec *otmr);",
    "long sys_timerfd_gettime(int ufd, struct itimerspec *otmr);",
    "long sys_eventfd(unsigned int count);",
    "long sys_eventfd2(unsigned int count, int flags);",
    "long sys_memfd_create(const char *uname_ptr, unsigned int flags);",
    "long sys_userfaultfd(int flags);",
    "long sys_fallocate(int fd, int mode, loff_t offset, loff_t len);",
    "long sys_old_readdir(unsigned int, struct old_linux_dirent *, unsigned int);",
    "long sys_pselect6(int, fd_set *, fd_set *, fd_set *, struct timespec *, void *);",
    "long sys_ppoll(struct pollfd *, unsigned int, struct timespec *, const sigset_t *, size_t);",
    "long sys_fanotify_init(unsigned int flags, unsigned int event_f_flags);",
    "long sys_fanotify_mark(int fanotify_fd, unsigned int flags, u64 mask, int fd, const char *pathname);",
    "long sys_syncfs(int fd);",
    "long sys_fork(void);",
    "long sys_vfork(void);",
    "long sys_clone(unsigned long, unsigned long, int *, int *, unsigned long);",
    "long sys_execve(const char *filename, const char *const *argv, const char *const *envp);",
    "long sys_perf_event_open( struct perf_event_attr *attr_uptr, pid_t pid, int cpu, int group_fd, unsigned long flags);",
    "long sys_mmap_pgoff(unsigned long addr, unsigned long len, unsigned long prot, unsigned long flags, unsigned long fd, unsigned long pgoff);",
    "long sys_old_mmap(struct mmap_arg_struct *arg);",
    "long sys_name_to_handle_at(int dfd, const char *name, struct file_handle *handle, int *mnt_id, int flag);",
    "long sys_open_by_handle_at(int mountdirfd, struct file_handle *handle, int flags);",
    "long sys_setns(int fd, int nstype);",
    "long sys_process_vm_readv(pid_t pid, const struct iovec *lvec, unsigned long liovcnt, const struct iovec *rvec, unsigned long riovcnt, unsigned long flags);",
    "long sys_process_vm_writev(pid_t pid, const struct iovec *lvec, unsigned long liovcnt, const struct iovec *rvec, unsigned long riovcnt, unsigned long flags);",
    "long sys_kcmp(pid_t pid1, pid_t pid2, int type, unsigned long idx1, unsigned long idx2);",
    "long sys_finit_module(int fd, const char *uargs, int flags);",
    "long sys_seccomp(unsigned int op, unsigned int flags, const char *uargs);",
    "long sys_getrandom(char *buf, size_t count, unsigned int flags);",
    "long sys_bpf(int cmd, struct bpf_attr *attr, unsigned int size);",
    "long sys_execveat(int dfd, const char *filename, const char *const *argv, const char *const *envp, int flags);",
    "long sys_membarrier(int cmd, int flags);",
    "long sys_copy_file_range(int fd_in, loff_t *off_in, int fd_out, loff_t *off_out, size_t len, unsigned int flags);",
    "long sys_mlock2(unsigned long start, size_t len, int flags);",
    "long sys_pkey_mprotect(unsigned long start, size_t len, unsigned long prot, int pkey);",
    "long sys_pkey_alloc(unsigned long flags, unsigned long init_val);",
    "long sys_pkey_free(int pkey);",
    "long sys_statx(int dfd, const char *path, unsigned flags, unsigned mask, struct statx *buffer);",
    # extra!
    "void *mmap(void *addr, size_t len, int prot, int flags, int fildes, off_t off);",
]

_amd64_fd_spots = \
{('accept', -1),
 ('accept', 0),
 ('accept4', -1),
 ('accept4', 0),
 ('bind', 0),
 ('bpf', -1),
 ('close', 0),
 ('connect', 0),
 ('creat', -1),
 ('dup', -1),
 ('dup', 0),
 ('dup2', -1),
 ('dup2', 0),
 ('dup2', 1),
 ('dup3', -1),
 ('dup3', 0),
 ('dup3', 1),
 ('epoll_create', -1),
 ('epoll_create1', -1),
 ('epoll_ctl', 0),
 ('epoll_ctl', 2),
 ('epoll_pwait', 0),
 ('epoll_wait', 0),
 ('eventfd', -1),
 ('eventfd2', -1),
 ('execveat', 0),
 ('faccessat', 0),
 ('fadvise64', 0),
 ('fallocate', 0),
 ('fanotify_init', -1),
 ('fanotify_mark', 0),
 ('fanotify_mark', 3),
 ('fchdir', 0),
 ('fchmod', 0),
 ('fchmodat', 0),
 ('fchown', 0),
 ('fchownat', 0),
 #('fcntl', -1),
 ('fcntl', 0),
 #('fcntl', 2),
 ('fdatasync', 0),
 ('fgetxattr', 0),
 ('finit_module', 0),
 ('flistxattr', 0),
 ('flock', 0),
 ('fremovexattr', 0),
 ('fsconfig', 0),
 ('fsconfig', 4),
 ('fsetxattr', 0),
 ('fsmount', -1),
 ('fsmount', 0),
 ('fsopen', -1),
 ('fspick', -1),
 ('fspick', 0),
 ('fstat', 0),
 ('fstatfs', 0),
 ('fsync', 0),
 ('ftruncate', 0),
 ('futimesat', 0),
 ('getdents', 0),
 ('getdents64', 0),
 ('getpeername', 0),
 ('getsockname', 0),
 ('getsockopt', 0),
 ('inotify_add_watch', 0),
 ('inotify_init', -1),
 ('inotify_init1', -1),
 ('inotify_rm_watch', 0),
 ('io_uring_enter', 0),
 ('io_uring_register', 0),
 ('io_uring_setup', -1),
 #('ioctl', -1),
 ('ioctl', 0),
 #('ioctl', 2),
 ('kcmp', 3),
 ('kcmp', 4),
 ('linkat', 0),
 ('linkat', 2),
 ('listen', 0),
 ('lseek', 0),
 ('memfd_create', -1),
 ('mkdirat', 0),
 ('mknodat', 0),
 ('mmap', 4),
 ('move_mount', 0),
 ('move_mount', 2),
 ('mq_getsetattr', 0),
 ('mq_notify', 0),
 ('mq_open', -1),
 ('mq_timedreceive', 0),
 ('mq_timedsend', 0),
 ('name_to_handle_at', 0),
 ('open', -1),
 ('open_by_handle_at', 0),
 ('open_tree', -1),
 ('open_tree', 0),
 ('openat', -1),
 ('openat', 0),
 ('perf_event_open', -1),
 ('perf_event_open', 1),
 ('perf_event_open', 3),
 ('pidfd_send_signal', 0),
 ('prctl', 2),
 ('pread64', 0),
 ('preadv', 0),
 ('pwrite64', 0),
 ('pwritev', 0),
 ('read', 0),
 ('readahead', 0),
 ('readlinkat', 0),
 ('readv', 0),
 ('recvfrom', 0),
 ('recvmmsg', 0),
 ('recvmsg', 0),
 ('renameat', 0),
 ('renameat', 2),
 ('renameat2', 0),
 ('renameat2', 2),
 ('sendfile', 0),
 ('sendfile', 1),
 ('sendmmsg', 0),
 ('sendmsg', 0),
 ('sendto', 0),
 ('setns', 0),
 ('setsockopt', 0),
 ('shutdown', 0),
 ('signalfd', -1),
 ('signalfd', 0),
 ('signalfd4', -1),
 ('signalfd4', 0),
 ('socket', -1),
 ('splice', 0),
 ('splice', 2),
 ('statx', 0),
 ('symlinkat', 1),
 ('sync_file_range', 0),
 ('syncfs', 0),
 ('syz_init_net_socket', -1),
 ('syz_kvm_setup_cpu', 0),
 ('syz_kvm_setup_cpu', 1),
 ('syz_open_dev', -1),
 ('syz_open_procfs', -1),
 ('syz_open_pts', -1),
 ('syz_open_pts', 0),
 ('tee', 0),
 ('tee', 1),
 ('timerfd_create', -1),
 ('timerfd_gettime', 0),
 ('timerfd_settime', 0),
 ('unlinkat', 0),
 ('userfaultfd', -1),
 ('utimensat', 0),
 ('vmsplice', 0),
 ('write', 0),
}


_syscall_abis: Dict[str,Dict[str,SimTypeFunction]] = {
    "amd64": {},
    "arm": {},
    "armhf": {},
    "i386": {},
    "mips-n32": {},
    "mips-n64": {},
    "mips-o32": {
        "set_thread_area": SimTypeFunction([SimTypeLong(signed=False)], SimTypeInt(signed=True), arg_names=["addr"]),
        "fstat64": SimTypeFunction([SimTypeFd(), SimTypePointer(SimStruct({}, name="stat64", pack=False, align=None), offset=0)], SimTypeLong(signed=True), arg_names=["fd", "statbuf"]),
    },
    "ppc": {},
    "ppc64": {},
    "s390": {},
    "s390x": {},
    "sparc": {},
    "sparc64": {},
    "riscv32": {},
}


for _abi, _special_decls in _syscall_abis.items():
    # first we initialize the syscall prototypes dict with the base syscalls
    lib.set_prototypes(_abi, _base_syscall_decls)
    # then we update the dict the abi-specific syscall prototypes
    lib.set_prototypes(_abi, _special_decls)


# python parse_syscalls_from_local_system.py >> linux_kernel.py

lib.add_number_mapping_from_dict("aarch64", {
    0: "io_setup",
    1: "io_destroy",
    2: "io_submit",
    3: "io_cancel",
    4: "io_getevents",
    5: "setxattr",
    6: "lsetxattr",
    7: "fsetxattr",
    8: "getxattr",
    9: "lgetxattr",
    10: "fgetxattr",
    11: "listxattr",
    12: "llistxattr",
    13: "flistxattr",
    14: "removexattr",
    15: "lremovexattr",
    16: "fremovexattr",
    17: "getcwd",
    18: "lookup_dcookie",
    19: "eventfd2",
    20: "epoll_create1",
    21: "epoll_ctl",
    22: "epoll_pwait",
    23: "dup",
    24: "dup3",
    25: "fcntl",
    26: "inotify_init1",
    27: "inotify_add_watch",
    28: "inotify_rm_watch",
    29: "ioctl",
    30: "ioprio_set",
    31: "ioprio_get",
    32: "flock",
    33: "mknodat",
    34: "mkdirat",
    35: "unlinkat",
    36: "symlinkat",
    37: "linkat",
    38: "renameat",
    39: "umount2",
    40: "mount",
    41: "pivot_root",
    42: "nfsservctl",
    43: "statfs",
    44: "fstatfs",
    45: "truncate",
    46: "ftruncate",
    47: "fallocate",
    48: "faccessat",
    49: "chdir",
    50: "fchdir",
    51: "chroot",
    52: "fchmod",
    53: "fchmodat",
    54: "fchownat",
    55: "fchown",
    56: "openat",
    57: "close",
    58: "vhangup",
    59: "pipe2",
    60: "quotactl",
    61: "getdents64",
    62: "lseek",
    63: "read",
    64: "write",
    65: "readv",
    66: "writev",
    67: "pread64",
    68: "pwrite64",
    69: "preadv",
    70: "pwritev",
    71: "sendfile",
    72: "pselect6",
    73: "ppoll",
    74: "signalfd4",
    75: "vmsplice",
    76: "splice",
    77: "tee",
    78: "readlinkat",
    79: "newfstatat",
    80: "fstat",
    81: "sync",
    82: "fsync",
    83: "fdatasync",
    84: "sync_file_range",
    85: "timerfd_create",
    86: "timerfd_settime",
    87: "timerfd_gettime",
    88: "utimensat",
    89: "acct",
    90: "capget",
    91: "capset",
    92: "personality",
    93: "exit",
    94: "exit_group",
    95: "waitid",
    96: "set_tid_address",
    97: "unshare",
    98: "futex",
    99: "set_robust_list",
    100: "get_robust_list",
    101: "nanosleep",
    102: "getitimer",
    103: "setitimer",
    104: "kexec_load",
    105: "init_module",
    106: "delete_module",
    107: "timer_create",
    108: "timer_gettime",
    109: "timer_getoverrun",
    110: "timer_settime",
    111: "timer_delete",
    112: "clock_settime",
    113: "clock_gettime",
    114: "clock_getres",
    115: "clock_nanosleep",
    116: "syslog",
    117: "ptrace",
    118: "sched_setparam",
    119: "sched_setscheduler",
    120: "sched_getscheduler",
    121: "sched_getparam",
    122: "sched_setaffinity",
    123: "sched_getaffinity",
    124: "sched_yield",
    125: "sched_get_priority_max",
    126: "sched_get_priority_min",
    127: "sched_rr_get_interval",
    128: "restart_syscall",
    129: "kill",
    130: "tkill",
    131: "tgkill",
    132: "sigaltstack",
    133: "rt_sigsuspend",
    134: "rt_sigaction",
    135: "rt_sigprocmask",
    136: "rt_sigpending",
    137: "rt_sigtimedwait",
    138: "rt_sigqueueinfo",
    139: "rt_sigreturn",
    140: "setpriority",
    141: "getpriority",
    142: "reboot",
    143: "setregid",
    144: "setgid",
    145: "setreuid",
    146: "setuid",
    147: "setresuid",
    148: "getresuid",
    149: "setresgid",
    150: "getresgid",
    151: "setfsuid",
    152: "setfsgid",
    153: "times",
    154: "setpgid",
    155: "getpgid",
    156: "getsid",
    157: "setsid",
    158: "getgroups",
    159: "setgroups",
    160: "uname",
    161: "sethostname",
    162: "setdomainname",
    163: "getrlimit",
    164: "setrlimit",
    165: "getrusage",
    166: "umask",
    167: "prctl",
    168: "getcpu",
    169: "gettimeofday",
    170: "settimeofday",
    171: "adjtimex",
    172: "getpid",
    173: "getppid",
    174: "getuid",
    175: "geteuid",
    176: "getgid",
    177: "getegid",
    178: "gettid",
    179: "sysinfo",
    180: "mq_open",
    181: "mq_unlink",
    182: "mq_timedsend",
    183: "mq_timedreceive",
    184: "mq_notify",
    185: "mq_getsetattr",
    186: "msgget",
    187: "msgctl",
    188: "msgrcv",
    189: "msgsnd",
    190: "semget",
    191: "semctl",
    192: "semtimedop",
    193: "semop",
    194: "shmget",
    195: "shmctl",
    196: "shmat",
    197: "shmdt",
    198: "socket",
    199: "socketpair",
    200: "bind",
    201: "listen",
    202: "accept",
    203: "connect",
    204: "getsockname",
    205: "getpeername",
    206: "sendto",
    207: "recvfrom",
    208: "setsockopt",
    209: "getsockopt",
    210: "shutdown",
    211: "sendmsg",
    212: "recvmsg",
    213: "readahead",
    214: "brk",
    215: "munmap",
    216: "mremap",
    217: "add_key",
    218: "request_key",
    219: "keyctl",
    220: "clone",
    221: "execve",
    222: "mmap",
    223: "fadvise64",
    224: "swapon",
    225: "swapoff",
    226: "mprotect",
    227: "msync",
    228: "mlock",
    229: "munlock",
    230: "mlockall",
    231: "munlockall",
    232: "mincore",
    233: "madvise",
    234: "remap_file_pages",
    235: "mbind",
    236: "get_mempolicy",
    237: "set_mempolicy",
    238: "migrate_pages",
    239: "move_pages",
    240: "rt_tgsigqueueinfo",
    241: "perf_event_open",
    242: "accept4",
    243: "recvmmsg",
    244: "arch_specific_syscall",
    260: "wait4",
    261: "prlimit64",
    262: "fanotify_init",
    263: "fanotify_mark",
    264: "name_to_handle_at",
    265: "open_by_handle_at",
    266: "clock_adjtime",
    267: "syncfs",
    268: "setns",
    269: "sendmmsg",
    270: "process_vm_readv",
    271: "process_vm_writev",
    272: "kcmp",
    273: "syscalls",
})

lib.add_number_mapping_from_dict("amd64", {
    0: "read",
    1: "write",
    2: "open",
    3: "close",
    4: "stat",
    5: "fstat",
    6: "lstat",
    7: "poll",
    8: "lseek",
    9: "mmap",
    10: "mprotect",
    11: "munmap",
    12: "brk",
    13: "rt_sigaction",
    14: "rt_sigprocmask",
    15: "rt_sigreturn",
    16: "ioctl",
    17: "pread64",
    18: "pwrite64",
    19: "readv",
    20: "writev",
    21: "access",
    22: "pipe",
    23: "select",
    24: "sched_yield",
    25: "mremap",
    26: "msync",
    27: "mincore",
    28: "madvise",
    29: "shmget",
    30: "shmat",
    31: "shmctl",
    32: "dup",
    33: "dup2",
    34: "pause",
    35: "nanosleep",
    36: "getitimer",
    37: "alarm",
    38: "setitimer",
    39: "getpid",
    40: "sendfile",
    41: "socket",
    42: "connect",
    43: "accept",
    44: "sendto",
    45: "recvfrom",
    46: "sendmsg",
    47: "recvmsg",
    48: "shutdown",
    49: "bind",
    50: "listen",
    51: "getsockname",
    52: "getpeername",
    53: "socketpair",
    54: "setsockopt",
    55: "getsockopt",
    56: "clone",
    57: "fork",
    58: "vfork",
    59: "execve",
    60: "exit",
    61: "wait4",
    62: "kill",
    63: "uname",
    64: "semget",
    65: "semop",
    66: "semctl",
    67: "shmdt",
    68: "msgget",
    69: "msgsnd",
    70: "msgrcv",
    71: "msgctl",
    72: "fcntl",
    73: "flock",
    74: "fsync",
    75: "fdatasync",
    76: "truncate",
    77: "ftruncate",
    78: "getdents",
    79: "getcwd",
    80: "chdir",
    81: "fchdir",
    82: "rename",
    83: "mkdir",
    84: "rmdir",
    85: "creat",
    86: "link",
    87: "unlink",
    88: "symlink",
    89: "readlink",
    90: "chmod",
    91: "fchmod",
    92: "chown",
    93: "fchown",
    94: "lchown",
    95: "umask",
    96: "gettimeofday",
    97: "getrlimit",
    98: "getrusage",
    99: "sysinfo",
    100: "times",
    101: "ptrace",
    102: "getuid",
    103: "syslog",
    104: "getgid",
    105: "setuid",
    106: "setgid",
    107: "geteuid",
    108: "getegid",
    109: "setpgid",
    110: "getppid",
    111: "getpgrp",
    112: "setsid",
    113: "setreuid",
    114: "setregid",
    115: "getgroups",
    116: "setgroups",
    117: "setresuid",
    118: "getresuid",
    119: "setresgid",
    120: "getresgid",
    121: "getpgid",
    122: "setfsuid",
    123: "setfsgid",
    124: "getsid",
    125: "capget",
    126: "capset",
    127: "rt_sigpending",
    128: "rt_sigtimedwait",
    129: "rt_sigqueueinfo",
    130: "rt_sigsuspend",
    131: "sigaltstack",
    132: "utime",
    133: "mknod",
    134: "uselib",
    135: "personality",
    136: "ustat",
    137: "statfs",
    138: "fstatfs",
    139: "sysfs",
    140: "getpriority",
    141: "setpriority",
    142: "sched_setparam",
    143: "sched_getparam",
    144: "sched_setscheduler",
    145: "sched_getscheduler",
    146: "sched_get_priority_max",
    147: "sched_get_priority_min",
    148: "sched_rr_get_interval",
    149: "mlock",
    150: "munlock",
    151: "mlockall",
    152: "munlockall",
    153: "vhangup",
    154: "modify_ldt",
    155: "pivot_root",
    156: "_sysctl",
    157: "prctl",
    158: "arch_prctl",
    159: "adjtimex",
    160: "setrlimit",
    161: "chroot",
    162: "sync",
    163: "acct",
    164: "settimeofday",
    165: "mount",
    166: "umount2",
    167: "swapon",
    168: "swapoff",
    169: "reboot",
    170: "sethostname",
    171: "setdomainname",
    172: "iopl",
    173: "ioperm",
    174: "create_module",
    175: "init_module",
    176: "delete_module",
    177: "get_kernel_syms",
    178: "query_module",
    179: "quotactl",
    180: "nfsservctl",
    181: "getpmsg",
    182: "putpmsg",
    183: "afs_syscall",
    184: "tuxcall",
    185: "security",
    186: "gettid",
    187: "readahead",
    188: "setxattr",
    189: "lsetxattr",
    190: "fsetxattr",
    191: "getxattr",
    192: "lgetxattr",
    193: "fgetxattr",
    194: "listxattr",
    195: "llistxattr",
    196: "flistxattr",
    197: "removexattr",
    198: "lremovexattr",
    199: "fremovexattr",
    200: "tkill",
    201: "time",
    202: "futex",
    203: "sched_setaffinity",
    204: "sched_getaffinity",
    205: "set_thread_area",
    206: "io_setup",
    207: "io_destroy",
    208: "io_getevents",
    209: "io_submit",
    210: "io_cancel",
    211: "get_thread_area",
    212: "lookup_dcookie",
    213: "epoll_create",
    214: "epoll_ctl_old",
    215: "epoll_wait_old",
    216: "remap_file_pages",
    217: "getdents64",
    218: "set_tid_address",
    219: "restart_syscall",
    220: "semtimedop",
    221: "fadvise64",
    222: "timer_create",
    223: "timer_settime",
    224: "timer_gettime",
    225: "timer_getoverrun",
    226: "timer_delete",
    227: "clock_settime",
    228: "clock_gettime",
    229: "clock_getres",
    230: "clock_nanosleep",
    231: "exit_group",
    232: "epoll_wait",
    233: "epoll_ctl",
    234: "tgkill",
    235: "utimes",
    236: "vserver",
    237: "mbind",
    238: "set_mempolicy",
    239: "get_mempolicy",
    240: "mq_open",
    241: "mq_unlink",
    242: "mq_timedsend",
    243: "mq_timedreceive",
    244: "mq_notify",
    245: "mq_getsetattr",
    246: "kexec_load",
    247: "waitid",
    248: "add_key",
    249: "request_key",
    250: "keyctl",
    251: "ioprio_set",
    252: "ioprio_get",
    253: "inotify_init",
    254: "inotify_add_watch",
    255: "inotify_rm_watch",
    256: "migrate_pages",
    257: "openat",
    258: "mkdirat",
    259: "mknodat",
    260: "fchownat",
    261: "futimesat",
    262: "newfstatat",
    263: "unlinkat",
    264: "renameat",
    265: "linkat",
    266: "symlinkat",
    267: "readlinkat",
    268: "fchmodat",
    269: "faccessat",
    270: "pselect6",
    271: "ppoll",
    272: "unshare",
    273: "set_robust_list",
    274: "get_robust_list",
    275: "splice",
    276: "tee",
    277: "sync_file_range",
    278: "vmsplice",
    279: "move_pages",
    280: "utimensat",
    281: "epoll_pwait",
    282: "signalfd",
    283: "timerfd_create",
    284: "eventfd",
    285: "fallocate",
    286: "timerfd_settime",
    287: "timerfd_gettime",
    288: "accept4",
    289: "signalfd4",
    290: "eventfd2",
    291: "epoll_create1",
    292: "dup3",
    293: "pipe2",
    294: "inotify_init1",
    295: "preadv",
    296: "pwritev",
    297: "rt_tgsigqueueinfo",
    298: "perf_event_open",
    299: "recvmmsg",
    300: "fanotify_init",
    301: "fanotify_mark",
    302: "prlimit64",
    303: "name_to_handle_at",
    304: "open_by_handle_at",
    305: "clock_adjtime",
    306: "syncfs",
    307: "sendmmsg",
    308: "setns",
    309: "getcpu",
    310: "process_vm_readv",
    311: "process_vm_writev",
    312: "kcmp",
    313: "finit_module",
    314: "sched_setattr",
    315: "sched_getattr",
    316: "renameat2",
    317: "seccomp",
    318: "getrandom",
    319: "memfd_create",
    320: "kexec_file_load",
    321: "bpf",
    322: "execveat",
    323: "userfaultfd",
    324: "membarrier",
    325: "mlock2",
    326: "copy_file_range",
    327: "preadv2",
    328: "pwritev2",
    329: "pkey_mprotect",
    330: "pkey_alloc",
    331: "pkey_free",
    332: "statx",
})

lib.add_number_mapping_from_dict("arm", {
    0: "restart_syscall",
    1: "exit",
    2: "fork",
    3: "read",
    4: "write",
    5: "open",
    6: "close",
    7: "waitpid",
    8: "creat",
    9: "link",
    10: "unlink",
    11: "execve",
    12: "chdir",
    13: "time",
    14: "mknod",
    15: "chmod",
    16: "lchown",
    17: "break",
    18: "oldstat",
    19: "lseek",
    20: "getpid",
    21: "mount",
    22: "umount",
    23: "setuid",
    24: "getuid",
    25: "stime",
    26: "ptrace",
    27: "alarm",
    28: "oldfstat",
    29: "pause",
    30: "utime",
    31: "stty",
    32: "gtty",
    33: "access",
    34: "nice",
    35: "ftime",
    36: "sync",
    37: "kill",
    38: "rename",
    39: "mkdir",
    40: "rmdir",
    41: "dup",
    42: "pipe",
    43: "times",
    44: "prof",
    45: "brk",
    46: "setgid",
    47: "getgid",
    48: "signal",
    49: "geteuid",
    50: "getegid",
    51: "acct",
    52: "umount2",
    53: "lock",
    54: "ioctl",
    55: "fcntl",
    56: "mpx",
    57: "setpgid",
    58: "ulimit",
    59: "oldolduname",
    60: "umask",
    61: "chroot",
    62: "ustat",
    63: "dup2",
    64: "getppid",
    65: "getpgrp",
    66: "setsid",
    67: "sigaction",
    68: "sgetmask",
    69: "ssetmask",
    70: "setreuid",
    71: "setregid",
    72: "sigsuspend",
    73: "sigpending",
    74: "sethostname",
    75: "setrlimit",
    76: "getrlimit",
    77: "getrusage",
    78: "gettimeofday",
    79: "settimeofday",
    80: "getgroups",
    81: "setgroups",
    82: "select",
    83: "symlink",
    84: "oldlstat",
    85: "readlink",
    86: "uselib",
    87: "swapon",
    88: "reboot",
    89: "readdir",
    90: "mmap",
    91: "munmap",
    92: "truncate",
    93: "ftruncate",
    94: "fchmod",
    95: "fchown",
    96: "getpriority",
    97: "setpriority",
    98: "profil",
    99: "statfs",
    100: "fstatfs",
    101: "ioperm",
    102: "socketcall",
    103: "syslog",
    104: "setitimer",
    105: "getitimer",
    106: "stat",
    107: "lstat",
    108: "fstat",
    109: "olduname",
    110: "iopl",
    111: "vhangup",
    112: "idle",
    113: "syscall",
    114: "wait4",
    115: "swapoff",
    116: "sysinfo",
    117: "ipc",
    118: "fsync",
    119: "sigreturn",
    120: "clone",
    121: "setdomainname",
    122: "uname",
    123: "modify_ldt",
    124: "adjtimex",
    125: "mprotect",
    126: "sigprocmask",
    127: "create_module",
    128: "init_module",
    129: "delete_module",
    130: "get_kernel_syms",
    131: "quotactl",
    132: "getpgid",
    133: "fchdir",
    134: "bdflush",
    135: "sysfs",
    136: "personality",
    137: "afs_syscall",
    138: "setfsuid",
    139: "setfsgid",
    140: "_llseek",
    141: "getdents",
    142: "_newselect",
    143: "flock",
    144: "msync",
    145: "readv",
    146: "writev",
    147: "getsid",
    148: "fdatasync",
    149: "_sysctl",
    150: "mlock",
    151: "munlock",
    152: "mlockall",
    153: "munlockall",
    154: "sched_setparam",
    155: "sched_getparam",
    156: "sched_setscheduler",
    157: "sched_getscheduler",
    158: "sched_yield",
    159: "sched_get_priority_max",
    160: "sched_get_priority_min",
    161: "sched_rr_get_interval",
    162: "nanosleep",
    163: "mremap",
    164: "setresuid",
    165: "getresuid",
    166: "vm86",
    167: "query_module",
    168: "poll",
    169: "nfsservctl",
    170: "setresgid",
    171: "getresgid",
    172: "prctl",
    173: "rt_sigreturn",
    174: "rt_sigaction",
    175: "rt_sigprocmask",
    176: "rt_sigpending",
    177: "rt_sigtimedwait",
    178: "rt_sigqueueinfo",
    179: "rt_sigsuspend",
    180: "pread64",
    181: "pwrite64",
    182: "chown",
    183: "getcwd",
    184: "capget",
    185: "capset",
    186: "sigaltstack",
    187: "sendfile",
    190: "vfork",
    191: "ugetrlimit",
    192: "mmap2",
    193: "truncate64",
    194: "ftruncate64",
    195: "stat64",
    196: "lstat64",
    197: "fstat64",
    198: "lchown32",
    199: "getuid32",
    200: "getgid32",
    201: "geteuid32",
    202: "getegid32",
    203: "setreuid32",
    204: "setregid32",
    205: "getgroups32",
    206: "setgroups32",
    207: "fchown32",
    208: "setresuid32",
    209: "getresuid32",
    210: "setresgid32",
    211: "getresgid32",
    212: "chown32",
    213: "setuid32",
    214: "setgid32",
    215: "setfsuid32",
    216: "setfsgid32",
    217: "getdents64",
    218: "pivot_root",
    219: "mincore",
    220: "madvise",
    221: "fcntl64",
    224: "gettid",
    225: "readahead",
    226: "setxattr",
    227: "lsetxattr",
    228: "fsetxattr",
    229: "getxattr",
    230: "lgetxattr",
    231: "fgetxattr",
    232: "listxattr",
    233: "llistxattr",
    234: "flistxattr",
    235: "removexattr",
    236: "lremovexattr",
    237: "fremovexattr",
    238: "tkill",
    239: "sendfile64",
    240: "futex",
    241: "sched_setaffinity",
    242: "sched_getaffinity",
    243: "io_setup",
    244: "io_destroy",
    245: "io_getevents",
    246: "io_submit",
    247: "io_cancel",
    248: "exit_group",
    249: "lookup_dcookie",
    250: "epoll_create",
    251: "epoll_ctl",
    252: "epoll_wait",
    253: "remap_file_pages",
    256: "set_tid_address",
    257: "timer_create",
    258: "timer_settime",
    259: "timer_gettime",
    260: "timer_getoverrun",
    261: "timer_delete",
    262: "clock_settime",
    263: "clock_gettime",
    264: "clock_getres",
    265: "clock_nanosleep",
    266: "statfs64",
    267: "fstatfs64",
    268: "tgkill",
    269: "utimes",
    270: "arm_fadvise64_64",
    271: "pciconfig_iobase",
    272: "pciconfig_read",
    273: "pciconfig_write",
    274: "mq_open",
    275: "mq_unlink",
    276: "mq_timedsend",
    277: "mq_timedreceive",
    278: "mq_notify",
    279: "mq_getsetattr",
    280: "waitid",
    281: "socket",
    282: "bind",
    283: "connect",
    284: "listen",
    285: "accept",
    286: "getsockname",
    287: "getpeername",
    288: "socketpair",
    289: "send",
    290: "sendto",
    291: "recv",
    292: "recvfrom",
    293: "shutdown",
    294: "setsockopt",
    295: "getsockopt",
    296: "sendmsg",
    297: "recvmsg",
    298: "semop",
    299: "semget",
    300: "semctl",
    301: "msgsnd",
    302: "msgrcv",
    303: "msgget",
    304: "msgctl",
    305: "shmat",
    306: "shmdt",
    307: "shmget",
    308: "shmctl",
    309: "add_key",
    310: "request_key",
    311: "keyctl",
    312: "semtimedop",
    313: "vserver",
    314: "ioprio_set",
    315: "ioprio_get",
    316: "inotify_init",
    317: "inotify_add_watch",
    318: "inotify_rm_watch",
    319: "mbind",
    320: "get_mempolicy",
    321: "set_mempolicy",
    322: "openat",
    323: "mkdirat",
    324: "mknodat",
    325: "fchownat",
    326: "futimesat",
    327: "fstatat64",
    328: "unlinkat",
    329: "renameat",
    330: "linkat",
    331: "symlinkat",
    332: "readlinkat",
    333: "fchmodat",
    334: "faccessat",
    335: "pselect6",
    336: "ppoll",
    337: "unshare",
    338: "set_robust_list",
    339: "get_robust_list",
    340: "splice",
    341: "arm_sync_file_range",
    342: "tee",
    343: "vmsplice",
    344: "move_pages",
    345: "getcpu",
    346: "epoll_pwait",
    347: "kexec_load",
    348: "utimensat",
    349: "signalfd",
    350: "timerfd_create",
    351: "eventfd",
    352: "fallocate",
    353: "timerfd_settime",
    354: "timerfd_gettime",
    355: "signalfd4",
    356: "eventfd2",
    357: "epoll_create1",
    358: "dup3",
    359: "pipe2",
    360: "inotify_init1",
    361: "preadv",
    362: "pwritev",
    363: "rt_tgsigqueueinfo",
    364: "perf_event_open",
    365: "recvmmsg",
    366: "accept4",
    367: "fanotify_init",
    368: "fanotify_mark",
    369: "prlimit64",
    370: "name_to_handle_at",
    371: "open_by_handle_at",
    372: "clock_adjtime",
    373: "syncfs",
    374: "sendmmsg",
    375: "setns",
    376: "process_vm_readv",
    377: "process_vm_writev",
    378: "kcmp",
    379: "finit_module",
})

lib.add_number_mapping_from_dict("armhf", {
    9437184: "restart_syscall",
    9437185: "exit",
    9437186: "fork",
    9437187: "read",
    9437188: "write",
    9437189: "open",
    9437190: "close",
    9437191: "waitpid",
    9437192: "creat",
    9437193: "link",
    9437194: "unlink",
    9437195: "execve",
    9437196: "chdir",
    9437197: "time",
    9437198: "mknod",
    9437199: "chmod",
    9437200: "lchown",
    9437201: "break",
    9437202: "oldstat",
    9437203: "lseek",
    9437204: "getpid",
    9437205: "mount",
    9437206: "umount",
    9437207: "setuid",
    9437208: "getuid",
    9437209: "stime",
    9437210: "ptrace",
    9437211: "alarm",
    9437212: "oldfstat",
    9437213: "pause",
    9437214: "utime",
    9437215: "stty",
    9437216: "gtty",
    9437217: "access",
    9437218: "nice",
    9437219: "ftime",
    9437220: "sync",
    9437221: "kill",
    9437222: "rename",
    9437223: "mkdir",
    9437224: "rmdir",
    9437225: "dup",
    9437226: "pipe",
    9437227: "times",
    9437228: "prof",
    9437229: "brk",
    9437230: "setgid",
    9437231: "getgid",
    9437232: "signal",
    9437233: "geteuid",
    9437234: "getegid",
    9437235: "acct",
    9437236: "umount2",
    9437237: "lock",
    9437238: "ioctl",
    9437239: "fcntl",
    9437240: "mpx",
    9437241: "setpgid",
    9437242: "ulimit",
    9437243: "oldolduname",
    9437244: "umask",
    9437245: "chroot",
    9437246: "ustat",
    9437247: "dup2",
    9437248: "getppid",
    9437249: "getpgrp",
    9437250: "setsid",
    9437251: "sigaction",
    9437252: "sgetmask",
    9437253: "ssetmask",
    9437254: "setreuid",
    9437255: "setregid",
    9437256: "sigsuspend",
    9437257: "sigpending",
    9437258: "sethostname",
    9437259: "setrlimit",
    9437260: "getrlimit",
    9437261: "getrusage",
    9437262: "gettimeofday",
    9437263: "settimeofday",
    9437264: "getgroups",
    9437265: "setgroups",
    9437266: "select",
    9437267: "symlink",
    9437268: "oldlstat",
    9437269: "readlink",
    9437270: "uselib",
    9437271: "swapon",
    9437272: "reboot",
    9437273: "readdir",
    9437274: "mmap",
    9437275: "munmap",
    9437276: "truncate",
    9437277: "ftruncate",
    9437278: "fchmod",
    9437279: "fchown",
    9437280: "getpriority",
    9437281: "setpriority",
    9437282: "profil",
    9437283: "statfs",
    9437284: "fstatfs",
    9437285: "ioperm",
    9437286: "socketcall",
    9437287: "syslog",
    9437288: "setitimer",
    9437289: "getitimer",
    9437290: "stat",
    9437291: "lstat",
    9437292: "fstat",
    9437293: "olduname",
    9437294: "iopl",
    9437295: "vhangup",
    9437296: "idle",
    9437297: "syscall",
    9437298: "wait4",
    9437299: "swapoff",
    9437300: "sysinfo",
    9437301: "ipc",
    9437302: "fsync",
    9437303: "sigreturn",
    9437304: "clone",
    9437305: "setdomainname",
    9437306: "uname",
    9437307: "modify_ldt",
    9437308: "adjtimex",
    9437309: "mprotect",
    9437310: "sigprocmask",
    9437311: "create_module",
    9437312: "init_module",
    9437313: "delete_module",
    9437314: "get_kernel_syms",
    9437315: "quotactl",
    9437316: "getpgid",
    9437317: "fchdir",
    9437318: "bdflush",
    9437319: "sysfs",
    9437320: "personality",
    9437321: "afs_syscall",
    9437322: "setfsuid",
    9437323: "setfsgid",
    9437324: "_llseek",
    9437325: "getdents",
    9437326: "_newselect",
    9437327: "flock",
    9437328: "msync",
    9437329: "readv",
    9437330: "writev",
    9437331: "getsid",
    9437332: "fdatasync",
    9437333: "_sysctl",
    9437334: "mlock",
    9437335: "munlock",
    9437336: "mlockall",
    9437337: "munlockall",
    9437338: "sched_setparam",
    9437339: "sched_getparam",
    9437340: "sched_setscheduler",
    9437341: "sched_getscheduler",
    9437342: "sched_yield",
    9437343: "sched_get_priority_max",
    9437344: "sched_get_priority_min",
    9437345: "sched_rr_get_interval",
    9437346: "nanosleep",
    9437347: "mremap",
    9437348: "setresuid",
    9437349: "getresuid",
    9437350: "vm86",
    9437351: "query_module",
    9437352: "poll",
    9437353: "nfsservctl",
    9437354: "setresgid",
    9437355: "getresgid",
    9437356: "prctl",
    9437357: "rt_sigreturn",
    9437358: "rt_sigaction",
    9437359: "rt_sigprocmask",
    9437360: "rt_sigpending",
    9437361: "rt_sigtimedwait",
    9437362: "rt_sigqueueinfo",
    9437363: "rt_sigsuspend",
    9437364: "pread64",
    9437365: "pwrite64",
    9437366: "chown",
    9437367: "getcwd",
    9437368: "capget",
    9437369: "capset",
    9437370: "sigaltstack",
    9437371: "sendfile",
    9437374: "vfork",
    9437375: "ugetrlimit",
    9437376: "mmap2",
    9437377: "truncate64",
    9437378: "ftruncate64",
    9437379: "stat64",
    9437380: "lstat64",
    9437381: "fstat64",
    9437382: "lchown32",
    9437383: "getuid32",
    9437384: "getgid32",
    9437385: "geteuid32",
    9437386: "getegid32",
    9437387: "setreuid32",
    9437388: "setregid32",
    9437389: "getgroups32",
    9437390: "setgroups32",
    9437391: "fchown32",
    9437392: "setresuid32",
    9437393: "getresuid32",
    9437394: "setresgid32",
    9437395: "getresgid32",
    9437396: "chown32",
    9437397: "setuid32",
    9437398: "setgid32",
    9437399: "setfsuid32",
    9437400: "setfsgid32",
    9437401: "getdents64",
    9437402: "pivot_root",
    9437403: "mincore",
    9437404: "madvise",
    9437405: "fcntl64",
    9437408: "gettid",
    9437409: "readahead",
    9437410: "setxattr",
    9437411: "lsetxattr",
    9437412: "fsetxattr",
    9437413: "getxattr",
    9437414: "lgetxattr",
    9437415: "fgetxattr",
    9437416: "listxattr",
    9437417: "llistxattr",
    9437418: "flistxattr",
    9437419: "removexattr",
    9437420: "lremovexattr",
    9437421: "fremovexattr",
    9437422: "tkill",
    9437423: "sendfile64",
    9437424: "futex",
    9437425: "sched_setaffinity",
    9437426: "sched_getaffinity",
    9437427: "io_setup",
    9437428: "io_destroy",
    9437429: "io_getevents",
    9437430: "io_submit",
    9437431: "io_cancel",
    9437432: "exit_group",
    9437433: "lookup_dcookie",
    9437434: "epoll_create",
    9437435: "epoll_ctl",
    9437436: "epoll_wait",
    9437437: "remap_file_pages",
    9437440: "set_tid_address",
    9437441: "timer_create",
    9437442: "timer_settime",
    9437443: "timer_gettime",
    9437444: "timer_getoverrun",
    9437445: "timer_delete",
    9437446: "clock_settime",
    9437447: "clock_gettime",
    9437448: "clock_getres",
    9437449: "clock_nanosleep",
    9437450: "statfs64",
    9437451: "fstatfs64",
    9437452: "tgkill",
    9437453: "utimes",
    9437454: "arm_fadvise64_64",
    9437455: "pciconfig_iobase",
    9437456: "pciconfig_read",
    9437457: "pciconfig_write",
    9437458: "mq_open",
    9437459: "mq_unlink",
    9437460: "mq_timedsend",
    9437461: "mq_timedreceive",
    9437462: "mq_notify",
    9437463: "mq_getsetattr",
    9437464: "waitid",
    9437465: "socket",
    9437466: "bind",
    9437467: "connect",
    9437468: "listen",
    9437469: "accept",
    9437470: "getsockname",
    9437471: "getpeername",
    9437472: "socketpair",
    9437473: "send",
    9437474: "sendto",
    9437475: "recv",
    9437476: "recvfrom",
    9437477: "shutdown",
    9437478: "setsockopt",
    9437479: "getsockopt",
    9437480: "sendmsg",
    9437481: "recvmsg",
    9437482: "semop",
    9437483: "semget",
    9437484: "semctl",
    9437485: "msgsnd",
    9437486: "msgrcv",
    9437487: "msgget",
    9437488: "msgctl",
    9437489: "shmat",
    9437490: "shmdt",
    9437491: "shmget",
    9437492: "shmctl",
    9437493: "add_key",
    9437494: "request_key",
    9437495: "keyctl",
    9437496: "semtimedop",
    9437497: "vserver",
    9437498: "ioprio_set",
    9437499: "ioprio_get",
    9437500: "inotify_init",
    9437501: "inotify_add_watch",
    9437502: "inotify_rm_watch",
    9437503: "mbind",
    9437504: "get_mempolicy",
    9437505: "set_mempolicy",
    9437506: "openat",
    9437507: "mkdirat",
    9437508: "mknodat",
    9437509: "fchownat",
    9437510: "futimesat",
    9437511: "fstatat64",
    9437512: "unlinkat",
    9437513: "renameat",
    9437514: "linkat",
    9437515: "symlinkat",
    9437516: "readlinkat",
    9437517: "fchmodat",
    9437518: "faccessat",
    9437519: "pselect6",
    9437520: "ppoll",
    9437521: "unshare",
    9437522: "set_robust_list",
    9437523: "get_robust_list",
    9437524: "splice",
    9437525: "arm_sync_file_range",
    9437526: "tee",
    9437527: "vmsplice",
    9437528: "move_pages",
    9437529: "getcpu",
    9437530: "epoll_pwait",
    9437531: "kexec_load",
    9437532: "utimensat",
    9437533: "signalfd",
    9437534: "timerfd_create",
    9437535: "eventfd",
    9437536: "fallocate",
    9437537: "timerfd_settime",
    9437538: "timerfd_gettime",
    9437539: "signalfd4",
    9437540: "eventfd2",
    9437541: "epoll_create1",
    9437542: "dup3",
    9437543: "pipe2",
    9437544: "inotify_init1",
    9437545: "preadv",
    9437546: "pwritev",
    9437547: "rt_tgsigqueueinfo",
    9437548: "perf_event_open",
    9437549: "recvmmsg",
    9437550: "accept4",
    9437551: "fanotify_init",
    9437552: "fanotify_mark",
    9437553: "prlimit64",
    9437554: "name_to_handle_at",
    9437555: "open_by_handle_at",
    9437556: "clock_adjtime",
    9437557: "syncfs",
    9437558: "sendmmsg",
    9437559: "setns",
    9437560: "process_vm_readv",
    9437561: "process_vm_writev",
    9437562: "kcmp",
    9437563: "finit_module",
})

lib.add_number_mapping_from_dict("i386", {
    0: "restart_syscall",
    1: "exit",
    2: "fork",
    3: "read",
    4: "write",
    5: "open",
    6: "close",
    7: "waitpid",
    8: "creat",
    9: "link",
    10: "unlink",
    11: "execve",
    12: "chdir",
    13: "time",
    14: "mknod",
    15: "chmod",
    16: "lchown",
    17: "break",
    18: "oldstat",
    19: "lseek",
    20: "getpid",
    21: "mount",
    22: "umount",
    23: "setuid",
    24: "getuid",
    25: "stime",
    26: "ptrace",
    27: "alarm",
    28: "oldfstat",
    29: "pause",
    30: "utime",
    31: "stty",
    32: "gtty",
    33: "access",
    34: "nice",
    35: "ftime",
    36: "sync",
    37: "kill",
    38: "rename",
    39: "mkdir",
    40: "rmdir",
    41: "dup",
    42: "pipe",
    43: "times",
    44: "prof",
    45: "brk",
    46: "setgid",
    47: "getgid",
    48: "signal",
    49: "geteuid",
    50: "getegid",
    51: "acct",
    52: "umount2",
    53: "lock",
    54: "ioctl",
    55: "fcntl",
    56: "mpx",
    57: "setpgid",
    58: "ulimit",
    59: "oldolduname",
    60: "umask",
    61: "chroot",
    62: "ustat",
    63: "dup2",
    64: "getppid",
    65: "getpgrp",
    66: "setsid",
    67: "sigaction",
    68: "sgetmask",
    69: "ssetmask",
    70: "setreuid",
    71: "setregid",
    72: "sigsuspend",
    73: "sigpending",
    74: "sethostname",
    75: "setrlimit",
    76: "getrlimit",
    77: "getrusage",
    78: "gettimeofday",
    79: "settimeofday",
    80: "getgroups",
    81: "setgroups",
    82: "select",
    83: "symlink",
    84: "oldlstat",
    85: "readlink",
    86: "uselib",
    87: "swapon",
    88: "reboot",
    89: "readdir",
    90: "old_mmap",
    91: "munmap",
    92: "truncate",
    93: "ftruncate",
    94: "fchmod",
    95: "fchown",
    96: "getpriority",
    97: "setpriority",
    98: "profil",
    99: "statfs",
    100: "fstatfs",
    101: "ioperm",
    102: "socketcall",
    103: "syslog",
    104: "setitimer",
    105: "getitimer",
    106: "stat",
    107: "lstat",
    108: "fstat",
    109: "olduname",
    110: "iopl",
    111: "vhangup",
    112: "idle",
    113: "vm86old",
    114: "wait4",
    115: "swapoff",
    116: "sysinfo",
    117: "ipc",
    118: "fsync",
    119: "sigreturn",
    120: "clone",
    121: "setdomainname",
    122: "uname",
    123: "modify_ldt",
    124: "adjtimex",
    125: "mprotect",
    126: "sigprocmask",
    127: "create_module",
    128: "init_module",
    129: "delete_module",
    130: "get_kernel_syms",
    131: "quotactl",
    132: "getpgid",
    133: "fchdir",
    134: "bdflush",
    135: "sysfs",
    136: "personality",
    137: "afs_syscall",
    138: "setfsuid",
    139: "setfsgid",
    140: "_llseek",
    141: "getdents",
    142: "_newselect",
    143: "flock",
    144: "msync",
    145: "readv",
    146: "writev",
    147: "getsid",
    148: "fdatasync",
    149: "_sysctl",
    150: "mlock",
    151: "munlock",
    152: "mlockall",
    153: "munlockall",
    154: "sched_setparam",
    155: "sched_getparam",
    156: "sched_setscheduler",
    157: "sched_getscheduler",
    158: "sched_yield",
    159: "sched_get_priority_max",
    160: "sched_get_priority_min",
    161: "sched_rr_get_interval",
    162: "nanosleep",
    163: "mremap",
    164: "setresuid",
    165: "getresuid",
    166: "vm86",
    167: "query_module",
    168: "poll",
    169: "nfsservctl",
    170: "setresgid",
    171: "getresgid",
    172: "prctl",
    173: "rt_sigreturn",
    174: "rt_sigaction",
    175: "rt_sigprocmask",
    176: "rt_sigpending",
    177: "rt_sigtimedwait",
    178: "rt_sigqueueinfo",
    179: "rt_sigsuspend",
    180: "pread64",
    181: "pwrite64",
    182: "chown",
    183: "getcwd",
    184: "capget",
    185: "capset",
    186: "sigaltstack",
    187: "sendfile",
    188: "getpmsg",
    189: "putpmsg",
    190: "vfork",
    191: "ugetrlimit",
    192: "mmap2",
    193: "truncate64",
    194: "ftruncate64",
    195: "stat64",
    196: "lstat64",
    197: "fstat64",
    198: "lchown32",
    199: "getuid32",
    200: "getgid32",
    201: "geteuid32",
    202: "getegid32",
    203: "setreuid32",
    204: "setregid32",
    205: "getgroups32",
    206: "setgroups32",
    207: "fchown32",
    208: "setresuid32",
    209: "getresuid32",
    210: "setresgid32",
    211: "getresgid32",
    212: "chown32",
    213: "setuid32",
    214: "setgid32",
    215: "setfsuid32",
    216: "setfsgid32",
    217: "pivot_root",
    218: "mincore",
    219: "madvise",
    220: "getdents64",
    221: "fcntl64",
    224: "gettid",
    225: "readahead",
    226: "setxattr",
    227: "lsetxattr",
    228: "fsetxattr",
    229: "getxattr",
    230: "lgetxattr",
    231: "fgetxattr",
    232: "listxattr",
    233: "llistxattr",
    234: "flistxattr",
    235: "removexattr",
    236: "lremovexattr",
    237: "fremovexattr",
    238: "tkill",
    239: "sendfile64",
    240: "futex",
    241: "sched_setaffinity",
    242: "sched_getaffinity",
    243: "set_thread_area",
    244: "get_thread_area",
    245: "io_setup",
    246: "io_destroy",
    247: "io_getevents",
    248: "io_submit",
    249: "io_cancel",
    250: "fadvise64",
    252: "exit_group",
    253: "lookup_dcookie",
    254: "epoll_create",
    255: "epoll_ctl",
    256: "epoll_wait",
    257: "remap_file_pages",
    258: "set_tid_address",
    259: "timer_create",
    260: "timer_settime",
    261: "timer_gettime",
    262: "timer_getoverrun",
    263: "timer_delete",
    264: "clock_settime",
    265: "clock_gettime",
    266: "clock_getres",
    267: "clock_nanosleep",
    268: "statfs64",
    269: "fstatfs64",
    270: "tgkill",
    271: "utimes",
    272: "fadvise64_64",
    273: "vserver",
    274: "mbind",
    275: "get_mempolicy",
    276: "set_mempolicy",
    277: "mq_open",
    278: "mq_unlink",
    279: "mq_timedsend",
    280: "mq_timedreceive",
    281: "mq_notify",
    282: "mq_getsetattr",
    283: "kexec_load",
    284: "waitid",
    286: "add_key",
    287: "request_key",
    288: "keyctl",
    289: "ioprio_set",
    290: "ioprio_get",
    291: "inotify_init",
    292: "inotify_add_watch",
    293: "inotify_rm_watch",
    294: "migrate_pages",
    295: "openat",
    296: "mkdirat",
    297: "mknodat",
    298: "fchownat",
    299: "futimesat",
    300: "fstatat64",
    301: "unlinkat",
    302: "renameat",
    303: "linkat",
    304: "symlinkat",
    305: "readlinkat",
    306: "fchmodat",
    307: "faccessat",
    308: "pselect6",
    309: "ppoll",
    310: "unshare",
    311: "set_robust_list",
    312: "get_robust_list",
    313: "splice",
    314: "sync_file_range",
    315: "tee",
    316: "vmsplice",
    317: "move_pages",
    318: "getcpu",
    319: "epoll_pwait",
    320: "utimensat",
    321: "signalfd",
    322: "timerfd_create",
    323: "eventfd",
    324: "fallocate",
    325: "timerfd_settime",
    326: "timerfd_gettime",
    327: "signalfd4",
    328: "eventfd2",
    329: "epoll_create1",
    330: "dup3",
    331: "pipe2",
    332: "inotify_init1",
    333: "preadv",
    334: "pwritev",
    335: "rt_tgsigqueueinfo",
    336: "perf_event_open",
    337: "recvmmsg",
    338: "fanotify_init",
    339: "fanotify_mark",
    340: "prlimit64",
    341: "name_to_handle_at",
    342: "open_by_handle_at",
    343: "clock_adjtime",
    344: "syncfs",
    345: "sendmmsg",
    346: "setns",
    347: "process_vm_readv",
    348: "process_vm_writev",
    349: "kcmp",
    350: "finit_module",
    351: "sched_setattr",
    352: "sched_getattr",
    353: "renameat2",
    354: "seccomp",
    355: "getrandom",
    356: "memfd_create",
    357: "bpf",
    358: "execveat",
    359: "socket",
    360: "socketpair",
    361: "bind",
    362: "connect",
    363: "listen",
    364: "accept4",
    365: "getsockopt",
    366: "setsockopt",
    367: "getsockname",
    368: "getpeername",
    369: "sendto",
    370: "sendmsg",
    371: "recvfrom",
    372: "recvmsg",
    373: "shutdown",
    374: "userfaultfd",
    375: "membarrier",
    376: "mlock2",
    377: "copy_file_range",
    378: "preadv2",
    379: "pwritev2",
    380: "pkey_mprotect",
    381: "pkey_alloc",
    382: "pkey_free",
    383: "statx",
    384: "arch_prctl",
})

lib.add_number_mapping_from_dict("mips-n32", {
    6000: "read",
    6001: "write",
    6002: "open",
    6003: "close",
    6004: "stat",
    6005: "fstat",
    6006: "lstat",
    6007: "poll",
    6008: "lseek",
    6009: "mmap",
    6010: "mprotect",
    6011: "munmap",
    6012: "brk",
    6013: "rt_sigaction",
    6014: "rt_sigprocmask",
    6015: "ioctl",
    6016: "pread64",
    6017: "pwrite64",
    6018: "readv",
    6019: "writev",
    6020: "access",
    6021: "pipe",
    6022: "_newselect",
    6023: "sched_yield",
    6024: "mremap",
    6025: "msync",
    6026: "mincore",
    6027: "madvise",
    6028: "shmget",
    6029: "shmat",
    6030: "shmctl",
    6031: "dup",
    6032: "dup2",
    6033: "pause",
    6034: "nanosleep",
    6035: "getitimer",
    6036: "setitimer",
    6037: "alarm",
    6038: "getpid",
    6039: "sendfile",
    6040: "socket",
    6041: "connect",
    6042: "accept",
    6043: "sendto",
    6044: "recvfrom",
    6045: "sendmsg",
    6046: "recvmsg",
    6047: "shutdown",
    6048: "bind",
    6049: "listen",
    6050: "getsockname",
    6051: "getpeername",
    6052: "socketpair",
    6053: "setsockopt",
    6054: "getsockopt",
    6055: "clone",
    6056: "fork",
    6057: "execve",
    6058: "exit",
    6059: "wait4",
    6060: "kill",
    6061: "uname",
    6062: "semget",
    6063: "semop",
    6064: "semctl",
    6065: "shmdt",
    6066: "msgget",
    6067: "msgsnd",
    6068: "msgrcv",
    6069: "msgctl",
    6070: "fcntl",
    6071: "flock",
    6072: "fsync",
    6073: "fdatasync",
    6074: "truncate",
    6075: "ftruncate",
    6076: "getdents",
    6077: "getcwd",
    6078: "chdir",
    6079: "fchdir",
    6080: "rename",
    6081: "mkdir",
    6082: "rmdir",
    6083: "creat",
    6084: "link",
    6085: "unlink",
    6086: "symlink",
    6087: "readlink",
    6088: "chmod",
    6089: "fchmod",
    6090: "chown",
    6091: "fchown",
    6092: "lchown",
    6093: "umask",
    6094: "gettimeofday",
    6095: "getrlimit",
    6096: "getrusage",
    6097: "sysinfo",
    6098: "times",
    6099: "ptrace",
    6100: "getuid",
    6101: "syslog",
    6102: "getgid",
    6103: "setuid",
    6104: "setgid",
    6105: "geteuid",
    6106: "getegid",
    6107: "setpgid",
    6108: "getppid",
    6109: "getpgrp",
    6110: "setsid",
    6111: "setreuid",
    6112: "setregid",
    6113: "getgroups",
    6114: "setgroups",
    6115: "setresuid",
    6116: "getresuid",
    6117: "setresgid",
    6118: "getresgid",
    6119: "getpgid",
    6120: "setfsuid",
    6121: "setfsgid",
    6122: "getsid",
    6123: "capget",
    6124: "capset",
    6125: "rt_sigpending",
    6126: "rt_sigtimedwait",
    6127: "rt_sigqueueinfo",
    6128: "rt_sigsuspend",
    6129: "sigaltstack",
    6130: "utime",
    6131: "mknod",
    6132: "personality",
    6133: "ustat",
    6134: "statfs",
    6135: "fstatfs",
    6136: "sysfs",
    6137: "getpriority",
    6138: "setpriority",
    6139: "sched_setparam",
    6140: "sched_getparam",
    6141: "sched_setscheduler",
    6142: "sched_getscheduler",
    6143: "sched_get_priority_max",
    6144: "sched_get_priority_min",
    6145: "sched_rr_get_interval",
    6146: "mlock",
    6147: "munlock",
    6148: "mlockall",
    6149: "munlockall",
    6150: "vhangup",
    6151: "pivot_root",
    6152: "_sysctl",
    6153: "prctl",
    6154: "adjtimex",
    6155: "setrlimit",
    6156: "chroot",
    6157: "sync",
    6158: "acct",
    6159: "settimeofday",
    6160: "mount",
    6161: "umount2",
    6162: "swapon",
    6163: "swapoff",
    6164: "reboot",
    6165: "sethostname",
    6166: "setdomainname",
    6167: "create_module",
    6168: "init_module",
    6169: "delete_module",
    6170: "get_kernel_syms",
    6171: "query_module",
    6172: "quotactl",
    6173: "nfsservctl",
    6174: "getpmsg",
    6175: "putpmsg",
    6176: "afs_syscall",
    6177: "reserved177",
    6178: "gettid",
    6179: "readahead",
    6180: "setxattr",
    6181: "lsetxattr",
    6182: "fsetxattr",
    6183: "getxattr",
    6184: "lgetxattr",
    6185: "fgetxattr",
    6186: "listxattr",
    6187: "llistxattr",
    6188: "flistxattr",
    6189: "removexattr",
    6190: "lremovexattr",
    6191: "fremovexattr",
    6192: "tkill",
    6193: "reserved193",
    6194: "futex",
    6195: "sched_setaffinity",
    6196: "sched_getaffinity",
    6197: "cacheflush",
    6198: "cachectl",
    6199: "sysmips",
    6200: "io_setup",
    6201: "io_destroy",
    6202: "io_getevents",
    6203: "io_submit",
    6204: "io_cancel",
    6205: "exit_group",
    6206: "lookup_dcookie",
    6207: "epoll_create",
    6208: "epoll_ctl",
    6209: "epoll_wait",
    6210: "remap_file_pages",
    6211: "rt_sigreturn",
    6212: "fcntl64",
    6213: "set_tid_address",
    6214: "restart_syscall",
    6215: "semtimedop",
    6216: "fadvise64",
    6217: "statfs64",
    6218: "fstatfs64",
    6219: "sendfile64",
    6220: "timer_create",
    6221: "timer_settime",
    6222: "timer_gettime",
    6223: "timer_getoverrun",
    6224: "timer_delete",
    6225: "clock_settime",
    6226: "clock_gettime",
    6227: "clock_getres",
    6228: "clock_nanosleep",
    6229: "tgkill",
    6230: "utimes",
    6231: "mbind",
    6232: "get_mempolicy",
    6233: "set_mempolicy",
    6234: "mq_open",
    6235: "mq_unlink",
    6236: "mq_timedsend",
    6237: "mq_timedreceive",
    6238: "mq_notify",
    6239: "mq_getsetattr",
    6240: "vserver",
    6241: "waitid",
    6243: "add_key",
    6244: "request_key",
    6245: "keyctl",
    6246: "set_thread_area",
    6247: "inotify_init",
    6248: "inotify_add_watch",
    6249: "inotify_rm_watch",
    6250: "migrate_pages",
    6251: "openat",
    6252: "mkdirat",
    6253: "mknodat",
    6254: "fchownat",
    6255: "futimesat",
    6256: "newfstatat",
    6257: "unlinkat",
    6258: "renameat",
    6259: "linkat",
    6260: "symlinkat",
    6261: "readlinkat",
    6262: "fchmodat",
    6263: "faccessat",
    6264: "pselect6",
    6265: "ppoll",
    6266: "unshare",
    6267: "splice",
    6268: "sync_file_range",
    6269: "tee",
    6270: "vmsplice",
    6271: "move_pages",
    6272: "set_robust_list",
    6273: "get_robust_list",
    6274: "kexec_load",
    6275: "getcpu",
    6276: "epoll_pwait",
    6277: "ioprio_set",
    6278: "ioprio_get",
    6279: "utimensat",
    6280: "signalfd",
    6281: "timerfd",
    6282: "eventfd",
    6283: "fallocate",
    6284: "timerfd_create",
    6285: "timerfd_gettime",
    6286: "timerfd_settime",
    6287: "signalfd4",
    6288: "eventfd2",
    6289: "epoll_create1",
    6290: "dup3",
    6291: "pipe2",
    6292: "inotify_init1",
    6293: "preadv",
    6294: "pwritev",
    6295: "rt_tgsigqueueinfo",
    6296: "perf_event_open",
    6297: "accept4",
    6298: "recvmmsg",
    6299: "getdents64",
    6300: "fanotify_init",
    6301: "fanotify_mark",
    6302: "prlimit64",
})

lib.add_number_mapping_from_dict("mips-n64", {
    5000: "read",
    5001: "write",
    5002: "open",
    5003: "close",
    5004: "stat",
    5005: "fstat",
    5006: "lstat",
    5007: "poll",
    5008: "lseek",
    5009: "mmap",
    5010: "mprotect",
    5011: "munmap",
    5012: "brk",
    5013: "rt_sigaction",
    5014: "rt_sigprocmask",
    5015: "ioctl",
    5016: "pread64",
    5017: "pwrite64",
    5018: "readv",
    5019: "writev",
    5020: "access",
    5021: "pipe",
    5022: "_newselect",
    5023: "sched_yield",
    5024: "mremap",
    5025: "msync",
    5026: "mincore",
    5027: "madvise",
    5028: "shmget",
    5029: "shmat",
    5030: "shmctl",
    5031: "dup",
    5032: "dup2",
    5033: "pause",
    5034: "nanosleep",
    5035: "getitimer",
    5036: "setitimer",
    5037: "alarm",
    5038: "getpid",
    5039: "sendfile",
    5040: "socket",
    5041: "connect",
    5042: "accept",
    5043: "sendto",
    5044: "recvfrom",
    5045: "sendmsg",
    5046: "recvmsg",
    5047: "shutdown",
    5048: "bind",
    5049: "listen",
    5050: "getsockname",
    5051: "getpeername",
    5052: "socketpair",
    5053: "setsockopt",
    5054: "getsockopt",
    5055: "clone",
    5056: "fork",
    5057: "execve",
    5058: "exit",
    5059: "wait4",
    5060: "kill",
    5061: "uname",
    5062: "semget",
    5063: "semop",
    5064: "semctl",
    5065: "shmdt",
    5066: "msgget",
    5067: "msgsnd",
    5068: "msgrcv",
    5069: "msgctl",
    5070: "fcntl",
    5071: "flock",
    5072: "fsync",
    5073: "fdatasync",
    5074: "truncate",
    5075: "ftruncate",
    5076: "getdents",
    5077: "getcwd",
    5078: "chdir",
    5079: "fchdir",
    5080: "rename",
    5081: "mkdir",
    5082: "rmdir",
    5083: "creat",
    5084: "link",
    5085: "unlink",
    5086: "symlink",
    5087: "readlink",
    5088: "chmod",
    5089: "fchmod",
    5090: "chown",
    5091: "fchown",
    5092: "lchown",
    5093: "umask",
    5094: "gettimeofday",
    5095: "getrlimit",
    5096: "getrusage",
    5097: "sysinfo",
    5098: "times",
    5099: "ptrace",
    5100: "getuid",
    5101: "syslog",
    5102: "getgid",
    5103: "setuid",
    5104: "setgid",
    5105: "geteuid",
    5106: "getegid",
    5107: "setpgid",
    5108: "getppid",
    5109: "getpgrp",
    5110: "setsid",
    5111: "setreuid",
    5112: "setregid",
    5113: "getgroups",
    5114: "setgroups",
    5115: "setresuid",
    5116: "getresuid",
    5117: "setresgid",
    5118: "getresgid",
    5119: "getpgid",
    5120: "setfsuid",
    5121: "setfsgid",
    5122: "getsid",
    5123: "capget",
    5124: "capset",
    5125: "rt_sigpending",
    5126: "rt_sigtimedwait",
    5127: "rt_sigqueueinfo",
    5128: "rt_sigsuspend",
    5129: "sigaltstack",
    5130: "utime",
    5131: "mknod",
    5132: "personality",
    5133: "ustat",
    5134: "statfs",
    5135: "fstatfs",
    5136: "sysfs",
    5137: "getpriority",
    5138: "setpriority",
    5139: "sched_setparam",
    5140: "sched_getparam",
    5141: "sched_setscheduler",
    5142: "sched_getscheduler",
    5143: "sched_get_priority_max",
    5144: "sched_get_priority_min",
    5145: "sched_rr_get_interval",
    5146: "mlock",
    5147: "munlock",
    5148: "mlockall",
    5149: "munlockall",
    5150: "vhangup",
    5151: "pivot_root",
    5152: "_sysctl",
    5153: "prctl",
    5154: "adjtimex",
    5155: "setrlimit",
    5156: "chroot",
    5157: "sync",
    5158: "acct",
    5159: "settimeofday",
    5160: "mount",
    5161: "umount2",
    5162: "swapon",
    5163: "swapoff",
    5164: "reboot",
    5165: "sethostname",
    5166: "setdomainname",
    5167: "create_module",
    5168: "init_module",
    5169: "delete_module",
    5170: "get_kernel_syms",
    5171: "query_module",
    5172: "quotactl",
    5173: "nfsservctl",
    5174: "getpmsg",
    5175: "putpmsg",
    5176: "afs_syscall",
    5178: "gettid",
    5179: "readahead",
    5180: "setxattr",
    5181: "lsetxattr",
    5182: "fsetxattr",
    5183: "getxattr",
    5184: "lgetxattr",
    5185: "fgetxattr",
    5186: "listxattr",
    5187: "llistxattr",
    5188: "flistxattr",
    5189: "removexattr",
    5190: "lremovexattr",
    5191: "fremovexattr",
    5192: "tkill",
    5194: "futex",
    5195: "sched_setaffinity",
    5196: "sched_getaffinity",
    5197: "cacheflush",
    5198: "cachectl",
    5199: "sysmips",
    5200: "io_setup",
    5201: "io_destroy",
    5202: "io_getevents",
    5203: "io_submit",
    5204: "io_cancel",
    5205: "exit_group",
    5206: "lookup_dcookie",
    5207: "epoll_create",
    5208: "epoll_ctl",
    5209: "epoll_wait",
    5210: "remap_file_pages",
    5211: "rt_sigreturn",
    5212: "set_tid_address",
    5213: "restart_syscall",
    5214: "semtimedop",
    5215: "fadvise64",
    5216: "timer_create",
    5217: "timer_settime",
    5218: "timer_gettime",
    5219: "timer_getoverrun",
    5220: "timer_delete",
    5221: "clock_settime",
    5222: "clock_gettime",
    5223: "clock_getres",
    5224: "clock_nanosleep",
    5225: "tgkill",
    5226: "utimes",
    5227: "mbind",
    5228: "get_mempolicy",
    5229: "set_mempolicy",
    5230: "mq_open",
    5231: "mq_unlink",
    5232: "mq_timedsend",
    5233: "mq_timedreceive",
    5234: "mq_notify",
    5235: "mq_getsetattr",
    5236: "vserver",
    5237: "waitid",
    5239: "add_key",
    5240: "request_key",
    5241: "keyctl",
    5242: "set_thread_area",
    5243: "inotify_init",
    5244: "inotify_add_watch",
    5245: "inotify_rm_watch",
    5246: "migrate_pages",
    5247: "openat",
    5248: "mkdirat",
    5249: "mknodat",
    5250: "fchownat",
    5251: "futimesat",
    5252: "newfstatat",
    5253: "unlinkat",
    5254: "renameat",
    5255: "linkat",
    5256: "symlinkat",
    5257: "readlinkat",
    5258: "fchmodat",
    5259: "faccessat",
    5260: "pselect6",
    5261: "ppoll",
    5262: "unshare",
    5263: "splice",
    5264: "sync_file_range",
    5265: "tee",
    5266: "vmsplice",
    5267: "move_pages",
    5268: "set_robust_list",
    5269: "get_robust_list",
    5270: "kexec_load",
    5271: "getcpu",
    5272: "epoll_pwait",
    5273: "ioprio_set",
    5274: "ioprio_get",
    5275: "utimensat",
    5276: "signalfd",
    5277: "timerfd",
    5278: "eventfd",
    5279: "fallocate",
    5280: "timerfd_create",
    5281: "timerfd_gettime",
    5282: "timerfd_settime",
    5283: "signalfd4",
    5284: "eventfd2",
    5285: "epoll_create1",
    5286: "dup3",
    5287: "pipe2",
    5288: "inotify_init1",
    5289: "preadv",
    5290: "pwritev",
    5291: "rt_tgsigqueueinfo",
    5292: "perf_event_open",
    5293: "accept4",
    5294: "recvmmsg",
    5295: "fanotify_init",
    5296: "fanotify_mark",
    5297: "prlimit64",
})

lib.add_number_mapping_from_dict("mips-o32", {
    4000: "syscall",
    4001: "exit",
    4002: "fork",
    4003: "read",
    4004: "write",
    4005: "open",
    4006: "close",
    4007: "waitpid",
    4008: "creat",
    4009: "link",
    4010: "unlink",
    4011: "execve",
    4012: "chdir",
    4013: "time",
    4014: "mknod",
    4015: "chmod",
    4016: "lchown",
    4017: "break",
    4019: "lseek",
    4020: "getpid",
    4021: "mount",
    4022: "umount",
    4023: "setuid",
    4024: "getuid",
    4025: "stime",
    4026: "ptrace",
    4027: "alarm",
    4029: "pause",
    4030: "utime",
    4031: "stty",
    4032: "gtty",
    4033: "access",
    4034: "nice",
    4035: "ftime",
    4036: "sync",
    4037: "kill",
    4038: "rename",
    4039: "mkdir",
    4040: "rmdir",
    4041: "dup",
    4042: "pipe",
    4043: "times",
    4044: "prof",
    4045: "brk",
    4046: "setgid",
    4047: "getgid",
    4048: "signal",
    4049: "geteuid",
    4050: "getegid",
    4051: "acct",
    4052: "umount2",
    4053: "lock",
    4054: "ioctl",
    4055: "fcntl",
    4056: "mpx",
    4057: "setpgid",
    4058: "ulimit",
    4060: "umask",
    4061: "chroot",
    4062: "ustat",
    4063: "dup2",
    4064: "getppid",
    4065: "getpgrp",
    4066: "setsid",
    4067: "sigaction",
    4068: "sgetmask",
    4069: "ssetmask",
    4070: "setreuid",
    4071: "setregid",
    4072: "sigsuspend",
    4073: "sigpending",
    4074: "sethostname",
    4075: "setrlimit",
    4076: "getrlimit",
    4077: "getrusage",
    4078: "gettimeofday",
    4079: "settimeofday",
    4080: "getgroups",
    4081: "setgroups",
    4083: "symlink",
    4085: "readlink",
    4086: "uselib",
    4087: "swapon",
    4088: "reboot",
    4089: "readdir",
    4090: "mmap",
    4091: "munmap",
    4092: "truncate",
    4093: "ftruncate",
    4094: "fchmod",
    4095: "fchown",
    4096: "getpriority",
    4097: "setpriority",
    4098: "profil",
    4099: "statfs",
    4100: "fstatfs",
    4101: "ioperm",
    4102: "socketcall",
    4103: "syslog",
    4104: "setitimer",
    4105: "getitimer",
    4106: "stat",
    4107: "lstat",
    4108: "fstat",
    4110: "iopl",
    4111: "vhangup",
    4112: "idle",
    4113: "vm86",
    4114: "wait4",
    4115: "swapoff",
    4116: "sysinfo",
    4117: "ipc",
    4118: "fsync",
    4119: "sigreturn",
    4120: "clone",
    4121: "setdomainname",
    4122: "uname",
    4123: "modify_ldt",
    4124: "adjtimex",
    4125: "mprotect",
    4126: "sigprocmask",
    4127: "create_module",
    4128: "init_module",
    4129: "delete_module",
    4130: "get_kernel_syms",
    4131: "quotactl",
    4132: "getpgid",
    4133: "fchdir",
    4134: "bdflush",
    4135: "sysfs",
    4136: "personality",
    4137: "afs_syscall",
    4138: "setfsuid",
    4139: "setfsgid",
    4140: "_llseek",
    4141: "getdents",
    4142: "_newselect",
    4143: "flock",
    4144: "msync",
    4145: "readv",
    4146: "writev",
    4147: "cacheflush",
    4148: "cachectl",
    4149: "sysmips",
    4151: "getsid",
    4152: "fdatasync",
    4153: "_sysctl",
    4154: "mlock",
    4155: "munlock",
    4156: "mlockall",
    4157: "munlockall",
    4158: "sched_setparam",
    4159: "sched_getparam",
    4160: "sched_setscheduler",
    4161: "sched_getscheduler",
    4162: "sched_yield",
    4163: "sched_get_priority_max",
    4164: "sched_get_priority_min",
    4165: "sched_rr_get_interval",
    4166: "nanosleep",
    4167: "mremap",
    4168: "accept",
    4169: "bind",
    4170: "connect",
    4171: "getpeername",
    4172: "getsockname",
    4173: "getsockopt",
    4174: "listen",
    4175: "recv",
    4176: "recvfrom",
    4177: "recvmsg",
    4178: "send",
    4179: "sendmsg",
    4180: "sendto",
    4181: "setsockopt",
    4182: "shutdown",
    4183: "socket",
    4184: "socketpair",
    4185: "setresuid",
    4186: "getresuid",
    4187: "query_module",
    4188: "poll",
    4189: "nfsservctl",
    4190: "setresgid",
    4191: "getresgid",
    4192: "prctl",
    4193: "rt_sigreturn",
    4194: "rt_sigaction",
    4195: "rt_sigprocmask",
    4196: "rt_sigpending",
    4197: "rt_sigtimedwait",
    4198: "rt_sigqueueinfo",
    4199: "rt_sigsuspend",
    4200: "pread64",
    4201: "pwrite64",
    4202: "chown",
    4203: "getcwd",
    4204: "capget",
    4205: "capset",
    4206: "sigaltstack",
    4207: "sendfile",
    4208: "getpmsg",
    4209: "putpmsg",
    4210: "mmap2",
    4211: "truncate64",
    4212: "ftruncate64",
    4213: "stat64",
    4214: "lstat64",
    4215: "fstat64",
    4216: "pivot_root",
    4217: "mincore",
    4218: "madvise",
    4219: "getdents64",
    4220: "fcntl64",
    4222: "gettid",
    4223: "readahead",
    4224: "setxattr",
    4225: "lsetxattr",
    4226: "fsetxattr",
    4227: "getxattr",
    4228: "lgetxattr",
    4229: "fgetxattr",
    4230: "listxattr",
    4231: "llistxattr",
    4232: "flistxattr",
    4233: "removexattr",
    4234: "lremovexattr",
    4235: "fremovexattr",
    4236: "tkill",
    4237: "sendfile64",
    4238: "futex",
    4239: "sched_setaffinity",
    4240: "sched_getaffinity",
    4241: "io_setup",
    4242: "io_destroy",
    4243: "io_getevents",
    4244: "io_submit",
    4245: "io_cancel",
    4246: "exit_group",
    4247: "lookup_dcookie",
    4248: "epoll_create",
    4249: "epoll_ctl",
    4250: "epoll_wait",
    4251: "remap_file_pages",
    4252: "set_tid_address",
    4253: "restart_syscall",
    4254: "fadvise64",
    4255: "statfs64",
    4256: "fstatfs64",
    4257: "timer_create",
    4258: "timer_settime",
    4259: "timer_gettime",
    4260: "timer_getoverrun",
    4261: "timer_delete",
    4262: "clock_settime",
    4263: "clock_gettime",
    4264: "clock_getres",
    4265: "clock_nanosleep",
    4266: "tgkill",
    4267: "utimes",
    4268: "mbind",
    4269: "get_mempolicy",
    4270: "set_mempolicy",
    4271: "mq_open",
    4272: "mq_unlink",
    4273: "mq_timedsend",
    4274: "mq_timedreceive",
    4275: "mq_notify",
    4276: "mq_getsetattr",
    4277: "vserver",
    4278: "waitid",
    4280: "add_key",
    4281: "request_key",
    4282: "keyctl",
    4283: "set_thread_area",
    4284: "inotify_init",
    4285: "inotify_add_watch",
    4286: "inotify_rm_watch",
    4287: "migrate_pages",
    4288: "openat",
    4289: "mkdirat",
    4290: "mknodat",
    4291: "fchownat",
    4292: "futimesat",
    4293: "fstatat64",
    4294: "unlinkat",
    4295: "renameat",
    4296: "linkat",
    4297: "symlinkat",
    4298: "readlinkat",
    4299: "fchmodat",
    4300: "faccessat",
    4301: "pselect6",
    4302: "ppoll",
    4303: "unshare",
    4304: "splice",
    4305: "sync_file_range",
    4306: "tee",
    4307: "vmsplice",
    4308: "move_pages",
    4309: "set_robust_list",
    4310: "get_robust_list",
    4311: "kexec_load",
    4312: "getcpu",
    4313: "epoll_pwait",
    4314: "ioprio_set",
    4315: "ioprio_get",
    4316: "utimensat",
    4317: "signalfd",
    4318: "timerfd",
    4319: "eventfd",
    4320: "fallocate",
    4321: "timerfd_create",
    4322: "timerfd_gettime",
    4323: "timerfd_settime",
    4324: "signalfd4",
    4325: "eventfd2",
    4326: "epoll_create1",
    4327: "dup3",
    4328: "pipe2",
    4329: "inotify_init1",
    4330: "preadv",
    4331: "pwritev",
    4332: "rt_tgsigqueueinfo",
    4333: "perf_event_open",
    4334: "accept4",
    4335: "recvmmsg",
    4336: "fanotify_init",
    4337: "fanotify_mark",
    4338: "prlimit64",
})

lib.add_number_mapping_from_dict("ppc", {
    0: "restart_syscall",
    1: "exit",
    2: "fork",
    3: "read",
    4: "write",
    5: "open",
    6: "close",
    7: "waitpid",
    8: "creat",
    9: "link",
    10: "unlink",
    11: "execve",
    12: "chdir",
    13: "time",
    14: "mknod",
    15: "chmod",
    16: "lchown",
    17: "break",
    18: "oldstat",
    19: "lseek",
    20: "getpid",
    21: "mount",
    22: "umount",
    23: "setuid",
    24: "getuid",
    25: "stime",
    26: "ptrace",
    27: "alarm",
    28: "oldfstat",
    29: "pause",
    30: "utime",
    31: "stty",
    32: "gtty",
    33: "access",
    34: "nice",
    35: "ftime",
    36: "sync",
    37: "kill",
    38: "rename",
    39: "mkdir",
    40: "rmdir",
    41: "dup",
    42: "pipe",
    43: "times",
    44: "prof",
    45: "brk",
    46: "setgid",
    47: "getgid",
    48: "signal",
    49: "geteuid",
    50: "getegid",
    51: "acct",
    52: "umount2",
    53: "lock",
    54: "ioctl",
    55: "fcntl",
    56: "mpx",
    57: "setpgid",
    58: "ulimit",
    59: "oldolduname",
    60: "umask",
    61: "chroot",
    62: "ustat",
    63: "dup2",
    64: "getppid",
    65: "getpgrp",
    66: "setsid",
    67: "sigaction",
    68: "sgetmask",
    69: "ssetmask",
    70: "setreuid",
    71: "setregid",
    72: "sigsuspend",
    73: "sigpending",
    74: "sethostname",
    75: "setrlimit",
    76: "getrlimit",
    77: "getrusage",
    78: "gettimeofday",
    79: "settimeofday",
    80: "getgroups",
    81: "setgroups",
    82: "select",
    83: "symlink",
    84: "oldlstat",
    85: "readlink",
    86: "uselib",
    87: "swapon",
    88: "reboot",
    89: "readdir",
    90: "mmap",
    91: "munmap",
    92: "truncate",
    93: "ftruncate",
    94: "fchmod",
    95: "fchown",
    96: "getpriority",
    97: "setpriority",
    98: "profil",
    99: "statfs",
    100: "fstatfs",
    101: "ioperm",
    102: "socketcall",
    103: "syslog",
    104: "setitimer",
    105: "getitimer",
    106: "stat",
    107: "lstat",
    108: "fstat",
    109: "olduname",
    110: "iopl",
    111: "vhangup",
    112: "idle",
    113: "vm86",
    114: "wait4",
    115: "swapoff",
    116: "sysinfo",
    117: "ipc",
    118: "fsync",
    119: "sigreturn",
    120: "clone",
    121: "setdomainname",
    122: "uname",
    123: "modify_ldt",
    124: "adjtimex",
    125: "mprotect",
    126: "sigprocmask",
    127: "create_module",
    128: "init_module",
    129: "delete_module",
    130: "get_kernel_syms",
    131: "quotactl",
    132: "getpgid",
    133: "fchdir",
    134: "bdflush",
    135: "sysfs",
    136: "personality",
    137: "afs_syscall",
    138: "setfsuid",
    139: "setfsgid",
    140: "_llseek",
    141: "getdents",
    142: "_newselect",
    143: "flock",
    144: "msync",
    145: "readv",
    146: "writev",
    147: "getsid",
    148: "fdatasync",
    149: "_sysctl",
    150: "mlock",
    151: "munlock",
    152: "mlockall",
    153: "munlockall",
    154: "sched_setparam",
    155: "sched_getparam",
    156: "sched_setscheduler",
    157: "sched_getscheduler",
    158: "sched_yield",
    159: "sched_get_priority_max",
    160: "sched_get_priority_min",
    161: "sched_rr_get_interval",
    162: "nanosleep",
    163: "mremap",
    164: "setresuid",
    165: "getresuid",
    166: "query_module",
    167: "poll",
    168: "nfsservctl",
    169: "setresgid",
    170: "getresgid",
    171: "prctl",
    172: "rt_sigreturn",
    173: "rt_sigaction",
    174: "rt_sigprocmask",
    175: "rt_sigpending",
    176: "rt_sigtimedwait",
    177: "rt_sigqueueinfo",
    178: "rt_sigsuspend",
    179: "pread64",
    180: "pwrite64",
    181: "chown",
    182: "getcwd",
    183: "capget",
    184: "capset",
    185: "sigaltstack",
    186: "sendfile",
    187: "getpmsg",
    188: "putpmsg",
    189: "vfork",
    190: "ugetrlimit",
    191: "readahead",
    192: "mmap2",
    193: "truncate64",
    194: "ftruncate64",
    195: "stat64",
    196: "lstat64",
    197: "fstat64",
    198: "pciconfig_read",
    199: "pciconfig_write",
    200: "pciconfig_iobase",
    201: "multiplexer",
    202: "getdents64",
    203: "pivot_root",
    204: "fcntl64",
    205: "madvise",
    206: "mincore",
    207: "gettid",
    208: "tkill",
    209: "setxattr",
    210: "lsetxattr",
    211: "fsetxattr",
    212: "getxattr",
    213: "lgetxattr",
    214: "fgetxattr",
    215: "listxattr",
    216: "llistxattr",
    217: "flistxattr",
    218: "removexattr",
    219: "lremovexattr",
    220: "fremovexattr",
    221: "futex",
    222: "sched_setaffinity",
    223: "sched_getaffinity",
    225: "tuxcall",
    226: "sendfile64",
    227: "io_setup",
    228: "io_destroy",
    229: "io_getevents",
    230: "io_submit",
    231: "io_cancel",
    232: "set_tid_address",
    233: "fadvise64",
    234: "exit_group",
    235: "lookup_dcookie",
    236: "epoll_create",
    237: "epoll_ctl",
    238: "epoll_wait",
    239: "remap_file_pages",
    240: "timer_create",
    241: "timer_settime",
    242: "timer_gettime",
    243: "timer_getoverrun",
    244: "timer_delete",
    245: "clock_settime",
    246: "clock_gettime",
    247: "clock_getres",
    248: "clock_nanosleep",
    249: "swapcontext",
    250: "tgkill",
    251: "utimes",
    252: "statfs64",
    253: "fstatfs64",
    254: "fadvise64_64",
    255: "rtas",
    256: "sys_debug_setcontext",
    259: "mbind",
    260: "get_mempolicy",
    261: "set_mempolicy",
    262: "mq_open",
    263: "mq_unlink",
    264: "mq_timedsend",
    265: "mq_timedreceive",
    266: "mq_notify",
    267: "mq_getsetattr",
    268: "kexec_load",
    269: "add_key",
    270: "request_key",
    271: "keyctl",
    272: "waitid",
    273: "ioprio_set",
    274: "ioprio_get",
    275: "inotify_init",
    276: "inotify_add_watch",
    277: "inotify_rm_watch",
    278: "spu_run",
    279: "spu_create",
    280: "pselect6",
    281: "ppoll",
    282: "unshare",
    286: "openat",
    287: "mkdirat",
    288: "mknodat",
    289: "fchownat",
    290: "futimesat",
    291: "fstatat64",
    292: "unlinkat",
    293: "renameat",
    294: "linkat",
    295: "symlinkat",
    296: "readlinkat",
    297: "fchmodat",
    298: "faccessat",
})

lib.add_number_mapping_from_dict("ppc64", {
    0: "restart_syscall",
    1: "exit",
    2: "fork",
    3: "read",
    4: "write",
    5: "open",
    6: "close",
    7: "waitpid",
    8: "creat",
    9: "link",
    10: "unlink",
    11: "execve",
    12: "chdir",
    13: "time",
    14: "mknod",
    15: "chmod",
    16: "lchown",
    17: "break",
    18: "oldstat",
    19: "lseek",
    20: "getpid",
    21: "mount",
    22: "umount",
    23: "setuid",
    24: "getuid",
    25: "stime",
    26: "ptrace",
    27: "alarm",
    28: "oldfstat",
    29: "pause",
    30: "utime",
    31: "stty",
    32: "gtty",
    33: "access",
    34: "nice",
    35: "ftime",
    36: "sync",
    37: "kill",
    38: "rename",
    39: "mkdir",
    40: "rmdir",
    41: "dup",
    42: "pipe",
    43: "times",
    44: "prof",
    45: "brk",
    46: "setgid",
    47: "getgid",
    48: "signal",
    49: "geteuid",
    50: "getegid",
    51: "acct",
    52: "umount2",
    53: "lock",
    54: "ioctl",
    55: "fcntl",
    56: "mpx",
    57: "setpgid",
    58: "ulimit",
    59: "oldolduname",
    60: "umask",
    61: "chroot",
    62: "ustat",
    63: "dup2",
    64: "getppid",
    65: "getpgrp",
    66: "setsid",
    67: "sigaction",
    68: "sgetmask",
    69: "ssetmask",
    70: "setreuid",
    71: "setregid",
    72: "sigsuspend",
    73: "sigpending",
    74: "sethostname",
    75: "setrlimit",
    76: "getrlimit",
    77: "getrusage",
    78: "gettimeofday",
    79: "settimeofday",
    80: "getgroups",
    81: "setgroups",
    82: "select",
    83: "symlink",
    84: "oldlstat",
    85: "readlink",
    86: "uselib",
    87: "swapon",
    88: "reboot",
    89: "readdir",
    90: "mmap",
    91: "munmap",
    92: "truncate",
    93: "ftruncate",
    94: "fchmod",
    95: "fchown",
    96: "getpriority",
    97: "setpriority",
    98: "profil",
    99: "statfs",
    100: "fstatfs",
    101: "ioperm",
    102: "socketcall",
    103: "syslog",
    104: "setitimer",
    105: "getitimer",
    106: "stat",
    107: "lstat",
    108: "fstat",
    109: "olduname",
    110: "iopl",
    111: "vhangup",
    112: "idle",
    113: "vm86",
    114: "wait4",
    115: "swapoff",
    116: "sysinfo",
    117: "ipc",
    118: "fsync",
    119: "sigreturn",
    120: "clone",
    121: "setdomainname",
    122: "uname",
    123: "modify_ldt",
    124: "adjtimex",
    125: "mprotect",
    126: "sigprocmask",
    127: "create_module",
    128: "init_module",
    129: "delete_module",
    130: "get_kernel_syms",
    131: "quotactl",
    132: "getpgid",
    133: "fchdir",
    134: "bdflush",
    135: "sysfs",
    136: "personality",
    137: "afs_syscall",
    138: "setfsuid",
    139: "setfsgid",
    140: "_llseek",
    141: "getdents",
    142: "_newselect",
    143: "flock",
    144: "msync",
    145: "readv",
    146: "writev",
    147: "getsid",
    148: "fdatasync",
    149: "_sysctl",
    150: "mlock",
    151: "munlock",
    152: "mlockall",
    153: "munlockall",
    154: "sched_setparam",
    155: "sched_getparam",
    156: "sched_setscheduler",
    157: "sched_getscheduler",
    158: "sched_yield",
    159: "sched_get_priority_max",
    160: "sched_get_priority_min",
    161: "sched_rr_get_interval",
    162: "nanosleep",
    163: "mremap",
    164: "setresuid",
    165: "getresuid",
    166: "query_module",
    167: "poll",
    168: "nfsservctl",
    169: "setresgid",
    170: "getresgid",
    171: "prctl",
    172: "rt_sigreturn",
    173: "rt_sigaction",
    174: "rt_sigprocmask",
    175: "rt_sigpending",
    176: "rt_sigtimedwait",
    177: "rt_sigqueueinfo",
    178: "rt_sigsuspend",
    179: "pread64",
    180: "pwrite64",
    181: "chown",
    182: "getcwd",
    183: "capget",
    184: "capset",
    185: "sigaltstack",
    186: "sendfile",
    187: "getpmsg",
    188: "putpmsg",
    189: "vfork",
    190: "ugetrlimit",
    191: "readahead",
    198: "pciconfig_read",
    199: "pciconfig_write",
    200: "pciconfig_iobase",
    201: "multiplexer",
    202: "getdents64",
    203: "pivot_root",
    205: "madvise",
    206: "mincore",
    207: "gettid",
    208: "tkill",
    209: "setxattr",
    210: "lsetxattr",
    211: "fsetxattr",
    212: "getxattr",
    213: "lgetxattr",
    214: "fgetxattr",
    215: "listxattr",
    216: "llistxattr",
    217: "flistxattr",
    218: "removexattr",
    219: "lremovexattr",
    220: "fremovexattr",
    221: "futex",
    222: "sched_setaffinity",
    223: "sched_getaffinity",
    225: "tuxcall",
    227: "io_setup",
    228: "io_destroy",
    229: "io_getevents",
    230: "io_submit",
    231: "io_cancel",
    232: "set_tid_address",
    233: "fadvise64",
    234: "exit_group",
    235: "lookup_dcookie",
    236: "epoll_create",
    237: "epoll_ctl",
    238: "epoll_wait",
    239: "remap_file_pages",
    240: "timer_create",
    241: "timer_settime",
    242: "timer_gettime",
    243: "timer_getoverrun",
    244: "timer_delete",
    245: "clock_settime",
    246: "clock_gettime",
    247: "clock_getres",
    248: "clock_nanosleep",
    249: "swapcontext",
    250: "tgkill",
    251: "utimes",
    252: "statfs64",
    253: "fstatfs64",
    255: "rtas",
    256: "sys_debug_setcontext",
    259: "mbind",
    260: "get_mempolicy",
    261: "set_mempolicy",
    262: "mq_open",
    263: "mq_unlink",
    264: "mq_timedsend",
    265: "mq_timedreceive",
    266: "mq_notify",
    267: "mq_getsetattr",
    268: "kexec_load",
    269: "add_key",
    270: "request_key",
    271: "keyctl",
    272: "waitid",
    273: "ioprio_set",
    274: "ioprio_get",
    275: "inotify_init",
    276: "inotify_add_watch",
    277: "inotify_rm_watch",
    278: "spu_run",
    279: "spu_create",
    280: "pselect6",
    281: "ppoll",
    282: "unshare",
    286: "unlinkat",
    287: "renameat",
    288: "linkat",
    289: "symlinkat",
    290: "readlinkat",
    291: "fchmodat",
    292: "faccessat",
})

lib.add_number_mapping_from_dict("s390", {
    1: "exit",
    2: "fork",
    3: "read",
    4: "write",
    5: "open",
    6: "close",
    7: "restart_syscall",
    8: "creat",
    9: "link",
    10: "unlink",
    11: "execve",
    12: "chdir",
    13: "time",
    14: "mknod",
    15: "chmod",
    16: "lchown",
    19: "lseek",
    20: "getpid",
    21: "mount",
    22: "umount",
    23: "setuid",
    24: "getuid",
    25: "stime",
    26: "ptrace",
    27: "alarm",
    29: "pause",
    30: "utime",
    33: "access",
    34: "nice",
    36: "sync",
    37: "kill",
    38: "rename",
    39: "mkdir",
    40: "rmdir",
    41: "dup",
    42: "pipe",
    43: "times",
    45: "brk",
    46: "setgid",
    47: "getgid",
    48: "signal",
    49: "geteuid",
    50: "getegid",
    51: "acct",
    52: "umount2",
    54: "ioctl",
    55: "fcntl",
    57: "setpgid",
    60: "umask",
    61: "chroot",
    62: "ustat",
    63: "dup2",
    64: "getppid",
    65: "getpgrp",
    66: "setsid",
    67: "sigaction",
    70: "setreuid",
    71: "setregid",
    72: "sigsuspend",
    73: "sigpending",
    74: "sethostname",
    75: "setrlimit",
    76: "getrlimit",
    77: "getrusage",
    78: "gettimeofday",
    79: "settimeofday",
    80: "getgroups",
    81: "setgroups",
    83: "symlink",
    85: "readlink",
    86: "uselib",
    87: "swapon",
    88: "reboot",
    89: "readdir",
    90: "mmap",
    91: "munmap",
    92: "truncate",
    93: "ftruncate",
    94: "fchmod",
    95: "fchown",
    96: "getpriority",
    97: "setpriority",
    99: "statfs",
    100: "fstatfs",
    101: "ioperm",
    102: "socketcall",
    103: "syslog",
    104: "setitimer",
    105: "getitimer",
    106: "stat",
    107: "lstat",
    108: "fstat",
    110: "lookup_dcookie",
    111: "vhangup",
    112: "idle",
    114: "wait4",
    115: "swapoff",
    116: "sysinfo",
    117: "ipc",
    118: "fsync",
    119: "sigreturn",
    120: "clone",
    121: "setdomainname",
    122: "uname",
    124: "adjtimex",
    125: "mprotect",
    126: "sigprocmask",
    127: "create_module",
    128: "init_module",
    129: "delete_module",
    130: "get_kernel_syms",
    131: "quotactl",
    132: "getpgid",
    133: "fchdir",
    134: "bdflush",
    135: "sysfs",
    136: "personality",
    137: "afs_syscall",
    138: "setfsuid",
    139: "setfsgid",
    140: "_llseek",
    141: "getdents",
    142: "_newselect",
    143: "flock",
    144: "msync",
    145: "readv",
    146: "writev",
    147: "getsid",
    148: "fdatasync",
    149: "_sysctl",
    150: "mlock",
    151: "munlock",
    152: "mlockall",
    153: "munlockall",
    154: "sched_setparam",
    155: "sched_getparam",
    156: "sched_setscheduler",
    157: "sched_getscheduler",
    158: "sched_yield",
    159: "sched_get_priority_max",
    160: "sched_get_priority_min",
    161: "sched_rr_get_interval",
    162: "nanosleep",
    163: "mremap",
    164: "setresuid",
    165: "getresuid",
    167: "query_module",
    168: "poll",
    169: "nfsservctl",
    170: "setresgid",
    171: "getresgid",
    172: "prctl",
    173: "rt_sigreturn",
    174: "rt_sigaction",
    175: "rt_sigprocmask",
    176: "rt_sigpending",
    177: "rt_sigtimedwait",
    178: "rt_sigqueueinfo",
    179: "rt_sigsuspend",
    180: "pread64",
    181: "pwrite64",
    182: "chown",
    183: "getcwd",
    184: "capget",
    185: "capset",
    186: "sigaltstack",
    187: "sendfile",
    188: "getpmsg",
    189: "putpmsg",
    190: "vfork",
    191: "ugetrlimit",
    192: "mmap2",
    193: "truncate64",
    194: "ftruncate64",
    195: "stat64",
    196: "lstat64",
    197: "fstat64",
    198: "lchown32",
    199: "getuid32",
    200: "getgid32",
    201: "geteuid32",
    202: "getegid32",
    203: "setreuid32",
    204: "setregid32",
    205: "getgroups32",
    206: "setgroups32",
    207: "fchown32",
    208: "setresuid32",
    209: "getresuid32",
    210: "setresgid32",
    211: "getresgid32",
    212: "chown32",
    213: "setuid32",
    214: "setgid32",
    215: "setfsuid32",
    216: "setfsgid32",
    217: "pivot_root",
    218: "mincore",
    219: "madvise",
    220: "getdents64",
    221: "fcntl64",
    222: "readahead",
    223: "sendfile64",
    224: "setxattr",
    225: "lsetxattr",
    226: "fsetxattr",
    227: "getxattr",
    228: "lgetxattr",
    229: "fgetxattr",
    230: "listxattr",
    231: "llistxattr",
    232: "flistxattr",
    233: "removexattr",
    234: "lremovexattr",
    235: "fremovexattr",
    236: "gettid",
    237: "tkill",
    238: "futex",
    239: "sched_setaffinity",
    240: "sched_getaffinity",
    241: "tgkill",
    243: "io_setup",
    244: "io_destroy",
    245: "io_getevents",
    246: "io_submit",
    247: "io_cancel",
    248: "exit_group",
    249: "epoll_create",
    250: "epoll_ctl",
    251: "epoll_wait",
    252: "set_tid_address",
    253: "fadvise64",
    254: "timer_create",
    255: "timer_settime",
    256: "timer_gettime",
    257: "timer_getoverrun",
    258: "timer_delete",
    259: "clock_settime",
    260: "clock_gettime",
    261: "clock_getres",
    262: "clock_nanosleep",
    264: "fadvise64_64",
    265: "statfs64",
    266: "fstatfs64",
    267: "remap_file_pages",
    268: "mbind",
    269: "get_mempolicy",
    270: "set_mempolicy",
    271: "mq_open",
    272: "mq_unlink",
    273: "mq_timedsend",
    274: "mq_timedreceive",
    275: "mq_notify",
    276: "mq_getsetattr",
    277: "kexec_load",
    278: "add_key",
    279: "request_key",
    280: "keyctl",
    281: "waitid",
    282: "ioprio_set",
    283: "ioprio_get",
    284: "inotify_init",
    285: "inotify_add_watch",
    286: "inotify_rm_watch",
    287: "migrate_pages",
    288: "openat",
    289: "mkdirat",
    290: "mknodat",
    291: "fchownat",
    292: "futimesat",
    293: "fstatat64",
    294: "unlinkat",
    295: "renameat",
    296: "linkat",
    297: "symlinkat",
    298: "readlinkat",
    299: "fchmodat",
    300: "faccessat",
    301: "pselect6",
    302: "ppoll",
    303: "unshare",
    304: "set_robust_list",
    305: "get_robust_list",
    306: "splice",
    307: "sync_file_range",
    308: "tee",
    309: "vmsplice",
    310: "move_pages",
    311: "getcpu",
    312: "epoll_pwait",
    313: "utimes",
    314: "fallocate",
    315: "utimensat",
    316: "signalfd",
    317: "timerfd",
    318: "eventfd",
    319: "timerfd_create",
    320: "timerfd_settime",
    321: "timerfd_gettime",
    322: "signalfd4",
    323: "eventfd2",
    324: "inotify_init1",
    325: "pipe2",
    326: "dup3",
    327: "epoll_create1",
    330: "rt_tgsigqueueinfo",
    331: "perf_event_open",
    332: "fanotify_init",
    333: "fanotify_mark",
    334: "prlimit64",
    335: "name_to_handle_at",
    336: "open_by_handle_at",
    337: "clock_adjtime",
    338: "syncfs",
    339: "setns",
    340: "process_vm_readv",
    341: "process_vm_writev",
    342: "s390_runtime_instr",
    343: "kcmp",
    344: "finit_module",
    345: "sched_setattr",
    346: "sched_getattr",
    347: "renameat2",
    348: "seccomp",
    349: "getrandom",
    350: "memfd_create",
    351: "bpf",
    352: "s390_pci_mmio_write",
    353: "s390_pci_mmio_read",
    354: "execveat",
    355: "userfaultfd",
    356: "membarrier",
    357: "recvmmsg",
    358: "sendmmsg",
    359: "socket",
    360: "socketpair",
    361: "bind",
    362: "connect",
    363: "listen",
    364: "accept4",
    365: "getsockopt",
    366: "setsockopt",
    367: "getsockname",
    368: "getpeername",
    369: "sendto",
    370: "sendmsg",
    371: "recvfrom",
    372: "recvmsg",
    373: "shutdown",
    374: "mlock2",
})

lib.add_number_mapping_from_dict("s390x", {
    1: "exit",
    2: "fork",
    3: "read",
    4: "write",
    5: "open",
    6: "close",
    7: "restart_syscall",
    8: "creat",
    9: "link",
    10: "unlink",
    11: "execve",
    12: "chdir",
    14: "mknod",
    15: "chmod",
    19: "lseek",
    20: "getpid",
    21: "mount",
    22: "umount",
    26: "ptrace",
    27: "alarm",
    29: "pause",
    30: "utime",
    33: "access",
    34: "nice",
    36: "sync",
    37: "kill",
    38: "rename",
    39: "mkdir",
    40: "rmdir",
    41: "dup",
    42: "pipe",
    43: "times",
    45: "brk",
    48: "signal",
    51: "acct",
    52: "umount2",
    54: "ioctl",
    55: "fcntl",
    57: "setpgid",
    60: "umask",
    61: "chroot",
    62: "ustat",
    63: "dup2",
    64: "getppid",
    65: "getpgrp",
    66: "setsid",
    67: "sigaction",
    72: "sigsuspend",
    73: "sigpending",
    74: "sethostname",
    75: "setrlimit",
    77: "getrusage",
    78: "gettimeofday",
    79: "settimeofday",
    83: "symlink",
    85: "readlink",
    86: "uselib",
    87: "swapon",
    88: "reboot",
    89: "readdir",
    90: "mmap",
    91: "munmap",
    92: "truncate",
    93: "ftruncate",
    94: "fchmod",
    96: "getpriority",
    97: "setpriority",
    99: "statfs",
    100: "fstatfs",
    102: "socketcall",
    103: "syslog",
    104: "setitimer",
    105: "getitimer",
    106: "stat",
    107: "lstat",
    108: "fstat",
    110: "lookup_dcookie",
    111: "vhangup",
    112: "idle",
    114: "wait4",
    115: "swapoff",
    116: "sysinfo",
    117: "ipc",
    118: "fsync",
    119: "sigreturn",
    120: "clone",
    121: "setdomainname",
    122: "uname",
    124: "adjtimex",
    125: "mprotect",
    126: "sigprocmask",
    127: "create_module",
    128: "init_module",
    129: "delete_module",
    130: "get_kernel_syms",
    131: "quotactl",
    132: "getpgid",
    133: "fchdir",
    134: "bdflush",
    135: "sysfs",
    136: "personality",
    137: "afs_syscall",
    141: "getdents",
    142: "select",
    143: "flock",
    144: "msync",
    145: "readv",
    146: "writev",
    147: "getsid",
    148: "fdatasync",
    149: "_sysctl",
    150: "mlock",
    151: "munlock",
    152: "mlockall",
    153: "munlockall",
    154: "sched_setparam",
    155: "sched_getparam",
    156: "sched_setscheduler",
    157: "sched_getscheduler",
    158: "sched_yield",
    159: "sched_get_priority_max",
    160: "sched_get_priority_min",
    161: "sched_rr_get_interval",
    162: "nanosleep",
    163: "mremap",
    167: "query_module",
    168: "poll",
    169: "nfsservctl",
    172: "prctl",
    173: "rt_sigreturn",
    174: "rt_sigaction",
    175: "rt_sigprocmask",
    176: "rt_sigpending",
    177: "rt_sigtimedwait",
    178: "rt_sigqueueinfo",
    179: "rt_sigsuspend",
    180: "pread64",
    181: "pwrite64",
    183: "getcwd",
    184: "capget",
    185: "capset",
    186: "sigaltstack",
    187: "sendfile",
    188: "getpmsg",
    189: "putpmsg",
    190: "vfork",
    191: "getrlimit",
    198: "lchown",
    199: "getuid",
    200: "getgid",
    201: "geteuid",
    202: "getegid",
    203: "setreuid",
    204: "setregid",
    205: "getgroups",
    206: "setgroups",
    207: "fchown",
    208: "setresuid",
    209: "getresuid",
    210: "setresgid",
    211: "getresgid",
    212: "chown",
    213: "setuid",
    214: "setgid",
    215: "setfsuid",
    216: "setfsgid",
    217: "pivot_root",
    218: "mincore",
    219: "madvise",
    220: "getdents64",
    222: "readahead",
    224: "setxattr",
    225: "lsetxattr",
    226: "fsetxattr",
    227: "getxattr",
    228: "lgetxattr",
    229: "fgetxattr",
    230: "listxattr",
    231: "llistxattr",
    232: "flistxattr",
    233: "removexattr",
    234: "lremovexattr",
    235: "fremovexattr",
    236: "gettid",
    237: "tkill",
    238: "futex",
    239: "sched_setaffinity",
    240: "sched_getaffinity",
    241: "tgkill",
    243: "io_setup",
    244: "io_destroy",
    245: "io_getevents",
    246: "io_submit",
    247: "io_cancel",
    248: "exit_group",
    249: "epoll_create",
    250: "epoll_ctl",
    251: "epoll_wait",
    252: "set_tid_address",
    253: "fadvise64",
    254: "timer_create",
    255: "timer_settime",
    256: "timer_gettime",
    257: "timer_getoverrun",
    258: "timer_delete",
    259: "clock_settime",
    260: "clock_gettime",
    261: "clock_getres",
    262: "clock_nanosleep",
    265: "statfs64",
    266: "fstatfs64",
    267: "remap_file_pages",
    268: "mbind",
    269: "get_mempolicy",
    270: "set_mempolicy",
    271: "mq_open",
    272: "mq_unlink",
    273: "mq_timedsend",
    274: "mq_timedreceive",
    275: "mq_notify",
    276: "mq_getsetattr",
    277: "kexec_load",
    278: "add_key",
    279: "request_key",
    280: "keyctl",
    281: "waitid",
    282: "ioprio_set",
    283: "ioprio_get",
    284: "inotify_init",
    285: "inotify_add_watch",
    286: "inotify_rm_watch",
    287: "migrate_pages",
    288: "openat",
    289: "mkdirat",
    290: "mknodat",
    291: "fchownat",
    292: "futimesat",
    293: "newfstatat",
    294: "unlinkat",
    295: "renameat",
    296: "linkat",
    297: "symlinkat",
    298: "readlinkat",
    299: "fchmodat",
    300: "faccessat",
    301: "pselect6",
    302: "ppoll",
    303: "unshare",
    304: "set_robust_list",
    305: "get_robust_list",
    306: "splice",
    307: "sync_file_range",
    308: "tee",
    309: "vmsplice",
    310: "move_pages",
    311: "getcpu",
    312: "epoll_pwait",
    313: "utimes",
    314: "fallocate",
    315: "utimensat",
    316: "signalfd",
    317: "timerfd",
    318: "eventfd",
    319: "timerfd_create",
    320: "timerfd_settime",
    321: "timerfd_gettime",
    322: "signalfd4",
    323: "eventfd2",
    324: "inotify_init1",
    325: "pipe2",
    326: "dup3",
    327: "epoll_create1",
    330: "rt_tgsigqueueinfo",
    331: "perf_event_open",
    332: "fanotify_init",
    333: "fanotify_mark",
    334: "prlimit64",
    335: "name_to_handle_at",
    336: "open_by_handle_at",
    337: "clock_adjtime",
    338: "syncfs",
    339: "setns",
    340: "process_vm_readv",
    341: "process_vm_writev",
    342: "s390_runtime_instr",
    343: "kcmp",
    344: "finit_module",
    345: "sched_setattr",
    346: "sched_getattr",
    347: "renameat2",
    348: "seccomp",
    349: "getrandom",
    350: "memfd_create",
    351: "bpf",
    352: "s390_pci_mmio_write",
    353: "s390_pci_mmio_read",
    354: "execveat",
    355: "userfaultfd",
    356: "membarrier",
    357: "recvmmsg",
    358: "sendmmsg",
    359: "socket",
    360: "socketpair",
    361: "bind",
    362: "connect",
    363: "listen",
    364: "accept4",
    365: "getsockopt",
    366: "setsockopt",
    367: "getsockname",
    368: "getpeername",
    369: "sendto",
    370: "sendmsg",
    371: "recvfrom",
    372: "recvmsg",
    373: "shutdown",
    374: "mlock2",
})

lib.add_number_mapping_from_dict("sparc", {
    0: "restart_syscall",
    1: "exit",
    2: "fork",
    3: "read",
    4: "write",
    5: "open",
    6: "close",
    7: "wait4",
    8: "creat",
    9: "link",
    10: "unlink",
    11: "execv",
    12: "chdir",
    13: "chown",
    14: "mknod",
    15: "chmod",
    16: "lchown",
    17: "brk",
    18: "perfctr",
    19: "lseek",
    20: "getpid",
    21: "capget",
    22: "capset",
    23: "setuid",
    24: "getuid",
    25: "vmsplice",
    26: "ptrace",
    27: "alarm",
    28: "sigaltstack",
    29: "pause",
    30: "utime",
    31: "lchown32",
    32: "fchown32",
    33: "access",
    34: "nice",
    35: "chown32",
    36: "sync",
    37: "kill",
    38: "stat",
    39: "sendfile",
    40: "lstat",
    41: "dup",
    42: "pipe",
    43: "times",
    44: "getuid32",
    45: "umount2",
    46: "setgid",
    47: "getgid",
    48: "signal",
    49: "geteuid",
    50: "getegid",
    51: "acct",
    53: "getgid32",
    54: "ioctl",
    55: "reboot",
    56: "mmap2",
    57: "symlink",
    58: "readlink",
    59: "execve",
    60: "umask",
    61: "chroot",
    62: "fstat",
    63: "fstat64",
    64: "getpagesize",
    65: "msync",
    66: "vfork",
    67: "pread64",
    68: "pwrite64",
    69: "geteuid32",
    70: "getegid32",
    71: "mmap",
    72: "setreuid32",
    73: "munmap",
    74: "mprotect",
    75: "madvise",
    76: "vhangup",
    77: "truncate64",
    78: "mincore",
    79: "getgroups",
    80: "setgroups",
    81: "getpgrp",
    82: "setgroups32",
    83: "setitimer",
    84: "ftruncate64",
    85: "swapon",
    86: "getitimer",
    87: "setuid32",
    88: "sethostname",
    89: "setgid32",
    90: "dup2",
    91: "setfsuid32",
    92: "fcntl",
    93: "select",
    94: "setfsgid32",
    95: "fsync",
    96: "setpriority",
    97: "socket",
    98: "connect",
    99: "accept",
    100: "getpriority",
    101: "rt_sigreturn",
    102: "rt_sigaction",
    103: "rt_sigprocmask",
    104: "rt_sigpending",
    105: "rt_sigtimedwait",
    106: "rt_sigqueueinfo",
    107: "rt_sigsuspend",
    108: "setresuid32",
    109: "getresuid32",
    110: "setresgid32",
    111: "getresgid32",
    112: "setregid32",
    113: "recvmsg",
    114: "sendmsg",
    115: "getgroups32",
    116: "gettimeofday",
    117: "getrusage",
    118: "getsockopt",
    119: "getcwd",
    120: "readv",
    121: "writev",
    122: "settimeofday",
    123: "fchown",
    124: "fchmod",
    125: "recvfrom",
    126: "setreuid",
    127: "setregid",
    128: "rename",
    129: "truncate",
    130: "ftruncate",
    131: "flock",
    132: "lstat64",
    133: "sendto",
    134: "shutdown",
    135: "socketpair",
    136: "mkdir",
    137: "rmdir",
    138: "utimes",
    139: "stat64",
    140: "sendfile64",
    141: "getpeername",
    142: "futex",
    143: "gettid",
    144: "getrlimit",
    145: "setrlimit",
    146: "pivot_root",
    147: "prctl",
    148: "pciconfig_read",
    149: "pciconfig_write",
    150: "getsockname",
    151: "inotify_init",
    152: "inotify_add_watch",
    153: "poll",
    154: "getdents64",
    155: "fcntl64",
    156: "inotify_rm_watch",
    157: "statfs",
    158: "fstatfs",
    159: "umount",
    160: "sched_set_affinity",
    161: "sched_get_affinity",
    162: "getdomainname",
    163: "setdomainname",
    165: "quotactl",
    166: "set_tid_address",
    167: "mount",
    168: "ustat",
    169: "setxattr",
    170: "lsetxattr",
    171: "fsetxattr",
    172: "getxattr",
    173: "lgetxattr",
    174: "getdents",
    175: "setsid",
    176: "fchdir",
    177: "fgetxattr",
    178: "listxattr",
    179: "llistxattr",
    180: "flistxattr",
    181: "removexattr",
    182: "lremovexattr",
    183: "sigpending",
    184: "query_module",
    185: "setpgid",
    186: "fremovexattr",
    187: "tkill",
    188: "exit_group",
    189: "uname",
    190: "init_module",
    191: "personality",
    192: "remap_file_pages",
    193: "epoll_create",
    194: "epoll_ctl",
    195: "epoll_wait",
    196: "ioprio_set",
    197: "getppid",
    198: "sigaction",
    199: "sgetmask",
    200: "ssetmask",
    201: "sigsuspend",
    202: "oldlstat",
    203: "uselib",
    204: "readdir",
    205: "readahead",
    206: "socketcall",
    207: "syslog",
    208: "lookup_dcookie",
    209: "fadvise64",
    210: "fadvise64_64",
    211: "tgkill",
    212: "waitpid",
    213: "swapoff",
    214: "sysinfo",
    215: "ipc",
    216: "sigreturn",
    217: "clone",
    218: "ioprio_get",
    219: "adjtimex",
    220: "sigprocmask",
    221: "create_module",
    222: "delete_module",
    223: "get_kernel_syms",
    224: "getpgid",
    225: "bdflush",
    226: "sysfs",
    227: "afs_syscall",
    228: "setfsuid",
    229: "setfsgid",
    230: "_newselect",
    231: "time",
    232: "splice",
    233: "stime",
    234: "statfs64",
    235: "fstatfs64",
    236: "_llseek",
    237: "mlock",
    238: "munlock",
    239: "mlockall",
    240: "munlockall",
    241: "sched_setparam",
    242: "sched_getparam",
    243: "sched_setscheduler",
    244: "sched_getscheduler",
    245: "sched_yield",
    246: "sched_get_priority_max",
    247: "sched_get_priority_min",
    248: "sched_rr_get_interval",
    249: "nanosleep",
    250: "mremap",
    251: "_sysctl",
    252: "getsid",
    253: "fdatasync",
    254: "nfsservctl",
    255: "sync_file_range",
    256: "clock_settime",
    257: "clock_gettime",
    258: "clock_getres",
    259: "clock_nanosleep",
    260: "sched_getaffinity",
    261: "sched_setaffinity",
    262: "timer_settime",
    263: "timer_gettime",
    264: "timer_getoverrun",
    265: "timer_delete",
    266: "timer_create",
    267: "vserver",
    268: "io_setup",
    269: "io_destroy",
    270: "io_submit",
    271: "io_cancel",
    272: "io_getevents",
    273: "mq_open",
    274: "mq_unlink",
    275: "mq_timedsend",
    276: "mq_timedreceive",
    277: "mq_notify",
    278: "mq_getsetattr",
    279: "waitid",
    280: "tee",
    281: "add_key",
    282: "request_key",
    283: "keyctl",
    284: "openat",
    285: "mkdirat",
    286: "mknodat",
    287: "fchownat",
    288: "futimesat",
    289: "fstatat64",
    290: "unlinkat",
    291: "renameat",
    292: "linkat",
    293: "symlinkat",
    294: "readlinkat",
    295: "fchmodat",
    296: "faccessat",
    297: "pselect6",
    298: "ppoll",
    299: "unshare",
    300: "set_robust_list",
    301: "get_robust_list",
    302: "migrate_pages",
    303: "mbind",
    304: "get_mempolicy",
    305: "set_mempolicy",
    306: "kexec_load",
    307: "move_pages",
    308: "getcpu",
    309: "epoll_pwait",
    310: "utimensat",
    311: "signalfd",
    312: "timerfd_create",
    313: "eventfd",
    314: "fallocate",
    315: "timerfd_settime",
    316: "timerfd_gettime",
    317: "signalfd4",
    318: "eventfd2",
    319: "epoll_create1",
    320: "dup3",
    321: "pipe2",
    322: "inotify_init1",
    323: "accept4",
    324: "preadv",
    325: "pwritev",
    326: "rt_tgsigqueueinfo",
    327: "perf_event_open",
    328: "recvmmsg",
})

lib.add_number_mapping_from_dict("sparc64", {
    0: "restart_syscall",
    1: "exit",
    2: "fork",
    3: "read",
    4: "write",
    5: "open",
    6: "close",
    7: "wait4",
    8: "creat",
    9: "link",
    10: "unlink",
    11: "execv",
    12: "chdir",
    13: "chown",
    14: "mknod",
    15: "chmod",
    16: "lchown",
    17: "brk",
    18: "perfctr",
    19: "lseek",
    20: "getpid",
    21: "capget",
    22: "capset",
    23: "setuid",
    24: "getuid",
    25: "vmsplice",
    26: "ptrace",
    27: "alarm",
    28: "sigaltstack",
    29: "pause",
    30: "utime",
    33: "access",
    34: "nice",
    36: "sync",
    37: "kill",
    38: "stat",
    39: "sendfile",
    40: "lstat",
    41: "dup",
    42: "pipe",
    43: "times",
    45: "umount2",
    46: "setgid",
    47: "getgid",
    48: "signal",
    49: "geteuid",
    50: "getegid",
    51: "acct",
    52: "memory_ordering",
    54: "ioctl",
    55: "reboot",
    57: "symlink",
    58: "readlink",
    59: "execve",
    60: "umask",
    61: "chroot",
    62: "fstat",
    63: "fstat64",
    64: "getpagesize",
    65: "msync",
    66: "vfork",
    67: "pread64",
    68: "pwrite64",
    71: "mmap",
    73: "munmap",
    74: "mprotect",
    75: "madvise",
    76: "vhangup",
    78: "mincore",
    79: "getgroups",
    80: "setgroups",
    81: "getpgrp",
    83: "setitimer",
    85: "swapon",
    86: "getitimer",
    88: "sethostname",
    90: "dup2",
    92: "fcntl",
    93: "select",
    95: "fsync",
    96: "setpriority",
    97: "socket",
    98: "connect",
    99: "accept",
    100: "getpriority",
    101: "rt_sigreturn",
    102: "rt_sigaction",
    103: "rt_sigprocmask",
    104: "rt_sigpending",
    105: "rt_sigtimedwait",
    106: "rt_sigqueueinfo",
    107: "rt_sigsuspend",
    108: "setresuid",
    109: "getresuid",
    110: "setresgid",
    111: "getresgid",
    113: "recvmsg",
    114: "sendmsg",
    116: "gettimeofday",
    117: "getrusage",
    118: "getsockopt",
    119: "getcwd",
    120: "readv",
    121: "writev",
    122: "settimeofday",
    123: "fchown",
    124: "fchmod",
    125: "recvfrom",
    126: "setreuid",
    127: "setregid",
    128: "rename",
    129: "truncate",
    130: "ftruncate",
    131: "flock",
    132: "lstat64",
    133: "sendto",
    134: "shutdown",
    135: "socketpair",
    136: "mkdir",
    137: "rmdir",
    138: "utimes",
    139: "stat64",
    140: "sendfile64",
    141: "getpeername",
    142: "futex",
    143: "gettid",
    144: "getrlimit",
    145: "setrlimit",
    146: "pivot_root",
    147: "prctl",
    148: "pciconfig_read",
    149: "pciconfig_write",
    150: "getsockname",
    151: "inotify_init",
    152: "inotify_add_watch",
    153: "poll",
    154: "getdents64",
    156: "inotify_rm_watch",
    157: "statfs",
    158: "fstatfs",
    159: "umount",
    160: "sched_set_affinity",
    161: "sched_get_affinity",
    162: "getdomainname",
    163: "setdomainname",
    164: "utrap_install",
    165: "quotactl",
    166: "set_tid_address",
    167: "mount",
    168: "ustat",
    169: "setxattr",
    170: "lsetxattr",
    171: "fsetxattr",
    172: "getxattr",
    173: "lgetxattr",
    174: "getdents",
    175: "setsid",
    176: "fchdir",
    177: "fgetxattr",
    178: "listxattr",
    179: "llistxattr",
    180: "flistxattr",
    181: "removexattr",
    182: "lremovexattr",
    183: "sigpending",
    184: "query_module",
    185: "setpgid",
    186: "fremovexattr",
    187: "tkill",
    188: "exit_group",
    189: "uname",
    190: "init_module",
    191: "personality",
    192: "remap_file_pages",
    193: "epoll_create",
    194: "epoll_ctl",
    195: "epoll_wait",
    196: "ioprio_set",
    197: "getppid",
    198: "sigaction",
    199: "sgetmask",
    200: "ssetmask",
    201: "sigsuspend",
    202: "oldlstat",
    203: "uselib",
    204: "readdir",
    205: "readahead",
    206: "socketcall",
    207: "syslog",
    208: "lookup_dcookie",
    209: "fadvise64",
    210: "fadvise64_64",
    211: "tgkill",
    212: "waitpid",
    213: "swapoff",
    214: "sysinfo",
    215: "ipc",
    216: "sigreturn",
    217: "clone",
    218: "ioprio_get",
    219: "adjtimex",
    220: "sigprocmask",
    221: "create_module",
    222: "delete_module",
    223: "get_kernel_syms",
    224: "getpgid",
    225: "bdflush",
    226: "sysfs",
    227: "afs_syscall",
    228: "setfsuid",
    229: "setfsgid",
    230: "_newselect",
    232: "splice",
    233: "stime",
    234: "statfs64",
    235: "fstatfs64",
    236: "_llseek",
    237: "mlock",
    238: "munlock",
    239: "mlockall",
    240: "munlockall",
    241: "sched_setparam",
    242: "sched_getparam",
    243: "sched_setscheduler",
    244: "sched_getscheduler",
    245: "sched_yield",
    246: "sched_get_priority_max",
    247: "sched_get_priority_min",
    248: "sched_rr_get_interval",
    249: "nanosleep",
    250: "mremap",
    251: "_sysctl",
    252: "getsid",
    253: "fdatasync",
    254: "nfsservctl",
    255: "sync_file_range",
    256: "clock_settime",
    257: "clock_gettime",
    258: "clock_getres",
    259: "clock_nanosleep",
    260: "sched_getaffinity",
    261: "sched_setaffinity",
    262: "timer_settime",
    263: "timer_gettime",
    264: "timer_getoverrun",
    265: "timer_delete",
    266: "timer_create",
    267: "vserver",
    268: "io_setup",
    269: "io_destroy",
    270: "io_submit",
    271: "io_cancel",
    272: "io_getevents",
    273: "mq_open",
    274: "mq_unlink",
    275: "mq_timedsend",
    276: "mq_timedreceive",
    277: "mq_notify",
    278: "mq_getsetattr",
    279: "waitid",
    280: "tee",
    281: "add_key",
    282: "request_key",
    283: "keyctl",
    284: "openat",
    285: "mkdirat",
    286: "mknodat",
    287: "fchownat",
    288: "futimesat",
    289: "fstatat64",
    290: "unlinkat",
    291: "renameat",
    292: "linkat",
    293: "symlinkat",
    294: "readlinkat",
    295: "fchmodat",
    296: "faccessat",
    297: "pselect6",
    298: "ppoll",
    299: "unshare",
    300: "set_robust_list",
    301: "get_robust_list",
    302: "migrate_pages",
    303: "mbind",
    304: "get_mempolicy",
    305: "set_mempolicy",
    306: "kexec_load",
    307: "move_pages",
    308: "getcpu",
    309: "epoll_pwait",
    310: "utimensat",
    311: "signalfd",
    312: "timerfd_create",
    313: "eventfd",
    314: "fallocate",
    315: "timerfd_settime",
    316: "timerfd_gettime",
    317: "signalfd4",
    318: "eventfd2",
    319: "epoll_create1",
    320: "dup3",
    321: "pipe2",
    322: "inotify_init1",
    323: "accept4",
    324: "preadv",
    325: "pwritev",
    326: "rt_tgsigqueueinfo",
    327: "perf_event_open",
    328: "recvmmsg",
})

lib.add_number_mapping_from_dict("riscv32", {
    0: "io_setup",
    1: "io_destroy",
    2: "io_submit",
    3: "io_cancel",
    4: "io_getevents",
    5: "setxattr",
    6: "lsetxattr",
    7: "fsopen",
    8: "getxattr",
    9: "lgetxattr",
    10: "fgetxattr",
    11: "listxattr",
    12: "llistxattr",
    13: "flistxattr",
    14: "removexattr",
    15: "lremovexattr",
    16: "fremovexattr",
    17: "getcwd",
    18: "lookup_dcookie",
    19: "eventfd2",
    20: "epoll_create1",
    21: "epoll_ctl",
    22: "epoll_pwait",
    23: "dup",
    24: "dup3",
    25: "fcntl",
    26: "inotify_init1",
    27: "inotify_add_watch",
    28: "inotify_rm_watch",
    29: "ioctl",
    30: "ioprio_set",
    31: "ioprio_get",
    32: "flock",
    33: "mknodat",
    34: "mkdirat",
    35: "unlinkat",
    36: "symlinkat",
    37: "linkat",
    39: "unmount2",
    40: "mount",
    41: "pivot_root",
    42: "nfsservctl",
    43: "statfs",
    44: "fstatfs",
    45: "truncate",
    46: "ftruncate",
    47: "fallocate",
    48: "faccessat",
    49: "chdir",
    50: "fchdir",
    51: "chroot",
    52: "fchmod",
    53: "fchmodat",
    54: "fchownat",
    55: "fchown",
    56: "openat",
    57: "close",
    58: "vhangup",
    59: "pipe2",
    60: "quotactl",
    61: "getdents64",
    62: "lseek",
    63: "read",
    64: "write",
    65: "readv",
    66: "writev",
    67: "pread64",
    68: "pwrite64",
    69: "preadv",
    70: "pwritev",
    71: "sendfile",
    72: "pselect6",
    73: "ppoll",
    74: "signalfd4",
    75: "vmsplice",
    76: "splice",
    77: "tee",
    78: "readlinkat",
    80: "fstat",
    81: "sync",
    82: "fsync",
    83: "fdatasync",
    84: "sync_file_range",
    85: "timer_fd_create",
    86: "timer_fd_settime",
    87: "timer_fd_gettime",
    88: "utimensat",
    89: "acct",
    90: "capget",
    91: "capset",
    92: "personality",
    93: "exit",
    94: "exit_group",
    95: "waitid",
    96: "set_tid_address",
    97: "unshare",
    98: "futex",
    99: "set_robust_list",
    100: "get_robust_list",
    101: "nanosleep",
    102: "getitimer",
    103: "setitimer",
    104: "kexec_load",
    105: "init_module",
    106: "delete_module",
    107: "timer_create",
    108: "timer_gettime",
    109: "timer_getoverrun",
    110: "timer_settime",
    111: "timer_delete",
    112: "clock_settime",
    113: "clock_gettime",
    114: "clock_getres",
    115: "clock_nanosleep",
    116: "syslog",
    117: "ptrace",
    118: "sched_setparam",
    119: "sched_setscheduler",
    120: "sched_getscheduler",
    121: "sched_getparam",
    122: "sched_setaffinity",
    123: "sched_getaffinity",
    124: "sched_yield",
    125: "sched_get_priority_max",
    126: "sched_get_priority_min",
    127: "sched_rr_get_interval",
    128: "restart_syscall",
    129: "kill",
    130: "tkill",
    131: "tgkill",
    132: "sigalstack",
    133: "rt_sigsuspend",
    134: "rt_sigaction",
    135: "rt_sigprocmask",
    136: "rt_sigpending",
    137: "rt_sigtimedwait",
    138: "rt_sigqueueinfo",
    139: "rt_sigreturn",
    140: "setpriority",
    141: "getpriority",
    142: "reboot",
    143: "setregid",
    144: "setgid",
    145: "setreuid",
    147: "setresuid",
    146: "setuid",
    148: "getresuid",
    149: "setresgid",
    150: "getresgid",
    151: "setfsuid",
    152: "setfsgid",
    153: "times",
    154: "setpgid",
    155: "getpgid",
    156: "getsid",
    157: "setsid",
    158: "get_groups",
    159: "set_groups",
    160: "uname",
    161: "sethostname",
    162: "setdomainname",
    165: "getrusage",
    166: "unmask",
    167: "prctl",
    168: "getcpu",
    169: "gettimeofday",
    170: "settimeofday",
    172: "getpid",
    173: "getppid",
    174: "getuid",
    175: "geteuid",
    176: "getgid",
    177: "getegid",
    178: "gettid",
    179: "sysinfo",
    180: "mq_open",
    181: "mq_unlink",
    182: "mq_timedsend",
    183: "mq_timedreceive",
    184: "mq_notify",
    185: "mq_getsetattr",
    186: "msgget",
    187: "msgctl",
    188: "msgrcv",
    189: "msgsnd",
    190: "semget",
    191: "semctl",
    192: "semtimedop",
    193: "semop",
    194: "shmget",
    195: "shmctl",
    196: "shmat",
    197: "shmdt",
    198: "socket",
    199: "socketpair",
    200: "bind",
    201: "listen",
    202: "accept",
    203: "connect",
    204: "getsockname",
    205: "getpeername",
    206: "sendto",
    207: "recvfrom",
    208: "setsockopt",
    209: "getsockopt",
    210: "shutdown",
    211: "sendmsg",
    212: "recvmsg",
    213: "readahead",
    214: "brk",
    215: "munmap",
    216: "mremap",
    217: "add_key",
    218: "request_key",
    219: "keyctl",
    220: "clone",
    221: "execve",
    222: "mmap",
    223: "fadvise_64",
    224: "swapon",
    225: "swapoff",
    226: "mprotect",
    227: "msync",
    228: "mlock",
    229: "munlock",
    230: "mlockall",
    231: "munlockall",
    232: "mincore",
    233: "madvise",
    234: "remap_file_pages",
    235: "mbind",
    236: "getmempolicy",
    237: "set_mempolicy",
    238: "migrate_pages",
    239: "move_pages",
    240: "rtas",
    241: "perf_event_open",
    242: "accept4",
    243: "recvmmsg",
    259: "riscv_flush_icache",
    260: "wait4",
    261: "prlimit64",
    262: "fanotify_init",
    263: "fanotify_mark",
    264: "name_to_handle_at",
    265: "open_by_handle_at",
    266: "clock_adjtime",
    267: "syncfs",
    268: "setns",
    269: "sendmmsg",
    270: "process_vm_readv",
    271: "process_vm_writev",
    272: "kcmp",
    273: "finit_module",
    274: "sched_setattr",
    275: "sched_getattr",
    276: "renameat2",
    277: "seccomp",
    278: "getrandom",
    279: "memfd_create",
    280: "bpf",
    281: "execveat",
    282: "userfaultfd",
    283: "membarrier",
    284: "mlock2",
    285: "copy_file_range",
    286: "preadv2",
    287: "pwritev2",
    288: "pkey_mprotect",
    289: "pkey_alloc",
    290: "pkey_free",
    291: "statx",
    292: "io_pgetevents",
    293: "rseq",
    294: "kexec_file_load",
    424: "pidfd_send_signal",
    425: "io_uring_setup",
    426: "io_uring_enter",
    427: "io_uring_register",
    428: "open_tree",
    429: "move_mount",
    430: "fstat",
    431: "fsconfig",
    432: "fsmount",
    433: "fspick",
    434: "pidfd_open",
    437: "openat2",
    438: "pidfd_getfd",
    439: "faccessat2",

})
