#!/bin/sh
# SPDX-License-Identifier: BSD-2-Clause
# SPDX-FileCopyrightText: © 2025 Antoine Le Gonidec <vv221@dotslashplay.it>
set -o errexit

###
# Age of Empires 2 expansions:
# - The Conquerors
# send your bug reports to vv221@dotslashplay.it
###

script_version=20251105.1

PLAYIT_COMPATIBILITY_LEVEL='2.33'

GAME_ID='age-of-empires-2'
GAME_NAME='Age of Empires II'

EXPANSION_ID='the-conquerors'
EXPANSION_NAME='The Conquerors'

ARCHIVE_BASE_FR_0_NAME='AGE2_X1.iso'
ARCHIVE_BASE_FR_0_MD5='e50d6f31de6f4a4daca15d3323cb2a51'
ARCHIVE_BASE_FR_0_SIZE='473052'
ARCHIVE_BASE_FR_0_VERSION='1.0-1'

ARCHIVE_BASE_EN_0_NAME='AGE2_X1.iso'
ARCHIVE_BASE_EN_0_MD5='82bb7757fa30328dd77d6dfed923d330'
ARCHIVE_BASE_EN_0_SIZE='460046'
ARCHIVE_BASE_EN_0_VERSION='1.0-1'

ARCHIVE_OPTIONAL_FR_PATCH_NAME='Age2XPatch.exe'
ARCHIVE_OPTIONAL_FR_PATCH_MD5='6e7f97ea81d54d0f5fe7f7ee38c2337b'
ARCHIVE_OPTIONAL_FR_PATCH_TYPE='cabinet'
ARCHIVE_OPTIONAL_FR_PATCH_SIZE='5460'
ARCHIVE_OPTIONAL_FR_PATCH_VERSION='1.0c-1'

ARCHIVE_OPTIONAL_EN_PATCH_NAME='Age2XPatch.exe'
ARCHIVE_OPTIONAL_EN_PATCH_MD5='8170ef5a8fa02725dcecc230b2f172ec'
ARCHIVE_OPTIONAL_EN_PATCH_TYPE='cabinet'
ARCHIVE_OPTIONAL_EN_PATCH_SIZE='5467'
ARCHIVE_OPTIONAL_EN_PATCH_VERSION='1.0c-1'

## Files found directly on the CD-ROM

CONTENT_PATH_DEFAULT='GAME'
## The game crashes on launch if any of the following libraries is missing.
CONTENT_LIBS_BIN_FILES='
ebueulax.dll
language_x1.dll'
## The game fails on launch with an empty error box if the following library is missing.
CONTENT_LIBS0_BIN_FILES='
language_x1_p1.dll'
CONTENT_GAME_BIN_FILES='
age2_x1/0000040c.016
age2_x1/0000040c.256
age2_x1/age2_x1.exe
age2_x1/age2_x1.icd
age2_x1/clokspl.exe'
CONTENT_GAME_L10N_FILES='
campaign/media/intro*
campaign/media/x*
sound/campaign'
CONTENT_GAME_DATA_FILES='
avi/age2_x1.avi
campaign/xcam?.cpx
data/empires2_x1.dat
data/empires2_x1_p1.dat
sound/stream/aztecs.mp3
sound/stream/huns.mp3
sound/stream/koreans.mp3
sound/stream/mayans.mp3
sound/stream/spanish.mp3
sound/stream/xcredits.mp3
sound/stream/xopen.mp3
sound/stream/xtown.mp3'

## Files found in the cabinet archive

CONTENT_GAME0_L10N_PATH='.'
CONTENT_GAME0_L10N_FILES='
data/list.crx
history
sound/scenario
taunt'
CONTENT_GAME0_DATA_PATH='.'
CONTENT_GAME0_DATA_FILES='
data/shadow.col
data/blendomatic.dat
data/view_icm.dat
data/gamedata_x1.drs
data/graphics.drs
data/interfac.drs
data/sounds_x1.drs
data/terrain.drs
random/random.txt
sound/terrain
support/support.txt'

## Files found in the 1.0c patch

CONTENT_LIBS1_BIN_PATH='.'
CONTENT_LIBS1_BIN_FILES='
language_x1_p1.dll'
CONTENT_GAME1_BIN_PATH='.'
CONTENT_GAME1_BIN_FILES='
age2_x1/age2_x1.exe
age2_x1/age2_x1.icd
age2_x1/clokspl.exe
age2_x1/secdrv.sys'
CONTENT_GAME1_DATA_PATH='.'
CONTENT_GAME1_DATA_FILES='
data/empires2_x1_p1.dat
data/gamedata_x1_p1.drs
random/*.rms'

USER_PERSISTENT_DIRECTORIES='
campaign/*.cpx
learn
savegame
scenario'
USER_PERSISTENT_FILES='
data/*.dat
*.nfp'

## Without a WINE virtual desktop, the game menu is badly rendered.
## (WINE 10.0)
WINE_VIRTUAL_DESKTOP='1280x1024'

APP_MAIN_ID="${GAME_ID}-${EXPANSION_ID}"
APP_MAIN_NAME="${GAME_NAME}: ${EXPANSION_NAME}"
APP_MAIN_EXE='age2_x1/age2_x1.exe'

PACKAGES_LIST='
PKG_BIN
PKG_L10N
PKG_DATA'

PKG_PARENT_ID="$GAME_ID"
PKG_PARENT_ID_FR="${PKG_PARENT_ID}-fr"
PKG_PARENT_ID_EN="${PKG_PARENT_ID}-en"

PKG_BIN_ID="${GAME_ID}-${EXPANSION_ID}"
PKG_BIN_ID_FR="${PKG_BIN_ID}-fr"
PKG_BIN_ID_EN="${PKG_BIN_ID}-en"
PKG_BIN_PROVIDES="
$PKG_BIN_ID"
PKG_BIN_ARCH='32'
PKG_BIN_DEPENDENCIES_SIBLINGS='
PKG_PARENT
PKG_L10N
PKG_DATA'
PKG_BIN_DEPENDENCIES_GSTREAMER_PLUGINS='
video/x-msvideo'

PKG_L10N_ID="${GAME_ID}-${EXPANSION_ID}-l10n"
PKG_L10N_ID_FR="${PKG_L10N_ID}-fr"
PKG_L10N_ID_EN="${PKG_L10N_ID}-en"
PKG_L10N_PROVIDES="
$PKG_L10N_ID
${GAME_ID}-l10n-common"
PKG_L10N_PROVIDES_FR="$PKG_L10N_PROVIDES
${GAME_ID}-l10n-common-fr"
PKG_L10N_PROVIDES_EN="$PKG_L10N_PROVIDES
${GAME_ID}-l10n-common-en"
PKG_L10N_FR_DESCRIPTION='French localisation'
PKG_L10N_FR_DESCRIPTION='English localisation'

PKG_DATA_ID="${GAME_ID}-${EXPANSION_ID}-data"
PKG_DATA_DESCRIPTION='data'
PKG_DATA_PROVIDES="
${GAME_ID}-data-common"

## - cabextract is used to extract files from a cabinet archive on the CD-ROM.
## - convmv is used to convert file paths to lower case.
REQUIREMENTS_LIST='
cabextract
convmv'

# Load common functions.

PLAYIT_LIB_PATHS="
$PWD
${XDG_DATA_HOME:="${HOME}/.local/share"}/play.it
/usr/local/share/games/play.it
/usr/local/share/play.it
/usr/share/games/play.it
/usr/share/play.it"

if [ -z "$PLAYIT_LIB2" ]; then
	for playit_lib_path in $PLAYIT_LIB_PATHS; do
		if [ -e "${playit_lib_path}/libplayit2.sh" ]; then
			PLAYIT_LIB2="${playit_lib_path}/libplayit2.sh"
			break
		fi
	done
fi
if [ -z "$PLAYIT_LIB2" ]; then
	printf '\n\033[1;31mError:\033[0m\n'
	printf 'libplayit2.sh not found.\n'
	exit 1
fi
# shellcheck source=libplayit2.sh
. "$PLAYIT_LIB2"

# Run default initialisation actions.

initialization_default "$@"

# Check for the presence of the 2.0c patch.

case "$(current_archive)" in
	('ARCHIVE_BASE_FR_'*)
		archive_initialize_optional 'ARCHIVE_PATCH' 'ARCHIVE_OPTIONAL_FR_PATCH'
	;;
	('ARCHIVE_BASE_EN_'*)
		archive_initialize_optional 'ARCHIVE_PATCH' 'ARCHIVE_OPTIONAL_EN_PATCH'
	;;
esac
if archive_is_available 'ARCHIVE_PATCH'; then
	export "$(current_archive)_VERSION=$(archive_version 'ARCHIVE_PATCH')"
fi

# Extract game data.

archive_extraction_default
## Convert all file paths to lowercase.
tolower "${PLAYIT_WORKDIR}/gamedata/$(content_path_default)"
## Rename some files with an abbreviated file name.
mv \
	"${PLAYIT_WORKDIR}/gamedata/$(content_path_default)/langua_1.dll" \
	"${PLAYIT_WORKDIR}/gamedata/$(content_path_default)/language_x1.dll"
if [ -e "${PLAYIT_WORKDIR}/gamedata/$(content_path_default)/langua_2.dll" ]; then
	mv \
		"${PLAYIT_WORKDIR}/gamedata/$(content_path_default)/langua_2.dll" \
		"${PLAYIT_WORKDIR}/gamedata/$(content_path_default)/language_x1_p1.dll"
fi
mv \
	"${PLAYIT_WORKDIR}/gamedata/$(content_path_default)/data/empire_1.dat" \
	"${PLAYIT_WORKDIR}/gamedata/$(content_path_default)/data/empires2_x1.dat"
if [ -e "${PLAYIT_WORKDIR}/gamedata/$(content_path_default)/empire_2.dll" ]; then
	mv \
		"${PLAYIT_WORKDIR}/gamedata/$(content_path_default)/data/empire_2.dat" \
		"${PLAYIT_WORKDIR}/gamedata/$(content_path_default)/data/empires2_x1_p1.dat"
fi
## Extract files from the cabinet archive.
ARCHIVE_CABINET_PATH="${PLAYIT_WORKDIR}/gamedata/MSGAME.CAB"
ARCHIVE_CABINET_TYPE='cabinet'
archive_extraction 'ARCHIVE_CABINET'
rm "$ARCHIVE_CABINET_PATH"
## Extract files from the 1.0c patch.
if archive_is_available 'ARCHIVE_PATCH'; then
	archive_extraction 'ARCHIVE_PATCH'
fi

# Include game data.

content_inclusion_icons 'PKG_DATA'
content_inclusion_default

# Write launchers.

## Install shipped libraries.
wineprefix_init_custom() {
	cat <<- EOF
	# Install shipped libraries.
	PATH_LIBRARIES_SYSTEM='$(path_libraries)'
	EOF
	cat <<- 'EOF'
	find "$PATH_LIBRARIES_SYSTEM" -iname '*.dll' | while read -r dll_source; do
		dll_destination="${PATH_PREFIX}${dll_source#"${PATH_LIBRARIES_SYSTEM}"}"
	    mkdir --parents "$(dirname "$dll_destination")"
	    ln --symbolic "$dll_source" "$dll_destination"
	done
	EOF
}

launchers_generation

# Build packages.

packages_generation
print_instructions

# Clean up.

working_directory_cleanup

exit 0
