// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

#include "gtest/gtest.h"

#include "test/providers/provider_test_utils.h"
#include "test/providers/run_options_config_keys.h"
#include "test/common/dnnl_op_test_utils.h"
#include "test/common/cuda_op_test_utils.h"
#include "test/common/tensor_op_test_utils.h"
#include "default_providers.h"

namespace onnxruntime {
namespace test {

namespace {

const onnxruntime::RunOptions run_options = []() {
  onnxruntime::RunOptions options{};
  ORT_THROW_IF_ERROR(options.config_options.AddConfigEntry(kOpTesterRunOptionsConfigTestTunableOp, "true"));
  return options;
}();

const constexpr auto run_with_tunable_op = &run_options;

}  // namespace

template <typename T>
struct MatMulTestData {
  std::string name;
  std::vector<int64_t> input0_dims;
  std::vector<int64_t> input1_dims;
  std::vector<int64_t> expected_dims;
  std::vector<T> expected_vals;
};

template <typename T>
std::vector<MatMulTestData<T>> GenerateTestCases() {
  std::vector<MatMulTestData<T>> test_cases;

  auto real_expected_vals = [](const std::vector<int32_t>& expected_vals) {
    if constexpr (std::is_same_v<T, int32_t>) {
      return expected_vals;
    } else if constexpr (std::is_same_v<T, MLFloat16>) {
      std::vector<MLFloat16> expected_vals_fp16(expected_vals.size());
      std::transform(expected_vals.begin(), expected_vals.end(), expected_vals_fp16.begin(),
                     [](int32_t num) { return MLFloat16(float(num)); });
      return expected_vals_fp16;
    } else {
      std::vector<T> real_expected_vals(expected_vals.size());
      std::transform(expected_vals.begin(), expected_vals.end(), real_expected_vals.begin(),
                     [](int32_t num) { return static_cast<T>(num); });
      return real_expected_vals;
    }
  };

  test_cases.push_back(
      {"test padding and broadcast A > B",
       {3, 1, 1, 2},
       {2, 2, 2},
       {3, 2, 1, 2},
       real_expected_vals({2, 3, 6, 7, 6, 11, 26, 31, 10, 19, 46, 55})});

  test_cases.push_back(
      {"test padding and broadcast A > B - no broadcast in B",
       {2, 2, 3, 2},
       {2, 1},
       {2, 2, 3, 1},
       real_expected_vals({1, 3, 5, 7, 9, 11, 13, 15, 17, 19, 21, 23})});

  test_cases.push_back(
      {"test padding and broadcast B > A",
       {2, 3, 2},
       {3, 2, 2, 1},
       {3, 2, 3, 1},
       real_expected_vals({1, 3, 5, 33, 43, 53, 5, 23, 41, 85, 111, 137, 9, 43, 77, 137, 179, 221})});

  test_cases.push_back(
      {"test left 1D",
       {2},
       {3, 2, 1},
       {3, 1},
       real_expected_vals({1, 3, 5})});

  test_cases.push_back(
      {"test right 1D",
       {3, 1, 2},
       {2},
       {3, 1},
       real_expected_vals({1, 3, 5})});

  test_cases.push_back(
      {"test left 1D right 2D",
       {2},
       {2, 3},
       {3},
       real_expected_vals({3, 4, 5})});

  test_cases.push_back(
      {"test scalar output",
       {3},
       {3},
       {},
       real_expected_vals({5})});

  test_cases.push_back(
      {"test 2D",
       {3, 4},
       {4, 3},
       {3, 3},
       real_expected_vals({42, 48, 54, 114, 136, 158, 186, 224, 262})});

  test_cases.push_back(
      {"test 2D special",
       {2, 2, 3},
       {3, 4},
       {2, 2, 4},
       real_expected_vals({20, 23, 26, 29, 56, 68, 80, 92, 92, 113, 134, 155, 128, 158, 188, 218})});

  test_cases.push_back(
      {"test 2D special 2",
       {2, 2, 3},
       {1, 3, 4},
       {2, 2, 4},
       real_expected_vals({20, 23, 26, 29, 56, 68, 80, 92, 92, 113, 134, 155, 128, 158, 188, 218})});

  test_cases.push_back(
      {"test 2D special 3",
       {2, 6},
       {1, 1, 6, 1},
       {1, 1, 2, 1},
       real_expected_vals({55, 145})});

  test_cases.push_back(
      {"test 2D empty input",
       {3, 4},
       {4, 0},
       {3, 0},
       real_expected_vals({})});

  test_cases.push_back(
      {"test 3D batch",
       {3, 1, 3},
       {3, 3, 2},
       {3, 1, 2},
       real_expected_vals({
           // clang-format off
              10,  13,
             100, 112,
             298, 319,
           // clang-format on
       })});

  test_cases.push_back(
      {"test 4D batch",
       {2, 2, 1, 3},
       {2, 2, 3, 2},
       {2, 2, 1, 2},
       real_expected_vals({
           // clang-format off
              10,  13,
             100, 112,
             298, 319,
             604, 634,
           // clang-format on
       })});

#ifdef USE_WEBGPU
  test_cases.push_back(
      {"test 3D tensors with M = 1",
       {6, 1, 8},
       {1, 8, 3},
       {6, 1, 3},
       real_expected_vals({
           // clang-format off
              420, 448, 476,
              1092, 1184, 1276,
              1764, 1920, 2076,
              2436, 2656, 2876,
              3108, 3392, 3676,
              3780, 4128, 4476,
           // clang-format on
       })});

  test_cases.push_back(
      {"test 4D tensors with M = 1",
       {2, 3, 1, 8},
       {1, 1, 8, 3},
       {2, 3, 1, 3},
       real_expected_vals({420, 448, 476, 1092, 1184, 1276, 1764, 1920, 2076, 2436, 2656, 2876, 3108, 3392, 3676, 3780, 4128, 4476})});

  test_cases.push_back(
      {"test 4D tensors",
       {2, 3, 4, 3},
       {2, 3, 3, 5},
       {2, 3, 4, 5},
       real_expected_vals({
           // clang-format off
            25, 28, 31, 34, 37, 70, 82, 94, 106, 118, 115, 136, 157, 178, 199, 160, 190, 220,
            250, 280, 790, 829, 868, 907, 946, 970, 1018, 1066, 1114, 1162, 1150, 1207, 1264,
            1321, 1378, 1330, 1396, 1462, 1528, 1594, 2635, 2710, 2785, 2860, 2935, 2950, 3034,
            3118, 3202, 3286, 3265, 3358, 3451, 3544, 3637, 3580, 3682, 3784, 3886, 3988, 5560,
            5671, 5782, 5893, 6004, 6010, 6130, 6250, 6370, 6490, 6460, 6589, 6718, 6847, 6976,
            6910, 7048, 7186, 7324, 7462, 9565, 9712, 9859, 10006, 10153, 10150, 10306, 10462,
            10618, 10774, 10735, 10900, 11065, 11230, 11395, 11320, 11494, 11668, 11842, 12016,
            14650, 14833, 15016, 15199, 15382, 15370, 15562, 15754, 15946, 16138, 16090, 16291,
            16492, 16693, 16894, 16810, 17020, 17230, 17440, 17650
           // clang-format on
       })});

  // Test case: multiplies 2D broadcasted to 4D tensors
  test_cases.push_back(
      {"test 2D broadcasted to 4D tensors",
       {2, 4},
       {4, 3, 2, 4, 2},
       {4, 3, 2, 2, 2},
       real_expected_vals({
           // clang-format off
            28, 34, 76, 98, 76, 82, 252, 274, 124, 130, 428, 450, 172, 178, 604, 626,
            220, 226, 780, 802, 268, 274, 956, 978, 316, 322, 1132, 1154, 364, 370,
            1308, 1330, 412, 418, 1484, 1506, 460, 466, 1660, 1682, 508, 514, 1836,
            1858, 556, 562, 2012, 2034, 604, 610, 2188, 2210, 652, 658, 2364, 2386,
            700, 706, 2540, 2562, 748, 754, 2716, 2738, 796, 802, 2892, 2914, 844,
            850, 3068, 3090, 892, 898, 3244, 3266, 940, 946, 3420, 3442, 988, 994,
            3596, 3618, 1036, 1042, 3772, 3794, 1084, 1090, 3948, 3970, 1132, 1138,
            4124, 4146
           // clang-format on
       })});

  // Test case: multiplies 4D broadcasted to 5D tensors
  test_cases.push_back(
      {"test 4D broadcasted to 5D tensors",
       {3, 1, 2, 4},
       {4, 3, 2, 4, 2},
       {4, 3, 2, 2, 2},
       real_expected_vals({
           // clang-format off
              28, 34, 76, 98, 76, 82, 252, 274, 732, 770, 1036, 1090, 1036, 1074, 1468,
              1522, 2460, 2530, 3020, 3106, 3020, 3090, 3708, 3794, 316, 322, 1132,
              1154, 364, 370, 1308, 1330, 2556, 2594, 3628, 3682, 2860, 2898, 4060,
              4114, 5820, 5890, 7148, 7234, 6380, 6450, 7836, 7922, 604, 610, 2188,
              2210, 652, 658, 2364, 2386, 4380, 4418, 6220, 6274, 4684, 4722, 6652,
              6706, 9180, 9250, 11276, 11362, 9740, 9810, 11964, 12050, 892, 898, 3244,
              3266, 940, 946, 3420, 3442, 6204, 6242, 8812, 8866, 6508, 6546, 9244,
              9298, 12540, 12610, 15404, 15490, 13100, 13170, 16092, 16178

           // clang-format on
       })});

  // Test case: same ranks different broadcast small 1
  test_cases.push_back(
      {"test same ranks different broadcast small 1",
       {2, 1, 2, 2},
       {1, 2, 2, 1},
       {2, 2, 2, 1},
       real_expected_vals({1, 3, 3, 13, 5, 7, 23, 33})});

  // Test case: same ranks different broadcast larger 0
  test_cases.push_back(
      {"test same ranks different broadcast larger 0",
       {1, 2, 2, 8},
       {2, 1, 8, 1},
       {2, 2, 2, 1},
       real_expected_vals({140, 364, 588, 812, 364, 1100, 1836, 2572})});

  // Test case: same ranks different broadcast larger 1
  test_cases.push_back(
      {"test same ranks different broadcast larger 1",
       {2, 1, 2, 8},
       {1, 2, 8, 1},
       {2, 2, 2, 1},
       real_expected_vals({140, 364, 364, 1100, 588, 812, 1836, 2572})});

  // Test case: 9 * 33, 33 * 10 matrix for vec1
  test_cases.push_back(
      {"test one batch 9 * 33, 33 * 10 matrix",
       {2, 1, 9, 33},
       {1, 2, 33, 10},
       {2, 2, 9, 10},
       real_expected_vals({114400, 114928, 115456, 115984, 116512, 117040, 117568, 118096, 118624, 119152, 288640, 290257, 291874, 293491, 295108, 296725, 298342, 299959, 301576, 303193, 462880, 465586, 468292, 470998, 473704, 476410, 479116, 481822, 484528, 487234, 637120, 640915, 644710, 648505, 652300, 656095, 659890, 663685, 667480, 671275, 811360, 816244, 821128, 826012, 830896, 835780, 840664, 845548, 850432, 855316, 985600, 991573, 997546, 1003519, 1009492, 1015465, 1021438, 1027411, 1033384, 1039357, 1159840, 1166902, 1173964, 1181026, 1188088, 1195150, 1202212, 1209274, 1216336, 1223398, 1334080, 1342231, 1350382, 1358533, 1366684, 1374835, 1382986, 1391137, 1399288, 1407439, 1508320, 1517560, 1526800, 1536040, 1545280, 1554520, 1563760, 1573000, 1582240, 1591480, 288640, 289168, 289696, 290224, 290752, 291280, 291808, 292336, 292864, 293392, 822250, 823867, 825484, 827101, 828718, 830335, 831952, 833569, 835186, 836803, 1355860, 1358566, 1361272, 1363978, 1366684, 1369390, 1372096, 1374802, 1377508, 1380214, 1889470, 1893265, 1897060, 1900855, 1904650, 1908445, 1912240, 1916035, 1919830, 1923625, 2423080, 2427964, 2432848, 2437732, 2442616, 2447500, 2452384, 2457268, 2462152, 2467036, 2956690, 2962663, 2968636, 2974609, 2980582, 2986555, 2992528, 2998501, 3004474, 3010447, 3490300, 3497362, 3504424, 3511486, 3518548, 3525610, 3532672, 3539734, 3546796, 3553858, 4023910, 4032061, 4040212, 4048363, 4056514, 4064665, 4072816, 4080967, 4089118, 4097269, 4557520, 4566760, 4576000, 4585240, 4594480, 4603720, 4612960, 4622200, 4631440, 4640680, 1682560, 1692889, 1703218, 1713547, 1723876, 1734205, 1744534, 1754863, 1765192, 1775521, 1856800, 1868218, 1879636, 1891054, 1902472, 1913890, 1925308, 1936726, 1948144, 1959562, 2031040, 2043547, 2056054, 2068561, 2081068, 2093575, 2106082, 2118589, 2131096, 2143603, 2205280, 2218876, 2232472, 2246068, 2259664, 2273260, 2286856, 2300452, 2314048, 2327644, 2379520, 2394205, 2408890, 2423575, 2438260, 2452945, 2467630, 2482315, 2497000, 2511685, 2553760, 2569534, 2585308, 2601082, 2616856, 2632630, 2648404, 2664178, 2679952, 2695726, 2728000, 2744863, 2761726, 2778589, 2795452, 2812315, 2829178, 2846041, 2862904, 2879767, 2902240, 2920192, 2938144, 2956096, 2974048, 2992000, 3009952, 3027904, 3045856, 3063808, 3076480, 3095521, 3114562, 3133603, 3152644, 3171685, 3190726, 3209767, 3228808, 3247849, 5091130, 5101459, 5111788, 5122117, 5132446, 5142775, 5153104, 5163433, 5173762, 5184091, 5624740, 5636158, 5647576, 5658994, 5670412, 5681830, 5693248, 5704666, 5716084, 5727502, 6158350, 6170857, 6183364, 6195871, 6208378, 6220885, 6233392, 6245899, 6258406, 6270913, 6691960, 6705556, 6719152, 6732748, 6746344, 6759940, 6773536, 6787132, 6800728, 6814324, 7225570, 7240255, 7254940, 7269625, 7284310, 7298995, 7313680, 7328365, 7343050, 7357735, 7759180, 7774954, 7790728, 7806502, 7822276, 7838050, 7853824, 7869598, 7885372, 7901146, 8292790, 8309653, 8326516, 8343379, 8360242, 8377105, 8393968, 8410831, 8427694, 8444557, 8826400, 8844352, 8862304, 8880256, 8898208, 8916160, 8934112, 8952064, 8970016, 8987968, 9360010, 9379051, 9398092, 9417133, 9436174, 9455215, 9474256, 9493297, 9512338, 9531379})});

  // Test case: 5* 32, 32 * 67 matrix for vec1
  test_cases.push_back(
      {"test one batch 5 * 32, 32 * 67 matrix",
       {1, 5, 32},
       {1, 32, 67},
       {1, 5, 67},
       real_expected_vals({697872, 698368, 698864, 699360, 699856, 700352, 700848, 701344, 701840, 702336, 702832, 703328, 703824, 704320, 704816, 705312, 705808, 706304, 706800, 707296, 707792, 708288, 708784, 709280, 709776, 710272, 710768, 711264, 711760, 712256, 712752, 713248, 713744, 714240, 714736, 715232, 715728, 716224, 716720, 717216, 717712, 718208, 718704, 719200, 719696, 720192, 720688, 721184, 721680, 722176, 722672, 723168, 723664, 724160, 724656, 725152, 725648, 726144, 726640, 727136, 727632, 728128, 728624, 729120, 729616, 730112, 730608, 1761296, 1762816, 1764336, 1765856, 1767376, 1768896, 1770416, 1771936, 1773456, 1774976, 1776496, 1778016, 1779536, 1781056, 1782576, 1784096, 1785616, 1787136, 1788656, 1790176, 1791696, 1793216, 1794736, 1796256, 1797776, 1799296, 1800816, 1802336, 1803856, 1805376, 1806896, 1808416, 1809936, 1811456, 1812976, 1814496, 1816016, 1817536, 1819056, 1820576, 1822096, 1823616, 1825136, 1826656, 1828176, 1829696, 1831216, 1832736, 1834256, 1835776, 1837296, 1838816, 1840336, 1841856, 1843376, 1844896, 1846416, 1847936, 1849456, 1850976, 1852496, 1854016, 1855536, 1857056, 1858576, 1860096, 1861616, 2824720, 2827264, 2829808, 2832352, 2834896, 2837440, 2839984, 2842528, 2845072, 2847616, 2850160, 2852704, 2855248, 2857792, 2860336, 2862880, 2865424, 2867968, 2870512, 2873056, 2875600, 2878144, 2880688, 2883232, 2885776, 2888320, 2890864, 2893408, 2895952, 2898496, 2901040, 2903584, 2906128, 2908672, 2911216, 2913760, 2916304, 2918848, 2921392, 2923936, 2926480, 2929024, 2931568, 2934112, 2936656, 2939200, 2941744, 2944288, 2946832, 2949376, 2951920, 2954464, 2957008, 2959552, 2962096, 2964640, 2967184, 2969728, 2972272, 2974816, 2977360, 2979904, 2982448, 2984992, 2987536, 2990080, 2992624, 3888144, 3891712, 3895280, 3898848, 3902416, 3905984, 3909552, 3913120, 3916688, 3920256, 3923824, 3927392, 3930960, 3934528, 3938096, 3941664, 3945232, 3948800, 3952368, 3955936, 3959504, 3963072, 3966640, 3970208, 3973776, 3977344, 3980912, 3984480, 3988048, 3991616, 3995184, 3998752, 4002320, 4005888, 4009456, 4013024, 4016592, 4020160, 4023728, 4027296, 4030864, 4034432, 4038000, 4041568, 4045136, 4048704, 4052272, 4055840, 4059408, 4062976, 4066544, 4070112, 4073680, 4077248, 4080816, 4084384, 4087952, 4091520, 4095088, 4098656, 4102224, 4105792, 4109360, 4112928, 4116496, 4120064, 4123632, 4951568, 4956160, 4960752, 4965344, 4969936, 4974528, 4979120, 4983712, 4988304, 4992896, 4997488, 5002080, 5006672, 5011264, 5015856, 5020448, 5025040, 5029632, 5034224, 5038816, 5043408, 5048000, 5052592, 5057184, 5061776, 5066368, 5070960, 5075552, 5080144, 5084736, 5089328, 5093920, 5098512, 5103104, 5107696, 5112288, 5116880, 5121472, 5126064, 5130656, 5135248, 5139840, 5144432, 5149024, 5153616, 5158208, 5162800, 5167392, 5171984, 5176576, 5181168, 5185760, 5190352, 5194944, 5199536, 5204128, 5208720, 5213312, 5217904, 5222496, 5227088, 5231680, 5236272, 5240864, 5245456, 5250048, 5254640})});

  // Test case: 32 * 32 matrix for vec4
  test_cases.push_back(
      {"test one batch 32 * 32 matrix",
       {1, 1, 32, 32},
       {1, 1, 32, 32},
       {1, 1, 32, 32},
       real_expected_vals({333312, 333808, 334304, 334800, 335296, 335792, 336288, 336784, 337280, 337776, 338272, 338768, 339264, 339760, 340256, 340752, 341248, 341744, 342240, 342736, 343232, 343728, 344224, 344720, 345216, 345712, 346208, 346704, 347200, 347696, 348192, 348688, 841216, 842736, 844256, 845776, 847296, 848816, 850336, 851856, 853376, 854896, 856416, 857936, 859456, 860976, 862496, 864016, 865536, 867056, 868576, 870096, 871616, 873136, 874656, 876176, 877696, 879216, 880736, 882256, 883776, 885296, 886816, 888336, 1349120, 1351664, 1354208, 1356752, 1359296, 1361840, 1364384, 1366928, 1369472, 1372016, 1374560, 1377104, 1379648, 1382192, 1384736, 1387280, 1389824, 1392368, 1394912, 1397456, 1400000, 1402544, 1405088, 1407632, 1410176, 1412720, 1415264, 1417808, 1420352, 1422896, 1425440, 1427984, 1857024, 1860592, 1864160, 1867728, 1871296, 1874864, 1878432, 1882000, 1885568, 1889136, 1892704, 1896272, 1899840, 1903408, 1906976, 1910544, 1914112, 1917680, 1921248, 1924816, 1928384, 1931952, 1935520, 1939088, 1942656, 1946224, 1949792, 1953360, 1956928, 1960496, 1964064, 1967632, 2364928, 2369520, 2374112, 2378704, 2383296, 2387888, 2392480, 2397072, 2401664, 2406256, 2410848, 2415440, 2420032, 2424624, 2429216, 2433808, 2438400, 2442992, 2447584, 2452176, 2456768, 2461360, 2465952, 2470544, 2475136, 2479728, 2484320, 2488912, 2493504, 2498096, 2502688, 2507280, 2872832, 2878448, 2884064, 2889680, 2895296, 2900912, 2906528, 2912144, 2917760, 2923376, 2928992, 2934608, 2940224, 2945840, 2951456, 2957072, 2962688, 2968304, 2973920, 2979536, 2985152, 2990768, 2996384, 3002000, 3007616, 3013232, 3018848, 3024464, 3030080, 3035696, 3041312, 3046928, 3380736, 3387376, 3394016, 3400656, 3407296, 3413936, 3420576, 3427216, 3433856, 3440496, 3447136, 3453776, 3460416, 3467056, 3473696, 3480336, 3486976, 3493616, 3500256, 3506896, 3513536, 3520176, 3526816, 3533456, 3540096, 3546736, 3553376, 3560016, 3566656, 3573296, 3579936, 3586576, 3888640, 3896304, 3903968, 3911632, 3919296, 3926960, 3934624, 3942288, 3949952, 3957616, 3965280, 3972944, 3980608, 3988272, 3995936, 4003600, 4011264, 4018928, 4026592, 4034256, 4041920, 4049584, 4057248, 4064912, 4072576, 4080240, 4087904, 4095568, 4103232, 4110896, 4118560, 4126224, 4396544, 4405232, 4413920, 4422608, 4431296, 4439984, 4448672, 4457360, 4466048, 4474736, 4483424, 4492112, 4500800, 4509488, 4518176, 4526864, 4535552, 4544240, 4552928, 4561616, 4570304, 4578992, 4587680, 4596368, 4605056, 4613744, 4622432, 4631120, 4639808, 4648496, 4657184, 4665872, 4904448, 4914160, 4923872, 4933584, 4943296, 4953008, 4962720, 4972432, 4982144, 4991856, 5001568, 5011280, 5020992, 5030704, 5040416, 5050128, 5059840, 5069552, 5079264, 5088976, 5098688, 5108400, 5118112, 5127824, 5137536, 5147248, 5156960, 5166672, 5176384, 5186096, 5195808, 5205520, 5412352, 5423088, 5433824, 5444560, 5455296, 5466032, 5476768, 5487504, 5498240, 5508976, 5519712, 5530448, 5541184, 5551920, 5562656, 5573392, 5584128, 5594864, 5605600, 5616336, 5627072, 5637808, 5648544, 5659280, 5670016, 5680752, 5691488, 5702224, 5712960, 5723696, 5734432, 5745168, 5920256, 5932016, 5943776, 5955536, 5967296, 5979056, 5990816, 6002576, 6014336, 6026096, 6037856, 6049616, 6061376, 6073136, 6084896, 6096656, 6108416, 6120176, 6131936, 6143696, 6155456, 6167216, 6178976, 6190736, 6202496, 6214256, 6226016, 6237776, 6249536, 6261296, 6273056, 6284816, 6428160, 6440944, 6453728, 6466512, 6479296, 6492080, 6504864, 6517648, 6530432, 6543216, 6556000, 6568784, 6581568, 6594352, 6607136, 6619920, 6632704, 6645488, 6658272, 6671056, 6683840, 6696624, 6709408, 6722192, 6734976, 6747760, 6760544, 6773328, 6786112, 6798896, 6811680, 6824464, 6936064, 6949872, 6963680, 6977488, 6991296, 7005104, 7018912, 7032720, 7046528, 7060336, 7074144, 7087952, 7101760, 7115568, 7129376, 7143184, 7156992, 7170800, 7184608, 7198416, 7212224, 7226032, 7239840, 7253648, 7267456, 7281264, 7295072, 7308880, 7322688, 7336496, 7350304, 7364112, 7443968, 7458800, 7473632, 7488464, 7503296, 7518128, 7532960, 7547792, 7562624, 7577456, 7592288, 7607120, 7621952, 7636784, 7651616, 7666448, 7681280, 7696112, 7710944, 7725776, 7740608, 7755440, 7770272, 7785104, 7799936, 7814768, 7829600, 7844432, 7859264, 7874096, 7888928, 7903760, 7951872, 7967728, 7983584, 7999440, 8015296, 8031152, 8047008, 8062864, 8078720, 8094576, 8110432, 8126288, 8142144, 8158000, 8173856, 8189712, 8205568, 8221424, 8237280, 8253136, 8268992, 8284848, 8300704, 8316560, 8332416, 8348272, 8364128, 8379984, 8395840, 8411696, 8427552, 8443408, 8459776, 8476656, 8493536, 8510416, 8527296, 8544176, 8561056, 8577936, 8594816, 8611696, 8628576, 8645456, 8662336, 8679216, 8696096, 8712976, 8729856, 8746736, 8763616, 8780496, 8797376, 8814256, 8831136, 8848016, 8864896, 8881776, 8898656, 8915536, 8932416, 8949296, 8966176, 8983056, 8967680, 8985584, 9003488, 9021392, 9039296, 9057200, 9075104, 9093008, 9110912, 9128816, 9146720, 9164624, 9182528, 9200432, 9218336, 9236240, 9254144, 9272048, 9289952, 9307856, 9325760, 9343664, 9361568, 9379472, 9397376, 9415280, 9433184, 9451088, 9468992, 9486896, 9504800, 9522704, 9475584, 9494512, 9513440, 9532368, 9551296, 9570224, 9589152, 9608080, 9627008, 9645936, 9664864, 9683792, 9702720, 9721648, 9740576, 9759504, 9778432, 9797360, 9816288, 9835216, 9854144, 9873072, 9892000, 9910928, 9929856, 9948784, 9967712, 9986640, 10005568, 10024496, 10043424, 10062352, 9983488, 10003440, 10023392, 10043344, 10063296, 10083248, 10103200, 10123152, 10143104, 10163056, 10183008, 10202960, 10222912, 10242864, 10262816, 10282768, 10302720, 10322672, 10342624, 10362576, 10382528, 10402480, 10422432, 10442384, 10462336, 10482288, 10502240, 10522192, 10542144, 10562096, 10582048, 10602000, 10491392, 10512368, 10533344, 10554320, 10575296, 10596272, 10617248, 10638224, 10659200, 10680176, 10701152, 10722128, 10743104, 10764080, 10785056, 10806032, 10827008, 10847984, 10868960, 10889936, 10910912, 10931888, 10952864, 10973840, 10994816, 11015792, 11036768, 11057744, 11078720, 11099696, 11120672, 11141648, 10999296, 11021296, 11043296, 11065296, 11087296, 11109296, 11131296, 11153296, 11175296, 11197296, 11219296, 11241296, 11263296, 11285296, 11307296, 11329296, 11351296, 11373296, 11395296, 11417296, 11439296, 11461296, 11483296, 11505296, 11527296, 11549296, 11571296, 11593296, 11615296, 11637296, 11659296, 11681296, 11507200, 11530224, 11553248, 11576272, 11599296, 11622320, 11645344, 11668368, 11691392, 11714416, 11737440, 11760464, 11783488, 11806512, 11829536, 11852560, 11875584, 11898608, 11921632, 11944656, 11967680, 11990704, 12013728, 12036752, 12059776, 12082800, 12105824, 12128848, 12151872, 12174896, 12197920, 12220944, 12015104, 12039152, 12063200, 12087248, 12111296, 12135344, 12159392, 12183440, 12207488, 12231536, 12255584, 12279632, 12303680, 12327728, 12351776, 12375824, 12399872, 12423920, 12447968, 12472016, 12496064, 12520112, 12544160, 12568208, 12592256, 12616304, 12640352, 12664400, 12688448, 12712496, 12736544, 12760592, 12523008, 12548080, 12573152, 12598224, 12623296, 12648368, 12673440, 12698512, 12723584, 12748656, 12773728, 12798800, 12823872, 12848944, 12874016, 12899088, 12924160, 12949232, 12974304, 12999376, 13024448, 13049520, 13074592, 13099664, 13124736, 13149808, 13174880, 13199952, 13225024, 13250096, 13275168, 13300240, 13030912, 13057008, 13083104, 13109200, 13135296, 13161392, 13187488, 13213584, 13239680, 13265776, 13291872, 13317968, 13344064, 13370160, 13396256, 13422352, 13448448, 13474544, 13500640, 13526736, 13552832, 13578928, 13605024, 13631120, 13657216, 13683312, 13709408, 13735504, 13761600, 13787696, 13813792, 13839888, 13538816, 13565936, 13593056, 13620176, 13647296, 13674416, 13701536, 13728656, 13755776, 13782896, 13810016, 13837136, 13864256, 13891376, 13918496, 13945616, 13972736, 13999856, 14026976, 14054096, 14081216, 14108336, 14135456, 14162576, 14189696, 14216816, 14243936, 14271056, 14298176, 14325296, 14352416, 14379536, 14046720, 14074864, 14103008, 14131152, 14159296, 14187440, 14215584, 14243728, 14271872, 14300016, 14328160, 14356304, 14384448, 14412592, 14440736, 14468880, 14497024, 14525168, 14553312, 14581456, 14609600, 14637744, 14665888, 14694032, 14722176, 14750320, 14778464, 14806608, 14834752, 14862896, 14891040, 14919184, 14554624, 14583792, 14612960, 14642128, 14671296, 14700464, 14729632, 14758800, 14787968, 14817136, 14846304, 14875472, 14904640, 14933808, 14962976, 14992144, 15021312, 15050480, 15079648, 15108816, 15137984, 15167152, 15196320, 15225488, 15254656, 15283824, 15312992, 15342160, 15371328, 15400496, 15429664, 15458832, 15062528, 15092720, 15122912, 15153104, 15183296, 15213488, 15243680, 15273872, 15304064, 15334256, 15364448, 15394640, 15424832, 15455024, 15485216, 15515408, 15545600, 15575792, 15605984, 15636176, 15666368, 15696560, 15726752, 15756944, 15787136, 15817328, 15847520, 15877712, 15907904, 15938096, 15968288, 15998480, 15570432, 15601648, 15632864, 15664080, 15695296, 15726512, 15757728, 15788944, 15820160, 15851376, 15882592, 15913808, 15945024, 15976240, 16007456, 16038672, 16069888, 16101104, 16132320, 16163536, 16194752, 16225968, 16257184, 16288400, 16319616, 16350832, 16382048, 16413264, 16444480, 16475696, 16506912, 16538128, 16078336, 16110576, 16142816, 16175056, 16207296, 16239536, 16271776, 16304016, 16336256, 16368496, 16400736, 16432976, 16465216, 16497456, 16529696, 16561936, 16594176, 16626416, 16658656, 16690896, 16723136, 16755376, 16787616, 16819856, 16852096, 16884336, 16916576, 16948816, 16981056, 17013296, 17045536, 17077776})});

  // Test case: 34 * 68, 68 * 72 for vec4 and multiple workgroups
  test_cases.push_back(
      {"test 34 * 68, 68 * 72",
       {34, 68},
       {68, 72},
       {34, 72},
       real_expected_vals({7380720, 7382998, 7385276, 7387554, 7389832, 7392110, 7394388, 7396666, 7398944, 7401222, 7403500, 7405778, 7408056, 7410334, 7412612, 7414890, 7417168, 7419446, 7421724, 7424002, 7426280, 7428558, 7430836, 7433114, 7435392, 7437670, 7439948, 7442226, 7444504, 7446782, 7449060, 7451338, 7453616, 7455894, 7458172, 7460450, 7462728, 7465006, 7467284, 7469562, 7471840, 7474118, 7476396, 7478674, 7480952, 7483230, 7485508, 7487786, 7490064, 7492342, 7494620, 7496898, 7499176, 7501454, 7503732, 7506010, 7508288, 7510566, 7512844, 7515122, 7517400, 7519678, 7521956, 7524234, 7526512, 7528790, 7531068, 7533346, 7535624, 7537902, 7540180, 7542458, 18533808, 18540710, 18547612, 18554514, 18561416, 18568318, 18575220, 18582122, 18589024, 18595926, 18602828, 18609730, 18616632, 18623534, 18630436, 18637338, 18644240, 18651142, 18658044, 18664946, 18671848, 18678750, 18685652, 18692554, 18699456, 18706358, 18713260, 18720162, 18727064, 18733966, 18740868, 18747770, 18754672, 18761574, 18768476, 18775378, 18782280, 18789182, 18796084, 18802986, 18809888, 18816790, 18823692, 18830594, 18837496, 18844398, 18851300, 18858202, 18865104, 18872006, 18878908, 18885810, 18892712, 18899614, 18906516, 18913418, 18920320, 18927222, 18934124, 18941026, 18947928, 18954830, 18961732, 18968634, 18975536, 18982438, 18989340, 18996242, 19003144, 19010046, 19016948, 19023850, 29686896, 29698422, 29709948, 29721474, 29733000, 29744526, 29756052, 29767578, 29779104, 29790630, 29802156, 29813682, 29825208, 29836734, 29848260, 29859786, 29871312, 29882838, 29894364, 29905890, 29917416, 29928942, 29940468, 29951994, 29963520, 29975046, 29986572, 29998098, 30009624, 30021150, 30032676, 30044202, 30055728, 30067254, 30078780, 30090306, 30101832, 30113358, 30124884, 30136410, 30147936, 30159462, 30170988, 30182514, 30194040, 30205566, 30217092, 30228618, 30240144, 30251670, 30263196, 30274722, 30286248, 30297774, 30309300, 30320826, 30332352, 30343878, 30355404, 30366930, 30378456, 30389982, 30401508, 30413034, 30424560, 30436086, 30447612, 30459138, 30470664, 30482190, 30493716, 30505242, 40839984, 40856134, 40872284, 40888434, 40904584, 40920734, 40936884, 40953034, 40969184, 40985334, 41001484, 41017634, 41033784, 41049934, 41066084, 41082234, 41098384, 41114534, 41130684, 41146834, 41162984, 41179134, 41195284, 41211434, 41227584, 41243734, 41259884, 41276034, 41292184, 41308334, 41324484, 41340634, 41356784, 41372934, 41389084, 41405234, 41421384, 41437534, 41453684, 41469834, 41485984, 41502134, 41518284, 41534434, 41550584, 41566734, 41582884, 41599034, 41615184, 41631334, 41647484, 41663634, 41679784, 41695934, 41712084, 41728234, 41744384, 41760534, 41776684, 41792834, 41808984, 41825134, 41841284, 41857434, 41873584, 41889734, 41905884, 41922034, 41938184, 41954334, 41970484, 41986634, 51993072, 52013846, 52034620, 52055394, 52076168, 52096942, 52117716, 52138490, 52159264, 52180038, 52200812, 52221586, 52242360, 52263134, 52283908, 52304682, 52325456, 52346230, 52367004, 52387778, 52408552, 52429326, 52450100, 52470874, 52491648, 52512422, 52533196, 52553970, 52574744, 52595518, 52616292, 52637066, 52657840, 52678614, 52699388, 52720162, 52740936, 52761710, 52782484, 52803258, 52824032, 52844806, 52865580, 52886354, 52907128, 52927902, 52948676, 52969450, 52990224, 53010998, 53031772, 53052546, 53073320, 53094094, 53114868, 53135642, 53156416, 53177190, 53197964, 53218738, 53239512, 53260286, 53281060, 53301834, 53322608, 53343382, 53364156, 53384930, 53405704, 53426478, 53447252, 53468026, 63146160, 63171558, 63196956, 63222354, 63247752, 63273150, 63298548, 63323946, 63349344, 63374742, 63400140, 63425538, 63450936, 63476334, 63501732, 63527130, 63552528, 63577926, 63603324, 63628722, 63654120, 63679518, 63704916, 63730314, 63755712, 63781110, 63806508, 63831906, 63857304, 63882702, 63908100, 63933498, 63958896, 63984294, 64009692, 64035090, 64060488, 64085886, 64111284, 64136682, 64162080, 64187478, 64212876, 64238274, 64263672, 64289070, 64314468, 64339866, 64365264, 64390662, 64416060, 64441458, 64466856, 64492254, 64517652, 64543050, 64568448, 64593846, 64619244, 64644642, 64670040, 64695438, 64720836, 64746234, 64771632, 64797030, 64822428, 64847826, 64873224, 64898622, 64924020, 64949418, 74299248, 74329270, 74359292, 74389314, 74419336, 74449358, 74479380, 74509402, 74539424, 74569446, 74599468, 74629490, 74659512, 74689534, 74719556, 74749578, 74779600, 74809622, 74839644, 74869666, 74899688, 74929710, 74959732, 74989754, 75019776, 75049798, 75079820, 75109842, 75139864, 75169886, 75199908, 75229930, 75259952, 75289974, 75319996, 75350018, 75380040, 75410062, 75440084, 75470106, 75500128, 75530150, 75560172, 75590194, 75620216, 75650238, 75680260, 75710282, 75740304, 75770326, 75800348, 75830370, 75860392, 75890414, 75920436, 75950458, 75980480, 76010502, 76040524, 76070546, 76100568, 76130590, 76160612, 76190634, 76220656, 76250678, 76280700, 76310722, 76340744, 76370766, 76400788, 76430810, 85452336, 85486982, 85521628, 85556274, 85590920, 85625566, 85660212, 85694858, 85729504, 85764150, 85798796, 85833442, 85868088, 85902734, 85937380, 85972026, 86006672, 86041318, 86075964, 86110610, 86145256, 86179902, 86214548, 86249194, 86283840, 86318486, 86353132, 86387778, 86422424, 86457070, 86491716, 86526362, 86561008, 86595654, 86630300, 86664946, 86699592, 86734238, 86768884, 86803530, 86838176, 86872822, 86907468, 86942114, 86976760, 87011406, 87046052, 87080698, 87115344, 87149990, 87184636, 87219282, 87253928, 87288574, 87323220, 87357866, 87392512, 87427158, 87461804, 87496450, 87531096, 87565742, 87600388, 87635034, 87669680, 87704326, 87738972, 87773618, 87808264, 87842910, 87877556, 87912202, 96605424, 96644694, 96683964, 96723234, 96762504, 96801774, 96841044, 96880314, 96919584, 96958854, 96998124, 97037394, 97076664, 97115934, 97155204, 97194474, 97233744, 97273014, 97312284, 97351554, 97390824, 97430094, 97469364, 97508634, 97547904, 97587174, 97626444, 97665714, 97704984, 97744254, 97783524, 97822794, 97862064, 97901334, 97940604, 97979874, 98019144, 98058414, 98097684, 98136954, 98176224, 98215494, 98254764, 98294034, 98333304, 98372574, 98411844, 98451114, 98490384, 98529654, 98568924, 98608194, 98647464, 98686734, 98726004, 98765274, 98804544, 98843814, 98883084, 98922354, 98961624, 99000894, 99040164, 99079434, 99118704, 99157974, 99197244, 99236514, 99275784, 99315054, 99354324, 99393594, 107758512, 107802406, 107846300, 107890194, 107934088, 107977982, 108021876, 108065770, 108109664, 108153558, 108197452, 108241346, 108285240, 108329134, 108373028, 108416922, 108460816, 108504710, 108548604, 108592498, 108636392, 108680286, 108724180, 108768074, 108811968, 108855862, 108899756, 108943650, 108987544, 109031438, 109075332, 109119226, 109163120, 109207014, 109250908, 109294802, 109338696, 109382590, 109426484, 109470378, 109514272, 109558166, 109602060, 109645954, 109689848, 109733742, 109777636, 109821530, 109865424, 109909318, 109953212, 109997106, 110041000, 110084894, 110128788, 110172682, 110216576, 110260470, 110304364, 110348258, 110392152, 110436046, 110479940, 110523834, 110567728, 110611622, 110655516, 110699410, 110743304, 110787198, 110831092, 110874986, 118911600, 118960118, 119008636, 119057154, 119105672, 119154190, 119202708, 119251226, 119299744, 119348262, 119396780, 119445298, 119493816, 119542334, 119590852, 119639370, 119687888, 119736406, 119784924, 119833442, 119881960, 119930478, 119978996, 120027514, 120076032, 120124550, 120173068, 120221586, 120270104, 120318622, 120367140, 120415658, 120464176, 120512694, 120561212, 120609730, 120658248, 120706766, 120755284, 120803802, 120852320, 120900838, 120949356, 120997874, 121046392, 121094910, 121143428, 121191946, 121240464, 121288982, 121337500, 121386018, 121434536, 121483054, 121531572, 121580090, 121628608, 121677126, 121725644, 121774162, 121822680, 121871198, 121919716, 121968234, 122016752, 122065270, 122113788, 122162306, 122210824, 122259342, 122307860, 122356378, 130064688, 130117830, 130170972, 130224114, 130277256, 130330398, 130383540, 130436682, 130489824, 130542966, 130596108, 130649250, 130702392, 130755534, 130808676, 130861818, 130914960, 130968102, 131021244, 131074386, 131127528, 131180670, 131233812, 131286954, 131340096, 131393238, 131446380, 131499522, 131552664, 131605806, 131658948, 131712090, 131765232, 131818374, 131871516, 131924658, 131977800, 132030942, 132084084, 132137226, 132190368, 132243510, 132296652, 132349794, 132402936, 132456078, 132509220, 132562362, 132615504, 132668646, 132721788, 132774930, 132828072, 132881214, 132934356, 132987498, 133040640, 133093782, 133146924, 133200066, 133253208, 133306350, 133359492, 133412634, 133465776, 133518918, 133572060, 133625202, 133678344, 133731486, 133784628, 133837770, 141217776, 141275542, 141333308, 141391074, 141448840, 141506606, 141564372, 141622138, 141679904, 141737670, 141795436, 141853202, 141910968, 141968734, 142026500, 142084266, 142142032, 142199798, 142257564, 142315330, 142373096, 142430862, 142488628, 142546394, 142604160, 142661926, 142719692, 142777458, 142835224, 142892990, 142950756, 143008522, 143066288, 143124054, 143181820, 143239586, 143297352, 143355118, 143412884, 143470650, 143528416, 143586182, 143643948, 143701714, 143759480, 143817246, 143875012, 143932778, 143990544, 144048310, 144106076, 144163842, 144221608, 144279374, 144337140, 144394906, 144452672, 144510438, 144568204, 144625970, 144683736, 144741502, 144799268, 144857034, 144914800, 144972566, 145030332, 145088098, 145145864, 145203630, 145261396, 145319162, 152370864, 152433254, 152495644, 152558034, 152620424, 152682814, 152745204, 152807594, 152869984, 152932374, 152994764, 153057154, 153119544, 153181934, 153244324, 153306714, 153369104, 153431494, 153493884, 153556274, 153618664, 153681054, 153743444, 153805834, 153868224, 153930614, 153993004, 154055394, 154117784, 154180174, 154242564, 154304954, 154367344, 154429734, 154492124, 154554514, 154616904, 154679294, 154741684, 154804074, 154866464, 154928854, 154991244, 155053634, 155116024, 155178414, 155240804, 155303194, 155365584, 155427974, 155490364, 155552754, 155615144, 155677534, 155739924, 155802314, 155864704, 155927094, 155989484, 156051874, 156114264, 156176654, 156239044, 156301434, 156363824, 156426214, 156488604, 156550994, 156613384, 156675774, 156738164, 156800554, 163523952, 163590966, 163657980, 163724994, 163792008, 163859022, 163926036, 163993050, 164060064, 164127078, 164194092, 164261106, 164328120, 164395134, 164462148, 164529162, 164596176, 164663190, 164730204, 164797218, 164864232, 164931246, 164998260, 165065274, 165132288, 165199302, 165266316, 165333330, 165400344, 165467358, 165534372, 165601386, 165668400, 165735414, 165802428, 165869442, 165936456, 166003470, 166070484, 166137498, 166204512, 166271526, 166338540, 166405554, 166472568, 166539582, 166606596, 166673610, 166740624, 166807638, 166874652, 166941666, 167008680, 167075694, 167142708, 167209722, 167276736, 167343750, 167410764, 167477778, 167544792, 167611806, 167678820, 167745834, 167812848, 167879862, 167946876, 168013890, 168080904, 168147918, 168214932, 168281946, 174677040, 174748678, 174820316, 174891954, 174963592, 175035230, 175106868, 175178506, 175250144, 175321782, 175393420, 175465058, 175536696, 175608334, 175679972, 175751610, 175823248, 175894886, 175966524, 176038162, 176109800, 176181438, 176253076, 176324714, 176396352, 176467990, 176539628, 176611266, 176682904, 176754542, 176826180, 176897818, 176969456, 177041094, 177112732, 177184370, 177256008, 177327646, 177399284, 177470922, 177542560, 177614198, 177685836, 177757474, 177829112, 177900750, 177972388, 178044026, 178115664, 178187302, 178258940, 178330578, 178402216, 178473854, 178545492, 178617130, 178688768, 178760406, 178832044, 178903682, 178975320, 179046958, 179118596, 179190234, 179261872, 179333510, 179405148, 179476786, 179548424, 179620062, 179691700, 179763338, 185830128, 185906390, 185982652, 186058914, 186135176, 186211438, 186287700, 186363962, 186440224, 186516486, 186592748, 186669010, 186745272, 186821534, 186897796, 186974058, 187050320, 187126582, 187202844, 187279106, 187355368, 187431630, 187507892, 187584154, 187660416, 187736678, 187812940, 187889202, 187965464, 188041726, 188117988, 188194250, 188270512, 188346774, 188423036, 188499298, 188575560, 188651822, 188728084, 188804346, 188880608, 188956870, 189033132, 189109394, 189185656, 189261918, 189338180, 189414442, 189490704, 189566966, 189643228, 189719490, 189795752, 189872014, 189948276, 190024538, 190100800, 190177062, 190253324, 190329586, 190405848, 190482110, 190558372, 190634634, 190710896, 190787158, 190863420, 190939682, 191015944, 191092206, 191168468, 191244730, 196983216, 197064102, 197144988, 197225874, 197306760, 197387646, 197468532, 197549418, 197630304, 197711190, 197792076, 197872962, 197953848, 198034734, 198115620, 198196506, 198277392, 198358278, 198439164, 198520050, 198600936, 198681822, 198762708, 198843594, 198924480, 199005366, 199086252, 199167138, 199248024, 199328910, 199409796, 199490682, 199571568, 199652454, 199733340, 199814226, 199895112, 199975998, 200056884, 200137770, 200218656, 200299542, 200380428, 200461314, 200542200, 200623086, 200703972, 200784858, 200865744, 200946630, 201027516, 201108402, 201189288, 201270174, 201351060, 201431946, 201512832, 201593718, 201674604, 201755490, 201836376, 201917262, 201998148, 202079034, 202159920, 202240806, 202321692, 202402578, 202483464, 202564350, 202645236, 202726122, 208136304, 208221814, 208307324, 208392834, 208478344, 208563854, 208649364, 208734874, 208820384, 208905894, 208991404, 209076914, 209162424, 209247934, 209333444, 209418954, 209504464, 209589974, 209675484, 209760994, 209846504, 209932014, 210017524, 210103034, 210188544, 210274054, 210359564, 210445074, 210530584, 210616094, 210701604, 210787114, 210872624, 210958134, 211043644, 211129154, 211214664, 211300174, 211385684, 211471194, 211556704, 211642214, 211727724, 211813234, 211898744, 211984254, 212069764, 212155274, 212240784, 212326294, 212411804, 212497314, 212582824, 212668334, 212753844, 212839354, 212924864, 213010374, 213095884, 213181394, 213266904, 213352414, 213437924, 213523434, 213608944, 213694454, 213779964, 213865474, 213950984, 214036494, 214122004, 214207514, 219289392, 219379526, 219469660, 219559794, 219649928, 219740062, 219830196, 219920330, 220010464, 220100598, 220190732, 220280866, 220371000, 220461134, 220551268, 220641402, 220731536, 220821670, 220911804, 221001938, 221092072, 221182206, 221272340, 221362474, 221452608, 221542742, 221632876, 221723010, 221813144, 221903278, 221993412, 222083546, 222173680, 222263814, 222353948, 222444082, 222534216, 222624350, 222714484, 222804618, 222894752, 222984886, 223075020, 223165154, 223255288, 223345422, 223435556, 223525690, 223615824, 223705958, 223796092, 223886226, 223976360, 224066494, 224156628, 224246762, 224336896, 224427030, 224517164, 224607298, 224697432, 224787566, 224877700, 224967834, 225057968, 225148102, 225238236, 225328370, 225418504, 225508638, 225598772, 225688906, 230442480, 230537238, 230631996, 230726754, 230821512, 230916270, 231011028, 231105786, 231200544, 231295302, 231390060, 231484818, 231579576, 231674334, 231769092, 231863850, 231958608, 232053366, 232148124, 232242882, 232337640, 232432398, 232527156, 232621914, 232716672, 232811430, 232906188, 233000946, 233095704, 233190462, 233285220, 233379978, 233474736, 233569494, 233664252, 233759010, 233853768, 233948526, 234043284, 234138042, 234232800, 234327558, 234422316, 234517074, 234611832, 234706590, 234801348, 234896106, 234990864, 235085622, 235180380, 235275138, 235369896, 235464654, 235559412, 235654170, 235748928, 235843686, 235938444, 236033202, 236127960, 236222718, 236317476, 236412234, 236506992, 236601750, 236696508, 236791266, 236886024, 236980782, 237075540, 237170298, 241595568, 241694950, 241794332, 241893714, 241993096, 242092478, 242191860, 242291242, 242390624, 242490006, 242589388, 242688770, 242788152, 242887534, 242986916, 243086298, 243185680, 243285062, 243384444, 243483826, 243583208, 243682590, 243781972, 243881354, 243980736, 244080118, 244179500, 244278882, 244378264, 244477646, 244577028, 244676410, 244775792, 244875174, 244974556, 245073938, 245173320, 245272702, 245372084, 245471466, 245570848, 245670230, 245769612, 245868994, 245968376, 246067758, 246167140, 246266522, 246365904, 246465286, 246564668, 246664050, 246763432, 246862814, 246962196, 247061578, 247160960, 247260342, 247359724, 247459106, 247558488, 247657870, 247757252, 247856634, 247956016, 248055398, 248154780, 248254162, 248353544, 248452926, 248552308, 248651690, 252748656, 252852662, 252956668, 253060674, 253164680, 253268686, 253372692, 253476698, 253580704, 253684710, 253788716, 253892722, 253996728, 254100734, 254204740, 254308746, 254412752, 254516758, 254620764, 254724770, 254828776, 254932782, 255036788, 255140794, 255244800, 255348806, 255452812, 255556818, 255660824, 255764830, 255868836, 255972842, 256076848, 256180854, 256284860, 256388866, 256492872, 256596878, 256700884, 256804890, 256908896, 257012902, 257116908, 257220914, 257324920, 257428926, 257532932, 257636938, 257740944, 257844950, 257948956, 258052962, 258156968, 258260974, 258364980, 258468986, 258572992, 258676998, 258781004, 258885010, 258989016, 259093022, 259197028, 259301034, 259405040, 259509046, 259613052, 259717058, 259821064, 259925070, 260029076, 260133082, 263901744, 264010374, 264119004, 264227634, 264336264, 264444894, 264553524, 264662154, 264770784, 264879414, 264988044, 265096674, 265205304, 265313934, 265422564, 265531194, 265639824, 265748454, 265857084, 265965714, 266074344, 266182974, 266291604, 266400234, 266508864, 266617494, 266726124, 266834754, 266943384, 267052014, 267160644, 267269274, 267377904, 267486534, 267595164, 267703794, 267812424, 267921054, 268029684, 268138314, 268246944, 268355574, 268464204, 268572834, 268681464, 268790094, 268898724, 269007354, 269115984, 269224614, 269333244, 269441874, 269550504, 269659134, 269767764, 269876394, 269985024, 270093654, 270202284, 270310914, 270419544, 270528174, 270636804, 270745434, 270854064, 270962694, 271071324, 271179954, 271288584, 271397214, 271505844, 271614474, 275054832, 275168086, 275281340, 275394594, 275507848, 275621102, 275734356, 275847610, 275960864, 276074118, 276187372, 276300626, 276413880, 276527134, 276640388, 276753642, 276866896, 276980150, 277093404, 277206658, 277319912, 277433166, 277546420, 277659674, 277772928, 277886182, 277999436, 278112690, 278225944, 278339198, 278452452, 278565706, 278678960, 278792214, 278905468, 279018722, 279131976, 279245230, 279358484, 279471738, 279584992, 279698246, 279811500, 279924754, 280038008, 280151262, 280264516, 280377770, 280491024, 280604278, 280717532, 280830786, 280944040, 281057294, 281170548, 281283802, 281397056, 281510310, 281623564, 281736818, 281850072, 281963326, 282076580, 282189834, 282303088, 282416342, 282529596, 282642850, 282756104, 282869358, 282982612, 283095866, 286207920, 286325798, 286443676, 286561554, 286679432, 286797310, 286915188, 287033066, 287150944, 287268822, 287386700, 287504578, 287622456, 287740334, 287858212, 287976090, 288093968, 288211846, 288329724, 288447602, 288565480, 288683358, 288801236, 288919114, 289036992, 289154870, 289272748, 289390626, 289508504, 289626382, 289744260, 289862138, 289980016, 290097894, 290215772, 290333650, 290451528, 290569406, 290687284, 290805162, 290923040, 291040918, 291158796, 291276674, 291394552, 291512430, 291630308, 291748186, 291866064, 291983942, 292101820, 292219698, 292337576, 292455454, 292573332, 292691210, 292809088, 292926966, 293044844, 293162722, 293280600, 293398478, 293516356, 293634234, 293752112, 293869990, 293987868, 294105746, 294223624, 294341502, 294459380, 294577258, 297361008, 297483510, 297606012, 297728514, 297851016, 297973518, 298096020, 298218522, 298341024, 298463526, 298586028, 298708530, 298831032, 298953534, 299076036, 299198538, 299321040, 299443542, 299566044, 299688546, 299811048, 299933550, 300056052, 300178554, 300301056, 300423558, 300546060, 300668562, 300791064, 300913566, 301036068, 301158570, 301281072, 301403574, 301526076, 301648578, 301771080, 301893582, 302016084, 302138586, 302261088, 302383590, 302506092, 302628594, 302751096, 302873598, 302996100, 303118602, 303241104, 303363606, 303486108, 303608610, 303731112, 303853614, 303976116, 304098618, 304221120, 304343622, 304466124, 304588626, 304711128, 304833630, 304956132, 305078634, 305201136, 305323638, 305446140, 305568642, 305691144, 305813646, 305936148, 306058650, 308514096, 308641222, 308768348, 308895474, 309022600, 309149726, 309276852, 309403978, 309531104, 309658230, 309785356, 309912482, 310039608, 310166734, 310293860, 310420986, 310548112, 310675238, 310802364, 310929490, 311056616, 311183742, 311310868, 311437994, 311565120, 311692246, 311819372, 311946498, 312073624, 312200750, 312327876, 312455002, 312582128, 312709254, 312836380, 312963506, 313090632, 313217758, 313344884, 313472010, 313599136, 313726262, 313853388, 313980514, 314107640, 314234766, 314361892, 314489018, 314616144, 314743270, 314870396, 314997522, 315124648, 315251774, 315378900, 315506026, 315633152, 315760278, 315887404, 316014530, 316141656, 316268782, 316395908, 316523034, 316650160, 316777286, 316904412, 317031538, 317158664, 317285790, 317412916, 317540042, 319667184, 319798934, 319930684, 320062434, 320194184, 320325934, 320457684, 320589434, 320721184, 320852934, 320984684, 321116434, 321248184, 321379934, 321511684, 321643434, 321775184, 321906934, 322038684, 322170434, 322302184, 322433934, 322565684, 322697434, 322829184, 322960934, 323092684, 323224434, 323356184, 323487934, 323619684, 323751434, 323883184, 324014934, 324146684, 324278434, 324410184, 324541934, 324673684, 324805434, 324937184, 325068934, 325200684, 325332434, 325464184, 325595934, 325727684, 325859434, 325991184, 326122934, 326254684, 326386434, 326518184, 326649934, 326781684, 326913434, 327045184, 327176934, 327308684, 327440434, 327572184, 327703934, 327835684, 327967434, 328099184, 328230934, 328362684, 328494434, 328626184, 328757934, 328889684, 329021434, 330820272, 330956646, 331093020, 331229394, 331365768, 331502142, 331638516, 331774890, 331911264, 332047638, 332184012, 332320386, 332456760, 332593134, 332729508, 332865882, 333002256, 333138630, 333275004, 333411378, 333547752, 333684126, 333820500, 333956874, 334093248, 334229622, 334365996, 334502370, 334638744, 334775118, 334911492, 335047866, 335184240, 335320614, 335456988, 335593362, 335729736, 335866110, 336002484, 336138858, 336275232, 336411606, 336547980, 336684354, 336820728, 336957102, 337093476, 337229850, 337366224, 337502598, 337638972, 337775346, 337911720, 338048094, 338184468, 338320842, 338457216, 338593590, 338729964, 338866338, 339002712, 339139086, 339275460, 339411834, 339548208, 339684582, 339820956, 339957330, 340093704, 340230078, 340366452, 340502826, 341973360, 342114358, 342255356, 342396354, 342537352, 342678350, 342819348, 342960346, 343101344, 343242342, 343383340, 343524338, 343665336, 343806334, 343947332, 344088330, 344229328, 344370326, 344511324, 344652322, 344793320, 344934318, 345075316, 345216314, 345357312, 345498310, 345639308, 345780306, 345921304, 346062302, 346203300, 346344298, 346485296, 346626294, 346767292, 346908290, 347049288, 347190286, 347331284, 347472282, 347613280, 347754278, 347895276, 348036274, 348177272, 348318270, 348459268, 348600266, 348741264, 348882262, 349023260, 349164258, 349305256, 349446254, 349587252, 349728250, 349869248, 350010246, 350151244, 350292242, 350433240, 350574238, 350715236, 350856234, 350997232, 351138230, 351279228, 351420226, 351561224, 351702222, 351843220, 351984218, 353126448, 353272070, 353417692, 353563314, 353708936, 353854558, 354000180, 354145802, 354291424, 354437046, 354582668, 354728290, 354873912, 355019534, 355165156, 355310778, 355456400, 355602022, 355747644, 355893266, 356038888, 356184510, 356330132, 356475754, 356621376, 356766998, 356912620, 357058242, 357203864, 357349486, 357495108, 357640730, 357786352, 357931974, 358077596, 358223218, 358368840, 358514462, 358660084, 358805706, 358951328, 359096950, 359242572, 359388194, 359533816, 359679438, 359825060, 359970682, 360116304, 360261926, 360407548, 360553170, 360698792, 360844414, 360990036, 361135658, 361281280, 361426902, 361572524, 361718146, 361863768, 362009390, 362155012, 362300634, 362446256, 362591878, 362737500, 362883122, 363028744, 363174366, 363319988, 363465610, 364279536, 364429782, 364580028, 364730274, 364880520, 365030766, 365181012, 365331258, 365481504, 365631750, 365781996, 365932242, 366082488, 366232734, 366382980, 366533226, 366683472, 366833718, 366983964, 367134210, 367284456, 367434702, 367584948, 367735194, 367885440, 368035686, 368185932, 368336178, 368486424, 368636670, 368786916, 368937162, 369087408, 369237654, 369387900, 369538146, 369688392, 369838638, 369988884, 370139130, 370289376, 370439622, 370589868, 370740114, 370890360, 371040606, 371190852, 371341098, 371491344, 371641590, 371791836, 371942082, 372092328, 372242574, 372392820, 372543066, 372693312, 372843558, 372993804, 373144050, 373294296, 373444542, 373594788, 373745034, 373895280, 374045526, 374195772, 374346018, 374496264, 374646510, 374796756, 374947002, 375432624, 375587494, 375742364, 375897234, 376052104, 376206974, 376361844, 376516714, 376671584, 376826454, 376981324, 377136194, 377291064, 377445934, 377600804, 377755674, 377910544, 378065414, 378220284, 378375154, 378530024, 378684894, 378839764, 378994634, 379149504, 379304374, 379459244, 379614114, 379768984, 379923854, 380078724, 380233594, 380388464, 380543334, 380698204, 380853074, 381007944, 381162814, 381317684, 381472554, 381627424, 381782294, 381937164, 382092034, 382246904, 382401774, 382556644, 382711514, 382866384, 383021254, 383176124, 383330994, 383485864, 383640734, 383795604, 383950474, 384105344, 384260214, 384415084, 384569954, 384724824, 384879694, 385034564, 385189434, 385344304, 385499174, 385654044, 385808914, 385963784, 386118654, 386273524, 386428394})});
#endif
  return test_cases;
}

template <typename T>
void RunMatMulTest(int32_t opset_version, bool is_a_constant, bool is_b_constant) {
  for (auto t : GenerateTestCases<T>()) {
    SCOPED_TRACE("test case: " + t.name);

    OpTester test("MatMul", opset_version);

    int64_t size0 = TensorShape::FromExistingBuffer(t.input0_dims).SizeHelper(0, t.input0_dims.size());
    std::vector<T> input0_vals = ValueRange<T>(size0);
    test.AddInput<T>("A", t.input0_dims, input0_vals, is_a_constant);

    int64_t size1 = TensorShape::FromExistingBuffer(t.input1_dims).SizeHelper(0, t.input1_dims.size());
    std::vector<T> input1_vals = ValueRange<T>(size1);
    test.AddInput<T>("B", t.input1_dims, input1_vals, is_b_constant);

    test.AddOutput<T>("Y", t.expected_dims, t.expected_vals);

    // OpenVINO EP: Disabled temporarily matmul broadcasting not fully supported
    // Disable TensorRT because of unsupported data type
    // QNN EP: Crash during graph execution for QNN's CPU backend on QNN SDK 2.22. Not a problem for QNN's HTP backend.
    std::unordered_set<std::string> excluded_providers{kTensorrtExecutionProvider,
                                                       kOpenVINOExecutionProvider,
                                                       kQnnExecutionProvider};
    if (t.name == "test 2D empty input") {
      // NNAPI: currently fails for the "test 2D empty input" case
      excluded_providers.insert(kNnapiExecutionProvider);
    }

    // TODO:: Change MatMulNaive Shader to support these test cases webgpu
    std::unordered_set<std::string> webgpu_excluded_test_cases{
        "test left 1D",
        "test right 1D",
        "test 2D empty input"};

    // if test in webgpu_excluded_test_cases, add webgpu to excluded_providers
    if (webgpu_excluded_test_cases.find(t.name) != webgpu_excluded_test_cases.end()) {
      excluded_providers.insert(kWebGpuExecutionProvider);
    }

    test.ConfigExcludeEps(excluded_providers)
        .Config(run_with_tunable_op)
        .RunWithConfig();
  }
}

template <typename T>
void RunMatMulTest(int32_t opset_version) {
  RunMatMulTest<T>(opset_version, false, false);
}

TEST(MathOpTest, MatMulFloatType) {
  // TODO: Unskip when fixed #41968513
  if (DefaultDmlExecutionProvider().get() != nullptr) {
    GTEST_SKIP() << "Skipping because of the following error: Assertion failed: m_bufferTensorDesc.TotalTensorSizeInBytes >= ComputeByteSizeFromDimensions(nonBroadcastDimensions, dataType)";
  }
  RunMatMulTest<float>(7, false, false);
  // Note. Xnnpack only supports matmul when Matrix B is constant
  RunMatMulTest<float>(7, false, true);
}

#if defined(USE_CUDA) || defined(USE_ROCM) || defined(USE_COREML) || defined(USE_XNNPACK)
TEST(MathOpTest, MatMulFloat16) {
#ifdef USE_CUDA
  int min_cuda_architecture = 530;
  if (!HasCudaEnvironment(min_cuda_architecture)) {
    LOGS_DEFAULT(WARNING) << "Hardware NOT support FP16";
    return;
  }
#endif
  // TODO: Unskip when fixed #41968513
  if (DefaultDmlExecutionProvider().get() != nullptr) {
    GTEST_SKIP() << "Skipping because of the following error: Assertion failed: m_bufferTensorDesc.TotalTensorSizeInBytes >= ComputeByteSizeFromDimensions(nonBroadcastDimensions, dataType)";
  }
  RunMatMulTest<MLFloat16>(14, false, false);
  // Note. Xnnpack only supports matmul when Matrix B is constant
  RunMatMulTest<MLFloat16>(14, false, true);
}
#endif

TEST(MathOpTest, MatMulDoubleType) {
  RunMatMulTest<double>(7);
}

TEST(MathOpTest, MatMulInt32Type) {
  // Webgpu does not support int32 matmul
  if (DefaultWebGpuExecutionProvider().get() != nullptr) {
    GTEST_SKIP() << "Skipping because of the following error: Webgpu does not support int32 matmul";
  }
  RunMatMulTest<int32_t>(9);
}

TEST(MathOpTest, MatMulUint32Type) {
  // Webgpu does not support uint32 matmul
  if (DefaultWebGpuExecutionProvider().get() != nullptr) {
    GTEST_SKIP() << "Skipping because of the following error: Webgpu does not support uint32 matmul";
  }
  RunMatMulTest<uint32_t>(9);
}

TEST(MathOpTest, MatMulInt64Type) {
  RunMatMulTest<int64_t>(9);
}

TEST(MathOpTest, MatMulUint64Type) {
  RunMatMulTest<uint64_t>(9);
}

template <typename T>
void RunMatMulZeroKTest() {
  // test with empty inputs and zero filled output
  constexpr const std::array<T, 0> empty_input{};
  const std::vector<T> expected_output(4 * 4, T{});
  OpTester test("MatMul", 13);

  test.AddInput<T>("A", {4, 0}, empty_input);
  test.AddInput<T>("B", {0, 4}, empty_input);
  test.AddOutput<T>("Y", {4, 4}, expected_output);

  // No special case is implemented.
  test.ConfigExcludeEps({kCoreMLExecutionProvider, kNnapiExecutionProvider,
                         kDmlExecutionProvider, kDnnlExecutionProvider, kQnnExecutionProvider,
                         kOpenVINOExecutionProvider, kWebGpuExecutionProvider})
      .Config(run_with_tunable_op)
      .RunWithConfig();
}

TEST(MathOpTest, MatMulZeroKFloatType) {
  if (DefaultWebGpuExecutionProvider().get() != nullptr) {
    GTEST_SKIP() << "Skipping because of the following error: Webgpu does not support zero-sized tensor";
  }
  RunMatMulZeroKTest<float>();
}

TEST(MathOpTest, MatMulZeroKInt32Type) {
  if (DefaultWebGpuExecutionProvider().get() != nullptr) {
    GTEST_SKIP() << "Skipping because of the following error: Webgpu does not support zero-sized tensor";
  }
  RunMatMulZeroKTest<int32_t>();
}

#if defined(USE_CUDA) || defined(USE_ROCM) || defined(USE_COREML) || defined(USE_XNNPACK)
TEST(MathOpTest, MatMul_Float16) {
#ifdef USE_CUDA
  int min_cuda_architecture = 530;
  if (!HasCudaEnvironment(min_cuda_architecture)) {
    LOGS_DEFAULT(WARNING) << "Hardware NOT support FP16";
    return;
  }
#endif
  std::vector<float> A{1.0f, 2.0f, 3.0f, 4.0f,
                       -1.0f, -2.0f, -3.0f, -4.0f};
  std::vector<float> B(12, 1.0f);
  std::vector<float> Y{10.0f, 10.0f, 10.0f,
                       -10.0f, -10.0f, -10.0f};

  std::vector<MLFloat16> f_A(8);
  std::vector<MLFloat16> f_B(12);
  std::vector<MLFloat16> f_Y(6);
  ConvertFloatToMLFloat16(A.data(), f_A.data(), 8);
  ConvertFloatToMLFloat16(B.data(), f_B.data(), 12);
  ConvertFloatToMLFloat16(Y.data(), f_Y.data(), 6);

  auto run_test = [&](bool B_is_constant) {
    // it needs Matrix B as constant to test XNNPack
    OpTester test("MatMul", 14);
    test.AddInput<MLFloat16>("A", {2, 4}, f_A);
    test.AddInput<MLFloat16>("B", {4, 3}, f_B, B_is_constant);
    test.AddOutput<MLFloat16>("Y", {2, 3}, f_Y);
    test.ConfigExcludeEps({kTensorrtExecutionProvider})  // TensorRT: fp16 is not supported
        .Config(run_with_tunable_op)
        .RunWithConfig();
  };
  run_test(true);
  run_test(false);
}
#endif

#if defined(USE_CUDA) || defined(USE_ROCM) || defined(USE_DNNL)
TEST(MathOpTest, MatMul_bfloat16) {
#ifdef USE_CUDA
  int min_cuda_architecture = 530;
  if (!HasCudaEnvironment(min_cuda_architecture)) {
    LOGS_DEFAULT(WARNING) << "Hardware NOT support BFP16";
    return;
  }
#endif
#ifdef USE_DNNL
  if (!DnnlHasBF16Support()) {
    LOGS_DEFAULT(WARNING) << "Hardware does NOT support BF16";
    return;
  }
#endif
  OpTester test("MatMul", 14);

  test.AddInput<BFloat16>("A", {2, 4}, MakeBFloat16({1.0f, 2.0f, 3.0f, 4.0f, -1.0f, -2.0f, -3.0f, -4.0f}));
  test.AddInput<BFloat16>("B", {4, 3}, MakeBFloat16({1.f, 1.f, 1.f, 1.f, 1.f, 1.f, 1.f, 1.f, 1.f, 1.f, 1.f, 1.f}));
  test.AddOutput<BFloat16>("Y", {2, 3}, MakeBFloat16({10.0f, 10.0f, 10.0f, -10.0f, -10.0f, -10.0f}));
  std::vector<std::unique_ptr<IExecutionProvider>> execution_providers;
  test.Config(run_with_tunable_op);
#ifdef USE_CUDA
  execution_providers.emplace_back(DefaultCudaExecutionProvider());
#elif USE_ROCM
  execution_providers.emplace_back(DefaultRocmExecutionProvider(/*test_tunable_op=*/true));
  test.ConfigEps(std::move(execution_providers))
      .RunWithConfig();

  execution_providers.clear();
  execution_providers.emplace_back(DefaultRocmExecutionProvider(/*test_tunable_op=*/false));
#elif USE_DNNL
  execution_providers.emplace_back(DefaultDnnlExecutionProvider());
#endif
  test.ConfigEps(std::move(execution_providers))
      .RunWithConfig();
}
#endif

#ifndef ENABLE_TRAINING
// Prepacking is disabled in full training build so no need to test the feature in a training build.
TEST(MathOpTest, MatMulSharedPrepackedWeights) {
  OpTester test("MatMul");

  std::vector<float> b_init_values(12, 1.0f);
  test.AddInput<float>("A", {2, 4},
                       {1.0f, 2.0f, 3.0f, 4.0f,
                        -1.0f, -2.0f, -3.0f, -4.0f});
  // B is to be an initializer for triggering pre-packing
  test.AddInput<float>("B", {4, 3}, b_init_values, true);

  test.AddOutput<float>("Y", {2, 3},
                        {10.0f, 10.0f, 10.0f,
                         -10.0f, -10.0f, -10.0f});

  OrtValue b;
  Tensor::InitOrtValue(DataTypeImpl::GetType<float>(), TensorShape({4, 3}),
                       b_init_values.data(), OrtMemoryInfo(CPU, OrtAllocatorType::OrtDeviceAllocator), b);

  SessionOptions so;
  // Set up B as a shared initializer to be shared between sessions
  ASSERT_EQ(so.AddInitializer("B", &b), Status::OK());

  // We want all sessions running using this OpTester to be able to share pre-packed weights if applicable
  test.EnableSharingOfPrePackedWeightsAcrossSessions();

  // Pre-packing is limited just to the CPU EP for now and we will only test the CPU EP
  // and we want to ensure that it is available in this build
  auto cpu_ep = []() -> std::vector<std::unique_ptr<IExecutionProvider>> {
    std::vector<std::unique_ptr<IExecutionProvider>> execution_providers;
    execution_providers.push_back(DefaultCpuExecutionProvider());
    return execution_providers;
  };

  size_t number_of_pre_packed_weights_counter_session_1 = 0;
  size_t number_of_shared_pre_packed_weights_counter = 0;

  // Session 1
  {
    test.Config(so)
        .Config(run_with_tunable_op)
        .ConfigEps(cpu_ep())
        .RunWithConfig(&number_of_pre_packed_weights_counter_session_1, &number_of_shared_pre_packed_weights_counter);
    // Assert that no pre-packed weights have been shared thus far
    ASSERT_EQ(number_of_shared_pre_packed_weights_counter, static_cast<size_t>(0));
  }

  auto number_of_elements_in_shared_prepacked_buffers_container =
      test.GetNumPrePackedWeightsShared();
  // Assert that the number of elements in the shared container
  // is the same as the number of weights that have been pre-packed
  ASSERT_EQ(number_of_pre_packed_weights_counter_session_1, number_of_elements_in_shared_prepacked_buffers_container);

  // On some platforms/architectures MLAS may choose to not do any pre-packing and the number of elements
  // that have been pre-packed will be zero in which case we do not continue with the testing
  // of "sharing" of pre-packed weights as there are no pre-packed weights to be shared at all.
  if (number_of_pre_packed_weights_counter_session_1 == 0)
    return;

  // Session 2
  {
    size_t number_of_pre_packed_weights_counter_session_2 = 0;
    test.Config(so)
        .Config(run_with_tunable_op)
        .ConfigEps(cpu_ep())
        .RunWithConfig(&number_of_pre_packed_weights_counter_session_2, &number_of_shared_pre_packed_weights_counter);

    // Assert that the same number of weights were pre-packed in both sessions
    ASSERT_EQ(number_of_pre_packed_weights_counter_session_1, number_of_pre_packed_weights_counter_session_2);

    // Assert that the number of pre-packed weights that were shared equals
    // the number of pre-packed weights in the second session
    ASSERT_EQ(number_of_pre_packed_weights_counter_session_2,
              static_cast<size_t>(number_of_shared_pre_packed_weights_counter));
  }
}

#endif

}  // namespace test
}  // namespace onnxruntime
