/*
 * Copyright (C) 2024 Igalia S.L. All rights reserved.
 * Copyright (C) 2025 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "UserScript.h"

#include <wtf/TZoneMallocInlines.h>
#include <wtf/text/MakeString.h>

namespace WebCore {

WTF_MAKE_TZONE_ALLOCATED_IMPL(UserScript);

static WTF::URL generateUserScriptUniqueURL()
{
    static uint64_t identifier;
    return { { }, makeString("user-script:"_s, ++identifier) };
}

UserScript::UserScript(String&& source, URL&& url, Vector<String>&& allowlist, Vector<String>&& blocklist, UserScriptInjectionTime injectionTime, UserContentInjectedFrames injectedFrames, UserContentMatchParentFrame matchParentFrame)
    : m_source(WTFMove(source))
    , m_url(url.isEmpty() ? generateUserScriptUniqueURL() : WTFMove(url))
    , m_allowlist(WTFMove(allowlist))
    , m_blocklist(WTFMove(blocklist))
    , m_injectionTime(injectionTime)
    , m_injectedFrames(injectedFrames)
    , m_matchParentFrame(matchParentFrame)
{
}

String UserScript::debugDescription() const
{
    String urlString = m_url.string();
    if (!urlString.isEmpty() && !urlString.startsWith("user-script:"_s))
        return urlString;

    StringView view { m_source };
    auto truncated = view.left(64);
    if (size_t pos = truncated.find('\n'); pos != notFound) {
        if (truncated.startsWith("//# sourceURL="_s))
            truncated = truncated.substring(14, pos);
        else
            truncated = truncated.substring(0, pos);
    }
    return truncated.toString();
}

} // namespace WebCore
