/*
 * ATRAC9 decoder
 * Copyright (c) 2018 Rostislav Pehlivanov <atomnuker@gmail.com>
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef AVCODEC_ATRAC9TAB_H
#define AVCODEC_ATRAC9TAB_H

#include <stdint.h>

#include "libavutil/channel_layout.h"

enum ATRAC9BlockType {
    ATRAC9_BLOCK_TYPE_SCE, /* Single channel */
    ATRAC9_BLOCK_TYPE_CPE, /* 2 coupled channels */
    ATRAC9_BLOCK_TYPE_LFE, /* Single LFE channel */
};

typedef struct ATRAC9BlockConfig {
    AVChannelLayout channel_layout;
    enum ATRAC9BlockType type[5];
    int plane_map[5][2];
    int count;
} ATRAC9BlockConfig;

static const ATRAC9BlockConfig at9_block_layout[] = {
    { /* Mono */
        .channel_layout  = AV_CHANNEL_LAYOUT_MONO,
        .type            = { ATRAC9_BLOCK_TYPE_SCE, },
        .count           = 1,
    },
    { /* Dual Mono */
        .channel_layout  = AV_CHANNEL_LAYOUT_STEREO,
        .type            = { ATRAC9_BLOCK_TYPE_SCE,
                             ATRAC9_BLOCK_TYPE_SCE, },
        .plane_map       = { { 0 }, { 1 }, },
        .count           = 2,
    },
    { /* Stereo */
        .channel_layout  = AV_CHANNEL_LAYOUT_STEREO,
        .type            = { ATRAC9_BLOCK_TYPE_CPE, },
        .plane_map       = { { 0, 1 }, },
        .count           = 1,
    },
    { /* 5.1 */
        .channel_layout  = AV_CHANNEL_LAYOUT_5POINT1,
        .type            = { ATRAC9_BLOCK_TYPE_CPE,
                             ATRAC9_BLOCK_TYPE_SCE,
                             ATRAC9_BLOCK_TYPE_LFE,
                             ATRAC9_BLOCK_TYPE_CPE, },
        .plane_map       = { { 0, 1 }, { 2 }, { 3 }, { 4, 5 }, },
        .count           = 4,
    },
    { /* 7.1 */
        .channel_layout  = AV_CHANNEL_LAYOUT_7POINT1,
        .type            = { ATRAC9_BLOCK_TYPE_CPE,
                             ATRAC9_BLOCK_TYPE_SCE,
                             ATRAC9_BLOCK_TYPE_LFE,
                             ATRAC9_BLOCK_TYPE_CPE,
                             ATRAC9_BLOCK_TYPE_CPE, },
        .plane_map       = { { 0, 1 }, { 2 }, { 3 }, { 4, 5 }, { 6, 7 }, },
        .count           = 5,
    },
    { /* Quad */
        .channel_layout  = AV_CHANNEL_LAYOUT_QUAD,
        .type            = { ATRAC9_BLOCK_TYPE_CPE,
                             ATRAC9_BLOCK_TYPE_CPE, },
        .plane_map       = { { 0, 1 }, { 2, 3 }, },
        .count           = 2,
    },
};

static const uint8_t at9_tab_sri_frame_log2[] = {
    6, 6, 7, 7, 7, 8, 8, 8, 6, 6, 7, 7, 7, 8, 8, 8,
};

static const uint8_t at9_tab_band_q_unit_map[] = {
    0, 4, 8, 10, 12, 13, 14, 15, 16, 18, 20, 21, 22, 23, 24, 25, 26, 28, 30,
};

static const uint8_t at9_q_unit_to_coeff_cnt[] = {
    2, 2, 2, 2, 2,  2,  2,  2,  4,  4,  4,  4,  8,  8,  8,
    8, 8, 8, 8, 8, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
};

static const int at9_q_unit_to_coeff_idx[] = {
    0, 2, 4, 6, 8, 10, 12, 14, 16, 20, 24, 28, 32, 40, 48, 56, 64,
    72, 80, 88, 96, 112, 128, 144, 160, 176, 192, 208, 224, 240, 256
};

static const uint8_t at9_q_unit_to_codebookidx[] = {
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2,
    2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
};

static const uint8_t at9_tab_sri_max_bands[] = {
    8, 8, 12, 12, 12, 18, 18, 18, 8, 8, 12, 12, 12, 16, 16, 16,
};

static const int at9_tab_samplerates[] = {
    11025, 12000, 16000, 22050, 24000, 32000, 44100, 48000, 44100, 48000,
    64000, 88200, 96000, 128000, 176400, 192000,
};

static const uint8_t at9_tab_band_ext_cnt[][6] = {
    { 0, 0, 0, 4, 4, 2 },
    { 0, 0, 0, 0, 0, 0 },
    { 0, 0, 0, 2, 2, 1 },
    { 0, 0, 0, 2, 2, 2 },
    { 1, 1, 1, 0, 0, 0 },
};

/* B unit, C unit, Band count */
static const uint8_t at9_tab_band_ext_group[][3] = {
    { 16, 21, 0 },
    { 18, 22, 1 },
    { 20, 22, 2 },
    { 21, 22, 3 },
    { 21, 22, 3 },
    { 23, 24, 4 },
    { 23, 24, 4 },
    { 24, 24, 5 },
};

static const uint8_t at9_tab_band_ext_lengths[][6][4] = {
    {
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 5, 4, 3, 3 },
        { 4, 4, 3, 4 },
        { 4, 5, 0, 0 },
    },
    {
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
    },
    {
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 6, 6, 0, 0 },
        { 6, 6, 0, 0 },
        { 6, 0, 0, 0 },
    },
    {
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 4, 4, 0, 0 },
        { 4, 4, 0, 0 },
        { 4, 4, 0, 0 },
    },
    {
        { 3, 0, 0, 0 },
        { 3, 0, 0, 0 },
        { 3, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
    },
};

static const float at9_band_ext_scales_m0[][5][32] = {
    {
        {
            0.000000e+0f, 1.988220e-1f, 2.514343e-1f, 2.960510e-1f,
            3.263550e-1f, 3.771362e-1f, 3.786926e-1f, 4.540405e-1f,
            4.877625e-1f, 5.262451e-1f, 5.447083e-1f, 5.737000e-1f,
            6.212158e-1f, 6.222839e-1f, 6.560974e-1f, 6.896667e-1f,
            7.555542e-1f, 7.677917e-1f, 7.918091e-1f, 7.971497e-1f,
            8.188171e-1f, 8.446045e-1f, 9.790649e-1f, 9.822083e-1f,
            9.846191e-1f, 9.859314e-1f, 9.863586e-1f, 9.863892e-1f,
            9.873352e-1f, 9.881287e-1f, 9.898682e-1f, 9.913330e-1f,
        }, {
            0.000000e+0f, 9.982910e-1f, 7.592773e-2f, 7.179565e-1f,
            9.851379e-1f, 5.340271e-1f, 9.013672e-1f, 6.349182e-1f,
            7.226257e-1f, 1.948547e-1f, 7.628174e-1f, 9.873657e-1f,
            8.112183e-1f, 2.715454e-1f, 9.734192e-1f, 1.443787e-1f,
            4.640198e-1f, 3.249207e-1f, 3.790894e-1f, 8.276367e-2f,
            5.954590e-1f, 2.864380e-1f, 9.806824e-1f, 7.929077e-1f,
            6.292114e-1f, 4.887085e-1f, 2.905273e-1f, 1.301880e-1f,
            3.140869e-1f, 5.482483e-1f, 4.210815e-1f, 1.182861e-1f,
        }, {
            0.000000e+0f, 3.155518e-2f, 8.581543e-2f, 1.364746e-1f,
            1.858826e-1f, 2.368469e-1f, 2.888184e-1f, 3.432617e-1f,
            4.012451e-1f, 4.623108e-1f, 5.271301e-1f, 5.954895e-1f,
            6.681213e-1f, 7.448425e-1f, 8.245239e-1f, 9.097290e-1f,
        }, {
            0.000000e+0f, 4.418945e-2f, 1.303711e-1f, 2.273560e-1f,
            3.395996e-1f, 4.735718e-1f, 6.267090e-1f, 8.003845e-1f,
        }, {
            0.000000e+0f, 2.804565e-2f, 9.683228e-2f, 1.849976e-1f,
            3.005981e-1f, 4.470520e-1f, 6.168518e-1f, 8.007813e-1f,
        },
    },
    {
        {
            0.000000e+0f, 2.708740e-1f, 3.479614e-1f, 3.578186e-1f,
            5.083618e-1f, 5.299072e-1f, 5.819092e-1f, 6.381836e-1f,
            7.276917e-1f, 7.595520e-1f, 7.878723e-1f, 9.707336e-1f,
            9.713135e-1f, 9.736023e-1f, 9.759827e-1f, 9.832458e-1f,
        }, {
            0.000000e+0f, 2.330627e-1f, 5.891418e-1f, 7.170410e-1f,
            2.036438e-1f, 1.613464e-1f, 6.668701e-1f, 9.481201e-1f,
            9.769897e-1f, 5.111694e-1f, 3.522644e-1f, 8.209534e-1f,
            2.933960e-1f, 9.757690e-1f, 5.289917e-1f, 4.372253e-1f,
        }, {
            0.000000e+0f, 4.360962e-2f, 1.056519e-1f, 1.590576e-1f,
            2.078857e-1f, 2.572937e-1f, 3.082581e-1f, 3.616028e-1f,
            4.191589e-1f, 4.792175e-1f, 5.438538e-1f, 6.125183e-1f,
            6.841125e-1f, 7.589417e-1f, 8.365173e-1f, 9.148254e-1f,
        }, {
            0.000000e+0f, 4.074097e-2f, 1.164551e-1f, 2.077026e-1f,
            3.184509e-1f, 4.532166e-1f, 6.124268e-1f, 7.932129e-1f,
        }, {
            0.000000e+0f, 8.880615e-3f, 2.932739e-2f, 5.593872e-2f,
            8.825684e-2f, 1.259155e-1f, 1.721497e-1f, 2.270813e-1f,
            2.901611e-1f, 3.579712e-1f, 4.334106e-1f, 5.147095e-1f,
            6.023254e-1f, 6.956177e-1f, 7.952881e-1f, 8.977356e-1f,
        },
    },
    {
        {
            0.000000e+0f, 7.379150e-2f, 1.806335e-1f, 2.687073e-1f,
            3.407898e-1f, 4.047546e-1f, 4.621887e-1f, 5.168762e-1f,
            5.703125e-1f, 6.237488e-1f, 6.763611e-1f, 7.288208e-1f,
            7.808533e-1f, 8.337708e-1f, 8.874512e-1f, 9.418030e-1f,
        }, {
            0.000000e+0f, 7.980347e-2f, 1.615295e-1f, 1.665649e-1f,
            1.822205e-1f, 2.185669e-1f, 2.292175e-1f, 2.456665e-1f,
            2.666321e-1f, 3.306580e-1f, 3.330688e-1f, 3.765259e-1f,
            4.085083e-1f, 4.400024e-1f, 4.407654e-1f, 4.817505e-1f,
            4.924011e-1f, 5.320740e-1f, 5.893860e-1f, 6.131287e-1f,
            6.212463e-1f, 6.278076e-1f, 6.308899e-1f, 7.660828e-1f,
            7.850647e-1f, 7.910461e-1f, 7.929382e-1f, 8.038330e-1f,
            9.834900e-1f, 9.846191e-1f, 9.852295e-1f, 9.862671e-1f,
        }, {
            0.000000e+0f, 6.084290e-1f, 3.672791e-1f, 3.151855e-1f,
            1.488953e-1f, 2.571716e-1f, 5.103455e-1f, 3.311157e-1f,
            5.426025e-2f, 4.254456e-1f, 7.998352e-1f, 7.873230e-1f,
            5.418701e-1f, 2.925110e-1f, 8.468628e-2f, 1.410522e-1f,
            9.819641e-1f, 9.609070e-1f, 3.530884e-2f, 9.729004e-2f,
            5.758362e-1f, 9.941711e-1f, 7.215576e-1f, 7.183228e-1f,
            2.028809e-1f, 9.588623e-2f, 2.032166e-1f, 1.338806e-1f,
            5.003357e-1f, 1.874390e-1f, 9.804993e-1f, 1.107788e-1f,
        },
    },
};

static const float at9_band_ext_scales_m2[] = {
    4.272461e-4f, 1.312256e-3f, 2.441406e-3f, 3.692627e-3f,
    4.913330e-3f, 6.134033e-3f, 7.507324e-3f, 8.972168e-3f,
    1.049805e-2f, 1.223755e-2f, 1.406860e-2f, 1.599121e-2f,
    1.800537e-2f, 2.026367e-2f, 2.264404e-2f, 2.517700e-2f,
    2.792358e-2f, 3.073120e-2f, 3.344727e-2f, 3.631592e-2f,
    3.952026e-2f, 4.275513e-2f, 4.608154e-2f, 4.968262e-2f,
    5.355835e-2f, 5.783081e-2f, 6.195068e-2f, 6.677246e-2f,
    7.196045e-2f, 7.745361e-2f, 8.319092e-2f, 8.993530e-2f,
    9.759521e-2f, 1.056213e-1f, 1.138916e-1f, 1.236267e-1f,
    1.348267e-1f, 1.470337e-1f, 1.603394e-1f, 1.755676e-1f,
    1.905823e-1f, 2.071228e-1f, 2.245178e-1f, 2.444153e-1f,
    2.658997e-1f, 2.897644e-1f, 3.146057e-1f, 3.450012e-1f,
    3.766174e-1f, 4.122620e-1f, 4.505615e-1f, 4.893799e-1f,
    5.305481e-1f, 5.731201e-1f, 6.157837e-1f, 6.580811e-1f,
    6.985168e-1f, 7.435303e-1f, 7.865906e-1f, 8.302612e-1f,
    8.718567e-1f, 9.125671e-1f, 9.575806e-1f, 9.996643e-1f,
};

static const float at9_band_ext_scales_m3[][2] = {
    { 3.491211e-1f, -2.913818e-1f, }, { 5.371094e-1f, -2.541504e-1f, },
    { 6.782227e-1f, -1.664429e-1f, }, { 7.910156e-1f, -1.476440e-1f, },
    { 9.057617e-1f, -1.342163e-1f, }, { 1.024902e+0f, -1.220703e-1f, },
    { 1.156250e+0f, -1.117554e-1f, }, { 1.290527e+0f, -1.026611e-1f, },
    { 1.458984e+0f, -9.436035e-2f, }, { 1.664551e+0f, -8.483887e-2f, },
    { 1.929688e+0f, -7.476807e-2f, }, { 2.278320e+0f, -6.304932e-2f, },
    { 2.831543e+0f, -4.492188e-2f, }, { 3.659180e+0f, -2.447510e-2f, },
    { 5.257813e+0f, +1.831055e-4f, }, { 8.373047e+0f, +4.174805e-2f, },
};

static const float at9_band_ext_scales_m4[] = {
    3.610229e-2f, 1.260681e-1f, 2.227478e-1f, 3.338318e-1f,
    4.662170e-1f, 6.221313e-1f, 7.989197e-1f, 9.939575e-1f,
};

static const float at9_quant_step_coarse[] = {
    2.0000000000000000e+0f, 6.6666666666666663e-1f, 2.8571428571428570e-1f,
    1.3333333333333333e-1f, 6.4516129032258063e-2f, 3.1746031746031744e-2f,
    1.5748031496062992e-2f, 7.8431372549019607e-3f, 3.9138943248532287e-3f,
    1.9550342130987292e-3f, 9.7703957010258913e-4f, 4.8840048840048840e-4f,
    2.4417043096081065e-4f, 1.2207776353537203e-4f, 6.1037018951994385e-5f,
    3.0518043793392844e-5f,
};

static const float at9_quant_step_fine[] = {
    3.0518043793392844e-05f, 1.0172681264464281e-05f, 4.3597205419132631e-06f,
    2.0345362528928561e-06f, 9.8445302559331759e-07f, 4.8441339354591809e-07f,
    2.4029955742829012e-07f, 1.1967860311134448e-07f, 5.9722199204291275e-08f,
    2.9831909866464167e-08f, 1.4908668194134265e-08f, 7.4525137468602791e-09f,
    3.7258019525568114e-09f, 1.8627872668859698e-09f, 9.3136520869755679e-10f,
    4.6567549848772173e-10f,
};

static const float at9_scalefactor_c[] = {
    3.0517578125e-5f, 6.1035156250e-5f, 1.2207031250e-4f, 2.4414062500e-4f,
    4.8828125000e-4f, 9.7656250000e-4f, 1.9531250000e-3f, 3.9062500000e-3f,
    7.8125000000e-3f, 1.5625000000e-2f, 3.1250000000e-2f, 6.2500000000e-2f,
    1.2500000000e-1f, 2.5000000000e-1f, 5.0000000000e-1f, 1.0000000000e+0f,
    2.0000000000e+0f, 4.0000000000e+0f, 8.0000000000e+0f, 1.6000000000e+1f,
    3.2000000000e+1f, 6.4000000000e+1f, 1.2800000000e+2f, 2.5600000000e+2f,
    5.1200000000e+2f, 1.0240000000e+3f, 2.0480000000e+3f, 4.0960000000e+3f,
    8.1920000000e+3f, 1.6384000000e+4f, 3.2768000000e+4f, 6.5536000000e+4f,
};

static const uint8_t at9_tab_sf_weights[][32] = {
    {
        0,  0,  0,  1,  1,  2,  2,  2,  2,  2,  2,  3,  2,  3,  3,  4,  4,  4,
        4,  4,  4,  5,  5,  6,  6,  7,  7,  8, 10, 12, 12, 12,
    },
    {
        3,  2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  0,  1,  0,  1,
        1,  1,  1,  1,  1,  2,  3,  3,  4,  5,  7, 10, 10, 10,
    },
    {
        0,  2,  4,  5,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
        6,  6,  6,  6,  6,  7,  7,  7,  7,  8,  9, 12, 12, 12,
    },
    {
        0,  1,  1,  2,  2,  2,  3,  3,  3,  3,  3,  4,  4,  4,  5,  5,  5,  6,
        6,  6,  6,  7,  8,  8, 10, 11, 11, 12, 13, 13, 13, 13,
    },
    {
        0,  2,  2,  3,  3,  4,  4,  5,  4,  5,  5,  5,  5,  6,  7,  8,  8,  8,
        8,  9,  9,  9, 10, 10, 11, 12, 12, 13, 13, 14, 14, 14,
    },
    {
        1,  1,  0,  0,  0,  0,  1,  0,  0,  1,  1,  1,  1,  1,  2,  2,  2,  2,
        2,  3,  3,  3,  4,  4,  5,  6,  7,  7,  9, 11, 11, 11,
    },
    {
        0,  5,  8, 10, 11, 11, 12, 12, 12, 13, 13, 13, 13, 13, 13, 13, 13, 13,
        13, 13, 13, 13, 13, 13, 12, 12, 12, 12, 13, 15, 15, 15,
    },
    {
        0,  2,  3,  4,  5,  6,  6,  7,  7,  8,  8,  8,  9,  9, 10, 10, 10, 11,
        11, 11, 11, 11, 11, 12, 12, 12, 12, 13, 13, 15, 15, 15,
    },
};

static const uint8_t at9_tab_b_dist[] = {
         1,  1,  1,  1,  2,  2,  2,  2,  3,  3,  3,  4,  4,  5,  5,  6,  7,  8,
         9, 10, 11, 12, 13, 15,  6, 18, 19, 20, 21, 22, 23, 24, 25, 26, 26, 27,
        27, 28, 28, 28, 29, 29, 29, 29, 30, 30, 30, 30,
};

static const uint8_t at9_sfb_a_tab[][2] = {
    /* Unsigned scalefactor table 1 - 2 entries */
    {  0, 1 }, {  1, 1 },
    /* Unsigned scalefactor table 2 - 4 entries */
    {  0, 1 }, {  3, 2 }, {  1, 3 }, {  2, 3 },
    /* Unsigned scalefactor table 3 - 8 entries */
    {  0, 2 }, {  1, 2 }, {  7, 2 }, {  6, 3 }, {  2, 4 }, {  5, 5 }, {  3, 6 },
    {  4, 6 },
    /* Unsigned scalefactor table 4 - 16 entries */
    {  2, 4 }, { 14, 4 }, { 11, 8 }, { 10, 8 }, {  9, 8 }, {  6, 8 }, {  7, 8 },
    {  8, 8 }, {  5, 7 }, { 13, 5 }, {  3, 5 }, { 12, 6 }, {  4, 6 }, {  0, 2 },
    {  1, 2 }, { 15, 2 },
    /* Unsigned scalefactor table 5 - 32 entries */
    { 31, 3 }, {  1, 3 }, { 26, 7 }, {  7, 7 }, { 20, 8 }, { 21, 8 }, { 23, 8 },
    { 22, 8 }, { 29, 5 }, { 30, 4 }, {  4, 5 }, { 11, 8 }, { 24, 8 }, {  9, 8 },
    { 10, 8 }, {  6, 6 }, { 25, 7 }, { 19, 8 }, { 12, 8 }, { 27, 6 }, { 18, 8 },
    { 13, 8 }, { 16, 8 }, { 17, 8 }, { 14, 8 }, { 15, 8 }, {  8, 7 }, {  0, 2 },
    {  5, 5 }, { 28, 5 }, {  3, 4 }, {  2, 3 },
    /* Unsigned scalefactor table 6 - 64 entries */
    {  0, 3 }, {  1, 3 }, {  2, 4 }, {  3, 4 }, { 61, 4 }, { 62, 4 }, { 63, 4 },
    {  4, 5 }, {  5, 5 }, { 58, 5 }, { 59, 5 }, { 60, 5 }, {  6, 6 }, {  7, 6 },
    {  8, 6 }, { 56, 6 }, { 57, 6 }, {  9, 7 }, { 10, 7 }, { 53, 7 }, { 54, 7 },
    { 55, 7 }, { 11, 8 }, { 12, 8 }, { 13, 8 }, { 14, 8 }, { 15, 8 }, { 16, 8 },
    { 17, 8 }, { 18, 8 }, { 19, 8 }, { 20, 8 }, { 21, 8 }, { 22, 8 }, { 23, 8 },
    { 24, 8 }, { 25, 8 }, { 26, 8 }, { 27, 8 }, { 28, 8 }, { 29, 8 }, { 30, 8 },
    { 31, 8 }, { 32, 8 }, { 33, 8 }, { 34, 8 }, { 35, 8 }, { 36, 8 }, { 37, 8 },
    { 38, 8 }, { 39, 8 }, { 40, 8 }, { 41, 8 }, { 42, 8 }, { 43, 8 }, { 44, 8 },
    { 45, 8 }, { 46, 8 }, { 47, 8 }, { 48, 8 }, { 49, 8 }, { 50, 8 }, { 51, 8 },
    { 52, 8 },
};

static const uint8_t at9_sfb_b_tab[][2] = {
    /* Signed scalefactor table 2 - 3 entries */
    { 16, 1 }, { 15, 2 }, { 17, 2 },
    /* Signed scalefactor table 3 - 7 entries */
    { 17, 3 }, { 18, 5 }, { 13, 6 }, { 19, 6 }, { 14, 4 }, { 15, 2 }, { 16, 1 },
    /* Signed scalefactor table 4 - 15 entries */
    { 15, 3 }, { 17, 3 }, { 18, 4 }, { 13, 6 }, { 12, 6 }, { 21, 7 }, { 22, 8 },
    { 10, 8 }, {  9, 8 }, { 23, 8 }, { 11, 7 }, { 14, 4 }, { 19, 5 }, { 20, 5 },
    { 16, 1 },
    /* Signed scalefactor table 5 - 32 entries */
    { 16, 3 }, { 21, 4 }, { 22, 4 }, { 20, 4 }, { 23, 4 }, { 30, 8 }, {  1, 8 },
    {  0, 8 }, {  2, 8 }, {  3, 8 }, { 31, 8 }, { 27, 7 }, { 14, 6 }, { 28, 7 },
    { 13, 7 }, { 18, 4 }, { 25, 5 }, { 12, 7 }, { 11, 7 }, { 26, 6 }, { 24, 4 },
    { 17, 3 }, { 19, 4 }, {  9, 7 }, { 10, 7 }, {  8, 7 }, {  7, 7 }, {  6, 7 },
    {  5, 7 }, { 29, 7 }, {  4, 7 }, { 15, 3 },
};

typedef struct HuffmanCodebook {
    const int size;
    const int value_cnt;
    const int value_cnt_pow;
    const int value_bits;
} HuffmanCodebook;

static const HuffmanCodebook at9_huffman_sf_unsigned[] = {
    { 0 },
    {  2,  1,  0,  1 },
    {  4,  1,  0,  2 },
    {  8,  1,  0,  3 },
    { 16,  1,  0,  4 },
    { 32,  1,  0,  5 },
    { 64,  1,  0,  6 },
};

static const HuffmanCodebook at9_huffman_sf_signed[] = {
    { 0 },
    { 0 },
    {  3,  1,  0,  2 },
    {  7,  1,  0,  3 },
    { 15,  1,  0,  4 },
    { 32,  1,  0,  5 },
};

static const uint8_t at9_coeffs_tab[][2] = {
    /* Coefficient table A.2.1 - 8 entries */
    {   1,  3 }, {   3,  3 }, {  12,  3 }, {   4,  3 }, {   7,  3 },
    {  13,  3 }, {  15,  3 }, {   5,  3 },
    /* Coefficient table A.2.2 - 80 entries */
    {  17,  6 }, {  51,  6 }, {  81,  7 }, { 243,  7 }, {  69,  7 },
    { 207,  7 }, { 205,  7 }, {  71,  7 }, { 209,  7 }, { 115,  7 },
    { 245,  7 }, {  95,  7 }, { 125,  7 }, { 215,  7 }, {   1,  4 },
    {   3,  4 }, { 116,  7 }, { 220,  7 }, {  84,  7 }, { 252,  7 },
    {  29,  7 }, {  55,  7 }, {  21,  7 }, {  63,  7 }, {  65,  6 },
    { 195,  6 }, { 193,  6 }, {  67,  6 }, {  20,  6 }, {  60,  6 },
    {  52,  6 }, {  28,  6 }, {  16,  5 }, {  48,  5 }, {   4,  5 },
    {  12,  5 }, { 221,  8 }, { 119,  8 }, { 117,  8 }, { 223,  8 },
    { 213,  8 }, { 127,  8 }, { 253,  8 }, {  87,  8 }, {  80,  6 },
    { 240,  6 }, {  93,  8 }, { 247,  8 }, {  85,  8 }, { 255,  8 },
    { 113,  7 }, { 211,  7 }, { 208,  6 }, { 112,  6 }, {  64,  5 },
    { 192,  5 }, { 212,  7 }, { 124,  7 }, { 241,  7 }, {  83,  7 },
    {  77,  7 }, { 199,  7 }, { 244,  7 }, {  92,  7 }, { 196,  6 },
    {  76,  6 }, {  53,  7 }, {  31,  7 }, { 197,  7 }, {  79,  7 },
    {  49,  6 }, {  19,  6 }, {  61,  7 }, {  23,  7 }, {  68,  7 },
    { 204,  7 }, {   5,  6 }, {  15,  6 }, {  13,  6 }, {   7,  6 },
    /* Coefficient table A.2.3 - 81 entries */
    {  65,  6 }, { 195,  6 }, { 193,  6 }, {  67,  6 }, { 197,  7 },
    {  79,  7 }, {  77,  7 }, { 199,  7 }, { 241,  7 }, {  83,  7 },
    { 244,  7 }, {  92,  7 }, {   1,  4 }, {   3,  4 }, {  20,  6 },
    {  60,  6 }, { 212,  7 }, { 124,  7 }, {  61,  7 }, {  23,  7 },
    {  52,  6 }, {  28,  6 }, {  53,  7 }, {  31,  7 }, { 245,  8 },
    {  95,  8 }, { 125,  8 }, { 215,  8 }, {  80,  6 }, { 240,  6 },
    { 208,  6 }, { 112,  6 }, { 205,  8 }, {  71,  8 }, {  81,  8 },
    { 243,  8 }, {  69,  8 }, { 207,  8 }, { 209,  8 }, { 115,  8 },
    {   5,  6 }, {  15,  6 }, {  13,  6 }, {   7,  6 }, { 196,  6 },
    {  76,  6 }, { 221,  9 }, { 119,  9 }, { 253,  9 }, {  87,  9 },
    { 116,  8 }, { 220,  8 }, {  68,  7 }, { 204,  7 }, {  49,  6 },
    {  19,  6 }, { 117,  9 }, { 223,  9 }, { 213,  9 }, { 127,  9 },
    {  84,  8 }, { 252,  8 }, {  29,  8 }, {  55,  8 }, {  21,  8 },
    {  63,  8 }, {  17,  7 }, {  51,  7 }, {  93,  9 }, { 247,  9 },
    {  85,  9 }, { 255,  9 }, { 113,  8 }, { 211,  8 }, {   4,  5 },
    {  12,  5 }, {   0,  3 }, {  16,  5 }, {  48,  5 }, {  64,  5 },
    { 192,  5 },
    /* Coefficient table A.2.4 - 81 entries */
    {  20,  6 }, {  60,  6 }, {  52,  6 }, {  28,  6 }, {  80,  6 },
    { 240,  6 }, { 208,  6 }, { 112,  6 }, {   1,  4 }, {   3,  4 },
    {   5,  6 }, {  15,  6 }, {  13,  6 }, {   7,  6 }, {  84,  8 },
    { 252,  8 }, {  29,  8 }, {  55,  8 }, {  68,  7 }, { 204,  7 },
    { 196,  6 }, {  76,  6 }, {  17,  7 }, {  51,  7 }, {  21,  8 },
    {  63,  8 }, { 113,  8 }, { 211,  8 }, {  49,  6 }, {  19,  6 },
    { 245,  9 }, {  95,  9 }, { 125,  9 }, { 215,  9 }, { 244,  8 },
    {  92,  8 }, { 212,  8 }, { 124,  8 }, { 241,  8 }, {  83,  8 },
    {  77,  8 }, { 199,  8 }, { 197,  8 }, {  79,  8 }, {  53,  8 },
    {  31,  8 }, { 253, 10 }, {  87, 10 }, { 213, 10 }, { 127, 10 },
    {  81,  9 }, { 243,  9 }, {   0,  2 }, {  61,  8 }, {  23,  8 },
    { 117, 10 }, { 223, 10 }, { 221, 10 }, { 119, 10 }, { 209,  9 },
    { 115,  9 }, {  65,  7 }, { 195,  7 }, { 193,  7 }, {  67,  7 },
    {  69,  9 }, { 207,  9 }, { 205,  9 }, {  71,  9 }, {  93, 10 },
    { 247, 10 }, {  85, 10 }, { 255, 10 }, { 116,  9 }, { 220,  9 },
    {  16,  5 }, {  48,  5 }, {  64,  5 }, { 192,  5 }, {   4,  5 },
    {  12,  5 },
    /* Coefficient table A.3.1 - 40 entries */
    {  50,  5 }, {  22,  5 }, {  11,  5 }, {  61,  5 }, {   2,  4 },
    {   6,  4 }, {  59,  5 }, {  13,  5 }, {  17,  5 }, {  55,  5 },
    {  49,  5 }, {  23,  5 }, {  24,  5 }, {  40,  5 }, {  26,  6 },
    {  46,  6 }, {  42,  6 }, {  30,  6 }, {  10,  5 }, {  62,  5 },
    {  58,  5 }, {  14,  5 }, {  19,  6 }, {  53,  6 }, {  51,  6 },
    {  21,  6 }, {   3,  5 }, {   5,  5 }, {  16,  5 }, {  48,  5 },
    {  25,  6 }, {  47,  6 }, {  41,  6 }, {  31,  6 }, {  27,  7 },
    {  45,  7 }, {  43,  7 }, {  29,  7 }, {  18,  6 }, {  54,  6 },
    /* Coefficient table A.3.2 - 49 entries */
    {  57,  5 }, {  15,  5 }, {  17,  5 }, {  55,  5 }, {  49,  5 },
    {  23,  5 }, {  58,  5 }, {  14,  5 }, {  10,  5 }, {  62,  5 },
    {  43,  7 }, {  29,  7 }, {  27,  7 }, {  45,  7 }, {  25,  6 },
    {  47,  6 }, {  41,  6 }, {  31,  6 }, {  59,  6 }, {  13,  6 },
    {  11,  6 }, {  61,  6 }, {  50,  6 }, {  22,  6 }, {  16,  5 },
    {  48,  5 }, {   8,  5 }, {  56,  5 }, {  18,  6 }, {  54,  6 },
    {  24,  6 }, {  40,  6 }, {   2,  5 }, {   6,  5 }, {   1,  5 },
    {   7,  5 }, {   0,  4 }, {  42,  7 }, {  30,  7 }, {  26,  7 },
    {  46,  7 }, {   3,  6 }, {   5,  6 }, {  19,  7 }, {  53,  7 },
    {  51,  7 }, {  21,  7 }, {   9,  6 }, {  63,  6 },
    /* Coefficient table A.3.3 - 49 entries */
    {  51,  7 }, {  21,  7 }, {  19,  7 }, {  53,  7 }, {  24,  6 },
    {  40,  6 }, {  50,  6 }, {  22,  6 }, {  18,  6 }, {  54,  6 },
    {   3,  6 }, {   5,  6 }, {  43,  8 }, {  29,  8 }, {  27,  8 },
    {  45,  8 }, {  25,  7 }, {  47,  7 }, {  16,  5 }, {  48,  5 },
    {   8,  4 }, {  56,  4 }, {   1,  4 }, {   7,  4 }, {   2,  5 },
    {   6,  5 }, {   9,  5 }, {  63,  5 }, {   0,  3 }, {  57,  5 },
    {  15,  5 }, {  41,  7 }, {  31,  7 }, {  11,  7 }, {  61,  7 },
    {  59,  7 }, {  13,  7 }, {  26,  8 }, {  46,  8 }, {  42,  8 },
    {  30,  8 }, {  49,  6 }, {  23,  6 }, {  17,  6 }, {  55,  6 },
    {  58,  6 }, {  14,  6 }, {  10,  6 }, {  62,  6 },
    /* Coefficient table A.3.4 - 49 entries */
    {   0,  2 }, {  16,  5 }, {  48,  5 }, {   2,  5 }, {   6,  5 },
    {  26,  8 }, {  46,  8 }, {  51,  8 }, {  21,  8 }, {  24,  7 },
    {  40,  7 }, {   3,  7 }, {   5,  7 }, {  19,  8 }, {  53,  8 },
    {  25,  8 }, {  47,  8 }, {  50,  7 }, {  22,  7 }, {  18,  7 },
    {  54,  7 }, {  17,  6 }, {  55,  6 }, {   8,  4 }, {  56,  4 },
    {   1,  4 }, {   7,  4 }, {  49,  6 }, {  23,  6 }, {  10,  6 },
    {  62,  6 }, {  58,  6 }, {  14,  6 }, {  41,  8 }, {  31,  8 },
    {  11,  8 }, {  61,  8 }, {  59,  8 }, {  13,  8 }, {  43, 10 },
    {  29, 10 }, {  27, 10 }, {  45, 10 }, {  42,  9 }, {  30,  9 },
    {   9,  5 }, {  63,  5 }, {  57,  5 }, {  15,  5 },
    /* Coefficient table A.4.1 - 176 entries */
    {  83,  7 }, { 189,  7 }, {  68,  7 }, { 204,  7 }, {  23,  7 },
    { 249,  7 }, { 166,  8 }, { 106,  8 }, { 183,  8 }, {  89,  8 },
    {  52,  7 }, { 220,  7 }, { 161,  7 }, { 111,  7 }, { 179,  7 },
    {  93,  7 }, { 196,  7 }, {  76,  7 }, {  38,  7 }, { 234,  7 },
    { 116,  8 }, { 156,  8 }, { 148,  8 }, { 124,  8 }, {  97,  7 },
    { 175,  7 }, { 247,  7 }, {  25,  7 }, {  53,  7 }, { 219,  7 },
    { 212,  7 }, {  60,  7 }, {   5,  6 }, {  11,  6 }, { 101,  8 },
    { 171,  8 }, { 165,  8 }, { 107,  8 }, { 230,  7 }, {  42,  7 },
    { 244,  6 }, {  28,  6 }, {   7,  7 }, {   9,  7 }, { 213,  7 },
    {  59,  7 }, {  86,  8 }, { 186,  8 }, { 115,  8 }, { 157,  8 },
    {  71,  8 }, { 201,  8 }, { 182,  8 }, {  90,  8 }, {   4,  6 },
    {  12,  6 }, { 147,  8 }, { 125,  8 }, { 199,  8 }, {  73,  8 },
    {  82,  7 }, { 190,  7 }, {  66,  7 }, { 206,  7 }, { 178,  7 },
    {  94,  7 }, { 194,  7 }, {  78,  7 }, {  22,  7 }, { 250,  7 },
    {  96,  7 }, { 160,  7 }, { 100,  8 }, { 172,  8 }, { 164,  8 },
    { 108,  8 }, { 246,  7 }, {  26,  7 }, {  85,  8 }, { 187,  8 },
    { 114,  8 }, { 158,  8 }, {  37,  7 }, { 235,  7 }, {  36,  7 },
    { 236,  7 }, { 146,  8 }, { 126,  8 }, {  70,  8 }, { 202,  8 },
    { 215,  8 }, {  57,  8 }, {  55,  8 }, { 217,  8 }, { 181,  8 },
    {  91,  8 }, { 198,  8 }, {  74,  8 }, { 228,  7 }, {  44,  7 },
    { 229,  7 }, {  43,  7 }, {  81,  7 }, { 191,  7 }, { 177,  7 },
    {  95,  7 }, { 119,  9 }, { 153,  9 }, { 151,  9 }, { 121,  9 },
    {  99,  8 }, { 173,  8 }, {  65,  7 }, { 207,  7 }, {   6,  7 },
    {  10,  7 }, { 193,  7 }, {  79,  7 }, { 113,  8 }, { 159,  8 },
    { 163,  8 }, { 109,  8 }, { 145,  8 }, { 127,  8 }, { 118,  9 },
    { 154,  9 }, { 103,  9 }, { 169,  9 }, {  84,  8 }, { 188,  8 },
    {  39,  8 }, { 233,  8 }, { 167,  9 }, { 105,  9 }, { 150,  9 },
    { 122,  9 }, {  69,  8 }, { 203,  8 }, { 180,  8 }, {  92,  8 },
    {  54,  8 }, { 218,  8 }, { 197,  8 }, {  75,  8 }, {  98,  8 },
    { 174,  8 }, {  21,  7 }, { 251,  7 }, { 231,  8 }, {  41,  8 },
    { 162,  8 }, { 110,  8 }, { 214,  8 }, {  58,  8 }, { 117,  9 },
    { 155,  9 }, { 149,  9 }, { 123,  9 }, {  80,  7 }, { 176,  7 },
    {  20,  7 }, { 252,  7 }, { 102,  9 }, { 170,  9 }, {  87,  9 },
    { 185,  9 }, { 112,  8 }, { 144,  8 }, { 245,  7 }, {  27,  7 },
    {  64,  7 }, { 192,  7 }, {  67,  8 }, { 205,  8 }, { 195,  8 },
    {  77,  8 },
    /* Coefficient table A.4.2 - 225 entries */
    {   3,  7 }, {  13,  7 }, { 113,  8 }, { 159,  8 }, { 163,  8 },
    { 109,  8 }, {  16,  6 }, { 240,  6 }, {  81,  7 }, { 191,  7 },
    { 177,  7 }, {  95,  7 }, {   0,  5 }, { 112,  8 }, { 144,  8 },
    {  99,  8 }, { 173,  8 }, { 149,  9 }, { 123,  9 }, { 166,  9 },
    { 106,  9 }, {  54,  8 }, { 218,  8 }, {  66,  7 }, { 206,  7 },
    {  23,  8 }, { 249,  8 }, { 214,  8 }, {  58,  8 }, { 194,  7 },
    {  78,  7 }, { 117,  9 }, { 155,  9 }, {  87,  9 }, { 185,  9 },
    { 247,  8 }, {  25,  8 }, {  21,  7 }, { 251,  7 }, { 211,  8 },
    {  61,  8 }, {  51,  8 }, { 221,  8 }, {  36,  7 }, { 236,  7 },
    { 228,  7 }, {  44,  7 }, { 183,  9 }, {  89,  9 }, { 102,  9 },
    { 170,  9 }, { 180,  8 }, {  92,  8 }, {  80,  7 }, { 176,  7 },
    { 245,  7 }, {  27,  7 }, {  69,  8 }, { 203,  8 }, {  84,  8 },
    { 188,  8 }, { 225,  7 }, {  47,  7 }, {  33,  7 }, { 239,  7 },
    {   1,  6 }, {  15,  6 }, { 197,  8 }, {  75,  8 }, {  98,  8 },
    { 174,  8 }, {  18,  7 }, { 254,  7 }, { 162,  8 }, { 110,  8 },
    { 116,  9 }, { 156,  9 }, { 148,  9 }, { 124,  9 }, { 242,  7 },
    {  30,  7 }, {   5,  7 }, {  11,  7 }, {   7,  8 }, {   9,  8 },
    { 230,  8 }, {  42,  8 }, {  65,  7 }, { 207,  7 }, {  71,  9 },
    { 201,  9 }, { 199,  9 }, {  73,  9 }, {  38,  8 }, { 234,  8 },
    {  32,  7 }, { 224,  7 }, { 193,  7 }, {  79,  7 }, { 165,  9 },
    { 107,  9 }, { 182,  9 }, {  90,  9 }, {  35,  8 }, { 237,  8 },
    { 244,  7 }, {  28,  7 }, {  20,  7 }, { 252,  7 }, {  50,  8 },
    { 222,  8 }, {  97,  8 }, { 175,  8 }, { 101,  9 }, { 171,  9 },
    {  86,  9 }, { 186,  9 }, {  83,  8 }, { 189,  8 }, { 161,  8 },
    { 111,  8 }, { 179,  8 }, {  93,  8 }, {  64,  7 }, { 192,  7 },
    { 227,  8 }, {  45,  8 }, { 210,  8 }, {  62,  8 }, {   2,  7 },
    {  14,  7 }, { 213,  8 }, {  59,  8 }, {  53,  8 }, { 219,  8 },
    { 196,  8 }, {  76,  8 }, { 115,  9 }, { 157,  9 }, { 151, 10 },
    { 121, 10 }, { 119, 10 }, { 153, 10 }, {  68,  8 }, { 204,  8 },
    {  22,  8 }, { 250,  8 }, {  96,  8 }, { 160,  8 }, { 147,  9 },
    { 125,  9 }, {  55,  9 }, { 217,  9 }, { 246,  8 }, {  26,  8 },
    { 215,  9 }, {  57,  9 }, { 164,  9 }, { 108,  9 }, {  17,  7 },
    { 255,  7 }, {   4,  7 }, {  12,  7 }, { 241,  7 }, {  31,  7 },
    { 114,  9 }, { 158,  9 }, { 100,  9 }, { 172,  9 }, {   6,  8 },
    {  10,  8 }, { 209,  8 }, {  63,  8 }, {  19,  8 }, { 253,  8 },
    {  49,  8 }, { 223,  8 }, { 195,  8 }, {  77,  8 }, { 212,  8 },
    {  60,  8 }, { 198,  9 }, {  74,  9 }, {  70,  9 }, { 202,  9 },
    {  82,  8 }, { 190,  8 }, { 243,  8 }, {  29,  8 }, { 146,  9 },
    { 126,  9 }, { 118, 10 }, { 154, 10 }, { 167, 10 }, { 105, 10 },
    { 178,  8 }, {  94,  8 }, {  67,  8 }, { 205,  8 }, {  85,  9 },
    { 187,  9 }, { 181,  9 }, {  91,  9 }, {  52,  8 }, { 220,  8 },
    { 150, 10 }, { 122, 10 }, { 103, 10 }, { 169, 10 }, { 231,  9 },
    {  41,  9 }, { 229,  8 }, {  43,  8 }, {  48,  8 }, { 208,  8 },
    {  37,  8 }, { 235,  8 }, {  34,  8 }, { 238,  8 }, { 226,  8 },
    {  46,  8 }, {  39,  9 }, { 233,  9 }, { 145,  9 }, { 127,  9 },
    /* Coefficient table A.4.3 - 225 entries */
    { 246,  8 }, {  26,  8 }, {  22,  8 }, { 250,  8 }, { 210,  7 },
    {  62,  7 }, {  50,  7 }, { 222,  7 }, {  53,  8 }, { 219,  8 },
    { 231,  9 }, {  41,  9 }, {  39,  9 }, { 233,  9 }, {  16,  5 },
    { 240,  5 }, {   0,  4 }, { 227,  7 }, {  45,  7 }, {  35,  7 },
    { 237,  7 }, { 196,  8 }, {  76,  8 }, { 151, 10 }, { 121, 10 },
    { 119, 10 }, { 153, 10 }, { 150, 10 }, { 122, 10 }, { 167, 10 },
    { 105, 10 }, { 118, 10 }, { 154, 10 }, { 103, 10 }, { 169, 10 },
    { 117, 10 }, { 155, 10 }, { 149, 10 }, { 123, 10 }, {  96,  8 },
    { 160,  8 }, { 166, 10 }, { 106, 10 }, { 183, 10 }, {  89, 10 },
    {  87, 10 }, { 185, 10 }, { 102, 10 }, { 170, 10 }, {  71, 10 },
    { 201, 10 }, { 116, 10 }, { 156, 10 }, { 148, 10 }, { 124, 10 },
    { 199, 10 }, {  73, 10 }, { 182, 10 }, {  90, 10 }, { 165, 10 },
    { 107, 10 }, { 181,  9 }, {  91,  9 }, {  68,  8 }, { 204,  8 },
    {  85,  9 }, { 187,  9 }, { 113,  9 }, { 159,  9 }, {   6,  8 },
    {  10,  8 }, {  65,  7 }, { 207,  7 }, {  32,  6 }, { 224,  6 },
    {   2,  6 }, {  14,  6 }, { 193,  7 }, {  79,  7 }, { 145,  9 },
    { 127,  9 }, { 163,  9 }, { 109,  9 }, {  99,  9 }, { 173,  9 },
    { 247,  9 }, {  25,  9 }, {  20,  7 }, { 252,  7 }, { 244,  7 },
    {  28,  7 }, { 178,  8 }, {  94,  8 }, {  82,  8 }, { 190,  8 },
    {  23,  9 }, { 249,  9 }, {  54,  9 }, { 218,  9 }, { 214,  9 },
    {  58,  9 }, { 112,  9 }, { 144,  9 }, {  49,  7 }, { 223,  7 },
    { 209,  7 }, {  63,  7 }, {  37,  8 }, { 235,  8 }, { 229,  8 },
    {  43,  8 }, {  19,  7 }, { 253,  7 }, { 243,  7 }, {  29,  7 },
    {  64,  7 }, { 192,  7 }, { 195,  8 }, {  77,  8 }, {  67,  8 },
    { 205,  8 }, { 212,  8 }, {  60,  8 }, {  52,  8 }, { 220,  8 },
    {   7,  9 }, {   9,  9 }, { 180,  9 }, {  92,  9 }, { 197,  9 },
    {  75,  9 }, {  84,  9 }, { 188,  9 }, {   4,  7 }, {  12,  7 },
    {  48,  7 }, { 208,  7 }, {  34,  7 }, { 238,  7 }, { 226,  7 },
    {  46,  7 }, { 101, 10 }, { 171, 10 }, {  86, 10 }, { 186, 10 },
    {  69,  9 }, { 203,  9 }, { 211,  8 }, {  61,  8 }, {  17,  6 },
    { 255,  6 }, { 241,  6 }, {  31,  6 }, {   3,  7 }, {  13,  7 },
    {  51,  8 }, { 221,  8 }, {  98,  9 }, { 174,  9 }, { 162,  9 },
    { 110,  9 }, { 177,  8 }, {  95,  8 }, {  81,  8 }, { 191,  8 },
    {  38,  9 }, { 234,  9 }, { 230,  9 }, {  42,  9 }, {  21,  8 },
    { 251,  8 }, { 245,  8 }, {  27,  8 }, { 115, 10 }, { 157, 10 },
    {  55, 10 }, { 217, 10 }, { 147, 10 }, { 125, 10 }, { 215, 10 },
    {  57, 10 }, {  80,  8 }, { 176,  8 }, {  66,  8 }, { 206,  8 },
    { 194,  8 }, {  78,  8 }, {  97,  9 }, { 175,  9 }, { 161,  9 },
    { 111,  9 }, {   5,  8 }, {  11,  8 }, { 228,  8 }, {  44,  8 },
    {  36,  8 }, { 236,  8 }, { 114, 10 }, { 158, 10 }, { 100, 10 },
    { 172, 10 }, {  83,  9 }, { 189,  9 }, {  33,  7 }, { 239,  7 },
    { 164, 10 }, { 108, 10 }, { 198, 10 }, {  74, 10 }, { 179,  9 },
    {  93,  9 }, {  70, 10 }, { 202, 10 }, { 146, 10 }, { 126, 10 },
    { 213,  9 }, {  59,  9 }, {  18,  7 }, { 254,  7 }, { 225,  7 },
    {  47,  7 }, { 242,  7 }, {  30,  7 }, {   1,  6 }, {  15,  6 },
    /* Coefficient table A.4.4 - 225 entries */
    { 193,  7 }, {  79,  7 }, {  20,  7 }, { 252,  7 }, { 151, 10 },
    { 121, 10 }, { 119, 10 }, { 153, 10 }, { 118, 10 }, { 154, 10 },
    { 150, 10 }, { 122, 10 }, { 167, 10 }, { 105, 10 }, { 103, 10 },
    { 169, 10 }, { 117, 10 }, { 155, 10 }, { 149, 10 }, { 123, 10 },
    { 244,  7 }, {  28,  7 }, {  87, 10 }, { 185, 10 }, { 183, 10 },
    {  89, 10 }, { 166, 10 }, { 106, 10 }, { 102, 10 }, { 170, 10 },
    {  71, 10 }, { 201, 10 }, { 148, 10 }, { 124, 10 }, { 199, 10 },
    {  73, 10 }, { 116, 10 }, { 156, 10 }, { 165, 10 }, { 107, 10 },
    { 182, 10 }, {  90, 10 }, {  86, 10 }, { 186, 10 }, { 115, 10 },
    { 157, 10 }, { 101, 10 }, { 171, 10 }, { 147, 10 }, { 125, 10 },
    {  55, 10 }, { 217, 10 }, { 215, 10 }, {  57, 10 }, { 114, 10 },
    { 158, 10 }, { 146, 10 }, { 126, 10 }, { 164, 10 }, { 108, 10 },
    {  39, 10 }, { 233, 10 }, { 100, 10 }, { 172, 10 }, { 231, 10 },
    {  41, 10 }, { 198, 10 }, {  74, 10 }, {  70, 10 }, { 202, 10 },
    { 145, 10 }, { 127, 10 }, { 181, 10 }, {  91, 10 }, {  85, 10 },
    { 187, 10 }, { 113, 10 }, { 159, 10 }, { 247, 10 }, {  25, 10 },
    { 112, 10 }, { 144, 10 }, {  23, 10 }, { 249, 10 }, {   7, 10 },
    {   9, 10 }, {  99, 10 }, { 173, 10 }, {  54, 10 }, { 218, 10 },
    { 163, 10 }, { 109, 10 }, { 214, 10 }, {  58, 10 }, { 197, 10 },
    {  75, 10 }, {  84, 10 }, { 188, 10 }, {  98, 10 }, { 174, 10 },
    {  69, 10 }, { 203, 10 }, {  67,  8 }, { 205,  8 }, { 180, 10 },
    {  92, 10 }, { 162, 10 }, { 110, 10 }, {  97,  9 }, { 175,  9 },
    {  81,  8 }, { 191,  8 }, { 177,  8 }, {  95,  8 }, { 210,  7 },
    {  62,  7 }, { 212,  8 }, {  60,  8 }, { 195,  8 }, {  77,  8 },
    {  35,  7 }, { 237,  7 }, {  64,  7 }, { 192,  7 }, {  50,  7 },
    { 222,  7 }, { 227,  7 }, {  45,  7 }, {  52,  8 }, { 220,  8 },
    {  21,  8 }, { 251,  8 }, {   4,  7 }, {  12,  7 }, { 245,  8 },
    {  27,  8 }, { 161,  9 }, { 111,  9 }, {  22,  9 }, { 250,  9 },
    { 225,  6 }, {  47,  6 }, {  33,  6 }, { 239,  6 }, {  18,  6 },
    { 254,  6 }, { 242,  6 }, {  30,  6 }, {  80,  8 }, { 176,  8 },
    { 246,  9 }, {  26,  9 }, {  96,  9 }, { 160,  9 }, {   5,  8 },
    {  11,  8 }, {   6,  9 }, {  10,  9 }, { 179,  9 }, {  93,  9 },
    {  83,  9 }, { 189,  9 }, {  53,  9 }, { 219,  9 }, { 213,  9 },
    {  59,  9 }, { 196,  9 }, {  76,  9 }, {  49,  7 }, { 223,  7 },
    {  16,  5 }, { 240,  5 }, {   1,  5 }, {  15,  5 }, {   0,  4 },
    { 209,  7 }, {  63,  7 }, {  19,  7 }, { 253,  7 }, {  32,  6 },
    { 224,  6 }, {   2,  6 }, {  14,  6 }, { 243,  7 }, {  29,  7 },
    {  51,  8 }, { 221,  8 }, { 211,  8 }, {  61,  8 }, {  66,  8 },
    { 206,  8 }, { 194,  8 }, {  78,  8 }, { 228,  8 }, {  44,  8 },
    {  36,  8 }, { 236,  8 }, {  48,  7 }, { 208,  7 }, {   3,  7 },
    {  13,  7 }, {  68,  9 }, { 204,  9 }, {  82,  9 }, { 190,  9 },
    { 178,  9 }, {  94,  9 }, {  37,  9 }, { 235,  9 }, {  34,  7 },
    { 238,  7 }, { 226,  7 }, {  46,  7 }, { 229,  9 }, {  43,  9 },
    {  38, 10 }, { 234, 10 }, { 230, 10 }, {  42, 10 }, {  65,  8 },
    { 207,  8 }, {  17,  6 }, { 255,  6 }, { 241,  6 }, {  31,  6 },
    /* Coefficient table A.5.1 - 31 entries */
    {   8,  4 }, {  24,  4 }, {  15,  5 }, {  17,  5 }, {   4,  5 },
    {  28,  5 }, {   9,  4 }, {  23,  4 }, {  14,  5 }, {  18,  5 },
    {   3,  5 }, {  29,  5 }, {  13,  5 }, {  19,  5 }, {   2,  5 },
    {  30,  5 }, {  12,  5 }, {  20,  5 }, {   1,  5 }, {  31,  5 },
    {   7,  6 }, {  25,  6 }, {   0,  5 }, {  11,  5 }, {  21,  5 },
    {   6,  6 }, {  26,  6 }, {   5,  6 }, {  27,  6 }, {  10,  5 },
    {  22,  5 },
    /* Coefficient table A.5.2 - 31 entries */
    {   2,  4 }, {  30,  4 }, {  11,  5 }, {  21,  5 }, {   7,  5 },
    {  25,  5 }, {   1,  4 }, {  31,  4 }, {   6,  5 }, {  26,  5 },
    {  10,  5 }, {  22,  5 }, {  15,  6 }, {  17,  6 }, {  14,  6 },
    {  18,  6 }, {   0,  4 }, {   5,  5 }, {  27,  5 }, {   9,  5 },
    {  23,  5 }, {   8,  5 }, {  24,  5 }, {   4,  5 }, {  28,  5 },
    {  13,  6 }, {  19,  6 }, {  12,  6 }, {  20,  6 }, {   3,  5 },
    {  29,  5 },
    /* Coefficient table A.5.3 - 31 entries */
    {   0,  3 }, {   9,  5 }, {  23,  5 }, {   7,  5 }, {  25,  5 },
    {   3,  4 }, {  29,  4 }, {   8,  5 }, {  24,  5 }, {  15,  7 },
    {  17,  7 }, {  14,  7 }, {  18,  7 }, {  11,  6 }, {  21,  6 },
    {   2,  4 }, {  30,  4 }, {   6,  5 }, {  26,  5 }, {   5,  5 },
    {  27,  5 }, {   1,  4 }, {  31,  4 }, {  10,  6 }, {  22,  6 },
    {  13,  7 }, {  19,  7 }, {  12,  7 }, {  20,  7 }, {   4,  5 },
    {  28,  5 },
    /* Coefficient table A.5.4 - 31 entries */
    {   8,  5 }, {  24,  5 }, {   7,  5 }, {  25,  5 }, {   4,  4 },
    {  28,  4 }, {   0,  3 }, {  10,  6 }, {  22,  6 }, {  13,  7 },
    {  19,  7 }, {  12,  7 }, {  20,  7 }, {   6,  5 }, {  26,  5 },
    {   3,  4 }, {  29,  4 }, {   2,  4 }, {  30,  4 }, {   9,  6 },
    {  23,  6 }, {  15,  8 }, {  17,  8 }, {  14,  8 }, {  18,  8 },
    {  11,  7 }, {  21,  7 }, {   5,  5 }, {  27,  5 }, {   1,  4 },
    {  31,  4 },
    /* Coefficient table A.6.1 - 63 entries */
    {  19,  5 }, {  45,  5 }, {  31,  6 }, {  33,  6 }, {  30,  6 },
    {  34,  6 }, {  16,  5 }, {  48,  5 }, {   8,  6 }, {  56,  6 },
    {   7,  6 }, {  57,  6 }, {  18,  5 }, {  46,  5 }, {  29,  6 },
    {  35,  6 }, {  28,  6 }, {  36,  6 }, {   6,  6 }, {  58,  6 },
    {  27,  6 }, {  37,  6 }, {  17,  5 }, {  47,  5 }, {   5,  6 },
    {  59,  6 }, {  26,  6 }, {  38,  6 }, {   4,  6 }, {  60,  6 },
    {  25,  6 }, {  39,  6 }, {   3,  6 }, {  61,  6 }, {   2,  6 },
    {  62,  6 }, {  24,  6 }, {  40,  6 }, {  15,  7 }, {  49,  7 },
    {  14,  7 }, {  50,  7 }, {   1,  6 }, {  63,  6 }, {  23,  6 },
    {  41,  6 }, {  13,  7 }, {  51,  7 }, {   0,  6 }, {  22,  6 },
    {  42,  6 }, {  12,  7 }, {  52,  7 }, {  11,  7 }, {  53,  7 },
    {  21,  6 }, {  43,  6 }, {  20,  6 }, {  44,  6 }, {  10,  7 },
    {  54,  7 }, {   9,  7 }, {  55,  7 },
    /* Coefficient table A.6.2 - 63 entries */
    {  23,  6 }, {  41,  6 }, {  15,  6 }, {  49,  6 }, {   4,  5 },
    {  60,  5 }, {   3,  5 }, {  61,  5 }, {  22,  6 }, {  42,  6 },
    {  14,  6 }, {  50,  6 }, {   2,  5 }, {  62,  5 }, {  31,  7 },
    {  33,  7 }, {  30,  7 }, {  34,  7 }, {  21,  6 }, {  43,  6 },
    {  13,  6 }, {  51,  6 }, {  12,  6 }, {  52,  6 }, {   1,  5 },
    {  63,  5 }, {  20,  6 }, {  44,  6 }, {  29,  7 }, {  35,  7 },
    {  28,  7 }, {  36,  7 }, {  11,  6 }, {  53,  6 }, {  19,  6 },
    {  45,  6 }, {   0,  5 }, {  10,  6 }, {  54,  6 }, {  18,  6 },
    {  46,  6 }, {  27,  7 }, {  37,  7 }, {  26,  7 }, {  38,  7 },
    {   9,  6 }, {  55,  6 }, {  16,  6 }, {  48,  6 }, {   8,  6 },
    {  56,  6 }, {  17,  6 }, {  47,  6 }, {   7,  6 }, {  57,  6 },
    {  25,  7 }, {  39,  7 }, {  24,  7 }, {  40,  7 }, {   6,  6 },
    {  58,  6 }, {   5,  6 }, {  59,  6 },
    /* Coefficient table A.6.3 - 63 entries */
    {   0,  4 }, {   7,  5 }, {  57,  5 }, {  15,  6 }, {  49,  6 },
    {  25,  7 }, {  39,  7 }, {  24,  7 }, {  40,  7 }, {  18,  6 },
    {  46,  6 }, {  14,  6 }, {  50,  6 }, {   6,  5 }, {  58,  5 },
    {   5,  5 }, {  59,  5 }, {  31,  8 }, {  33,  8 }, {  30,  8 },
    {  34,  8 }, {  23,  7 }, {  41,  7 }, {  17,  6 }, {  47,  6 },
    {  13,  6 }, {  51,  6 }, {  16,  6 }, {  48,  6 }, {   4,  5 },
    {  60,  5 }, {  12,  6 }, {  52,  6 }, {  22,  7 }, {  42,  7 },
    {  29,  8 }, {  35,  8 }, {  28,  8 }, {  36,  8 }, {   3,  5 },
    {  61,  5 }, {  11,  6 }, {  53,  6 }, {  10,  6 }, {  54,  6 },
    {   2,  5 }, {  62,  5 }, {  21,  7 }, {  43,  7 }, {  20,  7 },
    {  44,  7 }, {   9,  6 }, {  55,  6 }, {   1,  5 }, {  63,  5 },
    {  27,  8 }, {  37,  8 }, {  26,  8 }, {  38,  8 }, {  19,  7 },
    {  45,  7 }, {   8,  6 }, {  56,  6 },
    /* Coefficient table A.6.4 - 63 entries */
    {  16,  6 }, {  48,  6 }, {  27,  8 }, {  37,  8 }, {  26,  8 },
    {  38,  8 }, {  21,  7 }, {  43,  7 }, {   8,  5 }, {  56,  5 },
    {   1,  4 }, {  63,  4 }, {  14,  6 }, {  50,  6 }, {  20,  7 },
    {  44,  7 }, {  31,  9 }, {  33,  9 }, {  30,  9 }, {  34,  9 },
    {  25,  8 }, {  39,  8 }, {   7,  5 }, {  57,  5 }, {   0,  4 },
    {   6,  5 }, {  58,  5 }, {  13,  6 }, {  51,  6 }, {  12,  6 },
    {  52,  6 }, {   5,  5 }, {  59,  5 }, {  19,  7 }, {  45,  7 },
    {  24,  8 }, {  40,  8 }, {  29,  9 }, {  35,  9 }, {  28,  9 },
    {  36,  9 }, {  11,  6 }, {  53,  6 }, {   4,  5 }, {  60,  5 },
    {  18,  7 }, {  46,  7 }, {  23,  8 }, {  41,  8 }, {  22,  8 },
    {  42,  8 }, {  10,  6 }, {  54,  6 }, {   3,  5 }, {  61,  5 },
    {   2,  5 }, {  62,  5 }, {  17,  7 }, {  47,  7 }, {  15,  7 },
    {  49,  7 }, {   9,  6 }, {  55,  6 },
    /* Coefficient table A.7.1 - 127 entries */
    {  18,  7 }, { 110,  7 }, {  63,  7 }, {  65,  7 }, {  38,  6 },
    {  90,  6 }, {  17,  7 }, { 111,  7 }, {  62,  7 }, {  66,  7 },
    {  61,  7 }, {  67,  7 }, {  16,  7 }, { 112,  7 }, {  37,  6 },
    {  91,  6 }, {  60,  7 }, {  68,  7 }, {  15,  7 }, { 113,  7 },
    {  32,  6 }, {  96,  6 }, {  59,  7 }, {  69,  7 }, {  14,  7 },
    { 114,  7 }, {  36,  6 }, {  92,  6 }, {  58,  7 }, {  70,  7 },
    {  13,  7 }, { 115,  7 }, {  57,  7 }, {  71,  7 }, {  12,  7 },
    { 116,  7 }, {  35,  6 }, {  93,  6 }, {  56,  7 }, {  72,  7 },
    {  55,  7 }, {  73,  7 }, {  11,  7 }, { 117,  7 }, {  10,  7 },
    { 118,  7 }, {  34,  6 }, {  94,  6 }, {  54,  7 }, {  74,  7 },
    {   9,  7 }, { 119,  7 }, {  33,  6 }, {  95,  6 }, {  53,  7 },
    {  75,  7 }, {  52,  7 }, {  76,  7 }, {   8,  7 }, { 120,  7 },
    {  51,  7 }, {  77,  7 }, {  50,  7 }, {  78,  7 }, {   7,  7 },
    { 121,  7 }, {   5,  7 }, { 123,  7 }, {   6,  7 }, { 122,  7 },
    {  49,  7 }, {  79,  7 }, {   4,  7 }, { 124,  7 }, {  31,  8 },
    {  97,  8 }, {  30,  8 }, {  98,  8 }, {  48,  7 }, {  80,  7 },
    {   3,  7 }, { 125,  7 }, {  29,  8 }, {  99,  8 }, {  28,  8 },
    { 100,  8 }, {  47,  7 }, {  81,  7 }, {   2,  7 }, { 126,  7 },
    {  46,  7 }, {  82,  7 }, {   1,  7 }, { 127,  7 }, {  27,  8 },
    { 101,  8 }, {  26,  8 }, { 102,  8 }, {  45,  7 }, {  83,  7 },
    {   0,  7 }, {  25,  8 }, { 103,  8 }, {  44,  7 }, {  84,  7 },
    {  24,  8 }, { 104,  8 }, {  23,  8 }, { 105,  8 }, {  43,  7 },
    {  85,  7 }, {  42,  7 }, {  86,  7 }, {  22,  8 }, { 106,  8 },
    {  21,  8 }, { 107,  8 }, {  41,  7 }, {  87,  7 }, {  20,  8 },
    { 108,  8 }, {  19,  8 }, { 109,  8 }, {  40,  7 }, {  88,  7 },
    {  39,  7 }, {  89,  7 },
    /* Coefficient table A.7.2 - 127 entries */
    {  47,  7 }, {  81,  7 }, {  46,  7 }, {  82,  7 }, {   8,  6 },
    { 120,  6 }, {  31,  7 }, {  97,  7 }, {  45,  7 }, {  83,  7 },
    {   7,  6 }, { 121,  6 }, {  30,  7 }, {  98,  7 }, {  29,  7 },
    {  99,  7 }, {   6,  6 }, { 122,  6 }, {  63,  8 }, {  65,  8 },
    {  62,  8 }, {  66,  8 }, {  44,  7 }, {  84,  7 }, {   5,  6 },
    { 123,  6 }, {  28,  7 }, { 100,  7 }, {  43,  7 }, {  85,  7 },
    {   4,  6 }, { 124,  6 }, {  27,  7 }, { 101,  7 }, {  26,  7 },
    { 102,  7 }, {  42,  7 }, {  86,  7 }, {  61,  8 }, {  67,  8 },
    {  60,  8 }, {  68,  8 }, {   3,  6 }, { 125,  6 }, {  25,  7 },
    { 103,  7 }, {  41,  7 }, {  87,  7 }, {   2,  6 }, { 126,  6 },
    {  24,  7 }, { 104,  7 }, {  59,  8 }, {  69,  8 }, {  58,  8 },
    {  70,  8 }, {  40,  7 }, {  88,  7 }, {  23,  7 }, { 105,  7 },
    {  39,  7 }, {  89,  7 }, {  22,  7 }, { 106,  7 }, {   1,  6 },
    { 127,  6 }, {  57,  8 }, {  71,  8 }, {  56,  8 }, {  72,  8 },
    {  21,  7 }, { 107,  7 }, {  38,  7 }, {  90,  7 }, {  20,  7 },
    { 108,  7 }, {   0,  6 }, {  55,  8 }, {  73,  8 }, {  54,  8 },
    {  74,  8 }, {  37,  7 }, {  91,  7 }, {  19,  7 }, { 109,  7 },
    {  18,  7 }, { 110,  7 }, {  36,  7 }, {  92,  7 }, {  53,  8 },
    {  75,  8 }, {  52,  8 }, {  76,  8 }, {  17,  7 }, { 111,  7 },
    {  35,  7 }, {  93,  7 }, {  32,  7 }, {  96,  7 }, {  16,  7 },
    { 112,  7 }, {  15,  7 }, { 113,  7 }, {  34,  7 }, {  94,  7 },
    {  51,  8 }, {  77,  8 }, {  50,  8 }, {  78,  8 }, {  14,  7 },
    { 114,  7 }, {  13,  7 }, { 115,  7 }, {  33,  7 }, {  95,  7 },
    {  12,  7 }, { 116,  7 }, {  49,  8 }, {  79,  8 }, {  48,  8 },
    {  80,  8 }, {  11,  7 }, { 117,  7 }, {  10,  7 }, { 118,  7 },
    {   9,  7 }, { 119,  7 },
    /* Coefficient table A.7.3 - 127 entries */
    {  15,  6 }, { 113,  6 }, {  31,  7 }, {  97,  7 }, {  51,  8 },
    {  77,  8 }, {  50,  8 }, {  78,  8 }, {  14,  6 }, { 114,  6 },
    {   0,  5 }, {  37,  7 }, {  91,  7 }, {  30,  7 }, {  98,  7 },
    {  13,  6 }, { 115,  6 }, {  29,  7 }, {  99,  7 }, {  49,  8 },
    {  79,  8 }, {  63,  9 }, {  65,  9 }, {  62,  9 }, {  66,  9 },
    {  12,  6 }, { 116,  6 }, {  36,  7 }, {  92,  7 }, {  28,  7 },
    { 100,  7 }, {  35,  7 }, {  93,  7 }, {  48,  8 }, {  80,  8 },
    {  47,  8 }, {  81,  8 }, {  11,  6 }, { 117,  6 }, {  10,  6 },
    { 118,  6 }, {  27,  7 }, { 101,  7 }, {  34,  7 }, {  94,  7 },
    {  61,  9 }, {  67,  9 }, {  60,  9 }, {  68,  9 }, {  46,  8 },
    {  82,  8 }, {  26,  7 }, { 102,  7 }, {   9,  6 }, { 119,  6 },
    {  32,  7 }, {  96,  7 }, {  33,  7 }, {  95,  7 }, {   8,  6 },
    { 120,  6 }, {  25,  7 }, { 103,  7 }, {  45,  8 }, {  83,  8 },
    {  59,  9 }, {  69,  9 }, {  58,  9 }, {  70,  9 }, {   7,  6 },
    { 121,  6 }, {  24,  7 }, { 104,  7 }, {  23,  7 }, { 105,  7 },
    {   6,  6 }, { 122,  6 }, {  44,  8 }, {  84,  8 }, {  57,  9 },
    {  71,  9 }, {  56,  9 }, {  72,  9 }, {  22,  7 }, { 106,  7 },
    {   5,  6 }, { 123,  6 }, {  43,  8 }, {  85,  8 }, {  42,  8 },
    {  86,  8 }, {  21,  7 }, { 107,  7 }, {   4,  6 }, { 124,  6 },
    {  20,  7 }, { 108,  7 }, {  55,  9 }, {  73,  9 }, {  54,  9 },
    {  74,  9 }, {  41,  8 }, {  87,  8 }, {   3,  6 }, { 125,  6 },
    {  19,  7 }, { 109,  7 }, {  18,  7 }, { 110,  7 }, {   2,  6 },
    { 126,  6 }, {  40,  8 }, {  88,  8 }, {  53,  9 }, {  75,  9 },
    {  52,  9 }, {  76,  9 }, {  17,  7 }, { 111,  7 }, {  39,  8 },
    {  89,  8 }, {  38,  8 }, {  90,  8 }, {  16,  7 }, { 112,  7 },
    {   1,  6 }, { 127,  6 },
    /* Coefficient table A.7.4 - 127 entries */
    {   3,  5 }, { 125,  5 }, {  32,  7 }, {  96,  7 }, {  29,  7 },
    {  99,  7 }, {  16,  6 }, { 112,  6 }, {   2,  5 }, { 126,  5 },
    {  53,  9 }, {  75,  9 }, {  52,  9 }, {  76,  9 }, {  42,  8 },
    {  86,  8 }, {  28,  7 }, { 100,  7 }, {  15,  6 }, { 113,  6 },
    {  63, 10 }, {  65, 10 }, {  62, 10 }, {  66, 10 }, {  51,  9 },
    {  77,  9 }, {  41,  8 }, {  87,  8 }, {  27,  7 }, { 101,  7 },
    {  14,  6 }, { 114,  6 }, {   1,  5 }, { 127,  5 }, {  13,  6 },
    { 115,  6 }, {   0,  5 }, {  26,  7 }, { 102,  7 }, {  40,  8 },
    {  88,  8 }, {  50,  9 }, {  78,  9 }, {  61, 10 }, {  67, 10 },
    {  60, 10 }, {  68, 10 }, {  12,  6 }, { 116,  6 }, {  25,  7 },
    { 103,  7 }, {  39,  8 }, {  89,  8 }, {  49,  9 }, {  79,  9 },
    {  48,  9 }, {  80,  9 }, {  11,  6 }, { 117,  6 }, {  24,  7 },
    { 104,  7 }, {  38,  8 }, {  90,  8 }, {  59, 10 }, {  69, 10 },
    {  58, 10 }, {  70, 10 }, {  47,  9 }, {  81,  9 }, {  10,  6 },
    { 118,  6 }, {   9,  6 }, { 119,  6 }, {  23,  7 }, { 105,  7 },
    {  22,  7 }, { 106,  7 }, {   8,  6 }, { 120,  6 }, {  37,  8 },
    {  91,  8 }, {  36,  8 }, {  92,  8 }, {  21,  7 }, { 107,  7 },
    {   7,  6 }, { 121,  6 }, {   6,  6 }, { 122,  6 }, {  57, 10 },
    {  71, 10 }, {  56, 10 }, {  72, 10 }, {  46,  9 }, {  82,  9 },
    {  35,  8 }, {  93,  8 }, {  20,  7 }, { 108,  7 }, {   5,  6 },
    { 123,  6 }, {  19,  7 }, { 109,  7 }, {  45,  9 }, {  83,  9 },
    {  44,  9 }, {  84,  9 }, {  31,  8 }, {  97,  8 }, {   4,  6 },
    { 124,  6 }, {  34,  8 }, {  94,  8 }, {  55, 10 }, {  73, 10 },
    {  54, 10 }, {  74, 10 }, {  43,  9 }, {  85,  9 }, {  18,  7 },
    { 110,  7 }, {  30,  8 }, {  98,  8 }, {  33,  8 }, {  95,  8 },
    {  17,  7 }, { 111,  7 },
    /* Coefficient table B.2.2 - 80 entries */
    {  80,  5 }, { 240,  5 }, { 253, 10 }, {  87, 10 }, { 117, 10 },
    { 223, 10 }, {  93, 10 }, { 247, 10 }, { 221, 10 }, { 119, 10 },
    { 213, 10 }, { 127, 10 }, {  85, 10 }, { 255, 10 }, { 245, 10 },
    {  95, 10 }, {  81, 10 }, { 243, 10 }, {  17,  7 }, {  51,  7 },
    {  20,  6 }, {  60,  6 }, { 116,  9 }, { 220,  9 }, {  84,  9 },
    { 252,  9 }, {  68,  8 }, { 204,  8 }, { 212,  7 }, { 124,  7 },
    { 196,  6 }, {  76,  6 }, {   5,  5 }, {  15,  5 }, {  53,  7 },
    {  31,  7 }, {  61,  7 }, {  23,  7 }, {  52,  6 }, {  28,  6 },
    {  13,  5 }, {   7,  5 }, {  16,  4 }, {  48,  4 }, {   4,  4 },
    {  12,  4 }, {  49,  6 }, {  19,  6 }, { 244,  7 }, {  92,  7 },
    {  21,  9 }, {  63,  9 }, { 197,  9 }, {  79,  9 }, {  69, 10 },
    { 207, 10 }, { 205, 10 }, {  71, 10 }, { 241,  9 }, {  83,  9 },
    { 208,  6 }, { 112,  6 }, {  77,  9 }, { 199,  9 }, { 113,  9 },
    { 211,  9 }, {  29,  9 }, {  55,  9 }, { 209, 10 }, { 115, 10 },
    { 125, 10 }, { 215, 10 }, {  65,  8 }, { 195,  8 }, { 193,  8 },
    {  67,  8 }, {  64,  4 }, { 192,  4 }, {   1,  4 }, {   3,  4 },
    /* Coefficient table B.2.3 - 81 entries */
    {  16,  4 }, {  48,  4 }, {   4,  4 }, {  12,  4 }, {  64,  4 },
    { 192,  4 }, {   5,  6 }, {  15,  6 }, {  13,  6 }, {   7,  6 },
    { 212,  8 }, { 124,  8 }, { 213, 10 }, { 127, 10 }, {  93, 10 },
    { 247, 10 }, { 117, 10 }, { 223, 10 }, { 253, 10 }, {  87, 10 },
    {  21,  9 }, {  63,  9 }, { 221, 10 }, { 119, 10 }, {  81, 10 },
    { 243, 10 }, { 205, 10 }, {  71, 10 }, {  69, 10 }, { 207, 10 },
    { 245, 10 }, {  95, 10 }, { 209, 10 }, { 115, 10 }, { 125, 10 },
    { 215, 10 }, {  85, 10 }, { 255, 10 }, { 116, 10 }, { 220, 10 },
    { 241, 10 }, {  83, 10 }, {  77, 10 }, { 199, 10 }, {  84, 10 },
    { 252, 10 }, {  29,  9 }, {  55,  9 }, {  61,  8 }, {  23,  8 },
    {  53,  8 }, {  31,  8 }, {   1,  4 }, {   3,  4 }, { 193,  8 },
    {  67,  8 }, {  65,  8 }, { 195,  8 }, { 196,  7 }, {  76,  7 },
    {  49,  7 }, {  19,  7 }, {  20,  7 }, {  60,  7 }, {  52,  7 },
    {  28,  7 }, { 208,  7 }, { 112,  7 }, {  80,  7 }, { 240,  7 },
    { 113, 10 }, { 211, 10 }, { 197, 10 }, {  79, 10 }, {  68,  9 },
    { 204,  9 }, {  17,  9 }, {  51,  9 }, { 244,  9 }, {  92,  9 },
    {   0,  2 },
    /* Coefficient table B.2.4 - 81 entries */
    {   1,  4 }, {   3,  4 }, {  52,  7 }, {  28,  7 }, { 193,  8 },
    {  67,  8 }, { 213, 10 }, { 127, 10 }, { 253, 10 }, {  87, 10 },
    {  93, 10 }, { 247, 10 }, { 117, 10 }, { 223, 10 }, {  80,  7 },
    { 240,  7 }, { 196,  7 }, {  76,  7 }, { 208,  7 }, { 112,  7 },
    { 221, 10 }, { 119, 10 }, { 245, 10 }, {  95, 10 }, {  69, 10 },
    { 207, 10 }, {  85, 10 }, { 255, 10 }, { 209, 10 }, { 115, 10 },
    { 205, 10 }, {  71, 10 }, {  81, 10 }, { 243, 10 }, { 125, 10 },
    { 215, 10 }, {   5,  7 }, {  15,  7 }, { 116, 10 }, { 220, 10 },
    {  84, 10 }, { 252, 10 }, {  29, 10 }, {  55, 10 }, { 113, 10 },
    { 211, 10 }, { 241, 10 }, {  83, 10 }, {  21, 10 }, {  63, 10 },
    { 197, 10 }, {  79, 10 }, {  77, 10 }, { 199, 10 }, {  16,  5 },
    {  48,  5 }, {   4,  5 }, {  12,  5 }, {  49,  7 }, {  19,  7 },
    {  13,  7 }, {   7,  7 }, { 212, 10 }, { 124, 10 }, { 244, 10 },
    {  92, 10 }, {  68,  9 }, { 204,  9 }, {  53,  9 }, {  31,  9 },
    {  61,  9 }, {  23,  9 }, {  17,  9 }, {  51,  9 }, {  65,  9 },
    { 195,  9 }, {  20,  8 }, {  60,  8 }, {  64,  5 }, { 192,  5 },
    {   0,  1 },
    /* Coefficient table B.3.2 - 49 entries */
    {  58,  5 }, {  14,  5 }, {  59,  6 }, {  13,  6 }, {  19,  7 },
    {  53,  7 }, {  25,  7 }, {  47,  7 }, {   1,  4 }, {   7,  4 },
    {   0,  2 }, {   9,  6 }, {  63,  6 }, {  57,  6 }, {  15,  6 },
    {  24,  6 }, {  40,  6 }, {  17,  6 }, {  55,  6 }, {  42,  8 },
    {  30,  8 }, {  43,  9 }, {  29,  9 }, {  27,  9 }, {  45,  9 },
    {  41,  7 }, {  31,  7 }, {   3,  6 }, {   5,  6 }, {   8,  5 },
    {  56,  5 }, {  49,  6 }, {  23,  6 }, {  11,  7 }, {  61,  7 },
    {  26,  8 }, {  46,  8 }, {  51,  8 }, {  21,  8 }, {  16,  5 },
    {  48,  5 }, {  18,  7 }, {  54,  7 }, {  50,  7 }, {  22,  7 },
    {  10,  6 }, {  62,  6 }, {   2,  5 }, {   6,  5 },
    /* Coefficient table B.3.3 - 49 entries */
    {  57,  5 }, {  15,  5 }, {   9,  5 }, {  63,  5 }, {  59,  7 },
    {  13,  7 }, {  50,  7 }, {  22,  7 }, {  18,  7 }, {  54,  7 },
    {  51,  8 }, {  21,  8 }, {  27, 10 }, {  45, 10 }, {  43, 10 },
    {  29, 10 }, {  26,  9 }, {  46,  9 }, {  17,  6 }, {  55,  6 },
    {  49,  6 }, {  23,  6 }, {   8,  4 }, {  56,  4 }, {  16,  5 },
    {  48,  5 }, {  10,  6 }, {  62,  6 }, {  58,  6 }, {  14,  6 },
    {   1,  4 }, {   7,  4 }, {   2,  5 }, {   6,  5 }, {  25,  8 },
    {  47,  8 }, {  42,  9 }, {  30,  9 }, {  19,  9 }, {  53,  9 },
    {  24,  7 }, {  40,  7 }, {  41,  8 }, {  31,  8 }, {  11,  8 },
    {  61,  8 }, {   3,  7 }, {   5,  7 }, {   0,  2 },
    /* Coefficient table B.3.4 - 49 entries */
    {   0,  1 }, {   8,  4 }, {  56,  4 }, {   1,  4 }, {   7,  4 },
    {  59,  8 }, {  13,  8 }, {  27, 10 }, {  45, 10 }, {  43, 10 },
    {  29, 10 }, {  26, 10 }, {  46, 10 }, {  51, 10 }, {  21, 10 },
    {  18,  8 }, {  54,  8 }, {  50,  8 }, {  22,  8 }, {  16,  6 },
    {  48,  6 }, {  17,  7 }, {  55,  7 }, {  24,  8 }, {  40,  8 },
    {  19, 10 }, {  53, 10 }, {  42, 10 }, {  30, 10 }, {  25,  9 },
    {  47,  9 }, {  49,  7 }, {  23,  7 }, {  10,  7 }, {  62,  7 },
    {   2,  6 }, {   6,  6 }, {  58,  7 }, {  14,  7 }, {  11,  9 },
    {  61,  9 }, {  41,  9 }, {  31,  9 }, {   3,  8 }, {   5,  8 },
    {   9,  6 }, {  63,  6 }, {  57,  6 }, {  15,  6 },
    /* Coefficient table B.4.2 - 225 entries */
    {  33,  7 }, { 239,  7 }, {  36,  7 }, { 236,  7 }, { 119, 10 },
    { 153, 10 }, { 151, 10 }, { 121, 10 }, { 118, 10 }, { 154, 10 },
    { 167, 10 }, { 105, 10 }, { 247,  8 }, {  25,  8 }, { 112,  8 },
    { 144,  8 }, { 150, 10 }, { 122, 10 }, { 103, 10 }, { 169, 10 },
    {  87, 10 }, { 185, 10 }, { 183, 10 }, {  89, 10 }, {  53,  8 },
    { 219,  8 }, { 213,  8 }, {  59,  8 }, { 228,  7 }, {  44,  7 },
    { 225,  7 }, {  47,  7 }, { 166, 10 }, { 106, 10 }, { 102, 10 },
    { 170, 10 }, { 149, 10 }, { 123, 10 }, { 117, 10 }, { 155, 10 },
    { 114,  9 }, { 158,  9 }, { 165, 10 }, { 107, 10 }, { 148, 10 },
    { 124, 10 }, {  81,  7 }, { 191,  7 }, {  21,  7 }, { 251,  7 },
    {  64,  6 }, { 192,  6 }, { 242,  7 }, {  30,  7 }, { 116, 10 },
    { 156, 10 }, {  71, 10 }, { 201, 10 }, { 101, 10 }, { 171, 10 },
    { 199, 10 }, {  73, 10 }, {  19,  8 }, { 253,  8 }, {  18,  7 },
    { 254,  7 }, { 182, 10 }, {  90, 10 }, {  86, 10 }, { 186, 10 },
    {  39,  9 }, { 233,  9 }, {  34,  8 }, { 238,  8 }, {   2,  6 },
    {  14,  6 }, { 177,  7 }, {  95,  7 }, { 231,  9 }, {  41,  9 },
    { 146,  9 }, { 126,  9 }, {  67,  8 }, { 205,  8 }, {  17,  6 },
    { 255,  6 }, { 241,  6 }, {  31,  6 }, {  16,  5 }, { 240,  5 },
    { 243,  8 }, {  29,  8 }, { 195,  8 }, {  77,  8 }, { 245,  7 },
    {  27,  7 }, {   7,  8 }, {   9,  8 }, { 209,  8 }, {  63,  8 },
    {  99,  9 }, { 173,  9 }, { 163,  9 }, { 109,  9 }, { 226,  8 },
    {  46,  8 }, { 212,  8 }, {  60,  8 }, {  97,  8 }, { 175,  8 },
    { 161,  8 }, { 111,  8 }, {  52,  8 }, { 220,  8 }, { 180,  9 },
    {  92,  9 }, {  50,  9 }, { 222,  9 }, {  35,  9 }, { 237,  9 },
    { 211, 10 }, {  61, 10 }, {  51, 10 }, { 221, 10 }, { 210,  9 },
    {  62,  9 }, {  84,  9 }, { 188,  9 }, {  82,  8 }, { 190,  8 },
    {   4,  6 }, {  12,  6 }, { 227,  9 }, {  45,  9 }, { 214,  9 },
    {  58,  9 }, {  54,  9 }, { 218,  9 }, { 197,  9 }, {  75,  9 },
    { 178,  8 }, {  94,  8 }, {  22,  8 }, { 250,  8 }, {  65,  7 },
    { 207,  7 }, {  80,  7 }, { 176,  7 }, { 246,  8 }, {  26,  8 },
    {  69,  9 }, { 203,  9 }, { 145,  9 }, { 127,  9 }, { 193,  7 },
    {  79,  7 }, {  37,  8 }, { 235,  8 }, { 229,  8 }, {  43,  8 },
    {   5,  7 }, {  11,  7 }, {  20,  7 }, { 252,  7 }, { 244,  7 },
    {  28,  7 }, {   1,  5 }, {  15,  5 }, {  98,  9 }, { 174,  9 },
    { 147, 10 }, { 125, 10 }, { 115, 10 }, { 157, 10 }, {  96,  8 },
    { 160,  8 }, { 113,  9 }, { 159,  9 }, {  38,  9 }, { 234,  9 },
    { 162,  9 }, { 110,  9 }, {  55, 10 }, { 217, 10 }, { 215, 10 },
    {  57, 10 }, {  48,  8 }, { 208,  8 }, {  68,  9 }, { 204,  9 },
    { 164, 10 }, { 108, 10 }, { 100, 10 }, { 172, 10 }, {  32,  7 },
    { 224,  7 }, {   0,  4 }, {   3,  8 }, {  13,  8 }, {  23,  9 },
    { 249,  9 }, { 230,  9 }, {  42,  9 }, {  83,  9 }, { 189,  9 },
    { 181, 10 }, {  91, 10 }, { 198, 10 }, {  74, 10 }, {   6,  8 },
    {  10,  8 }, {  66,  8 }, { 206,  8 }, { 179,  9 }, {  93,  9 },
    {  70, 10 }, { 202, 10 }, {  85, 10 }, { 187, 10 }, { 194,  8 },
    {  78,  8 }, { 196,  9 }, {  76,  9 }, {  49,  9 }, { 223,  9 },
    /* Coefficient table B.4.3 - 225 entries */
    { 193,  7 }, {  79,  7 }, {  48,  7 }, { 208,  7 }, {   6,  8 },
    {  10,  8 }, { 151, 10 }, { 121, 10 }, { 119, 10 }, { 153, 10 },
    { 167, 10 }, { 105, 10 }, { 118, 10 }, { 154, 10 }, { 103, 10 },
    { 169, 10 }, { 150, 10 }, { 122, 10 }, { 102, 10 }, { 170, 10 },
    { 183, 10 }, {  89, 10 }, { 117, 10 }, { 155, 10 }, { 149, 10 },
    { 123, 10 }, { 166, 10 }, { 106, 10 }, {  87, 10 }, { 185, 10 },
    { 116, 10 }, { 156, 10 }, { 148, 10 }, { 124, 10 }, {  71, 10 },
    { 201, 10 }, { 199, 10 }, {  73, 10 }, { 165, 10 }, { 107, 10 },
    { 182, 10 }, {  90, 10 }, { 101, 10 }, { 171, 10 }, {  86, 10 },
    { 186, 10 }, {  20,  7 }, { 252,  7 }, {  55, 10 }, { 217, 10 },
    { 147, 10 }, { 125, 10 }, { 115, 10 }, { 157, 10 }, { 215, 10 },
    {  57, 10 }, { 100, 10 }, { 172, 10 }, { 181, 10 }, {  91, 10 },
    { 198, 10 }, {  74, 10 }, { 114, 10 }, { 158, 10 }, { 244,  7 },
    {  28,  7 }, {  70, 10 }, { 202, 10 }, { 164, 10 }, { 108, 10 },
    {  85, 10 }, { 187, 10 }, { 146, 10 }, { 126, 10 }, {  39, 10 },
    { 233, 10 }, { 231, 10 }, {  41, 10 }, {  53,  9 }, { 219,  9 },
    {  54, 10 }, { 218, 10 }, {  99, 10 }, { 173, 10 }, { 214, 10 },
    {  58, 10 }, { 163, 10 }, { 109, 10 }, { 197, 10 }, {  75, 10 },
    {  84, 10 }, { 188, 10 }, {  69, 10 }, { 203, 10 }, { 113, 10 },
    { 159, 10 }, { 180, 10 }, {  92, 10 }, { 145, 10 }, { 127, 10 },
    {  98, 10 }, { 174, 10 }, {  23, 10 }, { 249, 10 }, { 247, 10 },
    {  25, 10 }, { 162, 10 }, { 110, 10 }, {  38, 10 }, { 234, 10 },
    { 230, 10 }, {  42, 10 }, {  83, 10 }, { 189, 10 }, { 179, 10 },
    {  93, 10 }, { 211, 10 }, {  61, 10 }, { 112, 10 }, { 144, 10 },
    {  51, 10 }, { 221, 10 }, {  68, 10 }, { 204, 10 }, { 213, 10 },
    {  59, 10 }, { 196, 10 }, {  76, 10 }, {   3,  7 }, {  13,  7 },
    {  66,  8 }, { 206,  8 }, {  81,  8 }, { 191,  8 }, {  32,  6 },
    { 224,  6 }, {   0,  2 }, {   7,  9 }, {   9,  9 }, {  97,  9 },
    { 175,  9 }, {  36,  8 }, { 236,  8 }, { 194,  8 }, {  78,  8 },
    { 177,  8 }, {  95,  8 }, {   2,  6 }, {  14,  6 }, {  21,  8 },
    { 251,  8 }, { 245,  8 }, {  27,  8 }, { 228,  8 }, {  44,  8 },
    {  22,  9 }, { 250,  9 }, { 246,  9 }, {  26,  9 }, {  49,  8 },
    { 223,  8 }, {  19,  8 }, { 253,  8 }, { 209,  8 }, {  63,  8 },
    { 243,  8 }, {  29,  8 }, { 225,  7 }, {  47,  7 }, {  64,  7 },
    { 192,  7 }, {  33,  7 }, { 239,  7 }, {  18,  7 }, { 254,  7 },
    { 242,  7 }, {  30,  7 }, { 161,  9 }, { 111,  9 }, { 178,  9 },
    {  94,  9 }, {  82,  9 }, { 190,  9 }, { 195,  9 }, {  77,  9 },
    {  52,  9 }, { 220,  9 }, { 212,  9 }, {  60,  9 }, {  67,  9 },
    { 205,  9 }, {  37,  9 }, { 235,  9 }, {  34,  8 }, { 238,  8 },
    { 226,  8 }, {  46,  8 }, {  17,  6 }, { 255,  6 }, {   4,  7 },
    {  12,  7 }, {  80,  8 }, { 176,  8 }, { 229,  9 }, {  43,  9 },
    {  35,  9 }, { 237,  9 }, {  16,  5 }, { 240,  5 }, { 241,  6 },
    {  31,  6 }, { 227,  9 }, {  45,  9 }, { 210,  9 }, {  62,  9 },
    {  50,  9 }, { 222,  9 }, {  96,  9 }, { 160,  9 }, {   5,  8 },
    {  11,  8 }, {  65,  8 }, { 207,  8 }, {   1,  5 }, {  15,  5 },
    /* Coefficient table B.4.4 - 225 entries */
    {   3,  7 }, {  13,  7 }, {   4,  7 }, {  12,  7 }, { 151, 10 },
    { 121, 10 }, { 119, 10 }, { 153, 10 }, { 103, 10 }, { 169, 10 },
    { 167, 10 }, { 105, 10 }, { 150, 10 }, { 122, 10 }, { 118, 10 },
    { 154, 10 }, { 117, 10 }, { 155, 10 }, { 102, 10 }, { 170, 10 },
    {   5,  8 }, {  11,  8 }, { 183, 10 }, {  89, 10 }, {  87, 10 },
    { 185, 10 }, { 166, 10 }, { 106, 10 }, { 149, 10 }, { 123, 10 },
    {  17,  5 }, { 255,  5 }, {   1,  4 }, {  15,  4 }, { 241,  5 },
    {  31,  5 }, { 116, 10 }, { 156, 10 }, { 148, 10 }, { 124, 10 },
    { 199, 10 }, {  73, 10 }, { 182, 10 }, {  90, 10 }, {  71, 10 },
    { 201, 10 }, {  86, 10 }, { 186, 10 }, { 101, 10 }, { 171, 10 },
    { 115, 10 }, { 157, 10 }, { 165, 10 }, { 107, 10 }, { 215, 10 },
    {  57, 10 }, {  55, 10 }, { 217, 10 }, {  70, 10 }, { 202, 10 },
    { 147, 10 }, { 125, 10 }, {  85, 10 }, { 187, 10 }, { 164, 10 },
    { 108, 10 }, { 198, 10 }, {  74, 10 }, { 100, 10 }, { 172, 10 },
    { 114, 10 }, { 158, 10 }, {  39, 10 }, { 233, 10 }, { 146, 10 },
    { 126, 10 }, {  99, 10 }, { 173, 10 }, { 181, 10 }, {  91, 10 },
    { 231, 10 }, {  41, 10 }, { 163, 10 }, { 109, 10 }, {  54, 10 },
    { 218, 10 }, { 214, 10 }, {  58, 10 }, { 113, 10 }, { 159, 10 },
    {  23, 10 }, { 249, 10 }, { 145, 10 }, { 127, 10 }, { 180, 10 },
    {  92, 10 }, {  84, 10 }, { 188, 10 }, { 247, 10 }, {  25, 10 },
    {  32,  6 }, { 224,  6 }, { 197, 10 }, {  75, 10 }, {  98, 10 },
    { 174, 10 }, { 112, 10 }, { 144, 10 }, { 162, 10 }, { 110, 10 },
    {  69, 10 }, { 203, 10 }, {  38, 10 }, { 234, 10 }, { 230, 10 },
    {  42, 10 }, {   7, 10 }, {   9, 10 }, { 213, 10 }, {  59, 10 },
    {  83, 10 }, { 189, 10 }, { 179, 10 }, {  93, 10 }, {  53, 10 },
    { 219, 10 }, {  97, 10 }, { 175, 10 }, {  68, 10 }, { 204, 10 },
    { 196, 10 }, {  76, 10 }, { 161, 10 }, { 111, 10 }, {  22, 10 },
    { 250, 10 }, { 246, 10 }, {  26, 10 }, {  82, 10 }, { 190, 10 },
    {  96, 10 }, { 160, 10 }, { 178, 10 }, {  94, 10 }, {  37, 10 },
    { 235, 10 }, {  67, 10 }, { 205, 10 }, {  51, 10 }, { 221, 10 },
    { 229, 10 }, {  43, 10 }, {   6, 10 }, {  10, 10 }, { 211, 10 },
    {  61, 10 }, { 212, 10 }, {  60, 10 }, { 195, 10 }, {  77, 10 },
    {  52, 10 }, { 220, 10 }, {  81, 10 }, { 191, 10 }, { 177, 10 },
    {  95, 10 }, {  65,  8 }, { 207,  8 }, {  21, 10 }, { 251, 10 },
    { 245, 10 }, {  27, 10 }, { 194, 10 }, {  78, 10 }, { 228, 10 },
    {  44, 10 }, { 193,  8 }, {  79,  8 }, {  20,  8 }, { 252,  8 },
    {   0,  2 }, {   2,  6 }, {  14,  6 }, {  19,  8 }, { 253,  8 },
    {  49,  8 }, { 223,  8 }, { 209,  8 }, {  63,  8 }, { 244,  8 },
    {  28,  8 }, {  66, 10 }, { 206, 10 }, { 210, 10 }, {  62, 10 },
    {  36,  9 }, { 236,  9 }, { 243,  8 }, {  29,  8 }, {  33,  7 },
    { 239,  7 }, { 225,  7 }, {  47,  7 }, {  64,  8 }, { 192,  8 },
    {  80,  9 }, { 176,  9 }, {  50,  9 }, { 222,  9 }, {  18,  7 },
    { 254,  7 }, { 242,  7 }, {  30,  7 }, {  48,  8 }, { 208,  8 },
    { 226,  8 }, {  46,  8 }, {  34,  8 }, { 238,  8 }, { 227,  9 },
    {  45,  9 }, {  35,  9 }, { 237,  9 }, {  16,  5 }, { 240,  5 },
    /* Coefficient table B.5.2 - 31 entries */
    {   3,  4 }, {  29,  4 }, {   5,  5 }, {  27,  5 }, {  10,  5 },
    {  22,  5 }, {  13,  6 }, {  19,  6 }, {   7,  6 }, {  25,  6 },
    {   4,  5 }, {  28,  5 }, {   2,  4 }, {  30,  4 }, {   9,  5 },
    {  23,  5 }, {  12,  6 }, {  20,  6 }, {  15,  7 }, {  17,  7 },
    {  14,  7 }, {  18,  7 }, {   8,  5 }, {  24,  5 }, {   6,  6 },
    {  26,  6 }, {  11,  6 }, {  21,  6 }, {   0,  3 }, {   1,  4 },
    {  31,  4 },
    /* Coefficient table B.5.3 - 31 entries */
    {   1,  3 }, {  31,  3 }, {   7,  6 }, {  25,  6 }, {  13,  7 },
    {  19,  7 }, {  12,  7 }, {  20,  7 }, {  10,  6 }, {  22,  6 },
    {   6,  6 }, {  26,  6 }, {   2,  4 }, {  30,  4 }, {   0,  2 },
    {   4,  5 }, {  28,  5 }, {  15,  8 }, {  17,  8 }, {  14,  8 },
    {  18,  8 }, {  11,  7 }, {  21,  7 }, {   9,  6 }, {  23,  6 },
    {   3,  5 }, {  29,  5 }, {   8,  6 }, {  24,  6 }, {   5,  6 },
    {  27,  6 },
    /* Coefficient table B.5.4 - 31 entries */
    {   3,  4 }, {  29,  4 }, {   9,  6 }, {  23,  6 }, {   8,  6 },
    {  24,  6 }, {  13,  8 }, {  19,  8 }, {  12,  8 }, {  20,  8 },
    {  10,  7 }, {  22,  7 }, {   6,  6 }, {  26,  6 }, {   1,  3 },
    {  31,  3 }, {   2,  4 }, {  30,  4 }, {   4,  5 }, {  28,  5 },
    {  15,  9 }, {  17,  9 }, {  14,  9 }, {  18,  9 }, {  11,  8 },
    {  21,  8 }, {   7,  7 }, {  25,  7 }, {   5,  6 }, {  27,  6 },
    {   0,  2 },
    /* Coefficient table B.6.2 - 63 entries */
    {  11,  6 }, {  53,  6 }, {  21,  6 }, {  43,  6 }, {  27,  7 },
    {  37,  7 }, {  26,  7 }, {  38,  7 }, {  10,  6 }, {  54,  6 },
    {   5,  5 }, {  59,  5 }, {  20,  6 }, {  44,  6 }, {   9,  6 },
    {  55,  6 }, {  15,  7 }, {  49,  7 }, {  25,  7 }, {  39,  7 },
    {  19,  6 }, {  45,  6 }, {   4,  5 }, {  60,  5 }, {   1,  4 },
    {  63,  4 }, {  14,  7 }, {  50,  7 }, {  24,  7 }, {  40,  7 },
    {   8,  6 }, {  56,  6 }, {  18,  6 }, {  46,  6 }, {  31,  8 },
    {  33,  8 }, {  30,  8 }, {  34,  8 }, {  13,  7 }, {  51,  7 },
    {   3,  5 }, {  61,  5 }, {  23,  7 }, {  41,  7 }, {  12,  7 },
    {  52,  7 }, {   7,  6 }, {  57,  6 }, {  16,  6 }, {  48,  6 },
    {  17,  6 }, {  47,  6 }, {   0,  4 }, {   2,  5 }, {  62,  5 },
    {  22,  7 }, {  42,  7 }, {  29,  8 }, {  35,  8 }, {  28,  8 },
    {  36,  8 }, {   6,  6 }, {  58,  6 },
    /* Coefficient table B.6.3 - 63 entries */
    {   6,  5 }, {  58,  5 }, {  10,  6 }, {  54,  6 }, {  16,  6 },
    {  48,  6 }, {  17,  6 }, {  47,  6 }, {  21,  7 }, {  43,  7 },
    {  14,  7 }, {  50,  7 }, {   5,  5 }, {  59,  5 }, {   2,  4 },
    {  62,  4 }, {   9,  6 }, {  55,  6 }, {  25,  8 }, {  39,  8 },
    {  31,  9 }, {  33,  9 }, {  30,  9 }, {  34,  9 }, {  20,  7 },
    {  44,  7 }, {   8,  6 }, {  56,  6 }, {  13,  7 }, {  51,  7 },
    {  19,  7 }, {  45,  7 }, {   4,  5 }, {  60,  5 }, {  24,  8 },
    {  40,  8 }, {  29,  9 }, {  35,  9 }, {  28,  9 }, {  36,  9 },
    {  12,  7 }, {  52,  7 }, {   7,  6 }, {  57,  6 }, {   3,  5 },
    {  61,  5 }, {  18,  7 }, {  46,  7 }, {  23,  8 }, {  41,  8 },
    {  22,  8 }, {  42,  8 }, {  11,  7 }, {  53,  7 }, {  15,  8 },
    {  49,  8 }, {  27,  9 }, {  37,  9 }, {  26,  9 }, {  38,  9 },
    {   0,  3 }, {   1,  4 }, {  63,  4 },
    /* Coefficient table B.6.4 - 63 entries */
    {   1,  3 }, {  63,  3 }, {  14,  7 }, {  50,  7 }, {  18,  7 },
    {  46,  7 }, {  27,  9 }, {  37,  9 }, {  26,  9 }, {  38,  9 },
    {  21,  8 }, {  43,  8 }, {  13,  7 }, {  51,  7 }, {   9,  6 },
    {  55,  6 }, {  17,  7 }, {  47,  7 }, {  31, 10 }, {  33, 10 },
    {  30, 10 }, {  34, 10 }, {  25,  9 }, {  39,  9 }, {  20,  8 },
    {  44,  8 }, {   5,  5 }, {  59,  5 }, {  16,  7 }, {  48,  7 },
    {  12,  7 }, {  52,  7 }, {   8,  6 }, {  56,  6 }, {   2,  4 },
    {  62,  4 }, {   4,  5 }, {  60,  5 }, {  24,  9 }, {  40,  9 },
    {  29, 10 }, {  35, 10 }, {  28, 10 }, {  36, 10 }, {  15,  8 },
    {  49,  8 }, {  11,  7 }, {  53,  7 }, {   7,  6 }, {  57,  6 },
    {  19,  8 }, {  45,  8 }, {  23,  9 }, {  41,  9 }, {  22,  9 },
    {  42,  9 }, {  10,  7 }, {  54,  7 }, {   6,  6 }, {  58,  6 },
    {   3,  5 }, {  61,  5 }, {   0,  3 },
    /* Coefficient table B.7.2 - 127 entries */
    {   3,  5 }, { 125,  5 }, {  42,  7 }, {  86,  7 }, {  21,  7 },
    { 107,  7 }, {  11,  6 }, { 117,  6 }, {  41,  7 }, {  87,  7 },
    {  20,  7 }, { 108,  7 }, {  53,  8 }, {  75,  8 }, {  31,  8 },
    {  97,  8 }, {  40,  7 }, {  88,  7 }, {  52,  8 }, {  76,  8 },
    {  51,  8 }, {  77,  8 }, {  19,  7 }, { 109,  7 }, {  10,  6 },
    { 118,  6 }, {  30,  8 }, {  98,  8 }, {  29,  8 }, {  99,  8 },
    {  18,  7 }, { 110,  7 }, {   9,  6 }, { 119,  6 }, {  39,  7 },
    {  89,  7 }, {  17,  7 }, { 111,  7 }, {  28,  8 }, { 100,  8 },
    {  50,  8 }, {  78,  8 }, {  38,  7 }, {  90,  7 }, {   2,  5 },
    { 126,  5 }, {   8,  6 }, { 120,  6 }, {  49,  8 }, {  79,  8 },
    {  48,  8 }, {  80,  8 }, {  16,  7 }, { 112,  7 }, {  37,  7 },
    {  91,  7 }, {  63,  9 }, {  65,  9 }, {  62,  9 }, {  66,  9 },
    {  27,  8 }, { 101,  8 }, {   7,  6 }, { 121,  6 }, {  36,  7 },
    {  92,  7 }, {  26,  8 }, { 102,  8 }, {  47,  8 }, {  81,  8 },
    {  61,  9 }, {  67,  9 }, {  60,  9 }, {  68,  9 }, {  46,  8 },
    {  82,  8 }, {  15,  7 }, { 113,  7 }, {   6,  6 }, { 122,  6 },
    {  25,  8 }, { 103,  8 }, {  45,  8 }, {  83,  8 }, {  35,  7 },
    {  93,  7 }, {   1,  5 }, { 127,  5 }, {  14,  7 }, { 114,  7 },
    {  59,  9 }, {  69,  9 }, {  58,  9 }, {  70,  9 }, {  24,  8 },
    { 104,  8 }, {   5,  6 }, { 123,  6 }, {  32,  7 }, {  96,  7 },
    {  34,  7 }, {  94,  7 }, {  13,  7 }, { 115,  7 }, {  23,  8 },
    { 105,  8 }, {  44,  8 }, {  84,  8 }, {   4,  6 }, { 124,  6 },
    {  57,  9 }, {  71,  9 }, {  56,  9 }, {  72,  9 }, {  43,  8 },
    {  85,  8 }, {  12,  7 }, { 116,  7 }, {   0,  5 }, {  33,  7 },
    {  95,  7 }, {  22,  8 }, { 106,  8 }, {  55,  9 }, {  73,  9 },
    {  54,  9 }, {  74,  9 },
    /* Coefficient table B.7.3 - 127 entries */
    {   0,  3 }, {   4,  5 }, { 124,  5 }, {  19,  7 }, { 109,  7 },
    {  29,  8 }, {  99,  8 }, {  51,  9 }, {  77,  9 }, {  62, 10 },
    {  66, 10 }, {  63, 10 }, {  65, 10 }, {  28,  8 }, { 100,  8 },
    {  41,  8 }, {  87,  8 }, {  33,  7 }, {  95,  7 }, {  10,  6 },
    { 118,  6 }, {  18,  7 }, { 110,  7 }, {  40,  8 }, {  88,  8 },
    {  27,  8 }, { 101,  8 }, {   9,  6 }, { 119,  6 }, {  17,  7 },
    { 111,  7 }, {  50,  9 }, {  78,  9 }, {  61, 10 }, {  67, 10 },
    {  60, 10 }, {  68, 10 }, {  26,  8 }, { 102,  8 }, {   1,  4 },
    { 127,  4 }, {   3,  5 }, { 125,  5 }, {  39,  8 }, {  89,  8 },
    {  49,  9 }, {  79,  9 }, {  48,  9 }, {  80,  9 }, {  16,  7 },
    { 112,  7 }, {   8,  6 }, { 120,  6 }, {  25,  8 }, { 103,  8 },
    {  38,  8 }, {  90,  8 }, {  15,  7 }, { 113,  7 }, {   7,  6 },
    { 121,  6 }, {  59, 10 }, {  69, 10 }, {  58, 10 }, {  70, 10 },
    {  47,  9 }, {  81,  9 }, {  24,  8 }, { 104,  8 }, {  37,  8 },
    {  91,  8 }, {  23,  8 }, { 105,  8 }, {  14,  7 }, { 114,  7 },
    {  46,  9 }, {  82,  9 }, {  57, 10 }, {  71, 10 }, {  56, 10 },
    {  72, 10 }, {  36,  8 }, {  92,  8 }, {   2,  5 }, { 126,  5 },
    {   6,  6 }, { 122,  6 }, {  45,  9 }, {  83,  9 }, {  31,  9 },
    {  97,  9 }, {  22,  8 }, { 106,  8 }, {  13,  7 }, { 115,  7 },
    {  21,  8 }, { 107,  8 }, {  55, 10 }, {  73, 10 }, {  54, 10 },
    {  74, 10 }, {  44,  9 }, {  84,  9 }, {  12,  7 }, { 116,  7 },
    {   5,  6 }, { 123,  6 }, {  35,  8 }, {  93,  8 }, {  43,  9 },
    {  85,  9 }, {  30,  9 }, {  98,  9 }, {  20,  8 }, { 108,  8 },
    {  53, 10 }, {  75, 10 }, {  52, 10 }, {  76, 10 }, {  42,  9 },
    {  86,  9 }, {  34,  8 }, {  94,  8 }, {  32,  8 }, {  96,  8 },
    {  11,  7 }, { 117,  7 },
    /* Coefficient table B.7.4 - 127 entries */
    {  28,  8 }, { 100,  8 }, {  27,  8 }, { 101,  8 }, {  36,  8 },
    {  92,  8 }, {  63, 10 }, {  65, 10 }, {  61, 10 }, {  67, 10 },
    {  62, 10 }, {  66, 10 }, {  60, 10 }, {  68, 10 }, {  19,  7 },
    { 109,  7 }, {  59, 10 }, {  69, 10 }, {  58, 10 }, {  70, 10 },
    {  57, 10 }, {  71, 10 }, {  56, 10 }, {  72, 10 }, {  55, 10 },
    {  73, 10 }, {  53, 10 }, {  75, 10 }, {  54, 10 }, {  74, 10 },
    {  52, 10 }, {  76, 10 }, {   5,  5 }, { 123,  5 }, {   0,  3 },
    {  51, 10 }, {  77, 10 }, {  50, 10 }, {  78, 10 }, {  49, 10 },
    {  79, 10 }, {  48, 10 }, {  80, 10 }, {  47, 10 }, {  81, 10 },
    {  46, 10 }, {  82, 10 }, {  43,  9 }, {  85,  9 }, {  18,  7 },
    { 110,  7 }, {  11,  6 }, { 117,  6 }, {  26,  8 }, { 102,  8 },
    {  35,  8 }, {  93,  8 }, {  25,  8 }, { 103,  8 }, {  42,  9 },
    {  86,  9 }, {  41,  9 }, {  87,  9 }, {  10,  6 }, { 118,  6 },
    {   4,  5 }, { 124,  5 }, {  17,  7 }, { 111,  7 }, {  34,  8 },
    {  94,  8 }, {  24,  8 }, { 104,  8 }, {   9,  6 }, { 119,  6 },
    {   1,  4 }, { 127,  4 }, {  16,  7 }, { 112,  7 }, {  32,  8 },
    {  96,  8 }, {  33,  8 }, {  95,  8 }, {  23,  8 }, { 105,  8 },
    {  40,  9 }, {  88,  9 }, {  31,  9 }, {  97,  9 }, {  15,  7 },
    { 113,  7 }, {   8,  6 }, { 120,  6 }, {  39,  9 }, {  89,  9 },
    {  45, 10 }, {  83, 10 }, {  44, 10 }, {  84, 10 }, {  22,  8 },
    { 106,  8 }, {  14,  7 }, { 114,  7 }, {   3,  5 }, { 125,  5 },
    {   7,  6 }, { 121,  6 }, {  30,  9 }, {  98,  9 }, {  38,  9 },
    {  90,  9 }, {  21,  8 }, { 107,  8 }, {  13,  7 }, { 115,  7 },
    {  29,  9 }, {  99,  9 }, {  37,  9 }, {  91,  9 }, {  20,  8 },
    { 108,  8 }, {  12,  7 }, { 116,  7 }, {   6,  6 }, { 122,  6 },
    {   2,  5 }, { 126,  5 },
};

static const HuffmanCodebook at9_huffman_coeffs[][8][4] = {
    {
        { { 0 } },
        { { 0 } },
        {
            {   8,   2,   1,   2 },
            {  80,   4,   2,   2 },
            {  81,   4,   2,   2 },
            {  81,   4,   2,   2 },
        },
        {
            {  40,   2,   1,   3 },
            {  49,   2,   1,   3 },
            {  49,   2,   1,   3 },
            {  49,   2,   1,   3 },
        },
        {
            { 176,   2,   1,   4 },
            { 225,   2,   1,   4 },
            { 225,   2,   1,   4 },
            { 225,   2,   1,   4 },
        },
        {
            {  31,   1,   0,   5 },
            {  31,   1,   0,   5 },
            {  31,   1,   0,   5 },
            {  31,   1,   0,   5 },
        },
        {
            {  63,   1,   0,   6 },
            {  63,   1,   0,   6 },
            {  63,   1,   0,   6 },
            {  63,   1,   0,   6 },
        },
        {
            { 127,   1,   0,   7 },
            { 127,   1,   0,   7 },
            { 127,   1,   0,   7 },
            { 127,   1,   0,   7 },
        },
    },
    {
        { { 0 } },
        { { 0 } },
        {
            { 0 },
            {  80,   4,   2,   2 },
            {  81,   4,   2,   2 },
            {  81,   4,   2,   2 },
        },
        {
            { 0 },
            {  49,   2,   1,   3 },
            {  49,   2,   1,   3 },
            {  49,   2,   1,   3 },
        },
        {
            { 0 },
            { 225,   2,   1,   4 },
            { 225,   2,   1,   4 },
            { 225,   2,   1,   4 },
        },
        {
            { 0 },
            {  31,   1,   0,   5 },
            {  31,   1,   0,   5 },
            {  31,   1,   0,   5 },
        },
        {
            { 0 },
            {  63,   1,   0,   6 },
            {  63,   1,   0,   6 },
            {  63,   1,   0,   6 },
        },
        {
            { 0 },
            { 127,   1,   0,   7 },
            { 127,   1,   0,   7 },
            { 127,   1,   0,   7 },
        },
    },
};

#endif /* AVCODEC_ATRAC9TAB_H */
