% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TSPLIB.R
\name{TSPLIB}
\alias{TSPLIB}
\alias{read_TSPLIB}
\alias{write_TSPLIB}
\alias{write_TSPLIB.TSP}
\alias{write_TSPLIB.ATSP}
\alias{write_TSPLIB.ETSP}
\title{Read and write TSPLIB files}
\usage{
read_TSPLIB(file, precision = 0)

write_TSPLIB(x, file, precision = 6, inf = NULL, neg_inf = NULL)

\method{write_TSPLIB}{TSP}(x, file, precision = 6, inf = NULL, neg_inf = NULL)

\method{write_TSPLIB}{ATSP}(x, file, precision = 6, inf = NULL, neg_inf = NULL)

\method{write_TSPLIB}{ETSP}(x, file, precision = 6, inf = NULL, neg_inf = NULL)
}
\arguments{
\item{file}{file name or a \link{connection}.}

\item{precision}{controls the number of decimal places used to represent
distances (see details).  If \code{x} already is \code{integer}, this
argument is ignored and \code{x} is used as is.}

\item{x}{an object with a TSP problem.
\code{NA}s are not allowed.}

\item{inf}{replacement value for \code{Inf} (TSPLIB format cannot handle
\code{Inf}). If \code{inf} is \code{NULL}, a large value of \eqn{max(x) + 2
range(x)} (ignoring infinite entries) is used.}

\item{neg_inf}{replacement value for \code{-Inf}.  If no value is specified,
a small value of \eqn{min(x) - 2 range(x)} (ignoring infinite entries) is
used.}
}
\value{
returns an object of class \code{TSP} or
\code{ATSP}.
}
\description{
Reads and writes TSPLIB format files. TSPLIB files can be used by most TSP
solvers. Many problems in TSPLIB format can be found in the local
copy of the \href{https://github.com/mhahsler/TSP/tree/master/TSPLIB95}{TSPLIB95 problem library}.
}
\details{
In the TSPLIB format distances are represented by integer values. Therefore,
if \code{x} contains \code{double} values (which is normal in R) the values
given in \code{x} are multiplied by \eqn{10^{precision}} before coercion to
\code{integer}. Note that therefore all results produced by programs using
the TSPLIB file as input need to be divided by \eqn{10^{precision}} (i.e.,
the decimal point has to be shifted \code{precision} placed to the left).

Currently only the following \code{EDGE_WEIGHT_TYPE}s are implemented:
\code{EXPLICIT}, \code{EUC_2D} and \code{EUC_3D}.
}
\examples{

## Drilling problem from TSP
drill <- read_TSPLIB(system.file("examples/d493.tsp", package = "TSP"))
drill
tour <- solve_TSP(drill, method = "nn", two_opt = TRUE)
tour
plot(drill, tour, cex=.6, col = "red", pch= 3, main = "TSPLIB: d493")


## Write and read data in TSPLIB format
x <- data.frame(x=runif(5), y=runif(5))

## create TSP, ATSP and ETSP (2D)
tsp <- TSP(dist(x))
atsp <- ATSP(dist(x))
etsp <- ETSP(x[,1:2])

write_TSPLIB(tsp, file="example.tsp")
#file.show("example.tsp")
r <- read_TSPLIB("example.tsp")
r

write_TSPLIB(atsp, file="example.tsp")
#file.show("example.tsp")
r <- read_TSPLIB("example.tsp")
r

write_TSPLIB(etsp, file="example.tsp")
#file.show("example.tsp")
r <- read_TSPLIB("example.tsp")
r

## clean up
unlink("example.tsp")
}
\references{
Reinelt, Gerhard. 1991. “TSPLIB—a Traveling Salesman Problem Library.” ORSA Journal on Computing 3 (4): 376–84. \doi{10.1287/ijoc.3.4.376}
}
\seealso{
Other TSP: 
\code{\link{ATSP}()},
\code{\link{Concorde}},
\code{\link{ETSP}()},
\code{\link{TSP}()},
\code{\link{insert_dummy}()},
\code{\link{reformulate_ATSP_as_TSP}()},
\code{\link{solve_TSP}()}
}
\author{
Michael Hahsler
}
\concept{TSP}
\keyword{file}
