#![allow(unused, non_upper_case_globals, non_camel_case_types, missing_docs)]

use crate::common::{Error, Result};
use crate::CPUType;
use crate::Error::UnknownRegister;
use scroll::ctx::TryFromCtx;
use scroll::{Endian, LE};
use std::convert::{TryFrom, TryInto};
use std::fmt;

#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum Register {
    /// Register subset shared by all processor types,
    All(AllRegister),
    /// Register set for the Intel 80x86 and ix86 processor series (plus PCODE registers)
    X86(X86Register),
    /// Registers for the 68K processors
    M68K(M68KRegister),
    /// Register set for the MIPS 4000
    MIPS(MIPSRegister),
    /// Register set for the ALPHA AXP
    ALPHAAXP(ALPHAAXPRegister),
    /// Register Set for Motorola/IBM PowerPC
    PowerPC(PowerPCRegister),
    /// JAVA VM registers
    Java(JavaRegister),
    /// Register set for the Hitachi SH(3)
    HitachiSH(HitachiSHRegister),
    /// Register set for the ARM processor
    ARM(ARMRegister),
    /// Register set for ARM64
    ARM64(ARM64Register),
    /// Register set for Intel IA64
    IA64(IA64Register),
    /// Register set for the TriCore processor
    Tricore(TricoreRegister),
    /// Register set for the AM33 and related processors
    AM33(AM33Register),
    /// Register set for the Mitsubishi M32R
    MitsubishiM32R(MitsubishiM32RRegister),
    /// Register set for the SuperH SHMedia processor including compact mode
    SuperHSHMedia(SuperHSHMediaRegister),
    /// AMD64 registers
    AMD64(AMD64Register),
    /// HLSL registers
    HLSL(HLSLRegister),
}

impl fmt::Display for Register {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Register::All(r) => write!(f, "All({})", r),
            Register::X86(r) => write!(f, "X86({})", r),
            Register::M68K(r) => write!(f, "M68K({})", r),
            Register::MIPS(r) => write!(f, "MIPS({})", r),
            Register::ALPHAAXP(r) => write!(f, "ALPHAAXP({})", r),
            Register::PowerPC(r) => write!(f, "PowerPC({})", r),
            Register::Java(r) => write!(f, "Java({})", r),
            Register::HitachiSH(r) => write!(f, "HitachiSH({})", r),
            Register::ARM(r) => write!(f, "ARM({})", r),
            Register::ARM64(r) => write!(f, "ARM64({})", r),
            Register::IA64(r) => write!(f, "IA64({})", r),
            Register::Tricore(r) => write!(f, "Tricore({})", r),
            Register::AM33(r) => write!(f, "AM33({})", r),
            Register::MitsubishiM32R(r) => write!(f, "MitsubishiM32R({})", r),
            Register::SuperHSHMedia(r) => write!(f, "SuperHSHMedia({})", r),
            Register::AMD64(r) => write!(f, "AMD64({})", r),
            Register::HLSL(r) => write!(f, "HLSL({})", r),
            _ => Err(fmt::Error),
        }
    }
}

impl Register {
    pub fn new(value: crate::Register, cpu: crate::CPUType) -> Result<Self> {
        match cpu {
            CPUType::Intel8080
            | CPUType::Intel8086
            | CPUType::Intel80286
            | CPUType::Intel80386
            | CPUType::Intel80486
            | CPUType::Pentium
            | CPUType::PentiumPro
            | CPUType::Pentium3 => Ok(Self::X86(X86Register::try_from(value.0)?)),
            CPUType::M68000
            | CPUType::M68010
            | CPUType::M68020
            | CPUType::M68030
            | CPUType::M68040 => Ok(Self::M68K(M68KRegister::try_from(value.0)?)),
            CPUType::MIPS
            | CPUType::MIPS16
            | CPUType::MIPS32
            | CPUType::MIPS64
            | CPUType::MIPSI
            | CPUType::MIPSII
            | CPUType::MIPSIII
            | CPUType::MIPSIV
            | CPUType::MIPSV => Ok(Self::MIPS(MIPSRegister::try_from(value.0)?)),
            CPUType::Alpha
            | CPUType::Alpha21164
            | CPUType::Alpha21164A
            | CPUType::Alpha21264
            | CPUType::Alpha21364 => Ok(Self::ALPHAAXP(ALPHAAXPRegister::try_from(value.0)?)),
            CPUType::PPC601
            | CPUType::PPC603
            | CPUType::PPC604
            | CPUType::PPC620
            | CPUType::PPCFP
            | CPUType::PPCBE => Ok(Self::PowerPC(PowerPCRegister::try_from(value.0)?)),
            CPUType::SH3 | CPUType::SH3E | CPUType::SH3DSP | CPUType::SH4 => {
                Ok(Self::HitachiSH(HitachiSHRegister::try_from(value.0)?))
            }
            CPUType::SHMedia => Ok(Self::SuperHSHMedia(SuperHSHMediaRegister::try_from(
                value.0,
            )?)),
            CPUType::ARM3
            | CPUType::ARM4
            | CPUType::ARM4T
            | CPUType::ARM5
            | CPUType::ARM5T
            | CPUType::ARM6
            | CPUType::ARM_XMAC
            | CPUType::ARM_WMMX
            | CPUType::ARM7 => Ok(Self::ARM(ARMRegister::try_from(value.0)?)),
            CPUType::ARM64 => Ok(Self::ARM64(ARM64Register::try_from(value.0)?)),
            CPUType::Ia64 | CPUType::Ia64_2 => Ok(Self::IA64(IA64Register::try_from(value.0)?)),
            CPUType::AM33 => Ok(Self::AM33(AM33Register::try_from(value.0)?)),
            CPUType::M32R => Ok(Self::MitsubishiM32R(MitsubishiM32RRegister::try_from(
                value.0,
            )?)),
            CPUType::TriCore => Ok(Self::Tricore(TricoreRegister::try_from(value.0)?)),
            CPUType::X64 => Ok(Self::AMD64(AMD64Register::try_from(value.0)?)),
            CPUType::D3D11_Shader => Ok(Self::HLSL(HLSLRegister::try_from(value.0)?)),
            _ => Err(UnknownRegister(value.0)),
        }
    }
}

/// Register subset shared by all processor types,
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum AllRegister {
    Err = 30000,
    TEB = 30001,
    Timer = 30002,
    Efad1 = 30003,
    Efad2 = 30004,
    Efad3 = 30005,
    VFrame = 30006,
    Handle = 30007,
    Params = 30008,
    Locals = 30009,
    TID = 30010,
    Env = 30011,
    CmdLn = 30012,
}

impl fmt::Display for AllRegister {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::Err => write!(f, "Err"),
            Self::TEB => write!(f, "TEB"),
            Self::Timer => write!(f, "Timer"),
            Self::Efad1 => write!(f, "Efad1"),
            Self::Efad2 => write!(f, "Efad2"),
            Self::Efad3 => write!(f, "Efad3"),
            Self::VFrame => write!(f, "VFrame"),
            Self::Handle => write!(f, "Handle"),
            Self::Params => write!(f, "Params"),
            Self::Locals => write!(f, "Locals"),
            Self::TID => write!(f, "TID"),
            Self::Env => write!(f, "Env"),
            Self::CmdLn => write!(f, "CmdLn"),
        }
    }
}

impl TryFrom<u16> for AllRegister {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            30000 => Ok(Self::Err),
            30001 => Ok(Self::TEB),
            30002 => Ok(Self::Timer),
            30003 => Ok(Self::Efad1),
            30004 => Ok(Self::Efad2),
            30005 => Ok(Self::Efad3),
            30006 => Ok(Self::VFrame),
            30007 => Ok(Self::Handle),
            30008 => Ok(Self::Params),
            30009 => Ok(Self::Locals),
            30010 => Ok(Self::TID),
            30011 => Ok(Self::Env),
            30012 => Ok(Self::CmdLn),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for AllRegister {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// Register set for the Intel 80x86 and ix86 processor series (plus PCODE registers)
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum X86Register {
    AL = 1,
    CL = 2,
    DL = 3,
    BL = 4,
    AH = 5,
    CH = 6,
    DH = 7,
    BH = 8,
    AX = 9,
    CX = 10,
    DX = 11,
    BX = 12,
    SP = 13,
    BP = 14,
    SI = 15,
    DI = 16,
    EAX = 17,
    ECX = 18,
    EDX = 19,
    EBX = 20,
    ESP = 21,
    EBP = 22,
    ESI = 23,
    EDI = 24,
    ES = 25,
    CS = 26,
    SS = 27,
    DS = 28,
    FS = 29,
    GS = 30,
    IP = 31,
    FLAGS = 32,
    EIP = 33,
    EFLAGS = 34,
    TEMP = 40,  // PCODE Temp
    TEMPH = 41, // PCODE TempH
    QUOTE = 42, // PCODE Quote
    PCDR3 = 43, // PCODE reserved
    PCDR4 = 44, // PCODE reserved
    PCDR5 = 45, // PCODE reserved
    PCDR6 = 46, // PCODE reserved
    PCDR7 = 47, // PCODE reserved
    CR0 = 80,   // CR0 -- control registers
    CR1 = 81,
    CR2 = 82,
    CR3 = 83,
    CR4 = 84, // Pentium
    DR0 = 90, // Debug register
    DR1 = 91,
    DR2 = 92,
    DR3 = 93,
    DR4 = 94,
    DR5 = 95,
    DR6 = 96,
    DR7 = 97,
    GDTR = 110,
    GDTL = 111,
    IDTR = 112,
    IDTL = 113,
    LDTR = 114,
    TR = 115,

    PSEUDO1 = 116,
    PSEUDO2 = 117,
    PSEUDO3 = 118,
    PSEUDO4 = 119,
    PSEUDO5 = 120,
    PSEUDO6 = 121,
    PSEUDO7 = 122,
    PSEUDO8 = 123,
    PSEUDO9 = 124,

    ST0 = 128,
    ST1 = 129,
    ST2 = 130,
    ST3 = 131,
    ST4 = 132,
    ST5 = 133,
    ST6 = 134,
    ST7 = 135,
    CTRL = 136,
    STAT = 137,
    TAG = 138,
    FPIP = 139,
    FPCS = 140,
    FPDO = 141,
    FPDS = 142,
    ISEM = 143,
    FPEIP = 144,
    FPEDO = 145,

    MM0 = 146,
    MM1 = 147,
    MM2 = 148,
    MM3 = 149,
    MM4 = 150,
    MM5 = 151,
    MM6 = 152,
    MM7 = 153,

    XMM0 = 154, // KATMAI registers
    XMM1 = 155,
    XMM2 = 156,
    XMM3 = 157,
    XMM4 = 158,
    XMM5 = 159,
    XMM6 = 160,
    XMM7 = 161,

    XMM00 = 162, // KATMAI sub-registers
    XMM01 = 163,
    XMM02 = 164,
    XMM03 = 165,
    XMM10 = 166,
    XMM11 = 167,
    XMM12 = 168,
    XMM13 = 169,
    XMM20 = 170,
    XMM21 = 171,
    XMM22 = 172,
    XMM23 = 173,
    XMM30 = 174,
    XMM31 = 175,
    XMM32 = 176,
    XMM33 = 177,
    XMM40 = 178,
    XMM41 = 179,
    XMM42 = 180,
    XMM43 = 181,
    XMM50 = 182,
    XMM51 = 183,
    XMM52 = 184,
    XMM53 = 185,
    XMM60 = 186,
    XMM61 = 187,
    XMM62 = 188,
    XMM63 = 189,
    XMM70 = 190,
    XMM71 = 191,
    XMM72 = 192,
    XMM73 = 193,

    XMM0L = 194,
    XMM1L = 195,
    XMM2L = 196,
    XMM3L = 197,
    XMM4L = 198,
    XMM5L = 199,
    XMM6L = 200,
    XMM7L = 201,

    XMM0H = 202,
    XMM1H = 203,
    XMM2H = 204,
    XMM3H = 205,
    XMM4H = 206,
    XMM5H = 207,
    XMM6H = 208,
    XMM7H = 209,

    MXCSR = 211, // XMM status register

    EDXEAX = 212, // EDX:EAX pair

    EMM0L = 220, // XMM sub-registers (WNI integer)
    EMM1L = 221,
    EMM2L = 222,
    EMM3L = 223,
    EMM4L = 224,
    EMM5L = 225,
    EMM6L = 226,
    EMM7L = 227,

    EMM0H = 228,
    EMM1H = 229,
    EMM2H = 230,
    EMM3H = 231,
    EMM4H = 232,
    EMM5H = 233,
    EMM6H = 234,
    EMM7H = 235,

    // do not change the order of these regs, first one must be even too
    MM00 = 236,
    MM01 = 237,
    MM10 = 238,
    MM11 = 239,
    MM20 = 240,
    MM21 = 241,
    MM30 = 242,
    MM31 = 243,
    MM40 = 244,
    MM41 = 245,
    MM50 = 246,
    MM51 = 247,
    MM60 = 248,
    MM61 = 249,
    MM70 = 250,
    MM71 = 251,

    YMM0 = 252, // AVX registers
    YMM1 = 253,
    YMM2 = 254,
    YMM3 = 255,
    YMM4 = 256,
    YMM5 = 257,
    YMM6 = 258,
    YMM7 = 259,

    YMM0H = 260,
    YMM1H = 261,
    YMM2H = 262,
    YMM3H = 263,
    YMM4H = 264,
    YMM5H = 265,
    YMM6H = 266,
    YMM7H = 267,

    YMM0I0 = 268, // AVX integer registers
    YMM0I1 = 269,
    YMM0I2 = 270,
    YMM0I3 = 271,
    YMM1I0 = 272,
    YMM1I1 = 273,
    YMM1I2 = 274,
    YMM1I3 = 275,
    YMM2I0 = 276,
    YMM2I1 = 277,
    YMM2I2 = 278,
    YMM2I3 = 279,
    YMM3I0 = 280,
    YMM3I1 = 281,
    YMM3I2 = 282,
    YMM3I3 = 283,
    YMM4I0 = 284,
    YMM4I1 = 285,
    YMM4I2 = 286,
    YMM4I3 = 287,
    YMM5I0 = 288,
    YMM5I1 = 289,
    YMM5I2 = 290,
    YMM5I3 = 291,
    YMM6I0 = 292,
    YMM6I1 = 293,
    YMM6I2 = 294,
    YMM6I3 = 295,
    YMM7I0 = 296,
    YMM7I1 = 297,
    YMM7I2 = 298,
    YMM7I3 = 299,

    YMM0F0 = 300, // AVX floating-point single precise registers
    YMM0F1 = 301,
    YMM0F2 = 302,
    YMM0F3 = 303,
    YMM0F4 = 304,
    YMM0F5 = 305,
    YMM0F6 = 306,
    YMM0F7 = 307,
    YMM1F0 = 308,
    YMM1F1 = 309,
    YMM1F2 = 310,
    YMM1F3 = 311,
    YMM1F4 = 312,
    YMM1F5 = 313,
    YMM1F6 = 314,
    YMM1F7 = 315,
    YMM2F0 = 316,
    YMM2F1 = 317,
    YMM2F2 = 318,
    YMM2F3 = 319,
    YMM2F4 = 320,
    YMM2F5 = 321,
    YMM2F6 = 322,
    YMM2F7 = 323,
    YMM3F0 = 324,
    YMM3F1 = 325,
    YMM3F2 = 326,
    YMM3F3 = 327,
    YMM3F4 = 328,
    YMM3F5 = 329,
    YMM3F6 = 330,
    YMM3F7 = 331,
    YMM4F0 = 332,
    YMM4F1 = 333,
    YMM4F2 = 334,
    YMM4F3 = 335,
    YMM4F4 = 336,
    YMM4F5 = 337,
    YMM4F6 = 338,
    YMM4F7 = 339,
    YMM5F0 = 340,
    YMM5F1 = 341,
    YMM5F2 = 342,
    YMM5F3 = 343,
    YMM5F4 = 344,
    YMM5F5 = 345,
    YMM5F6 = 346,
    YMM5F7 = 347,
    YMM6F0 = 348,
    YMM6F1 = 349,
    YMM6F2 = 350,
    YMM6F3 = 351,
    YMM6F4 = 352,
    YMM6F5 = 353,
    YMM6F6 = 354,
    YMM6F7 = 355,
    YMM7F0 = 356,
    YMM7F1 = 357,
    YMM7F2 = 358,
    YMM7F3 = 359,
    YMM7F4 = 360,
    YMM7F5 = 361,
    YMM7F6 = 362,
    YMM7F7 = 363,

    YMM0D0 = 364, // AVX floating-point double precise registers
    YMM0D1 = 365,
    YMM0D2 = 366,
    YMM0D3 = 367,
    YMM1D0 = 368,
    YMM1D1 = 369,
    YMM1D2 = 370,
    YMM1D3 = 371,
    YMM2D0 = 372,
    YMM2D1 = 373,
    YMM2D2 = 374,
    YMM2D3 = 375,
    YMM3D0 = 376,
    YMM3D1 = 377,
    YMM3D2 = 378,
    YMM3D3 = 379,
    YMM4D0 = 380,
    YMM4D1 = 381,
    YMM4D2 = 382,
    YMM4D3 = 383,
    YMM5D0 = 384,
    YMM5D1 = 385,
    YMM5D2 = 386,
    YMM5D3 = 387,
    YMM6D0 = 388,
    YMM6D1 = 389,
    YMM6D2 = 390,
    YMM6D3 = 391,
    YMM7D0 = 392,
    YMM7D1 = 393,
    YMM7D2 = 394,
    YMM7D3 = 395,

    BND0 = 396,
    BND1 = 397,
    BND2 = 398,
    BND3 = 399,
}

impl fmt::Display for X86Register {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::AL => write!(f, "AL"),
            Self::CL => write!(f, "CL"),
            Self::DL => write!(f, "DL"),
            Self::BL => write!(f, "BL"),
            Self::AH => write!(f, "AH"),
            Self::CH => write!(f, "CH"),
            Self::DH => write!(f, "DH"),
            Self::BH => write!(f, "BH"),
            Self::AX => write!(f, "AX"),
            Self::CX => write!(f, "CX"),
            Self::DX => write!(f, "DX"),
            Self::BX => write!(f, "BX"),
            Self::SP => write!(f, "SP"),
            Self::BP => write!(f, "BP"),
            Self::SI => write!(f, "SI"),
            Self::DI => write!(f, "DI"),
            Self::EAX => write!(f, "EAX"),
            Self::ECX => write!(f, "ECX"),
            Self::EDX => write!(f, "EDX"),
            Self::EBX => write!(f, "EBX"),
            Self::ESP => write!(f, "ESP"),
            Self::EBP => write!(f, "EBP"),
            Self::ESI => write!(f, "ESI"),
            Self::EDI => write!(f, "EDI"),
            Self::ES => write!(f, "ES"),
            Self::CS => write!(f, "CS"),
            Self::SS => write!(f, "SS"),
            Self::DS => write!(f, "DS"),
            Self::FS => write!(f, "FS"),
            Self::GS => write!(f, "GS"),
            Self::IP => write!(f, "IP"),
            Self::FLAGS => write!(f, "FLAGS"),
            Self::EIP => write!(f, "EIP"),
            Self::EFLAGS => write!(f, "EFLAGS"),
            Self::TEMP => write!(f, "TEMP"),
            Self::TEMPH => write!(f, "TEMPH"),
            Self::QUOTE => write!(f, "QUOTE"),
            Self::PCDR3 => write!(f, "PCDR3"),
            Self::PCDR4 => write!(f, "PCDR4"),
            Self::PCDR5 => write!(f, "PCDR5"),
            Self::PCDR6 => write!(f, "PCDR6"),
            Self::PCDR7 => write!(f, "PCDR7"),
            Self::CR0 => write!(f, "CR0"),
            Self::CR1 => write!(f, "CR1"),
            Self::CR2 => write!(f, "CR2"),
            Self::CR3 => write!(f, "CR3"),
            Self::CR4 => write!(f, "CR4"),
            Self::DR0 => write!(f, "DR0"),
            Self::DR1 => write!(f, "DR1"),
            Self::DR2 => write!(f, "DR2"),
            Self::DR3 => write!(f, "DR3"),
            Self::DR4 => write!(f, "DR4"),
            Self::DR5 => write!(f, "DR5"),
            Self::DR6 => write!(f, "DR6"),
            Self::DR7 => write!(f, "DR7"),
            Self::GDTR => write!(f, "GDTR"),
            Self::GDTL => write!(f, "GDTL"),
            Self::IDTR => write!(f, "IDTR"),
            Self::IDTL => write!(f, "IDTL"),
            Self::LDTR => write!(f, "LDTR"),
            Self::TR => write!(f, "TR"),
            Self::PSEUDO1 => write!(f, "PSEUDO1"),
            Self::PSEUDO2 => write!(f, "PSEUDO2"),
            Self::PSEUDO3 => write!(f, "PSEUDO3"),
            Self::PSEUDO4 => write!(f, "PSEUDO4"),
            Self::PSEUDO5 => write!(f, "PSEUDO5"),
            Self::PSEUDO6 => write!(f, "PSEUDO6"),
            Self::PSEUDO7 => write!(f, "PSEUDO7"),
            Self::PSEUDO8 => write!(f, "PSEUDO8"),
            Self::PSEUDO9 => write!(f, "PSEUDO9"),
            Self::ST0 => write!(f, "ST0"),
            Self::ST1 => write!(f, "ST1"),
            Self::ST2 => write!(f, "ST2"),
            Self::ST3 => write!(f, "ST3"),
            Self::ST4 => write!(f, "ST4"),
            Self::ST5 => write!(f, "ST5"),
            Self::ST6 => write!(f, "ST6"),
            Self::ST7 => write!(f, "ST7"),
            Self::CTRL => write!(f, "CTRL"),
            Self::STAT => write!(f, "STAT"),
            Self::TAG => write!(f, "TAG"),
            Self::FPIP => write!(f, "FPIP"),
            Self::FPCS => write!(f, "FPCS"),
            Self::FPDO => write!(f, "FPDO"),
            Self::FPDS => write!(f, "FPDS"),
            Self::ISEM => write!(f, "ISEM"),
            Self::FPEIP => write!(f, "FPEIP"),
            Self::FPEDO => write!(f, "FPEDO"),
            Self::MM0 => write!(f, "MM0"),
            Self::MM1 => write!(f, "MM1"),
            Self::MM2 => write!(f, "MM2"),
            Self::MM3 => write!(f, "MM3"),
            Self::MM4 => write!(f, "MM4"),
            Self::MM5 => write!(f, "MM5"),
            Self::MM6 => write!(f, "MM6"),
            Self::MM7 => write!(f, "MM7"),
            Self::XMM0 => write!(f, "XMM0"),
            Self::XMM1 => write!(f, "XMM1"),
            Self::XMM2 => write!(f, "XMM2"),
            Self::XMM3 => write!(f, "XMM3"),
            Self::XMM4 => write!(f, "XMM4"),
            Self::XMM5 => write!(f, "XMM5"),
            Self::XMM6 => write!(f, "XMM6"),
            Self::XMM7 => write!(f, "XMM7"),
            Self::XMM00 => write!(f, "XMM00"),
            Self::XMM01 => write!(f, "XMM01"),
            Self::XMM02 => write!(f, "XMM02"),
            Self::XMM03 => write!(f, "XMM03"),
            Self::XMM10 => write!(f, "XMM10"),
            Self::XMM11 => write!(f, "XMM11"),
            Self::XMM12 => write!(f, "XMM12"),
            Self::XMM13 => write!(f, "XMM13"),
            Self::XMM20 => write!(f, "XMM20"),
            Self::XMM21 => write!(f, "XMM21"),
            Self::XMM22 => write!(f, "XMM22"),
            Self::XMM23 => write!(f, "XMM23"),
            Self::XMM30 => write!(f, "XMM30"),
            Self::XMM31 => write!(f, "XMM31"),
            Self::XMM32 => write!(f, "XMM32"),
            Self::XMM33 => write!(f, "XMM33"),
            Self::XMM40 => write!(f, "XMM40"),
            Self::XMM41 => write!(f, "XMM41"),
            Self::XMM42 => write!(f, "XMM42"),
            Self::XMM43 => write!(f, "XMM43"),
            Self::XMM50 => write!(f, "XMM50"),
            Self::XMM51 => write!(f, "XMM51"),
            Self::XMM52 => write!(f, "XMM52"),
            Self::XMM53 => write!(f, "XMM53"),
            Self::XMM60 => write!(f, "XMM60"),
            Self::XMM61 => write!(f, "XMM61"),
            Self::XMM62 => write!(f, "XMM62"),
            Self::XMM63 => write!(f, "XMM63"),
            Self::XMM70 => write!(f, "XMM70"),
            Self::XMM71 => write!(f, "XMM71"),
            Self::XMM72 => write!(f, "XMM72"),
            Self::XMM73 => write!(f, "XMM73"),
            Self::XMM0L => write!(f, "XMM0L"),
            Self::XMM1L => write!(f, "XMM1L"),
            Self::XMM2L => write!(f, "XMM2L"),
            Self::XMM3L => write!(f, "XMM3L"),
            Self::XMM4L => write!(f, "XMM4L"),
            Self::XMM5L => write!(f, "XMM5L"),
            Self::XMM6L => write!(f, "XMM6L"),
            Self::XMM7L => write!(f, "XMM7L"),
            Self::XMM0H => write!(f, "XMM0H"),
            Self::XMM1H => write!(f, "XMM1H"),
            Self::XMM2H => write!(f, "XMM2H"),
            Self::XMM3H => write!(f, "XMM3H"),
            Self::XMM4H => write!(f, "XMM4H"),
            Self::XMM5H => write!(f, "XMM5H"),
            Self::XMM6H => write!(f, "XMM6H"),
            Self::XMM7H => write!(f, "XMM7H"),
            Self::MXCSR => write!(f, "MXCSR"),
            Self::EDXEAX => write!(f, "EDXEAX"),
            Self::EMM0L => write!(f, "EMM0L"),
            Self::EMM1L => write!(f, "EMM1L"),
            Self::EMM2L => write!(f, "EMM2L"),
            Self::EMM3L => write!(f, "EMM3L"),
            Self::EMM4L => write!(f, "EMM4L"),
            Self::EMM5L => write!(f, "EMM5L"),
            Self::EMM6L => write!(f, "EMM6L"),
            Self::EMM7L => write!(f, "EMM7L"),
            Self::EMM0H => write!(f, "EMM0H"),
            Self::EMM1H => write!(f, "EMM1H"),
            Self::EMM2H => write!(f, "EMM2H"),
            Self::EMM3H => write!(f, "EMM3H"),
            Self::EMM4H => write!(f, "EMM4H"),
            Self::EMM5H => write!(f, "EMM5H"),
            Self::EMM6H => write!(f, "EMM6H"),
            Self::EMM7H => write!(f, "EMM7H"),
            Self::MM00 => write!(f, "MM00"),
            Self::MM01 => write!(f, "MM01"),
            Self::MM10 => write!(f, "MM10"),
            Self::MM11 => write!(f, "MM11"),
            Self::MM20 => write!(f, "MM20"),
            Self::MM21 => write!(f, "MM21"),
            Self::MM30 => write!(f, "MM30"),
            Self::MM31 => write!(f, "MM31"),
            Self::MM40 => write!(f, "MM40"),
            Self::MM41 => write!(f, "MM41"),
            Self::MM50 => write!(f, "MM50"),
            Self::MM51 => write!(f, "MM51"),
            Self::MM60 => write!(f, "MM60"),
            Self::MM61 => write!(f, "MM61"),
            Self::MM70 => write!(f, "MM70"),
            Self::MM71 => write!(f, "MM71"),
            Self::YMM0 => write!(f, "YMM0"),
            Self::YMM1 => write!(f, "YMM1"),
            Self::YMM2 => write!(f, "YMM2"),
            Self::YMM3 => write!(f, "YMM3"),
            Self::YMM4 => write!(f, "YMM4"),
            Self::YMM5 => write!(f, "YMM5"),
            Self::YMM6 => write!(f, "YMM6"),
            Self::YMM7 => write!(f, "YMM7"),
            Self::YMM0H => write!(f, "YMM0H"),
            Self::YMM1H => write!(f, "YMM1H"),
            Self::YMM2H => write!(f, "YMM2H"),
            Self::YMM3H => write!(f, "YMM3H"),
            Self::YMM4H => write!(f, "YMM4H"),
            Self::YMM5H => write!(f, "YMM5H"),
            Self::YMM6H => write!(f, "YMM6H"),
            Self::YMM7H => write!(f, "YMM7H"),
            Self::YMM0I0 => write!(f, "YMM0I0"),
            Self::YMM0I1 => write!(f, "YMM0I1"),
            Self::YMM0I2 => write!(f, "YMM0I2"),
            Self::YMM0I3 => write!(f, "YMM0I3"),
            Self::YMM1I0 => write!(f, "YMM1I0"),
            Self::YMM1I1 => write!(f, "YMM1I1"),
            Self::YMM1I2 => write!(f, "YMM1I2"),
            Self::YMM1I3 => write!(f, "YMM1I3"),
            Self::YMM2I0 => write!(f, "YMM2I0"),
            Self::YMM2I1 => write!(f, "YMM2I1"),
            Self::YMM2I2 => write!(f, "YMM2I2"),
            Self::YMM2I3 => write!(f, "YMM2I3"),
            Self::YMM3I0 => write!(f, "YMM3I0"),
            Self::YMM3I1 => write!(f, "YMM3I1"),
            Self::YMM3I2 => write!(f, "YMM3I2"),
            Self::YMM3I3 => write!(f, "YMM3I3"),
            Self::YMM4I0 => write!(f, "YMM4I0"),
            Self::YMM4I1 => write!(f, "YMM4I1"),
            Self::YMM4I2 => write!(f, "YMM4I2"),
            Self::YMM4I3 => write!(f, "YMM4I3"),
            Self::YMM5I0 => write!(f, "YMM5I0"),
            Self::YMM5I1 => write!(f, "YMM5I1"),
            Self::YMM5I2 => write!(f, "YMM5I2"),
            Self::YMM5I3 => write!(f, "YMM5I3"),
            Self::YMM6I0 => write!(f, "YMM6I0"),
            Self::YMM6I1 => write!(f, "YMM6I1"),
            Self::YMM6I2 => write!(f, "YMM6I2"),
            Self::YMM6I3 => write!(f, "YMM6I3"),
            Self::YMM7I0 => write!(f, "YMM7I0"),
            Self::YMM7I1 => write!(f, "YMM7I1"),
            Self::YMM7I2 => write!(f, "YMM7I2"),
            Self::YMM7I3 => write!(f, "YMM7I3"),
            Self::YMM0F0 => write!(f, "YMM0F0"),
            Self::YMM0F1 => write!(f, "YMM0F1"),
            Self::YMM0F2 => write!(f, "YMM0F2"),
            Self::YMM0F3 => write!(f, "YMM0F3"),
            Self::YMM0F4 => write!(f, "YMM0F4"),
            Self::YMM0F5 => write!(f, "YMM0F5"),
            Self::YMM0F6 => write!(f, "YMM0F6"),
            Self::YMM0F7 => write!(f, "YMM0F7"),
            Self::YMM1F0 => write!(f, "YMM1F0"),
            Self::YMM1F1 => write!(f, "YMM1F1"),
            Self::YMM1F2 => write!(f, "YMM1F2"),
            Self::YMM1F3 => write!(f, "YMM1F3"),
            Self::YMM1F4 => write!(f, "YMM1F4"),
            Self::YMM1F5 => write!(f, "YMM1F5"),
            Self::YMM1F6 => write!(f, "YMM1F6"),
            Self::YMM1F7 => write!(f, "YMM1F7"),
            Self::YMM2F0 => write!(f, "YMM2F0"),
            Self::YMM2F1 => write!(f, "YMM2F1"),
            Self::YMM2F2 => write!(f, "YMM2F2"),
            Self::YMM2F3 => write!(f, "YMM2F3"),
            Self::YMM2F4 => write!(f, "YMM2F4"),
            Self::YMM2F5 => write!(f, "YMM2F5"),
            Self::YMM2F6 => write!(f, "YMM2F6"),
            Self::YMM2F7 => write!(f, "YMM2F7"),
            Self::YMM3F0 => write!(f, "YMM3F0"),
            Self::YMM3F1 => write!(f, "YMM3F1"),
            Self::YMM3F2 => write!(f, "YMM3F2"),
            Self::YMM3F3 => write!(f, "YMM3F3"),
            Self::YMM3F4 => write!(f, "YMM3F4"),
            Self::YMM3F5 => write!(f, "YMM3F5"),
            Self::YMM3F6 => write!(f, "YMM3F6"),
            Self::YMM3F7 => write!(f, "YMM3F7"),
            Self::YMM4F0 => write!(f, "YMM4F0"),
            Self::YMM4F1 => write!(f, "YMM4F1"),
            Self::YMM4F2 => write!(f, "YMM4F2"),
            Self::YMM4F3 => write!(f, "YMM4F3"),
            Self::YMM4F4 => write!(f, "YMM4F4"),
            Self::YMM4F5 => write!(f, "YMM4F5"),
            Self::YMM4F6 => write!(f, "YMM4F6"),
            Self::YMM4F7 => write!(f, "YMM4F7"),
            Self::YMM5F0 => write!(f, "YMM5F0"),
            Self::YMM5F1 => write!(f, "YMM5F1"),
            Self::YMM5F2 => write!(f, "YMM5F2"),
            Self::YMM5F3 => write!(f, "YMM5F3"),
            Self::YMM5F4 => write!(f, "YMM5F4"),
            Self::YMM5F5 => write!(f, "YMM5F5"),
            Self::YMM5F6 => write!(f, "YMM5F6"),
            Self::YMM5F7 => write!(f, "YMM5F7"),
            Self::YMM6F0 => write!(f, "YMM6F0"),
            Self::YMM6F1 => write!(f, "YMM6F1"),
            Self::YMM6F2 => write!(f, "YMM6F2"),
            Self::YMM6F3 => write!(f, "YMM6F3"),
            Self::YMM6F4 => write!(f, "YMM6F4"),
            Self::YMM6F5 => write!(f, "YMM6F5"),
            Self::YMM6F6 => write!(f, "YMM6F6"),
            Self::YMM6F7 => write!(f, "YMM6F7"),
            Self::YMM7F0 => write!(f, "YMM7F0"),
            Self::YMM7F1 => write!(f, "YMM7F1"),
            Self::YMM7F2 => write!(f, "YMM7F2"),
            Self::YMM7F3 => write!(f, "YMM7F3"),
            Self::YMM7F4 => write!(f, "YMM7F4"),
            Self::YMM7F5 => write!(f, "YMM7F5"),
            Self::YMM7F6 => write!(f, "YMM7F6"),
            Self::YMM7F7 => write!(f, "YMM7F7"),
            Self::YMM0D0 => write!(f, "YMM0D0"),
            Self::YMM0D1 => write!(f, "YMM0D1"),
            Self::YMM0D2 => write!(f, "YMM0D2"),
            Self::YMM0D3 => write!(f, "YMM0D3"),
            Self::YMM1D0 => write!(f, "YMM1D0"),
            Self::YMM1D1 => write!(f, "YMM1D1"),
            Self::YMM1D2 => write!(f, "YMM1D2"),
            Self::YMM1D3 => write!(f, "YMM1D3"),
            Self::YMM2D0 => write!(f, "YMM2D0"),
            Self::YMM2D1 => write!(f, "YMM2D1"),
            Self::YMM2D2 => write!(f, "YMM2D2"),
            Self::YMM2D3 => write!(f, "YMM2D3"),
            Self::YMM3D0 => write!(f, "YMM3D0"),
            Self::YMM3D1 => write!(f, "YMM3D1"),
            Self::YMM3D2 => write!(f, "YMM3D2"),
            Self::YMM3D3 => write!(f, "YMM3D3"),
            Self::YMM4D0 => write!(f, "YMM4D0"),
            Self::YMM4D1 => write!(f, "YMM4D1"),
            Self::YMM4D2 => write!(f, "YMM4D2"),
            Self::YMM4D3 => write!(f, "YMM4D3"),
            Self::YMM5D0 => write!(f, "YMM5D0"),
            Self::YMM5D1 => write!(f, "YMM5D1"),
            Self::YMM5D2 => write!(f, "YMM5D2"),
            Self::YMM5D3 => write!(f, "YMM5D3"),
            Self::YMM6D0 => write!(f, "YMM6D0"),
            Self::YMM6D1 => write!(f, "YMM6D1"),
            Self::YMM6D2 => write!(f, "YMM6D2"),
            Self::YMM6D3 => write!(f, "YMM6D3"),
            Self::YMM7D0 => write!(f, "YMM7D0"),
            Self::YMM7D1 => write!(f, "YMM7D1"),
            Self::YMM7D2 => write!(f, "YMM7D2"),
            Self::YMM7D3 => write!(f, "YMM7D3"),
            Self::BND0 => write!(f, "BND0"),
            Self::BND1 => write!(f, "BND1"),
            Self::BND2 => write!(f, "BND2"),
            Self::BND3 => write!(f, "BND3"),
        }
    }
}

impl TryFrom<u16> for X86Register {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            1 => Ok(Self::AL),
            2 => Ok(Self::CL),
            3 => Ok(Self::DL),
            4 => Ok(Self::BL),
            5 => Ok(Self::AH),
            6 => Ok(Self::CH),
            7 => Ok(Self::DH),
            8 => Ok(Self::BH),
            9 => Ok(Self::AX),
            10 => Ok(Self::CX),
            11 => Ok(Self::DX),
            12 => Ok(Self::BX),
            13 => Ok(Self::SP),
            14 => Ok(Self::BP),
            15 => Ok(Self::SI),
            16 => Ok(Self::DI),
            17 => Ok(Self::EAX),
            18 => Ok(Self::ECX),
            19 => Ok(Self::EDX),
            20 => Ok(Self::EBX),
            21 => Ok(Self::ESP),
            22 => Ok(Self::EBP),
            23 => Ok(Self::ESI),
            24 => Ok(Self::EDI),
            25 => Ok(Self::ES),
            26 => Ok(Self::CS),
            27 => Ok(Self::SS),
            28 => Ok(Self::DS),
            29 => Ok(Self::FS),
            30 => Ok(Self::GS),
            31 => Ok(Self::IP),
            32 => Ok(Self::FLAGS),
            33 => Ok(Self::EIP),
            34 => Ok(Self::EFLAGS),
            40 => Ok(Self::TEMP),
            41 => Ok(Self::TEMPH),
            42 => Ok(Self::QUOTE),
            43 => Ok(Self::PCDR3),
            44 => Ok(Self::PCDR4),
            45 => Ok(Self::PCDR5),
            46 => Ok(Self::PCDR6),
            47 => Ok(Self::PCDR7),
            80 => Ok(Self::CR0),
            81 => Ok(Self::CR1),
            82 => Ok(Self::CR2),
            83 => Ok(Self::CR3),
            84 => Ok(Self::CR4),
            90 => Ok(Self::DR0),
            91 => Ok(Self::DR1),
            92 => Ok(Self::DR2),
            93 => Ok(Self::DR3),
            94 => Ok(Self::DR4),
            95 => Ok(Self::DR5),
            96 => Ok(Self::DR6),
            97 => Ok(Self::DR7),
            110 => Ok(Self::GDTR),
            111 => Ok(Self::GDTL),
            112 => Ok(Self::IDTR),
            113 => Ok(Self::IDTL),
            114 => Ok(Self::LDTR),
            115 => Ok(Self::TR),
            116 => Ok(Self::PSEUDO1),
            117 => Ok(Self::PSEUDO2),
            118 => Ok(Self::PSEUDO3),
            119 => Ok(Self::PSEUDO4),
            120 => Ok(Self::PSEUDO5),
            121 => Ok(Self::PSEUDO6),
            122 => Ok(Self::PSEUDO7),
            123 => Ok(Self::PSEUDO8),
            124 => Ok(Self::PSEUDO9),
            128 => Ok(Self::ST0),
            129 => Ok(Self::ST1),
            130 => Ok(Self::ST2),
            131 => Ok(Self::ST3),
            132 => Ok(Self::ST4),
            133 => Ok(Self::ST5),
            134 => Ok(Self::ST6),
            135 => Ok(Self::ST7),
            136 => Ok(Self::CTRL),
            137 => Ok(Self::STAT),
            138 => Ok(Self::TAG),
            139 => Ok(Self::FPIP),
            140 => Ok(Self::FPCS),
            141 => Ok(Self::FPDO),
            142 => Ok(Self::FPDS),
            143 => Ok(Self::ISEM),
            144 => Ok(Self::FPEIP),
            145 => Ok(Self::FPEDO),
            146 => Ok(Self::MM0),
            147 => Ok(Self::MM1),
            148 => Ok(Self::MM2),
            149 => Ok(Self::MM3),
            150 => Ok(Self::MM4),
            151 => Ok(Self::MM5),
            152 => Ok(Self::MM6),
            153 => Ok(Self::MM7),
            154 => Ok(Self::XMM0),
            155 => Ok(Self::XMM1),
            156 => Ok(Self::XMM2),
            157 => Ok(Self::XMM3),
            158 => Ok(Self::XMM4),
            159 => Ok(Self::XMM5),
            160 => Ok(Self::XMM6),
            161 => Ok(Self::XMM7),
            162 => Ok(Self::XMM00),
            163 => Ok(Self::XMM01),
            164 => Ok(Self::XMM02),
            165 => Ok(Self::XMM03),
            166 => Ok(Self::XMM10),
            167 => Ok(Self::XMM11),
            168 => Ok(Self::XMM12),
            169 => Ok(Self::XMM13),
            170 => Ok(Self::XMM20),
            171 => Ok(Self::XMM21),
            172 => Ok(Self::XMM22),
            173 => Ok(Self::XMM23),
            174 => Ok(Self::XMM30),
            175 => Ok(Self::XMM31),
            176 => Ok(Self::XMM32),
            177 => Ok(Self::XMM33),
            178 => Ok(Self::XMM40),
            179 => Ok(Self::XMM41),
            180 => Ok(Self::XMM42),
            181 => Ok(Self::XMM43),
            182 => Ok(Self::XMM50),
            183 => Ok(Self::XMM51),
            184 => Ok(Self::XMM52),
            185 => Ok(Self::XMM53),
            186 => Ok(Self::XMM60),
            187 => Ok(Self::XMM61),
            188 => Ok(Self::XMM62),
            189 => Ok(Self::XMM63),
            190 => Ok(Self::XMM70),
            191 => Ok(Self::XMM71),
            192 => Ok(Self::XMM72),
            193 => Ok(Self::XMM73),
            194 => Ok(Self::XMM0L),
            195 => Ok(Self::XMM1L),
            196 => Ok(Self::XMM2L),
            197 => Ok(Self::XMM3L),
            198 => Ok(Self::XMM4L),
            199 => Ok(Self::XMM5L),
            200 => Ok(Self::XMM6L),
            201 => Ok(Self::XMM7L),
            202 => Ok(Self::XMM0H),
            203 => Ok(Self::XMM1H),
            204 => Ok(Self::XMM2H),
            205 => Ok(Self::XMM3H),
            206 => Ok(Self::XMM4H),
            207 => Ok(Self::XMM5H),
            208 => Ok(Self::XMM6H),
            209 => Ok(Self::XMM7H),
            211 => Ok(Self::MXCSR),
            212 => Ok(Self::EDXEAX),
            220 => Ok(Self::EMM0L),
            221 => Ok(Self::EMM1L),
            222 => Ok(Self::EMM2L),
            223 => Ok(Self::EMM3L),
            224 => Ok(Self::EMM4L),
            225 => Ok(Self::EMM5L),
            226 => Ok(Self::EMM6L),
            227 => Ok(Self::EMM7L),
            228 => Ok(Self::EMM0H),
            229 => Ok(Self::EMM1H),
            230 => Ok(Self::EMM2H),
            231 => Ok(Self::EMM3H),
            232 => Ok(Self::EMM4H),
            233 => Ok(Self::EMM5H),
            234 => Ok(Self::EMM6H),
            235 => Ok(Self::EMM7H),
            236 => Ok(Self::MM00),
            237 => Ok(Self::MM01),
            238 => Ok(Self::MM10),
            239 => Ok(Self::MM11),
            240 => Ok(Self::MM20),
            241 => Ok(Self::MM21),
            242 => Ok(Self::MM30),
            243 => Ok(Self::MM31),
            244 => Ok(Self::MM40),
            245 => Ok(Self::MM41),
            246 => Ok(Self::MM50),
            247 => Ok(Self::MM51),
            248 => Ok(Self::MM60),
            249 => Ok(Self::MM61),
            250 => Ok(Self::MM70),
            251 => Ok(Self::MM71),
            252 => Ok(Self::YMM0),
            253 => Ok(Self::YMM1),
            254 => Ok(Self::YMM2),
            255 => Ok(Self::YMM3),
            256 => Ok(Self::YMM4),
            257 => Ok(Self::YMM5),
            258 => Ok(Self::YMM6),
            259 => Ok(Self::YMM7),
            260 => Ok(Self::YMM0H),
            261 => Ok(Self::YMM1H),
            262 => Ok(Self::YMM2H),
            263 => Ok(Self::YMM3H),
            264 => Ok(Self::YMM4H),
            265 => Ok(Self::YMM5H),
            266 => Ok(Self::YMM6H),
            267 => Ok(Self::YMM7H),
            268 => Ok(Self::YMM0I0),
            269 => Ok(Self::YMM0I1),
            270 => Ok(Self::YMM0I2),
            271 => Ok(Self::YMM0I3),
            272 => Ok(Self::YMM1I0),
            273 => Ok(Self::YMM1I1),
            274 => Ok(Self::YMM1I2),
            275 => Ok(Self::YMM1I3),
            276 => Ok(Self::YMM2I0),
            277 => Ok(Self::YMM2I1),
            278 => Ok(Self::YMM2I2),
            279 => Ok(Self::YMM2I3),
            280 => Ok(Self::YMM3I0),
            281 => Ok(Self::YMM3I1),
            282 => Ok(Self::YMM3I2),
            283 => Ok(Self::YMM3I3),
            284 => Ok(Self::YMM4I0),
            285 => Ok(Self::YMM4I1),
            286 => Ok(Self::YMM4I2),
            287 => Ok(Self::YMM4I3),
            288 => Ok(Self::YMM5I0),
            289 => Ok(Self::YMM5I1),
            290 => Ok(Self::YMM5I2),
            291 => Ok(Self::YMM5I3),
            292 => Ok(Self::YMM6I0),
            293 => Ok(Self::YMM6I1),
            294 => Ok(Self::YMM6I2),
            295 => Ok(Self::YMM6I3),
            296 => Ok(Self::YMM7I0),
            297 => Ok(Self::YMM7I1),
            298 => Ok(Self::YMM7I2),
            299 => Ok(Self::YMM7I3),
            300 => Ok(Self::YMM0F0),
            301 => Ok(Self::YMM0F1),
            302 => Ok(Self::YMM0F2),
            303 => Ok(Self::YMM0F3),
            304 => Ok(Self::YMM0F4),
            305 => Ok(Self::YMM0F5),
            306 => Ok(Self::YMM0F6),
            307 => Ok(Self::YMM0F7),
            308 => Ok(Self::YMM1F0),
            309 => Ok(Self::YMM1F1),
            310 => Ok(Self::YMM1F2),
            311 => Ok(Self::YMM1F3),
            312 => Ok(Self::YMM1F4),
            313 => Ok(Self::YMM1F5),
            314 => Ok(Self::YMM1F6),
            315 => Ok(Self::YMM1F7),
            316 => Ok(Self::YMM2F0),
            317 => Ok(Self::YMM2F1),
            318 => Ok(Self::YMM2F2),
            319 => Ok(Self::YMM2F3),
            320 => Ok(Self::YMM2F4),
            321 => Ok(Self::YMM2F5),
            322 => Ok(Self::YMM2F6),
            323 => Ok(Self::YMM2F7),
            324 => Ok(Self::YMM3F0),
            325 => Ok(Self::YMM3F1),
            326 => Ok(Self::YMM3F2),
            327 => Ok(Self::YMM3F3),
            328 => Ok(Self::YMM3F4),
            329 => Ok(Self::YMM3F5),
            330 => Ok(Self::YMM3F6),
            331 => Ok(Self::YMM3F7),
            332 => Ok(Self::YMM4F0),
            333 => Ok(Self::YMM4F1),
            334 => Ok(Self::YMM4F2),
            335 => Ok(Self::YMM4F3),
            336 => Ok(Self::YMM4F4),
            337 => Ok(Self::YMM4F5),
            338 => Ok(Self::YMM4F6),
            339 => Ok(Self::YMM4F7),
            340 => Ok(Self::YMM5F0),
            341 => Ok(Self::YMM5F1),
            342 => Ok(Self::YMM5F2),
            343 => Ok(Self::YMM5F3),
            344 => Ok(Self::YMM5F4),
            345 => Ok(Self::YMM5F5),
            346 => Ok(Self::YMM5F6),
            347 => Ok(Self::YMM5F7),
            348 => Ok(Self::YMM6F0),
            349 => Ok(Self::YMM6F1),
            350 => Ok(Self::YMM6F2),
            351 => Ok(Self::YMM6F3),
            352 => Ok(Self::YMM6F4),
            353 => Ok(Self::YMM6F5),
            354 => Ok(Self::YMM6F6),
            355 => Ok(Self::YMM6F7),
            356 => Ok(Self::YMM7F0),
            357 => Ok(Self::YMM7F1),
            358 => Ok(Self::YMM7F2),
            359 => Ok(Self::YMM7F3),
            360 => Ok(Self::YMM7F4),
            361 => Ok(Self::YMM7F5),
            362 => Ok(Self::YMM7F6),
            363 => Ok(Self::YMM7F7),
            364 => Ok(Self::YMM0D0),
            365 => Ok(Self::YMM0D1),
            366 => Ok(Self::YMM0D2),
            367 => Ok(Self::YMM0D3),
            368 => Ok(Self::YMM1D0),
            369 => Ok(Self::YMM1D1),
            370 => Ok(Self::YMM1D2),
            371 => Ok(Self::YMM1D3),
            372 => Ok(Self::YMM2D0),
            373 => Ok(Self::YMM2D1),
            374 => Ok(Self::YMM2D2),
            375 => Ok(Self::YMM2D3),
            376 => Ok(Self::YMM3D0),
            377 => Ok(Self::YMM3D1),
            378 => Ok(Self::YMM3D2),
            379 => Ok(Self::YMM3D3),
            380 => Ok(Self::YMM4D0),
            381 => Ok(Self::YMM4D1),
            382 => Ok(Self::YMM4D2),
            383 => Ok(Self::YMM4D3),
            384 => Ok(Self::YMM5D0),
            385 => Ok(Self::YMM5D1),
            386 => Ok(Self::YMM5D2),
            387 => Ok(Self::YMM5D3),
            388 => Ok(Self::YMM6D0),
            389 => Ok(Self::YMM6D1),
            390 => Ok(Self::YMM6D2),
            391 => Ok(Self::YMM6D3),
            392 => Ok(Self::YMM7D0),
            393 => Ok(Self::YMM7D1),
            394 => Ok(Self::YMM7D2),
            395 => Ok(Self::YMM7D3),
            396 => Ok(Self::BND0),
            397 => Ok(Self::BND1),
            398 => Ok(Self::BND2),
            399 => Ok(Self::BND3),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for X86Register {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// Registers for the 68K processors
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum M68KRegister {
    D0 = 0,
    D1 = 1,
    D2 = 2,
    D3 = 3,
    D4 = 4,
    D5 = 5,
    D6 = 6,
    D7 = 7,
    A0 = 8,
    A1 = 9,
    A2 = 10,
    A3 = 11,
    A4 = 12,
    A5 = 13,
    A6 = 14,
    A7 = 15,
    CCR = 16,
    SR = 17,
    USP = 18,
    MSP = 19,
    SFC = 20,
    DFC = 21,
    CACR = 22,
    VBR = 23,
    CAAR = 24,
    ISP = 25,
    PC = 26,
    //reserved  27
    FPCR = 28,
    FPSR = 29,
    FPIAR = 30,
    //reserved  31
    FP0 = 32,
    FP1 = 33,
    FP2 = 34,
    FP3 = 35,
    FP4 = 36,
    FP5 = 37,
    FP6 = 38,
    FP7 = 39,
    //reserved  40
    MMUSR030 = 41,
    MMUSR = 42,
    URP = 43,
    DTT0 = 44,
    DTT1 = 45,
    ITT0 = 46,
    ITT1 = 47,
    //reserved  50
    PSR = 51,
    PCSR = 52,
    VAL = 53,
    CRP = 54,
    SRP = 55,
    DRP = 56,
    TC = 57,
    AC = 58,
    SCC = 59,
    CAL = 60,
    TT0 = 61,
    TT1 = 62,
    //reserved  63
    BAD0 = 64,
    BAD1 = 65,
    BAD2 = 66,
    BAD3 = 67,
    BAD4 = 68,
    BAD5 = 69,
    BAD6 = 70,
    BAD7 = 71,
    BAC0 = 72,
    BAC1 = 73,
    BAC2 = 74,
    BAC3 = 75,
    BAC4 = 76,
    BAC5 = 77,
    BAC6 = 78,
    BAC7 = 79,
}

impl fmt::Display for M68KRegister {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::D0 => write!(f, "D0"),
            Self::D1 => write!(f, "D1"),
            Self::D2 => write!(f, "D2"),
            Self::D3 => write!(f, "D3"),
            Self::D4 => write!(f, "D4"),
            Self::D5 => write!(f, "D5"),
            Self::D6 => write!(f, "D6"),
            Self::D7 => write!(f, "D7"),
            Self::A0 => write!(f, "A0"),
            Self::A1 => write!(f, "A1"),
            Self::A2 => write!(f, "A2"),
            Self::A3 => write!(f, "A3"),
            Self::A4 => write!(f, "A4"),
            Self::A5 => write!(f, "A5"),
            Self::A6 => write!(f, "A6"),
            Self::A7 => write!(f, "A7"),
            Self::CCR => write!(f, "CCR"),
            Self::SR => write!(f, "SR"),
            Self::USP => write!(f, "USP"),
            Self::MSP => write!(f, "MSP"),
            Self::SFC => write!(f, "SFC"),
            Self::DFC => write!(f, "DFC"),
            Self::CACR => write!(f, "CACR"),
            Self::VBR => write!(f, "VBR"),
            Self::CAAR => write!(f, "CAAR"),
            Self::ISP => write!(f, "ISP"),
            Self::PC => write!(f, "PC"),
            Self::FPCR => write!(f, "FPCR"),
            Self::FPSR => write!(f, "FPSR"),
            Self::FPIAR => write!(f, "FPIAR"),
            Self::FP0 => write!(f, "FP0"),
            Self::FP1 => write!(f, "FP1"),
            Self::FP2 => write!(f, "FP2"),
            Self::FP3 => write!(f, "FP3"),
            Self::FP4 => write!(f, "FP4"),
            Self::FP5 => write!(f, "FP5"),
            Self::FP6 => write!(f, "FP6"),
            Self::FP7 => write!(f, "FP7"),
            Self::MMUSR030 => write!(f, "MMUSR030"),
            Self::MMUSR => write!(f, "MMUSR"),
            Self::URP => write!(f, "URP"),
            Self::DTT0 => write!(f, "DTT0"),
            Self::DTT1 => write!(f, "DTT1"),
            Self::ITT0 => write!(f, "ITT0"),
            Self::ITT1 => write!(f, "ITT1"),
            Self::PSR => write!(f, "PSR"),
            Self::PCSR => write!(f, "PCSR"),
            Self::VAL => write!(f, "VAL"),
            Self::CRP => write!(f, "CRP"),
            Self::SRP => write!(f, "SRP"),
            Self::DRP => write!(f, "DRP"),
            Self::TC => write!(f, "TC"),
            Self::AC => write!(f, "AC"),
            Self::SCC => write!(f, "SCC"),
            Self::CAL => write!(f, "CAL"),
            Self::TT0 => write!(f, "TT0"),
            Self::TT1 => write!(f, "TT1"),
            Self::BAD0 => write!(f, "BAD0"),
            Self::BAD1 => write!(f, "BAD1"),
            Self::BAD2 => write!(f, "BAD2"),
            Self::BAD3 => write!(f, "BAD3"),
            Self::BAD4 => write!(f, "BAD4"),
            Self::BAD5 => write!(f, "BAD5"),
            Self::BAD6 => write!(f, "BAD6"),
            Self::BAD7 => write!(f, "BAD7"),
            Self::BAC0 => write!(f, "BAC0"),
            Self::BAC1 => write!(f, "BAC1"),
            Self::BAC2 => write!(f, "BAC2"),
            Self::BAC3 => write!(f, "BAC3"),
            Self::BAC4 => write!(f, "BAC4"),
            Self::BAC5 => write!(f, "BAC5"),
            Self::BAC6 => write!(f, "BAC6"),
            Self::BAC7 => write!(f, "BAC7"),
        }
    }
}

impl TryFrom<u16> for M68KRegister {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            0 => Ok(Self::D0),
            1 => Ok(Self::D1),
            2 => Ok(Self::D2),
            3 => Ok(Self::D3),
            4 => Ok(Self::D4),
            5 => Ok(Self::D5),
            6 => Ok(Self::D6),
            7 => Ok(Self::D7),
            8 => Ok(Self::A0),
            9 => Ok(Self::A1),
            10 => Ok(Self::A2),
            11 => Ok(Self::A3),
            12 => Ok(Self::A4),
            13 => Ok(Self::A5),
            14 => Ok(Self::A6),
            15 => Ok(Self::A7),
            16 => Ok(Self::CCR),
            17 => Ok(Self::SR),
            18 => Ok(Self::USP),
            19 => Ok(Self::MSP),
            20 => Ok(Self::SFC),
            21 => Ok(Self::DFC),
            22 => Ok(Self::CACR),
            23 => Ok(Self::VBR),
            24 => Ok(Self::CAAR),
            25 => Ok(Self::ISP),
            26 => Ok(Self::PC),
            28 => Ok(Self::FPCR),
            29 => Ok(Self::FPSR),
            30 => Ok(Self::FPIAR),
            32 => Ok(Self::FP0),
            33 => Ok(Self::FP1),
            34 => Ok(Self::FP2),
            35 => Ok(Self::FP3),
            36 => Ok(Self::FP4),
            37 => Ok(Self::FP5),
            38 => Ok(Self::FP6),
            39 => Ok(Self::FP7),
            41 => Ok(Self::MMUSR030),
            42 => Ok(Self::MMUSR),
            43 => Ok(Self::URP),
            44 => Ok(Self::DTT0),
            45 => Ok(Self::DTT1),
            46 => Ok(Self::ITT0),
            47 => Ok(Self::ITT1),
            51 => Ok(Self::PSR),
            52 => Ok(Self::PCSR),
            53 => Ok(Self::VAL),
            54 => Ok(Self::CRP),
            55 => Ok(Self::SRP),
            56 => Ok(Self::DRP),
            57 => Ok(Self::TC),
            58 => Ok(Self::AC),
            59 => Ok(Self::SCC),
            60 => Ok(Self::CAL),
            61 => Ok(Self::TT0),
            62 => Ok(Self::TT1),
            64 => Ok(Self::BAD0),
            65 => Ok(Self::BAD1),
            66 => Ok(Self::BAD2),
            67 => Ok(Self::BAD3),
            68 => Ok(Self::BAD4),
            69 => Ok(Self::BAD5),
            70 => Ok(Self::BAD6),
            71 => Ok(Self::BAD7),
            72 => Ok(Self::BAC0),
            73 => Ok(Self::BAC1),
            74 => Ok(Self::BAC2),
            75 => Ok(Self::BAC3),
            76 => Ok(Self::BAC4),
            77 => Ok(Self::BAC5),
            78 => Ok(Self::BAC6),
            79 => Ok(Self::BAC7),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for M68KRegister {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// Register set for the MIPS 4000
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum MIPSRegister {
    IntZERO = 10, /* CPU REGISTER */
    IntAT = 11,
    IntV0 = 12,
    IntV1 = 13,
    IntA0 = 14,
    IntA1 = 15,
    IntA2 = 16,
    IntA3 = 17,
    IntT0 = 18,
    IntT1 = 19,
    IntT2 = 20,
    IntT3 = 21,
    IntT4 = 22,
    IntT5 = 23,
    IntT6 = 24,
    IntT7 = 25,
    IntS0 = 26,
    IntS1 = 27,
    IntS2 = 28,
    IntS3 = 29,
    IntS4 = 30,
    IntS5 = 31,
    IntS6 = 32,
    IntS7 = 33,
    IntT8 = 34,
    IntT9 = 35,
    IntKT0 = 36,
    IntKT1 = 37,
    IntGP = 38,
    IntSP = 39,
    IntS8 = 40,
    IntRA = 41,
    IntLO = 42,
    IntHI = 43,

    Fir = 50,
    Psr = 51,

    FltF0 = 60, /* Floating point registers */
    FltF1 = 61,
    FltF2 = 62,
    FltF3 = 63,
    FltF4 = 64,
    FltF5 = 65,
    FltF6 = 66,
    FltF7 = 67,
    FltF8 = 68,
    FltF9 = 69,
    FltF10 = 70,
    FltF11 = 71,
    FltF12 = 72,
    FltF13 = 73,
    FltF14 = 74,
    FltF15 = 75,
    FltF16 = 76,
    FltF17 = 77,
    FltF18 = 78,
    FltF19 = 79,
    FltF20 = 80,
    FltF21 = 81,
    FltF22 = 82,
    FltF23 = 83,
    FltF24 = 84,
    FltF25 = 85,
    FltF26 = 86,
    FltF27 = 87,
    FltF28 = 88,
    FltF29 = 89,
    FltF30 = 90,
    FltF31 = 91,
    FltFsr = 92,
}

impl fmt::Display for MIPSRegister {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::IntZERO => write!(f, "IntZERO"),
            Self::IntAT => write!(f, "IntAT"),
            Self::IntV0 => write!(f, "IntV0"),
            Self::IntV1 => write!(f, "IntV1"),
            Self::IntA0 => write!(f, "IntA0"),
            Self::IntA1 => write!(f, "IntA1"),
            Self::IntA2 => write!(f, "IntA2"),
            Self::IntA3 => write!(f, "IntA3"),
            Self::IntT0 => write!(f, "IntT0"),
            Self::IntT1 => write!(f, "IntT1"),
            Self::IntT2 => write!(f, "IntT2"),
            Self::IntT3 => write!(f, "IntT3"),
            Self::IntT4 => write!(f, "IntT4"),
            Self::IntT5 => write!(f, "IntT5"),
            Self::IntT6 => write!(f, "IntT6"),
            Self::IntT7 => write!(f, "IntT7"),
            Self::IntS0 => write!(f, "IntS0"),
            Self::IntS1 => write!(f, "IntS1"),
            Self::IntS2 => write!(f, "IntS2"),
            Self::IntS3 => write!(f, "IntS3"),
            Self::IntS4 => write!(f, "IntS4"),
            Self::IntS5 => write!(f, "IntS5"),
            Self::IntS6 => write!(f, "IntS6"),
            Self::IntS7 => write!(f, "IntS7"),
            Self::IntT8 => write!(f, "IntT8"),
            Self::IntT9 => write!(f, "IntT9"),
            Self::IntKT0 => write!(f, "IntKT0"),
            Self::IntKT1 => write!(f, "IntKT1"),
            Self::IntGP => write!(f, "IntGP"),
            Self::IntSP => write!(f, "IntSP"),
            Self::IntS8 => write!(f, "IntS8"),
            Self::IntRA => write!(f, "IntRA"),
            Self::IntLO => write!(f, "IntLO"),
            Self::IntHI => write!(f, "IntHI"),
            Self::Fir => write!(f, "Fir"),
            Self::Psr => write!(f, "Psr"),
            Self::FltF0 => write!(f, "FltF0"),
            Self::FltF1 => write!(f, "FltF1"),
            Self::FltF2 => write!(f, "FltF2"),
            Self::FltF3 => write!(f, "FltF3"),
            Self::FltF4 => write!(f, "FltF4"),
            Self::FltF5 => write!(f, "FltF5"),
            Self::FltF6 => write!(f, "FltF6"),
            Self::FltF7 => write!(f, "FltF7"),
            Self::FltF8 => write!(f, "FltF8"),
            Self::FltF9 => write!(f, "FltF9"),
            Self::FltF10 => write!(f, "FltF10"),
            Self::FltF11 => write!(f, "FltF11"),
            Self::FltF12 => write!(f, "FltF12"),
            Self::FltF13 => write!(f, "FltF13"),
            Self::FltF14 => write!(f, "FltF14"),
            Self::FltF15 => write!(f, "FltF15"),
            Self::FltF16 => write!(f, "FltF16"),
            Self::FltF17 => write!(f, "FltF17"),
            Self::FltF18 => write!(f, "FltF18"),
            Self::FltF19 => write!(f, "FltF19"),
            Self::FltF20 => write!(f, "FltF20"),
            Self::FltF21 => write!(f, "FltF21"),
            Self::FltF22 => write!(f, "FltF22"),
            Self::FltF23 => write!(f, "FltF23"),
            Self::FltF24 => write!(f, "FltF24"),
            Self::FltF25 => write!(f, "FltF25"),
            Self::FltF26 => write!(f, "FltF26"),
            Self::FltF27 => write!(f, "FltF27"),
            Self::FltF28 => write!(f, "FltF28"),
            Self::FltF29 => write!(f, "FltF29"),
            Self::FltF30 => write!(f, "FltF30"),
            Self::FltF31 => write!(f, "FltF31"),
            Self::FltFsr => write!(f, "FltFsr"),
        }
    }
}

impl TryFrom<u16> for MIPSRegister {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            10 => Ok(Self::IntZERO),
            11 => Ok(Self::IntAT),
            12 => Ok(Self::IntV0),
            13 => Ok(Self::IntV1),
            14 => Ok(Self::IntA0),
            15 => Ok(Self::IntA1),
            16 => Ok(Self::IntA2),
            17 => Ok(Self::IntA3),
            18 => Ok(Self::IntT0),
            19 => Ok(Self::IntT1),
            20 => Ok(Self::IntT2),
            21 => Ok(Self::IntT3),
            22 => Ok(Self::IntT4),
            23 => Ok(Self::IntT5),
            24 => Ok(Self::IntT6),
            25 => Ok(Self::IntT7),
            26 => Ok(Self::IntS0),
            27 => Ok(Self::IntS1),
            28 => Ok(Self::IntS2),
            29 => Ok(Self::IntS3),
            30 => Ok(Self::IntS4),
            31 => Ok(Self::IntS5),
            32 => Ok(Self::IntS6),
            33 => Ok(Self::IntS7),
            34 => Ok(Self::IntT8),
            35 => Ok(Self::IntT9),
            36 => Ok(Self::IntKT0),
            37 => Ok(Self::IntKT1),
            38 => Ok(Self::IntGP),
            39 => Ok(Self::IntSP),
            40 => Ok(Self::IntS8),
            41 => Ok(Self::IntRA),
            42 => Ok(Self::IntLO),
            43 => Ok(Self::IntHI),
            50 => Ok(Self::Fir),
            51 => Ok(Self::Psr),
            60 => Ok(Self::FltF0),
            61 => Ok(Self::FltF1),
            62 => Ok(Self::FltF2),
            63 => Ok(Self::FltF3),
            64 => Ok(Self::FltF4),
            65 => Ok(Self::FltF5),
            66 => Ok(Self::FltF6),
            67 => Ok(Self::FltF7),
            68 => Ok(Self::FltF8),
            69 => Ok(Self::FltF9),
            70 => Ok(Self::FltF10),
            71 => Ok(Self::FltF11),
            72 => Ok(Self::FltF12),
            73 => Ok(Self::FltF13),
            74 => Ok(Self::FltF14),
            75 => Ok(Self::FltF15),
            76 => Ok(Self::FltF16),
            77 => Ok(Self::FltF17),
            78 => Ok(Self::FltF18),
            79 => Ok(Self::FltF19),
            80 => Ok(Self::FltF20),
            81 => Ok(Self::FltF21),
            82 => Ok(Self::FltF22),
            83 => Ok(Self::FltF23),
            84 => Ok(Self::FltF24),
            85 => Ok(Self::FltF25),
            86 => Ok(Self::FltF26),
            87 => Ok(Self::FltF27),
            88 => Ok(Self::FltF28),
            89 => Ok(Self::FltF29),
            90 => Ok(Self::FltF30),
            91 => Ok(Self::FltF31),
            92 => Ok(Self::FltFsr),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for MIPSRegister {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// Register set for the ALPHA AXP
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum ALPHAAXPRegister {
    FltF0 = 10, // Floating point registers
    FltF1 = 11,
    FltF2 = 12,
    FltF3 = 13,
    FltF4 = 14,
    FltF5 = 15,
    FltF6 = 16,
    FltF7 = 17,
    FltF8 = 18,
    FltF9 = 19,
    FltF10 = 20,
    FltF11 = 21,
    FltF12 = 22,
    FltF13 = 23,
    FltF14 = 24,
    FltF15 = 25,
    FltF16 = 26,
    FltF17 = 27,
    FltF18 = 28,
    FltF19 = 29,
    FltF20 = 30,
    FltF21 = 31,
    FltF22 = 32,
    FltF23 = 33,
    FltF24 = 34,
    FltF25 = 35,
    FltF26 = 36,
    FltF27 = 37,
    FltF28 = 38,
    FltF29 = 39,
    FltF30 = 40,
    FltF31 = 41,

    IntV0 = 42, // Integer registers
    IntT0 = 43,
    IntT1 = 44,
    IntT2 = 45,
    IntT3 = 46,
    IntT4 = 47,
    IntT5 = 48,
    IntT6 = 49,
    IntT7 = 50,
    IntS0 = 51,
    IntS1 = 52,
    IntS2 = 53,
    IntS3 = 54,
    IntS4 = 55,
    IntS5 = 56,
    IntFP = 57,
    IntA0 = 58,
    IntA1 = 59,
    IntA2 = 60,
    IntA3 = 61,
    IntA4 = 62,
    IntA5 = 63,
    IntT8 = 64,
    IntT9 = 65,
    IntT10 = 66,
    IntT11 = 67,
    IntRA = 68,
    IntT12 = 69,
    IntAT = 70,
    IntGP = 71,
    IntSP = 72,
    IntZERO = 73,

    Fpcr = 74, // Control registers
    Fir = 75,
    Psr = 76,
    FltFsr = 77,
    SoftFpcr = 78,
}

impl fmt::Display for ALPHAAXPRegister {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::FltF0 => write!(f, "FltF0"),
            Self::FltF1 => write!(f, "FltF1"),
            Self::FltF2 => write!(f, "FltF2"),
            Self::FltF3 => write!(f, "FltF3"),
            Self::FltF4 => write!(f, "FltF4"),
            Self::FltF5 => write!(f, "FltF5"),
            Self::FltF6 => write!(f, "FltF6"),
            Self::FltF7 => write!(f, "FltF7"),
            Self::FltF8 => write!(f, "FltF8"),
            Self::FltF9 => write!(f, "FltF9"),
            Self::FltF10 => write!(f, "FltF10"),
            Self::FltF11 => write!(f, "FltF11"),
            Self::FltF12 => write!(f, "FltF12"),
            Self::FltF13 => write!(f, "FltF13"),
            Self::FltF14 => write!(f, "FltF14"),
            Self::FltF15 => write!(f, "FltF15"),
            Self::FltF16 => write!(f, "FltF16"),
            Self::FltF17 => write!(f, "FltF17"),
            Self::FltF18 => write!(f, "FltF18"),
            Self::FltF19 => write!(f, "FltF19"),
            Self::FltF20 => write!(f, "FltF20"),
            Self::FltF21 => write!(f, "FltF21"),
            Self::FltF22 => write!(f, "FltF22"),
            Self::FltF23 => write!(f, "FltF23"),
            Self::FltF24 => write!(f, "FltF24"),
            Self::FltF25 => write!(f, "FltF25"),
            Self::FltF26 => write!(f, "FltF26"),
            Self::FltF27 => write!(f, "FltF27"),
            Self::FltF28 => write!(f, "FltF28"),
            Self::FltF29 => write!(f, "FltF29"),
            Self::FltF30 => write!(f, "FltF30"),
            Self::FltF31 => write!(f, "FltF31"),
            Self::IntV0 => write!(f, "IntV0"),
            Self::IntT0 => write!(f, "IntT0"),
            Self::IntT1 => write!(f, "IntT1"),
            Self::IntT2 => write!(f, "IntT2"),
            Self::IntT3 => write!(f, "IntT3"),
            Self::IntT4 => write!(f, "IntT4"),
            Self::IntT5 => write!(f, "IntT5"),
            Self::IntT6 => write!(f, "IntT6"),
            Self::IntT7 => write!(f, "IntT7"),
            Self::IntS0 => write!(f, "IntS0"),
            Self::IntS1 => write!(f, "IntS1"),
            Self::IntS2 => write!(f, "IntS2"),
            Self::IntS3 => write!(f, "IntS3"),
            Self::IntS4 => write!(f, "IntS4"),
            Self::IntS5 => write!(f, "IntS5"),
            Self::IntFP => write!(f, "IntFP"),
            Self::IntA0 => write!(f, "IntA0"),
            Self::IntA1 => write!(f, "IntA1"),
            Self::IntA2 => write!(f, "IntA2"),
            Self::IntA3 => write!(f, "IntA3"),
            Self::IntA4 => write!(f, "IntA4"),
            Self::IntA5 => write!(f, "IntA5"),
            Self::IntT8 => write!(f, "IntT8"),
            Self::IntT9 => write!(f, "IntT9"),
            Self::IntT10 => write!(f, "IntT10"),
            Self::IntT11 => write!(f, "IntT11"),
            Self::IntRA => write!(f, "IntRA"),
            Self::IntT12 => write!(f, "IntT12"),
            Self::IntAT => write!(f, "IntAT"),
            Self::IntGP => write!(f, "IntGP"),
            Self::IntSP => write!(f, "IntSP"),
            Self::IntZERO => write!(f, "IntZERO"),
            Self::Fpcr => write!(f, "Fpcr"),
            Self::Fir => write!(f, "Fir"),
            Self::Psr => write!(f, "Psr"),
            Self::FltFsr => write!(f, "FltFsr"),
            Self::SoftFpcr => write!(f, "SoftFpcr"),
        }
    }
}

impl TryFrom<u16> for ALPHAAXPRegister {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            10 => Ok(Self::FltF0),
            11 => Ok(Self::FltF1),
            12 => Ok(Self::FltF2),
            13 => Ok(Self::FltF3),
            14 => Ok(Self::FltF4),
            15 => Ok(Self::FltF5),
            16 => Ok(Self::FltF6),
            17 => Ok(Self::FltF7),
            18 => Ok(Self::FltF8),
            19 => Ok(Self::FltF9),
            20 => Ok(Self::FltF10),
            21 => Ok(Self::FltF11),
            22 => Ok(Self::FltF12),
            23 => Ok(Self::FltF13),
            24 => Ok(Self::FltF14),
            25 => Ok(Self::FltF15),
            26 => Ok(Self::FltF16),
            27 => Ok(Self::FltF17),
            28 => Ok(Self::FltF18),
            29 => Ok(Self::FltF19),
            30 => Ok(Self::FltF20),
            31 => Ok(Self::FltF21),
            32 => Ok(Self::FltF22),
            33 => Ok(Self::FltF23),
            34 => Ok(Self::FltF24),
            35 => Ok(Self::FltF25),
            36 => Ok(Self::FltF26),
            37 => Ok(Self::FltF27),
            38 => Ok(Self::FltF28),
            39 => Ok(Self::FltF29),
            40 => Ok(Self::FltF30),
            41 => Ok(Self::FltF31),
            42 => Ok(Self::IntV0),
            43 => Ok(Self::IntT0),
            44 => Ok(Self::IntT1),
            45 => Ok(Self::IntT2),
            46 => Ok(Self::IntT3),
            47 => Ok(Self::IntT4),
            48 => Ok(Self::IntT5),
            49 => Ok(Self::IntT6),
            50 => Ok(Self::IntT7),
            51 => Ok(Self::IntS0),
            52 => Ok(Self::IntS1),
            53 => Ok(Self::IntS2),
            54 => Ok(Self::IntS3),
            55 => Ok(Self::IntS4),
            56 => Ok(Self::IntS5),
            57 => Ok(Self::IntFP),
            58 => Ok(Self::IntA0),
            59 => Ok(Self::IntA1),
            60 => Ok(Self::IntA2),
            61 => Ok(Self::IntA3),
            62 => Ok(Self::IntA4),
            63 => Ok(Self::IntA5),
            64 => Ok(Self::IntT8),
            65 => Ok(Self::IntT9),
            66 => Ok(Self::IntT10),
            67 => Ok(Self::IntT11),
            68 => Ok(Self::IntRA),
            69 => Ok(Self::IntT12),
            70 => Ok(Self::IntAT),
            71 => Ok(Self::IntGP),
            72 => Ok(Self::IntSP),
            73 => Ok(Self::IntZERO),
            74 => Ok(Self::Fpcr),
            75 => Ok(Self::Fir),
            76 => Ok(Self::Psr),
            77 => Ok(Self::FltFsr),
            78 => Ok(Self::SoftFpcr),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for ALPHAAXPRegister {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// Register Set for Motorola/IBM PowerPC
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum PowerPCRegister {
    /*
     ** PowerPC General Registers ( User Level )
     */
    GPR0 = 1,
    GPR1 = 2,
    GPR2 = 3,
    GPR3 = 4,
    GPR4 = 5,
    GPR5 = 6,
    GPR6 = 7,
    GPR7 = 8,
    GPR8 = 9,
    GPR9 = 10,
    GPR10 = 11,
    GPR11 = 12,
    GPR12 = 13,
    GPR13 = 14,
    GPR14 = 15,
    GPR15 = 16,
    GPR16 = 17,
    GPR17 = 18,
    GPR18 = 19,
    GPR19 = 20,
    GPR20 = 21,
    GPR21 = 22,
    GPR22 = 23,
    GPR23 = 24,
    GPR24 = 25,
    GPR25 = 26,
    GPR26 = 27,
    GPR27 = 28,
    GPR28 = 29,
    GPR29 = 30,
    GPR30 = 31,
    GPR31 = 32,

    /*
     ** PowerPC Condition Register ( User Level )
     */
    CR = 33,
    CR0 = 34,
    CR1 = 35,
    CR2 = 36,
    CR3 = 37,
    CR4 = 38,
    CR5 = 39,
    CR6 = 40,
    CR7 = 41,

    /*
     ** PowerPC Floating Point Registers ( User Level )
     */
    FPR0 = 42,
    FPR1 = 43,
    FPR2 = 44,
    FPR3 = 45,
    FPR4 = 46,
    FPR5 = 47,
    FPR6 = 48,
    FPR7 = 49,
    FPR8 = 50,
    FPR9 = 51,
    FPR10 = 52,
    FPR11 = 53,
    FPR12 = 54,
    FPR13 = 55,
    FPR14 = 56,
    FPR15 = 57,
    FPR16 = 58,
    FPR17 = 59,
    FPR18 = 60,
    FPR19 = 61,
    FPR20 = 62,
    FPR21 = 63,
    FPR22 = 64,
    FPR23 = 65,
    FPR24 = 66,
    FPR25 = 67,
    FPR26 = 68,
    FPR27 = 69,
    FPR28 = 70,
    FPR29 = 71,
    FPR30 = 72,
    FPR31 = 73,

    /*
     ** PowerPC Floating Point Status and Control Register ( User Level )
     */
    FPSCR = 74,

    /*
     ** PowerPC Machine State Register ( Supervisor Level )
     */
    MSR = 75,

    /*
     ** PowerPC Segment Registers ( Supervisor Level )
     */
    SR0 = 76,
    SR1 = 77,
    SR2 = 78,
    SR3 = 79,
    SR4 = 80,
    SR5 = 81,
    SR6 = 82,
    SR7 = 83,
    SR8 = 84,
    SR9 = 85,
    SR10 = 86,
    SR11 = 87,
    SR12 = 88,
    SR13 = 89,
    SR14 = 90,
    SR15 = 91,

    /*
    ** For all of the special purpose registers add 100 to the SPR# that the
    ** Motorola/IBM documentation gives with the exception of any imaginary
    ** registers.
    */

    /*
    ** PowerPC Special Purpose Registers ( User Level )
    */
    PC = 99, // PC (imaginary register)

    MQ = 100, // MPC601
    XER = 101,
    RTCU = 104, // MPC601
    RTCL = 105, // MPC601
    LR = 108,
    CTR = 109,

    COMPARE = 110, // part of XER (internal to the debugger only)
    COUNT = 111,   // part of XER (internal to the debugger only)

    /*
     ** PowerPC Special Purpose Registers ( Supervisor Level )
     */
    DSISR = 118,
    DAR = 119,
    DEC = 122,
    SDR1 = 125,
    SRR0 = 126,
    SRR1 = 127,
    SPRG0 = 372,
    SPRG1 = 373,
    SPRG2 = 374,
    SPRG3 = 375,
    ASR = 280, // 64-bit implementations only
    EAR = 382,
    PVR = 287,
    BAT0U = 628,
    BAT0L = 629,
    BAT1U = 630,
    BAT1L = 631,
    BAT2U = 632,
    BAT2L = 633,
    BAT3U = 634,
    BAT3L = 635,
    DBAT0U = 636,
    DBAT0L = 637,
    DBAT1U = 638,
    DBAT1L = 639,
    DBAT2U = 640,
    DBAT2L = 641,
    DBAT3U = 642,
    DBAT3L = 643,

    /*
    ** PowerPC Special Purpose Registers Implementation Dependent ( Supervisor Level )
    */

    /*
    ** Doesn't appear that IBM/Motorola has finished defining these.
    */
    PMR0 = 1044,  // MPC620,
    PMR1 = 1045,  // MPC620,
    PMR2 = 1046,  // MPC620,
    PMR3 = 1047,  // MPC620,
    PMR4 = 1048,  // MPC620,
    PMR5 = 1049,  // MPC620,
    PMR6 = 1050,  // MPC620,
    PMR7 = 1051,  // MPC620,
    PMR8 = 1052,  // MPC620,
    PMR9 = 1053,  // MPC620,
    PMR10 = 1054, // MPC620,
    PMR11 = 1055, // MPC620,
    PMR12 = 1056, // MPC620,
    PMR13 = 1057, // MPC620,
    PMR14 = 1058, // MPC620,
    PMR15 = 1059, // MPC620,

    DMISS = 1076, // MPC603
    DCMP = 1077,  // MPC603
    HASH1 = 1078, // MPC603
    HASH2 = 1079, // MPC603
    IMISS = 1080, // MPC603
    ICMP = 1081,  // MPC603
    RPA = 1082,   // MPC603

    HID0 = 1108,  // MPC601, MPC603, MPC620
    HID1 = 1109,  // MPC601
    HID2 = 1110,  // MPC601, MPC603, MPC620 ( IABR )
    HID3 = 1111,  // Not Defined
    HID4 = 1112,  // Not Defined
    HID5 = 1113,  // MPC601, MPC604, MPC620 ( DABR )
    HID6 = 1114,  // Not Defined
    HID7 = 1115,  // Not Defined
    HID8 = 1116,  // MPC620 ( BUSCSR )
    HID9 = 1117,  // MPC620 ( L2CSR )
    HID10 = 1118, // Not Defined
    HID11 = 1119, // Not Defined
    HID12 = 1120, // Not Defined
    HID13 = 1121, // MPC604 ( HCR )
    HID14 = 1122, // Not Defined
    HID15 = 1123, // MPC601, MPC604, MPC620 ( PIR )
}

impl fmt::Display for PowerPCRegister {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::GPR0 => write!(f, "GPR0"),
            Self::GPR1 => write!(f, "GPR1"),
            Self::GPR2 => write!(f, "GPR2"),
            Self::GPR3 => write!(f, "GPR3"),
            Self::GPR4 => write!(f, "GPR4"),
            Self::GPR5 => write!(f, "GPR5"),
            Self::GPR6 => write!(f, "GPR6"),
            Self::GPR7 => write!(f, "GPR7"),
            Self::GPR8 => write!(f, "GPR8"),
            Self::GPR9 => write!(f, "GPR9"),
            Self::GPR10 => write!(f, "GPR10"),
            Self::GPR11 => write!(f, "GPR11"),
            Self::GPR12 => write!(f, "GPR12"),
            Self::GPR13 => write!(f, "GPR13"),
            Self::GPR14 => write!(f, "GPR14"),
            Self::GPR15 => write!(f, "GPR15"),
            Self::GPR16 => write!(f, "GPR16"),
            Self::GPR17 => write!(f, "GPR17"),
            Self::GPR18 => write!(f, "GPR18"),
            Self::GPR19 => write!(f, "GPR19"),
            Self::GPR20 => write!(f, "GPR20"),
            Self::GPR21 => write!(f, "GPR21"),
            Self::GPR22 => write!(f, "GPR22"),
            Self::GPR23 => write!(f, "GPR23"),
            Self::GPR24 => write!(f, "GPR24"),
            Self::GPR25 => write!(f, "GPR25"),
            Self::GPR26 => write!(f, "GPR26"),
            Self::GPR27 => write!(f, "GPR27"),
            Self::GPR28 => write!(f, "GPR28"),
            Self::GPR29 => write!(f, "GPR29"),
            Self::GPR30 => write!(f, "GPR30"),
            Self::GPR31 => write!(f, "GPR31"),
            Self::CR => write!(f, "CR"),
            Self::CR0 => write!(f, "CR0"),
            Self::CR1 => write!(f, "CR1"),
            Self::CR2 => write!(f, "CR2"),
            Self::CR3 => write!(f, "CR3"),
            Self::CR4 => write!(f, "CR4"),
            Self::CR5 => write!(f, "CR5"),
            Self::CR6 => write!(f, "CR6"),
            Self::CR7 => write!(f, "CR7"),
            Self::FPR0 => write!(f, "FPR0"),
            Self::FPR1 => write!(f, "FPR1"),
            Self::FPR2 => write!(f, "FPR2"),
            Self::FPR3 => write!(f, "FPR3"),
            Self::FPR4 => write!(f, "FPR4"),
            Self::FPR5 => write!(f, "FPR5"),
            Self::FPR6 => write!(f, "FPR6"),
            Self::FPR7 => write!(f, "FPR7"),
            Self::FPR8 => write!(f, "FPR8"),
            Self::FPR9 => write!(f, "FPR9"),
            Self::FPR10 => write!(f, "FPR10"),
            Self::FPR11 => write!(f, "FPR11"),
            Self::FPR12 => write!(f, "FPR12"),
            Self::FPR13 => write!(f, "FPR13"),
            Self::FPR14 => write!(f, "FPR14"),
            Self::FPR15 => write!(f, "FPR15"),
            Self::FPR16 => write!(f, "FPR16"),
            Self::FPR17 => write!(f, "FPR17"),
            Self::FPR18 => write!(f, "FPR18"),
            Self::FPR19 => write!(f, "FPR19"),
            Self::FPR20 => write!(f, "FPR20"),
            Self::FPR21 => write!(f, "FPR21"),
            Self::FPR22 => write!(f, "FPR22"),
            Self::FPR23 => write!(f, "FPR23"),
            Self::FPR24 => write!(f, "FPR24"),
            Self::FPR25 => write!(f, "FPR25"),
            Self::FPR26 => write!(f, "FPR26"),
            Self::FPR27 => write!(f, "FPR27"),
            Self::FPR28 => write!(f, "FPR28"),
            Self::FPR29 => write!(f, "FPR29"),
            Self::FPR30 => write!(f, "FPR30"),
            Self::FPR31 => write!(f, "FPR31"),
            Self::FPSCR => write!(f, "FPSCR"),
            Self::MSR => write!(f, "MSR"),
            Self::SR0 => write!(f, "SR0"),
            Self::SR1 => write!(f, "SR1"),
            Self::SR2 => write!(f, "SR2"),
            Self::SR3 => write!(f, "SR3"),
            Self::SR4 => write!(f, "SR4"),
            Self::SR5 => write!(f, "SR5"),
            Self::SR6 => write!(f, "SR6"),
            Self::SR7 => write!(f, "SR7"),
            Self::SR8 => write!(f, "SR8"),
            Self::SR9 => write!(f, "SR9"),
            Self::SR10 => write!(f, "SR10"),
            Self::SR11 => write!(f, "SR11"),
            Self::SR12 => write!(f, "SR12"),
            Self::SR13 => write!(f, "SR13"),
            Self::SR14 => write!(f, "SR14"),
            Self::SR15 => write!(f, "SR15"),
            Self::PC => write!(f, "PC"),
            Self::MQ => write!(f, "MQ"),
            Self::XER => write!(f, "XER"),
            Self::RTCU => write!(f, "RTCU"),
            Self::RTCL => write!(f, "RTCL"),
            Self::LR => write!(f, "LR"),
            Self::CTR => write!(f, "CTR"),
            Self::COMPARE => write!(f, "COMPARE"),
            Self::COUNT => write!(f, "COUNT"),
            Self::DSISR => write!(f, "DSISR"),
            Self::DAR => write!(f, "DAR"),
            Self::DEC => write!(f, "DEC"),
            Self::SDR1 => write!(f, "SDR1"),
            Self::SRR0 => write!(f, "SRR0"),
            Self::SRR1 => write!(f, "SRR1"),
            Self::SPRG0 => write!(f, "SPRG0"),
            Self::SPRG1 => write!(f, "SPRG1"),
            Self::SPRG2 => write!(f, "SPRG2"),
            Self::SPRG3 => write!(f, "SPRG3"),
            Self::ASR => write!(f, "ASR"),
            Self::EAR => write!(f, "EAR"),
            Self::PVR => write!(f, "PVR"),
            Self::BAT0U => write!(f, "BAT0U"),
            Self::BAT0L => write!(f, "BAT0L"),
            Self::BAT1U => write!(f, "BAT1U"),
            Self::BAT1L => write!(f, "BAT1L"),
            Self::BAT2U => write!(f, "BAT2U"),
            Self::BAT2L => write!(f, "BAT2L"),
            Self::BAT3U => write!(f, "BAT3U"),
            Self::BAT3L => write!(f, "BAT3L"),
            Self::DBAT0U => write!(f, "DBAT0U"),
            Self::DBAT0L => write!(f, "DBAT0L"),
            Self::DBAT1U => write!(f, "DBAT1U"),
            Self::DBAT1L => write!(f, "DBAT1L"),
            Self::DBAT2U => write!(f, "DBAT2U"),
            Self::DBAT2L => write!(f, "DBAT2L"),
            Self::DBAT3U => write!(f, "DBAT3U"),
            Self::DBAT3L => write!(f, "DBAT3L"),
            Self::PMR0 => write!(f, "PMR0"),
            Self::PMR1 => write!(f, "PMR1"),
            Self::PMR2 => write!(f, "PMR2"),
            Self::PMR3 => write!(f, "PMR3"),
            Self::PMR4 => write!(f, "PMR4"),
            Self::PMR5 => write!(f, "PMR5"),
            Self::PMR6 => write!(f, "PMR6"),
            Self::PMR7 => write!(f, "PMR7"),
            Self::PMR8 => write!(f, "PMR8"),
            Self::PMR9 => write!(f, "PMR9"),
            Self::PMR10 => write!(f, "PMR10"),
            Self::PMR11 => write!(f, "PMR11"),
            Self::PMR12 => write!(f, "PMR12"),
            Self::PMR13 => write!(f, "PMR13"),
            Self::PMR14 => write!(f, "PMR14"),
            Self::PMR15 => write!(f, "PMR15"),
            Self::DMISS => write!(f, "DMISS"),
            Self::DCMP => write!(f, "DCMP"),
            Self::HASH1 => write!(f, "HASH1"),
            Self::HASH2 => write!(f, "HASH2"),
            Self::IMISS => write!(f, "IMISS"),
            Self::ICMP => write!(f, "ICMP"),
            Self::RPA => write!(f, "RPA"),
            Self::HID0 => write!(f, "HID0"),
            Self::HID1 => write!(f, "HID1"),
            Self::HID2 => write!(f, "HID2"),
            Self::HID3 => write!(f, "HID3"),
            Self::HID4 => write!(f, "HID4"),
            Self::HID5 => write!(f, "HID5"),
            Self::HID6 => write!(f, "HID6"),
            Self::HID7 => write!(f, "HID7"),
            Self::HID8 => write!(f, "HID8"),
            Self::HID9 => write!(f, "HID9"),
            Self::HID10 => write!(f, "HID10"),
            Self::HID11 => write!(f, "HID11"),
            Self::HID12 => write!(f, "HID12"),
            Self::HID13 => write!(f, "HID13"),
            Self::HID14 => write!(f, "HID14"),
            Self::HID15 => write!(f, "HID15"),
        }
    }
}

impl TryFrom<u16> for PowerPCRegister {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            1 => Ok(Self::GPR0),
            2 => Ok(Self::GPR1),
            3 => Ok(Self::GPR2),
            4 => Ok(Self::GPR3),
            5 => Ok(Self::GPR4),
            6 => Ok(Self::GPR5),
            7 => Ok(Self::GPR6),
            8 => Ok(Self::GPR7),
            9 => Ok(Self::GPR8),
            10 => Ok(Self::GPR9),
            11 => Ok(Self::GPR10),
            12 => Ok(Self::GPR11),
            13 => Ok(Self::GPR12),
            14 => Ok(Self::GPR13),
            15 => Ok(Self::GPR14),
            16 => Ok(Self::GPR15),
            17 => Ok(Self::GPR16),
            18 => Ok(Self::GPR17),
            19 => Ok(Self::GPR18),
            20 => Ok(Self::GPR19),
            21 => Ok(Self::GPR20),
            22 => Ok(Self::GPR21),
            23 => Ok(Self::GPR22),
            24 => Ok(Self::GPR23),
            25 => Ok(Self::GPR24),
            26 => Ok(Self::GPR25),
            27 => Ok(Self::GPR26),
            28 => Ok(Self::GPR27),
            29 => Ok(Self::GPR28),
            30 => Ok(Self::GPR29),
            31 => Ok(Self::GPR30),
            32 => Ok(Self::GPR31),
            33 => Ok(Self::CR),
            34 => Ok(Self::CR0),
            35 => Ok(Self::CR1),
            36 => Ok(Self::CR2),
            37 => Ok(Self::CR3),
            38 => Ok(Self::CR4),
            39 => Ok(Self::CR5),
            40 => Ok(Self::CR6),
            41 => Ok(Self::CR7),
            42 => Ok(Self::FPR0),
            43 => Ok(Self::FPR1),
            44 => Ok(Self::FPR2),
            45 => Ok(Self::FPR3),
            46 => Ok(Self::FPR4),
            47 => Ok(Self::FPR5),
            48 => Ok(Self::FPR6),
            49 => Ok(Self::FPR7),
            50 => Ok(Self::FPR8),
            51 => Ok(Self::FPR9),
            52 => Ok(Self::FPR10),
            53 => Ok(Self::FPR11),
            54 => Ok(Self::FPR12),
            55 => Ok(Self::FPR13),
            56 => Ok(Self::FPR14),
            57 => Ok(Self::FPR15),
            58 => Ok(Self::FPR16),
            59 => Ok(Self::FPR17),
            60 => Ok(Self::FPR18),
            61 => Ok(Self::FPR19),
            62 => Ok(Self::FPR20),
            63 => Ok(Self::FPR21),
            64 => Ok(Self::FPR22),
            65 => Ok(Self::FPR23),
            66 => Ok(Self::FPR24),
            67 => Ok(Self::FPR25),
            68 => Ok(Self::FPR26),
            69 => Ok(Self::FPR27),
            70 => Ok(Self::FPR28),
            71 => Ok(Self::FPR29),
            72 => Ok(Self::FPR30),
            73 => Ok(Self::FPR31),
            74 => Ok(Self::FPSCR),
            75 => Ok(Self::MSR),
            76 => Ok(Self::SR0),
            77 => Ok(Self::SR1),
            78 => Ok(Self::SR2),
            79 => Ok(Self::SR3),
            80 => Ok(Self::SR4),
            81 => Ok(Self::SR5),
            82 => Ok(Self::SR6),
            83 => Ok(Self::SR7),
            84 => Ok(Self::SR8),
            85 => Ok(Self::SR9),
            86 => Ok(Self::SR10),
            87 => Ok(Self::SR11),
            88 => Ok(Self::SR12),
            89 => Ok(Self::SR13),
            90 => Ok(Self::SR14),
            91 => Ok(Self::SR15),
            99 => Ok(Self::PC),
            100 => Ok(Self::MQ),
            101 => Ok(Self::XER),
            104 => Ok(Self::RTCU),
            105 => Ok(Self::RTCL),
            108 => Ok(Self::LR),
            109 => Ok(Self::CTR),
            110 => Ok(Self::COMPARE),
            111 => Ok(Self::COUNT),
            118 => Ok(Self::DSISR),
            119 => Ok(Self::DAR),
            122 => Ok(Self::DEC),
            125 => Ok(Self::SDR1),
            126 => Ok(Self::SRR0),
            127 => Ok(Self::SRR1),
            372 => Ok(Self::SPRG0),
            373 => Ok(Self::SPRG1),
            374 => Ok(Self::SPRG2),
            375 => Ok(Self::SPRG3),
            280 => Ok(Self::ASR),
            382 => Ok(Self::EAR),
            287 => Ok(Self::PVR),
            628 => Ok(Self::BAT0U),
            629 => Ok(Self::BAT0L),
            630 => Ok(Self::BAT1U),
            631 => Ok(Self::BAT1L),
            632 => Ok(Self::BAT2U),
            633 => Ok(Self::BAT2L),
            634 => Ok(Self::BAT3U),
            635 => Ok(Self::BAT3L),
            636 => Ok(Self::DBAT0U),
            637 => Ok(Self::DBAT0L),
            638 => Ok(Self::DBAT1U),
            639 => Ok(Self::DBAT1L),
            640 => Ok(Self::DBAT2U),
            641 => Ok(Self::DBAT2L),
            642 => Ok(Self::DBAT3U),
            643 => Ok(Self::DBAT3L),
            1044 => Ok(Self::PMR0),
            1045 => Ok(Self::PMR1),
            1046 => Ok(Self::PMR2),
            1047 => Ok(Self::PMR3),
            1048 => Ok(Self::PMR4),
            1049 => Ok(Self::PMR5),
            1050 => Ok(Self::PMR6),
            1051 => Ok(Self::PMR7),
            1052 => Ok(Self::PMR8),
            1053 => Ok(Self::PMR9),
            1054 => Ok(Self::PMR10),
            1055 => Ok(Self::PMR11),
            1056 => Ok(Self::PMR12),
            1057 => Ok(Self::PMR13),
            1058 => Ok(Self::PMR14),
            1059 => Ok(Self::PMR15),
            1076 => Ok(Self::DMISS),
            1077 => Ok(Self::DCMP),
            1078 => Ok(Self::HASH1),
            1079 => Ok(Self::HASH2),
            1080 => Ok(Self::IMISS),
            1081 => Ok(Self::ICMP),
            1082 => Ok(Self::RPA),
            1108 => Ok(Self::HID0),
            1109 => Ok(Self::HID1),
            1110 => Ok(Self::HID2),
            1111 => Ok(Self::HID3),
            1112 => Ok(Self::HID4),
            1113 => Ok(Self::HID5),
            1114 => Ok(Self::HID6),
            1115 => Ok(Self::HID7),
            1116 => Ok(Self::HID8),
            1117 => Ok(Self::HID9),
            1118 => Ok(Self::HID10),
            1119 => Ok(Self::HID11),
            1120 => Ok(Self::HID12),
            1121 => Ok(Self::HID13),
            1122 => Ok(Self::HID14),
            1123 => Ok(Self::HID15),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for PowerPCRegister {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// JAVA VM registers
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum JavaRegister {
    PC = 1,
}

impl fmt::Display for JavaRegister {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::PC => write!(f, "PC"),
        }
    }
}

impl TryFrom<u16> for JavaRegister {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            1 => Ok(Self::PC),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for JavaRegister {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// Register set for the Hitachi SH(3)
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum HitachiSHRegister {
    IntR0 = 10, // CPU REGISTER
    IntR1 = 11,
    IntR2 = 12,
    IntR3 = 13,
    IntR4 = 14,
    IntR5 = 15,
    IntR6 = 16,
    IntR7 = 17,
    IntR8 = 18,
    IntR9 = 19,
    IntR10 = 20,
    IntR11 = 21,
    IntR12 = 22,
    IntR13 = 23,
    IntFp = 24,
    IntSp = 25,
    Gbr = 38,
    Pr = 39,
    Mach = 40,
    Macl = 41,

    Pc = 50,
    Sr = 51,

    BarA = 60,
    BasrA = 61,
    BamrA = 62,
    BbrA = 63,
    BarB = 64,
    BasrB = 65,
    BamrB = 66,
    BbrB = 67,
    BdrB = 68,
    BdmrB = 69,
    Brcr = 70,

    //
    // Additional registers for Hitachi SH processors
    //
    Fpscr = 75, // floating point status/control register
    Fpul = 76,  // floating point communication register

    FpR0 = 80, // Floating point registers
    FpR1 = 81,
    FpR2 = 82,
    FpR3 = 83,
    FpR4 = 84,
    FpR5 = 85,
    FpR6 = 86,
    FpR7 = 87,
    FpR8 = 88,
    FpR9 = 89,
    FpR10 = 90,
    FpR11 = 91,
    FpR12 = 92,
    FpR13 = 93,
    FpR14 = 94,
    FpR15 = 95,

    XFpR0 = 96,
    XFpR1 = 97,
    XFpR2 = 98,
    XFpR3 = 99,
    XFpR4 = 100,
    XFpR5 = 101,
    XFpR6 = 102,
    XFpR7 = 103,
    XFpR8 = 104,
    XFpR9 = 105,
    XFpR10 = 106,
    XFpR11 = 107,
    XFpR12 = 108,
    XFpR13 = 109,
    XFpR14 = 110,
    XFpR15 = 111,
}

impl fmt::Display for HitachiSHRegister {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::IntR0 => write!(f, "IntR0"),
            Self::IntR1 => write!(f, "IntR1"),
            Self::IntR2 => write!(f, "IntR2"),
            Self::IntR3 => write!(f, "IntR3"),
            Self::IntR4 => write!(f, "IntR4"),
            Self::IntR5 => write!(f, "IntR5"),
            Self::IntR6 => write!(f, "IntR6"),
            Self::IntR7 => write!(f, "IntR7"),
            Self::IntR8 => write!(f, "IntR8"),
            Self::IntR9 => write!(f, "IntR9"),
            Self::IntR10 => write!(f, "IntR10"),
            Self::IntR11 => write!(f, "IntR11"),
            Self::IntR12 => write!(f, "IntR12"),
            Self::IntR13 => write!(f, "IntR13"),
            Self::IntFp => write!(f, "IntFp"),
            Self::IntSp => write!(f, "IntSp"),
            Self::Gbr => write!(f, "Gbr"),
            Self::Pr => write!(f, "Pr"),
            Self::Mach => write!(f, "Mach"),
            Self::Macl => write!(f, "Macl"),
            Self::Pc => write!(f, "Pc"),
            Self::Sr => write!(f, "Sr"),
            Self::BarA => write!(f, "BarA"),
            Self::BasrA => write!(f, "BasrA"),
            Self::BamrA => write!(f, "BamrA"),
            Self::BbrA => write!(f, "BbrA"),
            Self::BarB => write!(f, "BarB"),
            Self::BasrB => write!(f, "BasrB"),
            Self::BamrB => write!(f, "BamrB"),
            Self::BbrB => write!(f, "BbrB"),
            Self::BdrB => write!(f, "BdrB"),
            Self::BdmrB => write!(f, "BdmrB"),
            Self::Brcr => write!(f, "Brcr"),
            Self::Fpscr => write!(f, "Fpscr"),
            Self::Fpul => write!(f, "Fpul"),
            Self::FpR0 => write!(f, "FpR0"),
            Self::FpR1 => write!(f, "FpR1"),
            Self::FpR2 => write!(f, "FpR2"),
            Self::FpR3 => write!(f, "FpR3"),
            Self::FpR4 => write!(f, "FpR4"),
            Self::FpR5 => write!(f, "FpR5"),
            Self::FpR6 => write!(f, "FpR6"),
            Self::FpR7 => write!(f, "FpR7"),
            Self::FpR8 => write!(f, "FpR8"),
            Self::FpR9 => write!(f, "FpR9"),
            Self::FpR10 => write!(f, "FpR10"),
            Self::FpR11 => write!(f, "FpR11"),
            Self::FpR12 => write!(f, "FpR12"),
            Self::FpR13 => write!(f, "FpR13"),
            Self::FpR14 => write!(f, "FpR14"),
            Self::FpR15 => write!(f, "FpR15"),
            Self::XFpR0 => write!(f, "XFpR0"),
            Self::XFpR1 => write!(f, "XFpR1"),
            Self::XFpR2 => write!(f, "XFpR2"),
            Self::XFpR3 => write!(f, "XFpR3"),
            Self::XFpR4 => write!(f, "XFpR4"),
            Self::XFpR5 => write!(f, "XFpR5"),
            Self::XFpR6 => write!(f, "XFpR6"),
            Self::XFpR7 => write!(f, "XFpR7"),
            Self::XFpR8 => write!(f, "XFpR8"),
            Self::XFpR9 => write!(f, "XFpR9"),
            Self::XFpR10 => write!(f, "XFpR10"),
            Self::XFpR11 => write!(f, "XFpR11"),
            Self::XFpR12 => write!(f, "XFpR12"),
            Self::XFpR13 => write!(f, "XFpR13"),
            Self::XFpR14 => write!(f, "XFpR14"),
            Self::XFpR15 => write!(f, "XFpR15"),
        }
    }
}

impl TryFrom<u16> for HitachiSHRegister {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            10 => Ok(Self::IntR0),
            11 => Ok(Self::IntR1),
            12 => Ok(Self::IntR2),
            13 => Ok(Self::IntR3),
            14 => Ok(Self::IntR4),
            15 => Ok(Self::IntR5),
            16 => Ok(Self::IntR6),
            17 => Ok(Self::IntR7),
            18 => Ok(Self::IntR8),
            19 => Ok(Self::IntR9),
            20 => Ok(Self::IntR10),
            21 => Ok(Self::IntR11),
            22 => Ok(Self::IntR12),
            23 => Ok(Self::IntR13),
            24 => Ok(Self::IntFp),
            25 => Ok(Self::IntSp),
            38 => Ok(Self::Gbr),
            39 => Ok(Self::Pr),
            40 => Ok(Self::Mach),
            41 => Ok(Self::Macl),
            50 => Ok(Self::Pc),
            51 => Ok(Self::Sr),
            60 => Ok(Self::BarA),
            61 => Ok(Self::BasrA),
            62 => Ok(Self::BamrA),
            63 => Ok(Self::BbrA),
            64 => Ok(Self::BarB),
            65 => Ok(Self::BasrB),
            66 => Ok(Self::BamrB),
            67 => Ok(Self::BbrB),
            68 => Ok(Self::BdrB),
            69 => Ok(Self::BdmrB),
            70 => Ok(Self::Brcr),
            75 => Ok(Self::Fpscr),
            76 => Ok(Self::Fpul),
            80 => Ok(Self::FpR0),
            81 => Ok(Self::FpR1),
            82 => Ok(Self::FpR2),
            83 => Ok(Self::FpR3),
            84 => Ok(Self::FpR4),
            85 => Ok(Self::FpR5),
            86 => Ok(Self::FpR6),
            87 => Ok(Self::FpR7),
            88 => Ok(Self::FpR8),
            89 => Ok(Self::FpR9),
            90 => Ok(Self::FpR10),
            91 => Ok(Self::FpR11),
            92 => Ok(Self::FpR12),
            93 => Ok(Self::FpR13),
            94 => Ok(Self::FpR14),
            95 => Ok(Self::FpR15),
            96 => Ok(Self::XFpR0),
            97 => Ok(Self::XFpR1),
            98 => Ok(Self::XFpR2),
            99 => Ok(Self::XFpR3),
            100 => Ok(Self::XFpR4),
            101 => Ok(Self::XFpR5),
            102 => Ok(Self::XFpR6),
            103 => Ok(Self::XFpR7),
            104 => Ok(Self::XFpR8),
            105 => Ok(Self::XFpR9),
            106 => Ok(Self::XFpR10),
            107 => Ok(Self::XFpR11),
            108 => Ok(Self::XFpR12),
            109 => Ok(Self::XFpR13),
            110 => Ok(Self::XFpR14),
            111 => Ok(Self::XFpR15),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for HitachiSHRegister {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// Register set for the ARM processor
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum ARMRegister {
    R0 = 10,
    R1 = 11,
    R2 = 12,
    R3 = 13,
    R4 = 14,
    R5 = 15,
    R6 = 16,
    R7 = 17,
    R8 = 18,
    R9 = 19,
    R10 = 20,
    R11 = 21, // Frame pointer, if allocated
    R12 = 22,
    SP = 23,   // Stack pointer
    LR = 24,   // Link Register
    PC = 25,   // Program counter
    CPSR = 26, // Current program status register

    ACC0 = 27, // DSP co-processor 0 40 bit accumulator

    //
    // Registers for ARM VFP10 support
    //
    FPSCR = 40,
    FPEXC = 41,

    FS0 = 50,
    FS1 = 51,
    FS2 = 52,
    FS3 = 53,
    FS4 = 54,
    FS5 = 55,
    FS6 = 56,
    FS7 = 57,
    FS8 = 58,
    FS9 = 59,
    FS10 = 60,
    FS11 = 61,
    FS12 = 62,
    FS13 = 63,
    FS14 = 64,
    FS15 = 65,
    FS16 = 66,
    FS17 = 67,
    FS18 = 68,
    FS19 = 69,
    FS20 = 70,
    FS21 = 71,
    FS22 = 72,
    FS23 = 73,
    FS24 = 74,
    FS25 = 75,
    FS26 = 76,
    FS27 = 77,
    FS28 = 78,
    FS29 = 79,
    FS30 = 80,
    FS31 = 81,

    //
    // ARM VFP Floating Point Extra control registers
    //
    FPEXTRA0 = 90,
    FPEXTRA1 = 91,
    FPEXTRA2 = 92,
    FPEXTRA3 = 93,
    FPEXTRA4 = 94,
    FPEXTRA5 = 95,
    FPEXTRA6 = 96,
    FPEXTRA7 = 97,

    // XSCALE Concan co-processor registers
    WR0 = 128,
    WR1 = 129,
    WR2 = 130,
    WR3 = 131,
    WR4 = 132,
    WR5 = 133,
    WR6 = 134,
    WR7 = 135,
    WR8 = 136,
    WR9 = 137,
    WR10 = 138,
    WR11 = 139,
    WR12 = 140,
    WR13 = 141,
    WR14 = 142,
    WR15 = 143,

    // XSCALE Concan co-processor control registers
    WCID = 144,
    WCON = 145,
    WCSSF = 146,
    WCASF = 147,
    WC4 = 148,
    WC5 = 149,
    WC6 = 150,
    WC7 = 151,
    WCGR0 = 152,
    WCGR1 = 153,
    WCGR2 = 154,
    WCGR3 = 155,
    WC12 = 156,
    WC13 = 157,
    WC14 = 158,
    WC15 = 159,

    //
    // ARM VFPv3/Neon extended floating Point
    //
    FS32 = 200,
    FS33 = 201,
    FS34 = 202,
    FS35 = 203,
    FS36 = 204,
    FS37 = 205,
    FS38 = 206,
    FS39 = 207,
    FS40 = 208,
    FS41 = 209,
    FS42 = 210,
    FS43 = 211,
    FS44 = 212,
    FS45 = 213,
    FS46 = 214,
    FS47 = 215,
    FS48 = 216,
    FS49 = 217,
    FS50 = 218,
    FS51 = 219,
    FS52 = 220,
    FS53 = 221,
    FS54 = 222,
    FS55 = 223,
    FS56 = 224,
    FS57 = 225,
    FS58 = 226,
    FS59 = 227,
    FS60 = 228,
    FS61 = 229,
    FS62 = 230,
    FS63 = 231,

    // ARM double-precision floating point
    ND0 = 300,
    ND1 = 301,
    ND2 = 302,
    ND3 = 303,
    ND4 = 304,
    ND5 = 305,
    ND6 = 306,
    ND7 = 307,
    ND8 = 308,
    ND9 = 309,
    ND10 = 310,
    ND11 = 311,
    ND12 = 312,
    ND13 = 313,
    ND14 = 314,
    ND15 = 315,
    ND16 = 316,
    ND17 = 317,
    ND18 = 318,
    ND19 = 319,
    ND20 = 320,
    ND21 = 321,
    ND22 = 322,
    ND23 = 323,
    ND24 = 324,
    ND25 = 325,
    ND26 = 326,
    ND27 = 327,
    ND28 = 328,
    ND29 = 329,
    ND30 = 330,
    ND31 = 331,

    // ARM extended precision floating point
    NQ0 = 400,
    NQ1 = 401,
    NQ2 = 402,
    NQ3 = 403,
    NQ4 = 404,
    NQ5 = 405,
    NQ6 = 406,
    NQ7 = 407,
    NQ8 = 408,
    NQ9 = 409,
    NQ10 = 410,
    NQ11 = 411,
    NQ12 = 412,
    NQ13 = 413,
    NQ14 = 414,
    NQ15 = 415,
}

impl fmt::Display for ARMRegister {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::R0 => write!(f, "R0"),
            Self::R1 => write!(f, "R1"),
            Self::R2 => write!(f, "R2"),
            Self::R3 => write!(f, "R3"),
            Self::R4 => write!(f, "R4"),
            Self::R5 => write!(f, "R5"),
            Self::R6 => write!(f, "R6"),
            Self::R7 => write!(f, "R7"),
            Self::R8 => write!(f, "R8"),
            Self::R9 => write!(f, "R9"),
            Self::R10 => write!(f, "R10"),
            Self::R11 => write!(f, "R11"),
            Self::R12 => write!(f, "R12"),
            Self::SP => write!(f, "SP"),
            Self::LR => write!(f, "LR"),
            Self::PC => write!(f, "PC"),
            Self::CPSR => write!(f, "CPSR"),
            Self::ACC0 => write!(f, "ACC0"),
            Self::FPSCR => write!(f, "FPSCR"),
            Self::FPEXC => write!(f, "FPEXC"),
            Self::FS0 => write!(f, "FS0"),
            Self::FS1 => write!(f, "FS1"),
            Self::FS2 => write!(f, "FS2"),
            Self::FS3 => write!(f, "FS3"),
            Self::FS4 => write!(f, "FS4"),
            Self::FS5 => write!(f, "FS5"),
            Self::FS6 => write!(f, "FS6"),
            Self::FS7 => write!(f, "FS7"),
            Self::FS8 => write!(f, "FS8"),
            Self::FS9 => write!(f, "FS9"),
            Self::FS10 => write!(f, "FS10"),
            Self::FS11 => write!(f, "FS11"),
            Self::FS12 => write!(f, "FS12"),
            Self::FS13 => write!(f, "FS13"),
            Self::FS14 => write!(f, "FS14"),
            Self::FS15 => write!(f, "FS15"),
            Self::FS16 => write!(f, "FS16"),
            Self::FS17 => write!(f, "FS17"),
            Self::FS18 => write!(f, "FS18"),
            Self::FS19 => write!(f, "FS19"),
            Self::FS20 => write!(f, "FS20"),
            Self::FS21 => write!(f, "FS21"),
            Self::FS22 => write!(f, "FS22"),
            Self::FS23 => write!(f, "FS23"),
            Self::FS24 => write!(f, "FS24"),
            Self::FS25 => write!(f, "FS25"),
            Self::FS26 => write!(f, "FS26"),
            Self::FS27 => write!(f, "FS27"),
            Self::FS28 => write!(f, "FS28"),
            Self::FS29 => write!(f, "FS29"),
            Self::FS30 => write!(f, "FS30"),
            Self::FS31 => write!(f, "FS31"),
            Self::FPEXTRA0 => write!(f, "FPEXTRA0"),
            Self::FPEXTRA1 => write!(f, "FPEXTRA1"),
            Self::FPEXTRA2 => write!(f, "FPEXTRA2"),
            Self::FPEXTRA3 => write!(f, "FPEXTRA3"),
            Self::FPEXTRA4 => write!(f, "FPEXTRA4"),
            Self::FPEXTRA5 => write!(f, "FPEXTRA5"),
            Self::FPEXTRA6 => write!(f, "FPEXTRA6"),
            Self::FPEXTRA7 => write!(f, "FPEXTRA7"),
            Self::WR0 => write!(f, "WR0"),
            Self::WR1 => write!(f, "WR1"),
            Self::WR2 => write!(f, "WR2"),
            Self::WR3 => write!(f, "WR3"),
            Self::WR4 => write!(f, "WR4"),
            Self::WR5 => write!(f, "WR5"),
            Self::WR6 => write!(f, "WR6"),
            Self::WR7 => write!(f, "WR7"),
            Self::WR8 => write!(f, "WR8"),
            Self::WR9 => write!(f, "WR9"),
            Self::WR10 => write!(f, "WR10"),
            Self::WR11 => write!(f, "WR11"),
            Self::WR12 => write!(f, "WR12"),
            Self::WR13 => write!(f, "WR13"),
            Self::WR14 => write!(f, "WR14"),
            Self::WR15 => write!(f, "WR15"),
            Self::WCID => write!(f, "WCID"),
            Self::WCON => write!(f, "WCON"),
            Self::WCSSF => write!(f, "WCSSF"),
            Self::WCASF => write!(f, "WCASF"),
            Self::WC4 => write!(f, "WC4"),
            Self::WC5 => write!(f, "WC5"),
            Self::WC6 => write!(f, "WC6"),
            Self::WC7 => write!(f, "WC7"),
            Self::WCGR0 => write!(f, "WCGR0"),
            Self::WCGR1 => write!(f, "WCGR1"),
            Self::WCGR2 => write!(f, "WCGR2"),
            Self::WCGR3 => write!(f, "WCGR3"),
            Self::WC12 => write!(f, "WC12"),
            Self::WC13 => write!(f, "WC13"),
            Self::WC14 => write!(f, "WC14"),
            Self::WC15 => write!(f, "WC15"),
            Self::FS32 => write!(f, "FS32"),
            Self::FS33 => write!(f, "FS33"),
            Self::FS34 => write!(f, "FS34"),
            Self::FS35 => write!(f, "FS35"),
            Self::FS36 => write!(f, "FS36"),
            Self::FS37 => write!(f, "FS37"),
            Self::FS38 => write!(f, "FS38"),
            Self::FS39 => write!(f, "FS39"),
            Self::FS40 => write!(f, "FS40"),
            Self::FS41 => write!(f, "FS41"),
            Self::FS42 => write!(f, "FS42"),
            Self::FS43 => write!(f, "FS43"),
            Self::FS44 => write!(f, "FS44"),
            Self::FS45 => write!(f, "FS45"),
            Self::FS46 => write!(f, "FS46"),
            Self::FS47 => write!(f, "FS47"),
            Self::FS48 => write!(f, "FS48"),
            Self::FS49 => write!(f, "FS49"),
            Self::FS50 => write!(f, "FS50"),
            Self::FS51 => write!(f, "FS51"),
            Self::FS52 => write!(f, "FS52"),
            Self::FS53 => write!(f, "FS53"),
            Self::FS54 => write!(f, "FS54"),
            Self::FS55 => write!(f, "FS55"),
            Self::FS56 => write!(f, "FS56"),
            Self::FS57 => write!(f, "FS57"),
            Self::FS58 => write!(f, "FS58"),
            Self::FS59 => write!(f, "FS59"),
            Self::FS60 => write!(f, "FS60"),
            Self::FS61 => write!(f, "FS61"),
            Self::FS62 => write!(f, "FS62"),
            Self::FS63 => write!(f, "FS63"),
            Self::ND0 => write!(f, "ND0"),
            Self::ND1 => write!(f, "ND1"),
            Self::ND2 => write!(f, "ND2"),
            Self::ND3 => write!(f, "ND3"),
            Self::ND4 => write!(f, "ND4"),
            Self::ND5 => write!(f, "ND5"),
            Self::ND6 => write!(f, "ND6"),
            Self::ND7 => write!(f, "ND7"),
            Self::ND8 => write!(f, "ND8"),
            Self::ND9 => write!(f, "ND9"),
            Self::ND10 => write!(f, "ND10"),
            Self::ND11 => write!(f, "ND11"),
            Self::ND12 => write!(f, "ND12"),
            Self::ND13 => write!(f, "ND13"),
            Self::ND14 => write!(f, "ND14"),
            Self::ND15 => write!(f, "ND15"),
            Self::ND16 => write!(f, "ND16"),
            Self::ND17 => write!(f, "ND17"),
            Self::ND18 => write!(f, "ND18"),
            Self::ND19 => write!(f, "ND19"),
            Self::ND20 => write!(f, "ND20"),
            Self::ND21 => write!(f, "ND21"),
            Self::ND22 => write!(f, "ND22"),
            Self::ND23 => write!(f, "ND23"),
            Self::ND24 => write!(f, "ND24"),
            Self::ND25 => write!(f, "ND25"),
            Self::ND26 => write!(f, "ND26"),
            Self::ND27 => write!(f, "ND27"),
            Self::ND28 => write!(f, "ND28"),
            Self::ND29 => write!(f, "ND29"),
            Self::ND30 => write!(f, "ND30"),
            Self::ND31 => write!(f, "ND31"),
            Self::NQ0 => write!(f, "NQ0"),
            Self::NQ1 => write!(f, "NQ1"),
            Self::NQ2 => write!(f, "NQ2"),
            Self::NQ3 => write!(f, "NQ3"),
            Self::NQ4 => write!(f, "NQ4"),
            Self::NQ5 => write!(f, "NQ5"),
            Self::NQ6 => write!(f, "NQ6"),
            Self::NQ7 => write!(f, "NQ7"),
            Self::NQ8 => write!(f, "NQ8"),
            Self::NQ9 => write!(f, "NQ9"),
            Self::NQ10 => write!(f, "NQ10"),
            Self::NQ11 => write!(f, "NQ11"),
            Self::NQ12 => write!(f, "NQ12"),
            Self::NQ13 => write!(f, "NQ13"),
            Self::NQ14 => write!(f, "NQ14"),
            Self::NQ15 => write!(f, "NQ15"),
        }
    }
}

impl TryFrom<u16> for ARMRegister {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            10 => Ok(Self::R0),
            11 => Ok(Self::R1),
            12 => Ok(Self::R2),
            13 => Ok(Self::R3),
            14 => Ok(Self::R4),
            15 => Ok(Self::R5),
            16 => Ok(Self::R6),
            17 => Ok(Self::R7),
            18 => Ok(Self::R8),
            19 => Ok(Self::R9),
            20 => Ok(Self::R10),
            21 => Ok(Self::R11),
            22 => Ok(Self::R12),
            23 => Ok(Self::SP),
            24 => Ok(Self::LR),
            25 => Ok(Self::PC),
            26 => Ok(Self::CPSR),
            27 => Ok(Self::ACC0),
            40 => Ok(Self::FPSCR),
            41 => Ok(Self::FPEXC),
            50 => Ok(Self::FS0),
            51 => Ok(Self::FS1),
            52 => Ok(Self::FS2),
            53 => Ok(Self::FS3),
            54 => Ok(Self::FS4),
            55 => Ok(Self::FS5),
            56 => Ok(Self::FS6),
            57 => Ok(Self::FS7),
            58 => Ok(Self::FS8),
            59 => Ok(Self::FS9),
            60 => Ok(Self::FS10),
            61 => Ok(Self::FS11),
            62 => Ok(Self::FS12),
            63 => Ok(Self::FS13),
            64 => Ok(Self::FS14),
            65 => Ok(Self::FS15),
            66 => Ok(Self::FS16),
            67 => Ok(Self::FS17),
            68 => Ok(Self::FS18),
            69 => Ok(Self::FS19),
            70 => Ok(Self::FS20),
            71 => Ok(Self::FS21),
            72 => Ok(Self::FS22),
            73 => Ok(Self::FS23),
            74 => Ok(Self::FS24),
            75 => Ok(Self::FS25),
            76 => Ok(Self::FS26),
            77 => Ok(Self::FS27),
            78 => Ok(Self::FS28),
            79 => Ok(Self::FS29),
            80 => Ok(Self::FS30),
            81 => Ok(Self::FS31),
            90 => Ok(Self::FPEXTRA0),
            91 => Ok(Self::FPEXTRA1),
            92 => Ok(Self::FPEXTRA2),
            93 => Ok(Self::FPEXTRA3),
            94 => Ok(Self::FPEXTRA4),
            95 => Ok(Self::FPEXTRA5),
            96 => Ok(Self::FPEXTRA6),
            97 => Ok(Self::FPEXTRA7),
            128 => Ok(Self::WR0),
            129 => Ok(Self::WR1),
            130 => Ok(Self::WR2),
            131 => Ok(Self::WR3),
            132 => Ok(Self::WR4),
            133 => Ok(Self::WR5),
            134 => Ok(Self::WR6),
            135 => Ok(Self::WR7),
            136 => Ok(Self::WR8),
            137 => Ok(Self::WR9),
            138 => Ok(Self::WR10),
            139 => Ok(Self::WR11),
            140 => Ok(Self::WR12),
            141 => Ok(Self::WR13),
            142 => Ok(Self::WR14),
            143 => Ok(Self::WR15),
            144 => Ok(Self::WCID),
            145 => Ok(Self::WCON),
            146 => Ok(Self::WCSSF),
            147 => Ok(Self::WCASF),
            148 => Ok(Self::WC4),
            149 => Ok(Self::WC5),
            150 => Ok(Self::WC6),
            151 => Ok(Self::WC7),
            152 => Ok(Self::WCGR0),
            153 => Ok(Self::WCGR1),
            154 => Ok(Self::WCGR2),
            155 => Ok(Self::WCGR3),
            156 => Ok(Self::WC12),
            157 => Ok(Self::WC13),
            158 => Ok(Self::WC14),
            159 => Ok(Self::WC15),
            200 => Ok(Self::FS32),
            201 => Ok(Self::FS33),
            202 => Ok(Self::FS34),
            203 => Ok(Self::FS35),
            204 => Ok(Self::FS36),
            205 => Ok(Self::FS37),
            206 => Ok(Self::FS38),
            207 => Ok(Self::FS39),
            208 => Ok(Self::FS40),
            209 => Ok(Self::FS41),
            210 => Ok(Self::FS42),
            211 => Ok(Self::FS43),
            212 => Ok(Self::FS44),
            213 => Ok(Self::FS45),
            214 => Ok(Self::FS46),
            215 => Ok(Self::FS47),
            216 => Ok(Self::FS48),
            217 => Ok(Self::FS49),
            218 => Ok(Self::FS50),
            219 => Ok(Self::FS51),
            220 => Ok(Self::FS52),
            221 => Ok(Self::FS53),
            222 => Ok(Self::FS54),
            223 => Ok(Self::FS55),
            224 => Ok(Self::FS56),
            225 => Ok(Self::FS57),
            226 => Ok(Self::FS58),
            227 => Ok(Self::FS59),
            228 => Ok(Self::FS60),
            229 => Ok(Self::FS61),
            230 => Ok(Self::FS62),
            231 => Ok(Self::FS63),
            300 => Ok(Self::ND0),
            301 => Ok(Self::ND1),
            302 => Ok(Self::ND2),
            303 => Ok(Self::ND3),
            304 => Ok(Self::ND4),
            305 => Ok(Self::ND5),
            306 => Ok(Self::ND6),
            307 => Ok(Self::ND7),
            308 => Ok(Self::ND8),
            309 => Ok(Self::ND9),
            310 => Ok(Self::ND10),
            311 => Ok(Self::ND11),
            312 => Ok(Self::ND12),
            313 => Ok(Self::ND13),
            314 => Ok(Self::ND14),
            315 => Ok(Self::ND15),
            316 => Ok(Self::ND16),
            317 => Ok(Self::ND17),
            318 => Ok(Self::ND18),
            319 => Ok(Self::ND19),
            320 => Ok(Self::ND20),
            321 => Ok(Self::ND21),
            322 => Ok(Self::ND22),
            323 => Ok(Self::ND23),
            324 => Ok(Self::ND24),
            325 => Ok(Self::ND25),
            326 => Ok(Self::ND26),
            327 => Ok(Self::ND27),
            328 => Ok(Self::ND28),
            329 => Ok(Self::ND29),
            330 => Ok(Self::ND30),
            331 => Ok(Self::ND31),
            400 => Ok(Self::NQ0),
            401 => Ok(Self::NQ1),
            402 => Ok(Self::NQ2),
            403 => Ok(Self::NQ3),
            404 => Ok(Self::NQ4),
            405 => Ok(Self::NQ5),
            406 => Ok(Self::NQ6),
            407 => Ok(Self::NQ7),
            408 => Ok(Self::NQ8),
            409 => Ok(Self::NQ9),
            410 => Ok(Self::NQ10),
            411 => Ok(Self::NQ11),
            412 => Ok(Self::NQ12),
            413 => Ok(Self::NQ13),
            414 => Ok(Self::NQ14),
            415 => Ok(Self::NQ15),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for ARMRegister {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// Register set for ARM64
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum ARM64Register {
    // General purpose 32-bit integer registers
    W0 = 10,
    W1 = 11,
    W2 = 12,
    W3 = 13,
    W4 = 14,
    W5 = 15,
    W6 = 16,
    W7 = 17,
    W8 = 18,
    W9 = 19,
    W10 = 20,
    W11 = 21,
    W12 = 22,
    W13 = 23,
    W14 = 24,
    W15 = 25,
    W16 = 26,
    W17 = 27,
    W18 = 28,
    W19 = 29,
    W20 = 30,
    W21 = 31,
    W22 = 32,
    W23 = 33,
    W24 = 34,
    W25 = 35,
    W26 = 36,
    W27 = 37,
    W28 = 38,
    W29 = 39,
    W30 = 40,
    WZR = 41,

    // General purpose 64-bit integer registers
    X0 = 50,
    X1 = 51,
    X2 = 52,
    X3 = 53,
    X4 = 54,
    X5 = 55,
    X6 = 56,
    X7 = 57,
    X8 = 58,
    X9 = 59,
    X10 = 60,
    X11 = 61,
    X12 = 62,
    X13 = 63,
    X14 = 64,
    X15 = 65,
    IP0 = 66,
    IP1 = 67,
    X18 = 68,
    X19 = 69,
    X20 = 70,
    X21 = 71,
    X22 = 72,
    X23 = 73,
    X24 = 74,
    X25 = 75,
    X26 = 76,
    X27 = 77,
    X28 = 78,
    FP = 79,
    LR = 80,
    SP = 81,
    ZR = 82,

    // statue register
    NZCV = 90,

    // 32-bit floating point registers
    S0 = 100,
    S1 = 101,
    S2 = 102,
    S3 = 103,
    S4 = 104,
    S5 = 105,
    S6 = 106,
    S7 = 107,
    S8 = 108,
    S9 = 109,
    S10 = 110,
    S11 = 111,
    S12 = 112,
    S13 = 113,
    S14 = 114,
    S15 = 115,
    S16 = 116,
    S17 = 117,
    S18 = 118,
    S19 = 119,
    S20 = 120,
    S21 = 121,
    S22 = 122,
    S23 = 123,
    S24 = 124,
    S25 = 125,
    S26 = 126,
    S27 = 127,
    S28 = 128,
    S29 = 129,
    S30 = 130,
    S31 = 131,

    // 64-bit floating point registers
    D0 = 140,
    D1 = 141,
    D2 = 142,
    D3 = 143,
    D4 = 144,
    D5 = 145,
    D6 = 146,
    D7 = 147,
    D8 = 148,
    D9 = 149,
    D10 = 150,
    D11 = 151,
    D12 = 152,
    D13 = 153,
    D14 = 154,
    D15 = 155,
    D16 = 156,
    D17 = 157,
    D18 = 158,
    D19 = 159,
    D20 = 160,
    D21 = 161,
    D22 = 162,
    D23 = 163,
    D24 = 164,
    D25 = 165,
    D26 = 166,
    D27 = 167,
    D28 = 168,
    D29 = 169,
    D30 = 170,
    D31 = 171,

    // 128-bit SIMD registers
    Q0 = 180,
    Q1 = 181,
    Q2 = 182,
    Q3 = 183,
    Q4 = 184,
    Q5 = 185,
    Q6 = 186,
    Q7 = 187,
    Q8 = 188,
    Q9 = 189,
    Q10 = 190,
    Q11 = 191,
    Q12 = 192,
    Q13 = 193,
    Q14 = 194,
    Q15 = 195,
    Q16 = 196,
    Q17 = 197,
    Q18 = 198,
    Q19 = 199,
    Q20 = 200,
    Q21 = 201,
    Q22 = 202,
    Q23 = 203,
    Q24 = 204,
    Q25 = 205,
    Q26 = 206,
    Q27 = 207,
    Q28 = 208,
    Q29 = 209,
    Q30 = 210,
    Q31 = 211,

    // Floating point status register
    FPSR = 220,
}

impl fmt::Display for ARM64Register {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::W0 => write!(f, "W0"),
            Self::W1 => write!(f, "W1"),
            Self::W2 => write!(f, "W2"),
            Self::W3 => write!(f, "W3"),
            Self::W4 => write!(f, "W4"),
            Self::W5 => write!(f, "W5"),
            Self::W6 => write!(f, "W6"),
            Self::W7 => write!(f, "W7"),
            Self::W8 => write!(f, "W8"),
            Self::W9 => write!(f, "W9"),
            Self::W10 => write!(f, "W10"),
            Self::W11 => write!(f, "W11"),
            Self::W12 => write!(f, "W12"),
            Self::W13 => write!(f, "W13"),
            Self::W14 => write!(f, "W14"),
            Self::W15 => write!(f, "W15"),
            Self::W16 => write!(f, "W16"),
            Self::W17 => write!(f, "W17"),
            Self::W18 => write!(f, "W18"),
            Self::W19 => write!(f, "W19"),
            Self::W20 => write!(f, "W20"),
            Self::W21 => write!(f, "W21"),
            Self::W22 => write!(f, "W22"),
            Self::W23 => write!(f, "W23"),
            Self::W24 => write!(f, "W24"),
            Self::W25 => write!(f, "W25"),
            Self::W26 => write!(f, "W26"),
            Self::W27 => write!(f, "W27"),
            Self::W28 => write!(f, "W28"),
            Self::W29 => write!(f, "W29"),
            Self::W30 => write!(f, "W30"),
            Self::WZR => write!(f, "WZR"),
            Self::X0 => write!(f, "X0"),
            Self::X1 => write!(f, "X1"),
            Self::X2 => write!(f, "X2"),
            Self::X3 => write!(f, "X3"),
            Self::X4 => write!(f, "X4"),
            Self::X5 => write!(f, "X5"),
            Self::X6 => write!(f, "X6"),
            Self::X7 => write!(f, "X7"),
            Self::X8 => write!(f, "X8"),
            Self::X9 => write!(f, "X9"),
            Self::X10 => write!(f, "X10"),
            Self::X11 => write!(f, "X11"),
            Self::X12 => write!(f, "X12"),
            Self::X13 => write!(f, "X13"),
            Self::X14 => write!(f, "X14"),
            Self::X15 => write!(f, "X15"),
            Self::IP0 => write!(f, "IP0"),
            Self::IP1 => write!(f, "IP1"),
            Self::X18 => write!(f, "X18"),
            Self::X19 => write!(f, "X19"),
            Self::X20 => write!(f, "X20"),
            Self::X21 => write!(f, "X21"),
            Self::X22 => write!(f, "X22"),
            Self::X23 => write!(f, "X23"),
            Self::X24 => write!(f, "X24"),
            Self::X25 => write!(f, "X25"),
            Self::X26 => write!(f, "X26"),
            Self::X27 => write!(f, "X27"),
            Self::X28 => write!(f, "X28"),
            Self::FP => write!(f, "FP"),
            Self::LR => write!(f, "LR"),
            Self::SP => write!(f, "SP"),
            Self::ZR => write!(f, "ZR"),
            Self::NZCV => write!(f, "NZCV"),
            Self::S0 => write!(f, "S0"),
            Self::S1 => write!(f, "S1"),
            Self::S2 => write!(f, "S2"),
            Self::S3 => write!(f, "S3"),
            Self::S4 => write!(f, "S4"),
            Self::S5 => write!(f, "S5"),
            Self::S6 => write!(f, "S6"),
            Self::S7 => write!(f, "S7"),
            Self::S8 => write!(f, "S8"),
            Self::S9 => write!(f, "S9"),
            Self::S10 => write!(f, "S10"),
            Self::S11 => write!(f, "S11"),
            Self::S12 => write!(f, "S12"),
            Self::S13 => write!(f, "S13"),
            Self::S14 => write!(f, "S14"),
            Self::S15 => write!(f, "S15"),
            Self::S16 => write!(f, "S16"),
            Self::S17 => write!(f, "S17"),
            Self::S18 => write!(f, "S18"),
            Self::S19 => write!(f, "S19"),
            Self::S20 => write!(f, "S20"),
            Self::S21 => write!(f, "S21"),
            Self::S22 => write!(f, "S22"),
            Self::S23 => write!(f, "S23"),
            Self::S24 => write!(f, "S24"),
            Self::S25 => write!(f, "S25"),
            Self::S26 => write!(f, "S26"),
            Self::S27 => write!(f, "S27"),
            Self::S28 => write!(f, "S28"),
            Self::S29 => write!(f, "S29"),
            Self::S30 => write!(f, "S30"),
            Self::S31 => write!(f, "S31"),
            Self::D0 => write!(f, "D0"),
            Self::D1 => write!(f, "D1"),
            Self::D2 => write!(f, "D2"),
            Self::D3 => write!(f, "D3"),
            Self::D4 => write!(f, "D4"),
            Self::D5 => write!(f, "D5"),
            Self::D6 => write!(f, "D6"),
            Self::D7 => write!(f, "D7"),
            Self::D8 => write!(f, "D8"),
            Self::D9 => write!(f, "D9"),
            Self::D10 => write!(f, "D10"),
            Self::D11 => write!(f, "D11"),
            Self::D12 => write!(f, "D12"),
            Self::D13 => write!(f, "D13"),
            Self::D14 => write!(f, "D14"),
            Self::D15 => write!(f, "D15"),
            Self::D16 => write!(f, "D16"),
            Self::D17 => write!(f, "D17"),
            Self::D18 => write!(f, "D18"),
            Self::D19 => write!(f, "D19"),
            Self::D20 => write!(f, "D20"),
            Self::D21 => write!(f, "D21"),
            Self::D22 => write!(f, "D22"),
            Self::D23 => write!(f, "D23"),
            Self::D24 => write!(f, "D24"),
            Self::D25 => write!(f, "D25"),
            Self::D26 => write!(f, "D26"),
            Self::D27 => write!(f, "D27"),
            Self::D28 => write!(f, "D28"),
            Self::D29 => write!(f, "D29"),
            Self::D30 => write!(f, "D30"),
            Self::D31 => write!(f, "D31"),
            Self::Q0 => write!(f, "Q0"),
            Self::Q1 => write!(f, "Q1"),
            Self::Q2 => write!(f, "Q2"),
            Self::Q3 => write!(f, "Q3"),
            Self::Q4 => write!(f, "Q4"),
            Self::Q5 => write!(f, "Q5"),
            Self::Q6 => write!(f, "Q6"),
            Self::Q7 => write!(f, "Q7"),
            Self::Q8 => write!(f, "Q8"),
            Self::Q9 => write!(f, "Q9"),
            Self::Q10 => write!(f, "Q10"),
            Self::Q11 => write!(f, "Q11"),
            Self::Q12 => write!(f, "Q12"),
            Self::Q13 => write!(f, "Q13"),
            Self::Q14 => write!(f, "Q14"),
            Self::Q15 => write!(f, "Q15"),
            Self::Q16 => write!(f, "Q16"),
            Self::Q17 => write!(f, "Q17"),
            Self::Q18 => write!(f, "Q18"),
            Self::Q19 => write!(f, "Q19"),
            Self::Q20 => write!(f, "Q20"),
            Self::Q21 => write!(f, "Q21"),
            Self::Q22 => write!(f, "Q22"),
            Self::Q23 => write!(f, "Q23"),
            Self::Q24 => write!(f, "Q24"),
            Self::Q25 => write!(f, "Q25"),
            Self::Q26 => write!(f, "Q26"),
            Self::Q27 => write!(f, "Q27"),
            Self::Q28 => write!(f, "Q28"),
            Self::Q29 => write!(f, "Q29"),
            Self::Q30 => write!(f, "Q30"),
            Self::Q31 => write!(f, "Q31"),
            Self::FPSR => write!(f, "FPSR"),
        }
    }
}

impl TryFrom<u16> for ARM64Register {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            10 => Ok(Self::W0),
            11 => Ok(Self::W1),
            12 => Ok(Self::W2),
            13 => Ok(Self::W3),
            14 => Ok(Self::W4),
            15 => Ok(Self::W5),
            16 => Ok(Self::W6),
            17 => Ok(Self::W7),
            18 => Ok(Self::W8),
            19 => Ok(Self::W9),
            20 => Ok(Self::W10),
            21 => Ok(Self::W11),
            22 => Ok(Self::W12),
            23 => Ok(Self::W13),
            24 => Ok(Self::W14),
            25 => Ok(Self::W15),
            26 => Ok(Self::W16),
            27 => Ok(Self::W17),
            28 => Ok(Self::W18),
            29 => Ok(Self::W19),
            30 => Ok(Self::W20),
            31 => Ok(Self::W21),
            32 => Ok(Self::W22),
            33 => Ok(Self::W23),
            34 => Ok(Self::W24),
            35 => Ok(Self::W25),
            36 => Ok(Self::W26),
            37 => Ok(Self::W27),
            38 => Ok(Self::W28),
            39 => Ok(Self::W29),
            40 => Ok(Self::W30),
            41 => Ok(Self::WZR),
            50 => Ok(Self::X0),
            51 => Ok(Self::X1),
            52 => Ok(Self::X2),
            53 => Ok(Self::X3),
            54 => Ok(Self::X4),
            55 => Ok(Self::X5),
            56 => Ok(Self::X6),
            57 => Ok(Self::X7),
            58 => Ok(Self::X8),
            59 => Ok(Self::X9),
            60 => Ok(Self::X10),
            61 => Ok(Self::X11),
            62 => Ok(Self::X12),
            63 => Ok(Self::X13),
            64 => Ok(Self::X14),
            65 => Ok(Self::X15),
            66 => Ok(Self::IP0),
            67 => Ok(Self::IP1),
            68 => Ok(Self::X18),
            69 => Ok(Self::X19),
            70 => Ok(Self::X20),
            71 => Ok(Self::X21),
            72 => Ok(Self::X22),
            73 => Ok(Self::X23),
            74 => Ok(Self::X24),
            75 => Ok(Self::X25),
            76 => Ok(Self::X26),
            77 => Ok(Self::X27),
            78 => Ok(Self::X28),
            79 => Ok(Self::FP),
            80 => Ok(Self::LR),
            81 => Ok(Self::SP),
            82 => Ok(Self::ZR),
            90 => Ok(Self::NZCV),
            100 => Ok(Self::S0),
            101 => Ok(Self::S1),
            102 => Ok(Self::S2),
            103 => Ok(Self::S3),
            104 => Ok(Self::S4),
            105 => Ok(Self::S5),
            106 => Ok(Self::S6),
            107 => Ok(Self::S7),
            108 => Ok(Self::S8),
            109 => Ok(Self::S9),
            110 => Ok(Self::S10),
            111 => Ok(Self::S11),
            112 => Ok(Self::S12),
            113 => Ok(Self::S13),
            114 => Ok(Self::S14),
            115 => Ok(Self::S15),
            116 => Ok(Self::S16),
            117 => Ok(Self::S17),
            118 => Ok(Self::S18),
            119 => Ok(Self::S19),
            120 => Ok(Self::S20),
            121 => Ok(Self::S21),
            122 => Ok(Self::S22),
            123 => Ok(Self::S23),
            124 => Ok(Self::S24),
            125 => Ok(Self::S25),
            126 => Ok(Self::S26),
            127 => Ok(Self::S27),
            128 => Ok(Self::S28),
            129 => Ok(Self::S29),
            130 => Ok(Self::S30),
            131 => Ok(Self::S31),
            140 => Ok(Self::D0),
            141 => Ok(Self::D1),
            142 => Ok(Self::D2),
            143 => Ok(Self::D3),
            144 => Ok(Self::D4),
            145 => Ok(Self::D5),
            146 => Ok(Self::D6),
            147 => Ok(Self::D7),
            148 => Ok(Self::D8),
            149 => Ok(Self::D9),
            150 => Ok(Self::D10),
            151 => Ok(Self::D11),
            152 => Ok(Self::D12),
            153 => Ok(Self::D13),
            154 => Ok(Self::D14),
            155 => Ok(Self::D15),
            156 => Ok(Self::D16),
            157 => Ok(Self::D17),
            158 => Ok(Self::D18),
            159 => Ok(Self::D19),
            160 => Ok(Self::D20),
            161 => Ok(Self::D21),
            162 => Ok(Self::D22),
            163 => Ok(Self::D23),
            164 => Ok(Self::D24),
            165 => Ok(Self::D25),
            166 => Ok(Self::D26),
            167 => Ok(Self::D27),
            168 => Ok(Self::D28),
            169 => Ok(Self::D29),
            170 => Ok(Self::D30),
            171 => Ok(Self::D31),
            180 => Ok(Self::Q0),
            181 => Ok(Self::Q1),
            182 => Ok(Self::Q2),
            183 => Ok(Self::Q3),
            184 => Ok(Self::Q4),
            185 => Ok(Self::Q5),
            186 => Ok(Self::Q6),
            187 => Ok(Self::Q7),
            188 => Ok(Self::Q8),
            189 => Ok(Self::Q9),
            190 => Ok(Self::Q10),
            191 => Ok(Self::Q11),
            192 => Ok(Self::Q12),
            193 => Ok(Self::Q13),
            194 => Ok(Self::Q14),
            195 => Ok(Self::Q15),
            196 => Ok(Self::Q16),
            197 => Ok(Self::Q17),
            198 => Ok(Self::Q18),
            199 => Ok(Self::Q19),
            200 => Ok(Self::Q20),
            201 => Ok(Self::Q21),
            202 => Ok(Self::Q22),
            203 => Ok(Self::Q23),
            204 => Ok(Self::Q24),
            205 => Ok(Self::Q25),
            206 => Ok(Self::Q26),
            207 => Ok(Self::Q27),
            208 => Ok(Self::Q28),
            209 => Ok(Self::Q29),
            210 => Ok(Self::Q30),
            211 => Ok(Self::Q31),
            220 => Ok(Self::FPSR),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for ARM64Register {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// Register set for Intel IA64
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum IA64Register {
    // Branch Registers
    Br0 = 512,
    Br1 = 513,
    Br2 = 514,
    Br3 = 515,
    Br4 = 516,
    Br5 = 517,
    Br6 = 518,
    Br7 = 519,

    // Predicate Registers
    P0 = 704,
    P1 = 705,
    P2 = 706,
    P3 = 707,
    P4 = 708,
    P5 = 709,
    P6 = 710,
    P7 = 711,
    P8 = 712,
    P9 = 713,
    P10 = 714,
    P11 = 715,
    P12 = 716,
    P13 = 717,
    P14 = 718,
    P15 = 719,
    P16 = 720,
    P17 = 721,
    P18 = 722,
    P19 = 723,
    P20 = 724,
    P21 = 725,
    P22 = 726,
    P23 = 727,
    P24 = 728,
    P25 = 729,
    P26 = 730,
    P27 = 731,
    P28 = 732,
    P29 = 733,
    P30 = 734,
    P31 = 735,
    P32 = 736,
    P33 = 737,
    P34 = 738,
    P35 = 739,
    P36 = 740,
    P37 = 741,
    P38 = 742,
    P39 = 743,
    P40 = 744,
    P41 = 745,
    P42 = 746,
    P43 = 747,
    P44 = 748,
    P45 = 749,
    P46 = 750,
    P47 = 751,
    P48 = 752,
    P49 = 753,
    P50 = 754,
    P51 = 755,
    P52 = 756,
    P53 = 757,
    P54 = 758,
    P55 = 759,
    P56 = 760,
    P57 = 761,
    P58 = 762,
    P59 = 763,
    P60 = 764,
    P61 = 765,
    P62 = 766,
    P63 = 767,

    Preds = 768,

    // Banked General Registers
    IntH0 = 832,
    IntH1 = 833,
    IntH2 = 834,
    IntH3 = 835,
    IntH4 = 836,
    IntH5 = 837,
    IntH6 = 838,
    IntH7 = 839,
    IntH8 = 840,
    IntH9 = 841,
    IntH10 = 842,
    IntH11 = 843,
    IntH12 = 844,
    IntH13 = 845,
    IntH14 = 846,
    IntH15 = 847,

    // Special Registers
    Ip = 1016,
    Umask = 1017,
    Cfm = 1018,
    Psr = 1019,

    // Banked General Registers
    Nats = 1020,
    Nats2 = 1021,
    Nats3 = 1022,

    // General-Purpose Registers

    // Integer registers
    IntR0 = 1024,
    IntR1 = 1025,
    IntR2 = 1026,
    IntR3 = 1027,
    IntR4 = 1028,
    IntR5 = 1029,
    IntR6 = 1030,
    IntR7 = 1031,
    IntR8 = 1032,
    IntR9 = 1033,
    IntR10 = 1034,
    IntR11 = 1035,
    IntR12 = 1036,
    IntR13 = 1037,
    IntR14 = 1038,
    IntR15 = 1039,
    IntR16 = 1040,
    IntR17 = 1041,
    IntR18 = 1042,
    IntR19 = 1043,
    IntR20 = 1044,
    IntR21 = 1045,
    IntR22 = 1046,
    IntR23 = 1047,
    IntR24 = 1048,
    IntR25 = 1049,
    IntR26 = 1050,
    IntR27 = 1051,
    IntR28 = 1052,
    IntR29 = 1053,
    IntR30 = 1054,
    IntR31 = 1055,

    // Register Stack
    IntR32 = 1056,
    IntR33 = 1057,
    IntR34 = 1058,
    IntR35 = 1059,
    IntR36 = 1060,
    IntR37 = 1061,
    IntR38 = 1062,
    IntR39 = 1063,
    IntR40 = 1064,
    IntR41 = 1065,
    IntR42 = 1066,
    IntR43 = 1067,
    IntR44 = 1068,
    IntR45 = 1069,
    IntR46 = 1070,
    IntR47 = 1071,
    IntR48 = 1072,
    IntR49 = 1073,
    IntR50 = 1074,
    IntR51 = 1075,
    IntR52 = 1076,
    IntR53 = 1077,
    IntR54 = 1078,
    IntR55 = 1079,
    IntR56 = 1080,
    IntR57 = 1081,
    IntR58 = 1082,
    IntR59 = 1083,
    IntR60 = 1084,
    IntR61 = 1085,
    IntR62 = 1086,
    IntR63 = 1087,
    IntR64 = 1088,
    IntR65 = 1089,
    IntR66 = 1090,
    IntR67 = 1091,
    IntR68 = 1092,
    IntR69 = 1093,
    IntR70 = 1094,
    IntR71 = 1095,
    IntR72 = 1096,
    IntR73 = 1097,
    IntR74 = 1098,
    IntR75 = 1099,
    IntR76 = 1100,
    IntR77 = 1101,
    IntR78 = 1102,
    IntR79 = 1103,
    IntR80 = 1104,
    IntR81 = 1105,
    IntR82 = 1106,
    IntR83 = 1107,
    IntR84 = 1108,
    IntR85 = 1109,
    IntR86 = 1110,
    IntR87 = 1111,
    IntR88 = 1112,
    IntR89 = 1113,
    IntR90 = 1114,
    IntR91 = 1115,
    IntR92 = 1116,
    IntR93 = 1117,
    IntR94 = 1118,
    IntR95 = 1119,
    IntR96 = 1120,
    IntR97 = 1121,
    IntR98 = 1122,
    IntR99 = 1123,
    IntR100 = 1124,
    IntR101 = 1125,
    IntR102 = 1126,
    IntR103 = 1127,
    IntR104 = 1128,
    IntR105 = 1129,
    IntR106 = 1130,
    IntR107 = 1131,
    IntR108 = 1132,
    IntR109 = 1133,
    IntR110 = 1134,
    IntR111 = 1135,
    IntR112 = 1136,
    IntR113 = 1137,
    IntR114 = 1138,
    IntR115 = 1139,
    IntR116 = 1140,
    IntR117 = 1141,
    IntR118 = 1142,
    IntR119 = 1143,
    IntR120 = 1144,
    IntR121 = 1145,
    IntR122 = 1146,
    IntR123 = 1147,
    IntR124 = 1148,
    IntR125 = 1149,
    IntR126 = 1150,
    IntR127 = 1151,

    // Floating-Point Registers

    // Low Floating Point Registers
    FltF0 = 2048,
    FltF1 = 2049,
    FltF2 = 2050,
    FltF3 = 2051,
    FltF4 = 2052,
    FltF5 = 2053,
    FltF6 = 2054,
    FltF7 = 2055,
    FltF8 = 2056,
    FltF9 = 2057,
    FltF10 = 2058,
    FltF11 = 2059,
    FltF12 = 2060,
    FltF13 = 2061,
    FltF14 = 2062,
    FltF15 = 2063,
    FltF16 = 2064,
    FltF17 = 2065,
    FltF18 = 2066,
    FltF19 = 2067,
    FltF20 = 2068,
    FltF21 = 2069,
    FltF22 = 2070,
    FltF23 = 2071,
    FltF24 = 2072,
    FltF25 = 2073,
    FltF26 = 2074,
    FltF27 = 2075,
    FltF28 = 2076,
    FltF29 = 2077,
    FltF30 = 2078,
    FltF31 = 2079,

    // High Floating Point Registers
    FltF32 = 2080,
    FltF33 = 2081,
    FltF34 = 2082,
    FltF35 = 2083,
    FltF36 = 2084,
    FltF37 = 2085,
    FltF38 = 2086,
    FltF39 = 2087,
    FltF40 = 2088,
    FltF41 = 2089,
    FltF42 = 2090,
    FltF43 = 2091,
    FltF44 = 2092,
    FltF45 = 2093,
    FltF46 = 2094,
    FltF47 = 2095,
    FltF48 = 2096,
    FltF49 = 2097,
    FltF50 = 2098,
    FltF51 = 2099,
    FltF52 = 2100,
    FltF53 = 2101,
    FltF54 = 2102,
    FltF55 = 2103,
    FltF56 = 2104,
    FltF57 = 2105,
    FltF58 = 2106,
    FltF59 = 2107,
    FltF60 = 2108,
    FltF61 = 2109,
    FltF62 = 2110,
    FltF63 = 2111,
    FltF64 = 2112,
    FltF65 = 2113,
    FltF66 = 2114,
    FltF67 = 2115,
    FltF68 = 2116,
    FltF69 = 2117,
    FltF70 = 2118,
    FltF71 = 2119,
    FltF72 = 2120,
    FltF73 = 2121,
    FltF74 = 2122,
    FltF75 = 2123,
    FltF76 = 2124,
    FltF77 = 2125,
    FltF78 = 2126,
    FltF79 = 2127,
    FltF80 = 2128,
    FltF81 = 2129,
    FltF82 = 2130,
    FltF83 = 2131,
    FltF84 = 2132,
    FltF85 = 2133,
    FltF86 = 2134,
    FltF87 = 2135,
    FltF88 = 2136,
    FltF89 = 2137,
    FltF90 = 2138,
    FltF91 = 2139,
    FltF92 = 2140,
    FltF93 = 2141,
    FltF94 = 2142,
    FltF95 = 2143,
    FltF96 = 2144,
    FltF97 = 2145,
    FltF98 = 2146,
    FltF99 = 2147,
    FltF100 = 2148,
    FltF101 = 2149,
    FltF102 = 2150,
    FltF103 = 2151,
    FltF104 = 2152,
    FltF105 = 2153,
    FltF106 = 2154,
    FltF107 = 2155,
    FltF108 = 2156,
    FltF109 = 2157,
    FltF110 = 2158,
    FltF111 = 2159,
    FltF112 = 2160,
    FltF113 = 2161,
    FltF114 = 2162,
    FltF115 = 2163,
    FltF116 = 2164,
    FltF117 = 2165,
    FltF118 = 2166,
    FltF119 = 2167,
    FltF120 = 2168,
    FltF121 = 2169,
    FltF122 = 2170,
    FltF123 = 2171,
    FltF124 = 2172,
    FltF125 = 2173,
    FltF126 = 2174,
    FltF127 = 2175,

    // Application Registers
    ApKR0 = 3072,
    ApKR1 = 3073,
    ApKR2 = 3074,
    ApKR3 = 3075,
    ApKR4 = 3076,
    ApKR5 = 3077,
    ApKR6 = 3078,
    ApKR7 = 3079,
    AR8 = 3080,
    AR9 = 3081,
    AR10 = 3082,
    AR11 = 3083,
    AR12 = 3084,
    AR13 = 3085,
    AR14 = 3086,
    AR15 = 3087,
    RsRSC = 3088,
    RsBSP = 3089,
    RsBSPSTORE = 3090,
    RsRNAT = 3091,
    AR20 = 3092,
    StFCR = 3093,
    AR22 = 3094,
    AR23 = 3095,
    EFLAG = 3096,
    CSD = 3097,
    SSD = 3098,
    CFLG = 3099,
    StFSR = 3100,
    StFIR = 3101,
    StFDR = 3102,
    AR31 = 3103,
    ApCCV = 3104,
    AR33 = 3105,
    AR34 = 3106,
    AR35 = 3107,
    ApUNAT = 3108,
    AR37 = 3109,
    AR38 = 3110,
    AR39 = 3111,
    StFPSR = 3112,
    AR41 = 3113,
    AR42 = 3114,
    AR43 = 3115,
    ApITC = 3116,
    AR45 = 3117,
    AR46 = 3118,
    AR47 = 3119,
    AR48 = 3120,
    AR49 = 3121,
    AR50 = 3122,
    AR51 = 3123,
    AR52 = 3124,
    AR53 = 3125,
    AR54 = 3126,
    AR55 = 3127,
    AR56 = 3128,
    AR57 = 3129,
    AR58 = 3130,
    AR59 = 3131,
    AR60 = 3132,
    AR61 = 3133,
    AR62 = 3134,
    AR63 = 3135,
    RsPFS = 3136,
    ApLC = 3137,
    ApEC = 3138,
    AR67 = 3139,
    AR68 = 3140,
    AR69 = 3141,
    AR70 = 3142,
    AR71 = 3143,
    AR72 = 3144,
    AR73 = 3145,
    AR74 = 3146,
    AR75 = 3147,
    AR76 = 3148,
    AR77 = 3149,
    AR78 = 3150,
    AR79 = 3151,
    AR80 = 3152,
    AR81 = 3153,
    AR82 = 3154,
    AR83 = 3155,
    AR84 = 3156,
    AR85 = 3157,
    AR86 = 3158,
    AR87 = 3159,
    AR88 = 3160,
    AR89 = 3161,
    AR90 = 3162,
    AR91 = 3163,
    AR92 = 3164,
    AR93 = 3165,
    AR94 = 3166,
    AR95 = 3167,
    AR96 = 3168,
    AR97 = 3169,
    AR98 = 3170,
    AR99 = 3171,
    AR100 = 3172,
    AR101 = 3173,
    AR102 = 3174,
    AR103 = 3175,
    AR104 = 3176,
    AR105 = 3177,
    AR106 = 3178,
    AR107 = 3179,
    AR108 = 3180,
    AR109 = 3181,
    AR110 = 3182,
    AR111 = 3183,
    AR112 = 3184,
    AR113 = 3185,
    AR114 = 3186,
    AR115 = 3187,
    AR116 = 3188,
    AR117 = 3189,
    AR118 = 3190,
    AR119 = 3191,
    AR120 = 3192,
    AR121 = 3193,
    AR122 = 3194,
    AR123 = 3195,
    AR124 = 3196,
    AR125 = 3197,
    AR126 = 3198,
    AR127 = 3199,

    // CPUID Registers
    CPUID0 = 3328,
    CPUID1 = 3329,
    CPUID2 = 3330,
    CPUID3 = 3331,
    CPUID4 = 3332,

    // Control Registers
    ApDCR = 4096,
    ApITM = 4097,
    ApIVA = 4098,
    CR3 = 4099,
    CR4 = 4100,
    CR5 = 4101,
    CR6 = 4102,
    CR7 = 4103,
    ApPTA = 4104,
    ApGPTA = 4105,
    CR10 = 4106,
    CR11 = 4107,
    CR12 = 4108,
    CR13 = 4109,
    CR14 = 4110,
    CR15 = 4111,
    StIPSR = 4112,
    StISR = 4113,
    CR18 = 4114,
    StIIP = 4115,
    StIFA = 4116,
    StITIR = 4117,
    StIIPA = 4118,
    StIFS = 4119,
    StIIM = 4120,
    StIHA = 4121,
    CR26 = 4122,
    CR27 = 4123,
    CR28 = 4124,
    CR29 = 4125,
    CR30 = 4126,
    CR31 = 4127,
    CR32 = 4128,
    CR33 = 4129,
    CR34 = 4130,
    CR35 = 4131,
    CR36 = 4132,
    CR37 = 4133,
    CR38 = 4134,
    CR39 = 4135,
    CR40 = 4136,
    CR41 = 4137,
    CR42 = 4138,
    CR43 = 4139,
    CR44 = 4140,
    CR45 = 4141,
    CR46 = 4142,
    CR47 = 4143,
    CR48 = 4144,
    CR49 = 4145,
    CR50 = 4146,
    CR51 = 4147,
    CR52 = 4148,
    CR53 = 4149,
    CR54 = 4150,
    CR55 = 4151,
    CR56 = 4152,
    CR57 = 4153,
    CR58 = 4154,
    CR59 = 4155,
    CR60 = 4156,
    CR61 = 4157,
    CR62 = 4158,
    CR63 = 4159,
    SaLID = 4160,
    SaIVR = 4161,
    SaTPR = 4162,
    SaEOI = 4163,
    SaIRR0 = 4164,
    SaIRR1 = 4165,
    SaIRR2 = 4166,
    SaIRR3 = 4167,
    SaITV = 4168,
    SaPMV = 4169,
    SaCMCV = 4170,
    CR75 = 4171,
    CR76 = 4172,
    CR77 = 4173,
    CR78 = 4174,
    CR79 = 4175,
    SaLRR0 = 4176,
    SaLRR1 = 4177,
    CR82 = 4178,
    CR83 = 4179,
    CR84 = 4180,
    CR85 = 4181,
    CR86 = 4182,
    CR87 = 4183,
    CR88 = 4184,
    CR89 = 4185,
    CR90 = 4186,
    CR91 = 4187,
    CR92 = 4188,
    CR93 = 4189,
    CR94 = 4190,
    CR95 = 4191,
    CR96 = 4192,
    CR97 = 4193,
    CR98 = 4194,
    CR99 = 4195,
    CR100 = 4196,
    CR101 = 4197,
    CR102 = 4198,
    CR103 = 4199,
    CR104 = 4200,
    CR105 = 4201,
    CR106 = 4202,
    CR107 = 4203,
    CR108 = 4204,
    CR109 = 4205,
    CR110 = 4206,
    CR111 = 4207,
    CR112 = 4208,
    CR113 = 4209,
    CR114 = 4210,
    CR115 = 4211,
    CR116 = 4212,
    CR117 = 4213,
    CR118 = 4214,
    CR119 = 4215,
    CR120 = 4216,
    CR121 = 4217,
    CR122 = 4218,
    CR123 = 4219,
    CR124 = 4220,
    CR125 = 4221,
    CR126 = 4222,
    CR127 = 4223,

    // Protection Key Registers
    Pkr0 = 5120,
    Pkr1 = 5121,
    Pkr2 = 5122,
    Pkr3 = 5123,
    Pkr4 = 5124,
    Pkr5 = 5125,
    Pkr6 = 5126,
    Pkr7 = 5127,
    Pkr8 = 5128,
    Pkr9 = 5129,
    Pkr10 = 5130,
    Pkr11 = 5131,
    Pkr12 = 5132,
    Pkr13 = 5133,
    Pkr14 = 5134,
    Pkr15 = 5135,

    // Region Registers
    Rr0 = 6144,
    Rr1 = 6145,
    Rr2 = 6146,
    Rr3 = 6147,
    Rr4 = 6148,
    Rr5 = 6149,
    Rr6 = 6150,
    Rr7 = 6151,

    // Performance Monitor Data Registers
    PFD0 = 7168,
    PFD1 = 7169,
    PFD2 = 7170,
    PFD3 = 7171,
    PFD4 = 7172,
    PFD5 = 7173,
    PFD6 = 7174,
    PFD7 = 7175,
    PFD8 = 7176,
    PFD9 = 7177,
    PFD10 = 7178,
    PFD11 = 7179,
    PFD12 = 7180,
    PFD13 = 7181,
    PFD14 = 7182,
    PFD15 = 7183,
    PFD16 = 7184,
    PFD17 = 7185,

    // Performance Monitor Config Registers
    PFC0 = 7424,
    PFC1 = 7425,
    PFC2 = 7426,
    PFC3 = 7427,
    PFC4 = 7428,
    PFC5 = 7429,
    PFC6 = 7430,
    PFC7 = 7431,
    PFC8 = 7432,
    PFC9 = 7433,
    PFC10 = 7434,
    PFC11 = 7435,
    PFC12 = 7436,
    PFC13 = 7437,
    PFC14 = 7438,
    PFC15 = 7439,

    // Instruction Translation Registers
    TrI0 = 8192,
    TrI1 = 8193,
    TrI2 = 8194,
    TrI3 = 8195,
    TrI4 = 8196,
    TrI5 = 8197,
    TrI6 = 8198,
    TrI7 = 8199,

    // Data Translation Registers
    TrD0 = 8320,
    TrD1 = 8321,
    TrD2 = 8322,
    TrD3 = 8323,
    TrD4 = 8324,
    TrD5 = 8325,
    TrD6 = 8326,
    TrD7 = 8327,

    // Instruction Breakpoint Registers
    DbI0 = 8448,
    DbI1 = 8449,
    DbI2 = 8450,
    DbI3 = 8451,
    DbI4 = 8452,
    DbI5 = 8453,
    DbI6 = 8454,
    DbI7 = 8455,

    // Data Breakpoint Registers
    DbD0 = 8576,
    DbD1 = 8577,
    DbD2 = 8578,
    DbD3 = 8579,
    DbD4 = 8580,
    DbD5 = 8581,
    DbD6 = 8582,
    DbD7 = 8583,
}

impl fmt::Display for IA64Register {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::Br0 => write!(f, "Br0"),
            Self::Br1 => write!(f, "Br1"),
            Self::Br2 => write!(f, "Br2"),
            Self::Br3 => write!(f, "Br3"),
            Self::Br4 => write!(f, "Br4"),
            Self::Br5 => write!(f, "Br5"),
            Self::Br6 => write!(f, "Br6"),
            Self::Br7 => write!(f, "Br7"),
            Self::P0 => write!(f, "P0"),
            Self::P1 => write!(f, "P1"),
            Self::P2 => write!(f, "P2"),
            Self::P3 => write!(f, "P3"),
            Self::P4 => write!(f, "P4"),
            Self::P5 => write!(f, "P5"),
            Self::P6 => write!(f, "P6"),
            Self::P7 => write!(f, "P7"),
            Self::P8 => write!(f, "P8"),
            Self::P9 => write!(f, "P9"),
            Self::P10 => write!(f, "P10"),
            Self::P11 => write!(f, "P11"),
            Self::P12 => write!(f, "P12"),
            Self::P13 => write!(f, "P13"),
            Self::P14 => write!(f, "P14"),
            Self::P15 => write!(f, "P15"),
            Self::P16 => write!(f, "P16"),
            Self::P17 => write!(f, "P17"),
            Self::P18 => write!(f, "P18"),
            Self::P19 => write!(f, "P19"),
            Self::P20 => write!(f, "P20"),
            Self::P21 => write!(f, "P21"),
            Self::P22 => write!(f, "P22"),
            Self::P23 => write!(f, "P23"),
            Self::P24 => write!(f, "P24"),
            Self::P25 => write!(f, "P25"),
            Self::P26 => write!(f, "P26"),
            Self::P27 => write!(f, "P27"),
            Self::P28 => write!(f, "P28"),
            Self::P29 => write!(f, "P29"),
            Self::P30 => write!(f, "P30"),
            Self::P31 => write!(f, "P31"),
            Self::P32 => write!(f, "P32"),
            Self::P33 => write!(f, "P33"),
            Self::P34 => write!(f, "P34"),
            Self::P35 => write!(f, "P35"),
            Self::P36 => write!(f, "P36"),
            Self::P37 => write!(f, "P37"),
            Self::P38 => write!(f, "P38"),
            Self::P39 => write!(f, "P39"),
            Self::P40 => write!(f, "P40"),
            Self::P41 => write!(f, "P41"),
            Self::P42 => write!(f, "P42"),
            Self::P43 => write!(f, "P43"),
            Self::P44 => write!(f, "P44"),
            Self::P45 => write!(f, "P45"),
            Self::P46 => write!(f, "P46"),
            Self::P47 => write!(f, "P47"),
            Self::P48 => write!(f, "P48"),
            Self::P49 => write!(f, "P49"),
            Self::P50 => write!(f, "P50"),
            Self::P51 => write!(f, "P51"),
            Self::P52 => write!(f, "P52"),
            Self::P53 => write!(f, "P53"),
            Self::P54 => write!(f, "P54"),
            Self::P55 => write!(f, "P55"),
            Self::P56 => write!(f, "P56"),
            Self::P57 => write!(f, "P57"),
            Self::P58 => write!(f, "P58"),
            Self::P59 => write!(f, "P59"),
            Self::P60 => write!(f, "P60"),
            Self::P61 => write!(f, "P61"),
            Self::P62 => write!(f, "P62"),
            Self::P63 => write!(f, "P63"),
            Self::Preds => write!(f, "Preds"),
            Self::IntH0 => write!(f, "IntH0"),
            Self::IntH1 => write!(f, "IntH1"),
            Self::IntH2 => write!(f, "IntH2"),
            Self::IntH3 => write!(f, "IntH3"),
            Self::IntH4 => write!(f, "IntH4"),
            Self::IntH5 => write!(f, "IntH5"),
            Self::IntH6 => write!(f, "IntH6"),
            Self::IntH7 => write!(f, "IntH7"),
            Self::IntH8 => write!(f, "IntH8"),
            Self::IntH9 => write!(f, "IntH9"),
            Self::IntH10 => write!(f, "IntH10"),
            Self::IntH11 => write!(f, "IntH11"),
            Self::IntH12 => write!(f, "IntH12"),
            Self::IntH13 => write!(f, "IntH13"),
            Self::IntH14 => write!(f, "IntH14"),
            Self::IntH15 => write!(f, "IntH15"),
            Self::Ip => write!(f, "Ip"),
            Self::Umask => write!(f, "Umask"),
            Self::Cfm => write!(f, "Cfm"),
            Self::Psr => write!(f, "Psr"),
            Self::Nats => write!(f, "Nats"),
            Self::Nats2 => write!(f, "Nats2"),
            Self::Nats3 => write!(f, "Nats3"),
            Self::IntR0 => write!(f, "IntR0"),
            Self::IntR1 => write!(f, "IntR1"),
            Self::IntR2 => write!(f, "IntR2"),
            Self::IntR3 => write!(f, "IntR3"),
            Self::IntR4 => write!(f, "IntR4"),
            Self::IntR5 => write!(f, "IntR5"),
            Self::IntR6 => write!(f, "IntR6"),
            Self::IntR7 => write!(f, "IntR7"),
            Self::IntR8 => write!(f, "IntR8"),
            Self::IntR9 => write!(f, "IntR9"),
            Self::IntR10 => write!(f, "IntR10"),
            Self::IntR11 => write!(f, "IntR11"),
            Self::IntR12 => write!(f, "IntR12"),
            Self::IntR13 => write!(f, "IntR13"),
            Self::IntR14 => write!(f, "IntR14"),
            Self::IntR15 => write!(f, "IntR15"),
            Self::IntR16 => write!(f, "IntR16"),
            Self::IntR17 => write!(f, "IntR17"),
            Self::IntR18 => write!(f, "IntR18"),
            Self::IntR19 => write!(f, "IntR19"),
            Self::IntR20 => write!(f, "IntR20"),
            Self::IntR21 => write!(f, "IntR21"),
            Self::IntR22 => write!(f, "IntR22"),
            Self::IntR23 => write!(f, "IntR23"),
            Self::IntR24 => write!(f, "IntR24"),
            Self::IntR25 => write!(f, "IntR25"),
            Self::IntR26 => write!(f, "IntR26"),
            Self::IntR27 => write!(f, "IntR27"),
            Self::IntR28 => write!(f, "IntR28"),
            Self::IntR29 => write!(f, "IntR29"),
            Self::IntR30 => write!(f, "IntR30"),
            Self::IntR31 => write!(f, "IntR31"),
            Self::IntR32 => write!(f, "IntR32"),
            Self::IntR33 => write!(f, "IntR33"),
            Self::IntR34 => write!(f, "IntR34"),
            Self::IntR35 => write!(f, "IntR35"),
            Self::IntR36 => write!(f, "IntR36"),
            Self::IntR37 => write!(f, "IntR37"),
            Self::IntR38 => write!(f, "IntR38"),
            Self::IntR39 => write!(f, "IntR39"),
            Self::IntR40 => write!(f, "IntR40"),
            Self::IntR41 => write!(f, "IntR41"),
            Self::IntR42 => write!(f, "IntR42"),
            Self::IntR43 => write!(f, "IntR43"),
            Self::IntR44 => write!(f, "IntR44"),
            Self::IntR45 => write!(f, "IntR45"),
            Self::IntR46 => write!(f, "IntR46"),
            Self::IntR47 => write!(f, "IntR47"),
            Self::IntR48 => write!(f, "IntR48"),
            Self::IntR49 => write!(f, "IntR49"),
            Self::IntR50 => write!(f, "IntR50"),
            Self::IntR51 => write!(f, "IntR51"),
            Self::IntR52 => write!(f, "IntR52"),
            Self::IntR53 => write!(f, "IntR53"),
            Self::IntR54 => write!(f, "IntR54"),
            Self::IntR55 => write!(f, "IntR55"),
            Self::IntR56 => write!(f, "IntR56"),
            Self::IntR57 => write!(f, "IntR57"),
            Self::IntR58 => write!(f, "IntR58"),
            Self::IntR59 => write!(f, "IntR59"),
            Self::IntR60 => write!(f, "IntR60"),
            Self::IntR61 => write!(f, "IntR61"),
            Self::IntR62 => write!(f, "IntR62"),
            Self::IntR63 => write!(f, "IntR63"),
            Self::IntR64 => write!(f, "IntR64"),
            Self::IntR65 => write!(f, "IntR65"),
            Self::IntR66 => write!(f, "IntR66"),
            Self::IntR67 => write!(f, "IntR67"),
            Self::IntR68 => write!(f, "IntR68"),
            Self::IntR69 => write!(f, "IntR69"),
            Self::IntR70 => write!(f, "IntR70"),
            Self::IntR71 => write!(f, "IntR71"),
            Self::IntR72 => write!(f, "IntR72"),
            Self::IntR73 => write!(f, "IntR73"),
            Self::IntR74 => write!(f, "IntR74"),
            Self::IntR75 => write!(f, "IntR75"),
            Self::IntR76 => write!(f, "IntR76"),
            Self::IntR77 => write!(f, "IntR77"),
            Self::IntR78 => write!(f, "IntR78"),
            Self::IntR79 => write!(f, "IntR79"),
            Self::IntR80 => write!(f, "IntR80"),
            Self::IntR81 => write!(f, "IntR81"),
            Self::IntR82 => write!(f, "IntR82"),
            Self::IntR83 => write!(f, "IntR83"),
            Self::IntR84 => write!(f, "IntR84"),
            Self::IntR85 => write!(f, "IntR85"),
            Self::IntR86 => write!(f, "IntR86"),
            Self::IntR87 => write!(f, "IntR87"),
            Self::IntR88 => write!(f, "IntR88"),
            Self::IntR89 => write!(f, "IntR89"),
            Self::IntR90 => write!(f, "IntR90"),
            Self::IntR91 => write!(f, "IntR91"),
            Self::IntR92 => write!(f, "IntR92"),
            Self::IntR93 => write!(f, "IntR93"),
            Self::IntR94 => write!(f, "IntR94"),
            Self::IntR95 => write!(f, "IntR95"),
            Self::IntR96 => write!(f, "IntR96"),
            Self::IntR97 => write!(f, "IntR97"),
            Self::IntR98 => write!(f, "IntR98"),
            Self::IntR99 => write!(f, "IntR99"),
            Self::IntR100 => write!(f, "IntR100"),
            Self::IntR101 => write!(f, "IntR101"),
            Self::IntR102 => write!(f, "IntR102"),
            Self::IntR103 => write!(f, "IntR103"),
            Self::IntR104 => write!(f, "IntR104"),
            Self::IntR105 => write!(f, "IntR105"),
            Self::IntR106 => write!(f, "IntR106"),
            Self::IntR107 => write!(f, "IntR107"),
            Self::IntR108 => write!(f, "IntR108"),
            Self::IntR109 => write!(f, "IntR109"),
            Self::IntR110 => write!(f, "IntR110"),
            Self::IntR111 => write!(f, "IntR111"),
            Self::IntR112 => write!(f, "IntR112"),
            Self::IntR113 => write!(f, "IntR113"),
            Self::IntR114 => write!(f, "IntR114"),
            Self::IntR115 => write!(f, "IntR115"),
            Self::IntR116 => write!(f, "IntR116"),
            Self::IntR117 => write!(f, "IntR117"),
            Self::IntR118 => write!(f, "IntR118"),
            Self::IntR119 => write!(f, "IntR119"),
            Self::IntR120 => write!(f, "IntR120"),
            Self::IntR121 => write!(f, "IntR121"),
            Self::IntR122 => write!(f, "IntR122"),
            Self::IntR123 => write!(f, "IntR123"),
            Self::IntR124 => write!(f, "IntR124"),
            Self::IntR125 => write!(f, "IntR125"),
            Self::IntR126 => write!(f, "IntR126"),
            Self::IntR127 => write!(f, "IntR127"),
            Self::FltF0 => write!(f, "FltF0"),
            Self::FltF1 => write!(f, "FltF1"),
            Self::FltF2 => write!(f, "FltF2"),
            Self::FltF3 => write!(f, "FltF3"),
            Self::FltF4 => write!(f, "FltF4"),
            Self::FltF5 => write!(f, "FltF5"),
            Self::FltF6 => write!(f, "FltF6"),
            Self::FltF7 => write!(f, "FltF7"),
            Self::FltF8 => write!(f, "FltF8"),
            Self::FltF9 => write!(f, "FltF9"),
            Self::FltF10 => write!(f, "FltF10"),
            Self::FltF11 => write!(f, "FltF11"),
            Self::FltF12 => write!(f, "FltF12"),
            Self::FltF13 => write!(f, "FltF13"),
            Self::FltF14 => write!(f, "FltF14"),
            Self::FltF15 => write!(f, "FltF15"),
            Self::FltF16 => write!(f, "FltF16"),
            Self::FltF17 => write!(f, "FltF17"),
            Self::FltF18 => write!(f, "FltF18"),
            Self::FltF19 => write!(f, "FltF19"),
            Self::FltF20 => write!(f, "FltF20"),
            Self::FltF21 => write!(f, "FltF21"),
            Self::FltF22 => write!(f, "FltF22"),
            Self::FltF23 => write!(f, "FltF23"),
            Self::FltF24 => write!(f, "FltF24"),
            Self::FltF25 => write!(f, "FltF25"),
            Self::FltF26 => write!(f, "FltF26"),
            Self::FltF27 => write!(f, "FltF27"),
            Self::FltF28 => write!(f, "FltF28"),
            Self::FltF29 => write!(f, "FltF29"),
            Self::FltF30 => write!(f, "FltF30"),
            Self::FltF31 => write!(f, "FltF31"),
            Self::FltF32 => write!(f, "FltF32"),
            Self::FltF33 => write!(f, "FltF33"),
            Self::FltF34 => write!(f, "FltF34"),
            Self::FltF35 => write!(f, "FltF35"),
            Self::FltF36 => write!(f, "FltF36"),
            Self::FltF37 => write!(f, "FltF37"),
            Self::FltF38 => write!(f, "FltF38"),
            Self::FltF39 => write!(f, "FltF39"),
            Self::FltF40 => write!(f, "FltF40"),
            Self::FltF41 => write!(f, "FltF41"),
            Self::FltF42 => write!(f, "FltF42"),
            Self::FltF43 => write!(f, "FltF43"),
            Self::FltF44 => write!(f, "FltF44"),
            Self::FltF45 => write!(f, "FltF45"),
            Self::FltF46 => write!(f, "FltF46"),
            Self::FltF47 => write!(f, "FltF47"),
            Self::FltF48 => write!(f, "FltF48"),
            Self::FltF49 => write!(f, "FltF49"),
            Self::FltF50 => write!(f, "FltF50"),
            Self::FltF51 => write!(f, "FltF51"),
            Self::FltF52 => write!(f, "FltF52"),
            Self::FltF53 => write!(f, "FltF53"),
            Self::FltF54 => write!(f, "FltF54"),
            Self::FltF55 => write!(f, "FltF55"),
            Self::FltF56 => write!(f, "FltF56"),
            Self::FltF57 => write!(f, "FltF57"),
            Self::FltF58 => write!(f, "FltF58"),
            Self::FltF59 => write!(f, "FltF59"),
            Self::FltF60 => write!(f, "FltF60"),
            Self::FltF61 => write!(f, "FltF61"),
            Self::FltF62 => write!(f, "FltF62"),
            Self::FltF63 => write!(f, "FltF63"),
            Self::FltF64 => write!(f, "FltF64"),
            Self::FltF65 => write!(f, "FltF65"),
            Self::FltF66 => write!(f, "FltF66"),
            Self::FltF67 => write!(f, "FltF67"),
            Self::FltF68 => write!(f, "FltF68"),
            Self::FltF69 => write!(f, "FltF69"),
            Self::FltF70 => write!(f, "FltF70"),
            Self::FltF71 => write!(f, "FltF71"),
            Self::FltF72 => write!(f, "FltF72"),
            Self::FltF73 => write!(f, "FltF73"),
            Self::FltF74 => write!(f, "FltF74"),
            Self::FltF75 => write!(f, "FltF75"),
            Self::FltF76 => write!(f, "FltF76"),
            Self::FltF77 => write!(f, "FltF77"),
            Self::FltF78 => write!(f, "FltF78"),
            Self::FltF79 => write!(f, "FltF79"),
            Self::FltF80 => write!(f, "FltF80"),
            Self::FltF81 => write!(f, "FltF81"),
            Self::FltF82 => write!(f, "FltF82"),
            Self::FltF83 => write!(f, "FltF83"),
            Self::FltF84 => write!(f, "FltF84"),
            Self::FltF85 => write!(f, "FltF85"),
            Self::FltF86 => write!(f, "FltF86"),
            Self::FltF87 => write!(f, "FltF87"),
            Self::FltF88 => write!(f, "FltF88"),
            Self::FltF89 => write!(f, "FltF89"),
            Self::FltF90 => write!(f, "FltF90"),
            Self::FltF91 => write!(f, "FltF91"),
            Self::FltF92 => write!(f, "FltF92"),
            Self::FltF93 => write!(f, "FltF93"),
            Self::FltF94 => write!(f, "FltF94"),
            Self::FltF95 => write!(f, "FltF95"),
            Self::FltF96 => write!(f, "FltF96"),
            Self::FltF97 => write!(f, "FltF97"),
            Self::FltF98 => write!(f, "FltF98"),
            Self::FltF99 => write!(f, "FltF99"),
            Self::FltF100 => write!(f, "FltF100"),
            Self::FltF101 => write!(f, "FltF101"),
            Self::FltF102 => write!(f, "FltF102"),
            Self::FltF103 => write!(f, "FltF103"),
            Self::FltF104 => write!(f, "FltF104"),
            Self::FltF105 => write!(f, "FltF105"),
            Self::FltF106 => write!(f, "FltF106"),
            Self::FltF107 => write!(f, "FltF107"),
            Self::FltF108 => write!(f, "FltF108"),
            Self::FltF109 => write!(f, "FltF109"),
            Self::FltF110 => write!(f, "FltF110"),
            Self::FltF111 => write!(f, "FltF111"),
            Self::FltF112 => write!(f, "FltF112"),
            Self::FltF113 => write!(f, "FltF113"),
            Self::FltF114 => write!(f, "FltF114"),
            Self::FltF115 => write!(f, "FltF115"),
            Self::FltF116 => write!(f, "FltF116"),
            Self::FltF117 => write!(f, "FltF117"),
            Self::FltF118 => write!(f, "FltF118"),
            Self::FltF119 => write!(f, "FltF119"),
            Self::FltF120 => write!(f, "FltF120"),
            Self::FltF121 => write!(f, "FltF121"),
            Self::FltF122 => write!(f, "FltF122"),
            Self::FltF123 => write!(f, "FltF123"),
            Self::FltF124 => write!(f, "FltF124"),
            Self::FltF125 => write!(f, "FltF125"),
            Self::FltF126 => write!(f, "FltF126"),
            Self::FltF127 => write!(f, "FltF127"),
            Self::ApKR0 => write!(f, "ApKR0"),
            Self::ApKR1 => write!(f, "ApKR1"),
            Self::ApKR2 => write!(f, "ApKR2"),
            Self::ApKR3 => write!(f, "ApKR3"),
            Self::ApKR4 => write!(f, "ApKR4"),
            Self::ApKR5 => write!(f, "ApKR5"),
            Self::ApKR6 => write!(f, "ApKR6"),
            Self::ApKR7 => write!(f, "ApKR7"),
            Self::AR8 => write!(f, "AR8"),
            Self::AR9 => write!(f, "AR9"),
            Self::AR10 => write!(f, "AR10"),
            Self::AR11 => write!(f, "AR11"),
            Self::AR12 => write!(f, "AR12"),
            Self::AR13 => write!(f, "AR13"),
            Self::AR14 => write!(f, "AR14"),
            Self::AR15 => write!(f, "AR15"),
            Self::RsRSC => write!(f, "RsRSC"),
            Self::RsBSP => write!(f, "RsBSP"),
            Self::RsBSPSTORE => write!(f, "RsBSPSTORE"),
            Self::RsRNAT => write!(f, "RsRNAT"),
            Self::AR20 => write!(f, "AR20"),
            Self::StFCR => write!(f, "StFCR"),
            Self::AR22 => write!(f, "AR22"),
            Self::AR23 => write!(f, "AR23"),
            Self::EFLAG => write!(f, "EFLAG"),
            Self::CSD => write!(f, "CSD"),
            Self::SSD => write!(f, "SSD"),
            Self::CFLG => write!(f, "CFLG"),
            Self::StFSR => write!(f, "StFSR"),
            Self::StFIR => write!(f, "StFIR"),
            Self::StFDR => write!(f, "StFDR"),
            Self::AR31 => write!(f, "AR31"),
            Self::ApCCV => write!(f, "ApCCV"),
            Self::AR33 => write!(f, "AR33"),
            Self::AR34 => write!(f, "AR34"),
            Self::AR35 => write!(f, "AR35"),
            Self::ApUNAT => write!(f, "ApUNAT"),
            Self::AR37 => write!(f, "AR37"),
            Self::AR38 => write!(f, "AR38"),
            Self::AR39 => write!(f, "AR39"),
            Self::StFPSR => write!(f, "StFPSR"),
            Self::AR41 => write!(f, "AR41"),
            Self::AR42 => write!(f, "AR42"),
            Self::AR43 => write!(f, "AR43"),
            Self::ApITC => write!(f, "ApITC"),
            Self::AR45 => write!(f, "AR45"),
            Self::AR46 => write!(f, "AR46"),
            Self::AR47 => write!(f, "AR47"),
            Self::AR48 => write!(f, "AR48"),
            Self::AR49 => write!(f, "AR49"),
            Self::AR50 => write!(f, "AR50"),
            Self::AR51 => write!(f, "AR51"),
            Self::AR52 => write!(f, "AR52"),
            Self::AR53 => write!(f, "AR53"),
            Self::AR54 => write!(f, "AR54"),
            Self::AR55 => write!(f, "AR55"),
            Self::AR56 => write!(f, "AR56"),
            Self::AR57 => write!(f, "AR57"),
            Self::AR58 => write!(f, "AR58"),
            Self::AR59 => write!(f, "AR59"),
            Self::AR60 => write!(f, "AR60"),
            Self::AR61 => write!(f, "AR61"),
            Self::AR62 => write!(f, "AR62"),
            Self::AR63 => write!(f, "AR63"),
            Self::RsPFS => write!(f, "RsPFS"),
            Self::ApLC => write!(f, "ApLC"),
            Self::ApEC => write!(f, "ApEC"),
            Self::AR67 => write!(f, "AR67"),
            Self::AR68 => write!(f, "AR68"),
            Self::AR69 => write!(f, "AR69"),
            Self::AR70 => write!(f, "AR70"),
            Self::AR71 => write!(f, "AR71"),
            Self::AR72 => write!(f, "AR72"),
            Self::AR73 => write!(f, "AR73"),
            Self::AR74 => write!(f, "AR74"),
            Self::AR75 => write!(f, "AR75"),
            Self::AR76 => write!(f, "AR76"),
            Self::AR77 => write!(f, "AR77"),
            Self::AR78 => write!(f, "AR78"),
            Self::AR79 => write!(f, "AR79"),
            Self::AR80 => write!(f, "AR80"),
            Self::AR81 => write!(f, "AR81"),
            Self::AR82 => write!(f, "AR82"),
            Self::AR83 => write!(f, "AR83"),
            Self::AR84 => write!(f, "AR84"),
            Self::AR85 => write!(f, "AR85"),
            Self::AR86 => write!(f, "AR86"),
            Self::AR87 => write!(f, "AR87"),
            Self::AR88 => write!(f, "AR88"),
            Self::AR89 => write!(f, "AR89"),
            Self::AR90 => write!(f, "AR90"),
            Self::AR91 => write!(f, "AR91"),
            Self::AR92 => write!(f, "AR92"),
            Self::AR93 => write!(f, "AR93"),
            Self::AR94 => write!(f, "AR94"),
            Self::AR95 => write!(f, "AR95"),
            Self::AR96 => write!(f, "AR96"),
            Self::AR97 => write!(f, "AR97"),
            Self::AR98 => write!(f, "AR98"),
            Self::AR99 => write!(f, "AR99"),
            Self::AR100 => write!(f, "AR100"),
            Self::AR101 => write!(f, "AR101"),
            Self::AR102 => write!(f, "AR102"),
            Self::AR103 => write!(f, "AR103"),
            Self::AR104 => write!(f, "AR104"),
            Self::AR105 => write!(f, "AR105"),
            Self::AR106 => write!(f, "AR106"),
            Self::AR107 => write!(f, "AR107"),
            Self::AR108 => write!(f, "AR108"),
            Self::AR109 => write!(f, "AR109"),
            Self::AR110 => write!(f, "AR110"),
            Self::AR111 => write!(f, "AR111"),
            Self::AR112 => write!(f, "AR112"),
            Self::AR113 => write!(f, "AR113"),
            Self::AR114 => write!(f, "AR114"),
            Self::AR115 => write!(f, "AR115"),
            Self::AR116 => write!(f, "AR116"),
            Self::AR117 => write!(f, "AR117"),
            Self::AR118 => write!(f, "AR118"),
            Self::AR119 => write!(f, "AR119"),
            Self::AR120 => write!(f, "AR120"),
            Self::AR121 => write!(f, "AR121"),
            Self::AR122 => write!(f, "AR122"),
            Self::AR123 => write!(f, "AR123"),
            Self::AR124 => write!(f, "AR124"),
            Self::AR125 => write!(f, "AR125"),
            Self::AR126 => write!(f, "AR126"),
            Self::AR127 => write!(f, "AR127"),
            Self::CPUID0 => write!(f, "CPUID0"),
            Self::CPUID1 => write!(f, "CPUID1"),
            Self::CPUID2 => write!(f, "CPUID2"),
            Self::CPUID3 => write!(f, "CPUID3"),
            Self::CPUID4 => write!(f, "CPUID4"),
            Self::ApDCR => write!(f, "ApDCR"),
            Self::ApITM => write!(f, "ApITM"),
            Self::ApIVA => write!(f, "ApIVA"),
            Self::CR3 => write!(f, "CR3"),
            Self::CR4 => write!(f, "CR4"),
            Self::CR5 => write!(f, "CR5"),
            Self::CR6 => write!(f, "CR6"),
            Self::CR7 => write!(f, "CR7"),
            Self::ApPTA => write!(f, "ApPTA"),
            Self::ApGPTA => write!(f, "ApGPTA"),
            Self::CR10 => write!(f, "CR10"),
            Self::CR11 => write!(f, "CR11"),
            Self::CR12 => write!(f, "CR12"),
            Self::CR13 => write!(f, "CR13"),
            Self::CR14 => write!(f, "CR14"),
            Self::CR15 => write!(f, "CR15"),
            Self::StIPSR => write!(f, "StIPSR"),
            Self::StISR => write!(f, "StISR"),
            Self::CR18 => write!(f, "CR18"),
            Self::StIIP => write!(f, "StIIP"),
            Self::StIFA => write!(f, "StIFA"),
            Self::StITIR => write!(f, "StITIR"),
            Self::StIIPA => write!(f, "StIIPA"),
            Self::StIFS => write!(f, "StIFS"),
            Self::StIIM => write!(f, "StIIM"),
            Self::StIHA => write!(f, "StIHA"),
            Self::CR26 => write!(f, "CR26"),
            Self::CR27 => write!(f, "CR27"),
            Self::CR28 => write!(f, "CR28"),
            Self::CR29 => write!(f, "CR29"),
            Self::CR30 => write!(f, "CR30"),
            Self::CR31 => write!(f, "CR31"),
            Self::CR32 => write!(f, "CR32"),
            Self::CR33 => write!(f, "CR33"),
            Self::CR34 => write!(f, "CR34"),
            Self::CR35 => write!(f, "CR35"),
            Self::CR36 => write!(f, "CR36"),
            Self::CR37 => write!(f, "CR37"),
            Self::CR38 => write!(f, "CR38"),
            Self::CR39 => write!(f, "CR39"),
            Self::CR40 => write!(f, "CR40"),
            Self::CR41 => write!(f, "CR41"),
            Self::CR42 => write!(f, "CR42"),
            Self::CR43 => write!(f, "CR43"),
            Self::CR44 => write!(f, "CR44"),
            Self::CR45 => write!(f, "CR45"),
            Self::CR46 => write!(f, "CR46"),
            Self::CR47 => write!(f, "CR47"),
            Self::CR48 => write!(f, "CR48"),
            Self::CR49 => write!(f, "CR49"),
            Self::CR50 => write!(f, "CR50"),
            Self::CR51 => write!(f, "CR51"),
            Self::CR52 => write!(f, "CR52"),
            Self::CR53 => write!(f, "CR53"),
            Self::CR54 => write!(f, "CR54"),
            Self::CR55 => write!(f, "CR55"),
            Self::CR56 => write!(f, "CR56"),
            Self::CR57 => write!(f, "CR57"),
            Self::CR58 => write!(f, "CR58"),
            Self::CR59 => write!(f, "CR59"),
            Self::CR60 => write!(f, "CR60"),
            Self::CR61 => write!(f, "CR61"),
            Self::CR62 => write!(f, "CR62"),
            Self::CR63 => write!(f, "CR63"),
            Self::SaLID => write!(f, "SaLID"),
            Self::SaIVR => write!(f, "SaIVR"),
            Self::SaTPR => write!(f, "SaTPR"),
            Self::SaEOI => write!(f, "SaEOI"),
            Self::SaIRR0 => write!(f, "SaIRR0"),
            Self::SaIRR1 => write!(f, "SaIRR1"),
            Self::SaIRR2 => write!(f, "SaIRR2"),
            Self::SaIRR3 => write!(f, "SaIRR3"),
            Self::SaITV => write!(f, "SaITV"),
            Self::SaPMV => write!(f, "SaPMV"),
            Self::SaCMCV => write!(f, "SaCMCV"),
            Self::CR75 => write!(f, "CR75"),
            Self::CR76 => write!(f, "CR76"),
            Self::CR77 => write!(f, "CR77"),
            Self::CR78 => write!(f, "CR78"),
            Self::CR79 => write!(f, "CR79"),
            Self::SaLRR0 => write!(f, "SaLRR0"),
            Self::SaLRR1 => write!(f, "SaLRR1"),
            Self::CR82 => write!(f, "CR82"),
            Self::CR83 => write!(f, "CR83"),
            Self::CR84 => write!(f, "CR84"),
            Self::CR85 => write!(f, "CR85"),
            Self::CR86 => write!(f, "CR86"),
            Self::CR87 => write!(f, "CR87"),
            Self::CR88 => write!(f, "CR88"),
            Self::CR89 => write!(f, "CR89"),
            Self::CR90 => write!(f, "CR90"),
            Self::CR91 => write!(f, "CR91"),
            Self::CR92 => write!(f, "CR92"),
            Self::CR93 => write!(f, "CR93"),
            Self::CR94 => write!(f, "CR94"),
            Self::CR95 => write!(f, "CR95"),
            Self::CR96 => write!(f, "CR96"),
            Self::CR97 => write!(f, "CR97"),
            Self::CR98 => write!(f, "CR98"),
            Self::CR99 => write!(f, "CR99"),
            Self::CR100 => write!(f, "CR100"),
            Self::CR101 => write!(f, "CR101"),
            Self::CR102 => write!(f, "CR102"),
            Self::CR103 => write!(f, "CR103"),
            Self::CR104 => write!(f, "CR104"),
            Self::CR105 => write!(f, "CR105"),
            Self::CR106 => write!(f, "CR106"),
            Self::CR107 => write!(f, "CR107"),
            Self::CR108 => write!(f, "CR108"),
            Self::CR109 => write!(f, "CR109"),
            Self::CR110 => write!(f, "CR110"),
            Self::CR111 => write!(f, "CR111"),
            Self::CR112 => write!(f, "CR112"),
            Self::CR113 => write!(f, "CR113"),
            Self::CR114 => write!(f, "CR114"),
            Self::CR115 => write!(f, "CR115"),
            Self::CR116 => write!(f, "CR116"),
            Self::CR117 => write!(f, "CR117"),
            Self::CR118 => write!(f, "CR118"),
            Self::CR119 => write!(f, "CR119"),
            Self::CR120 => write!(f, "CR120"),
            Self::CR121 => write!(f, "CR121"),
            Self::CR122 => write!(f, "CR122"),
            Self::CR123 => write!(f, "CR123"),
            Self::CR124 => write!(f, "CR124"),
            Self::CR125 => write!(f, "CR125"),
            Self::CR126 => write!(f, "CR126"),
            Self::CR127 => write!(f, "CR127"),
            Self::Pkr0 => write!(f, "Pkr0"),
            Self::Pkr1 => write!(f, "Pkr1"),
            Self::Pkr2 => write!(f, "Pkr2"),
            Self::Pkr3 => write!(f, "Pkr3"),
            Self::Pkr4 => write!(f, "Pkr4"),
            Self::Pkr5 => write!(f, "Pkr5"),
            Self::Pkr6 => write!(f, "Pkr6"),
            Self::Pkr7 => write!(f, "Pkr7"),
            Self::Pkr8 => write!(f, "Pkr8"),
            Self::Pkr9 => write!(f, "Pkr9"),
            Self::Pkr10 => write!(f, "Pkr10"),
            Self::Pkr11 => write!(f, "Pkr11"),
            Self::Pkr12 => write!(f, "Pkr12"),
            Self::Pkr13 => write!(f, "Pkr13"),
            Self::Pkr14 => write!(f, "Pkr14"),
            Self::Pkr15 => write!(f, "Pkr15"),
            Self::Rr0 => write!(f, "Rr0"),
            Self::Rr1 => write!(f, "Rr1"),
            Self::Rr2 => write!(f, "Rr2"),
            Self::Rr3 => write!(f, "Rr3"),
            Self::Rr4 => write!(f, "Rr4"),
            Self::Rr5 => write!(f, "Rr5"),
            Self::Rr6 => write!(f, "Rr6"),
            Self::Rr7 => write!(f, "Rr7"),
            Self::PFD0 => write!(f, "PFD0"),
            Self::PFD1 => write!(f, "PFD1"),
            Self::PFD2 => write!(f, "PFD2"),
            Self::PFD3 => write!(f, "PFD3"),
            Self::PFD4 => write!(f, "PFD4"),
            Self::PFD5 => write!(f, "PFD5"),
            Self::PFD6 => write!(f, "PFD6"),
            Self::PFD7 => write!(f, "PFD7"),
            Self::PFD8 => write!(f, "PFD8"),
            Self::PFD9 => write!(f, "PFD9"),
            Self::PFD10 => write!(f, "PFD10"),
            Self::PFD11 => write!(f, "PFD11"),
            Self::PFD12 => write!(f, "PFD12"),
            Self::PFD13 => write!(f, "PFD13"),
            Self::PFD14 => write!(f, "PFD14"),
            Self::PFD15 => write!(f, "PFD15"),
            Self::PFD16 => write!(f, "PFD16"),
            Self::PFD17 => write!(f, "PFD17"),
            Self::PFC0 => write!(f, "PFC0"),
            Self::PFC1 => write!(f, "PFC1"),
            Self::PFC2 => write!(f, "PFC2"),
            Self::PFC3 => write!(f, "PFC3"),
            Self::PFC4 => write!(f, "PFC4"),
            Self::PFC5 => write!(f, "PFC5"),
            Self::PFC6 => write!(f, "PFC6"),
            Self::PFC7 => write!(f, "PFC7"),
            Self::PFC8 => write!(f, "PFC8"),
            Self::PFC9 => write!(f, "PFC9"),
            Self::PFC10 => write!(f, "PFC10"),
            Self::PFC11 => write!(f, "PFC11"),
            Self::PFC12 => write!(f, "PFC12"),
            Self::PFC13 => write!(f, "PFC13"),
            Self::PFC14 => write!(f, "PFC14"),
            Self::PFC15 => write!(f, "PFC15"),
            Self::TrI0 => write!(f, "TrI0"),
            Self::TrI1 => write!(f, "TrI1"),
            Self::TrI2 => write!(f, "TrI2"),
            Self::TrI3 => write!(f, "TrI3"),
            Self::TrI4 => write!(f, "TrI4"),
            Self::TrI5 => write!(f, "TrI5"),
            Self::TrI6 => write!(f, "TrI6"),
            Self::TrI7 => write!(f, "TrI7"),
            Self::TrD0 => write!(f, "TrD0"),
            Self::TrD1 => write!(f, "TrD1"),
            Self::TrD2 => write!(f, "TrD2"),
            Self::TrD3 => write!(f, "TrD3"),
            Self::TrD4 => write!(f, "TrD4"),
            Self::TrD5 => write!(f, "TrD5"),
            Self::TrD6 => write!(f, "TrD6"),
            Self::TrD7 => write!(f, "TrD7"),
            Self::DbI0 => write!(f, "DbI0"),
            Self::DbI1 => write!(f, "DbI1"),
            Self::DbI2 => write!(f, "DbI2"),
            Self::DbI3 => write!(f, "DbI3"),
            Self::DbI4 => write!(f, "DbI4"),
            Self::DbI5 => write!(f, "DbI5"),
            Self::DbI6 => write!(f, "DbI6"),
            Self::DbI7 => write!(f, "DbI7"),
            Self::DbD0 => write!(f, "DbD0"),
            Self::DbD1 => write!(f, "DbD1"),
            Self::DbD2 => write!(f, "DbD2"),
            Self::DbD3 => write!(f, "DbD3"),
            Self::DbD4 => write!(f, "DbD4"),
            Self::DbD5 => write!(f, "DbD5"),
            Self::DbD6 => write!(f, "DbD6"),
            Self::DbD7 => write!(f, "DbD7"),
        }
    }
}

impl TryFrom<u16> for IA64Register {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            512 => Ok(Self::Br0),
            513 => Ok(Self::Br1),
            514 => Ok(Self::Br2),
            515 => Ok(Self::Br3),
            516 => Ok(Self::Br4),
            517 => Ok(Self::Br5),
            518 => Ok(Self::Br6),
            519 => Ok(Self::Br7),
            704 => Ok(Self::P0),
            705 => Ok(Self::P1),
            706 => Ok(Self::P2),
            707 => Ok(Self::P3),
            708 => Ok(Self::P4),
            709 => Ok(Self::P5),
            710 => Ok(Self::P6),
            711 => Ok(Self::P7),
            712 => Ok(Self::P8),
            713 => Ok(Self::P9),
            714 => Ok(Self::P10),
            715 => Ok(Self::P11),
            716 => Ok(Self::P12),
            717 => Ok(Self::P13),
            718 => Ok(Self::P14),
            719 => Ok(Self::P15),
            720 => Ok(Self::P16),
            721 => Ok(Self::P17),
            722 => Ok(Self::P18),
            723 => Ok(Self::P19),
            724 => Ok(Self::P20),
            725 => Ok(Self::P21),
            726 => Ok(Self::P22),
            727 => Ok(Self::P23),
            728 => Ok(Self::P24),
            729 => Ok(Self::P25),
            730 => Ok(Self::P26),
            731 => Ok(Self::P27),
            732 => Ok(Self::P28),
            733 => Ok(Self::P29),
            734 => Ok(Self::P30),
            735 => Ok(Self::P31),
            736 => Ok(Self::P32),
            737 => Ok(Self::P33),
            738 => Ok(Self::P34),
            739 => Ok(Self::P35),
            740 => Ok(Self::P36),
            741 => Ok(Self::P37),
            742 => Ok(Self::P38),
            743 => Ok(Self::P39),
            744 => Ok(Self::P40),
            745 => Ok(Self::P41),
            746 => Ok(Self::P42),
            747 => Ok(Self::P43),
            748 => Ok(Self::P44),
            749 => Ok(Self::P45),
            750 => Ok(Self::P46),
            751 => Ok(Self::P47),
            752 => Ok(Self::P48),
            753 => Ok(Self::P49),
            754 => Ok(Self::P50),
            755 => Ok(Self::P51),
            756 => Ok(Self::P52),
            757 => Ok(Self::P53),
            758 => Ok(Self::P54),
            759 => Ok(Self::P55),
            760 => Ok(Self::P56),
            761 => Ok(Self::P57),
            762 => Ok(Self::P58),
            763 => Ok(Self::P59),
            764 => Ok(Self::P60),
            765 => Ok(Self::P61),
            766 => Ok(Self::P62),
            767 => Ok(Self::P63),
            768 => Ok(Self::Preds),
            832 => Ok(Self::IntH0),
            833 => Ok(Self::IntH1),
            834 => Ok(Self::IntH2),
            835 => Ok(Self::IntH3),
            836 => Ok(Self::IntH4),
            837 => Ok(Self::IntH5),
            838 => Ok(Self::IntH6),
            839 => Ok(Self::IntH7),
            840 => Ok(Self::IntH8),
            841 => Ok(Self::IntH9),
            842 => Ok(Self::IntH10),
            843 => Ok(Self::IntH11),
            844 => Ok(Self::IntH12),
            845 => Ok(Self::IntH13),
            846 => Ok(Self::IntH14),
            847 => Ok(Self::IntH15),
            1016 => Ok(Self::Ip),
            1017 => Ok(Self::Umask),
            1018 => Ok(Self::Cfm),
            1019 => Ok(Self::Psr),
            1020 => Ok(Self::Nats),
            1021 => Ok(Self::Nats2),
            1022 => Ok(Self::Nats3),
            1024 => Ok(Self::IntR0),
            1025 => Ok(Self::IntR1),
            1026 => Ok(Self::IntR2),
            1027 => Ok(Self::IntR3),
            1028 => Ok(Self::IntR4),
            1029 => Ok(Self::IntR5),
            1030 => Ok(Self::IntR6),
            1031 => Ok(Self::IntR7),
            1032 => Ok(Self::IntR8),
            1033 => Ok(Self::IntR9),
            1034 => Ok(Self::IntR10),
            1035 => Ok(Self::IntR11),
            1036 => Ok(Self::IntR12),
            1037 => Ok(Self::IntR13),
            1038 => Ok(Self::IntR14),
            1039 => Ok(Self::IntR15),
            1040 => Ok(Self::IntR16),
            1041 => Ok(Self::IntR17),
            1042 => Ok(Self::IntR18),
            1043 => Ok(Self::IntR19),
            1044 => Ok(Self::IntR20),
            1045 => Ok(Self::IntR21),
            1046 => Ok(Self::IntR22),
            1047 => Ok(Self::IntR23),
            1048 => Ok(Self::IntR24),
            1049 => Ok(Self::IntR25),
            1050 => Ok(Self::IntR26),
            1051 => Ok(Self::IntR27),
            1052 => Ok(Self::IntR28),
            1053 => Ok(Self::IntR29),
            1054 => Ok(Self::IntR30),
            1055 => Ok(Self::IntR31),
            1056 => Ok(Self::IntR32),
            1057 => Ok(Self::IntR33),
            1058 => Ok(Self::IntR34),
            1059 => Ok(Self::IntR35),
            1060 => Ok(Self::IntR36),
            1061 => Ok(Self::IntR37),
            1062 => Ok(Self::IntR38),
            1063 => Ok(Self::IntR39),
            1064 => Ok(Self::IntR40),
            1065 => Ok(Self::IntR41),
            1066 => Ok(Self::IntR42),
            1067 => Ok(Self::IntR43),
            1068 => Ok(Self::IntR44),
            1069 => Ok(Self::IntR45),
            1070 => Ok(Self::IntR46),
            1071 => Ok(Self::IntR47),
            1072 => Ok(Self::IntR48),
            1073 => Ok(Self::IntR49),
            1074 => Ok(Self::IntR50),
            1075 => Ok(Self::IntR51),
            1076 => Ok(Self::IntR52),
            1077 => Ok(Self::IntR53),
            1078 => Ok(Self::IntR54),
            1079 => Ok(Self::IntR55),
            1080 => Ok(Self::IntR56),
            1081 => Ok(Self::IntR57),
            1082 => Ok(Self::IntR58),
            1083 => Ok(Self::IntR59),
            1084 => Ok(Self::IntR60),
            1085 => Ok(Self::IntR61),
            1086 => Ok(Self::IntR62),
            1087 => Ok(Self::IntR63),
            1088 => Ok(Self::IntR64),
            1089 => Ok(Self::IntR65),
            1090 => Ok(Self::IntR66),
            1091 => Ok(Self::IntR67),
            1092 => Ok(Self::IntR68),
            1093 => Ok(Self::IntR69),
            1094 => Ok(Self::IntR70),
            1095 => Ok(Self::IntR71),
            1096 => Ok(Self::IntR72),
            1097 => Ok(Self::IntR73),
            1098 => Ok(Self::IntR74),
            1099 => Ok(Self::IntR75),
            1100 => Ok(Self::IntR76),
            1101 => Ok(Self::IntR77),
            1102 => Ok(Self::IntR78),
            1103 => Ok(Self::IntR79),
            1104 => Ok(Self::IntR80),
            1105 => Ok(Self::IntR81),
            1106 => Ok(Self::IntR82),
            1107 => Ok(Self::IntR83),
            1108 => Ok(Self::IntR84),
            1109 => Ok(Self::IntR85),
            1110 => Ok(Self::IntR86),
            1111 => Ok(Self::IntR87),
            1112 => Ok(Self::IntR88),
            1113 => Ok(Self::IntR89),
            1114 => Ok(Self::IntR90),
            1115 => Ok(Self::IntR91),
            1116 => Ok(Self::IntR92),
            1117 => Ok(Self::IntR93),
            1118 => Ok(Self::IntR94),
            1119 => Ok(Self::IntR95),
            1120 => Ok(Self::IntR96),
            1121 => Ok(Self::IntR97),
            1122 => Ok(Self::IntR98),
            1123 => Ok(Self::IntR99),
            1124 => Ok(Self::IntR100),
            1125 => Ok(Self::IntR101),
            1126 => Ok(Self::IntR102),
            1127 => Ok(Self::IntR103),
            1128 => Ok(Self::IntR104),
            1129 => Ok(Self::IntR105),
            1130 => Ok(Self::IntR106),
            1131 => Ok(Self::IntR107),
            1132 => Ok(Self::IntR108),
            1133 => Ok(Self::IntR109),
            1134 => Ok(Self::IntR110),
            1135 => Ok(Self::IntR111),
            1136 => Ok(Self::IntR112),
            1137 => Ok(Self::IntR113),
            1138 => Ok(Self::IntR114),
            1139 => Ok(Self::IntR115),
            1140 => Ok(Self::IntR116),
            1141 => Ok(Self::IntR117),
            1142 => Ok(Self::IntR118),
            1143 => Ok(Self::IntR119),
            1144 => Ok(Self::IntR120),
            1145 => Ok(Self::IntR121),
            1146 => Ok(Self::IntR122),
            1147 => Ok(Self::IntR123),
            1148 => Ok(Self::IntR124),
            1149 => Ok(Self::IntR125),
            1150 => Ok(Self::IntR126),
            1151 => Ok(Self::IntR127),
            2048 => Ok(Self::FltF0),
            2049 => Ok(Self::FltF1),
            2050 => Ok(Self::FltF2),
            2051 => Ok(Self::FltF3),
            2052 => Ok(Self::FltF4),
            2053 => Ok(Self::FltF5),
            2054 => Ok(Self::FltF6),
            2055 => Ok(Self::FltF7),
            2056 => Ok(Self::FltF8),
            2057 => Ok(Self::FltF9),
            2058 => Ok(Self::FltF10),
            2059 => Ok(Self::FltF11),
            2060 => Ok(Self::FltF12),
            2061 => Ok(Self::FltF13),
            2062 => Ok(Self::FltF14),
            2063 => Ok(Self::FltF15),
            2064 => Ok(Self::FltF16),
            2065 => Ok(Self::FltF17),
            2066 => Ok(Self::FltF18),
            2067 => Ok(Self::FltF19),
            2068 => Ok(Self::FltF20),
            2069 => Ok(Self::FltF21),
            2070 => Ok(Self::FltF22),
            2071 => Ok(Self::FltF23),
            2072 => Ok(Self::FltF24),
            2073 => Ok(Self::FltF25),
            2074 => Ok(Self::FltF26),
            2075 => Ok(Self::FltF27),
            2076 => Ok(Self::FltF28),
            2077 => Ok(Self::FltF29),
            2078 => Ok(Self::FltF30),
            2079 => Ok(Self::FltF31),
            2080 => Ok(Self::FltF32),
            2081 => Ok(Self::FltF33),
            2082 => Ok(Self::FltF34),
            2083 => Ok(Self::FltF35),
            2084 => Ok(Self::FltF36),
            2085 => Ok(Self::FltF37),
            2086 => Ok(Self::FltF38),
            2087 => Ok(Self::FltF39),
            2088 => Ok(Self::FltF40),
            2089 => Ok(Self::FltF41),
            2090 => Ok(Self::FltF42),
            2091 => Ok(Self::FltF43),
            2092 => Ok(Self::FltF44),
            2093 => Ok(Self::FltF45),
            2094 => Ok(Self::FltF46),
            2095 => Ok(Self::FltF47),
            2096 => Ok(Self::FltF48),
            2097 => Ok(Self::FltF49),
            2098 => Ok(Self::FltF50),
            2099 => Ok(Self::FltF51),
            2100 => Ok(Self::FltF52),
            2101 => Ok(Self::FltF53),
            2102 => Ok(Self::FltF54),
            2103 => Ok(Self::FltF55),
            2104 => Ok(Self::FltF56),
            2105 => Ok(Self::FltF57),
            2106 => Ok(Self::FltF58),
            2107 => Ok(Self::FltF59),
            2108 => Ok(Self::FltF60),
            2109 => Ok(Self::FltF61),
            2110 => Ok(Self::FltF62),
            2111 => Ok(Self::FltF63),
            2112 => Ok(Self::FltF64),
            2113 => Ok(Self::FltF65),
            2114 => Ok(Self::FltF66),
            2115 => Ok(Self::FltF67),
            2116 => Ok(Self::FltF68),
            2117 => Ok(Self::FltF69),
            2118 => Ok(Self::FltF70),
            2119 => Ok(Self::FltF71),
            2120 => Ok(Self::FltF72),
            2121 => Ok(Self::FltF73),
            2122 => Ok(Self::FltF74),
            2123 => Ok(Self::FltF75),
            2124 => Ok(Self::FltF76),
            2125 => Ok(Self::FltF77),
            2126 => Ok(Self::FltF78),
            2127 => Ok(Self::FltF79),
            2128 => Ok(Self::FltF80),
            2129 => Ok(Self::FltF81),
            2130 => Ok(Self::FltF82),
            2131 => Ok(Self::FltF83),
            2132 => Ok(Self::FltF84),
            2133 => Ok(Self::FltF85),
            2134 => Ok(Self::FltF86),
            2135 => Ok(Self::FltF87),
            2136 => Ok(Self::FltF88),
            2137 => Ok(Self::FltF89),
            2138 => Ok(Self::FltF90),
            2139 => Ok(Self::FltF91),
            2140 => Ok(Self::FltF92),
            2141 => Ok(Self::FltF93),
            2142 => Ok(Self::FltF94),
            2143 => Ok(Self::FltF95),
            2144 => Ok(Self::FltF96),
            2145 => Ok(Self::FltF97),
            2146 => Ok(Self::FltF98),
            2147 => Ok(Self::FltF99),
            2148 => Ok(Self::FltF100),
            2149 => Ok(Self::FltF101),
            2150 => Ok(Self::FltF102),
            2151 => Ok(Self::FltF103),
            2152 => Ok(Self::FltF104),
            2153 => Ok(Self::FltF105),
            2154 => Ok(Self::FltF106),
            2155 => Ok(Self::FltF107),
            2156 => Ok(Self::FltF108),
            2157 => Ok(Self::FltF109),
            2158 => Ok(Self::FltF110),
            2159 => Ok(Self::FltF111),
            2160 => Ok(Self::FltF112),
            2161 => Ok(Self::FltF113),
            2162 => Ok(Self::FltF114),
            2163 => Ok(Self::FltF115),
            2164 => Ok(Self::FltF116),
            2165 => Ok(Self::FltF117),
            2166 => Ok(Self::FltF118),
            2167 => Ok(Self::FltF119),
            2168 => Ok(Self::FltF120),
            2169 => Ok(Self::FltF121),
            2170 => Ok(Self::FltF122),
            2171 => Ok(Self::FltF123),
            2172 => Ok(Self::FltF124),
            2173 => Ok(Self::FltF125),
            2174 => Ok(Self::FltF126),
            2175 => Ok(Self::FltF127),
            3072 => Ok(Self::ApKR0),
            3073 => Ok(Self::ApKR1),
            3074 => Ok(Self::ApKR2),
            3075 => Ok(Self::ApKR3),
            3076 => Ok(Self::ApKR4),
            3077 => Ok(Self::ApKR5),
            3078 => Ok(Self::ApKR6),
            3079 => Ok(Self::ApKR7),
            3080 => Ok(Self::AR8),
            3081 => Ok(Self::AR9),
            3082 => Ok(Self::AR10),
            3083 => Ok(Self::AR11),
            3084 => Ok(Self::AR12),
            3085 => Ok(Self::AR13),
            3086 => Ok(Self::AR14),
            3087 => Ok(Self::AR15),
            3088 => Ok(Self::RsRSC),
            3089 => Ok(Self::RsBSP),
            3090 => Ok(Self::RsBSPSTORE),
            3091 => Ok(Self::RsRNAT),
            3092 => Ok(Self::AR20),
            3093 => Ok(Self::StFCR),
            3094 => Ok(Self::AR22),
            3095 => Ok(Self::AR23),
            3096 => Ok(Self::EFLAG),
            3097 => Ok(Self::CSD),
            3098 => Ok(Self::SSD),
            3099 => Ok(Self::CFLG),
            3100 => Ok(Self::StFSR),
            3101 => Ok(Self::StFIR),
            3102 => Ok(Self::StFDR),
            3103 => Ok(Self::AR31),
            3104 => Ok(Self::ApCCV),
            3105 => Ok(Self::AR33),
            3106 => Ok(Self::AR34),
            3107 => Ok(Self::AR35),
            3108 => Ok(Self::ApUNAT),
            3109 => Ok(Self::AR37),
            3110 => Ok(Self::AR38),
            3111 => Ok(Self::AR39),
            3112 => Ok(Self::StFPSR),
            3113 => Ok(Self::AR41),
            3114 => Ok(Self::AR42),
            3115 => Ok(Self::AR43),
            3116 => Ok(Self::ApITC),
            3117 => Ok(Self::AR45),
            3118 => Ok(Self::AR46),
            3119 => Ok(Self::AR47),
            3120 => Ok(Self::AR48),
            3121 => Ok(Self::AR49),
            3122 => Ok(Self::AR50),
            3123 => Ok(Self::AR51),
            3124 => Ok(Self::AR52),
            3125 => Ok(Self::AR53),
            3126 => Ok(Self::AR54),
            3127 => Ok(Self::AR55),
            3128 => Ok(Self::AR56),
            3129 => Ok(Self::AR57),
            3130 => Ok(Self::AR58),
            3131 => Ok(Self::AR59),
            3132 => Ok(Self::AR60),
            3133 => Ok(Self::AR61),
            3134 => Ok(Self::AR62),
            3135 => Ok(Self::AR63),
            3136 => Ok(Self::RsPFS),
            3137 => Ok(Self::ApLC),
            3138 => Ok(Self::ApEC),
            3139 => Ok(Self::AR67),
            3140 => Ok(Self::AR68),
            3141 => Ok(Self::AR69),
            3142 => Ok(Self::AR70),
            3143 => Ok(Self::AR71),
            3144 => Ok(Self::AR72),
            3145 => Ok(Self::AR73),
            3146 => Ok(Self::AR74),
            3147 => Ok(Self::AR75),
            3148 => Ok(Self::AR76),
            3149 => Ok(Self::AR77),
            3150 => Ok(Self::AR78),
            3151 => Ok(Self::AR79),
            3152 => Ok(Self::AR80),
            3153 => Ok(Self::AR81),
            3154 => Ok(Self::AR82),
            3155 => Ok(Self::AR83),
            3156 => Ok(Self::AR84),
            3157 => Ok(Self::AR85),
            3158 => Ok(Self::AR86),
            3159 => Ok(Self::AR87),
            3160 => Ok(Self::AR88),
            3161 => Ok(Self::AR89),
            3162 => Ok(Self::AR90),
            3163 => Ok(Self::AR91),
            3164 => Ok(Self::AR92),
            3165 => Ok(Self::AR93),
            3166 => Ok(Self::AR94),
            3167 => Ok(Self::AR95),
            3168 => Ok(Self::AR96),
            3169 => Ok(Self::AR97),
            3170 => Ok(Self::AR98),
            3171 => Ok(Self::AR99),
            3172 => Ok(Self::AR100),
            3173 => Ok(Self::AR101),
            3174 => Ok(Self::AR102),
            3175 => Ok(Self::AR103),
            3176 => Ok(Self::AR104),
            3177 => Ok(Self::AR105),
            3178 => Ok(Self::AR106),
            3179 => Ok(Self::AR107),
            3180 => Ok(Self::AR108),
            3181 => Ok(Self::AR109),
            3182 => Ok(Self::AR110),
            3183 => Ok(Self::AR111),
            3184 => Ok(Self::AR112),
            3185 => Ok(Self::AR113),
            3186 => Ok(Self::AR114),
            3187 => Ok(Self::AR115),
            3188 => Ok(Self::AR116),
            3189 => Ok(Self::AR117),
            3190 => Ok(Self::AR118),
            3191 => Ok(Self::AR119),
            3192 => Ok(Self::AR120),
            3193 => Ok(Self::AR121),
            3194 => Ok(Self::AR122),
            3195 => Ok(Self::AR123),
            3196 => Ok(Self::AR124),
            3197 => Ok(Self::AR125),
            3198 => Ok(Self::AR126),
            3199 => Ok(Self::AR127),
            3328 => Ok(Self::CPUID0),
            3329 => Ok(Self::CPUID1),
            3330 => Ok(Self::CPUID2),
            3331 => Ok(Self::CPUID3),
            3332 => Ok(Self::CPUID4),
            4096 => Ok(Self::ApDCR),
            4097 => Ok(Self::ApITM),
            4098 => Ok(Self::ApIVA),
            4099 => Ok(Self::CR3),
            4100 => Ok(Self::CR4),
            4101 => Ok(Self::CR5),
            4102 => Ok(Self::CR6),
            4103 => Ok(Self::CR7),
            4104 => Ok(Self::ApPTA),
            4105 => Ok(Self::ApGPTA),
            4106 => Ok(Self::CR10),
            4107 => Ok(Self::CR11),
            4108 => Ok(Self::CR12),
            4109 => Ok(Self::CR13),
            4110 => Ok(Self::CR14),
            4111 => Ok(Self::CR15),
            4112 => Ok(Self::StIPSR),
            4113 => Ok(Self::StISR),
            4114 => Ok(Self::CR18),
            4115 => Ok(Self::StIIP),
            4116 => Ok(Self::StIFA),
            4117 => Ok(Self::StITIR),
            4118 => Ok(Self::StIIPA),
            4119 => Ok(Self::StIFS),
            4120 => Ok(Self::StIIM),
            4121 => Ok(Self::StIHA),
            4122 => Ok(Self::CR26),
            4123 => Ok(Self::CR27),
            4124 => Ok(Self::CR28),
            4125 => Ok(Self::CR29),
            4126 => Ok(Self::CR30),
            4127 => Ok(Self::CR31),
            4128 => Ok(Self::CR32),
            4129 => Ok(Self::CR33),
            4130 => Ok(Self::CR34),
            4131 => Ok(Self::CR35),
            4132 => Ok(Self::CR36),
            4133 => Ok(Self::CR37),
            4134 => Ok(Self::CR38),
            4135 => Ok(Self::CR39),
            4136 => Ok(Self::CR40),
            4137 => Ok(Self::CR41),
            4138 => Ok(Self::CR42),
            4139 => Ok(Self::CR43),
            4140 => Ok(Self::CR44),
            4141 => Ok(Self::CR45),
            4142 => Ok(Self::CR46),
            4143 => Ok(Self::CR47),
            4144 => Ok(Self::CR48),
            4145 => Ok(Self::CR49),
            4146 => Ok(Self::CR50),
            4147 => Ok(Self::CR51),
            4148 => Ok(Self::CR52),
            4149 => Ok(Self::CR53),
            4150 => Ok(Self::CR54),
            4151 => Ok(Self::CR55),
            4152 => Ok(Self::CR56),
            4153 => Ok(Self::CR57),
            4154 => Ok(Self::CR58),
            4155 => Ok(Self::CR59),
            4156 => Ok(Self::CR60),
            4157 => Ok(Self::CR61),
            4158 => Ok(Self::CR62),
            4159 => Ok(Self::CR63),
            4160 => Ok(Self::SaLID),
            4161 => Ok(Self::SaIVR),
            4162 => Ok(Self::SaTPR),
            4163 => Ok(Self::SaEOI),
            4164 => Ok(Self::SaIRR0),
            4165 => Ok(Self::SaIRR1),
            4166 => Ok(Self::SaIRR2),
            4167 => Ok(Self::SaIRR3),
            4168 => Ok(Self::SaITV),
            4169 => Ok(Self::SaPMV),
            4170 => Ok(Self::SaCMCV),
            4171 => Ok(Self::CR75),
            4172 => Ok(Self::CR76),
            4173 => Ok(Self::CR77),
            4174 => Ok(Self::CR78),
            4175 => Ok(Self::CR79),
            4176 => Ok(Self::SaLRR0),
            4177 => Ok(Self::SaLRR1),
            4178 => Ok(Self::CR82),
            4179 => Ok(Self::CR83),
            4180 => Ok(Self::CR84),
            4181 => Ok(Self::CR85),
            4182 => Ok(Self::CR86),
            4183 => Ok(Self::CR87),
            4184 => Ok(Self::CR88),
            4185 => Ok(Self::CR89),
            4186 => Ok(Self::CR90),
            4187 => Ok(Self::CR91),
            4188 => Ok(Self::CR92),
            4189 => Ok(Self::CR93),
            4190 => Ok(Self::CR94),
            4191 => Ok(Self::CR95),
            4192 => Ok(Self::CR96),
            4193 => Ok(Self::CR97),
            4194 => Ok(Self::CR98),
            4195 => Ok(Self::CR99),
            4196 => Ok(Self::CR100),
            4197 => Ok(Self::CR101),
            4198 => Ok(Self::CR102),
            4199 => Ok(Self::CR103),
            4200 => Ok(Self::CR104),
            4201 => Ok(Self::CR105),
            4202 => Ok(Self::CR106),
            4203 => Ok(Self::CR107),
            4204 => Ok(Self::CR108),
            4205 => Ok(Self::CR109),
            4206 => Ok(Self::CR110),
            4207 => Ok(Self::CR111),
            4208 => Ok(Self::CR112),
            4209 => Ok(Self::CR113),
            4210 => Ok(Self::CR114),
            4211 => Ok(Self::CR115),
            4212 => Ok(Self::CR116),
            4213 => Ok(Self::CR117),
            4214 => Ok(Self::CR118),
            4215 => Ok(Self::CR119),
            4216 => Ok(Self::CR120),
            4217 => Ok(Self::CR121),
            4218 => Ok(Self::CR122),
            4219 => Ok(Self::CR123),
            4220 => Ok(Self::CR124),
            4221 => Ok(Self::CR125),
            4222 => Ok(Self::CR126),
            4223 => Ok(Self::CR127),
            5120 => Ok(Self::Pkr0),
            5121 => Ok(Self::Pkr1),
            5122 => Ok(Self::Pkr2),
            5123 => Ok(Self::Pkr3),
            5124 => Ok(Self::Pkr4),
            5125 => Ok(Self::Pkr5),
            5126 => Ok(Self::Pkr6),
            5127 => Ok(Self::Pkr7),
            5128 => Ok(Self::Pkr8),
            5129 => Ok(Self::Pkr9),
            5130 => Ok(Self::Pkr10),
            5131 => Ok(Self::Pkr11),
            5132 => Ok(Self::Pkr12),
            5133 => Ok(Self::Pkr13),
            5134 => Ok(Self::Pkr14),
            5135 => Ok(Self::Pkr15),
            6144 => Ok(Self::Rr0),
            6145 => Ok(Self::Rr1),
            6146 => Ok(Self::Rr2),
            6147 => Ok(Self::Rr3),
            6148 => Ok(Self::Rr4),
            6149 => Ok(Self::Rr5),
            6150 => Ok(Self::Rr6),
            6151 => Ok(Self::Rr7),
            7168 => Ok(Self::PFD0),
            7169 => Ok(Self::PFD1),
            7170 => Ok(Self::PFD2),
            7171 => Ok(Self::PFD3),
            7172 => Ok(Self::PFD4),
            7173 => Ok(Self::PFD5),
            7174 => Ok(Self::PFD6),
            7175 => Ok(Self::PFD7),
            7176 => Ok(Self::PFD8),
            7177 => Ok(Self::PFD9),
            7178 => Ok(Self::PFD10),
            7179 => Ok(Self::PFD11),
            7180 => Ok(Self::PFD12),
            7181 => Ok(Self::PFD13),
            7182 => Ok(Self::PFD14),
            7183 => Ok(Self::PFD15),
            7184 => Ok(Self::PFD16),
            7185 => Ok(Self::PFD17),
            7424 => Ok(Self::PFC0),
            7425 => Ok(Self::PFC1),
            7426 => Ok(Self::PFC2),
            7427 => Ok(Self::PFC3),
            7428 => Ok(Self::PFC4),
            7429 => Ok(Self::PFC5),
            7430 => Ok(Self::PFC6),
            7431 => Ok(Self::PFC7),
            7432 => Ok(Self::PFC8),
            7433 => Ok(Self::PFC9),
            7434 => Ok(Self::PFC10),
            7435 => Ok(Self::PFC11),
            7436 => Ok(Self::PFC12),
            7437 => Ok(Self::PFC13),
            7438 => Ok(Self::PFC14),
            7439 => Ok(Self::PFC15),
            8192 => Ok(Self::TrI0),
            8193 => Ok(Self::TrI1),
            8194 => Ok(Self::TrI2),
            8195 => Ok(Self::TrI3),
            8196 => Ok(Self::TrI4),
            8197 => Ok(Self::TrI5),
            8198 => Ok(Self::TrI6),
            8199 => Ok(Self::TrI7),
            8320 => Ok(Self::TrD0),
            8321 => Ok(Self::TrD1),
            8322 => Ok(Self::TrD2),
            8323 => Ok(Self::TrD3),
            8324 => Ok(Self::TrD4),
            8325 => Ok(Self::TrD5),
            8326 => Ok(Self::TrD6),
            8327 => Ok(Self::TrD7),
            8448 => Ok(Self::DbI0),
            8449 => Ok(Self::DbI1),
            8450 => Ok(Self::DbI2),
            8451 => Ok(Self::DbI3),
            8452 => Ok(Self::DbI4),
            8453 => Ok(Self::DbI5),
            8454 => Ok(Self::DbI6),
            8455 => Ok(Self::DbI7),
            8576 => Ok(Self::DbD0),
            8577 => Ok(Self::DbD1),
            8578 => Ok(Self::DbD2),
            8579 => Ok(Self::DbD3),
            8580 => Ok(Self::DbD4),
            8581 => Ok(Self::DbD5),
            8582 => Ok(Self::DbD6),
            8583 => Ok(Self::DbD7),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for IA64Register {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// Register set for the TriCore processor
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum TricoreRegister {
    // General Purpose Data Registers
    D0 = 10,
    D1 = 11,
    D2 = 12,
    D3 = 13,
    D4 = 14,
    D5 = 15,
    D6 = 16,
    D7 = 17,
    D8 = 18,
    D9 = 19,
    D10 = 20,
    D11 = 21,
    D12 = 22,
    D13 = 23,
    D14 = 24,
    D15 = 25,

    // General Purpose Address Registers
    A0 = 26,
    A1 = 27,
    A2 = 28,
    A3 = 29,
    A4 = 30,
    A5 = 31,
    A6 = 32,
    A7 = 33,
    A8 = 34,
    A9 = 35,
    A10 = 36,
    A11 = 37,
    A12 = 38,
    A13 = 39,
    A14 = 40,
    A15 = 41,

    // Extended (64-bit) data registers
    E0 = 42,
    E2 = 43,
    E4 = 44,
    E6 = 45,
    E8 = 46,
    E10 = 47,
    E12 = 48,
    E14 = 49,

    // Extended (64-bit) address registers
    EA0 = 50,
    EA2 = 51,
    EA4 = 52,
    EA6 = 53,
    EA8 = 54,
    EA10 = 55,
    EA12 = 56,
    EA14 = 57,

    PSW = 58,
    PCXI = 59,
    PC = 60,
    FCX = 61,
    LCX = 62,
    ISP = 63,
    ICR = 64,
    BIV = 65,
    BTV = 66,
    SYSCON = 67,
    DPRx_0 = 68,
    DPRx_1 = 69,
    DPRx_2 = 70,
    DPRx_3 = 71,
    // CPRx_0 = 68,
    // CPRx_1 = 69,
    // CPRx_2 = 70,
    // CPRx_3 = 71,
    // DPMx_0 = 68,
    // DPMx_1 = 69,
    // DPMx_2 = 70,
    // DPMx_3 = 71,
    // CPMx_0 = 68,
    // CPMx_1 = 69,
    // CPMx_2 = 70,
    // CPMx_3 = 71,
    DBGSSR = 72,
    EXEVT = 73,
    SWEVT = 74,
    CREVT = 75,
    TRnEVT = 76,
    MMUCON = 77,
    ASI = 78,
    TVA = 79,
    TPA = 80,
    TPX = 81,
    TFA = 82,
}

impl fmt::Display for TricoreRegister {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::D0 => write!(f, "D0"),
            Self::D1 => write!(f, "D1"),
            Self::D2 => write!(f, "D2"),
            Self::D3 => write!(f, "D3"),
            Self::D4 => write!(f, "D4"),
            Self::D5 => write!(f, "D5"),
            Self::D6 => write!(f, "D6"),
            Self::D7 => write!(f, "D7"),
            Self::D8 => write!(f, "D8"),
            Self::D9 => write!(f, "D9"),
            Self::D10 => write!(f, "D10"),
            Self::D11 => write!(f, "D11"),
            Self::D12 => write!(f, "D12"),
            Self::D13 => write!(f, "D13"),
            Self::D14 => write!(f, "D14"),
            Self::D15 => write!(f, "D15"),
            Self::A0 => write!(f, "A0"),
            Self::A1 => write!(f, "A1"),
            Self::A2 => write!(f, "A2"),
            Self::A3 => write!(f, "A3"),
            Self::A4 => write!(f, "A4"),
            Self::A5 => write!(f, "A5"),
            Self::A6 => write!(f, "A6"),
            Self::A7 => write!(f, "A7"),
            Self::A8 => write!(f, "A8"),
            Self::A9 => write!(f, "A9"),
            Self::A10 => write!(f, "A10"),
            Self::A11 => write!(f, "A11"),
            Self::A12 => write!(f, "A12"),
            Self::A13 => write!(f, "A13"),
            Self::A14 => write!(f, "A14"),
            Self::A15 => write!(f, "A15"),
            Self::E0 => write!(f, "E0"),
            Self::E2 => write!(f, "E2"),
            Self::E4 => write!(f, "E4"),
            Self::E6 => write!(f, "E6"),
            Self::E8 => write!(f, "E8"),
            Self::E10 => write!(f, "E10"),
            Self::E12 => write!(f, "E12"),
            Self::E14 => write!(f, "E14"),
            Self::EA0 => write!(f, "EA0"),
            Self::EA2 => write!(f, "EA2"),
            Self::EA4 => write!(f, "EA4"),
            Self::EA6 => write!(f, "EA6"),
            Self::EA8 => write!(f, "EA8"),
            Self::EA10 => write!(f, "EA10"),
            Self::EA12 => write!(f, "EA12"),
            Self::EA14 => write!(f, "EA14"),
            Self::PSW => write!(f, "PSW"),
            Self::PCXI => write!(f, "PCXI"),
            Self::PC => write!(f, "PC"),
            Self::FCX => write!(f, "FCX"),
            Self::LCX => write!(f, "LCX"),
            Self::ISP => write!(f, "ISP"),
            Self::ICR => write!(f, "ICR"),
            Self::BIV => write!(f, "BIV"),
            Self::BTV => write!(f, "BTV"),
            Self::SYSCON => write!(f, "SYSCON"),
            Self::DPRx_0 => write!(f, "DPRx_0"),
            Self::DPRx_1 => write!(f, "DPRx_1"),
            Self::DPRx_2 => write!(f, "DPRx_2"),
            Self::DPRx_3 => write!(f, "DPRx_3"),
            Self::DBGSSR => write!(f, "DBGSSR"),
            Self::EXEVT => write!(f, "EXEVT"),
            Self::SWEVT => write!(f, "SWEVT"),
            Self::CREVT => write!(f, "CREVT"),
            Self::TRnEVT => write!(f, "TRnEVT"),
            Self::MMUCON => write!(f, "MMUCON"),
            Self::ASI => write!(f, "ASI"),
            Self::TVA => write!(f, "TVA"),
            Self::TPA => write!(f, "TPA"),
            Self::TPX => write!(f, "TPX"),
            Self::TFA => write!(f, "TFA"),
        }
    }
}

impl TryFrom<u16> for TricoreRegister {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            10 => Ok(Self::D0),
            11 => Ok(Self::D1),
            12 => Ok(Self::D2),
            13 => Ok(Self::D3),
            14 => Ok(Self::D4),
            15 => Ok(Self::D5),
            16 => Ok(Self::D6),
            17 => Ok(Self::D7),
            18 => Ok(Self::D8),
            19 => Ok(Self::D9),
            20 => Ok(Self::D10),
            21 => Ok(Self::D11),
            22 => Ok(Self::D12),
            23 => Ok(Self::D13),
            24 => Ok(Self::D14),
            25 => Ok(Self::D15),
            26 => Ok(Self::A0),
            27 => Ok(Self::A1),
            28 => Ok(Self::A2),
            29 => Ok(Self::A3),
            30 => Ok(Self::A4),
            31 => Ok(Self::A5),
            32 => Ok(Self::A6),
            33 => Ok(Self::A7),
            34 => Ok(Self::A8),
            35 => Ok(Self::A9),
            36 => Ok(Self::A10),
            37 => Ok(Self::A11),
            38 => Ok(Self::A12),
            39 => Ok(Self::A13),
            40 => Ok(Self::A14),
            41 => Ok(Self::A15),
            42 => Ok(Self::E0),
            43 => Ok(Self::E2),
            44 => Ok(Self::E4),
            45 => Ok(Self::E6),
            46 => Ok(Self::E8),
            47 => Ok(Self::E10),
            48 => Ok(Self::E12),
            49 => Ok(Self::E14),
            50 => Ok(Self::EA0),
            51 => Ok(Self::EA2),
            52 => Ok(Self::EA4),
            53 => Ok(Self::EA6),
            54 => Ok(Self::EA8),
            55 => Ok(Self::EA10),
            56 => Ok(Self::EA12),
            57 => Ok(Self::EA14),
            58 => Ok(Self::PSW),
            59 => Ok(Self::PCXI),
            60 => Ok(Self::PC),
            61 => Ok(Self::FCX),
            62 => Ok(Self::LCX),
            63 => Ok(Self::ISP),
            64 => Ok(Self::ICR),
            65 => Ok(Self::BIV),
            66 => Ok(Self::BTV),
            67 => Ok(Self::SYSCON),
            68 => Ok(Self::DPRx_0),
            69 => Ok(Self::DPRx_1),
            70 => Ok(Self::DPRx_2),
            71 => Ok(Self::DPRx_3),
            72 => Ok(Self::DBGSSR),
            73 => Ok(Self::EXEVT),
            74 => Ok(Self::SWEVT),
            75 => Ok(Self::CREVT),
            76 => Ok(Self::TRnEVT),
            77 => Ok(Self::MMUCON),
            78 => Ok(Self::ASI),
            79 => Ok(Self::TVA),
            80 => Ok(Self::TPA),
            81 => Ok(Self::TPX),
            82 => Ok(Self::TFA),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for TricoreRegister {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// Register set for the AM33 and related processors
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum AM33Register {
    // "Extended" (general purpose integer) registers
    E0 = 10,
    E1 = 11,
    E2 = 12,
    E3 = 13,
    E4 = 14,
    E5 = 15,
    E6 = 16,
    E7 = 17,

    // Address registers
    A0 = 20,
    A1 = 21,
    A2 = 22,
    A3 = 23,

    // Integer data registers
    D0 = 30,
    D1 = 31,
    D2 = 32,
    D3 = 33,

    // (Single-precision) floating-point registers
    FS0 = 40,
    FS1 = 41,
    FS2 = 42,
    FS3 = 43,
    FS4 = 44,
    FS5 = 45,
    FS6 = 46,
    FS7 = 47,
    FS8 = 48,
    FS9 = 49,
    FS10 = 50,
    FS11 = 51,
    FS12 = 52,
    FS13 = 53,
    FS14 = 54,
    FS15 = 55,
    FS16 = 56,
    FS17 = 57,
    FS18 = 58,
    FS19 = 59,
    FS20 = 60,
    FS21 = 61,
    FS22 = 62,
    FS23 = 63,
    FS24 = 64,
    FS25 = 65,
    FS26 = 66,
    FS27 = 67,
    FS28 = 68,
    FS29 = 69,
    FS30 = 70,
    FS31 = 71,

    // Special purpose registers

    // Stack pointer
    SP = 80,

    // Program counter
    PC = 81,

    // Multiply-divide/accumulate registers
    MDR = 82,
    MDRQ = 83,
    MCRH = 84,
    MCRL = 85,
    MCVF = 86,

    // CPU status words
    EPSW = 87,
    FPCR = 88,

    // Loop buffer registers
    LIR = 89,
    LAR = 90,
}

impl fmt::Display for AM33Register {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::E0 => write!(f, "E0"),
            Self::E1 => write!(f, "E1"),
            Self::E2 => write!(f, "E2"),
            Self::E3 => write!(f, "E3"),
            Self::E4 => write!(f, "E4"),
            Self::E5 => write!(f, "E5"),
            Self::E6 => write!(f, "E6"),
            Self::E7 => write!(f, "E7"),
            Self::A0 => write!(f, "A0"),
            Self::A1 => write!(f, "A1"),
            Self::A2 => write!(f, "A2"),
            Self::A3 => write!(f, "A3"),
            Self::D0 => write!(f, "D0"),
            Self::D1 => write!(f, "D1"),
            Self::D2 => write!(f, "D2"),
            Self::D3 => write!(f, "D3"),
            Self::FS0 => write!(f, "FS0"),
            Self::FS1 => write!(f, "FS1"),
            Self::FS2 => write!(f, "FS2"),
            Self::FS3 => write!(f, "FS3"),
            Self::FS4 => write!(f, "FS4"),
            Self::FS5 => write!(f, "FS5"),
            Self::FS6 => write!(f, "FS6"),
            Self::FS7 => write!(f, "FS7"),
            Self::FS8 => write!(f, "FS8"),
            Self::FS9 => write!(f, "FS9"),
            Self::FS10 => write!(f, "FS10"),
            Self::FS11 => write!(f, "FS11"),
            Self::FS12 => write!(f, "FS12"),
            Self::FS13 => write!(f, "FS13"),
            Self::FS14 => write!(f, "FS14"),
            Self::FS15 => write!(f, "FS15"),
            Self::FS16 => write!(f, "FS16"),
            Self::FS17 => write!(f, "FS17"),
            Self::FS18 => write!(f, "FS18"),
            Self::FS19 => write!(f, "FS19"),
            Self::FS20 => write!(f, "FS20"),
            Self::FS21 => write!(f, "FS21"),
            Self::FS22 => write!(f, "FS22"),
            Self::FS23 => write!(f, "FS23"),
            Self::FS24 => write!(f, "FS24"),
            Self::FS25 => write!(f, "FS25"),
            Self::FS26 => write!(f, "FS26"),
            Self::FS27 => write!(f, "FS27"),
            Self::FS28 => write!(f, "FS28"),
            Self::FS29 => write!(f, "FS29"),
            Self::FS30 => write!(f, "FS30"),
            Self::FS31 => write!(f, "FS31"),
            Self::SP => write!(f, "SP"),
            Self::PC => write!(f, "PC"),
            Self::MDR => write!(f, "MDR"),
            Self::MDRQ => write!(f, "MDRQ"),
            Self::MCRH => write!(f, "MCRH"),
            Self::MCRL => write!(f, "MCRL"),
            Self::MCVF => write!(f, "MCVF"),
            Self::EPSW => write!(f, "EPSW"),
            Self::FPCR => write!(f, "FPCR"),
            Self::LIR => write!(f, "LIR"),
            Self::LAR => write!(f, "LAR"),
        }
    }
}

impl TryFrom<u16> for AM33Register {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            10 => Ok(Self::E0),
            11 => Ok(Self::E1),
            12 => Ok(Self::E2),
            13 => Ok(Self::E3),
            14 => Ok(Self::E4),
            15 => Ok(Self::E5),
            16 => Ok(Self::E6),
            17 => Ok(Self::E7),
            20 => Ok(Self::A0),
            21 => Ok(Self::A1),
            22 => Ok(Self::A2),
            23 => Ok(Self::A3),
            30 => Ok(Self::D0),
            31 => Ok(Self::D1),
            32 => Ok(Self::D2),
            33 => Ok(Self::D3),
            40 => Ok(Self::FS0),
            41 => Ok(Self::FS1),
            42 => Ok(Self::FS2),
            43 => Ok(Self::FS3),
            44 => Ok(Self::FS4),
            45 => Ok(Self::FS5),
            46 => Ok(Self::FS6),
            47 => Ok(Self::FS7),
            48 => Ok(Self::FS8),
            49 => Ok(Self::FS9),
            50 => Ok(Self::FS10),
            51 => Ok(Self::FS11),
            52 => Ok(Self::FS12),
            53 => Ok(Self::FS13),
            54 => Ok(Self::FS14),
            55 => Ok(Self::FS15),
            56 => Ok(Self::FS16),
            57 => Ok(Self::FS17),
            58 => Ok(Self::FS18),
            59 => Ok(Self::FS19),
            60 => Ok(Self::FS20),
            61 => Ok(Self::FS21),
            62 => Ok(Self::FS22),
            63 => Ok(Self::FS23),
            64 => Ok(Self::FS24),
            65 => Ok(Self::FS25),
            66 => Ok(Self::FS26),
            67 => Ok(Self::FS27),
            68 => Ok(Self::FS28),
            69 => Ok(Self::FS29),
            70 => Ok(Self::FS30),
            71 => Ok(Self::FS31),
            80 => Ok(Self::SP),
            81 => Ok(Self::PC),
            82 => Ok(Self::MDR),
            83 => Ok(Self::MDRQ),
            84 => Ok(Self::MCRH),
            85 => Ok(Self::MCRL),
            86 => Ok(Self::MCVF),
            87 => Ok(Self::EPSW),
            88 => Ok(Self::FPCR),
            89 => Ok(Self::LIR),
            90 => Ok(Self::LAR),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for AM33Register {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// Register set for the Mitsubishi M32R
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum MitsubishiM32RRegister {
    R0 = 10,
    R1 = 11,
    R2 = 12,
    R3 = 13,
    R4 = 14,
    R5 = 15,
    R6 = 16,
    R7 = 17,
    R8 = 18,
    R9 = 19,
    R10 = 20,
    R11 = 21,
    R12 = 22,  // Gloabal Pointer, if used
    R13 = 23,  // Frame Pointer, if allocated
    R14 = 24,  // Link Register
    R15 = 25,  // Stack Pointer
    PSW = 26,  // Preocessor Status Register
    CBR = 27,  // Condition Bit Register
    SPI = 28,  // Interrupt Stack Pointer
    SPU = 29,  // User Stack Pointer
    SPO = 30,  // OS Stack Pointer
    BPC = 31,  // Backup Program Counter
    ACHI = 32, // Accumulator High
    ACLO = 33, // Accumulator Low
    PC = 34,   // Program Counter
}

impl fmt::Display for MitsubishiM32RRegister {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::R0 => write!(f, "R0"),
            Self::R1 => write!(f, "R1"),
            Self::R2 => write!(f, "R2"),
            Self::R3 => write!(f, "R3"),
            Self::R4 => write!(f, "R4"),
            Self::R5 => write!(f, "R5"),
            Self::R6 => write!(f, "R6"),
            Self::R7 => write!(f, "R7"),
            Self::R8 => write!(f, "R8"),
            Self::R9 => write!(f, "R9"),
            Self::R10 => write!(f, "R10"),
            Self::R11 => write!(f, "R11"),
            Self::R12 => write!(f, "R12"),
            Self::R13 => write!(f, "R13"),
            Self::R14 => write!(f, "R14"),
            Self::R15 => write!(f, "R15"),
            Self::PSW => write!(f, "PSW"),
            Self::CBR => write!(f, "CBR"),
            Self::SPI => write!(f, "SPI"),
            Self::SPU => write!(f, "SPU"),
            Self::SPO => write!(f, "SPO"),
            Self::BPC => write!(f, "BPC"),
            Self::ACHI => write!(f, "ACHI"),
            Self::ACLO => write!(f, "ACLO"),
            Self::PC => write!(f, "PC"),
        }
    }
}

impl TryFrom<u16> for MitsubishiM32RRegister {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            10 => Ok(Self::R0),
            11 => Ok(Self::R1),
            12 => Ok(Self::R2),
            13 => Ok(Self::R3),
            14 => Ok(Self::R4),
            15 => Ok(Self::R5),
            16 => Ok(Self::R6),
            17 => Ok(Self::R7),
            18 => Ok(Self::R8),
            19 => Ok(Self::R9),
            20 => Ok(Self::R10),
            21 => Ok(Self::R11),
            22 => Ok(Self::R12),
            23 => Ok(Self::R13),
            24 => Ok(Self::R14),
            25 => Ok(Self::R15),
            26 => Ok(Self::PSW),
            27 => Ok(Self::CBR),
            28 => Ok(Self::SPI),
            29 => Ok(Self::SPU),
            30 => Ok(Self::SPO),
            31 => Ok(Self::BPC),
            32 => Ok(Self::ACHI),
            33 => Ok(Self::ACLO),
            34 => Ok(Self::PC),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for MitsubishiM32RRegister {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// Register set for the SuperH SHMedia processor including compact mode
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum SuperHSHMediaRegister {
    // Integer - 64 bit general registers
    R0 = 10,
    R1 = 11,
    R2 = 12,
    R3 = 13,
    R4 = 14,
    R5 = 15,
    R6 = 16,
    R7 = 17,
    R8 = 18,
    R9 = 19,
    R10 = 20,
    R11 = 21,
    R12 = 22,
    R13 = 23,
    R14 = 24,
    R15 = 25,
    R16 = 26,
    R17 = 27,
    R18 = 28,
    R19 = 29,
    R20 = 30,
    R21 = 31,
    R22 = 32,
    R23 = 33,
    R24 = 34,
    R25 = 35,
    R26 = 36,
    R27 = 37,
    R28 = 38,
    R29 = 39,
    R30 = 40,
    R31 = 41,
    R32 = 42,
    R33 = 43,
    R34 = 44,
    R35 = 45,
    R36 = 46,
    R37 = 47,
    R38 = 48,
    R39 = 49,
    R40 = 50,
    R41 = 51,
    R42 = 52,
    R43 = 53,
    R44 = 54,
    R45 = 55,
    R46 = 56,
    R47 = 57,
    R48 = 58,
    R49 = 59,
    R50 = 60,
    R51 = 61,
    R52 = 62,
    R53 = 63,
    R54 = 64,
    R55 = 65,
    R56 = 66,
    R57 = 67,
    R58 = 68,
    R59 = 69,
    R60 = 70,
    R61 = 71,
    R62 = 72,
    R63 = 73,

    // Target Registers - 32 bit
    TR0 = 74,
    TR1 = 75,
    TR2 = 76,
    TR3 = 77,
    TR4 = 78,
    TR5 = 79,
    TR6 = 80,
    TR7 = 81,
    TR8 = 82,  // future-proof
    TR9 = 83,  // future-proof
    TR10 = 84, // future-proof
    TR11 = 85, // future-proof
    TR12 = 86, // future-proof
    TR13 = 87, // future-proof
    TR14 = 88, // future-proof
    TR15 = 89, // future-proof

    // Single - 32 bit fp registers
    FR0 = 128,
    FR1 = 129,
    FR2 = 130,
    FR3 = 131,
    FR4 = 132,
    FR5 = 133,
    FR6 = 134,
    FR7 = 135,
    FR8 = 136,
    FR9 = 137,
    FR10 = 138,
    FR11 = 139,
    FR12 = 140,
    FR13 = 141,
    FR14 = 142,
    FR15 = 143,
    FR16 = 144,
    FR17 = 145,
    FR18 = 146,
    FR19 = 147,
    FR20 = 148,
    FR21 = 149,
    FR22 = 150,
    FR23 = 151,
    FR24 = 152,
    FR25 = 153,
    FR26 = 154,
    FR27 = 155,
    FR28 = 156,
    FR29 = 157,
    FR30 = 158,
    FR31 = 159,
    FR32 = 160,
    FR33 = 161,
    FR34 = 162,
    FR35 = 163,
    FR36 = 164,
    FR37 = 165,
    FR38 = 166,
    FR39 = 167,
    FR40 = 168,
    FR41 = 169,
    FR42 = 170,
    FR43 = 171,
    FR44 = 172,
    FR45 = 173,
    FR46 = 174,
    FR47 = 175,
    FR48 = 176,
    FR49 = 177,
    FR50 = 178,
    FR51 = 179,
    FR52 = 180,
    FR53 = 181,
    FR54 = 182,
    FR55 = 183,
    FR56 = 184,
    FR57 = 185,
    FR58 = 186,
    FR59 = 187,
    FR60 = 188,
    FR61 = 189,
    FR62 = 190,
    FR63 = 191,

    // Double - 64 bit synonyms for 32bit fp register pairs
    //          subtract 128 to find first base single register
    DR0 = 256,
    DR2 = 258,
    DR4 = 260,
    DR6 = 262,
    DR8 = 264,
    DR10 = 266,
    DR12 = 268,
    DR14 = 270,
    DR16 = 272,
    DR18 = 274,
    DR20 = 276,
    DR22 = 278,
    DR24 = 280,
    DR26 = 282,
    DR28 = 284,
    DR30 = 286,
    DR32 = 288,
    DR34 = 290,
    DR36 = 292,
    DR38 = 294,
    DR40 = 296,
    DR42 = 298,
    DR44 = 300,
    DR46 = 302,
    DR48 = 304,
    DR50 = 306,
    DR52 = 308,
    DR54 = 310,
    DR56 = 312,
    DR58 = 314,
    DR60 = 316,
    DR62 = 318,

    // Vector - 128 bit synonyms for 32bit fp register quads
    //          subtract 384 to find first base single register
    FV0 = 512,
    FV4 = 516,
    FV8 = 520,
    FV12 = 524,
    FV16 = 528,
    FV20 = 532,
    FV24 = 536,
    FV28 = 540,
    FV32 = 544,
    FV36 = 548,
    FV40 = 552,
    FV44 = 556,
    FV48 = 560,
    FV52 = 564,
    FV56 = 568,
    FV60 = 572,

    // Matrix - 512 bit synonyms for 16 adjacent 32bit fp registers
    //          subtract 896 to find first base single register
    MTRX0 = 1024,
    MTRX16 = 1040,
    MTRX32 = 1056,
    MTRX48 = 1072,

    // Control - Implementation defined 64bit control registers
    CR0 = 2000,
    CR1 = 2001,
    CR2 = 2002,
    CR3 = 2003,
    CR4 = 2004,
    CR5 = 2005,
    CR6 = 2006,
    CR7 = 2007,
    CR8 = 2008,
    CR9 = 2009,
    CR10 = 2010,
    CR11 = 2011,
    CR12 = 2012,
    CR13 = 2013,
    CR14 = 2014,
    CR15 = 2015,
    CR16 = 2016,
    CR17 = 2017,
    CR18 = 2018,
    CR19 = 2019,
    CR20 = 2020,
    CR21 = 2021,
    CR22 = 2022,
    CR23 = 2023,
    CR24 = 2024,
    CR25 = 2025,
    CR26 = 2026,
    CR27 = 2027,
    CR28 = 2028,
    CR29 = 2029,
    CR30 = 2030,
    CR31 = 2031,
    CR32 = 2032,
    CR33 = 2033,
    CR34 = 2034,
    CR35 = 2035,
    CR36 = 2036,
    CR37 = 2037,
    CR38 = 2038,
    CR39 = 2039,
    CR40 = 2040,
    CR41 = 2041,
    CR42 = 2042,
    CR43 = 2043,
    CR44 = 2044,
    CR45 = 2045,
    CR46 = 2046,
    CR47 = 2047,
    CR48 = 2048,
    CR49 = 2049,
    CR50 = 2050,
    CR51 = 2051,
    CR52 = 2052,
    CR53 = 2053,
    CR54 = 2054,
    CR55 = 2055,
    CR56 = 2056,
    CR57 = 2057,
    CR58 = 2058,
    CR59 = 2059,
    CR60 = 2060,
    CR61 = 2061,
    CR62 = 2062,
    CR63 = 2063,

    FPSCR = 2064,
    // Compact mode synonyms
    // GBR = 26,
    // MACL = 90, // synonym for lower 32bits of media R17
    // MACH = 91, // synonym for upper 32bits of media R17
    // PR = 28,
    // T = 92, // synonym for lowest bit of media R19
    // FPUL = 160,
    // PC = 93,
    // SR = 2000,
}

impl fmt::Display for SuperHSHMediaRegister {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::R0 => write!(f, "R0"),
            Self::R1 => write!(f, "R1"),
            Self::R2 => write!(f, "R2"),
            Self::R3 => write!(f, "R3"),
            Self::R4 => write!(f, "R4"),
            Self::R5 => write!(f, "R5"),
            Self::R6 => write!(f, "R6"),
            Self::R7 => write!(f, "R7"),
            Self::R8 => write!(f, "R8"),
            Self::R9 => write!(f, "R9"),
            Self::R10 => write!(f, "R10"),
            Self::R11 => write!(f, "R11"),
            Self::R12 => write!(f, "R12"),
            Self::R13 => write!(f, "R13"),
            Self::R14 => write!(f, "R14"),
            Self::R15 => write!(f, "R15"),
            Self::R16 => write!(f, "R16"),
            Self::R17 => write!(f, "R17"),
            Self::R18 => write!(f, "R18"),
            Self::R19 => write!(f, "R19"),
            Self::R20 => write!(f, "R20"),
            Self::R21 => write!(f, "R21"),
            Self::R22 => write!(f, "R22"),
            Self::R23 => write!(f, "R23"),
            Self::R24 => write!(f, "R24"),
            Self::R25 => write!(f, "R25"),
            Self::R26 => write!(f, "R26"),
            Self::R27 => write!(f, "R27"),
            Self::R28 => write!(f, "R28"),
            Self::R29 => write!(f, "R29"),
            Self::R30 => write!(f, "R30"),
            Self::R31 => write!(f, "R31"),
            Self::R32 => write!(f, "R32"),
            Self::R33 => write!(f, "R33"),
            Self::R34 => write!(f, "R34"),
            Self::R35 => write!(f, "R35"),
            Self::R36 => write!(f, "R36"),
            Self::R37 => write!(f, "R37"),
            Self::R38 => write!(f, "R38"),
            Self::R39 => write!(f, "R39"),
            Self::R40 => write!(f, "R40"),
            Self::R41 => write!(f, "R41"),
            Self::R42 => write!(f, "R42"),
            Self::R43 => write!(f, "R43"),
            Self::R44 => write!(f, "R44"),
            Self::R45 => write!(f, "R45"),
            Self::R46 => write!(f, "R46"),
            Self::R47 => write!(f, "R47"),
            Self::R48 => write!(f, "R48"),
            Self::R49 => write!(f, "R49"),
            Self::R50 => write!(f, "R50"),
            Self::R51 => write!(f, "R51"),
            Self::R52 => write!(f, "R52"),
            Self::R53 => write!(f, "R53"),
            Self::R54 => write!(f, "R54"),
            Self::R55 => write!(f, "R55"),
            Self::R56 => write!(f, "R56"),
            Self::R57 => write!(f, "R57"),
            Self::R58 => write!(f, "R58"),
            Self::R59 => write!(f, "R59"),
            Self::R60 => write!(f, "R60"),
            Self::R61 => write!(f, "R61"),
            Self::R62 => write!(f, "R62"),
            Self::R63 => write!(f, "R63"),
            Self::TR0 => write!(f, "TR0"),
            Self::TR1 => write!(f, "TR1"),
            Self::TR2 => write!(f, "TR2"),
            Self::TR3 => write!(f, "TR3"),
            Self::TR4 => write!(f, "TR4"),
            Self::TR5 => write!(f, "TR5"),
            Self::TR6 => write!(f, "TR6"),
            Self::TR7 => write!(f, "TR7"),
            Self::TR8 => write!(f, "TR8"),
            Self::TR9 => write!(f, "TR9"),
            Self::TR10 => write!(f, "TR10"),
            Self::TR11 => write!(f, "TR11"),
            Self::TR12 => write!(f, "TR12"),
            Self::TR13 => write!(f, "TR13"),
            Self::TR14 => write!(f, "TR14"),
            Self::TR15 => write!(f, "TR15"),
            Self::FR0 => write!(f, "FR0"),
            Self::FR1 => write!(f, "FR1"),
            Self::FR2 => write!(f, "FR2"),
            Self::FR3 => write!(f, "FR3"),
            Self::FR4 => write!(f, "FR4"),
            Self::FR5 => write!(f, "FR5"),
            Self::FR6 => write!(f, "FR6"),
            Self::FR7 => write!(f, "FR7"),
            Self::FR8 => write!(f, "FR8"),
            Self::FR9 => write!(f, "FR9"),
            Self::FR10 => write!(f, "FR10"),
            Self::FR11 => write!(f, "FR11"),
            Self::FR12 => write!(f, "FR12"),
            Self::FR13 => write!(f, "FR13"),
            Self::FR14 => write!(f, "FR14"),
            Self::FR15 => write!(f, "FR15"),
            Self::FR16 => write!(f, "FR16"),
            Self::FR17 => write!(f, "FR17"),
            Self::FR18 => write!(f, "FR18"),
            Self::FR19 => write!(f, "FR19"),
            Self::FR20 => write!(f, "FR20"),
            Self::FR21 => write!(f, "FR21"),
            Self::FR22 => write!(f, "FR22"),
            Self::FR23 => write!(f, "FR23"),
            Self::FR24 => write!(f, "FR24"),
            Self::FR25 => write!(f, "FR25"),
            Self::FR26 => write!(f, "FR26"),
            Self::FR27 => write!(f, "FR27"),
            Self::FR28 => write!(f, "FR28"),
            Self::FR29 => write!(f, "FR29"),
            Self::FR30 => write!(f, "FR30"),
            Self::FR31 => write!(f, "FR31"),
            Self::FR32 => write!(f, "FR32"),
            Self::FR33 => write!(f, "FR33"),
            Self::FR34 => write!(f, "FR34"),
            Self::FR35 => write!(f, "FR35"),
            Self::FR36 => write!(f, "FR36"),
            Self::FR37 => write!(f, "FR37"),
            Self::FR38 => write!(f, "FR38"),
            Self::FR39 => write!(f, "FR39"),
            Self::FR40 => write!(f, "FR40"),
            Self::FR41 => write!(f, "FR41"),
            Self::FR42 => write!(f, "FR42"),
            Self::FR43 => write!(f, "FR43"),
            Self::FR44 => write!(f, "FR44"),
            Self::FR45 => write!(f, "FR45"),
            Self::FR46 => write!(f, "FR46"),
            Self::FR47 => write!(f, "FR47"),
            Self::FR48 => write!(f, "FR48"),
            Self::FR49 => write!(f, "FR49"),
            Self::FR50 => write!(f, "FR50"),
            Self::FR51 => write!(f, "FR51"),
            Self::FR52 => write!(f, "FR52"),
            Self::FR53 => write!(f, "FR53"),
            Self::FR54 => write!(f, "FR54"),
            Self::FR55 => write!(f, "FR55"),
            Self::FR56 => write!(f, "FR56"),
            Self::FR57 => write!(f, "FR57"),
            Self::FR58 => write!(f, "FR58"),
            Self::FR59 => write!(f, "FR59"),
            Self::FR60 => write!(f, "FR60"),
            Self::FR61 => write!(f, "FR61"),
            Self::FR62 => write!(f, "FR62"),
            Self::FR63 => write!(f, "FR63"),
            Self::DR0 => write!(f, "DR0"),
            Self::DR2 => write!(f, "DR2"),
            Self::DR4 => write!(f, "DR4"),
            Self::DR6 => write!(f, "DR6"),
            Self::DR8 => write!(f, "DR8"),
            Self::DR10 => write!(f, "DR10"),
            Self::DR12 => write!(f, "DR12"),
            Self::DR14 => write!(f, "DR14"),
            Self::DR16 => write!(f, "DR16"),
            Self::DR18 => write!(f, "DR18"),
            Self::DR20 => write!(f, "DR20"),
            Self::DR22 => write!(f, "DR22"),
            Self::DR24 => write!(f, "DR24"),
            Self::DR26 => write!(f, "DR26"),
            Self::DR28 => write!(f, "DR28"),
            Self::DR30 => write!(f, "DR30"),
            Self::DR32 => write!(f, "DR32"),
            Self::DR34 => write!(f, "DR34"),
            Self::DR36 => write!(f, "DR36"),
            Self::DR38 => write!(f, "DR38"),
            Self::DR40 => write!(f, "DR40"),
            Self::DR42 => write!(f, "DR42"),
            Self::DR44 => write!(f, "DR44"),
            Self::DR46 => write!(f, "DR46"),
            Self::DR48 => write!(f, "DR48"),
            Self::DR50 => write!(f, "DR50"),
            Self::DR52 => write!(f, "DR52"),
            Self::DR54 => write!(f, "DR54"),
            Self::DR56 => write!(f, "DR56"),
            Self::DR58 => write!(f, "DR58"),
            Self::DR60 => write!(f, "DR60"),
            Self::DR62 => write!(f, "DR62"),
            Self::FV0 => write!(f, "FV0"),
            Self::FV4 => write!(f, "FV4"),
            Self::FV8 => write!(f, "FV8"),
            Self::FV12 => write!(f, "FV12"),
            Self::FV16 => write!(f, "FV16"),
            Self::FV20 => write!(f, "FV20"),
            Self::FV24 => write!(f, "FV24"),
            Self::FV28 => write!(f, "FV28"),
            Self::FV32 => write!(f, "FV32"),
            Self::FV36 => write!(f, "FV36"),
            Self::FV40 => write!(f, "FV40"),
            Self::FV44 => write!(f, "FV44"),
            Self::FV48 => write!(f, "FV48"),
            Self::FV52 => write!(f, "FV52"),
            Self::FV56 => write!(f, "FV56"),
            Self::FV60 => write!(f, "FV60"),
            Self::MTRX0 => write!(f, "MTRX0"),
            Self::MTRX16 => write!(f, "MTRX16"),
            Self::MTRX32 => write!(f, "MTRX32"),
            Self::MTRX48 => write!(f, "MTRX48"),
            Self::CR0 => write!(f, "CR0"),
            Self::CR1 => write!(f, "CR1"),
            Self::CR2 => write!(f, "CR2"),
            Self::CR3 => write!(f, "CR3"),
            Self::CR4 => write!(f, "CR4"),
            Self::CR5 => write!(f, "CR5"),
            Self::CR6 => write!(f, "CR6"),
            Self::CR7 => write!(f, "CR7"),
            Self::CR8 => write!(f, "CR8"),
            Self::CR9 => write!(f, "CR9"),
            Self::CR10 => write!(f, "CR10"),
            Self::CR11 => write!(f, "CR11"),
            Self::CR12 => write!(f, "CR12"),
            Self::CR13 => write!(f, "CR13"),
            Self::CR14 => write!(f, "CR14"),
            Self::CR15 => write!(f, "CR15"),
            Self::CR16 => write!(f, "CR16"),
            Self::CR17 => write!(f, "CR17"),
            Self::CR18 => write!(f, "CR18"),
            Self::CR19 => write!(f, "CR19"),
            Self::CR20 => write!(f, "CR20"),
            Self::CR21 => write!(f, "CR21"),
            Self::CR22 => write!(f, "CR22"),
            Self::CR23 => write!(f, "CR23"),
            Self::CR24 => write!(f, "CR24"),
            Self::CR25 => write!(f, "CR25"),
            Self::CR26 => write!(f, "CR26"),
            Self::CR27 => write!(f, "CR27"),
            Self::CR28 => write!(f, "CR28"),
            Self::CR29 => write!(f, "CR29"),
            Self::CR30 => write!(f, "CR30"),
            Self::CR31 => write!(f, "CR31"),
            Self::CR32 => write!(f, "CR32"),
            Self::CR33 => write!(f, "CR33"),
            Self::CR34 => write!(f, "CR34"),
            Self::CR35 => write!(f, "CR35"),
            Self::CR36 => write!(f, "CR36"),
            Self::CR37 => write!(f, "CR37"),
            Self::CR38 => write!(f, "CR38"),
            Self::CR39 => write!(f, "CR39"),
            Self::CR40 => write!(f, "CR40"),
            Self::CR41 => write!(f, "CR41"),
            Self::CR42 => write!(f, "CR42"),
            Self::CR43 => write!(f, "CR43"),
            Self::CR44 => write!(f, "CR44"),
            Self::CR45 => write!(f, "CR45"),
            Self::CR46 => write!(f, "CR46"),
            Self::CR47 => write!(f, "CR47"),
            Self::CR48 => write!(f, "CR48"),
            Self::CR49 => write!(f, "CR49"),
            Self::CR50 => write!(f, "CR50"),
            Self::CR51 => write!(f, "CR51"),
            Self::CR52 => write!(f, "CR52"),
            Self::CR53 => write!(f, "CR53"),
            Self::CR54 => write!(f, "CR54"),
            Self::CR55 => write!(f, "CR55"),
            Self::CR56 => write!(f, "CR56"),
            Self::CR57 => write!(f, "CR57"),
            Self::CR58 => write!(f, "CR58"),
            Self::CR59 => write!(f, "CR59"),
            Self::CR60 => write!(f, "CR60"),
            Self::CR61 => write!(f, "CR61"),
            Self::CR62 => write!(f, "CR62"),
            Self::CR63 => write!(f, "CR63"),
            Self::FPSCR => write!(f, "FPSCR"),
        }
    }
}

impl TryFrom<u16> for SuperHSHMediaRegister {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            10 => Ok(Self::R0),
            11 => Ok(Self::R1),
            12 => Ok(Self::R2),
            13 => Ok(Self::R3),
            14 => Ok(Self::R4),
            15 => Ok(Self::R5),
            16 => Ok(Self::R6),
            17 => Ok(Self::R7),
            18 => Ok(Self::R8),
            19 => Ok(Self::R9),
            20 => Ok(Self::R10),
            21 => Ok(Self::R11),
            22 => Ok(Self::R12),
            23 => Ok(Self::R13),
            24 => Ok(Self::R14),
            25 => Ok(Self::R15),
            26 => Ok(Self::R16),
            27 => Ok(Self::R17),
            28 => Ok(Self::R18),
            29 => Ok(Self::R19),
            30 => Ok(Self::R20),
            31 => Ok(Self::R21),
            32 => Ok(Self::R22),
            33 => Ok(Self::R23),
            34 => Ok(Self::R24),
            35 => Ok(Self::R25),
            36 => Ok(Self::R26),
            37 => Ok(Self::R27),
            38 => Ok(Self::R28),
            39 => Ok(Self::R29),
            40 => Ok(Self::R30),
            41 => Ok(Self::R31),
            42 => Ok(Self::R32),
            43 => Ok(Self::R33),
            44 => Ok(Self::R34),
            45 => Ok(Self::R35),
            46 => Ok(Self::R36),
            47 => Ok(Self::R37),
            48 => Ok(Self::R38),
            49 => Ok(Self::R39),
            50 => Ok(Self::R40),
            51 => Ok(Self::R41),
            52 => Ok(Self::R42),
            53 => Ok(Self::R43),
            54 => Ok(Self::R44),
            55 => Ok(Self::R45),
            56 => Ok(Self::R46),
            57 => Ok(Self::R47),
            58 => Ok(Self::R48),
            59 => Ok(Self::R49),
            60 => Ok(Self::R50),
            61 => Ok(Self::R51),
            62 => Ok(Self::R52),
            63 => Ok(Self::R53),
            64 => Ok(Self::R54),
            65 => Ok(Self::R55),
            66 => Ok(Self::R56),
            67 => Ok(Self::R57),
            68 => Ok(Self::R58),
            69 => Ok(Self::R59),
            70 => Ok(Self::R60),
            71 => Ok(Self::R61),
            72 => Ok(Self::R62),
            73 => Ok(Self::R63),
            74 => Ok(Self::TR0),
            75 => Ok(Self::TR1),
            76 => Ok(Self::TR2),
            77 => Ok(Self::TR3),
            78 => Ok(Self::TR4),
            79 => Ok(Self::TR5),
            80 => Ok(Self::TR6),
            81 => Ok(Self::TR7),
            82 => Ok(Self::TR8),
            83 => Ok(Self::TR9),
            84 => Ok(Self::TR10),
            85 => Ok(Self::TR11),
            86 => Ok(Self::TR12),
            87 => Ok(Self::TR13),
            88 => Ok(Self::TR14),
            89 => Ok(Self::TR15),
            128 => Ok(Self::FR0),
            129 => Ok(Self::FR1),
            130 => Ok(Self::FR2),
            131 => Ok(Self::FR3),
            132 => Ok(Self::FR4),
            133 => Ok(Self::FR5),
            134 => Ok(Self::FR6),
            135 => Ok(Self::FR7),
            136 => Ok(Self::FR8),
            137 => Ok(Self::FR9),
            138 => Ok(Self::FR10),
            139 => Ok(Self::FR11),
            140 => Ok(Self::FR12),
            141 => Ok(Self::FR13),
            142 => Ok(Self::FR14),
            143 => Ok(Self::FR15),
            144 => Ok(Self::FR16),
            145 => Ok(Self::FR17),
            146 => Ok(Self::FR18),
            147 => Ok(Self::FR19),
            148 => Ok(Self::FR20),
            149 => Ok(Self::FR21),
            150 => Ok(Self::FR22),
            151 => Ok(Self::FR23),
            152 => Ok(Self::FR24),
            153 => Ok(Self::FR25),
            154 => Ok(Self::FR26),
            155 => Ok(Self::FR27),
            156 => Ok(Self::FR28),
            157 => Ok(Self::FR29),
            158 => Ok(Self::FR30),
            159 => Ok(Self::FR31),
            160 => Ok(Self::FR32),
            161 => Ok(Self::FR33),
            162 => Ok(Self::FR34),
            163 => Ok(Self::FR35),
            164 => Ok(Self::FR36),
            165 => Ok(Self::FR37),
            166 => Ok(Self::FR38),
            167 => Ok(Self::FR39),
            168 => Ok(Self::FR40),
            169 => Ok(Self::FR41),
            170 => Ok(Self::FR42),
            171 => Ok(Self::FR43),
            172 => Ok(Self::FR44),
            173 => Ok(Self::FR45),
            174 => Ok(Self::FR46),
            175 => Ok(Self::FR47),
            176 => Ok(Self::FR48),
            177 => Ok(Self::FR49),
            178 => Ok(Self::FR50),
            179 => Ok(Self::FR51),
            180 => Ok(Self::FR52),
            181 => Ok(Self::FR53),
            182 => Ok(Self::FR54),
            183 => Ok(Self::FR55),
            184 => Ok(Self::FR56),
            185 => Ok(Self::FR57),
            186 => Ok(Self::FR58),
            187 => Ok(Self::FR59),
            188 => Ok(Self::FR60),
            189 => Ok(Self::FR61),
            190 => Ok(Self::FR62),
            191 => Ok(Self::FR63),
            256 => Ok(Self::DR0),
            258 => Ok(Self::DR2),
            260 => Ok(Self::DR4),
            262 => Ok(Self::DR6),
            264 => Ok(Self::DR8),
            266 => Ok(Self::DR10),
            268 => Ok(Self::DR12),
            270 => Ok(Self::DR14),
            272 => Ok(Self::DR16),
            274 => Ok(Self::DR18),
            276 => Ok(Self::DR20),
            278 => Ok(Self::DR22),
            280 => Ok(Self::DR24),
            282 => Ok(Self::DR26),
            284 => Ok(Self::DR28),
            286 => Ok(Self::DR30),
            288 => Ok(Self::DR32),
            290 => Ok(Self::DR34),
            292 => Ok(Self::DR36),
            294 => Ok(Self::DR38),
            296 => Ok(Self::DR40),
            298 => Ok(Self::DR42),
            300 => Ok(Self::DR44),
            302 => Ok(Self::DR46),
            304 => Ok(Self::DR48),
            306 => Ok(Self::DR50),
            308 => Ok(Self::DR52),
            310 => Ok(Self::DR54),
            312 => Ok(Self::DR56),
            314 => Ok(Self::DR58),
            316 => Ok(Self::DR60),
            318 => Ok(Self::DR62),
            512 => Ok(Self::FV0),
            516 => Ok(Self::FV4),
            520 => Ok(Self::FV8),
            524 => Ok(Self::FV12),
            528 => Ok(Self::FV16),
            532 => Ok(Self::FV20),
            536 => Ok(Self::FV24),
            540 => Ok(Self::FV28),
            544 => Ok(Self::FV32),
            548 => Ok(Self::FV36),
            552 => Ok(Self::FV40),
            556 => Ok(Self::FV44),
            560 => Ok(Self::FV48),
            564 => Ok(Self::FV52),
            568 => Ok(Self::FV56),
            572 => Ok(Self::FV60),
            1024 => Ok(Self::MTRX0),
            1040 => Ok(Self::MTRX16),
            1056 => Ok(Self::MTRX32),
            1072 => Ok(Self::MTRX48),
            2000 => Ok(Self::CR0),
            2001 => Ok(Self::CR1),
            2002 => Ok(Self::CR2),
            2003 => Ok(Self::CR3),
            2004 => Ok(Self::CR4),
            2005 => Ok(Self::CR5),
            2006 => Ok(Self::CR6),
            2007 => Ok(Self::CR7),
            2008 => Ok(Self::CR8),
            2009 => Ok(Self::CR9),
            2010 => Ok(Self::CR10),
            2011 => Ok(Self::CR11),
            2012 => Ok(Self::CR12),
            2013 => Ok(Self::CR13),
            2014 => Ok(Self::CR14),
            2015 => Ok(Self::CR15),
            2016 => Ok(Self::CR16),
            2017 => Ok(Self::CR17),
            2018 => Ok(Self::CR18),
            2019 => Ok(Self::CR19),
            2020 => Ok(Self::CR20),
            2021 => Ok(Self::CR21),
            2022 => Ok(Self::CR22),
            2023 => Ok(Self::CR23),
            2024 => Ok(Self::CR24),
            2025 => Ok(Self::CR25),
            2026 => Ok(Self::CR26),
            2027 => Ok(Self::CR27),
            2028 => Ok(Self::CR28),
            2029 => Ok(Self::CR29),
            2030 => Ok(Self::CR30),
            2031 => Ok(Self::CR31),
            2032 => Ok(Self::CR32),
            2033 => Ok(Self::CR33),
            2034 => Ok(Self::CR34),
            2035 => Ok(Self::CR35),
            2036 => Ok(Self::CR36),
            2037 => Ok(Self::CR37),
            2038 => Ok(Self::CR38),
            2039 => Ok(Self::CR39),
            2040 => Ok(Self::CR40),
            2041 => Ok(Self::CR41),
            2042 => Ok(Self::CR42),
            2043 => Ok(Self::CR43),
            2044 => Ok(Self::CR44),
            2045 => Ok(Self::CR45),
            2046 => Ok(Self::CR46),
            2047 => Ok(Self::CR47),
            2048 => Ok(Self::CR48),
            2049 => Ok(Self::CR49),
            2050 => Ok(Self::CR50),
            2051 => Ok(Self::CR51),
            2052 => Ok(Self::CR52),
            2053 => Ok(Self::CR53),
            2054 => Ok(Self::CR54),
            2055 => Ok(Self::CR55),
            2056 => Ok(Self::CR56),
            2057 => Ok(Self::CR57),
            2058 => Ok(Self::CR58),
            2059 => Ok(Self::CR59),
            2060 => Ok(Self::CR60),
            2061 => Ok(Self::CR61),
            2062 => Ok(Self::CR62),
            2063 => Ok(Self::CR63),
            2064 => Ok(Self::FPSCR),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for SuperHSHMediaRegister {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// AMD64 registers
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum AMD64Register {
    AL = 1,
    CL = 2,
    DL = 3,
    BL = 4,
    AH = 5,
    CH = 6,
    DH = 7,
    BH = 8,
    AX = 9,
    CX = 10,
    DX = 11,
    BX = 12,
    SP = 13,
    BP = 14,
    SI = 15,
    DI = 16,
    EAX = 17,
    ECX = 18,
    EDX = 19,
    EBX = 20,
    ESP = 21,
    EBP = 22,
    ESI = 23,
    EDI = 24,
    ES = 25,
    CS = 26,
    SS = 27,
    DS = 28,
    FS = 29,
    GS = 30,
    FLAGS = 32,
    RIP = 33,
    EFLAGS = 34,

    // Control registers
    CR0 = 80,
    CR1 = 81,
    CR2 = 82,
    CR3 = 83,
    CR4 = 84,
    CR8 = 88,

    // Debug registers
    DR0 = 90,
    DR1 = 91,
    DR2 = 92,
    DR3 = 93,
    DR4 = 94,
    DR5 = 95,
    DR6 = 96,
    DR7 = 97,
    DR8 = 98,
    DR9 = 99,
    DR10 = 100,
    DR11 = 101,
    DR12 = 102,
    DR13 = 103,
    DR14 = 104,
    DR15 = 105,

    GDTR = 110,
    GDTL = 111,
    IDTR = 112,
    IDTL = 113,
    LDTR = 114,
    TR = 115,

    ST0 = 128,
    ST1 = 129,
    ST2 = 130,
    ST3 = 131,
    ST4 = 132,
    ST5 = 133,
    ST6 = 134,
    ST7 = 135,
    CTRL = 136,
    STAT = 137,
    TAG = 138,
    FPIP = 139,
    FPCS = 140,
    FPDO = 141,
    FPDS = 142,
    ISEM = 143,
    FPEIP = 144,
    FPEDO = 145,

    MM0 = 146,
    MM1 = 147,
    MM2 = 148,
    MM3 = 149,
    MM4 = 150,
    MM5 = 151,
    MM6 = 152,
    MM7 = 153,

    XMM0 = 154, // KATMAI registers
    XMM1 = 155,
    XMM2 = 156,
    XMM3 = 157,
    XMM4 = 158,
    XMM5 = 159,
    XMM6 = 160,
    XMM7 = 161,

    XMM0_0 = 162, // KATMAI sub-registers
    XMM0_1 = 163,
    XMM0_2 = 164,
    XMM0_3 = 165,
    XMM1_0 = 166,
    XMM1_1 = 167,
    XMM1_2 = 168,
    XMM1_3 = 169,
    XMM2_0 = 170,
    XMM2_1 = 171,
    XMM2_2 = 172,
    XMM2_3 = 173,
    XMM3_0 = 174,
    XMM3_1 = 175,
    XMM3_2 = 176,
    XMM3_3 = 177,
    XMM4_0 = 178,
    XMM4_1 = 179,
    XMM4_2 = 180,
    XMM4_3 = 181,
    XMM5_0 = 182,
    XMM5_1 = 183,
    XMM5_2 = 184,
    XMM5_3 = 185,
    XMM6_0 = 186,
    XMM6_1 = 187,
    XMM6_2 = 188,
    XMM6_3 = 189,
    XMM7_0 = 190,
    XMM7_1 = 191,
    XMM7_2 = 192,
    XMM7_3 = 193,

    XMM0L = 194,
    XMM1L = 195,
    XMM2L = 196,
    XMM3L = 197,
    XMM4L = 198,
    XMM5L = 199,
    XMM6L = 200,
    XMM7L = 201,

    XMM0H = 202,
    XMM1H = 203,
    XMM2H = 204,
    XMM3H = 205,
    XMM4H = 206,
    XMM5H = 207,
    XMM6H = 208,
    XMM7H = 209,

    MXCSR = 211, // XMM status register

    EMM0L = 220, // XMM sub-registers (WNI integer)
    EMM1L = 221,
    EMM2L = 222,
    EMM3L = 223,
    EMM4L = 224,
    EMM5L = 225,
    EMM6L = 226,
    EMM7L = 227,

    EMM0H = 228,
    EMM1H = 229,
    EMM2H = 230,
    EMM3H = 231,
    EMM4H = 232,
    EMM5H = 233,
    EMM6H = 234,
    EMM7H = 235,

    // do not change the order of these regs, first one must be even too
    MM00 = 236,
    MM01 = 237,
    MM10 = 238,
    MM11 = 239,
    MM20 = 240,
    MM21 = 241,
    MM30 = 242,
    MM31 = 243,
    MM40 = 244,
    MM41 = 245,
    MM50 = 246,
    MM51 = 247,
    MM60 = 248,
    MM61 = 249,
    MM70 = 250,
    MM71 = 251,

    // Extended KATMAI registers
    XMM8 = 252, // KATMAI registers
    XMM9 = 253,
    XMM10 = 254,
    XMM11 = 255,
    XMM12 = 256,
    XMM13 = 257,
    XMM14 = 258,
    XMM15 = 259,

    XMM8_0 = 260, // KATMAI sub-registers
    XMM8_1 = 261,
    XMM8_2 = 262,
    XMM8_3 = 263,
    XMM9_0 = 264,
    XMM9_1 = 265,
    XMM9_2 = 266,
    XMM9_3 = 267,
    XMM10_0 = 268,
    XMM10_1 = 269,
    XMM10_2 = 270,
    XMM10_3 = 271,
    XMM11_0 = 272,
    XMM11_1 = 273,
    XMM11_2 = 274,
    XMM11_3 = 275,
    XMM12_0 = 276,
    XMM12_1 = 277,
    XMM12_2 = 278,
    XMM12_3 = 279,
    XMM13_0 = 280,
    XMM13_1 = 281,
    XMM13_2 = 282,
    XMM13_3 = 283,
    XMM14_0 = 284,
    XMM14_1 = 285,
    XMM14_2 = 286,
    XMM14_3 = 287,
    XMM15_0 = 288,
    XMM15_1 = 289,
    XMM15_2 = 290,
    XMM15_3 = 291,

    XMM8L = 292,
    XMM9L = 293,
    XMM10L = 294,
    XMM11L = 295,
    XMM12L = 296,
    XMM13L = 297,
    XMM14L = 298,
    XMM15L = 299,

    XMM8H = 300,
    XMM9H = 301,
    XMM10H = 302,
    XMM11H = 303,
    XMM12H = 304,
    XMM13H = 305,
    XMM14H = 306,
    XMM15H = 307,

    EMM8L = 308, // XMM sub-registers (WNI integer)
    EMM9L = 309,
    EMM10L = 310,
    EMM11L = 311,
    EMM12L = 312,
    EMM13L = 313,
    EMM14L = 314,
    EMM15L = 315,

    EMM8H = 316,
    EMM9H = 317,
    EMM10H = 318,
    EMM11H = 319,
    EMM12H = 320,
    EMM13H = 321,
    EMM14H = 322,
    EMM15H = 323,

    // Low byte forms of some standard registers
    SIL = 324,
    DIL = 325,
    BPL = 326,
    SPL = 327,

    // 64-bit regular registers
    RAX = 328,
    RBX = 329,
    RCX = 330,
    RDX = 331,
    RSI = 332,
    RDI = 333,
    RBP = 334,
    RSP = 335,

    // 64-bit integer registers with 8-, 16-, and 32-bit forms (B, W, and D)
    R8 = 336,
    R9 = 337,
    R10 = 338,
    R11 = 339,
    R12 = 340,
    R13 = 341,
    R14 = 342,
    R15 = 343,

    R8B = 344,
    R9B = 345,
    R10B = 346,
    R11B = 347,
    R12B = 348,
    R13B = 349,
    R14B = 350,
    R15B = 351,

    R8W = 352,
    R9W = 353,
    R10W = 354,
    R11W = 355,
    R12W = 356,
    R13W = 357,
    R14W = 358,
    R15W = 359,

    R8D = 360,
    R9D = 361,
    R10D = 362,
    R11D = 363,
    R12D = 364,
    R13D = 365,
    R14D = 366,
    R15D = 367,

    // AVX registers 256 bits
    YMM0 = 368,
    YMM1 = 369,
    YMM2 = 370,
    YMM3 = 371,
    YMM4 = 372,
    YMM5 = 373,
    YMM6 = 374,
    YMM7 = 375,
    YMM8 = 376,
    YMM9 = 377,
    YMM10 = 378,
    YMM11 = 379,
    YMM12 = 380,
    YMM13 = 381,
    YMM14 = 382,
    YMM15 = 383,

    // AVX registers upper 128 bits
    YMM0H = 384,
    YMM1H = 385,
    YMM2H = 386,
    YMM3H = 387,
    YMM4H = 388,
    YMM5H = 389,
    YMM6H = 390,
    YMM7H = 391,
    YMM8H = 392,
    YMM9H = 393,
    YMM10H = 394,
    YMM11H = 395,
    YMM12H = 396,
    YMM13H = 397,
    YMM14H = 398,
    YMM15H = 399,

    //Lower/upper 8 bytes of XMM registers.  Unlike XMM<regnum><H/L>, these
    //values reprsesent the bit patterns of the registers as 64-bit integers, not
    //the representation of these registers as a double.
    XMM0IL = 400,
    XMM1IL = 401,
    XMM2IL = 402,
    XMM3IL = 403,
    XMM4IL = 404,
    XMM5IL = 405,
    XMM6IL = 406,
    XMM7IL = 407,
    XMM8IL = 408,
    XMM9IL = 409,
    XMM10IL = 410,
    XMM11IL = 411,
    XMM12IL = 412,
    XMM13IL = 413,
    XMM14IL = 414,
    XMM15IL = 415,

    XMM0IH = 416,
    XMM1IH = 417,
    XMM2IH = 418,
    XMM3IH = 419,
    XMM4IH = 420,
    XMM5IH = 421,
    XMM6IH = 422,
    XMM7IH = 423,
    XMM8IH = 424,
    XMM9IH = 425,
    XMM10IH = 426,
    XMM11IH = 427,
    XMM12IH = 428,
    XMM13IH = 429,
    XMM14IH = 430,
    XMM15IH = 431,

    YMM0I0 = 432, // AVX integer registers
    YMM0I1 = 433,
    YMM0I2 = 434,
    YMM0I3 = 435,
    YMM1I0 = 436,
    YMM1I1 = 437,
    YMM1I2 = 438,
    YMM1I3 = 439,
    YMM2I0 = 440,
    YMM2I1 = 441,
    YMM2I2 = 442,
    YMM2I3 = 443,
    YMM3I0 = 444,
    YMM3I1 = 445,
    YMM3I2 = 446,
    YMM3I3 = 447,
    YMM4I0 = 448,
    YMM4I1 = 449,
    YMM4I2 = 450,
    YMM4I3 = 451,
    YMM5I0 = 452,
    YMM5I1 = 453,
    YMM5I2 = 454,
    YMM5I3 = 455,
    YMM6I0 = 456,
    YMM6I1 = 457,
    YMM6I2 = 458,
    YMM6I3 = 459,
    YMM7I0 = 460,
    YMM7I1 = 461,
    YMM7I2 = 462,
    YMM7I3 = 463,
    YMM8I0 = 464,
    YMM8I1 = 465,
    YMM8I2 = 466,
    YMM8I3 = 467,
    YMM9I0 = 468,
    YMM9I1 = 469,
    YMM9I2 = 470,
    YMM9I3 = 471,
    YMM10I0 = 472,
    YMM10I1 = 473,
    YMM10I2 = 474,
    YMM10I3 = 475,
    YMM11I0 = 476,
    YMM11I1 = 477,
    YMM11I2 = 478,
    YMM11I3 = 479,
    YMM12I0 = 480,
    YMM12I1 = 481,
    YMM12I2 = 482,
    YMM12I3 = 483,
    YMM13I0 = 484,
    YMM13I1 = 485,
    YMM13I2 = 486,
    YMM13I3 = 487,
    YMM14I0 = 488,
    YMM14I1 = 489,
    YMM14I2 = 490,
    YMM14I3 = 491,
    YMM15I0 = 492,
    YMM15I1 = 493,
    YMM15I2 = 494,
    YMM15I3 = 495,

    YMM0F0 = 496, // AVX floating-point single precise registers
    YMM0F1 = 497,
    YMM0F2 = 498,
    YMM0F3 = 499,
    YMM0F4 = 500,
    YMM0F5 = 501,
    YMM0F6 = 502,
    YMM0F7 = 503,
    YMM1F0 = 504,
    YMM1F1 = 505,
    YMM1F2 = 506,
    YMM1F3 = 507,
    YMM1F4 = 508,
    YMM1F5 = 509,
    YMM1F6 = 510,
    YMM1F7 = 511,
    YMM2F0 = 512,
    YMM2F1 = 513,
    YMM2F2 = 514,
    YMM2F3 = 515,
    YMM2F4 = 516,
    YMM2F5 = 517,
    YMM2F6 = 518,
    YMM2F7 = 519,
    YMM3F0 = 520,
    YMM3F1 = 521,
    YMM3F2 = 522,
    YMM3F3 = 523,
    YMM3F4 = 524,
    YMM3F5 = 525,
    YMM3F6 = 526,
    YMM3F7 = 527,
    YMM4F0 = 528,
    YMM4F1 = 529,
    YMM4F2 = 530,
    YMM4F3 = 531,
    YMM4F4 = 532,
    YMM4F5 = 533,
    YMM4F6 = 534,
    YMM4F7 = 535,
    YMM5F0 = 536,
    YMM5F1 = 537,
    YMM5F2 = 538,
    YMM5F3 = 539,
    YMM5F4 = 540,
    YMM5F5 = 541,
    YMM5F6 = 542,
    YMM5F7 = 543,
    YMM6F0 = 544,
    YMM6F1 = 545,
    YMM6F2 = 546,
    YMM6F3 = 547,
    YMM6F4 = 548,
    YMM6F5 = 549,
    YMM6F6 = 550,
    YMM6F7 = 551,
    YMM7F0 = 552,
    YMM7F1 = 553,
    YMM7F2 = 554,
    YMM7F3 = 555,
    YMM7F4 = 556,
    YMM7F5 = 557,
    YMM7F6 = 558,
    YMM7F7 = 559,
    YMM8F0 = 560,
    YMM8F1 = 561,
    YMM8F2 = 562,
    YMM8F3 = 563,
    YMM8F4 = 564,
    YMM8F5 = 565,
    YMM8F6 = 566,
    YMM8F7 = 567,
    YMM9F0 = 568,
    YMM9F1 = 569,
    YMM9F2 = 570,
    YMM9F3 = 571,
    YMM9F4 = 572,
    YMM9F5 = 573,
    YMM9F6 = 574,
    YMM9F7 = 575,
    YMM10F0 = 576,
    YMM10F1 = 577,
    YMM10F2 = 578,
    YMM10F3 = 579,
    YMM10F4 = 580,
    YMM10F5 = 581,
    YMM10F6 = 582,
    YMM10F7 = 583,
    YMM11F0 = 584,
    YMM11F1 = 585,
    YMM11F2 = 586,
    YMM11F3 = 587,
    YMM11F4 = 588,
    YMM11F5 = 589,
    YMM11F6 = 590,
    YMM11F7 = 591,
    YMM12F0 = 592,
    YMM12F1 = 593,
    YMM12F2 = 594,
    YMM12F3 = 595,
    YMM12F4 = 596,
    YMM12F5 = 597,
    YMM12F6 = 598,
    YMM12F7 = 599,
    YMM13F0 = 600,
    YMM13F1 = 601,
    YMM13F2 = 602,
    YMM13F3 = 603,
    YMM13F4 = 604,
    YMM13F5 = 605,
    YMM13F6 = 606,
    YMM13F7 = 607,
    YMM14F0 = 608,
    YMM14F1 = 609,
    YMM14F2 = 610,
    YMM14F3 = 611,
    YMM14F4 = 612,
    YMM14F5 = 613,
    YMM14F6 = 614,
    YMM14F7 = 615,
    YMM15F0 = 616,
    YMM15F1 = 617,
    YMM15F2 = 618,
    YMM15F3 = 619,
    YMM15F4 = 620,
    YMM15F5 = 621,
    YMM15F6 = 622,
    YMM15F7 = 623,

    YMM0D0 = 624, // AVX floating-point double precise registers
    YMM0D1 = 625,
    YMM0D2 = 626,
    YMM0D3 = 627,
    YMM1D0 = 628,
    YMM1D1 = 629,
    YMM1D2 = 630,
    YMM1D3 = 631,
    YMM2D0 = 632,
    YMM2D1 = 633,
    YMM2D2 = 634,
    YMM2D3 = 635,
    YMM3D0 = 636,
    YMM3D1 = 637,
    YMM3D2 = 638,
    YMM3D3 = 639,
    YMM4D0 = 640,
    YMM4D1 = 641,
    YMM4D2 = 642,
    YMM4D3 = 643,
    YMM5D0 = 644,
    YMM5D1 = 645,
    YMM5D2 = 646,
    YMM5D3 = 647,
    YMM6D0 = 648,
    YMM6D1 = 649,
    YMM6D2 = 650,
    YMM6D3 = 651,
    YMM7D0 = 652,
    YMM7D1 = 653,
    YMM7D2 = 654,
    YMM7D3 = 655,
    YMM8D0 = 656,
    YMM8D1 = 657,
    YMM8D2 = 658,
    YMM8D3 = 659,
    YMM9D0 = 660,
    YMM9D1 = 661,
    YMM9D2 = 662,
    YMM9D3 = 663,
    YMM10D0 = 664,
    YMM10D1 = 665,
    YMM10D2 = 666,
    YMM10D3 = 667,
    YMM11D0 = 668,
    YMM11D1 = 669,
    YMM11D2 = 670,
    YMM11D3 = 671,
    YMM12D0 = 672,
    YMM12D1 = 673,
    YMM12D2 = 674,
    YMM12D3 = 675,
    YMM13D0 = 676,
    YMM13D1 = 677,
    YMM13D2 = 678,
    YMM13D3 = 679,
    YMM14D0 = 680,
    YMM14D1 = 681,
    YMM14D2 = 682,
    YMM14D3 = 683,
    YMM15D0 = 684,
    YMM15D1 = 685,
    YMM15D2 = 686,
    YMM15D3 = 687,
}

impl fmt::Display for AMD64Register {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::AL => write!(f, "AL"),
            Self::CL => write!(f, "CL"),
            Self::DL => write!(f, "DL"),
            Self::BL => write!(f, "BL"),
            Self::AH => write!(f, "AH"),
            Self::CH => write!(f, "CH"),
            Self::DH => write!(f, "DH"),
            Self::BH => write!(f, "BH"),
            Self::AX => write!(f, "AX"),
            Self::CX => write!(f, "CX"),
            Self::DX => write!(f, "DX"),
            Self::BX => write!(f, "BX"),
            Self::SP => write!(f, "SP"),
            Self::BP => write!(f, "BP"),
            Self::SI => write!(f, "SI"),
            Self::DI => write!(f, "DI"),
            Self::EAX => write!(f, "EAX"),
            Self::ECX => write!(f, "ECX"),
            Self::EDX => write!(f, "EDX"),
            Self::EBX => write!(f, "EBX"),
            Self::ESP => write!(f, "ESP"),
            Self::EBP => write!(f, "EBP"),
            Self::ESI => write!(f, "ESI"),
            Self::EDI => write!(f, "EDI"),
            Self::ES => write!(f, "ES"),
            Self::CS => write!(f, "CS"),
            Self::SS => write!(f, "SS"),
            Self::DS => write!(f, "DS"),
            Self::FS => write!(f, "FS"),
            Self::GS => write!(f, "GS"),
            Self::FLAGS => write!(f, "FLAGS"),
            Self::RIP => write!(f, "RIP"),
            Self::EFLAGS => write!(f, "EFLAGS"),
            Self::CR0 => write!(f, "CR0"),
            Self::CR1 => write!(f, "CR1"),
            Self::CR2 => write!(f, "CR2"),
            Self::CR3 => write!(f, "CR3"),
            Self::CR4 => write!(f, "CR4"),
            Self::CR8 => write!(f, "CR8"),
            Self::DR0 => write!(f, "DR0"),
            Self::DR1 => write!(f, "DR1"),
            Self::DR2 => write!(f, "DR2"),
            Self::DR3 => write!(f, "DR3"),
            Self::DR4 => write!(f, "DR4"),
            Self::DR5 => write!(f, "DR5"),
            Self::DR6 => write!(f, "DR6"),
            Self::DR7 => write!(f, "DR7"),
            Self::DR8 => write!(f, "DR8"),
            Self::DR9 => write!(f, "DR9"),
            Self::DR10 => write!(f, "DR10"),
            Self::DR11 => write!(f, "DR11"),
            Self::DR12 => write!(f, "DR12"),
            Self::DR13 => write!(f, "DR13"),
            Self::DR14 => write!(f, "DR14"),
            Self::DR15 => write!(f, "DR15"),
            Self::GDTR => write!(f, "GDTR"),
            Self::GDTL => write!(f, "GDTL"),
            Self::IDTR => write!(f, "IDTR"),
            Self::IDTL => write!(f, "IDTL"),
            Self::LDTR => write!(f, "LDTR"),
            Self::TR => write!(f, "TR"),
            Self::ST0 => write!(f, "ST0"),
            Self::ST1 => write!(f, "ST1"),
            Self::ST2 => write!(f, "ST2"),
            Self::ST3 => write!(f, "ST3"),
            Self::ST4 => write!(f, "ST4"),
            Self::ST5 => write!(f, "ST5"),
            Self::ST6 => write!(f, "ST6"),
            Self::ST7 => write!(f, "ST7"),
            Self::CTRL => write!(f, "CTRL"),
            Self::STAT => write!(f, "STAT"),
            Self::TAG => write!(f, "TAG"),
            Self::FPIP => write!(f, "FPIP"),
            Self::FPCS => write!(f, "FPCS"),
            Self::FPDO => write!(f, "FPDO"),
            Self::FPDS => write!(f, "FPDS"),
            Self::ISEM => write!(f, "ISEM"),
            Self::FPEIP => write!(f, "FPEIP"),
            Self::FPEDO => write!(f, "FPEDO"),
            Self::MM0 => write!(f, "MM0"),
            Self::MM1 => write!(f, "MM1"),
            Self::MM2 => write!(f, "MM2"),
            Self::MM3 => write!(f, "MM3"),
            Self::MM4 => write!(f, "MM4"),
            Self::MM5 => write!(f, "MM5"),
            Self::MM6 => write!(f, "MM6"),
            Self::MM7 => write!(f, "MM7"),
            Self::XMM0 => write!(f, "XMM0"),
            Self::XMM1 => write!(f, "XMM1"),
            Self::XMM2 => write!(f, "XMM2"),
            Self::XMM3 => write!(f, "XMM3"),
            Self::XMM4 => write!(f, "XMM4"),
            Self::XMM5 => write!(f, "XMM5"),
            Self::XMM6 => write!(f, "XMM6"),
            Self::XMM7 => write!(f, "XMM7"),
            Self::XMM0_0 => write!(f, "XMM0_0"),
            Self::XMM0_1 => write!(f, "XMM0_1"),
            Self::XMM0_2 => write!(f, "XMM0_2"),
            Self::XMM0_3 => write!(f, "XMM0_3"),
            Self::XMM1_0 => write!(f, "XMM1_0"),
            Self::XMM1_1 => write!(f, "XMM1_1"),
            Self::XMM1_2 => write!(f, "XMM1_2"),
            Self::XMM1_3 => write!(f, "XMM1_3"),
            Self::XMM2_0 => write!(f, "XMM2_0"),
            Self::XMM2_1 => write!(f, "XMM2_1"),
            Self::XMM2_2 => write!(f, "XMM2_2"),
            Self::XMM2_3 => write!(f, "XMM2_3"),
            Self::XMM3_0 => write!(f, "XMM3_0"),
            Self::XMM3_1 => write!(f, "XMM3_1"),
            Self::XMM3_2 => write!(f, "XMM3_2"),
            Self::XMM3_3 => write!(f, "XMM3_3"),
            Self::XMM4_0 => write!(f, "XMM4_0"),
            Self::XMM4_1 => write!(f, "XMM4_1"),
            Self::XMM4_2 => write!(f, "XMM4_2"),
            Self::XMM4_3 => write!(f, "XMM4_3"),
            Self::XMM5_0 => write!(f, "XMM5_0"),
            Self::XMM5_1 => write!(f, "XMM5_1"),
            Self::XMM5_2 => write!(f, "XMM5_2"),
            Self::XMM5_3 => write!(f, "XMM5_3"),
            Self::XMM6_0 => write!(f, "XMM6_0"),
            Self::XMM6_1 => write!(f, "XMM6_1"),
            Self::XMM6_2 => write!(f, "XMM6_2"),
            Self::XMM6_3 => write!(f, "XMM6_3"),
            Self::XMM7_0 => write!(f, "XMM7_0"),
            Self::XMM7_1 => write!(f, "XMM7_1"),
            Self::XMM7_2 => write!(f, "XMM7_2"),
            Self::XMM7_3 => write!(f, "XMM7_3"),
            Self::XMM0L => write!(f, "XMM0L"),
            Self::XMM1L => write!(f, "XMM1L"),
            Self::XMM2L => write!(f, "XMM2L"),
            Self::XMM3L => write!(f, "XMM3L"),
            Self::XMM4L => write!(f, "XMM4L"),
            Self::XMM5L => write!(f, "XMM5L"),
            Self::XMM6L => write!(f, "XMM6L"),
            Self::XMM7L => write!(f, "XMM7L"),
            Self::XMM0H => write!(f, "XMM0H"),
            Self::XMM1H => write!(f, "XMM1H"),
            Self::XMM2H => write!(f, "XMM2H"),
            Self::XMM3H => write!(f, "XMM3H"),
            Self::XMM4H => write!(f, "XMM4H"),
            Self::XMM5H => write!(f, "XMM5H"),
            Self::XMM6H => write!(f, "XMM6H"),
            Self::XMM7H => write!(f, "XMM7H"),
            Self::MXCSR => write!(f, "MXCSR"),
            Self::EMM0L => write!(f, "EMM0L"),
            Self::EMM1L => write!(f, "EMM1L"),
            Self::EMM2L => write!(f, "EMM2L"),
            Self::EMM3L => write!(f, "EMM3L"),
            Self::EMM4L => write!(f, "EMM4L"),
            Self::EMM5L => write!(f, "EMM5L"),
            Self::EMM6L => write!(f, "EMM6L"),
            Self::EMM7L => write!(f, "EMM7L"),
            Self::EMM0H => write!(f, "EMM0H"),
            Self::EMM1H => write!(f, "EMM1H"),
            Self::EMM2H => write!(f, "EMM2H"),
            Self::EMM3H => write!(f, "EMM3H"),
            Self::EMM4H => write!(f, "EMM4H"),
            Self::EMM5H => write!(f, "EMM5H"),
            Self::EMM6H => write!(f, "EMM6H"),
            Self::EMM7H => write!(f, "EMM7H"),
            Self::MM00 => write!(f, "MM00"),
            Self::MM01 => write!(f, "MM01"),
            Self::MM10 => write!(f, "MM10"),
            Self::MM11 => write!(f, "MM11"),
            Self::MM20 => write!(f, "MM20"),
            Self::MM21 => write!(f, "MM21"),
            Self::MM30 => write!(f, "MM30"),
            Self::MM31 => write!(f, "MM31"),
            Self::MM40 => write!(f, "MM40"),
            Self::MM41 => write!(f, "MM41"),
            Self::MM50 => write!(f, "MM50"),
            Self::MM51 => write!(f, "MM51"),
            Self::MM60 => write!(f, "MM60"),
            Self::MM61 => write!(f, "MM61"),
            Self::MM70 => write!(f, "MM70"),
            Self::MM71 => write!(f, "MM71"),
            Self::XMM8 => write!(f, "XMM8"),
            Self::XMM9 => write!(f, "XMM9"),
            Self::XMM10 => write!(f, "XMM10"),
            Self::XMM11 => write!(f, "XMM11"),
            Self::XMM12 => write!(f, "XMM12"),
            Self::XMM13 => write!(f, "XMM13"),
            Self::XMM14 => write!(f, "XMM14"),
            Self::XMM15 => write!(f, "XMM15"),
            Self::XMM8_0 => write!(f, "XMM8_0"),
            Self::XMM8_1 => write!(f, "XMM8_1"),
            Self::XMM8_2 => write!(f, "XMM8_2"),
            Self::XMM8_3 => write!(f, "XMM8_3"),
            Self::XMM9_0 => write!(f, "XMM9_0"),
            Self::XMM9_1 => write!(f, "XMM9_1"),
            Self::XMM9_2 => write!(f, "XMM9_2"),
            Self::XMM9_3 => write!(f, "XMM9_3"),
            Self::XMM10_0 => write!(f, "XMM10_0"),
            Self::XMM10_1 => write!(f, "XMM10_1"),
            Self::XMM10_2 => write!(f, "XMM10_2"),
            Self::XMM10_3 => write!(f, "XMM10_3"),
            Self::XMM11_0 => write!(f, "XMM11_0"),
            Self::XMM11_1 => write!(f, "XMM11_1"),
            Self::XMM11_2 => write!(f, "XMM11_2"),
            Self::XMM11_3 => write!(f, "XMM11_3"),
            Self::XMM12_0 => write!(f, "XMM12_0"),
            Self::XMM12_1 => write!(f, "XMM12_1"),
            Self::XMM12_2 => write!(f, "XMM12_2"),
            Self::XMM12_3 => write!(f, "XMM12_3"),
            Self::XMM13_0 => write!(f, "XMM13_0"),
            Self::XMM13_1 => write!(f, "XMM13_1"),
            Self::XMM13_2 => write!(f, "XMM13_2"),
            Self::XMM13_3 => write!(f, "XMM13_3"),
            Self::XMM14_0 => write!(f, "XMM14_0"),
            Self::XMM14_1 => write!(f, "XMM14_1"),
            Self::XMM14_2 => write!(f, "XMM14_2"),
            Self::XMM14_3 => write!(f, "XMM14_3"),
            Self::XMM15_0 => write!(f, "XMM15_0"),
            Self::XMM15_1 => write!(f, "XMM15_1"),
            Self::XMM15_2 => write!(f, "XMM15_2"),
            Self::XMM15_3 => write!(f, "XMM15_3"),
            Self::XMM8L => write!(f, "XMM8L"),
            Self::XMM9L => write!(f, "XMM9L"),
            Self::XMM10L => write!(f, "XMM10L"),
            Self::XMM11L => write!(f, "XMM11L"),
            Self::XMM12L => write!(f, "XMM12L"),
            Self::XMM13L => write!(f, "XMM13L"),
            Self::XMM14L => write!(f, "XMM14L"),
            Self::XMM15L => write!(f, "XMM15L"),
            Self::XMM8H => write!(f, "XMM8H"),
            Self::XMM9H => write!(f, "XMM9H"),
            Self::XMM10H => write!(f, "XMM10H"),
            Self::XMM11H => write!(f, "XMM11H"),
            Self::XMM12H => write!(f, "XMM12H"),
            Self::XMM13H => write!(f, "XMM13H"),
            Self::XMM14H => write!(f, "XMM14H"),
            Self::XMM15H => write!(f, "XMM15H"),
            Self::EMM8L => write!(f, "EMM8L"),
            Self::EMM9L => write!(f, "EMM9L"),
            Self::EMM10L => write!(f, "EMM10L"),
            Self::EMM11L => write!(f, "EMM11L"),
            Self::EMM12L => write!(f, "EMM12L"),
            Self::EMM13L => write!(f, "EMM13L"),
            Self::EMM14L => write!(f, "EMM14L"),
            Self::EMM15L => write!(f, "EMM15L"),
            Self::EMM8H => write!(f, "EMM8H"),
            Self::EMM9H => write!(f, "EMM9H"),
            Self::EMM10H => write!(f, "EMM10H"),
            Self::EMM11H => write!(f, "EMM11H"),
            Self::EMM12H => write!(f, "EMM12H"),
            Self::EMM13H => write!(f, "EMM13H"),
            Self::EMM14H => write!(f, "EMM14H"),
            Self::EMM15H => write!(f, "EMM15H"),
            Self::SIL => write!(f, "SIL"),
            Self::DIL => write!(f, "DIL"),
            Self::BPL => write!(f, "BPL"),
            Self::SPL => write!(f, "SPL"),
            Self::RAX => write!(f, "RAX"),
            Self::RBX => write!(f, "RBX"),
            Self::RCX => write!(f, "RCX"),
            Self::RDX => write!(f, "RDX"),
            Self::RSI => write!(f, "RSI"),
            Self::RDI => write!(f, "RDI"),
            Self::RBP => write!(f, "RBP"),
            Self::RSP => write!(f, "RSP"),
            Self::R8 => write!(f, "R8"),
            Self::R9 => write!(f, "R9"),
            Self::R10 => write!(f, "R10"),
            Self::R11 => write!(f, "R11"),
            Self::R12 => write!(f, "R12"),
            Self::R13 => write!(f, "R13"),
            Self::R14 => write!(f, "R14"),
            Self::R15 => write!(f, "R15"),
            Self::R8B => write!(f, "R8B"),
            Self::R9B => write!(f, "R9B"),
            Self::R10B => write!(f, "R10B"),
            Self::R11B => write!(f, "R11B"),
            Self::R12B => write!(f, "R12B"),
            Self::R13B => write!(f, "R13B"),
            Self::R14B => write!(f, "R14B"),
            Self::R15B => write!(f, "R15B"),
            Self::R8W => write!(f, "R8W"),
            Self::R9W => write!(f, "R9W"),
            Self::R10W => write!(f, "R10W"),
            Self::R11W => write!(f, "R11W"),
            Self::R12W => write!(f, "R12W"),
            Self::R13W => write!(f, "R13W"),
            Self::R14W => write!(f, "R14W"),
            Self::R15W => write!(f, "R15W"),
            Self::R8D => write!(f, "R8D"),
            Self::R9D => write!(f, "R9D"),
            Self::R10D => write!(f, "R10D"),
            Self::R11D => write!(f, "R11D"),
            Self::R12D => write!(f, "R12D"),
            Self::R13D => write!(f, "R13D"),
            Self::R14D => write!(f, "R14D"),
            Self::R15D => write!(f, "R15D"),
            Self::YMM0 => write!(f, "YMM0"),
            Self::YMM1 => write!(f, "YMM1"),
            Self::YMM2 => write!(f, "YMM2"),
            Self::YMM3 => write!(f, "YMM3"),
            Self::YMM4 => write!(f, "YMM4"),
            Self::YMM5 => write!(f, "YMM5"),
            Self::YMM6 => write!(f, "YMM6"),
            Self::YMM7 => write!(f, "YMM7"),
            Self::YMM8 => write!(f, "YMM8"),
            Self::YMM9 => write!(f, "YMM9"),
            Self::YMM10 => write!(f, "YMM10"),
            Self::YMM11 => write!(f, "YMM11"),
            Self::YMM12 => write!(f, "YMM12"),
            Self::YMM13 => write!(f, "YMM13"),
            Self::YMM14 => write!(f, "YMM14"),
            Self::YMM15 => write!(f, "YMM15"),
            Self::YMM0H => write!(f, "YMM0H"),
            Self::YMM1H => write!(f, "YMM1H"),
            Self::YMM2H => write!(f, "YMM2H"),
            Self::YMM3H => write!(f, "YMM3H"),
            Self::YMM4H => write!(f, "YMM4H"),
            Self::YMM5H => write!(f, "YMM5H"),
            Self::YMM6H => write!(f, "YMM6H"),
            Self::YMM7H => write!(f, "YMM7H"),
            Self::YMM8H => write!(f, "YMM8H"),
            Self::YMM9H => write!(f, "YMM9H"),
            Self::YMM10H => write!(f, "YMM10H"),
            Self::YMM11H => write!(f, "YMM11H"),
            Self::YMM12H => write!(f, "YMM12H"),
            Self::YMM13H => write!(f, "YMM13H"),
            Self::YMM14H => write!(f, "YMM14H"),
            Self::YMM15H => write!(f, "YMM15H"),
            Self::XMM0IL => write!(f, "XMM0IL"),
            Self::XMM1IL => write!(f, "XMM1IL"),
            Self::XMM2IL => write!(f, "XMM2IL"),
            Self::XMM3IL => write!(f, "XMM3IL"),
            Self::XMM4IL => write!(f, "XMM4IL"),
            Self::XMM5IL => write!(f, "XMM5IL"),
            Self::XMM6IL => write!(f, "XMM6IL"),
            Self::XMM7IL => write!(f, "XMM7IL"),
            Self::XMM8IL => write!(f, "XMM8IL"),
            Self::XMM9IL => write!(f, "XMM9IL"),
            Self::XMM10IL => write!(f, "XMM10IL"),
            Self::XMM11IL => write!(f, "XMM11IL"),
            Self::XMM12IL => write!(f, "XMM12IL"),
            Self::XMM13IL => write!(f, "XMM13IL"),
            Self::XMM14IL => write!(f, "XMM14IL"),
            Self::XMM15IL => write!(f, "XMM15IL"),
            Self::XMM0IH => write!(f, "XMM0IH"),
            Self::XMM1IH => write!(f, "XMM1IH"),
            Self::XMM2IH => write!(f, "XMM2IH"),
            Self::XMM3IH => write!(f, "XMM3IH"),
            Self::XMM4IH => write!(f, "XMM4IH"),
            Self::XMM5IH => write!(f, "XMM5IH"),
            Self::XMM6IH => write!(f, "XMM6IH"),
            Self::XMM7IH => write!(f, "XMM7IH"),
            Self::XMM8IH => write!(f, "XMM8IH"),
            Self::XMM9IH => write!(f, "XMM9IH"),
            Self::XMM10IH => write!(f, "XMM10IH"),
            Self::XMM11IH => write!(f, "XMM11IH"),
            Self::XMM12IH => write!(f, "XMM12IH"),
            Self::XMM13IH => write!(f, "XMM13IH"),
            Self::XMM14IH => write!(f, "XMM14IH"),
            Self::XMM15IH => write!(f, "XMM15IH"),
            Self::YMM0I0 => write!(f, "YMM0I0"),
            Self::YMM0I1 => write!(f, "YMM0I1"),
            Self::YMM0I2 => write!(f, "YMM0I2"),
            Self::YMM0I3 => write!(f, "YMM0I3"),
            Self::YMM1I0 => write!(f, "YMM1I0"),
            Self::YMM1I1 => write!(f, "YMM1I1"),
            Self::YMM1I2 => write!(f, "YMM1I2"),
            Self::YMM1I3 => write!(f, "YMM1I3"),
            Self::YMM2I0 => write!(f, "YMM2I0"),
            Self::YMM2I1 => write!(f, "YMM2I1"),
            Self::YMM2I2 => write!(f, "YMM2I2"),
            Self::YMM2I3 => write!(f, "YMM2I3"),
            Self::YMM3I0 => write!(f, "YMM3I0"),
            Self::YMM3I1 => write!(f, "YMM3I1"),
            Self::YMM3I2 => write!(f, "YMM3I2"),
            Self::YMM3I3 => write!(f, "YMM3I3"),
            Self::YMM4I0 => write!(f, "YMM4I0"),
            Self::YMM4I1 => write!(f, "YMM4I1"),
            Self::YMM4I2 => write!(f, "YMM4I2"),
            Self::YMM4I3 => write!(f, "YMM4I3"),
            Self::YMM5I0 => write!(f, "YMM5I0"),
            Self::YMM5I1 => write!(f, "YMM5I1"),
            Self::YMM5I2 => write!(f, "YMM5I2"),
            Self::YMM5I3 => write!(f, "YMM5I3"),
            Self::YMM6I0 => write!(f, "YMM6I0"),
            Self::YMM6I1 => write!(f, "YMM6I1"),
            Self::YMM6I2 => write!(f, "YMM6I2"),
            Self::YMM6I3 => write!(f, "YMM6I3"),
            Self::YMM7I0 => write!(f, "YMM7I0"),
            Self::YMM7I1 => write!(f, "YMM7I1"),
            Self::YMM7I2 => write!(f, "YMM7I2"),
            Self::YMM7I3 => write!(f, "YMM7I3"),
            Self::YMM8I0 => write!(f, "YMM8I0"),
            Self::YMM8I1 => write!(f, "YMM8I1"),
            Self::YMM8I2 => write!(f, "YMM8I2"),
            Self::YMM8I3 => write!(f, "YMM8I3"),
            Self::YMM9I0 => write!(f, "YMM9I0"),
            Self::YMM9I1 => write!(f, "YMM9I1"),
            Self::YMM9I2 => write!(f, "YMM9I2"),
            Self::YMM9I3 => write!(f, "YMM9I3"),
            Self::YMM10I0 => write!(f, "YMM10I0"),
            Self::YMM10I1 => write!(f, "YMM10I1"),
            Self::YMM10I2 => write!(f, "YMM10I2"),
            Self::YMM10I3 => write!(f, "YMM10I3"),
            Self::YMM11I0 => write!(f, "YMM11I0"),
            Self::YMM11I1 => write!(f, "YMM11I1"),
            Self::YMM11I2 => write!(f, "YMM11I2"),
            Self::YMM11I3 => write!(f, "YMM11I3"),
            Self::YMM12I0 => write!(f, "YMM12I0"),
            Self::YMM12I1 => write!(f, "YMM12I1"),
            Self::YMM12I2 => write!(f, "YMM12I2"),
            Self::YMM12I3 => write!(f, "YMM12I3"),
            Self::YMM13I0 => write!(f, "YMM13I0"),
            Self::YMM13I1 => write!(f, "YMM13I1"),
            Self::YMM13I2 => write!(f, "YMM13I2"),
            Self::YMM13I3 => write!(f, "YMM13I3"),
            Self::YMM14I0 => write!(f, "YMM14I0"),
            Self::YMM14I1 => write!(f, "YMM14I1"),
            Self::YMM14I2 => write!(f, "YMM14I2"),
            Self::YMM14I3 => write!(f, "YMM14I3"),
            Self::YMM15I0 => write!(f, "YMM15I0"),
            Self::YMM15I1 => write!(f, "YMM15I1"),
            Self::YMM15I2 => write!(f, "YMM15I2"),
            Self::YMM15I3 => write!(f, "YMM15I3"),
            Self::YMM0F0 => write!(f, "YMM0F0"),
            Self::YMM0F1 => write!(f, "YMM0F1"),
            Self::YMM0F2 => write!(f, "YMM0F2"),
            Self::YMM0F3 => write!(f, "YMM0F3"),
            Self::YMM0F4 => write!(f, "YMM0F4"),
            Self::YMM0F5 => write!(f, "YMM0F5"),
            Self::YMM0F6 => write!(f, "YMM0F6"),
            Self::YMM0F7 => write!(f, "YMM0F7"),
            Self::YMM1F0 => write!(f, "YMM1F0"),
            Self::YMM1F1 => write!(f, "YMM1F1"),
            Self::YMM1F2 => write!(f, "YMM1F2"),
            Self::YMM1F3 => write!(f, "YMM1F3"),
            Self::YMM1F4 => write!(f, "YMM1F4"),
            Self::YMM1F5 => write!(f, "YMM1F5"),
            Self::YMM1F6 => write!(f, "YMM1F6"),
            Self::YMM1F7 => write!(f, "YMM1F7"),
            Self::YMM2F0 => write!(f, "YMM2F0"),
            Self::YMM2F1 => write!(f, "YMM2F1"),
            Self::YMM2F2 => write!(f, "YMM2F2"),
            Self::YMM2F3 => write!(f, "YMM2F3"),
            Self::YMM2F4 => write!(f, "YMM2F4"),
            Self::YMM2F5 => write!(f, "YMM2F5"),
            Self::YMM2F6 => write!(f, "YMM2F6"),
            Self::YMM2F7 => write!(f, "YMM2F7"),
            Self::YMM3F0 => write!(f, "YMM3F0"),
            Self::YMM3F1 => write!(f, "YMM3F1"),
            Self::YMM3F2 => write!(f, "YMM3F2"),
            Self::YMM3F3 => write!(f, "YMM3F3"),
            Self::YMM3F4 => write!(f, "YMM3F4"),
            Self::YMM3F5 => write!(f, "YMM3F5"),
            Self::YMM3F6 => write!(f, "YMM3F6"),
            Self::YMM3F7 => write!(f, "YMM3F7"),
            Self::YMM4F0 => write!(f, "YMM4F0"),
            Self::YMM4F1 => write!(f, "YMM4F1"),
            Self::YMM4F2 => write!(f, "YMM4F2"),
            Self::YMM4F3 => write!(f, "YMM4F3"),
            Self::YMM4F4 => write!(f, "YMM4F4"),
            Self::YMM4F5 => write!(f, "YMM4F5"),
            Self::YMM4F6 => write!(f, "YMM4F6"),
            Self::YMM4F7 => write!(f, "YMM4F7"),
            Self::YMM5F0 => write!(f, "YMM5F0"),
            Self::YMM5F1 => write!(f, "YMM5F1"),
            Self::YMM5F2 => write!(f, "YMM5F2"),
            Self::YMM5F3 => write!(f, "YMM5F3"),
            Self::YMM5F4 => write!(f, "YMM5F4"),
            Self::YMM5F5 => write!(f, "YMM5F5"),
            Self::YMM5F6 => write!(f, "YMM5F6"),
            Self::YMM5F7 => write!(f, "YMM5F7"),
            Self::YMM6F0 => write!(f, "YMM6F0"),
            Self::YMM6F1 => write!(f, "YMM6F1"),
            Self::YMM6F2 => write!(f, "YMM6F2"),
            Self::YMM6F3 => write!(f, "YMM6F3"),
            Self::YMM6F4 => write!(f, "YMM6F4"),
            Self::YMM6F5 => write!(f, "YMM6F5"),
            Self::YMM6F6 => write!(f, "YMM6F6"),
            Self::YMM6F7 => write!(f, "YMM6F7"),
            Self::YMM7F0 => write!(f, "YMM7F0"),
            Self::YMM7F1 => write!(f, "YMM7F1"),
            Self::YMM7F2 => write!(f, "YMM7F2"),
            Self::YMM7F3 => write!(f, "YMM7F3"),
            Self::YMM7F4 => write!(f, "YMM7F4"),
            Self::YMM7F5 => write!(f, "YMM7F5"),
            Self::YMM7F6 => write!(f, "YMM7F6"),
            Self::YMM7F7 => write!(f, "YMM7F7"),
            Self::YMM8F0 => write!(f, "YMM8F0"),
            Self::YMM8F1 => write!(f, "YMM8F1"),
            Self::YMM8F2 => write!(f, "YMM8F2"),
            Self::YMM8F3 => write!(f, "YMM8F3"),
            Self::YMM8F4 => write!(f, "YMM8F4"),
            Self::YMM8F5 => write!(f, "YMM8F5"),
            Self::YMM8F6 => write!(f, "YMM8F6"),
            Self::YMM8F7 => write!(f, "YMM8F7"),
            Self::YMM9F0 => write!(f, "YMM9F0"),
            Self::YMM9F1 => write!(f, "YMM9F1"),
            Self::YMM9F2 => write!(f, "YMM9F2"),
            Self::YMM9F3 => write!(f, "YMM9F3"),
            Self::YMM9F4 => write!(f, "YMM9F4"),
            Self::YMM9F5 => write!(f, "YMM9F5"),
            Self::YMM9F6 => write!(f, "YMM9F6"),
            Self::YMM9F7 => write!(f, "YMM9F7"),
            Self::YMM10F0 => write!(f, "YMM10F0"),
            Self::YMM10F1 => write!(f, "YMM10F1"),
            Self::YMM10F2 => write!(f, "YMM10F2"),
            Self::YMM10F3 => write!(f, "YMM10F3"),
            Self::YMM10F4 => write!(f, "YMM10F4"),
            Self::YMM10F5 => write!(f, "YMM10F5"),
            Self::YMM10F6 => write!(f, "YMM10F6"),
            Self::YMM10F7 => write!(f, "YMM10F7"),
            Self::YMM11F0 => write!(f, "YMM11F0"),
            Self::YMM11F1 => write!(f, "YMM11F1"),
            Self::YMM11F2 => write!(f, "YMM11F2"),
            Self::YMM11F3 => write!(f, "YMM11F3"),
            Self::YMM11F4 => write!(f, "YMM11F4"),
            Self::YMM11F5 => write!(f, "YMM11F5"),
            Self::YMM11F6 => write!(f, "YMM11F6"),
            Self::YMM11F7 => write!(f, "YMM11F7"),
            Self::YMM12F0 => write!(f, "YMM12F0"),
            Self::YMM12F1 => write!(f, "YMM12F1"),
            Self::YMM12F2 => write!(f, "YMM12F2"),
            Self::YMM12F3 => write!(f, "YMM12F3"),
            Self::YMM12F4 => write!(f, "YMM12F4"),
            Self::YMM12F5 => write!(f, "YMM12F5"),
            Self::YMM12F6 => write!(f, "YMM12F6"),
            Self::YMM12F7 => write!(f, "YMM12F7"),
            Self::YMM13F0 => write!(f, "YMM13F0"),
            Self::YMM13F1 => write!(f, "YMM13F1"),
            Self::YMM13F2 => write!(f, "YMM13F2"),
            Self::YMM13F3 => write!(f, "YMM13F3"),
            Self::YMM13F4 => write!(f, "YMM13F4"),
            Self::YMM13F5 => write!(f, "YMM13F5"),
            Self::YMM13F6 => write!(f, "YMM13F6"),
            Self::YMM13F7 => write!(f, "YMM13F7"),
            Self::YMM14F0 => write!(f, "YMM14F0"),
            Self::YMM14F1 => write!(f, "YMM14F1"),
            Self::YMM14F2 => write!(f, "YMM14F2"),
            Self::YMM14F3 => write!(f, "YMM14F3"),
            Self::YMM14F4 => write!(f, "YMM14F4"),
            Self::YMM14F5 => write!(f, "YMM14F5"),
            Self::YMM14F6 => write!(f, "YMM14F6"),
            Self::YMM14F7 => write!(f, "YMM14F7"),
            Self::YMM15F0 => write!(f, "YMM15F0"),
            Self::YMM15F1 => write!(f, "YMM15F1"),
            Self::YMM15F2 => write!(f, "YMM15F2"),
            Self::YMM15F3 => write!(f, "YMM15F3"),
            Self::YMM15F4 => write!(f, "YMM15F4"),
            Self::YMM15F5 => write!(f, "YMM15F5"),
            Self::YMM15F6 => write!(f, "YMM15F6"),
            Self::YMM15F7 => write!(f, "YMM15F7"),
            Self::YMM0D0 => write!(f, "YMM0D0"),
            Self::YMM0D1 => write!(f, "YMM0D1"),
            Self::YMM0D2 => write!(f, "YMM0D2"),
            Self::YMM0D3 => write!(f, "YMM0D3"),
            Self::YMM1D0 => write!(f, "YMM1D0"),
            Self::YMM1D1 => write!(f, "YMM1D1"),
            Self::YMM1D2 => write!(f, "YMM1D2"),
            Self::YMM1D3 => write!(f, "YMM1D3"),
            Self::YMM2D0 => write!(f, "YMM2D0"),
            Self::YMM2D1 => write!(f, "YMM2D1"),
            Self::YMM2D2 => write!(f, "YMM2D2"),
            Self::YMM2D3 => write!(f, "YMM2D3"),
            Self::YMM3D0 => write!(f, "YMM3D0"),
            Self::YMM3D1 => write!(f, "YMM3D1"),
            Self::YMM3D2 => write!(f, "YMM3D2"),
            Self::YMM3D3 => write!(f, "YMM3D3"),
            Self::YMM4D0 => write!(f, "YMM4D0"),
            Self::YMM4D1 => write!(f, "YMM4D1"),
            Self::YMM4D2 => write!(f, "YMM4D2"),
            Self::YMM4D3 => write!(f, "YMM4D3"),
            Self::YMM5D0 => write!(f, "YMM5D0"),
            Self::YMM5D1 => write!(f, "YMM5D1"),
            Self::YMM5D2 => write!(f, "YMM5D2"),
            Self::YMM5D3 => write!(f, "YMM5D3"),
            Self::YMM6D0 => write!(f, "YMM6D0"),
            Self::YMM6D1 => write!(f, "YMM6D1"),
            Self::YMM6D2 => write!(f, "YMM6D2"),
            Self::YMM6D3 => write!(f, "YMM6D3"),
            Self::YMM7D0 => write!(f, "YMM7D0"),
            Self::YMM7D1 => write!(f, "YMM7D1"),
            Self::YMM7D2 => write!(f, "YMM7D2"),
            Self::YMM7D3 => write!(f, "YMM7D3"),
            Self::YMM8D0 => write!(f, "YMM8D0"),
            Self::YMM8D1 => write!(f, "YMM8D1"),
            Self::YMM8D2 => write!(f, "YMM8D2"),
            Self::YMM8D3 => write!(f, "YMM8D3"),
            Self::YMM9D0 => write!(f, "YMM9D0"),
            Self::YMM9D1 => write!(f, "YMM9D1"),
            Self::YMM9D2 => write!(f, "YMM9D2"),
            Self::YMM9D3 => write!(f, "YMM9D3"),
            Self::YMM10D0 => write!(f, "YMM10D0"),
            Self::YMM10D1 => write!(f, "YMM10D1"),
            Self::YMM10D2 => write!(f, "YMM10D2"),
            Self::YMM10D3 => write!(f, "YMM10D3"),
            Self::YMM11D0 => write!(f, "YMM11D0"),
            Self::YMM11D1 => write!(f, "YMM11D1"),
            Self::YMM11D2 => write!(f, "YMM11D2"),
            Self::YMM11D3 => write!(f, "YMM11D3"),
            Self::YMM12D0 => write!(f, "YMM12D0"),
            Self::YMM12D1 => write!(f, "YMM12D1"),
            Self::YMM12D2 => write!(f, "YMM12D2"),
            Self::YMM12D3 => write!(f, "YMM12D3"),
            Self::YMM13D0 => write!(f, "YMM13D0"),
            Self::YMM13D1 => write!(f, "YMM13D1"),
            Self::YMM13D2 => write!(f, "YMM13D2"),
            Self::YMM13D3 => write!(f, "YMM13D3"),
            Self::YMM14D0 => write!(f, "YMM14D0"),
            Self::YMM14D1 => write!(f, "YMM14D1"),
            Self::YMM14D2 => write!(f, "YMM14D2"),
            Self::YMM14D3 => write!(f, "YMM14D3"),
            Self::YMM15D0 => write!(f, "YMM15D0"),
            Self::YMM15D1 => write!(f, "YMM15D1"),
            Self::YMM15D2 => write!(f, "YMM15D2"),
            Self::YMM15D3 => write!(f, "YMM15D3"),
        }
    }
}

impl TryFrom<u16> for AMD64Register {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            1 => Ok(Self::AL),
            2 => Ok(Self::CL),
            3 => Ok(Self::DL),
            4 => Ok(Self::BL),
            5 => Ok(Self::AH),
            6 => Ok(Self::CH),
            7 => Ok(Self::DH),
            8 => Ok(Self::BH),
            9 => Ok(Self::AX),
            10 => Ok(Self::CX),
            11 => Ok(Self::DX),
            12 => Ok(Self::BX),
            13 => Ok(Self::SP),
            14 => Ok(Self::BP),
            15 => Ok(Self::SI),
            16 => Ok(Self::DI),
            17 => Ok(Self::EAX),
            18 => Ok(Self::ECX),
            19 => Ok(Self::EDX),
            20 => Ok(Self::EBX),
            21 => Ok(Self::ESP),
            22 => Ok(Self::EBP),
            23 => Ok(Self::ESI),
            24 => Ok(Self::EDI),
            25 => Ok(Self::ES),
            26 => Ok(Self::CS),
            27 => Ok(Self::SS),
            28 => Ok(Self::DS),
            29 => Ok(Self::FS),
            30 => Ok(Self::GS),
            32 => Ok(Self::FLAGS),
            33 => Ok(Self::RIP),
            34 => Ok(Self::EFLAGS),
            80 => Ok(Self::CR0),
            81 => Ok(Self::CR1),
            82 => Ok(Self::CR2),
            83 => Ok(Self::CR3),
            84 => Ok(Self::CR4),
            88 => Ok(Self::CR8),
            90 => Ok(Self::DR0),
            91 => Ok(Self::DR1),
            92 => Ok(Self::DR2),
            93 => Ok(Self::DR3),
            94 => Ok(Self::DR4),
            95 => Ok(Self::DR5),
            96 => Ok(Self::DR6),
            97 => Ok(Self::DR7),
            98 => Ok(Self::DR8),
            99 => Ok(Self::DR9),
            100 => Ok(Self::DR10),
            101 => Ok(Self::DR11),
            102 => Ok(Self::DR12),
            103 => Ok(Self::DR13),
            104 => Ok(Self::DR14),
            105 => Ok(Self::DR15),
            110 => Ok(Self::GDTR),
            111 => Ok(Self::GDTL),
            112 => Ok(Self::IDTR),
            113 => Ok(Self::IDTL),
            114 => Ok(Self::LDTR),
            115 => Ok(Self::TR),
            128 => Ok(Self::ST0),
            129 => Ok(Self::ST1),
            130 => Ok(Self::ST2),
            131 => Ok(Self::ST3),
            132 => Ok(Self::ST4),
            133 => Ok(Self::ST5),
            134 => Ok(Self::ST6),
            135 => Ok(Self::ST7),
            136 => Ok(Self::CTRL),
            137 => Ok(Self::STAT),
            138 => Ok(Self::TAG),
            139 => Ok(Self::FPIP),
            140 => Ok(Self::FPCS),
            141 => Ok(Self::FPDO),
            142 => Ok(Self::FPDS),
            143 => Ok(Self::ISEM),
            144 => Ok(Self::FPEIP),
            145 => Ok(Self::FPEDO),
            146 => Ok(Self::MM0),
            147 => Ok(Self::MM1),
            148 => Ok(Self::MM2),
            149 => Ok(Self::MM3),
            150 => Ok(Self::MM4),
            151 => Ok(Self::MM5),
            152 => Ok(Self::MM6),
            153 => Ok(Self::MM7),
            154 => Ok(Self::XMM0),
            155 => Ok(Self::XMM1),
            156 => Ok(Self::XMM2),
            157 => Ok(Self::XMM3),
            158 => Ok(Self::XMM4),
            159 => Ok(Self::XMM5),
            160 => Ok(Self::XMM6),
            161 => Ok(Self::XMM7),
            162 => Ok(Self::XMM0_0),
            163 => Ok(Self::XMM0_1),
            164 => Ok(Self::XMM0_2),
            165 => Ok(Self::XMM0_3),
            166 => Ok(Self::XMM1_0),
            167 => Ok(Self::XMM1_1),
            168 => Ok(Self::XMM1_2),
            169 => Ok(Self::XMM1_3),
            170 => Ok(Self::XMM2_0),
            171 => Ok(Self::XMM2_1),
            172 => Ok(Self::XMM2_2),
            173 => Ok(Self::XMM2_3),
            174 => Ok(Self::XMM3_0),
            175 => Ok(Self::XMM3_1),
            176 => Ok(Self::XMM3_2),
            177 => Ok(Self::XMM3_3),
            178 => Ok(Self::XMM4_0),
            179 => Ok(Self::XMM4_1),
            180 => Ok(Self::XMM4_2),
            181 => Ok(Self::XMM4_3),
            182 => Ok(Self::XMM5_0),
            183 => Ok(Self::XMM5_1),
            184 => Ok(Self::XMM5_2),
            185 => Ok(Self::XMM5_3),
            186 => Ok(Self::XMM6_0),
            187 => Ok(Self::XMM6_1),
            188 => Ok(Self::XMM6_2),
            189 => Ok(Self::XMM6_3),
            190 => Ok(Self::XMM7_0),
            191 => Ok(Self::XMM7_1),
            192 => Ok(Self::XMM7_2),
            193 => Ok(Self::XMM7_3),
            194 => Ok(Self::XMM0L),
            195 => Ok(Self::XMM1L),
            196 => Ok(Self::XMM2L),
            197 => Ok(Self::XMM3L),
            198 => Ok(Self::XMM4L),
            199 => Ok(Self::XMM5L),
            200 => Ok(Self::XMM6L),
            201 => Ok(Self::XMM7L),
            202 => Ok(Self::XMM0H),
            203 => Ok(Self::XMM1H),
            204 => Ok(Self::XMM2H),
            205 => Ok(Self::XMM3H),
            206 => Ok(Self::XMM4H),
            207 => Ok(Self::XMM5H),
            208 => Ok(Self::XMM6H),
            209 => Ok(Self::XMM7H),
            211 => Ok(Self::MXCSR),
            220 => Ok(Self::EMM0L),
            221 => Ok(Self::EMM1L),
            222 => Ok(Self::EMM2L),
            223 => Ok(Self::EMM3L),
            224 => Ok(Self::EMM4L),
            225 => Ok(Self::EMM5L),
            226 => Ok(Self::EMM6L),
            227 => Ok(Self::EMM7L),
            228 => Ok(Self::EMM0H),
            229 => Ok(Self::EMM1H),
            230 => Ok(Self::EMM2H),
            231 => Ok(Self::EMM3H),
            232 => Ok(Self::EMM4H),
            233 => Ok(Self::EMM5H),
            234 => Ok(Self::EMM6H),
            235 => Ok(Self::EMM7H),
            236 => Ok(Self::MM00),
            237 => Ok(Self::MM01),
            238 => Ok(Self::MM10),
            239 => Ok(Self::MM11),
            240 => Ok(Self::MM20),
            241 => Ok(Self::MM21),
            242 => Ok(Self::MM30),
            243 => Ok(Self::MM31),
            244 => Ok(Self::MM40),
            245 => Ok(Self::MM41),
            246 => Ok(Self::MM50),
            247 => Ok(Self::MM51),
            248 => Ok(Self::MM60),
            249 => Ok(Self::MM61),
            250 => Ok(Self::MM70),
            251 => Ok(Self::MM71),
            252 => Ok(Self::XMM8),
            253 => Ok(Self::XMM9),
            254 => Ok(Self::XMM10),
            255 => Ok(Self::XMM11),
            256 => Ok(Self::XMM12),
            257 => Ok(Self::XMM13),
            258 => Ok(Self::XMM14),
            259 => Ok(Self::XMM15),
            260 => Ok(Self::XMM8_0),
            261 => Ok(Self::XMM8_1),
            262 => Ok(Self::XMM8_2),
            263 => Ok(Self::XMM8_3),
            264 => Ok(Self::XMM9_0),
            265 => Ok(Self::XMM9_1),
            266 => Ok(Self::XMM9_2),
            267 => Ok(Self::XMM9_3),
            268 => Ok(Self::XMM10_0),
            269 => Ok(Self::XMM10_1),
            270 => Ok(Self::XMM10_2),
            271 => Ok(Self::XMM10_3),
            272 => Ok(Self::XMM11_0),
            273 => Ok(Self::XMM11_1),
            274 => Ok(Self::XMM11_2),
            275 => Ok(Self::XMM11_3),
            276 => Ok(Self::XMM12_0),
            277 => Ok(Self::XMM12_1),
            278 => Ok(Self::XMM12_2),
            279 => Ok(Self::XMM12_3),
            280 => Ok(Self::XMM13_0),
            281 => Ok(Self::XMM13_1),
            282 => Ok(Self::XMM13_2),
            283 => Ok(Self::XMM13_3),
            284 => Ok(Self::XMM14_0),
            285 => Ok(Self::XMM14_1),
            286 => Ok(Self::XMM14_2),
            287 => Ok(Self::XMM14_3),
            288 => Ok(Self::XMM15_0),
            289 => Ok(Self::XMM15_1),
            290 => Ok(Self::XMM15_2),
            291 => Ok(Self::XMM15_3),
            292 => Ok(Self::XMM8L),
            293 => Ok(Self::XMM9L),
            294 => Ok(Self::XMM10L),
            295 => Ok(Self::XMM11L),
            296 => Ok(Self::XMM12L),
            297 => Ok(Self::XMM13L),
            298 => Ok(Self::XMM14L),
            299 => Ok(Self::XMM15L),
            300 => Ok(Self::XMM8H),
            301 => Ok(Self::XMM9H),
            302 => Ok(Self::XMM10H),
            303 => Ok(Self::XMM11H),
            304 => Ok(Self::XMM12H),
            305 => Ok(Self::XMM13H),
            306 => Ok(Self::XMM14H),
            307 => Ok(Self::XMM15H),
            308 => Ok(Self::EMM8L),
            309 => Ok(Self::EMM9L),
            310 => Ok(Self::EMM10L),
            311 => Ok(Self::EMM11L),
            312 => Ok(Self::EMM12L),
            313 => Ok(Self::EMM13L),
            314 => Ok(Self::EMM14L),
            315 => Ok(Self::EMM15L),
            316 => Ok(Self::EMM8H),
            317 => Ok(Self::EMM9H),
            318 => Ok(Self::EMM10H),
            319 => Ok(Self::EMM11H),
            320 => Ok(Self::EMM12H),
            321 => Ok(Self::EMM13H),
            322 => Ok(Self::EMM14H),
            323 => Ok(Self::EMM15H),
            324 => Ok(Self::SIL),
            325 => Ok(Self::DIL),
            326 => Ok(Self::BPL),
            327 => Ok(Self::SPL),
            328 => Ok(Self::RAX),
            329 => Ok(Self::RBX),
            330 => Ok(Self::RCX),
            331 => Ok(Self::RDX),
            332 => Ok(Self::RSI),
            333 => Ok(Self::RDI),
            334 => Ok(Self::RBP),
            335 => Ok(Self::RSP),
            336 => Ok(Self::R8),
            337 => Ok(Self::R9),
            338 => Ok(Self::R10),
            339 => Ok(Self::R11),
            340 => Ok(Self::R12),
            341 => Ok(Self::R13),
            342 => Ok(Self::R14),
            343 => Ok(Self::R15),
            344 => Ok(Self::R8B),
            345 => Ok(Self::R9B),
            346 => Ok(Self::R10B),
            347 => Ok(Self::R11B),
            348 => Ok(Self::R12B),
            349 => Ok(Self::R13B),
            350 => Ok(Self::R14B),
            351 => Ok(Self::R15B),
            352 => Ok(Self::R8W),
            353 => Ok(Self::R9W),
            354 => Ok(Self::R10W),
            355 => Ok(Self::R11W),
            356 => Ok(Self::R12W),
            357 => Ok(Self::R13W),
            358 => Ok(Self::R14W),
            359 => Ok(Self::R15W),
            360 => Ok(Self::R8D),
            361 => Ok(Self::R9D),
            362 => Ok(Self::R10D),
            363 => Ok(Self::R11D),
            364 => Ok(Self::R12D),
            365 => Ok(Self::R13D),
            366 => Ok(Self::R14D),
            367 => Ok(Self::R15D),
            368 => Ok(Self::YMM0),
            369 => Ok(Self::YMM1),
            370 => Ok(Self::YMM2),
            371 => Ok(Self::YMM3),
            372 => Ok(Self::YMM4),
            373 => Ok(Self::YMM5),
            374 => Ok(Self::YMM6),
            375 => Ok(Self::YMM7),
            376 => Ok(Self::YMM8),
            377 => Ok(Self::YMM9),
            378 => Ok(Self::YMM10),
            379 => Ok(Self::YMM11),
            380 => Ok(Self::YMM12),
            381 => Ok(Self::YMM13),
            382 => Ok(Self::YMM14),
            383 => Ok(Self::YMM15),
            384 => Ok(Self::YMM0H),
            385 => Ok(Self::YMM1H),
            386 => Ok(Self::YMM2H),
            387 => Ok(Self::YMM3H),
            388 => Ok(Self::YMM4H),
            389 => Ok(Self::YMM5H),
            390 => Ok(Self::YMM6H),
            391 => Ok(Self::YMM7H),
            392 => Ok(Self::YMM8H),
            393 => Ok(Self::YMM9H),
            394 => Ok(Self::YMM10H),
            395 => Ok(Self::YMM11H),
            396 => Ok(Self::YMM12H),
            397 => Ok(Self::YMM13H),
            398 => Ok(Self::YMM14H),
            399 => Ok(Self::YMM15H),
            400 => Ok(Self::XMM0IL),
            401 => Ok(Self::XMM1IL),
            402 => Ok(Self::XMM2IL),
            403 => Ok(Self::XMM3IL),
            404 => Ok(Self::XMM4IL),
            405 => Ok(Self::XMM5IL),
            406 => Ok(Self::XMM6IL),
            407 => Ok(Self::XMM7IL),
            408 => Ok(Self::XMM8IL),
            409 => Ok(Self::XMM9IL),
            410 => Ok(Self::XMM10IL),
            411 => Ok(Self::XMM11IL),
            412 => Ok(Self::XMM12IL),
            413 => Ok(Self::XMM13IL),
            414 => Ok(Self::XMM14IL),
            415 => Ok(Self::XMM15IL),
            416 => Ok(Self::XMM0IH),
            417 => Ok(Self::XMM1IH),
            418 => Ok(Self::XMM2IH),
            419 => Ok(Self::XMM3IH),
            420 => Ok(Self::XMM4IH),
            421 => Ok(Self::XMM5IH),
            422 => Ok(Self::XMM6IH),
            423 => Ok(Self::XMM7IH),
            424 => Ok(Self::XMM8IH),
            425 => Ok(Self::XMM9IH),
            426 => Ok(Self::XMM10IH),
            427 => Ok(Self::XMM11IH),
            428 => Ok(Self::XMM12IH),
            429 => Ok(Self::XMM13IH),
            430 => Ok(Self::XMM14IH),
            431 => Ok(Self::XMM15IH),
            432 => Ok(Self::YMM0I0),
            433 => Ok(Self::YMM0I1),
            434 => Ok(Self::YMM0I2),
            435 => Ok(Self::YMM0I3),
            436 => Ok(Self::YMM1I0),
            437 => Ok(Self::YMM1I1),
            438 => Ok(Self::YMM1I2),
            439 => Ok(Self::YMM1I3),
            440 => Ok(Self::YMM2I0),
            441 => Ok(Self::YMM2I1),
            442 => Ok(Self::YMM2I2),
            443 => Ok(Self::YMM2I3),
            444 => Ok(Self::YMM3I0),
            445 => Ok(Self::YMM3I1),
            446 => Ok(Self::YMM3I2),
            447 => Ok(Self::YMM3I3),
            448 => Ok(Self::YMM4I0),
            449 => Ok(Self::YMM4I1),
            450 => Ok(Self::YMM4I2),
            451 => Ok(Self::YMM4I3),
            452 => Ok(Self::YMM5I0),
            453 => Ok(Self::YMM5I1),
            454 => Ok(Self::YMM5I2),
            455 => Ok(Self::YMM5I3),
            456 => Ok(Self::YMM6I0),
            457 => Ok(Self::YMM6I1),
            458 => Ok(Self::YMM6I2),
            459 => Ok(Self::YMM6I3),
            460 => Ok(Self::YMM7I0),
            461 => Ok(Self::YMM7I1),
            462 => Ok(Self::YMM7I2),
            463 => Ok(Self::YMM7I3),
            464 => Ok(Self::YMM8I0),
            465 => Ok(Self::YMM8I1),
            466 => Ok(Self::YMM8I2),
            467 => Ok(Self::YMM8I3),
            468 => Ok(Self::YMM9I0),
            469 => Ok(Self::YMM9I1),
            470 => Ok(Self::YMM9I2),
            471 => Ok(Self::YMM9I3),
            472 => Ok(Self::YMM10I0),
            473 => Ok(Self::YMM10I1),
            474 => Ok(Self::YMM10I2),
            475 => Ok(Self::YMM10I3),
            476 => Ok(Self::YMM11I0),
            477 => Ok(Self::YMM11I1),
            478 => Ok(Self::YMM11I2),
            479 => Ok(Self::YMM11I3),
            480 => Ok(Self::YMM12I0),
            481 => Ok(Self::YMM12I1),
            482 => Ok(Self::YMM12I2),
            483 => Ok(Self::YMM12I3),
            484 => Ok(Self::YMM13I0),
            485 => Ok(Self::YMM13I1),
            486 => Ok(Self::YMM13I2),
            487 => Ok(Self::YMM13I3),
            488 => Ok(Self::YMM14I0),
            489 => Ok(Self::YMM14I1),
            490 => Ok(Self::YMM14I2),
            491 => Ok(Self::YMM14I3),
            492 => Ok(Self::YMM15I0),
            493 => Ok(Self::YMM15I1),
            494 => Ok(Self::YMM15I2),
            495 => Ok(Self::YMM15I3),
            496 => Ok(Self::YMM0F0),
            497 => Ok(Self::YMM0F1),
            498 => Ok(Self::YMM0F2),
            499 => Ok(Self::YMM0F3),
            500 => Ok(Self::YMM0F4),
            501 => Ok(Self::YMM0F5),
            502 => Ok(Self::YMM0F6),
            503 => Ok(Self::YMM0F7),
            504 => Ok(Self::YMM1F0),
            505 => Ok(Self::YMM1F1),
            506 => Ok(Self::YMM1F2),
            507 => Ok(Self::YMM1F3),
            508 => Ok(Self::YMM1F4),
            509 => Ok(Self::YMM1F5),
            510 => Ok(Self::YMM1F6),
            511 => Ok(Self::YMM1F7),
            512 => Ok(Self::YMM2F0),
            513 => Ok(Self::YMM2F1),
            514 => Ok(Self::YMM2F2),
            515 => Ok(Self::YMM2F3),
            516 => Ok(Self::YMM2F4),
            517 => Ok(Self::YMM2F5),
            518 => Ok(Self::YMM2F6),
            519 => Ok(Self::YMM2F7),
            520 => Ok(Self::YMM3F0),
            521 => Ok(Self::YMM3F1),
            522 => Ok(Self::YMM3F2),
            523 => Ok(Self::YMM3F3),
            524 => Ok(Self::YMM3F4),
            525 => Ok(Self::YMM3F5),
            526 => Ok(Self::YMM3F6),
            527 => Ok(Self::YMM3F7),
            528 => Ok(Self::YMM4F0),
            529 => Ok(Self::YMM4F1),
            530 => Ok(Self::YMM4F2),
            531 => Ok(Self::YMM4F3),
            532 => Ok(Self::YMM4F4),
            533 => Ok(Self::YMM4F5),
            534 => Ok(Self::YMM4F6),
            535 => Ok(Self::YMM4F7),
            536 => Ok(Self::YMM5F0),
            537 => Ok(Self::YMM5F1),
            538 => Ok(Self::YMM5F2),
            539 => Ok(Self::YMM5F3),
            540 => Ok(Self::YMM5F4),
            541 => Ok(Self::YMM5F5),
            542 => Ok(Self::YMM5F6),
            543 => Ok(Self::YMM5F7),
            544 => Ok(Self::YMM6F0),
            545 => Ok(Self::YMM6F1),
            546 => Ok(Self::YMM6F2),
            547 => Ok(Self::YMM6F3),
            548 => Ok(Self::YMM6F4),
            549 => Ok(Self::YMM6F5),
            550 => Ok(Self::YMM6F6),
            551 => Ok(Self::YMM6F7),
            552 => Ok(Self::YMM7F0),
            553 => Ok(Self::YMM7F1),
            554 => Ok(Self::YMM7F2),
            555 => Ok(Self::YMM7F3),
            556 => Ok(Self::YMM7F4),
            557 => Ok(Self::YMM7F5),
            558 => Ok(Self::YMM7F6),
            559 => Ok(Self::YMM7F7),
            560 => Ok(Self::YMM8F0),
            561 => Ok(Self::YMM8F1),
            562 => Ok(Self::YMM8F2),
            563 => Ok(Self::YMM8F3),
            564 => Ok(Self::YMM8F4),
            565 => Ok(Self::YMM8F5),
            566 => Ok(Self::YMM8F6),
            567 => Ok(Self::YMM8F7),
            568 => Ok(Self::YMM9F0),
            569 => Ok(Self::YMM9F1),
            570 => Ok(Self::YMM9F2),
            571 => Ok(Self::YMM9F3),
            572 => Ok(Self::YMM9F4),
            573 => Ok(Self::YMM9F5),
            574 => Ok(Self::YMM9F6),
            575 => Ok(Self::YMM9F7),
            576 => Ok(Self::YMM10F0),
            577 => Ok(Self::YMM10F1),
            578 => Ok(Self::YMM10F2),
            579 => Ok(Self::YMM10F3),
            580 => Ok(Self::YMM10F4),
            581 => Ok(Self::YMM10F5),
            582 => Ok(Self::YMM10F6),
            583 => Ok(Self::YMM10F7),
            584 => Ok(Self::YMM11F0),
            585 => Ok(Self::YMM11F1),
            586 => Ok(Self::YMM11F2),
            587 => Ok(Self::YMM11F3),
            588 => Ok(Self::YMM11F4),
            589 => Ok(Self::YMM11F5),
            590 => Ok(Self::YMM11F6),
            591 => Ok(Self::YMM11F7),
            592 => Ok(Self::YMM12F0),
            593 => Ok(Self::YMM12F1),
            594 => Ok(Self::YMM12F2),
            595 => Ok(Self::YMM12F3),
            596 => Ok(Self::YMM12F4),
            597 => Ok(Self::YMM12F5),
            598 => Ok(Self::YMM12F6),
            599 => Ok(Self::YMM12F7),
            600 => Ok(Self::YMM13F0),
            601 => Ok(Self::YMM13F1),
            602 => Ok(Self::YMM13F2),
            603 => Ok(Self::YMM13F3),
            604 => Ok(Self::YMM13F4),
            605 => Ok(Self::YMM13F5),
            606 => Ok(Self::YMM13F6),
            607 => Ok(Self::YMM13F7),
            608 => Ok(Self::YMM14F0),
            609 => Ok(Self::YMM14F1),
            610 => Ok(Self::YMM14F2),
            611 => Ok(Self::YMM14F3),
            612 => Ok(Self::YMM14F4),
            613 => Ok(Self::YMM14F5),
            614 => Ok(Self::YMM14F6),
            615 => Ok(Self::YMM14F7),
            616 => Ok(Self::YMM15F0),
            617 => Ok(Self::YMM15F1),
            618 => Ok(Self::YMM15F2),
            619 => Ok(Self::YMM15F3),
            620 => Ok(Self::YMM15F4),
            621 => Ok(Self::YMM15F5),
            622 => Ok(Self::YMM15F6),
            623 => Ok(Self::YMM15F7),
            624 => Ok(Self::YMM0D0),
            625 => Ok(Self::YMM0D1),
            626 => Ok(Self::YMM0D2),
            627 => Ok(Self::YMM0D3),
            628 => Ok(Self::YMM1D0),
            629 => Ok(Self::YMM1D1),
            630 => Ok(Self::YMM1D2),
            631 => Ok(Self::YMM1D3),
            632 => Ok(Self::YMM2D0),
            633 => Ok(Self::YMM2D1),
            634 => Ok(Self::YMM2D2),
            635 => Ok(Self::YMM2D3),
            636 => Ok(Self::YMM3D0),
            637 => Ok(Self::YMM3D1),
            638 => Ok(Self::YMM3D2),
            639 => Ok(Self::YMM3D3),
            640 => Ok(Self::YMM4D0),
            641 => Ok(Self::YMM4D1),
            642 => Ok(Self::YMM4D2),
            643 => Ok(Self::YMM4D3),
            644 => Ok(Self::YMM5D0),
            645 => Ok(Self::YMM5D1),
            646 => Ok(Self::YMM5D2),
            647 => Ok(Self::YMM5D3),
            648 => Ok(Self::YMM6D0),
            649 => Ok(Self::YMM6D1),
            650 => Ok(Self::YMM6D2),
            651 => Ok(Self::YMM6D3),
            652 => Ok(Self::YMM7D0),
            653 => Ok(Self::YMM7D1),
            654 => Ok(Self::YMM7D2),
            655 => Ok(Self::YMM7D3),
            656 => Ok(Self::YMM8D0),
            657 => Ok(Self::YMM8D1),
            658 => Ok(Self::YMM8D2),
            659 => Ok(Self::YMM8D3),
            660 => Ok(Self::YMM9D0),
            661 => Ok(Self::YMM9D1),
            662 => Ok(Self::YMM9D2),
            663 => Ok(Self::YMM9D3),
            664 => Ok(Self::YMM10D0),
            665 => Ok(Self::YMM10D1),
            666 => Ok(Self::YMM10D2),
            667 => Ok(Self::YMM10D3),
            668 => Ok(Self::YMM11D0),
            669 => Ok(Self::YMM11D1),
            670 => Ok(Self::YMM11D2),
            671 => Ok(Self::YMM11D3),
            672 => Ok(Self::YMM12D0),
            673 => Ok(Self::YMM12D1),
            674 => Ok(Self::YMM12D2),
            675 => Ok(Self::YMM12D3),
            676 => Ok(Self::YMM13D0),
            677 => Ok(Self::YMM13D1),
            678 => Ok(Self::YMM13D2),
            679 => Ok(Self::YMM13D3),
            680 => Ok(Self::YMM14D0),
            681 => Ok(Self::YMM14D1),
            682 => Ok(Self::YMM14D2),
            683 => Ok(Self::YMM14D3),
            684 => Ok(Self::YMM15D0),
            685 => Ok(Self::YMM15D1),
            686 => Ok(Self::YMM15D2),
            687 => Ok(Self::YMM15D3),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for AMD64Register {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}

/// HLSL registers
#[non_exhaustive]
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum HLSLRegister {
    TEMP = 0,
    INPUT = 1,
    OUTPUT = 2,
    INDEXABLE_TEMP = 3,
    IMMEDIATE32 = 4,
    IMMEDIATE64 = 5,
    SAMPLER = 6,
    RESOURCE = 7,
    CONSTANT_BUFFER = 8,
    IMMEDIATE_CONSTANT_BUFFER = 9,
    LABEL = 10,
    INPUT_PRIMITIVEID = 11,
    OUTPUT_DEPTH = 12,
    NULL = 13,
    RASTERIZER = 14,
    OUTPUT_COVERAGE_MASK = 15,
    STREAM = 16,
    FUNCTION_BODY = 17,
    FUNCTION_TABLE = 18,
    INTERFACE = 19,
    FUNCTION_INPUT = 20,
    FUNCTION_OUTPUT = 21,
    OUTPUT_CONTROL_POINT_ID = 22,
    INPUT_FORK_INSTANCE_ID = 23,
    INPUT_JOIN_INSTANCE_ID = 24,
    INPUT_CONTROL_POINT = 25,
    OUTPUT_CONTROL_POINT = 26,
    INPUT_PATCH_CONSTANT = 27,
    INPUT_DOMAIN_POINT = 28,
    THIS_POINTER = 29,
    UNORDERED_ACCESS_VIEW = 30,
    THREAD_GROUP_SHARED_MEMORY = 31,
    INPUT_THREAD_ID = 32,
    INPUT_THREAD_GROUP_ID = 33,
    INPUT_THREAD_ID_IN_GROUP = 34,
    INPUT_COVERAGE_MASK = 35,
    INPUT_THREAD_ID_IN_GROUP_FLATTENED = 36,
    INPUT_GS_INSTANCE_ID = 37,
    OUTPUT_DEPTH_GREATER_EQUAL = 38,
    OUTPUT_DEPTH_LESS_EQUAL = 39,
    CYCLE_COUNTER = 40,
}

impl fmt::Display for HLSLRegister {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::TEMP => write!(f, "TEMP"),
            Self::INPUT => write!(f, "INPUT"),
            Self::OUTPUT => write!(f, "OUTPUT"),
            Self::INDEXABLE_TEMP => write!(f, "INDEXABLE_TEMP"),
            Self::IMMEDIATE32 => write!(f, "IMMEDIATE32"),
            Self::IMMEDIATE64 => write!(f, "IMMEDIATE64"),
            Self::SAMPLER => write!(f, "SAMPLER"),
            Self::RESOURCE => write!(f, "RESOURCE"),
            Self::CONSTANT_BUFFER => write!(f, "CONSTANT_BUFFER"),
            Self::IMMEDIATE_CONSTANT_BUFFER => write!(f, "IMMEDIATE_CONSTANT_BUFFER"),
            Self::LABEL => write!(f, "LABEL"),
            Self::INPUT_PRIMITIVEID => write!(f, "INPUT_PRIMITIVEID"),
            Self::OUTPUT_DEPTH => write!(f, "OUTPUT_DEPTH"),
            Self::NULL => write!(f, "NULL"),
            Self::RASTERIZER => write!(f, "RASTERIZER"),
            Self::OUTPUT_COVERAGE_MASK => write!(f, "OUTPUT_COVERAGE_MASK"),
            Self::STREAM => write!(f, "STREAM"),
            Self::FUNCTION_BODY => write!(f, "FUNCTION_BODY"),
            Self::FUNCTION_TABLE => write!(f, "FUNCTION_TABLE"),
            Self::INTERFACE => write!(f, "INTERFACE"),
            Self::FUNCTION_INPUT => write!(f, "FUNCTION_INPUT"),
            Self::FUNCTION_OUTPUT => write!(f, "FUNCTION_OUTPUT"),
            Self::OUTPUT_CONTROL_POINT_ID => write!(f, "OUTPUT_CONTROL_POINT_ID"),
            Self::INPUT_FORK_INSTANCE_ID => write!(f, "INPUT_FORK_INSTANCE_ID"),
            Self::INPUT_JOIN_INSTANCE_ID => write!(f, "INPUT_JOIN_INSTANCE_ID"),
            Self::INPUT_CONTROL_POINT => write!(f, "INPUT_CONTROL_POINT"),
            Self::OUTPUT_CONTROL_POINT => write!(f, "OUTPUT_CONTROL_POINT"),
            Self::INPUT_PATCH_CONSTANT => write!(f, "INPUT_PATCH_CONSTANT"),
            Self::INPUT_DOMAIN_POINT => write!(f, "INPUT_DOMAIN_POINT"),
            Self::THIS_POINTER => write!(f, "THIS_POINTER"),
            Self::UNORDERED_ACCESS_VIEW => write!(f, "UNORDERED_ACCESS_VIEW"),
            Self::THREAD_GROUP_SHARED_MEMORY => write!(f, "THREAD_GROUP_SHARED_MEMORY"),
            Self::INPUT_THREAD_ID => write!(f, "INPUT_THREAD_ID"),
            Self::INPUT_THREAD_GROUP_ID => write!(f, "INPUT_THREAD_GROUP_ID"),
            Self::INPUT_THREAD_ID_IN_GROUP => write!(f, "INPUT_THREAD_ID_IN_GROUP"),
            Self::INPUT_COVERAGE_MASK => write!(f, "INPUT_COVERAGE_MASK"),
            Self::INPUT_THREAD_ID_IN_GROUP_FLATTENED => {
                write!(f, "INPUT_THREAD_ID_IN_GROUP_FLATTENED")
            }
            Self::INPUT_GS_INSTANCE_ID => write!(f, "INPUT_GS_INSTANCE_ID"),
            Self::OUTPUT_DEPTH_GREATER_EQUAL => write!(f, "OUTPUT_DEPTH_GREATER_EQUAL"),
            Self::OUTPUT_DEPTH_LESS_EQUAL => write!(f, "OUTPUT_DEPTH_LESS_EQUAL"),
            Self::CYCLE_COUNTER => write!(f, "CYCLE_COUNTER"),
        }
    }
}

impl TryFrom<u16> for HLSLRegister {
    type Error = Error;

    fn try_from(value: u16) -> Result<Self> {
        match value {
            0 => Ok(Self::TEMP),
            1 => Ok(Self::INPUT),
            2 => Ok(Self::OUTPUT),
            3 => Ok(Self::INDEXABLE_TEMP),
            4 => Ok(Self::IMMEDIATE32),
            5 => Ok(Self::IMMEDIATE64),
            6 => Ok(Self::SAMPLER),
            7 => Ok(Self::RESOURCE),
            8 => Ok(Self::CONSTANT_BUFFER),
            9 => Ok(Self::IMMEDIATE_CONSTANT_BUFFER),
            10 => Ok(Self::LABEL),
            11 => Ok(Self::INPUT_PRIMITIVEID),
            12 => Ok(Self::OUTPUT_DEPTH),
            13 => Ok(Self::NULL),
            14 => Ok(Self::RASTERIZER),
            15 => Ok(Self::OUTPUT_COVERAGE_MASK),
            16 => Ok(Self::STREAM),
            17 => Ok(Self::FUNCTION_BODY),
            18 => Ok(Self::FUNCTION_TABLE),
            19 => Ok(Self::INTERFACE),
            20 => Ok(Self::FUNCTION_INPUT),
            21 => Ok(Self::FUNCTION_OUTPUT),
            22 => Ok(Self::OUTPUT_CONTROL_POINT_ID),
            23 => Ok(Self::INPUT_FORK_INSTANCE_ID),
            24 => Ok(Self::INPUT_JOIN_INSTANCE_ID),
            25 => Ok(Self::INPUT_CONTROL_POINT),
            26 => Ok(Self::OUTPUT_CONTROL_POINT),
            27 => Ok(Self::INPUT_PATCH_CONSTANT),
            28 => Ok(Self::INPUT_DOMAIN_POINT),
            29 => Ok(Self::THIS_POINTER),
            30 => Ok(Self::UNORDERED_ACCESS_VIEW),
            31 => Ok(Self::THREAD_GROUP_SHARED_MEMORY),
            32 => Ok(Self::INPUT_THREAD_ID),
            33 => Ok(Self::INPUT_THREAD_GROUP_ID),
            34 => Ok(Self::INPUT_THREAD_ID_IN_GROUP),
            35 => Ok(Self::INPUT_COVERAGE_MASK),
            36 => Ok(Self::INPUT_THREAD_ID_IN_GROUP_FLATTENED),
            37 => Ok(Self::INPUT_GS_INSTANCE_ID),
            38 => Ok(Self::OUTPUT_DEPTH_GREATER_EQUAL),
            39 => Ok(Self::OUTPUT_DEPTH_LESS_EQUAL),
            40 => Ok(Self::CYCLE_COUNTER),
            _ => Err(UnknownRegister(value)),
        }
    }
}

impl<'a> TryFromCtx<'a, Endian> for HLSLRegister {
    type Error = Error;

    fn try_from_ctx(this: &'a [u8], le: Endian) -> Result<(Self, usize)> {
        let (v, l) = u16::try_from_ctx(this, le)?;
        Ok((v.try_into()?, l))
    }
}
