#pragma once

#if RESHADE_LOCALIZATION

#include <imgui.h>

static void UnpackAccumulativeOffsetsIntoRanges(int base_codepoint, const short *accumulative_offsets, int accumulative_offsets_count, ImWchar *out_ranges)
{
	for (int n = 0; n < accumulative_offsets_count; n++, out_ranges += 2)
	{
		out_ranges[0] = out_ranges[1] = (ImWchar)(base_codepoint + accumulative_offsets[n]);
		base_codepoint += accumulative_offsets[n];
	}
	out_ranges[0] = 0;
}

// Original GetGlyphRangesChineseSimplifiedCommon() is not enough for Simplified Chinese actually, but the "Full" one is too heavy.
// This one contains all the Chinese characters (Hanzi) defined in GB2312. However this does not contain Greek/Cyrillic/some Hanzi numbers/..., which are also defined in GB2312.
static const ImWchar *GetGlyphRangesChineseSimplifiedGB2312()
{
	// All Simplified Chinese characters contained in GB2312 charset.
	// Include some fullwidth numbers/letters/symbols larger than U+4E00.
	static const short accumulative_offsets_from_0x4E00[] =
	{
		0,1,2,4,1,1,1,1,1,1,1,2,1,2,1,1,1,2,1,1,1,1,1,1,4,2,1,2,1,2,1,1,1,3,2,2,2,2,1,1,1,2,1,1,4,2,2,1,1,2,1,1,1,1,1,2,1,1,2,2,1,3,1,1,1,1,1,5,3,7,
		1,2,11,8,2,1,2,1,1,1,1,2,1,1,1,1,3,2,1,4,1,1,1,2,1,1,1,1,1,2,1,1,1,4,1,2,5,1,4,1,1,1,1,1,1,1,1,2,1,1,2,1,3,2,1,1,1,1,1,1,4,1,1,2,2,1,1,3,2,
		1,1,4,2,1,2,1,1,4,2,2,2,8,1,3,1,1,1,1,6,1,1,1,1,3,1,1,2,2,1,1,1,3,1,4,1,2,2,2,2,2,2,1,6,3,7,1,1,1,1,2,2,2,1,1,1,1,1,1,1,1,1,3,1,1,2,2,3,3,1,
		3,1,2,5,1,2,1,4,1,4,1,2,2,2,2,3,3,4,2,3,3,2,1,1,1,1,1,2,2,1,6,10,4,1,1,5,4,1,1,1,6,1,2,2,1,1,1,2,2,3,2,1,1,3,1,1,2,2,5,2,4,14,1,2,3,2,4,1,1,
		2,3,2,4,1,2,1,1,2,1,1,12,2,2,5,4,1,4,2,1,6,5,2,9,7,10,1,4,3,1,1,5,3,5,22,2,2,1,1,3,6,8,1,20,7,4,12,1,5,1,1,5,2,6,11,1,4,7,15,30,1,1,2,1,1,1,
		2,1,2,2,4,3,1,1,4,2,6,3,3,3,1,1,1,2,1,2,1,1,1,1,1,1,2,1,1,3,1,1,3,3,1,3,1,5,3,1,1,2,2,1,4,2,2,1,6,1,3,1,1,1,1,2,1,1,4,1,1,3,4,2,1,2,2,1,3,2,
		10,2,3,1,3,7,2,2,1,3,2,1,2,1,1,1,1,1,2,1,1,1,1,3,1,1,2,3,1,3,1,4,1,1,1,1,1,2,3,4,4,1,2,2,1,2,3,3,1,1,1,1,1,2,2,1,1,1,1,7,2,1,3,1,3,2,6,2,3,
		4,2,2,1,5,3,11,2,2,1,6,8,3,8,2,1,1,1,1,1,1,5,1,1,1,1,1,4,1,1,11,1,4,4,2,2,5,2,4,2,7,5,12,9,1,4,1,1,5,1,2,5,2,1,5,1,1,2,1,3,3,1,2,3,4,4,11,1,
		1,3,1,2,2,2,2,1,1,1,4,1,2,1,1,2,1,1,3,2,2,1,1,1,1,1,1,2,1,2,2,3,1,1,1,2,1,1,2,1,2,5,3,2,1,1,3,2,1,1,8,3,2,3,2,3,1,2,1,2,1,5,8,5,4,2,1,6,1,1,
		1,1,1,4,3,2,1,1,1,2,4,1,3,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,2,1,1,4,2,1,2,2,1,1,2,1,1,1,1,1,1,1,2,1,1,4,2,1,1,1,1,2,3,1,1,1,2,1,1,1,1,
		2,1,2,1,3,1,2,1,2,2,3,3,2,2,1,5,2,1,1,1,1,1,1,1,2,1,6,2,2,2,9,1,1,2,1,1,1,3,1,1,3,2,2,2,5,1,2,1,1,2,2,1,1,3,1,1,2,6,1,2,1,1,1,1,1,1,1,2,2,2,
		1,4,3,2,2,1,1,1,2,2,1,1,1,3,1,1,1,1,1,1,1,1,1,2,2,1,2,1,1,1,6,1,1,1,1,1,3,1,4,1,7,2,1,2,2,5,1,2,6,1,1,3,7,5,2,1,1,3,3,2,2,1,1,1,2,4,5,2,1,2,
		2,1,2,3,1,6,5,1,6,5,3,1,1,1,3,2,1,1,7,1,1,1,3,1,2,2,1,1,1,1,3,1,1,1,1,4,2,3,4,1,3,1,2,8,10,2,2,2,2,2,2,1,6,1,4,3,1,4,1,1,1,2,6,1,2,2,2,1,1,
		1,2,2,1,1,3,3,1,2,2,6,1,2,1,7,1,3,2,1,9,3,3,1,4,1,3,5,1,4,1,2,2,3,2,4,13,1,1,6,3,1,1,3,6,2,4,1,1,1,1,5,5,5,1,9,1,8,1,4,16,12,8,5,14,10,3,3,
		1,2,1,1,1,1,1,2,7,2,3,1,3,1,4,1,3,1,1,4,1,2,2,1,1,18,3,4,5,1,1,2,1,1,1,1,3,6,1,1,3,2,2,5,3,2,1,1,1,1,1,6,3,1,1,1,1,1,1,1,3,2,2,1,1,1,2,2,4,
		3,1,4,1,6,1,1,1,1,5,1,6,1,8,5,1,1,1,1,2,1,2,2,2,1,4,2,4,10,1,8,3,1,3,2,1,3,1,1,3,2,1,4,9,2,5,4,1,1,3,3,2,4,1,4,2,4,4,4,5,2,1,3,6,6,5,15,8,1,
		4,3,4,6,7,6,1,18,2,1,2,2,4,9,1,6,1,4,1,9,1,19,5,4,12,4,15,7,1,2,2,3,3,3,9,2,3,6,2,5,1,1,3,1,2,3,5,1,2,2,1,1,2,1,1,2,3,3,1,1,1,2,5,1,2,3,1,1,
		2,3,1,2,3,1,1,1,1,2,6,2,3,14,1,2,2,1,4,4,1,1,1,2,1,1,2,3,5,1,3,1,2,5,5,1,1,3,1,1,1,3,1,3,7,2,3,8,4,1,5,1,1,1,1,3,1,2,2,1,6,2,3,4,13,2,4,2,2,
		1,1,1,1,1,1,3,5,2,5,4,3,1,3,2,4,8,1,2,2,6,4,6,3,1,11,5,8,5,3,10,1,1,1,3,5,19,8,1,15,8,1,2,3,6,3,1,7,3,6,2,2,2,4,4,1,2,3,3,2,6,24,13,1,27,2,
		3,9,16,1,2,1,1,2,1,1,1,1,1,1,2,3,1,1,1,1,3,1,2,4,1,2,2,5,3,3,1,2,1,1,2,1,1,2,1,3,4,2,2,1,1,1,1,1,1,1,2,1,1,1,1,1,1,4,1,5,3,1,1,1,2,1,4,1,1,
		3,2,1,1,1,5,4,2,1,10,1,1,2,3,1,3,6,2,8,1,1,1,1,3,2,3,2,3,1,5,2,3,1,1,2,2,2,1,5,2,1,2,5,5,3,4,1,1,1,1,1,1,1,1,1,1,3,3,1,1,1,3,1,1,1,4,4,5,2,
		1,2,2,1,8,1,2,8,1,5,2,1,6,4,1,3,1,3,2,1,1,1,1,1,1,6,1,6,2,1,1,4,2,2,2,1,5,2,2,3,7,7,7,8,3,1,1,2,2,3,3,11,7,1,3,1,7,6,2,5,3,6,2,1,2,4,1,6,9,
		1,9,3,1,1,12,3,2,12,2,1,3,5,14,23,4,26,14,8,14,2,1,3,1,3,1,1,1,1,2,3,1,2,1,1,1,3,6,1,3,1,1,2,1,2,4,3,1,1,3,1,1,2,1,1,1,1,1,9,1,6,1,3,6,1,3,
		1,1,5,2,1,7,8,1,6,3,3,1,16,1,1,2,2,2,1,1,1,2,1,4,2,1,3,1,4,1,1,2,1,1,1,1,2,1,2,2,1,1,5,1,1,6,6,2,1,1,1,1,5,11,1,7,1,1,3,5,13,2,10,2,1,3,4,1,
		1,1,1,1,1,4,2,1,4,2,2,2,2,1,3,4,1,5,1,1,2,1,3,2,2,8,1,2,4,16,2,1,2,1,1,1,5,4,1,2,2,3,1,2,1,3,1,2,4,2,2,1,4,1,1,2,1,2,1,1,1,1,1,4,2,3,2,1,1,
		3,5,7,2,3,1,7,2,5,1,6,1,1,1,3,3,1,2,1,1,1,4,1,1,1,4,3,1,3,3,3,1,2,1,3,4,3,3,2,1,1,1,1,1,1,1,1,1,4,3,1,1,3,2,1,1,3,2,1,1,3,1,4,1,1,1,1,1,1,4,
		6,6,1,3,2,1,1,8,2,3,2,3,4,1,3,5,1,1,3,1,1,1,1,1,1,2,1,3,3,2,1,1,1,1,1,2,4,1,5,3,1,5,2,2,4,1,2,2,1,3,3,2,3,1,1,2,2,1,2,4,3,1,9,1,4,1,6,4,3,2,
		2,1,2,1,6,1,1,1,2,1,1,1,1,1,3,1,2,3,1,6,1,5,2,1,4,1,1,6,5,5,1,3,1,2,1,4,20,9,2,2,2,3,4,8,5,5,1,8,5,2,20,3,6,9,10,1,1,3,11,7,4,6,2,1,6,1,2,
		18,15,10,7,2,2,1,1,1,1,1,1,1,1,3,1,1,1,2,1,4,2,1,2,1,5,2,2,5,1,3,6,1,1,1,1,2,4,1,1,2,1,1,1,3,1,1,1,4,3,8,3,1,2,1,1,1,1,1,1,1,3,3,3,3,2,1,1,
		4,5,1,7,1,1,2,1,1,1,2,1,4,1,1,1,2,1,3,3,1,5,4,4,2,1,1,2,3,2,1,1,1,1,1,1,2,1,1,2,2,1,1,2,1,1,1,1,1,1,3,3,1,2,1,1,1,1,3,1,1,1,2,2,1,2,1,5,1,1,
		1,2,1,5,1,1,5,3,5,4,1,2,1,1,1,1,1,1,1,1,3,2,1,4,3,7,1,3,5,1,2,1,3,2,1,1,1,1,1,5,9,1,2,1,1,4,2,4,1,3,5,1,3,1,5,2,5,1,1,1,2,2,1,1,2,4,2,8,2,1,
		2,2,1,1,1,2,1,1,2,3,1,3,1,2,2,2,6,2,3,4,1,1,1,2,4,8,2,1,2,6,1,3,5,2,2,2,5,2,1,1,1,3,6,1,3,1,3,1,7,1,2,2,1,5,4,2,1,7,6,3,2,2,3,1,1,1,1,2,8,2,
		4,6,9,2,4,11,1,1,8,2,1,2,10,1,3,1,3,5,6,5,3,1,1,2,5,2,1,2,2,4,2,3,8,1,2,2,5,1,6,4,2,2,26,9,9,6,13,6,3,1,5,1,1,2,1,2,3,1,6,3,1,3,3,2,4,1,3,2,
		2,1,4,1,3,5,1,4,2,2,3,16,4,1,4,1,2,4,2,2,1,3,2,3,1,2,2,2,2,2,1,9,3,1,4,2,1,1,1,5,1,2,1,3,4,1,9,2,3,1,1,1,1,3,1,1,1,1,1,5,1,3,6,2,1,3,4,2,2,
		1,4,1,1,4,4,2,1,5,2,1,5,2,2,3,1,1,6,2,3,2,8,1,3,3,1,1,1,1,1,3,5,2,3,2,2,6,1,1,4,2,1,3,4,4,2,3,5,5,5,1,6,10,1,6,6,5,5,27,2,1,1,9,3,7,2,1,1,3,
		2,3,2,1,1,8,1,1,1,2,3,4,1,2,4,2,2,7,2,2,1,1,1,2,2,3,1,5,3,3,2,1,3,2,1,3,2,1,1,1,2,3,6,2,1,1,1,4,3,1,1,3,2,1,2,1,2,2,5,2,1,2,3,3,2,2,5,5,2,1,
		2,2,1,1,4,1,2,2,1,2,1,2,2,1,3,2,1,1,1,9,2,1,11,1,1,1,1,1,4,1,1,2,1,1,2,2,3,4,3,3,1,1,2,1,9,2,1,5,2,1,1,1,1,1,2,1,2,2,3,1,6,4,8,1,8,1,4,1,1,
		3,1,1,2,1,1,1,1,1,1,2,1,1,2,2,2,1,2,1,1,11,1,1,1,1,1,1,1,1,1,2,9,2,1,10,2,2,1,9,4,4,11,4,1,1,5,2,1,3,2,11,2,7,2,2,5,3,3,2,6,3,11,2,1,4,4,1,
		2,5,4,6,2,1,2,2,13,1,4,9,3,4,5,4,2,3,18,3,3,3,1,2,3,3,5,3,3,6,1,1,3,4,2,2,2,1,1,1,4,7,1,3,3,1,11,1,3,2,4,3,3,4,6,9,2,2,2,4,7,4,1,13,5,11,2,
		11,13,1,7,2,7,2,5,2,4,8,1,6,3,9,8,1,2,6,4,12,8,3,4,4,10,2,1,6,9,9,2,1,116,1,1,1,1,3,11,5,2,1,4,5,3,1,2,3,13,9,1,1,1,1,1,3,15,2,1,5,1,1,1,2,
		1,2,1,1,2,5,1,3,4,1,6,9,9,1,1,2,8,2,1,3,6,2,2,3,1,1,1,1,1,2,2,6,9,1,4,4,2,4,4,8,1,1,6,2,1,1,2,1,1,1,2,1,1,1,4,2,1,2,2,1,1,1,1,4,1,1,2,2,1,3,
		5,3,1,1,4,1,2,1,6,4,1,2,4,1,1,1,1,1,1,3,4,1,1,6,2,2,2,3,4,1,3,1,1,2,1,2,1,3,3,1,3,1,5,2,4,2,2,1,1,1,1,2,1,1,2,1,3,1,1,5,1,2,1,1,1,1,5,1,4,1,
		2,4,3,1,1,1,1,4,3,2,1,1,1,2,3,2,1,3,1,1,1,2,2,1,1,1,2,1,1,1,1,3,3,3,4,1,2,4,5,2,1,1,3,7,2,3,1,3,3,1,1,2,4,2,1,1,1,3,2,2,1,1,1,2,1,2,1,1,2,1,
		1,1,5,1,2,2,2,3,3,3,1,4,1,5,3,1,4,6,3,3,1,3,2,3,2,1,1,6,2,1,1,1,1,2,1,2,1,1,1,1,1,3,1,3,3,3,7,1,4,1,1,1,4,1,5,5,2,1,4,1,2,1,3,2,5,1,2,3,2,4,
		2,2,1,9,5,2,1,1,2,1,3,2,1,3,3,3,1,2,1,1,4,2,2,2,3,2,4,2,9,1,9,1,5,1,2,2,3,4,12,3,16,1,4,2,1,3,6,1,8,4,3,3,3,2,3,5,2,1,2,2,1,3,1,2,4,1,5,4,4,
		2,2,1,1,2,3,4,1,1,1,1,2,1,1,2,1,11,5,9,4,3,6,4,1,1,11,4,5,1,1,2,2,2,2,3,8,8,1,4,2,11,4,2,4,4,7,1,4,2,4,2,2,8,4,1,3,1,1,14,5,6,12,3,3,7,2,7,
		8,1,12,2,1,13,1,34,9,1,8,18,4,19,3,15,13,1,1,2,1,5,1,2,4,2,1,1,5,4,1,4,4,2,1,1,3,3,1,14,1,1,1,1,2,2,4,1,1,2,1,1,3,1,1,1,5,2,14,1,2,4,5,2,1,
		1,1,2,1,1,2,8,2,4,12,1,6,3,2,1,2,1,1,12,9,1,1,5,15,5,2,2,14,2,6,2,1,1,6,4,1,5,2,3,7,6,5,5,4,1,6,1,8,4,7,2,4,10,11,6,1,11,5,2,7,11,13,23,11,
		2,2,4,1,4,1,1,1,1,2,2,2,8,1,4,1,5,4,3,2,2,2,2,1,4,1,2,5,1,3,3,4,1,4,1,1,1,3,6,1,2,2,3,13,13,1,2,1,4,2,1,1,1,8,1,1,1,4,5,1,2,2,5,2,5,2,1,7,1,
		3,1,1,1,1,1,1,1,1,3,1,2,1,1,5,2,7,4,5,2,1,1,4,1,1,1,3,1,3,4,1,1,1,2,3,3,3,1,1,5,1,14,3,2,5,9,12,1,2,15,6,3,2,2,3,3,5,5,4,3,7,2,3,1,1,2,1,4,
		3,1,5,2,6,1,1,3,2,1,1,8,5,2,5,2,2,4,5,17,2,1,3,1,5,1,10,1,7,3,1,2,2,2,2,2,3,1,1,1,6,5,20,2,2,1,1,1,2,1,14,3,6,1,7,2,1,2,4,4,3,2,12,2,9,1,1,
		17,24,10,2,2,2,1,1,2,8,1,5,2,1,8,5,9,2,2,2,5,1,1,2,3,6,3,1,2,1,1,2,1,1,1,1,2,2,1,2,1,3,2,5,3,3,1,2,1,5,5,1,1,1,9,1,4,8,2,4,1,6,4,3,5,4,2,1,
		2,2,1,2,1,3,2,1,1,2,1,1,6,1,2,1,1,1,1,1,1,1,3,1,3,1,1,4,1,1,1,2,1,1,1,3,5,2,1,1,2,3,3,4,1,1,2,1,1,2,1,5,1,3,5,3,3,1,1,2,2,5,2,4,5,2,1,1,2,4,
		1,2,2,1,1,3,1,1,2,3,3,1,1,3,4,2,1,1,3,9,1,7,2,6,2,5,8,4,9,3,2,1,4,2,2,1,1,3,3,3,2,3,3,11,10,3,1,2,11,3,3,1,2,2,3,1,1,1,1,1,2,2,1,1,3,4,6,9,
		1,2,1,2,4,1,3,2,3,3,3,1,1,2,2,12,1,5,1,2,4,2,1,4,2,6,1,1,1,2,2,4,1,2,4,9,1,9,1,6,1,1,2,1,2,3,1,13,4,1,1,1,4,1,7,1,1,3,1,13,1,2,3,2,2,1,2,4,
		3,2,6,2,2,14,10,4,1,6,1,2,2,2,2,1,1,1,5,3,2,5,1,1,1,1,1,7,3,1,4,1,2,2,1,1,2,2,1,2,4,2,1,1,2,3,1,3,4,3,1,1,1,1,1,2,2,5,2,5,2,2,2,3,1,1,6,13,
		1,1,1,1,9,5,11,2,3,1,1,3,2,4,1,2,1,1,3,2,2,2,2,9,1,1,1,1,5,5,3,4,4,1,1,5,4,1,4,15,4,6,2,3,3,7,4,14,11,6,16,6,1,1,2,2,1,5,2,1,10,3,1,3,1,1,1,
		1,1,1,2,3,2,1,5,2,8,1,2,6,1,3,1,5,5,11,13,12,6,1,1,2,1,2,1,2,3,3,2,2,4,1,3,3,7,4,1,2,1,2,2,2,2,1,8,3,2,3,2,1,3,5,2,1,6,3,3,4,2,3,16,4,2,2,1,
		1,2,7,11,6,25,4,2,1,1,1,1,5,1,1,2,1,2,2,5,4,1,3,1,1,1,4,1,2,1,5,1,2,4,1,6,12,12,11,3,5,1,1,3,2,1,7,2,10,1,3,2,4,1,2,2,2,1,4,2,3,1,4,2,3,2,4,
		1,1,2,2,1,1,2,3,2,5,2,2,2,7,2,2,2,1,3,1,1,1,2,2,2,2,1,2,3,2,12,3,1,3,2,2,2,3,2,5,8,3,4,1,2,5,1,4,1,4,1,1,1,1,1,1,1,4,3,4,9,5,1,5,5,2,6,1,3,
		4,4,1,4,4,3,6,5,2,9,4,1,3,7,9,7,1,3,14,7,1,1,12,38,1,7,1,1,12,8,1,3,2,1,4,1,1,1,3,2,1,5,4,3,1,1,6,3,1,1,3,4,2,1,2,2,1,8,1,1,2,3,3,1,8,7,9,3,
		15,22,2,5,4,4,63,9,47,8,141,6,6,59,25,4,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,2,1,1,1,1,1,3,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		2,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,
		1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,8,2,1,11,1,3,1,2,1,2,5,2,1,6,1,1,4,3,1,2,5,5,3,9,2,2,6,6,3,1,1,2,3,3,8,1,2,6,1,3,1,2,2,4,5,2,2,
		6,1,3,7,1,1,4,1,3,5,2,1,2,8,1,4,1,2,1,1,1,5,1,1,3,2,2,1,1,1,1,1,3,4,2,3,1,1,1,1,1,7,2,2,1,1,1,3,2,2,3,1,3,4,1,1,1,5,2,4,2,15,1,7,14,1,3,1,2,
		1,2,2,1,7,3,2,2,1,1,1,2,1,1,1,2,1,4,1,1,2,1,1,2,1,2,3,3,2,1,1,1,1,1,1,1,1,2,6,1,1,8,1,2,1,1,1,1,1,3,3,1,2,1,1,1,1,1,1,2,1,1,1,1,1,2,2,2,2,1,
		5,4,3,1,3,1,1,1,1,1,1,1,2,2,2,4,14,3,2,1,4,2,6,8,2,2,1,1,4,1,2,1,1,4,1,6,5,2,2,4,1,2,1,3,5,1,1,1,1,1,1,1,2,6,2,5,2,7,3,1,1,6,3,2,2,9,7,1,5,
		1,1,1,3,4,2,23,4,3,2,1,6,1,7,1,2,2,1,1,2,1,1,6,1,3,2,2,7,1,2,1,2,1,1,5,2,1,1,1,2,1,1,2,1,1,1,1,1,1,2,3,6,3,2,2,4,9,2,5,9,6,1,1,2,1,1,5,1,3,
		1,1,3,2,4,2,1,2,1,1,2,1,5,1,1,3,1,2,2,3,1,1,2,1,1,1,1,1,1,1,1,1,2,1,3,1,1,4,1,3,3,3,1,2,1,1,1,1,1,2,1,1,1,1,2,1,3,1,2,1,1,1,2,1,1,1,5,4,2,3,
		3,2,2,6,1,1,1,1,1,1,1,1,3,2,1,2,3,1,2,3,1,1,11,1,3,1,1,2,2,2,1,1,1,2,1,1,2,4,3,3,1,2,6,1,1,1,2,6,1,1,2,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,8,1,
		3,1,1,8,1,3,5,4,1,5,3,1,2,2,8,1,1,1,5,1,1,1,1,2,1,1,1,1,2,1,3,1,4,2,3,2,3,5,2,2,4,1,2,1,1,4,4,1,6,1,1,6,1,4,4,2,1,2,5,1,1,1,1,2,7,4,1,7,1,1,
		1,1,9,7,4,1,9,11,6,2,1,1,1,5,2,6,2,1,1,4,2,2,1,2,2,8,5,1,1,2,1,2,9,2,3,5,14,3,2,4,1,1,3,2,2,3,5,1,3,3,1,2,3,7,2,1,3,2,1,6,4,12,3,13,5,2,4,3,
		5,2,10,1,11,1,1,1,1,1,1,6,5,1,1,12,3,5,6,4,10,2,5,1,1,3,6,1,2,8,12,1,8,4,2,4,1,1,7,2,8,8,6,1,3,2,7,8,5,18,4,6,12,5,17,2,15,4,17,1,1,1,1,3,6,
		4,4,9,1,2,3,8,1,1,1,1,1,1,1,1,1,8,1,1,1,6,2,7,1,6,1,3,1,1,1,2,3,1,1,3,1,1,4,6,4,2,1,2,1,1,3,1,1,1,3,4,1,2,3,1,5,5,4,1,2,1,1,1,1,4,1,5,2,2,1,
		4,1,1,1,3,5,1,2,2,1,2,2,2,3,1,1,2,4,5,3,3,3,4,3,1,8,1,1,3,2,5,4,2,7,3,1,1,9,2,4,2,5,1,1,1,4,1,2,3,3,2,6,4,8,9,3,1,1,2,4,2,5,3,3,6,4,1,6,1,1,
		8,5,5,5,4,11,5,5,7,9,2,1,5,6,1,16,7,3,4,4,1,7,1,7,3,2,8,1,1,1,1,3,1,2,1,2,2,2,5,6,1,1,2,1,2,1,3,3,2,5,4,6,6,2,7,2,4,6,5,5,1,3,1,3,5,4,2,1,3,
		1,6,3,1,1,1,3,8,1,2,1,4,1,3,2,4,8,6,2,1,6,1,1,6,4,5,1,5,4,2,11,3,26,1,7,21,4,2,2,3,59,1,2,1,1,1,1,1,1,1,1,2,1,1,1,1,4,4,2,2,5,2,1,5,4,4,13,
		7,55,10,49,16,1,9,116,31,64,6,52,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,2,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,
		1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,
		1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,5,1,2,3,9,5,7,1,8,1,8,5,1,1,8,3,4,1,2,8,4,133,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,3,1,1,1,2,1,1,1,1,1,1,2,1,1,1,1,2,1,4,2,3,3,1,1,1,1,10,3,1,5,1,6,3,11,4,14,2,1,1,3,2,2,2,
		1,4,1,2,5,1,2,1,2,5,1,3,1,2,1,1,4,1,4,2,1,1,3,4,2,2,1,1,1,1,2,8,4,1,2,3,5,9,1,1,3,1,6,1,2,2,1,2,4,4,1,3,3,1,1,2,3,1,1,1,1,6,1,7,13,3,3,1,2,
		1,2,2,2,6,3,2,4,2,8,1,4,8,2,13,1,3,3,8,20,152,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,2,1,1,1,1,1,1,
		1,1,1,2,1,4,5,1,2,5,1,5,3,4,1,3,1,2,1,2,1,6,2,1,2,1,1,3,1,2,1,1,1,1,3,2,1,1,2,1,1,1,2,1,2,3,4,1,1,4,3,1,1,1,1,1,1,3,1,1,2,2,1,1,1,2,2,1,3,1,
		2,1,1,1,1,1,4,7,1,1,6,1,2,3,1,2,3,1,2,3,6,2,1,1,1,1,4,1,3,7,1,2,3,5,1,6,1,8,2,1,2,1,5,3,6,2,2,2,2,2,2,4,1,1,3,4,2,2,1,1,1,2,1,1,1,2,1,1,1,3,
		3,3,1,2,3,4,1,1,1,2,4,4,1,1,4,1,4,1,1,3,2,2,5,4,5,1,4,2,21,5,4,1,12,2,8,10,3,3,1,1,1,1,1,1,1,2,5,3,3,1,3,1,1,1,1,4,1,2,2,1,1,1,1,2,1,1,1,1,
		1,4,1,1,6,2,2,2,1,1,3,1,1,8,1,7,1,7,3,1,1,5,1,5,13,2,1,2,1,1,1,2,11,152,26,32,26,118,44,37,59,12,104,45,26,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,2,
		2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,2,1,1,3,1,1,1,1,2,1,1,1,1,1,1,1,1,
		1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,2,1,1,1,1,1,2,1,1,1,1,1,1,
		1,1,1,1,2,1,1,1,1,1,1,1,1,2,1,1,1,1,2,1,1,1,2,1,2,1,1,1,1,1,1,1,1,2,1,1,1,2,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,3,9,105,1,1,1,2,1,1,
		1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,2,1,1,1,2,1,2,1,2,2,1,8,4,3,1,1,1,1,1,5,1,1,2,1,2,2,1,1,1,1,1,2,1,1,3,4,1,
		6,4,2,1,2,3,1,1,1,2,6,2,1,1,1,14,1,2,3,2,2,1,4,3,1,1,3,11,9,3,3,3,3,1,1,2,1,3,1,1,1,2,3,1,1,1,3,3,11,8,1,1,5,4,3,1,2,5,2,1,3,2,1,1,1,4,1,1,
		4,3,6,2,12,3,3,2,6,1,5,20,1,3,3,2,3,2,1,1,3,4,10,1,2,6,9,6,2,4,1,2,4,8,3,8,2,2,3,2,50,1,2,1,1,1,1,6,2,1,127,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		1,1,1,1,1,2,1,2,1,1,2,1,2,1,1,2,1,1,1,1,1,1,1,1,2,1,1,1,39,3,1,1,2,3,1,1,4,1,8,1,37,3,30,38,1,14,2,2,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,3,2,1,2,
		1,1,3,1,1,1,1,2,1,2,2,1,1,1,1,1,1,1,1,1,1,12,3,196,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,2,1,1,3,1,1,1,1,3,1,1,2,1,1,1,1,1,1,
		1,1,1,1,1,2,1,8,1,5,1,1,2,2,4,1,1,3,6,1,5,2,5,7,2,5,5,2,2,10,2,8,5,7,4,12,4,12,3,9,1,5,1,1,1,1,2,1,1,4,2,2,3,296,3,2,1,3,1,1,1,3,2,1,2,1,1,
		2,1,5,1,1,2,1,1,1,1,1,1,1,1,1,1,1,2,2,1,2,1,1,1,1,1,1,1,1,2,1,1,1,6,1,1,1,1,3,1,1,1,1,1,1,3,1,1,1,1,1,1,3,1,1,1,3,317,1,1,1,1,2,1,2,1,1,1,1,
		1,2,2,1,1,2,1,1,1,1,1,3,1,1,2,1,1,1,1,1,1,1,1,1,1,1,2,1,2,4,2,1,2,1,1,2,5,1,2,1,1,1,1,1,1,1,3,1,2,11,1,3,5,1,3,7,1,10,2,7,14,4,3,2,1,6,5,4,
		1,1,2,3,4,3,1,1,2,1,2,3,2,2,1,5,10,2,1,2,13,2,1,2,3,4,2,7,2,10,3,8,2,2,2,1,6,12,1,46,1,3,1,1,1,1,1,1,1,1,1,13,1,1,4,1,24209,2,1,1,1,1,1,1,1,
		1,1,1,1,1,1,1,1,1,1,189,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,130,1,2,2
	};
	static ImWchar base_ranges[] = // not zero-terminated
	{
		0x0020, 0x00FF, // Basic Latin + Latin Supplement
		0x2000, 0x206F, // General Punctuation
		0x3000, 0x30FF, // CJK Symbols and Punctuations, Hiragana, Katakana
		0x31F0, 0x31FF, // Katakana Phonetic Extensions
		0xFF00, 0xFFEF, // Half-width characters
		0xFFFD, 0xFFFD  // Invalid
	};
	static ImWchar full_ranges[IM_ARRAYSIZE(base_ranges) + IM_ARRAYSIZE(accumulative_offsets_from_0x4E00) * 2 + 1] = { 0 };
	if (*full_ranges == '\0')
	{
		memcpy(full_ranges, base_ranges, sizeof(base_ranges));
		UnpackAccumulativeOffsetsIntoRanges(0x4E00, accumulative_offsets_from_0x4E00, IM_ARRAYSIZE(accumulative_offsets_from_0x4E00), full_ranges + IM_ARRAYSIZE(base_ranges));
	}
	return full_ranges;
}

#endif
