#!/usr/bin/env python3

import importlib.util
import os

from fenrirscreenreader.core.i18n import _

# Load base configuration class
_base_path = os.path.join(os.path.dirname(__file__), "..", "config_base.py")
_spec = importlib.util.spec_from_file_location("config_base", _base_path)
_module = importlib.util.module_from_spec(_spec)
_spec.loader.exec_module(_module)
config_command = _module.config_command


class command(config_command):
    def __init__(self):
        super().__init__()

    def get_description(self):
        return "Select screen driver"

    def run(self):
        current_driver = self.get_setting("screen", "driver", "vcsaDriver")

        # Present current driver
        driver_descriptions = {
            "vcsaDriver": "VCSA Driver - Linux TTY console access",
            "ptyDriver": "PTY Driver - terminal emulation",
            "dummyDriver": "Dummy Driver - for testing only",
            "debugDriver": "Debug Driver - development/debugging",
        }

        current_description = driver_descriptions.get(
            current_driver, "Unknown driver"
        )
        self.present_text(f"Current screen driver: {current_description}")

        # Cycle through the available drivers
        drivers = ["vcsaDriver", "ptyDriver", "dummyDriver", "debugDriver"]
        try:
            current_index = drivers.index(current_driver)
            next_index = (current_index + 1) % len(drivers)
            new_driver = drivers[next_index]
        except ValueError:
            new_driver = "vcsaDriver"  # Default to VCSA

        success = self.set_setting("screen", "driver", new_driver)

        if success:
            new_description = driver_descriptions[new_driver]
            self.present_text(f"Screen driver changed to: {new_description}")
            self.present_text(
                "Please restart Fenrir for this change to take effect."
            )
            self.play_sound("Accept")
        else:
            self.present_text("Failed to change screen driver")
            self.play_sound("Error")
